#!/usr/bin/perl
#############################################################################
## Crytek Source File
## Copyright (C) 2007, Crytek Studios
##
## Creator: Sascha Demetrio
## Date: Jul 17, 2007
## Description: GNU-make based build system
#############################################################################

# This script perfroms the post-scan actions.

use warnings;
use strict;

use Elements;
use Getopt::Long qw(:config no_auto_abbrev bundling);

# Stage clean variables names.  For every rule generating output files, the
# resulting files are added to the *_CLEAN variable of the corresponding
# stage.  The script assumes that the C++ code is generated in the post-scan
# step and that the compiling and linking is done in the post-link step (the
# rationale behind moving the compilation into the post-link is that we might
# want to generate some include files from the compiled PPU ELF binary).

# Parse the command line.
my $elementFileName;
my $makeFileName;
my $targetName;
my %optionPrefix = ( );
my $optionHelp = 0;
my $optionVerbose = 0;
GetOptions(
		'E|element-file=s' => \$elementFileName,
		'm|makefile=s' => \$makeFileName,
		't|target=s' => \$targetName,
		'P|prefix=s' => \%optionPrefix,
		'h|help' => \$optionHelp,
		'v|verbose!' => \$optionVerbose
		) or exit 1;
if ($optionHelp)
{
	print <<EOF;
postscan.pl: Script executing the post-scan action.
Synopsis:
  \$PERL postscan.pl (Options)
Options:
-E|--element-file FILE
  Specify the element file name.  This option is required.
-m|--makefile FILE
  Specify the output makefile to be generated.  This option is required.
-P|--prefix PREFIX=SUBST
  Specify a 
-v|--verbose
  Verbose operation.
-h|--help
  Display this help screen and exit.
EOF
	exit 0;
}
if (not defined $elementFileName)
{
  print STDERR "postscan: no element file specified!\n";
	exit 1;
}
if (not defined $makeFileName)
{
	print STDERR "postscan: no output makefile specified!\n";
	exit 1;
}
if (not defined $targetName)
{
	print STDERR "postscan: no target name specified!\n";
	exit 1;
}
my $spuJobDir = "$targetName/spujob";

# The clean variable collection all generated C++ files.
my $genCleanVar = "SCAN_CLEAN_$targetName";

# The clean variable for compiled output files, not including compiled files
# generated in a combined compile/link rule.
my $compileCleanVar = "LINK_CLEAN_$targetName";

# The clean variable for linker output files.
my $linkCleanVar = "LINK_CLEAN_$targetName";

# Substitute matching filename prefixes.
#
# Parameters:
# - The filename.
sub substPrefix ($)
{
	my $fileName = shift;
	my @matches = ( );

	foreach my $prefix (keys %optionPrefix)
	{
		my $expanded = $optionPrefix{$prefix};
		if (index($fileName, $expanded) == 0) { push @matches, $prefix; }
	}
	if (@matches)
	{
		my $longestMatch = pop @matches;
		my $longestMatchExpanded = $optionPrefix{$longestMatch};
		foreach my $match (@matches)
		{
			my $matchExpanded = $optionPrefix{$match};
			if (length($matchExpanded) > length($longestMatchExpanded))
			{
				$longestMatch = $match;
				$longestMatchExpanded = $matchExpanded;
			}
		}
		$fileName = $longestMatch.substr($fileName, length $longestMatchExpanded);
	}
	$fileName =~ s#/\./#/#g;
	return $fileName;
}

# Substitute a file name extension.
#
# Parameters.
# - The file name.
# - The expected extension (without the dot).
# - The new extension (without the dot).  If this is empty, then the extension
#   dot is stripped.
sub substExt ($$$)
{
	my $fileName = shift;
	my $ext = shift;
	my $newExt = shift;

	my $dotIndex = rindex $fileName, '.'.$ext;
	if ($dotIndex == -1)
	{
		die "expected extension .$ext on file name '$fileName'";
	}
	$fileName = substr($fileName, 0, $dotIndex);
	if ($newExt) { $fileName .= '.'.$newExt; }
	return $fileName;
}

# Generate the job makefile.
#
# Parameters:
# - The unit set.
# - The output makefile name.
sub genJobMakefile ($$)
{
  my $unitSet = shift;
  my $outputFileName = shift;
  my @list = $unitSet->getFunctionElements;
  my $element;
  my @entryList = ( );
	my %jobMap = ( );
	my $error = 0;
  local *OUT;

	# Generate a list of all job entry points.  A job entry point is a function
	# or method with an 'entry' attribute and a 'job' sub-attribute.
  foreach $element (@list)
  {
    my $attrs = $element->attrs;
    next unless (defined $attrs->{entry});
    my $jobName = getSubAttr $attrs->{entry}, 'job';
    next unless defined $jobName;
    my $elementName = $element->name;
    $elementName =~ s/\(.*//;
    if ($optionVerbose)
    {
      print "postscan: found entry point '$elementName', job '$jobName'\n";
    }
		if (defined $jobMap{$jobName})
		{
			print STDERR
				"postscan: ERROR: multiple definitions for job '$jobName'!\n";
			$error = 1;
			next;
		}
		$jobMap{$jobName} = $element;
  }

	# Locate the directory containing the job dependency files.
	my $sepIndex = rindex $elementFileName, '/';
	if ($sepIndex == -1) { $sepIndex = rindex $elementFileName, '\\'; }
	if ($sepIndex == 0) { die "bad element file name '$elementFileName'"; }
	my $dirName = '.';
	if ($sepIndex > 0) { $dirName = substr $elementFileName, 0, $sepIndex; }

	# Generate the makefile.
  open(OUT, '>', $outputFileName)
    or die "can not open output file '$outputFileName': $!";
	my $localtime = localtime;
	my $elementFileNameSubst = substPrefix $elementFileName;
	print OUT <<EOF;
# SPU job makefile, generated automatically by postscan.pl
# Time: $localtime
# Element file: $elementFileNameSubst

FLAGS_SUFFIX := _SPU
include \$(MAKE_ROOT)/Lib/setcflags.mk
FLAGS_SUFFIX := _SPUJOB
include \$(MAKE_ROOT)/Lib/setcflags.mk
FLAGS_SUFFIX :=

.SUFFIXES: .c .cpp .o .d .ccg .s .S .page .job .elf .bin

EOF

	foreach my $jobName (keys %jobMap)
	{
		my $entryElement = $jobMap{$jobName};

		# NOTE - CODE PAGES
		#
		# The script currently supports only one page per job, which is named
		# "${jobName}_0".  In general, the page number is appended to the job name
		# separated with a single underline character.  Page numbering starts at
		# 0.  The entry point of the job is _always_ located in page number 0.
		#
		# Without the _${pageNum} suffix, the name dentes the job execute file
		# (source or SPU compiled .o file).
		#
		# For now, we'll use ${jobName}_${pageNum} with ${pageNum} == 0 whenever a
		# page name is required!
		my $pageNum = 0;

		# Get the entry point element name.
    my $entryElementName = $entryElement->name;
    $entryElementName =~ s/\(.*//;

		# The rule for generating the SPU job execute file. 
		my $jobDir = substPrefix "$dirName/$spuJobDir";
		my $jobExecCPPFile = "$jobDir/$jobName.cpp";
		my $jobExecDepFile = substPrefix("$dirName/$spuJobDir/$jobName.d");
		print OUT<<EOF;
-include $jobExecDepFile
$jobExecCPPFile:
\t\$(BUILD_SILENT) mkdir -p '$jobDir'
\t\$(BUILD_SILENT) rm -f '$jobExecDepFile'
\t\$(BUILD_SILENT) rm -f '\$\@'
\t\$(call SPUJOB_GENPAGE,\$\@,$entryElementName,$jobName)

$genCleanVar += $jobExecCPPFile

EOF

		# The rule for generating the SPU job page implementation file.
		my $jobPageCPPFile = "$jobDir/${jobName}_${pageNum}.cpp";
		my $jobPageDepFile = "$jobDir/${jobName}_${pageNum}.d";
		print OUT <<EOF;
-include $jobPageDepFile
$jobPageCPPFile: $jobExecCPPFile
\t\$(BUILD_SILENT) rm -f '$jobPageDepFile'
\t\$(BUILD_SILENT) rm -f '\$\@'
\t\$(call SPUJOB_GENEXEC,\$\@,$entryElementName,$jobName)

$genCleanVar += $jobPageCPPFile

EOF

		# The rule for compiling the SPU job execute file to an ELF object file.
		# Note that we don't support XFLAGS_SPUJOB for the execute file.
		my $jobExecOFile = substExt $jobExecCPPFile, 'cpp', 'o';
		print OUT <<EOF;
$jobExecOFile: $jobExecCPPFile
\t\$(BUILD_SILENT) rm -f '\$\@'
\t\$(BUILD) \\
\t  \$(call COMPILE_CXX_SPUJOB,$jobName,\$<,\$\@,\$(XFLAGS_SPUJOB))

$compileCleanVar += $jobExecOFile

EOF

		# The rule for compiling the SPU job page to an assembly file.  The SPU
		# job C++ files are completely self contained, so there's no point in
		# including a special dependency file here.  The %.s only depends on the
		# %.cpp.
		my $jobPageAsmFile = substExt $jobPageCPPFile, 'cpp', 's';
		my $jobXFlagsVar = "PROJECT_XFLAGS_SPUJOB_$jobName";
		print OUT <<EOF;
$jobPageAsmFile: xflags_spu := \$($jobXFlagsVar)
$jobPageAsmFile: $jobPageCPPFile
\t\$(BUILD_SILENT) rm -f '\$\@'
\t\$(SILENT) if [ -n \"\$(xflags_spu)\" ]; then \\
\t  echo \"SPU job $jobName (page $pageNum): extra flags \$(xflags_spu)\"; \\
\t  \$(_BUILD) \\
\t    \$(call COMPILE_CXX_SPUJOB,$jobName page $pageNum,\$<,\$\@,\\
\t      -S \$(xflags_spu)); \\
\telse \\
\t  \$(_BUILD) \\
\t    \$(call COMPILE_CXX_SPUJOB,$jobName page $pageNum,\$<,\$\@,\\
\t      -S \$(XFLAGS_SPU)); \\
\tfi


$compileCleanVar += $jobPageAsmFile ${jobPageAsmFile}_tmp

EOF

		# The rule for assembling the SPU job.
		my $jobPageOFile = substExt $jobPageCPPFile, 'cpp', 'o';
		print OUT <<EOF;
$jobPageOFile: $jobPageAsmFile
\t\$(BUILD) \$(call COMPILE_C_SPU,$jobName page $pageNum,\$<,\$\@,)

$compileCleanVar += $jobPageOFile

EOF

    # The rule for running the page generator.  The page generator creates a
		# %.page file (which is a text file containing the page description).
		my $jobPageDescFile = substExt $jobPageCPPFile, 'cpp', 'page';
		my $jobPagePatchedOFile = $jobPageOFile;
		$jobPagePatchedOFile =~ s/\.o$/_patched.o/;
		print OUT <<EOF;
$jobPageDescFile: $jobPageOFile
\t\$(BUILD_SILENT) rm -f '$jobPageDescFile'
\t\$(BUILD_SILENT) rm -f '$jobPagePatchedOFile'
\t\$(BUILD_SILENT) rm -f '\$<.demangled.txt'
\t\$(BUILD_SILENT) rm -f '\$<.mangled.txt'
\t\$(BUILD_SILENT) rm -f '\$(patsubst \%.o,\%_crossbubblecalls.txt,\$<)'
\t\$(BUILD_SILENT) rm -f '\$(patsubst \%.o,\%_entrypoints.txt,\$<)'
\t\$(BUILD_LINK) \$(NM_SPU) --demangle '\$<' \\
\t  |sed -ne 's/^[0-9a-f]*\\s\\+[UT]\\s\\+//p' >'\$<.demangled.txt'
\t\$(BUILD_LINK) \$(NM_SPU) '\$<' \\
\t  |sed -ne 's/^[0-9a-f]*\\s\\+[UT]\\s\\+//p' >'\$<.mangled.txt'
\t\$(BUILD_SILENT) echo -n \\
\t  >'\$(patsubst \%.o,\%_crossbubblecalls.txt,\$<)'
\t\$(BUILD_SILENT) sed -ne 's,^//\\s*\@ENTRY=:*\\([^\@]\\+\\)\@.*,\\1 1,p' \\
\t  <'$jobExecCPPFile' \\
\t  >'\$(patsubst \%.o,\%_entrypoints.txt,\$<)'
\t\$(call SPUJOB_GENPAGEDESC,\$<,$jobPageDescFile)

$jobPagePatchedOFile: $jobPageDescFile
\t\$(BUILD_SILENT) test -f '\$\@' && touch '\$\@'

$linkCleanVar += \\
\t$jobPageDescFile \\
\t$jobPagePatchedOFile \\
\t${jobPageOFile}.demangled.txt \\
\t${jobPageOFile}.mangled.txt \\
\t\$(patsubst \%.o,\%_crossbubblecalls.txt,$jobPageOFile) \\
\t\$(patsubst \%.o,\%_entrypoints.txt,$jobPageOFile) \\
\t\$(patsubst \%.o,\%_patch.txt,$jobPageOFile) \\
\t\$(patsubst \%.o,\%.log,$jobPageOFile)

EOF

		# The rule for building the ELF file from the page .o file.
		my $jobPageELFFile = substExt $jobPageCPPFile, 'cpp', 'elf';
		print OUT <<EOF;
$jobPageELFFile: $jobPagePatchedOFile $jobPageDescFile \$(LINKPAGE_DEPS)
\t\$(BUILD_SILENT) rm -f '\$\@'
\t\$(call SPUJOB_LINKPAGE,\$<,\$\@)

$linkCleanVar += $jobPageELFFile

EOF

		# The rule for building the page BIN file.  The .bin files are required
		# for the SN debugger.  The rules are defined unconditionally, but they
		# are triggered only when SN debugging is enabled.
		my $jobPageBINFile = substExt $jobPageCPPFile, 'cpp', 'bin';
		print OUT <<EOF;
$jobPageBINFile: $jobPageELFFile
\t\$(BUILD_SILENT) rm -f '\$\@'
\t\$(call SPUJOB_SN_MODPAGE,\$<,\$\@)
\t\$(BUILD_SILENT) cp $jobPageELFFile \$(output_dir)

$linkCleanVar += $jobPageBINFile

EOF

		# The rule for generating the job description file.  For now, all prefetch
		# information is -1 (no prefetch), so we only need the job name and the
		# entry point.  We _could_ generate the job description files directly -
		# but the final workflow will determine the prefetch pages which can not
		# be done here.
		#
		# NOTE: For now we assume that the entry function receives a single main
		# memory pointer as an argument!
		my $jobExecDescFile = substExt $jobExecCPPFile, 'cpp', 'job';
		{
			my $pageNum = 0;
			my $entryElement = $jobMap{$jobName};
			my $entryElementName = $entryElement->name;
			$entryElementName =~ s/\(.*//;
			print OUT <<EOF;
$jobExecDescFile: $jobExecOFile
\t\$(BUILD_SILENT) rm -f '\$\@'
\t\$(BUILD_SILENT) echo '${jobName}_${pageNum}' >'\$\@'
\t\$(BUILD_SILENT) echo -1 >>'\$\@'
\t\$(BUILD_SILENT) echo -1 >>'\$\@'
\t\$(BUILD_SILENT) echo -1 >>'\$\@'
\t\$(BUILD_SILENT) sed -ne 's,^//\\s*\@ENTRY=:*\\([^\@]\\+\\)\@.*,\\1,p' \\
\t  <'$jobExecCPPFile' \\
\t  >>'\$\@'

$linkCleanVar += $jobExecDescFile

EOF
		}

		# The rule for building the execute BIN file (required for the SN
		# debugger).
		my $jobExecBINFile = substExt $jobExecCPPFile, 'cpp', 'bin';
		my $jobExecEntryCPPFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}_entry.cpp");
		my $jobExecEntryOFile = substExt $jobExecEntryCPPFile, 'cpp', 'o';
		my $jobExecELFFile = substExt $jobExecCPPFile, 'cpp', 'elf';
		print OUT <<EOF;
$jobExecBINFile: $jobExecOFile
\t\$(BUILD_SILENT) rm -f '\$\@'
\t\$(BUILD_SILENT) rm -f '$jobExecEntryCPPFile'
\t\$(BUILD_SILENT) echo 'int ' >'$jobExecEntryCPPFile'
\t\$(BUILD_LINK) \$(NM_SPU) '\$<' \\
\t  |sed -ne '/^[0-9a-f]*\\s\\+U/{s/\\s\\+U\\s\\+//p;q}' \\
\t  >>'$jobExecEntryCPPFile'
\t\$(BUILD_SILENT) echo '= 0;' >>'$jobExecEntryCPPFile'
\t\$(BUILD) \$(call COMPILE_CXX_SPU,$jobName entry,$jobExecEntryCPPFile,$jobExecEntryOFile,)
\t\$(BUILD_SILENT) rm -f '$jobExecEntryCPPFile'
\t\$(call SPUJOB_LINKEXEC,\$< $jobExecEntryOFile,$jobExecELFFile)
\t\$(BUILD_SILENT) rm -f '$jobExecEntryOFile'
\t\$(call SPUJOB_SN_MODEXEC,$jobExecELFFile,\$\@)
\t\$(BUILD_SILENT) cp $jobExecELFFile \$(output_dir)

$linkCleanVar += \\
\t$jobExecEntryCPPFile \\
\t$jobExecEntryOFile \\
\t$jobExecELFFile \\
\t$jobExecBINFile

EOF
	} # foreach $jobName

	print OUT "SPUJOB_ENABLE := 1\n\n";

	# Create the source and object file lists.
	print OUT "SPUJOB_SOURCES :=";
	foreach my $jobName (keys %jobMap)
	{
		my $pageNum = 0;
		my $jobExecCPPFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}.cpp");
		my $jobPageCPPFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}_${pageNum}.cpp");
		print OUT " \\\n\t$jobExecCPPFile \\\n\t$jobPageCPPFile";
	}
	print OUT "\n\nSPUJOB_OBJECTS :=";
	foreach my $jobName (keys %jobMap)
	{
		my $pageNum = 0;
		my $jobExecOFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}.o");
		my $jobPageOFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}_${pageNum}.o");
		print OUT " \\\n\t$jobExecOFile \\\n\t$jobPageOFile";
	}
	print OUT "\n\nSPUJOB_PAGEDESC_FILES :=";
	foreach my $jobName (keys %jobMap)
	{
		my $pageNum = 0;
		my $jobPageDescFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}_${pageNum}.page");
		print OUT " \\\n\t$jobPageDescFile";
	}
	print OUT "\n\nSPUJOB_JOBDESC_FILES :=";
	foreach my $jobName (keys %jobMap)
	{
		my $jobExecDescFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}.job");
		print OUT " \\\n\t$jobExecDescFile";
	}
	print OUT "\n\nSPUJOB_ELF_FILES :=";
	foreach my $jobName (keys %jobMap)
	{
		my $pageNum = 0;
		my $jobPageELFFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}_${pageNum}.elf");
		print OUT " \\\n\t$jobPageELFFile";
	}
	print OUT "\n\nSPUJOB_BIN_FILES :=";
	foreach my $jobName (keys %jobMap)
	{
		my $pageNum = 0;
		my $jobExecBINFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}.bin");
		my $jobPageBINFile = substPrefix(
				"$dirName/$spuJobDir/${jobName}_${pageNum}.bin");
		print OUT " \\\n\t$jobExecBINFile \\\n\t$jobPageBINFile";
	}

	print OUT "\n\n# vim", ":ts=8", ":sw=8\n\n";
  close OUT;
}

my $unitSet = UnitSet->new();
$unitSet->load($elementFileName);
genJobMakefile($unitSet, $makeFileName);

# Tools/postscan.pl
# vim:ts=2:sw=2

