//////////////////////////////////////////////////////////////////////////////
//  FILE       :  Log.cpp
//  AUTHORS    :  Sergei V. Migdalskiy
//  REQUIRES   :
//  CONTAINS   :  Implementation of Logging Functions
//	DESCRIPTION:
//	    Implements initialization of log file and logging of critical and
//      informational messages and Structured Exception dumps
//
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#pragma warning (disable: 4786)
#include "Synchronize.h"
#include "Log.h"

extern "C"
{
// Global denoting the instance handle of the module for which the logging is performed
// This is required for Dumps in the case of GPF
// See Also: function "LogException"
HINSTANCE g_hLogInst = 0;
char g_szModulePath[MAX_PATH] = "";
const char *g_szAppDbgState;
char g_szAppDbgStateV[0x800] = "";

}

// This mutex locks the log, so that two processes that mapped this dll cannot write simultaneously.
static util::CMutex g_LogMutex("LOGBv", util::CMutex::nFlagAllowAccessAll);
//#define AUTO_LOCK_MTX(x)

// uncomment the following line to put the log near the dll module
#define _LOG_NEAR_DLL_

// (normally absolute) path to the log file
static char g_szLogPath[MAX_PATH] = "";

// The last time when the log message was written
// used to determine the necessity to write date and month into the log.
static SYSTEMTIME g_Time;

// Informational message level. 0 means no informational messages are logged.
static int g_nInfoLevel = 4;

// Log intend: number of extra spaces placed before the log message
static int g_nLogIntend = 0;

// the log intend string: the extra spaces before the log message
// the length of this string buffer determines the max log intend
static char g_szLogIntend[0x100] = "";

// global list of lost messages
typedef std::vector<std::string> StringArray;
static StringArray g_vLostMessages;


//////////////////////////////////////////////////////////////////////////////
// InitLog
//
// TAKES   :
//     szModulePath  - path to the module for which logging is performed.
//                     module file name extension is replaced with ".log"
//                     and the resuling pathname is used as the log file name
// ASSUMES :
//     
// NOTES   :
//     call this function passing the dll or exe filepath to tell the library
//     where to place the default log file. Otherwise, it will not be written
//////////////////////////////////////////////////////////////////////////////

extern "C" void InitLog(HINSTANCE hInst)
{
	g_hLogInst = hInst;

	GetModuleFileName(hInst, g_szModulePath, sizeof(g_szModulePath));

	#ifdef _LOG_NEAR_DLL_
	InitLogByModuleName (g_szModulePath);
	#else
	// no normal path. use default log file path
	strcpy(g_szLogPath, "C:\\log.log");
	#endif
}

void SetLogPath (const char* szLogPath)
{
	strcpy(g_szLogPath, szLogPath);
}


void InitLogByModuleName (const char* szExePath)
{
	AUTO_LOCK_MTX(g_LogMutex);

	// g_szLogPath will hold the new log file path
	
	strcpy (g_szLogPath, szExePath);
	
	// find the last slash or dot in the path, and append log to it
	
	char* p;
	for (p = g_szLogPath + strlen(g_szLogPath) - 1; p > g_szLogPath && *p != '.' && *p != '\\'; p--);


	if (p > g_szLogPath)
	{
		if (*p == '.')
			// replace extension with .log
			strcpy (p, ".log");
		else
			// no extension. add extension .log
			strcat (g_szLogPath, ".log");
	}
	// the time is set to 0 to force the next Log call write the date
	memset (&g_Time, 0, sizeof(g_Time));
}



// the mapping of thread id to name
typedef std::map<DWORD, std::string> ThreadNameMap;
// this static is protected by the same lock as the whole log
static ThreadNameMap g_mapThreadNames;

void LogSetThreadName (std::string sName)
{
	AUTO_LOCK_MTX(g_LogMutex);
	g_mapThreadNames[GetCurrentThreadId()] = sName;
}

//////////////////////////////////////////////////////////////////////////////
// VLog
//
// TAKES   :
//     szFormat		- format of the message, in the form printf takes
//     parameters	- va_list, in the form vprintf takes
// ASSUMES :
//     Log should have been initialized with InitLog
// NOTES   :
//     Logs the message, taking it in the same form as vprintf
//////////////////////////////////////////////////////////////////////////////

extern "C" void VLog(const char* szFormat, va_list parameters)
{
	static LONG g_nMessage = 0;
	AUTO_LOCK_MTX(g_LogMutex);
	LONG nMessage = InterlockedIncrement(&g_nMessage);

	static char szBuffer[0x1000];

	// if we could open the file, log the process:thread
	DWORD dwThreadId = GetCurrentThreadId();
	DWORD dwProcessId = GetCurrentProcessId();
	const char* szThreadName = "";
	{
		ThreadNameMap::iterator it = g_mapThreadNames.find (dwThreadId);
		if (it != g_mapThreadNames.end())
			szThreadName = it->second.c_str();
	}

	// log date if the last message was not in this date
	SYSTEMTIME Time;
	GetLocalTime(&Time);
	if (Time.wYear == g_Time.wYear &&
		Time.wMonth == g_Time.wMonth &&
		Time.wDay == g_Time.wDay)
		_snprintf (szBuffer, sizeof(szBuffer)-8, "%s%d\t[%02d:%02d.%02d.%03d]\t%d:%d%s\t", g_szLogIntend, nMessage, Time.wHour, Time.wMinute, Time.wSecond, Time.wMilliseconds, dwProcessId, dwThreadId, szThreadName);
	else
		_snprintf (szBuffer, sizeof(szBuffer)-8, "%s%d\t[%02d/%02d/%02d|%02d:%02d.%02d.%03d]\t%d:%d%s\t", g_szLogIntend, nMessage, Time.wMonth, Time.wDay, Time.wYear, Time.wHour, Time.wMinute, Time.wSecond, Time.wMilliseconds, dwProcessId, dwThreadId, szThreadName);
	
	// memorize the last time logged
	g_Time = Time;
	
	unsigned nLen = strlen(szBuffer);
	char* pBuffer = szBuffer + nLen;
	
	_vsnprintf (pBuffer, sizeof(szBuffer)-nLen-8, szFormat, parameters);

	bool bWritten = false;
	// log the message
	if (g_szLogPath[0])
	{
		// try to open the file in shared mode
		FILE* f = fopen (g_szLogPath, "at");
		
		if (f)
		{
			for (StringArray::iterator it = g_vLostMessages.begin(); it != g_vLostMessages.end(); )
			{
				if (fprintf (f, "r%s\n", it->c_str()) > 0)
					it = g_vLostMessages.erase(it);
				else
					it++;
			}
			if (fprintf (f, "%s\n", szBuffer) > 0)
				bWritten = true;
			// close the log file each time
			fclose(f);
		}
	}
	
	if (!bWritten)
		g_vLostMessages.push_back(szBuffer);
}

extern "C" void VLogSimple(const char* szFormat, va_list parameters)
{
	FILE* f = fopen (g_szLogPath, "at");
	
	if (f)
	{	
		// if we could open the file, log the process:thread
		DWORD dwThreadId = GetCurrentThreadId();
		DWORD dwProcessId = GetCurrentProcessId();
		
		// log date if the last message was not in this date
		SYSTEMTIME Time;
		GetLocalTime(&Time);
		if (Time.wYear == g_Time.wYear &&
			Time.wMonth == g_Time.wMonth &&
			Time.wDay == g_Time.wDay)
			fprintf (f, "[%02d:%02d.%02d.%03d]\t%d:%d\t", Time.wHour, Time.wMinute, Time.wSecond, Time.wMilliseconds, dwProcessId, dwThreadId);
		else
			fprintf (f, "[%02d/%02d/%02d|%02d:%02d.%02d.%03d]\t%d:%d\t", Time.wMonth, Time.wDay, Time.wYear, Time.wHour, Time.wMinute, Time.wSecond, Time.wMilliseconds, dwProcessId, dwThreadId);
		
		// memorize the last time logged
		g_Time = Time;

		// log the message
		vfprintf (f, szFormat, parameters);
		// end of line
		fprintf (f, "\n");
		// close the log file each time
		fclose(f);
	}
}

// Utility function: passes parameters to VLog
extern "C" void Log(const char* szFormat, ...)
{
	va_list parameters;
	va_start(parameters, szFormat);
	VLog (szFormat, parameters);
	va_end (parameters);
}

void LogSimple(const char* szFormat, ...)
{
	va_list parameters;
	va_start(parameters, szFormat);
	VLogSimple (szFormat, parameters);
	va_end (parameters);
}

// Sets the informational message level
// See Also: global "g_nInfoLevel"
void SetInfoLevel(int nLevel)
{
	g_nInfoLevel = nLevel;
}

// increments or decrements the log intendation value: number of spaces placed before each log message
void LogIntend (int nDelta)
{
	g_nLogIntend += nDelta;

	if (g_nLogIntend <= 0)
		g_szLogIntend[0] = '\0';
	else
	if (g_nLogIntend >= sizeof(g_szLogIntend)-1)
	{
		memset (g_szLogIntend, ' ', sizeof(g_szLogIntend)-1);
		g_szLogIntend[sizeof(g_szLogIntend)-1] = '\0';
	}
	else
	{
		memset (g_szLogIntend, ' ', g_nLogIntend);
		g_szLogIntend[g_nLogIntend] = '\0';
	}
}

//////////////////////////////////////////////////////////////////////////////
// VInfo
//
// TAKES   :
//     szFormat    - format in the form of vprintf
//     parameters	 - parameters in the form of vprintf
// ASSUMES :
//     
// NOTES   :
//     puts informational message into the log file, if the leading space
//     count is less than the current informational level
//////////////////////////////////////////////////////////////////////////////

void VInfo(const char* szFormat, va_list parameters)
{
	int i;
	for (i = 0; szFormat[i] && szFormat[i]==' '; i++);
	
	if (i < g_nInfoLevel)
		VLog(szFormat, parameters);
}


// Logs informational message, just passes parameters to VLog
void Info(const char* szFormat, ...)
{
	va_list parameters;
	va_start(parameters, szFormat);
	VInfo (szFormat, parameters);
	va_end (parameters);
}


// converts the given exception code into verbal exception description
const char* FormatExceptionCode (DWORD dwExceptionCode)
{
	// some of the most widely-occurring exceptions are converted to their verbal equivalents,
	// and by default it's presented with 0xXXX code, XXX is the hex representation of the code

	switch (dwExceptionCode)
	{
		case EXCEPTION_ACCESS_VIOLATION:
			return "Access Violation";
		case EXCEPTION_BREAKPOINT:
			return "Breakpoint";
		case EXCEPTION_ILLEGAL_INSTRUCTION:
			return "Illegal Instruction";
		case EXCEPTION_IN_PAGE_ERROR:
			return "Page not accessible";
		case EXCEPTION_INT_DIVIDE_BY_ZERO:
			return "Integer division by zero";
		case EXCEPTION_INT_OVERFLOW:
			return "Integer overflow";
		case EXCEPTION_PRIV_INSTRUCTION:
			return "Priviledged instruction";
		case EXCEPTION_SINGLE_STEP:
			return "A trace trap or other single-instruction mechanism signaled that one instruction has been executed";
		case EXCEPTION_STACK_OVERFLOW:
			return "Stack overflow";
		default:
			{
				// first 2 chars is the 0x code, the others are the hex exception code
				static char szResult[16] = "0x";
				itoa (dwExceptionCode, szResult+2, 16);
				return szResult;
			}
	}
}


////////////////////////////////////////////////////////////////////////////////////////////////////
// LogException 
//
// TAKES:
//  szDescription - the description of the exception
//  pExceptionInfo - the exception information
// REUTRNS:
//  code enabling the __except() block handler execution (EXCEPTION_EXECUTE_HANDLER)
// NOTES:
//  this is the SEH exception filter function. It receives the exception info structure and logs it.
////////////////////////////////////////////////////////////////////////////////////////////////////
int LogException (LPEXCEPTION_POINTERS pExceptionInfo, const char* szDescription)
{
	Log ("*ERROR* Structured Exception: %s\n"
		"Exception Code: %s\n"
		"Exception Flags: %s (%d)\n"
		"Additional Exception Record: %s (0x%X)\n"
		"Address: 0x%X (base 0x%X)(pid %d)\n"
		"Parameter count: %d"
		,
		szDescription, 
		FormatExceptionCode(pExceptionInfo->ExceptionRecord->ExceptionCode),
		pExceptionInfo->ExceptionRecord->ExceptionFlags ? pExceptionInfo->ExceptionRecord->ExceptionFlags == EXCEPTION_NONCONTINUABLE ? "EXCEPTION_NONCONTINUABLE" : "UNKNOWN" : "NULL",
    pExceptionInfo->ExceptionRecord->ExceptionFlags,
		pExceptionInfo->ExceptionRecord->ExceptionRecord ? "PRESENT" : "NULL",
		pExceptionInfo->ExceptionRecord->ExceptionRecord,
		pExceptionInfo->ExceptionRecord->ExceptionAddress,
		g_hLogInst,
		GetCurrentProcessId(),
		pExceptionInfo->ExceptionRecord->NumberParameters
		);
	return EXCEPTION_EXECUTE_HANDLER;	
}


std::string IntToString (int nNumber)
// returns the decimal string representation of the given int
{
	char szNumber[16];
	//	itoa (nNumber, szNumber, 10);
	sprintf (szNumber, "%d", nNumber);
	return szNumber;
}

std::string UIntToHexString(DWORD dwNumber)
// reuturns hexadecimal string representation of the given dword
{
	char szNumber[24];
	sprintf (szNumber, "0x%X", dwNumber);
	return szNumber;
}



std::string FormatWinError(DWORD dwError)
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// returns the string representation (in natural language) of the last error retrieved by GetLastError()
// if the error couldn't be formatted, returns a string in form "Error #NNN, couldn't format"
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
{
	std::string sResult = TryFormatWinError(dwError);
	
	if (sResult.empty())
		// error. return both the user error and the error received during formatting the user error
		sResult = "Error " + IntToString (GetLastError()) + " while formatting error message";

	return sResult + " (" + (dwError & 0x80000000 ? UIntToHexString(dwError):IntToString(dwError)) + ")";
}

// tries to format the error, if the description wasn't found, returns an empty string
std::string TryFormatWinError(DWORD dwError)
{
	LPVOID lpMsgBuf;   // pointer to the buffer that will accept the formatted message
	//DWORD dwLastError = OsGetLastError(); // the last user error for which the description should be formatted
	DWORD dwFormattedMsgLen = FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS, NULL, dwError, MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR) &lpMsgBuf, 0, NULL );
	
	if (!dwFormattedMsgLen)
		// error. return both the user error and the error received during formatting the user error
		return std::string();
	else
	{// the lpMsgBuf contains allocated by the system call message that is to be returned.
		// we'll copy it into sResult and free it and return sResult
		std::string sResult = (LPCTSTR) lpMsgBuf;
		LocalFree (lpMsgBuf);
		while (!sResult.empty() && ((unsigned char)sResult[sResult.length()-1]) < 0x20)
			sResult.resize(sResult.length()-1);
		return sResult;
	}
}


///////////////////////////////////////////////////////////////////
// the same as LogException, but takes vprintf-like parameters
///////////////////////////////////////////////////////////////////
int LogExceptionV (LPEXCEPTION_POINTERS pExceptionInfo, const char* szFormat, ...)
{
	LogException(pExceptionInfo, "<Logged later>");
	va_list parameters;
	va_start(parameters, szFormat);
	VLog (szFormat, parameters);
	va_end (parameters);
	return EXCEPTION_EXECUTE_HANDLER;	
}

// set the debug state of the application to the given string.
// this state is only put to log if an error condition like GPF occurs
extern "C" void DbgState(const char* sz)
{
	g_szAppDbgState = sz;
}

// set the debug state of the application to the given string in form of printf.
// this state is only put to log if an error condition like GPF occurs
extern "C" void DbgStateEx(const char* szFormat, ...)
{
	va_list params;
	va_start(params, szFormat);
	DbgStateV(szFormat, params);
	va_end(params);
}

// set the debug state of the application to the given string in form of vprintf.
// this state is only put to log if an error condition like GPF occurs
extern "C" void DbgStateV(const char* szFormat, va_list params)
{
	vsprintf (g_szAppDbgStateV, szFormat, params);
	g_szAppDbgState = g_szAppDbgStateV;
}
