/////////////////////////////////////////////////////////////////
// Misc. common utilities, constants and definitions used by most applications
// GL stands for Graphics library, but not OpenGL
/////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "GLUtils.h"

const double gRadianToDegree = 5.7295779513082320876798154814105e+1;
const double gDegreeToRadian = 1.7453292519943295769236907684886e-2;
const double gPi = 3.1415926535897932384626433832795;
const double gEpsilon = 1e-7;


// utility - normalizes the given angle so that it fits into the normal interval
// [-pi,pi)
double GLNormalizeAngle (double fAngle)
{
	fAngle = fmod (fAngle + gPi, 2*gPi) - gPi;
	if (fAngle < -gPi)
		return fAngle + 2*gPi;
	else
		return fAngle;
}

// normalizes the angle to the[-pi,pi) interval
// ASSUMES: the angle is relatively close the target interval
double GLNormalizeAngle2(double fAngle)
{
	while (fAngle >= gPi)
		fAngle -= 2*gPi;
	while (fAngle < -gPi)
		fAngle += 2*gPi;
	return fAngle;
}

/////////////////////////////////////////////////////////////////////////////////
// blends two angles, which are in the [-pi,pi) interval, by the smaller arc
// fTime==0 corresponds to fAngle1, fTime==1 corresponds to the fAngle2
// ASSUMES: the input angles are normalized
// WARNING: the returned angle may not be normalized
/////////////////////////////////////////////////////////////////////////////////
double GLBlendAngle (double fAngle1, double fAngle2, double fTime)
{
	assert(fAngle1 >= -gPi && fAngle1 <= gPi);
	assert(fAngle2 >= -gPi && fAngle2 <= gPi);
	if (fAngle1 < fAngle2)
	{
		if (fAngle2-fAngle1 < fAngle1+2*gPi-fAngle2)
			// fAngle1->fAngle2
			return fAngle1*(1-fTime)+fAngle2*fTime;
		else
			// fAngle1+2*gPi->fAngle2
			return (fAngle1+2*gPi)*(1-fTime)+fAngle2*fTime;
	}
	else
	{
		if (fAngle1 - fAngle2 < fAngle2+2*gPi-fAngle1)
			// fAngle1->fAngle2
			return fAngle1*(1-fTime)+fAngle2*fTime;
		else
			// fAngle1->fAngle2+2*gPi
			return fAngle1*(1-fTime)+(fAngle2+2*gPi)*fTime;
	}
}

///////////////////////////////////////////////////////////////////////////////////
// compares the two angles in the interval [-pi,pi) and returns the 
// absolute distance between them, by the smaller arc (i.e. the distance
// between pi-0.01 and -pi+0.01 is 0.02
// ASSUMES: the input angles are normalized
// NOTE: the distance between the angles may not be greater than pi
///////////////////////////////////////////////////////////////////////////////////
double GLAngleDistance (double fAngle1, double fAngle2)
{
	assert(fAngle1 >= -gPi && fAngle1 <= gPi);
	assert(fAngle2 >= -gPi && fAngle2 <= gPi);

	double fDelta = fAngle1-fAngle2;
	if (fDelta > 0)
		return tmin ( fDelta, 2*gPi-fDelta);
	else
		return tmin (-fDelta, 2*gPi+fDelta);
}

///////////////////////////////////////////////////////////////////////////////////
// returns the angle equivalent to the angle 2, the closest to the fAngle1
// (this absolute value can be used to linearly blend the two angles
//  via the shortest arc)
// NOTE: this function accepts and outputs non-normalized angles
///////////////////////////////////////////////////////////////////////////////////
double GLBlendAngleTarget (double fAngle1, double fAngle2)
{
	while (tabs(fAngle2-2*gPi-fAngle1) < tabs(fAngle2-fAngle1))
		fAngle2 -= 2*gPi;

	while (tabs(fAngle2+2*gPi-fAngle1) < tabs(fAngle2-fAngle1))
		fAngle2 += 2*gPi;

	if (fAngle1 < fAngle2)
	{
		if (fAngle2-fAngle1 < fAngle1+2*gPi-fAngle2)
			// fAngle1->fAngle2
			return fAngle2;
		else
			// fAngle1+2*gPi->fAngle2,
			// or: fAngle1->fAngle2-2*gPi
			return fAngle2-2*gPi;
	}
	else
	{
		if (fAngle1 - fAngle2 < fAngle2+2*gPi-fAngle1)
			// fAngle1->fAngle2
			return fAngle2;
		else
			// fAngle1->fAngle2+2*gPi
			return fAngle2+2*gPi;
	}
}

#ifdef _MIGDALSKIY_STUFF_GLVECTOR_HDR_
///////////////////////////////////////////////////////////////////////////////////
// constructs a normal vector to the given vector
// guarantees no singularity as long as the input vector is not 0
// if it's 0, still returns default vector (not an error)
///////////////////////////////////////////////////////////////////////////////////
CGLVector ConstructNormal(const CGLVector& v)
{
	CGLVector ptResult(0,0,0);
	// find two greatest components and make the normal
	// in their plane
	if (tabs(v[0])<tabs(v[1])) // X < Y ?
	{
		if (tabs(v[2])<tabs(v[0])) // Z < X < Y ?
		{
			// use XY plane
			ptResult[1] =  v[0];
			ptResult[0] = -v[1];
		}
		else // Z > X, Y > X
		{
			// use YZ plane
			ptResult[2] =  v[1];
			ptResult[1] = -v[2];
		}
	}
	else // Y < X
	{
		if (tabs(v[2])< tabs(v[1])) // Z < Y < X
		{
			// use YX plane
			ptResult[1] =  v[0];
			ptResult[0] = -v[1];
		}
		else // Y < X, Y < Z
		{
			// use XZ plane
			ptResult[0] =  v[2];
			ptResult[2] = -v[0];
		}
	}
	if (ptResult.isSmall(1e-4f))
	{
		return CGLVector(1,0,0);
	}
	else
	{
		ptResult.Unify();
		return ptResult;
	}
}
#endif

unsigned GCD (unsigned p, unsigned q)
{
	while (q > 0)
	{
		unsigned r = p % q;
		p = q;
		q = r;
	}
	return p;
}

// treating both arrays as vectors, adds
FloatArray& operator += (FloatArray& arrDst, const FloatArray& arrSrc)
{
	assert (arrDst.size () == arrSrc.size());
	for (unsigned i = 0;i < tmin(arrDst.size(), arrSrc.size()); ++i)
		arrDst[i] += arrSrc[i];
	return arrDst;
}

// treating both arrays as vectors, subtracts
FloatArray& operator -= (FloatArray& arrDst, const FloatArray& arrSrc)
{
	assert (arrDst.size () == arrSrc.size());
	for (unsigned i = 0;i < tmin(arrDst.size(), arrSrc.size()); ++i)
		arrDst[i] -= arrSrc[i];
	return arrDst;
}

FloatArray& operator *= (FloatArray& arrDst, double k)
{
	for (FloatArray::iterator it = arrDst.begin(); it != arrDst.end(); ++it)
		*it = float(*it*k);
	return arrDst;
}

double Length (const FloatArray& arrSrc)
{
	double fSum = 0;
	for (FloatArray::const_iterator it = arrSrc.begin(); it != arrSrc.end(); ++it)
	{
		fSum += tsqr(*it);
	}
	return fSum;
}