/////////////////////////////////////////////////////////////////
// Misc. common utilities, constants and definitions used by most applications
// GL stands for Graphics library, but not OpenGL
/////////////////////////////////////////////////////////////////

#ifndef _GLUTILS_HDR_
#define _GLUTILS_HDR_

// graphics library utilities (not OpenGL utils!)

// size of array
#ifndef SIZE_OF_ARRAY
#define SIZE_OF_ARRAY(x) (sizeof(x)/sizeof(x[0]))
#endif

template <typename T> T tmax(T a, T b) {return a > b ? a : b;}
template <typename T> T tmax(T a, T b, T c) {return tmax(c,tmax(a,b));}
template <typename T> T tmax(T a, T b, T c, T d) {return tmax(d, tmax(c,tmax(a,b)));}
template <typename T> T tmin(T a, T b) {return a < b ? a : b;}
template <typename T> T tmin(T a, T b, T c) {return tmin(c,tmin(a,b));}
template <typename T> T tmin(T a, T b, T c, T d) {return tmin(d, tmin(c,tmin(a,b)));}
//template <typename T> T tmax(T a, T b, T c) {return tmax(tmax(a,b),c);}
//template <typename T> T tmin(T a, T b, T c) {return tmin(tmin(a,b),c);}
template <typename T> T tabs(T x) {return x >= 0 ? x : -x;}
template <typename T> void tswap(T&a, T&b) {T x = a; a = b; b = x;}
template <typename T> T tsqr (T x) {return x*x;}

template <typename T> void tforceOrder(T&a, T&b)
{
	if (a > b)
		tswap(a,b);
}

///////////////////////////////////////////////////////////////
// force range inclusive: forces a in [fBegin,fEnd]
///////////////////////////////////////////////////////////////
template <typename T> void tforceRangeIncl (T&a, T fBegin, T fEnd)
{
	if (a < fBegin)
		a = fBegin;
	else
	if (a > fEnd)
		a = fEnd;
}

extern const double gDegreeToRadian; // 0.017453292519943295769236907684886
extern const double gRadianToDegree; // 57.295779513082320876798154814105
extern const double gPi;
extern const double gEpsilon;

template<typename T>
T DegreeToRadian(T x) {return x*gDegreeToRadian;}
template<typename T>
T RadianToDegree(T x) {return x*gRadianToDegree;}

/////////////////////////////////////////////////////////////////////////////////////
// utility function that returns the value fValue limited by fMax maximum
// this is taken from the Irix sources that used it for angular computations
/////////////////////////////////////////////////////////////////////////////////////
template <typename T> T limit0(T fValue, T fMax)
{
   while(fValue < 0.0)
      fValue += fMax;
   while(fValue >= fMax)
      fValue -= fMax;
   return(fValue);
}

/////////////////////////////////////////////////////////////////////////////////////
// utility function that returns the value fValue limited by fMax maximum
// and fMin minimum
// this is taken from the Irix sources that used it for angular computations
/////////////////////////////////////////////////////////////////////////////////////
template <typename T> T limit(T fValue, T fMin, T fMax)
{
	T diff = fMax - fMin;
	while(fValue < fMin)
		fValue = -(fValue + diff);
	while(fValue > fMax)
		fValue = -(fValue - diff);
	return fValue;
}

/////////////////////////////////////////////////////////
// makes new fAspectNew, for smooth zooming, which is:
//  == fAspect if higher than fCenter
//  == fMin if lower than 0
//  has 0 derivative at 0 (new fMin)
//  has 1 derivative at fCenter (new fCenter)
/////////////////////////////////////////////////////////
inline double SmoothLimitZoom (double fAspect, double fCenter, double fMin)
{
	if (fAspect < 0)
		return fMin;

	if (fAspect < fCenter)
	{
		double fCenter2 = fCenter*fCenter;
		double fCenter3 = fCenter*fCenter2;

		double a = (2*fMin - fCenter)/fCenter3, b = (2*fCenter - 3*fMin)/fCenter2;
		return a * fAspect + b * fAspect + fMin;
	}

	return fAspect;
}

// utility - normalizes the given angle so that it fits into the normal interval
// [-pi,pi)
extern double GLNormalizeAngle (double fAngle);

// normalizes the angle to the [-pi, pi) interval 
// ASSUMES: the angle is already in the [-3*pi,3*pi) interval
inline double GLNormalizeAngle1(double fAngle)
{
	return fAngle >= gPi? fAngle - 2*gPi : fAngle < -gPi? fAngle + 2*gPi: fAngle;
}

// normalizes the angle to the[-pi,pi) interval
// ASSUMES: the angle is relatively close the target interval
extern double GLNormalizeAngle2(double fAngle);

// blends two angles, which are in the [-pi,pi) interval, by the smaller arc
// fTime==0 corresponds to fAngle1, fTime==1 corresponds to the fAngle2
// ASSUMES: the input angles are normalized
// WARNING: the returned angle may not be normalized
extern double GLBlendAngle (double fAngle1, double fAngle2, double fTime);

// returns the angle equivalent to the angle 2, the closest to the fAngle1
// (this absolute value can be used to linearly blend the two angles
//  via the shortest arc)
// NOTE: this function accepts non-normalized angles and normalizes them
extern double GLBlendAngleTarget (double fAngle1, double fAngle2);

// compares the two angles in the interval [-pi,pi) and returns the 
// absolute distance between them, by the smaller arc (i.e. the distance
// between pi-0.01 and -pi+0.01 is 0.02
// ASSUMES: the input angles are normalized
// NOTE: the distance between the angles may not be greater than pi
extern double GLAngleDistance (double fAngle1, double fAngle2);

#ifdef _MIGDALSKIY_STUFF_GLVECTOR_HDR_
// constructs a normal vector to the given vector
// guarantees no singularity as long as the input vector is not 0
// if it's 0, still returns default vector (not an error)
extern CGLVector ConstructNormal (const CGLVector& v);
#endif

// finds the Greatest Common Denominator of two integers
extern unsigned GCD (unsigned a, unsigned b);

inline unsigned GCD (int a, int b)
{
	return GCD((unsigned)tabs (a), (unsigned)tabs(b));
}

typedef std::vector<float> FloatArray;
typedef std::vector<double> DoubleArray;
typedef std::set<float> FloatSet;

// treating both arrays as vectors, performs corresponding operation of linear algebra
FloatArray& operator += (FloatArray& arrDst, const FloatArray& arrSrc);
FloatArray& operator -= (FloatArray& arrDst, const FloatArray& arrSrc);
FloatArray& operator *= (FloatArray& arrDst, double k);
double Length (const FloatArray& arrSrc);

#endif