/*
ScPl - A plotting library for .NET

ArrayAdapter.cs
Copyright (C) 2004
Paolo Pierini, Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

using System;

namespace scpl
{
	/// <summary>
	/// A class derived from ArrayAdapter, to provide text label capabilities.
	/// </summary>
	public class ArrayAdapterText : ArrayAdapter 
	{
		private string [] strings_;

		#region PointText this
		/// <summary>
		/// Indexer for returning the datapoint.
		/// </summary>
		public override PointD this[int i]
		{
			get
			{
				if ( xValues_ == null )
				{
					double xPos = this.Start + i*this.Step;
					return new PointText ( (double)xPos, (double)yValues_[i], strings_[i] );
				}
				else
				{
					return new PointText( (double)xValues_[i], (double)yValues_[i], strings_[i] );
				}			
			}
		}
		#endregion

		#region Constructors
		private void Init_Helper(int datalength, string[] s)
		{
			if ( datalength != s.Length )
			{
				throw new System.Exception( "xs/ys and strings not same length." );
			}
			strings_ = s;
		}

		/// <summary>
		/// The parameterless constructor.
		/// </summary>
		private ArrayAdapterText() : base()
		{
		}
		/// <summary>
		/// ArrayAdapterText constructor, taking two float arrays as arguments.
		/// <param name="xs">The float array of abscissa values.</param>
		/// <param name="ys">The float array of ordinate values.</param>
		/// <param name="strings">The array of strings.</param>
		/// </summary>
		public ArrayAdapterText( float[] xs, float[] ys, string[] strings ) : base(xs,ys)
		{
			this.Init_Helper(xs.Length, strings);
		}
		/// <summary>
		/// ArrayAdapterText constructor, taking float arrays as arguments.
		/// Causes the data to be replicated if DoClone is true. 
		/// Otherwise the original data is used. if DoClone is false and the data changes
		/// or goes out of scope before the plot is drawn, results are unpredictable.
		/// <param name="xs">The float array of abscissa values.</param>
		/// <param name="ys">The float array of ordinate values.</param>
		/// <param name="strings">The array of strings.</param>
		/// <param name="DoClone">The boolean flag used to replicate the data.</param>
		/// </summary>
		public ArrayAdapterText( double[] xs, double[] ys, string[] strings, bool DoClone ) : this(xs, ys, strings, DoClone, DoClone)
		{
			this.Init_Helper(xs.Length, strings);
		}

		/// <summary>
		/// ArrayAdapterText constructor, taking float arrays as arguments.
		/// Causes the data to be replicated if DoClone is true. 
		/// Otherwise the original data is used. if DoClone is false and the data changes
		/// or goes out of scope before the plot is drawn, results are unpredictable.
		/// <param name="xs">The float array of abscissa values.</param>
		/// <param name="ys">The float array of ordinate values.</param>
		/// <param name="strings">The array of strings.</param>
		/// <param name="DoCloneX">The boolean flag used to replicate the X data.</param>
		/// <param name="DoCloneY">The boolean flag used to replicate the Y data.</param>
		/// </summary>
		public ArrayAdapterText( double[] xs, double[] ys, string[] strings, bool DoCloneX, bool DoCloneY ) : base(xs, ys, DoCloneX, DoCloneY)
		{
			this.Init_Helper(xs.Length, strings);
		}

		/// <summary>
		/// ArrayAdapterText constructor, taking double arrays as arguments.
		/// <param name="xs">The double array of abscissa values.</param>
		/// <param name="ys">The double array of ordinate values.</param>
		/// <param name="strings">The array of strings.</param>
		/// </summary>
		public ArrayAdapterText( double[] xs, double[] ys, string[] strings ) : base(xs, ys)
		{
			this.Init_Helper(xs.Length, strings);
		}

		/// <summary>
		/// ArrayAdapter constructor, taking float arrays as argument.
		/// <param name="ys">The float array of ordinate values.</param>
		/// <param name="strings">The array of strings.</param>
		/// </summary>
		public ArrayAdapterText( float[] ys, string[] strings ) : base(ys)
		{
			this.Init_Helper(ys.Length, strings);
		}

		/// <summary>
		/// ArrayAdapter constructor, taking double arrays as argument.
		/// <param name="ys">The double array of ordinate values.</param>
		/// <param name="strings">The array of strings.</param>
		/// </summary>
		public ArrayAdapterText( double[] ys, string[] strings ) : base(ys)
		{
			this.Init_Helper(ys.Length, strings);
		}

		/// <summary>
		/// ArrayAdapterText constructor, taking one float array, and 
		/// the start and step values for the x array.
		/// </summary>
		/// <param name="ys">The float array of ordinate values.</param>
		/// <param name="strings">The array of strings.</param>
		/// <param name="start">The starting point of the abscissa values.</param>
		/// <param name="step">The step of the abscissa values.</param>
		public ArrayAdapterText( float[] ys, string[] strings, float start, float step ): base( ys, start, step)
		{
			this.Init_Helper(ys.Length, strings);
		}

		/// <summary>
		/// ArrayAdapter constructor, taking one double array, and 
		/// the start and step values for the x array.
		/// </summary>
		/// <param name="ys">The double array of ordinate values.</param>
		/// <param name="strings">The array of strings.</param>
		/// <param name="start">The starting point of the abscissa values.</param>
		/// <param name="step">The step of the abscissa values.</param>
		public ArrayAdapterText( double[] ys, string[] strings, double start, double step ): base(ys, start, step)
		{
			this.Init_Helper(ys.Length, strings);
		}
		#endregion

		#region get/set TextValues
		/// <summary>
		/// Accessor for the Text values.
		/// </summary>
		public string[] TextValues
		{
			get
			{
				return strings_;
			}
			set
			{
				strings_ = value;
			}
		}
		#endregion
	}
}
