/*
ScPl - A plotting library for .NET

Axis.cs
Copyright (C) 2003
Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: Axis.cs,v 1.33 2004/05/17 11:57:43 mhowlett Exp $

*/

using System.Drawing.Drawing2D;
using System.Drawing;
using System;
using System.Collections;

namespace scpl
{
	/// <summary>
	///  The generic Axis class. 
	///  Contains all general axis data except the physical extents of the axis.
	/// </summary>
	public class Axis : System.ICloneable
	{
		// NOTE: remember to add to clone method if add variable here..

		protected object worldMax_;
		protected object worldMin_;
		private object ticksAngle_;
		private float largeTickSize_;
		private float smallTickSize_;
		private Font font_;
		private float fontScale_;
		private float tickScale_;
		private bool tickTextNextToAxis_;
		private bool hideTickText_;
		private bool hidden_;
		private System.Collections.ArrayList drawPositions_;
		private string label_;
		private Font labelFont_;
		private Brush labelBrush_;
		private Brush tickTextBrush_;
		private bool reversed_;

		private string numFormat_;
		/*
		private double[] gridMantissas_ = {1.0, 2.0, 2.5, 5.0};
		private double[] mantissaLogs_;
		*/

		#region Clone
		/// <summary>
		/// Deep copy of Axis. Includes check to see that this isn't being called by
		/// a derived type. The derived type should provide it's own Clone method.
		/// </summary>
		/// <returns>A copy of the Axis Class</returns>
		public virtual object Clone()
		{
			Axis a = new Axis();
			// ensure that this isn't being called on a derived type. If it is, then oh no!
			if (this.GetType() != a.GetType())
			{
				throw new System.Exception( "Clone not defined in derived type. Help!" );
			}
			DoClone( this, a );
			return a;
		}
		#endregion
		#region DoClone
		/// <summary>
		/// Helper method for Clone. Does all the copying - can be called by derived
		/// types so they don't need to implement this part of the copying themselves.
		/// also useful in constructor of derived types that takes Axis class.
		/// </summary>
		protected static void DoClone( Axis b, Axis a )
		{

			// copy these boxed values by value if they are set, else set to null.

			if (b.ticksAngle_ != null)
			{
				a.ticksAngle_ = (double)b.ticksAngle_;
			}
			else
			{
				a.ticksAngle_ = null;
			}

			if (b.worldMax_ != null)
			{
				a.worldMax_ = (double)b.worldMax_;
			}
			else
			{
				a.worldMax_ = null;
			}

			if (b.worldMin_ != null)
			{
				a.worldMin_ = (double)b.worldMin_;
			}
			else
			{
				a.worldMin_ = null;
			}

			// copy unboxed values.
			a.largeTickSize_ = b.largeTickSize_;
			a.smallTickSize_ = b.smallTickSize_;
			a.fontScale_ = b.fontScale_;
			a.tickScale_ = b.tickScale_;
			a.tickTextNextToAxis_ = b.tickTextNextToAxis_;
			a.hidden_ = b.hidden_;
			a.hideTickText_ = b.hideTickText_;
			a.reversed_ = b.reversed_;
			a.numFormat_ = b.numFormat_;

			// copy IClonable objects.
			a.font_ = (Font)b.font_.Clone();
			a.drawPositions_ = (System.Collections.ArrayList)b.drawPositions_.Clone();
			a.label_ = (string)b.label_.Clone();
			a.labelFont_ = (Font)b.labelFont_.Clone();
			a.tickTextBrush_ = (Brush)b.tickTextBrush_.Clone();
			a.labelBrush_ = (Brush)b.labelBrush_.Clone();
			a.linePen_ = (Pen)b.linePen_.Clone();
		}
		#endregion

		#region Init
		/// <summary>
		/// Helper function for constructors.
		/// Do initialization here so that Clear() method is handled properly
		/// </summary>
		private void Init()
		{
			this.worldMax_ = null;
			this.worldMin_ = null;
			this.Hidden = false;
			this.SmallTickSize = 2.0f;
			this.LargeTickSize = 5.0f;
			this.fontScale_ = 1.0f;
			this.tickScale_ = 1.0f;
			this.tickTextNextToAxis_ = true;
			this.hideTickText_ = false;
			this.drawPositions_ = new System.Collections.ArrayList();

			this.label_ = "" ;
			this.numFormat_ = "{0:g5}";
			this.reversed_ = false;

			//UpdateMantissaLogs();

			FontFamily fontFamily = new FontFamily("Arial");
			this.TickTextFont = new Font(fontFamily, 10, FontStyle.Regular, GraphicsUnit.Pixel);
			this.LabelFont = new Font(fontFamily, 12, FontStyle.Regular, GraphicsUnit.Pixel);
			this.LabelBrush = new SolidBrush( Color.Black );
			this.tickTextBrush_ = new SolidBrush( Color.Black );
			this.linePen_ = new Pen( Color.Black );
			this.linePen_.Width = 1.0f;
		}
		#endregion
		#region Constructor
		/// <summary>
		/// Default constructor
		/// </summary>
		public Axis( )
		{
			this.Init();
		}
		#endregion
		#region Constructor
		/// <summary>
		/// Constructor that takes only world min and max values.
		/// </summary>
		/// <param name="worldMin">The minimum world coordinate.</param>
		/// <param name="worldMax">The maximum world coordinate.</param>
		public Axis( double worldMin, double worldMax )
		{
			this.Init();
			this.WorldMin = worldMin;
			this.WorldMax = worldMax;
		}
		#endregion
		#region Constructor
		/// <summary>
		/// Copy constructor.
		/// </summary>
		/// <param name="a">The Axis to clone.</param>
		public Axis( Axis a )
		{
			Axis.DoClone( a, this );
		}
		#endregion

		#region OutOfRange
		/// <summary>
		/// Determines whether a world value outside range WorldMin - WorldMax
		/// </summary>
		/// <param name="coord">the world value to test</param>
		/// <returns>true if outside limits, false otherwise</returns>
		public bool OutOfRange( double coord )
		{
			if (worldMin_ == null || worldMax_ == null)
			{
				throw new System.Exception( "world min / max not set" );
			}

			if (coord > this.WorldMax || coord < this.WorldMin)
			{
				return true;
			}
			else
			{
				return false;
			}
		}
		#endregion
		#region LUB
		/// <summary>
		/// Sets the world extent of the current axis to be just large enough
		/// to encompas the current world extent of the axis, and the world
		/// extent of the passed in axis
		/// </summary>
		/// <param name="a">The other Axis instance.</param>
		public void LUB( Axis a )
		{
			// mins
			if (a.worldMin_ != null)
			{
				if (worldMin_ == null)
				{
					WorldMin = a.WorldMin;
				}
				else
				{
					if (a.WorldMin < WorldMin)
					{
						WorldMin = a.WorldMin;
					}
				}
			}

			// maxs.
			if (a.worldMax_ != null)
			{
				if (worldMax_ == null)
				{
					WorldMax = a.WorldMax;
				}
				else
				{
					if (a.WorldMax > WorldMax)
					{
						WorldMax = a.WorldMax;
					}
				}
			}
		}
		#endregion

		#region WorldToPhysical
		/// <summary>
		/// World to physical coordinate transform.
		/// </summary>
		/// <param name="coord">The coordinate value to transform.</param>
		/// <param name="physicalMinTmp"></param>
		/// <param name="physicalMaxTmp"></param>
		/// <param name="clip">if false, then physical value may extend outside worldMin / worldMax.</param>
		/// <returns>The transformed coordinates.</returns>
		public virtual PointF WorldToPhysical( double coord, PointF physicalMinTmp, PointF physicalMaxTmp, bool clip )
		{
			// inefficient in this important function. oh well.
			PointF physicalMin;
			PointF physicalMax;

			if ( this.reversed_ == true )
			{
				physicalMin = physicalMaxTmp;
				physicalMax = physicalMinTmp;
			}
			else
			{
				physicalMin = physicalMinTmp;
				physicalMax = physicalMaxTmp;
			}

			// if want clipped value, return extrema if outside range.
			if ( clip )
			{
				if ( coord > WorldMax )
				{
					return physicalMax;
				}
				if ( coord < WorldMin )
				{
					return physicalMin;
				}
			}

			// we are inside range or don't want to clip.
			PointF offset;
			double difference = WorldMax - WorldMin;
			double prop = (coord - WorldMin) / difference;

			// Force clipping at bounding box largeClip times that of real bounding box 
			// anyway. This prevents many problems.
			const float largeClip = 100.0f;
			if (prop > largeClip)
				prop = largeClip;

			if (prop < -largeClip)
				prop = -largeClip;

			if (difference == 0)
			{
				if (coord >= WorldMin)
					prop = largeClip;
				if (coord < WorldMin)
					prop = -largeClip;
			}

			// calculate the physical coordinate.
			offset = new PointF( (float)(prop * (double)(physicalMax.X - physicalMin.X)),
				(float)(prop * (double)(physicalMax.Y - physicalMin.Y) ));

			return new PointF( (float)(physicalMin.X + offset.X), (float)(physicalMin.Y + offset.Y) );
		}
		#endregion
		#region PhysicalToWorld
		/// <summary>
		/// Transformation from physical (screen) coordinates to  world (user) coordinates.
		/// </summary>
		/// <param name="p">A physical coordinate.</param>
		/// <param name="physicalMinTmp"></param>
		/// <param name="physicalMaxTmp"></param>
		/// <param name="clip">Flag to indicate clipping.</param>
		/// <returns>The transformed coordinates.</returns>
		public virtual double PhysicalToWorld( PointF p, PointF physicalMinTmp, PointF physicalMaxTmp, bool clip )
		{
			// inefficient in this important function. oh well.
			PointF physicalMin;
			PointF physicalMax;

			if ( this.reversed_ == true )
			{
				physicalMin = physicalMaxTmp;
				physicalMax = physicalMinTmp;
			}
			else
			{
				physicalMin = physicalMinTmp;
				physicalMax = physicalMaxTmp;
			}

			// axis dir vector
			double axisX = ((PointF)physicalMax).X - ((PointF)physicalMin).X;
			double axisY = ((PointF)physicalMax).Y - ((PointF)physicalMin).Y;
			double len = Math.Sqrt( axisX * axisX + axisY * axisY );
			axisX /= len;
			axisY /= len;

			// point relative to axis.
			double pointX = ((PointF)p).X - ((PointF)physicalMin).X;
			double pointY = ((PointF)p).Y - ((PointF)physicalMin).Y;

			// dist of point projection on axis, normalised.
			double prop = ( axisX * pointX + axisY * pointY ) / len;

			double ret = prop * (this.WorldMax-this.WorldMin) + this.WorldMin;

			// if want clipped value, return extrema if outside range.
			if (clip)
			{
				ret = Math.Max( ret, this.WorldMin );
				ret = Math.Min( ret, this.WorldMax );
			}
			return ret;
		}
		#endregion
		#region DrawLabel
		/// <summary>
		/// Draw Axis Label
		/// </summary>
		/// <param name="g">The GDI+ drawing surface on which to draw.</param>
		/// <param name="offset">offset calculated by derived class that makes sure axis label
		/// misses tick labels.</param>
		/// <param name="axisPhysMin">The minimum physical extent of the axis.</param>
		/// <param name="axisPhysMax">The maximum physical extent of the axis.</param>
		/// <returns>boxed RectangleF indicating bounding box of label. null if no label printed.</returns>
		public object DrawLabel( Graphics g, PointF offset, PointF axisPhysMin, PointF axisPhysMax )
		{
			if ( label_ != "" )
			{
				// determine angle of axis.
				double theta = (double) System.Math.Atan2(
					axisPhysMax.Y - axisPhysMin.Y,
					axisPhysMax.X - axisPhysMin.X );

				double perpTheta = theta - Math.PI / 2.0f;  // want to move text this way to center on axis.
				double y = Math.Sin(perpTheta);
				double x = Math.Cos(perpTheta);

				PointF average = new PointF(
					(axisPhysMax.X + axisPhysMin.X)/2.0f,
					(axisPhysMax.Y + axisPhysMin.Y)/2.0f );

				g.TranslateTransform( offset.X , offset.Y ); // this is done last.
				g.TranslateTransform( average.X, average.Y );
				theta = theta * 180.0f / Math.PI; // convert to degrees.
				g.RotateTransform((float)theta); // this is done first.

				double lHt = g.MeasureString( label_, FontScaler.scaleFont(labelFont_,FontScale) ).Height;
				double lWd = g.MeasureString( label_, FontScaler.scaleFont(labelFont_,FontScale) ).Width;

				// bb centered around zero.
				RectangleF drawRect = new RectangleF( (float)(-lWd/2.0f), (float)(-lHt/2.0f), (float)lWd, (float)lHt );

				StringFormat drawFormat = new StringFormat();
				drawFormat.Alignment = StringAlignment.Center;
				g.DrawString( label_, FontScaler.scaleFont(this.LabelFont,this.FontScale),
					this.labelBrush_, drawRect, drawFormat);

				// now work out physical bounds of label. and return.
				Matrix m = g.Transform;
				Point[] recPoints = new Point[2];
				recPoints[0] = new Point( (int)(-lWd/2.0f), (int)(-lHt/2.0f) );
				recPoints[1] = new Point( (int)(lWd/2.0f), (int)(lHt/2.0f) );
				m.TransformPoints( recPoints );

				double x1 = Math.Min( recPoints[0].X, recPoints[1].X );
				double x2 = Math.Max( recPoints[0].X, recPoints[1].X );
				double y1 = Math.Min( recPoints[0].Y, recPoints[1].Y );
				double y2 = Math.Max( recPoints[0].Y, recPoints[1].Y );
				RectangleF bounds = new RectangleF( (float)x1, (float)y1, (float)(x2-x1), (float)(y2-y1) );

				g.ResetTransform(); // reset transform before we return.

				return bounds;
			}

			return null;
		}
		#endregion

		#region DrawTick
		/// <summary>
		/// Draws a tick on the axis
		/// </summary>
		/// <param name="g">The drawing surface</param>
		/// <param name="w">The tick position in world coordinates</param>
		/// <param name="size">The size of the tick (in pixels)</param>
		/// <param name="text">The text associated with the tick</param>
		/// <param name="textOffset">The Offset to draw from the auto calculated position</param>
		/// <param name="axisPhysMin">The minimum physical extent of the axis</param>
		/// <param name="axisPhysMax">The maximum physical extent of the axis</param>
		/// <returns> An ArrayList containing the offset from the axis required for an axis label
		/// to miss this tick, followed by a bounding rectangle for the tick and tickLabel drawn </returns>
		public virtual void DrawTick( 
			Graphics g, double w, double size, string text,
			Point textOffset, PointF axisPhysMin, PointF axisPhysMax,
			out PointF labelOffset,
			out RectangleF boundingBox )
		{

			// determine start point.
			PointF s = WorldToPhysical( w, axisPhysMin, axisPhysMax, true );

			// determine offset from start point.
			PointF dir=this.AxisNormVector(axisPhysMin,axisPhysMax);

			// rotate clockwise by angle radians.
			double x1 = Math.Cos( -this.TicksAngle ) * dir.X + Math.Sin( -this.TicksAngle ) * dir.Y;
			double y1 = -Math.Sin( -this.TicksAngle ) * dir.X + Math.Cos( -this.TicksAngle ) * dir.Y;

			// scaling tick.
			dir = new PointF( (float)(this.TickScale * size * x1), (float)(this.TickScale * size * y1) );

			// draw it!
			g.DrawLine( this.linePen_, s.X, s.Y, s.X + dir.X, s.Y + dir.Y );

			// calculate bounds.
			double minx = Math.Min(s.X,s.X+dir.X);
			double miny = Math.Min(s.Y,s.Y+dir.Y);
			double maxx = Math.Max(s.X,s.X+dir.X);
			double maxy = Math.Max(s.Y,s.Y+dir.Y);
			boundingBox = new RectangleF( (float)minx, (float)miny, (float)(maxx-minx), (float)(maxy-miny) );
			labelOffset = new PointF( 0.0f, 0.0f );

			// now draw associated text.
			if (text != "" && !HideTickText )
			{
				double lHt = g.MeasureString( text, FontScaler.scaleFont(font_,this.FontScale) ).Height;
				double lWd = g.MeasureString( text, FontScaler.scaleFont(font_,this.FontScale) ).Width;

				double textCenterX;
				double textCenterY;

				// if text is at pointy end of tick.
				if (!this.TickTextNextToAxis)
				{
					// offset due to tick.
					textCenterX = s.X + dir.X*1.2f;
					textCenterY = s.Y + dir.Y*1.2f;

					// offset due to text box size.
					textCenterX += 0.5f * x1 * lWd;
					textCenterY += 0.5f * y1 * lHt;
				}
					// else it's next to the axis.
				else
				{
					// start location.
					textCenterX = s.X;
					textCenterY = s.Y;

					// offset due to text box size.
					textCenterX -= 0.5f * x1 * lWd;
					textCenterY -= 0.5f * y1 * lHt;

					// bring text away from the axis a little bit.
					textCenterX -= x1*(2.0f+FontScale);
					textCenterY -= y1*(2.0f+FontScale);
				}

				double bx1 = textCenterX-lWd/2.0f;
				double by1 = textCenterY-lHt/2.0f;
				double bx2 = lWd;
				double by2 = lHt;

				RectangleF drawRect = new RectangleF( (float)bx1, (float)by1, (float)bx2, (float)by2 );

				// g.DrawRectangle( new Pen(Color.Green),bx1, by1, bx2, by2 );

				boundingBox = RectangleF.Union( (RectangleF)boundingBox, drawRect );

				// g.DrawRectangle( new Pen(Color.Purple), bounds.X, bounds.Y, bounds.Width, bounds.Height );

				StringFormat drawFormat = new StringFormat();
				drawFormat.Alignment = StringAlignment.Center;
				g.DrawString( text, FontScaler.scaleFont(this.TickTextFont,this.FontScale),
					this.tickTextBrush_, drawRect, drawFormat);

				textCenterX -= s.X;
				textCenterY -= s.Y;
				textCenterX *= 2.3f;
				textCenterY *= 2.3f;

				labelOffset = new PointF((float)textCenterX, (float)textCenterY);

			} 

		}
		#endregion

		#region Draw
		/// <summary>
		/// Draw the axis.
		/// </summary>
		/// <remarks>Polymorphically calls the DrawTicks method of subclasses.</remarks>
		/// <param name="g">The drawing surface on which to draw.</param>
		/// <param name="physicalMin">The minimum physical extent of the axis.</param>
		/// <param name="physicalMax">The maximum physical extent of the axis.</param>
		/// <returns>The bounding rectangle of the axis including axis line, label, tick marks and tick mark labels</returns>
		public virtual object Draw( System.Drawing.Graphics g, PointF physicalMin, PointF physicalMax )
		{
			if (!Hidden)
			{
				object labelOffset;
				object tickBounds;
				this.DrawTicks( g, physicalMin, physicalMax, out labelOffset, out tickBounds );
				
				RectangleF baseBB = this.DoDraw( g, (PointF)labelOffset, physicalMin, physicalMax );

				if (tickBounds == null)
				{
					tickBounds = baseBB;
				}

				return RectangleF.Union( baseBB, (RectangleF)tickBounds );
			}

			return new RectangleF( 
				physicalMin.X, 
				physicalMin.Y, 
				physicalMax.X-physicalMin.X, 
				physicalMax.Y-physicalMin.Y );
		}
		#endregion

		#region InitOffsetAndBounds
		protected void InitOffsetAndBounds( out object labelOffset, out object boundingBox )
		{
			labelOffset = new PointF( 0.0f, 0.0f );
			boundingBox = null;
		}
		#endregion

		#region UpdateOffsetAndBounds
		// TODO: change object labelOffset to PointF etc? 
		protected void UpdateOffsetAndBounds( 
			ref object labelOffset, ref object boundingBox, 
			PointF tLabelOffset, RectangleF tBoundingBox )
		{
			// determining largest label offset.
			PointF lo = (PointF)labelOffset;
			double norm1 = Math.Sqrt( lo.X*lo.X + lo.Y*lo.Y );
			double norm2 = Math.Sqrt( tLabelOffset.X*tLabelOffset.X + tLabelOffset.Y*tLabelOffset.Y );
			if (norm1<norm2)
			{
				labelOffset = tLabelOffset;
			}

			// determining bounding box.
			RectangleF b = tBoundingBox;
			if (boundingBox == null)
			{
				boundingBox = b;
			}
			else
			{
				boundingBox = RectangleF.Union( (RectangleF)boundingBox, b );
			}
		}
		#endregion

		#region DrawTicks
		/// <summary>
		/// DrawTicks method. In base axis class this does nothing.
		/// </summary>
		/// <param name="g">The graphics surface on which to draw</param>
		/// <param name="physicalMin"></param>
		/// <param name="physicalMax"></param>
		/// <param name="labelOffset"></param>
		/// <param name="boundingBox"></param>
		protected virtual void DrawTicks( 
			Graphics g, 
			PointF physicalMin, 
			PointF physicalMax, 
			out object labelOffset,
			out object boundingBox )
		{
			labelOffset = null;
			boundingBox = null;
			// do nothing. This class is not abstract because a subclass may
			// want to override the Axis.Draw method to one that doesn't 
			// require DrawTicks.
		}
		#endregion

		#region DoDraw
		/// <summary>
		/// Draws axis line, labels and determines the used space.
		/// </summary>
		/// <param name="g">The drawing surface on which to draw.</param>
		/// <param name="labelOffset">The label offset.</param>
		/// <param name="physicalMin">The minimum physical extent of the axis.</param>
		/// <param name="physicalMax">The maximum physical extent of the axis.</param>
		/// <returns>The RectangleF containing the axis and labels.</returns>
		protected virtual RectangleF DoDraw( System.Drawing.Graphics g, PointF labelOffset,
			PointF physicalMin, PointF physicalMax )
		{
			double x1 = Math.Min( physicalMin.X, physicalMax.X );
			double x2 = Math.Max( physicalMin.X, physicalMax.X );
			double y1 = Math.Min( physicalMin.Y, physicalMax.Y );
			double y2 = Math.Max( physicalMin.Y, physicalMax.Y );
			RectangleF bounds = new RectangleF( (float)x1, (float)y1, (float)(x2-x1), (float)(y2-y1) );

			if (!Hidden)
			{
				g.DrawLine( this.linePen_, physicalMin.X, physicalMin.Y, physicalMax.X, physicalMax.Y );

				if (!this.hideTickText_)
				{
					object labelBounds = this.DrawLabel(g, labelOffset, physicalMin, physicalMax );
					if (labelBounds != null)
					{
						bounds = RectangleF.Union( bounds, (RectangleF)labelBounds );
					}

				}
			}

			return bounds;
		}
		#endregion

		#region LargeTickSize
		/// <summary>
		/// Accessor for the size of the large tick.
		/// </summary>
		public float LargeTickSize
		{
			get
			{
				return largeTickSize_;
			}
			set
			{
				largeTickSize_ = value;
			}
		}
		#endregion

		#region SmallTickSize
		/// <summary>
		/// Accessor for the size of the small ticks.
		/// </summary>
		public float SmallTickSize
		{
			get
			{
				return smallTickSize_;
			}
			set
			{
				smallTickSize_ = value;
			}
		}
		#endregion 

		#region WorldMax
		/// <summary>
		/// Accessor for the maximum extent of the world (user) coordinates.
		/// </summary>
		public virtual double WorldMax
		{
			get
			{
				if (this.worldMax_ == null)
				{
					return (double)worldMax_;
				}
				else
				{
					return (double)this.worldMax_; // unbox.
				}
			}
			set
			{
				this.worldMax_ = value; // box (or set to null)
			}
		}
		#endregion
		#region WorldMin
		/// <summary>
		/// Accessor for the minumum extent of the world (user) coordinates.
		/// </summary>
		public virtual double WorldMin
		{
			get
			{
				return (double)this.worldMin_; // un box
			}
			set
			{
				this.worldMin_ = value; // box (or set to null).
			}
		}
		#endregion

		#region Length
		/// <summary>
		/// Length of the Axis (in world coordinates).
		/// </summary>
		public double Length
		{
			get
			{
				return Math.Abs( (double)this.worldMax_ - (double)this.worldMin_ );
			}
		}
		#endregion

		#region TickTextFont
		/// <summary>
		/// Accessor for the tick text Font.
		/// </summary>
		public Font TickTextFont
		{
			get
			{
				return this.font_;
			}
			set
			{
				this.font_ = value;
			}
		}
		#endregion
		#region LabelFont
		/// <summary>
		/// Accessor for the label Font.
		/// </summary>
		public Font LabelFont
		{
			get
			{
				return this.labelFont_;
			}
			set
			{
				this.labelFont_ = value;
			}
		}
		#endregion

		#region LabelBrush
		/// <summary>
		/// Accessor for the Brush used to draw the label.
		/// </summary>
		public Brush LabelBrush
		{
			get
			{
				return labelBrush_;
			}
			set
			{
				labelBrush_ = value;
			}
		}
		#endregion

		#region TickTextBrush
		/// <summary>
		/// Accessor for the Brush used to draw the tick text.
		/// </summary>
		public Brush TickTextBrush
		{
			get
			{
				return tickTextBrush_;
			}
			set
			{
				tickTextBrush_ = value;
			}
		}
		#endregion

		#region Label
		/// <summary>
		/// Accessor for the label to display on the axis.
		/// </summary>
		public string Label
		{
			get
			{
				return this.label_;
			}
			set
			{
				this.label_ = value;
			}
		}
		#endregion

		#region FontScale
		/// <summary>
		/// Accessor for the font scale factor.
		/// </summary>
		public float FontScale
		{
			get
			{
				return this.fontScale_;
			}
			set
			{
				this.fontScale_ = value;
			}
		}
		#endregion
		#region AxisNormVector
		/// <summary>
		/// Returns unitary vector oriented along axis.
		/// </summary>
		/// <param name="min"></param>
		/// <param name="max"></param>
		/// <returns>The unitary vector.</returns>
		protected PointF AxisNormVector(PointF min,PointF max)
		{
			PointF dir = new PointF( max.X - min.X,max.Y - min.Y );
			double dirNorm = Math.Sqrt(dir.X*dir.X+dir.Y*dir.Y);
			if ( dirNorm > 0.0f)
			{
				dir = new PointF( (float)((1.0f/dirNorm)*dir.X), (float)((1.0f/dirNorm)*dir.Y) ); // normalised axis direction vector
			}
			return dir;
		}
		#endregion
		#region TickScale
		/// <summary>
		/// Accessor for the distance between ticks.
		/// </summary>
		public float TickScale
		{
			get
			{
				return this.tickScale_;
			}
			set
			{
				this.tickScale_ = value;
			}
		}
		#endregion
		#region TickTextNextToAxis
		/// <summary>
		/// If true, the text associated with the ticks will appear on the opposite side of the axis to the tick.
		/// If false, the text will appear at the end of the tick.
		/// </summary>
		public bool TickTextNextToAxis
		{
			get
			{
				return tickTextNextToAxis_;
			}
			set
			{
				tickTextNextToAxis_ = value;
			}
		}
		#endregion
		#region Hidden
		/// <summary>
		/// Flag to hide the axis.
		/// </summary>
		public bool Hidden
		{
			get
			{
				return this.hidden_;
			}
			set
			{
				this.hidden_ = value;
			}
		}
		#endregion
		#region Reversed
		/// <summary>
		/// Reverse the display of the axis so that the values are
		/// printed from right to left instead of left to right
		/// </summary>
		public bool Reversed
		{
			get
			{
				return this.reversed_;
			}
			set
			{
				this.reversed_ = value;
			}
		}
		#endregion
		#region HideTickText
		/// <summary>
		/// Flag to hide the tick text.
		/// </summary>
		public bool HideTickText
		{
			get
			{
				return this.hideTickText_;
			}
			set
			{
				this.hideTickText_ = value;
			}
		}
		#endregion
		#region LinePen
		/// <summary>
		/// Accessor for the Pen used to draw the line
		/// </summary>
		public System.Drawing.Pen LinePen
		{
			get
			{
				return this.linePen_;
			}
			set
			{
				this.linePen_ = value;
			}
		}
		private System.Drawing.Pen linePen_;
		#endregion

		#region TicksAngle
		/// <summary>
		/// The angle used to draw ticks.
		/// </summary>
		public double TicksAngle
		{
			get
			{
				if (ticksAngle_ == null)
				{
					return (double)(Math.PI / 2.0);
				}
				else
				{
					return (double)ticksAngle_;
				}
			}
			set
			{
				ticksAngle_ = (double)value;
			}
		}
		#endregion
		#region LargeTickPositions
		/// <summary>
		/// The ArrayList containing the positions of the large ticks.
		/// </summary>
		public virtual ArrayList LargeTickPositions
		{
			get
			{
				return new ArrayList();
			}
		}
		#endregion
		#region SmallTickPositions
		/// <summary>
		/// The ArrayList containing the positions of the small ticks.
		/// </summary>
		public virtual ArrayList SmallTickPositions
		{
			get
			{
				return new ArrayList();
			}
		}
		#endregion



		#region NumberFormat
		/// <summary>
		/// The .NET number format to use for labeling the axis
		/// </summary>
		public string NumberFormat
		{
			get
			{
				return numFormat_;
			}
			set
			{
				numFormat_ = value;
			}
		}
		#endregion


		/*


		private void UpdateMantissaLogs()
		{
			mantissaLogs_ = new double[gridMantissas_.Length];
			for (int i=0; i < gridMantissas_.Length; i++)
			{
				mantissaLogs_[i] = Math.Log10(gridMantissas_[i]);
			}
		}

		/// <summary>
		/// Specify the values to use for the grid tick marks
		/// The actual GridMantissa used will be the one
		/// that puts the GridLineCount as close as possible to the selected value
		/// </summary>
		public double[] GridMantissas
		{
			get
			{
				return gridMantissas_;
			}
			set
			{
				// Resolve value to between 0 and 10
				for (int i=0; i<value.Length; i++)
				{
					if (value[i] < 0)
					{
						value[i] = - value[i];
					}
					if (value[i] == 0)
					{
						value[i] = 10;
					}
					else if (value[i] > 10)
					{
						double d = Math.Log10(value[i]);
						value[i] = Math.Pow(10, d - (int)d);
					}
				}
				gridMantissas_ = value;
				Array.Sort(gridMantissas_, Comparer.Default);
				UpdateMantissaLogs();
			}
		}

		/// <summary>
		/// Set the approximate number of Grids.
		/// The actual value will depend on the data range and the GridMantissa calculated
		/// so that the GridLineCount is as close as possible to the selected value.
		/// Get returns the actual value in use.
		/// </summary>
		public int GridLineCount
		{
			get
			{
				return gridlinecount_;
			}
			set
			{
				gridlinecount_ = value;
			}
		}

		/// <summary>
		/// Get the calculated grid interval based on the data range, GridLineCount
		/// and GridMantissa used.
		/// </summary>
		public double GridInterval
		{
			get
			{
				return gridinterval_;
			}
		}

		internal double[] MantissaLogs
		{
			get
			{
				return mantissaLogs_;
			}
		}


		/// <summary>
		///
		/// </summary>
		public double Scale
		{
			get
			{
				return scale_;
			}
			set
			{
				scale_ = value;
			}
		}

		/// <summary>
		///
		/// </summary>
		public double Offset
		{
			get
			{
				return offset_;
			}
			set
			{
				offset_ = value;
			}
		}

*/
	}
}
