/*
ScPl - A plotting library for .NET

Grid.cs
Copyright (C) 2003
Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: Grid.cs,v 1.3 2004/05/03 08:45:29 mhowlett Exp $

*/

using System;
using System.Drawing;
using System.Drawing.Drawing2D;

namespace scpl
{

	/// <summary>
	/// Class that provides grid drawing functionality.
	/// </summary>
	public class Grid : IDrawable
	{
		#region enum GridType
		/// <summary>
		/// GridType enumeration.
		/// </summary>
		public enum GridType
		{
			/// <summary>
			/// No grid.
			/// </summary>
			None = 0,
			/// <summary>
			/// Coarse grid.
			/// </summary>
			Coarse = 1,
			/// <summary>
			/// Fine grid.
			/// </summary>
			Fine = 2
		}
		#endregion

		#region Constructor
		/// <summary>
		/// Default constructor
		/// </summary>
		public Grid()
		{
			minorGridPen_ = new Pen( Color.LightGray );
			float[] pattern = {1.0f, 2.0f};
			minorGridPen_.DashPattern = pattern;
			
			majorGridPen_ = new Pen( Color.LightGray );

			horizontalGridType_ = GridType.Fine;
			
			verticalGridType_ = GridType.Fine;
		}

		// TODO other constructors.
		#endregion

		#region HorizontalGridType
		/// <summary>
		/// The coarseness of the horizontal grid lines.
		/// </summary>
		public GridType HorizontalGridType
		{
			get
			{
				return horizontalGridType_;
			}
			set
			{
				horizontalGridType_ = value;
			}
		}
		GridType horizontalGridType_;
		#endregion
		#region VerticalGridType
		/// <summary>
		/// The coarseness of the vertical grid lines.
		/// </summary>
		public GridType VerticalGridType
		{
			get
			{
				return verticalGridType_;
			}
			set
			{
				verticalGridType_ = value;
			}
		}
		GridType verticalGridType_;
		#endregion
		#region MajorGridPen
		/// <summary>
		/// The Pen used for the major grid lines.
		/// </summary>
		public System.Drawing.Pen MajorGridPen
		{
			get
			{
				return majorGridPen_;
			}
			set
			{
				majorGridPen_ = value;
			}
		}
		private Pen majorGridPen_;
		#endregion
		#region MinorGridPen
		/// <summary>
		/// The Pen used for the minor grid lines.
		/// </summary>
		public System.Drawing.Pen MinorGridPen
		{
			get
			{
				return minorGridPen_;
			}
			set
			{
				minorGridPen_ = value;
			}
		}
		private Pen minorGridPen_;
		#endregion

		#region Draw
		#region DrawGridLines
		/// <summary>
		/// Does all the work in drawing grid lines.
		/// </summary>
		/// <param name="g">The graphics surface on which to render.</param>
		/// <param name="axis">TODO</param>
		/// <param name="orthogonalAxis">TODO</param>
		/// <param name="a">the list of world values to draw grid lines at.</param>
		/// <param name="horizontal">true if want horizontal lines, false otherwise.</param>
		/// <param name="p">the pen to use to draw the grid lines.</param>
		private void DrawGridLines( Graphics g, PhysicalAxis axis, PhysicalAxis orthogonalAxis,
			System.Collections.ArrayList a, bool horizontal, Pen p )
		{
			for (int i=0; i<a.Count; ++i)
			{
				PointF p1 = axis.WorldToPhysical((double)a[i], true);
				PointF p2 = p1;
				PointF p3 = orthogonalAxis.PhysicalMax;
				PointF p4 = orthogonalAxis.PhysicalMin;
				if (horizontal)
				{
					p1.Y = p4.Y;
					p2.Y = p3.Y;
				}
				else
				{
					p1.X = p4.X;
					p2.X = p3.X;
				}
				g.DrawLine( p, p1, p2 );
			}
		}
		#endregion

		/// <summary>
		/// Draws the grid
		/// </summary>
		/// <param name="g">The graphics surface on which to draw</param>
		/// <param name="xAxis">The physical x axis to draw horizontal lines parallel to.</param>
		/// <param name="yAxis">The physical y axis to draw vertical lines parallel to.</param>
		public void Draw( Graphics g, PhysicalAxis xAxis, PhysicalAxis yAxis )
		{
			if (this.horizontalGridType_ == GridType.Fine)
			{
				System.Collections.ArrayList a = xAxis.Axis.SmallTickPositions;
				DrawGridLines( g, xAxis, yAxis, a, true, this.MinorGridPen );
			}

			if (this.verticalGridType_ == GridType.Fine)
			{
				System.Collections.ArrayList a = yAxis.Axis.SmallTickPositions;
				DrawGridLines( g, yAxis, xAxis, a, false, this.MinorGridPen );
			}

			if (this.horizontalGridType_ != GridType.None)
			{
				System.Collections.ArrayList a = xAxis.Axis.LargeTickPositions;
				DrawGridLines( g, xAxis, yAxis, a, true, this.MajorGridPen );	
			}

			if (this.verticalGridType_ != GridType.None)
			{
				System.Collections.ArrayList a = yAxis.Axis.LargeTickPositions;
				DrawGridLines( g, yAxis, xAxis, a, false, this.MajorGridPen );
			}
		}
		#endregion

	}
}
