/*
ScPl - A plotting library for .NET

HistogramPlot.cs
Copyright (C) 2003
Matt Howlett, Paolo Pierini

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: HistogramPlot.cs,v 1.17 2004/04/25 12:58:06 mhowlett Exp $

*/

using System;
using System.Drawing;
using System.Drawing.Drawing2D;

namespace scpl
{

	/// <summary>
	/// Provides ability to trace histogram plots.
	/// </summary>
	public class HistogramPlot : BasePlot, ISequencePlot
	{
		#region DataSource
		public object DataSource
		{
			get
			{
				return this.dataSource_;
			}
			set
			{
				this.dataSource_ = value;
			}
		}
		private object dataSource_ = null;
		#endregion
		#region ValueData
		public object ValueData
		{
			get
			{
				return this.valueData_;
			}
			set
			{
				this.valueData_ = value;
			}
		}
		private object valueData_ = null;
		#endregion		
		#region AbscissaData
		public object AbscissaData
		{
			get
			{
				return this.abscissaData_;
			}
			set
			{
				this.abscissaData_ = value;
			}
		}
		private object abscissaData_ = null;
		#endregion

		#region BrushOrientations
		/// <summary>
		/// The enumeration to provide a linear gradient brush.
		/// </summary>
		public enum BrushOrientations
		{
			/// <summary>
			/// No brush filling.
			/// </summary>
			None,
			/// <summary>
			/// Horizontal gradient filling from the chosen color to White.
			/// </summary>
			Horizontal,
			/// <summary>
			/// Horizontal gradient filling, with white at the center.
			/// </summary>
			HorizontalFadeIn,
			/// <summary>
			/// Horizontal gradient filling, with white at the edges.
			/// </summary>
			HorizontalFadeOut,
			/// <summary>
			/// Vertical gradient filling from the chosen color to White.
			/// </summary>
			Vertical, 
			/// <summary>
			/// Vertical gradient filling, with white at the center.
			/// </summary>
			VerticalFadeIn,
			/// <summary>
			/// Vertical gradient filling, with white at the edges.
			/// </summary>
			VerticalFadeOut
		}
		#endregion

		#region BrushOrientation
		/// <summary>
		/// Set/Get the orientation of the linear color gradient.
		/// </summary>
		public BrushOrientations BrushOrientation
		{
			get
			{
				return brushOrientation_;
			}
			set
			{
				brushOrientation_ = value;
			}

		}
		private BrushOrientations brushOrientation_ = BrushOrientations.None;
		#endregion
		/// <summary>
		/// Constructor for the HistogramPlot.
		/// </summary>
		public HistogramPlot()
		{
			this.Center = true;
		}

		#region Draw
		/// <summary>
		/// Renders the histogram.
		/// </summary>
		/// <param name="g">The Graphics surface.</param>
		/// <param name="xAxis">The X axis where the trace plot is attached to.</param>
		/// <param name="yAxis">The Y axis where the trace plot is attached to.</param>
		public void Draw( Graphics g, PhysicalAxis xAxis, PhysicalAxis yAxis )
		{
			SequenceAdapter data = 
				new SequenceAdapter( this.DataSource, this.DataMember, this.ValueData, this.AbscissaData );

			float yoff;
			bool needToDisposePen = false;
			System.Drawing.Pen p;
			if ( this.pen_ != null )
			{
				p = (System.Drawing.Pen)this.pen_;
			}
			else
			{
				p =	new	Pen( this.color_ );
				needToDisposePen = true;
			}

			for ( int i=0; i<data.Count; ++i )
			{
				PointD p1 = data[i];
				if ( !Double.IsNaN(p1.X) && !Double.IsNaN(p1.Y) )
				{
					// TODO: probably should check for p2 too.
					PointD p2;
					if (i+1 != data.Count)
					{
						p2 = data[i+1];
						p2.Y = p1.Y;
					}
					else
					{
						p2 = data[i-1];
						double offset = p1.X - p2.X;
						p2.X = p1.X + offset;
						p2.Y = p1.Y;
					}

					HistogramPlot currentPlot = this;

					yoff = 0.0f;
					double yval = 0.0f;
					while (currentPlot.isStacked_)
					{
						SequenceAdapter stackedToData = new SequenceAdapter(
							currentPlot.stackedTo_.DataSource, 
							currentPlot.stackedTo_.DataMember,
							currentPlot.stackedTo_.ValueData, 
							currentPlot.stackedTo_.AbscissaData 
						);
						yval += stackedToData[i].Y;
						yoff = yAxis.WorldToPhysical( yval, false ).Y;
						p1.Y += stackedToData[i].Y;
						p2.Y += stackedToData[i].Y;
						currentPlot = currentPlot.stackedTo_;
					}

					if ( center_ )
					{
						double offset = ( p2.X - p1.X ) / 2.0f;
						p1.X -= offset;
						p2.X -= offset;
					}

					PointF xPos1 = xAxis.WorldToPhysical( p1.X, false );
                    PointF yPos1 = yAxis.WorldToPhysical( p1.Y, false );
					PointF xPos2 = xAxis.WorldToPhysical( p2.X, false );
					PointF yPos2 = yAxis.WorldToPhysical( p2.Y, false );

					if (isStacked_)
					{
						currentPlot = this;
						while (currentPlot.isStacked_)
						{
							currentPlot = currentPlot.stackedTo_;
						}
						this.baseWidth_ = currentPlot.baseWidth_;
					}

					float width = xPos2.X-xPos1.X;
					float height;
					if (isStacked_)
					{
						height= -yPos1.Y+yoff;
					}
					else
					{
						height= -yPos1.Y+yAxis.PhysicalMin.Y;
					}

					float xoff = (1.0f - baseWidth_)/2.0f*width;
					RectangleF r = new RectangleF(xPos1.X+xoff,yPos1.Y,width-2*xoff,height);

					if (this.Filled)
					{
						if ( r.Height != 0.0f && r.Width != 0.0f)
						{
							Brush brush = this.GetProperBrush(r);
							g.FillRectangle(brush,r);
							brush.Dispose();
						}
					}
					g.DrawRectangle(p,r.X,r.Y,r.Width,r.Height);
				}
			}
			if (needToDisposePen) p.Dispose();
		}
		#endregion

		#region GetProperBrush
		private Brush GetProperBrush(RectangleF r)
		{
			Brush brush;
			bool needToDisposePen = false;
			System.Drawing.Pen p;
			if ( this.pen_ != null )
			{
				p = (System.Drawing.Pen)this.pen_;
			}
			else
			{
				p =	new	Pen( this.color_ );
				needToDisposePen = true;
			}
			switch ( this.brushOrientation_ )
			{
				case (BrushOrientations.None) :
				{
					brush = new SolidBrush(p.Color);
					break;
				}
				case (BrushOrientations.Horizontal) :
				{
					brush = new LinearGradientBrush(r,p.Color,Pens.White.Color,LinearGradientMode.Horizontal);
					break;
				}
				case (BrushOrientations.HorizontalFadeIn) :
				{
					brush = new LinearGradientBrush(r,p.Color,Pens.White.Color,LinearGradientMode.Horizontal);
					float[] relativeIntensities = {0.0f, 0.9f, 1.0f, 0.9f, 0.0f};
					float[] relativePositions   = {0.0f, 0.4f, 0.5f, 0.6f, 1.0f};
					Blend blend = new Blend();
					blend.Factors = relativeIntensities;
					blend.Positions = relativePositions;
					((LinearGradientBrush)brush).Blend = blend;
					break;
				}
				case (BrushOrientations.HorizontalFadeOut) :
				{
					brush = new LinearGradientBrush(r,p.Color,Pens.White.Color,LinearGradientMode.Horizontal);
					float[] relativeIntensities = {1.0f, 0.1f, 0.0f, 0.1f, 1.0f};
					float[] relativePositions   = {0.0f, 0.4f, 0.5f, 0.6f, 1.0f};
					Blend blend = new Blend();
					blend.Factors = relativeIntensities;
					blend.Positions = relativePositions;
					((LinearGradientBrush)brush).Blend = blend;
					break;
				}
				case (BrushOrientations.Vertical) :
				{
					brush = new LinearGradientBrush(r,p.Color,Pens.White.Color,LinearGradientMode.Vertical);
					break;
				}
				case (BrushOrientations.VerticalFadeIn) :
				{
					brush = new LinearGradientBrush(r,p.Color,Pens.White.Color,LinearGradientMode.Vertical);
					float[] relativeIntensities = {0.0f, 0.9f, 1.0f, 0.9f, 0.0f};
					float[] relativePositions   = {0.0f, 0.4f, 0.5f, 0.6f, 1.0f};
					Blend blend = new Blend();
					blend.Factors = relativeIntensities;
					blend.Positions = relativePositions;
					((LinearGradientBrush)brush).Blend = blend;
					break;
				}
				case (BrushOrientations.VerticalFadeOut) :
				{
					brush = new LinearGradientBrush(r,p.Color,Pens.White.Color,LinearGradientMode.Vertical);
					float[] relativeIntensities = {1.0f, 0.1f, 0.0f, 0.1f, 1.0f};
					float[] relativePositions   = {0.0f, 0.4f, 0.5f, 0.6f, 1.0f};
					Blend blend = new Blend();
					blend.Factors = relativeIntensities;
					blend.Positions = relativePositions;
					((LinearGradientBrush)brush).Blend = blend;
					break;
				}
				default :
				{
					brush = new SolidBrush(p.Color);
					break;
				}
			}
			if(needToDisposePen) p.Dispose();
			return brush;
		}
		#endregion

		#region Filled
		/// <summary>
		/// Flag for filling the interior of the histogram bars.
		/// </summary>
		public bool Filled
		{
			get 
			{
				return filled_;
			}
			set
			{
				filled_ = value;
			}
		}
		private bool filled_ = false;
		#endregion
		#region BaseWidth
		private float baseWidth_ = 1.0f;
        /// <summary>
        /// The width of the histogram bar with respect to the data spacing 
        /// (in percentage 0-1).
        /// </summary>
		public float BaseWidth
		{
			get
			{
				return baseWidth_;
			}
			set
			{
				if(value > 0.0 && value <= 1.0)
				{
					baseWidth_ = value;
				}
			}
		}
		#endregion
		#region SuggestXAxis
		/// <summary>
		/// Provides a hint for the X axis.
		/// </summary>
		/// <returns>The X axis.</returns>
		public Axis SuggestXAxis()
		{
			SequenceAdapter data = 
				new SequenceAdapter( this.DataSource, this.DataMember, this.ValueData, this.AbscissaData );

			if (data.Count < 2)
			{
				return data.SuggestXAxis();
			}

			// else

			Axis a = data.SuggestXAxis();

			PointD p1 = data[0];
			PointD p2 = data[1];
			PointD p3 = data[data.Count-2];
			PointD p4 = data[data.Count-1];

			double offset1;
			double offset2;

			if (!center_)
			{
				offset1 = 0.0f;
				offset2 = p4.X - p3.X;
			}
			else
			{
				offset1 = (p2.X - p1.X)/2.0f;
				offset2 = (p4.X - p3.X)/2.0f;
			}

			a.WorldMin -= offset1;
			a.WorldMax += offset2;

			return a;
		}
		#endregion
		#region SuggestYAxis
		/// <summary>
		/// Provides a hint for the Y axis.
		/// </summary>
		/// <returns>The Y axis.</returns>
		public Axis SuggestYAxis()
		{

			if ( this.isStacked_ )
			{
				// I have to iterate through all the stacked series
				HistogramPlot currentPlot = this;
				double tmpMin = 0.0f;
				double tmpMax = 0.0f;
				while ( currentPlot.isStacked_ )
				{
					SequenceAdapter data = new SequenceAdapter( 
						currentPlot.DataSource,
						currentPlot.DataMember,
						currentPlot.ValueData, 
						currentPlot.AbscissaData 
					);

					Axis tmpAxis;
					tmpAxis = data.SuggestYAxis();
					tmpMin = Math.Min(tmpMin,tmpAxis.WorldMin);
					tmpMax += tmpAxis.WorldMax;
					currentPlot = currentPlot.stackedTo_;
				}
				return new LinearAxis(tmpMin,tmpMax);
			}
			else
			{
				SequenceAdapter data = 
					new SequenceAdapter( this.DataSource, this.DataMember, this.ValueData, this.AbscissaData );

				return data.SuggestYAxis();
			}
		}
		#endregion
		private bool center_;
		#region Center
		/// <summary>
		/// Flag for centering the histogram.
		/// </summary>
		public bool Center
		{
			set
			{
				center_ = value;
			}
			get
			{
				return center_;
			}
		}
		#endregion
		private bool isStacked_;
		#region isStacked
		/// <summary>
		/// Flag to determine if the histogram is stacked to another instance.
		/// </summary>
		public bool isStacked
		{
			get
			{
				return isStacked_;
			}
		}
		#endregion
		private HistogramPlot stackedTo_;
		#region StackedTo
		/// <summary>
		/// Stack the histogram to another HistogramPlot.
		/// </summary>
		/// <param name="hp">The instance to stack data to.</param>
		public void StackedTo(HistogramPlot hp)
		{
			SequenceAdapter data = 
				new SequenceAdapter( this.DataSource, this.DataMember, this.ValueData, this.AbscissaData );

			SequenceAdapter hpData = 
				new SequenceAdapter( hp.DataSource, hp.DataMember, hp.ValueData, hp.AbscissaData );

				if ( hp != null )
				{
					isStacked_ = true;
					if ( hpData.Count != data.Count )
					{
						throw new Exception("Can stack HistogramPlot data only with the same number of datapoints.");
					}
					for ( int i=0; i < data.Count; ++i )
					{
						if ( data[i].X != hpData[i].X )
						{
							throw new Exception("Can stack HistogramPlot data only with the same X coordinates.");
						}
						if ( hpData[i].Y < 0.0f)
						{
							throw new Exception("Can stack HistogramPlot data only with positive Y coordinates.");
						}
					}
				}
				stackedTo_ = hp;
		}
		#endregion

		#region DrawLegendLine override
		/// <summary>
		/// Draws the HistogramPlot legend in the same style as the data plot.
		/// </summary>
		/// <param name="g">The Graphics surface to render.</param>
		/// <param name="startEnd">The rectangle where to plot the legend.</param>
		public override void DrawLegendLine(Graphics g, RectangleF startEnd)
		{
			bool needToDisposePen = false;
			Pen	p;
			if ( this.pen_ != null )
			{
				p = (System.Drawing.Pen)this.pen_;
			}
			else
			{
				p =	new	Pen(this.color_);
				needToDisposePen = true;
			}

			if (this.Filled)
			{
				Brush brush = this.GetProperBrush(startEnd);
				g.FillRectangle(brush,startEnd);
				brush.Dispose();
			}
			g.DrawRectangle(p,startEnd.X,startEnd.Y,startEnd.Width,startEnd.Height);
			if (needToDisposePen) p.Dispose();
		}
		#endregion
	}
}
