/*
ScPl - A plotting library for .NET

ImagePlot.cs
Copyright (C) 2003-2004
Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: ImagePlot.cs,v 1.10 2004/04/26 04:18:41 mhowlett Exp $

*/

using System;
using System.Drawing;
using System.Drawing.Drawing2D;

namespace scpl
{
	public class ImagePlot : IMeshPlot
	{
		private double[,] data_;
		private double xStart_ = 0.0;
		private double xStep_ = 1.0;
		private double yStart_ = 0.0;
		private double yStep_ = 1.0;
		private double dataMin_;
		private double dataMax_;

		#region calculateMinMax
		private void calculateMinMax()
		{
			dataMin_ = data_[0,0];
			dataMax_ = data_[0,0];
			for (int i=0; i<data_.GetLength(0); ++i)
			{
				for (int j=0; j<data_.GetLength(1); ++j)
				{
					if (data_[i,j]<dataMin_) 
					{
						dataMin_ = data_[i,j];
					}
					if (data_[i,j]>dataMax_) 
					{
						dataMax_ = data_[i,j];
					}
				}
			}
		}
		#endregion

		#region Constructor
		// no adapters for this yet - when we get some more 2d plotting functionality, then
		// perhaps create some.
		public ImagePlot( double[,] data, double xStart, double xStep, double yStart, double yStep )
		{

#if CHECK_ERRORS
			if (data == null || data.GetLength(0) == 0 || data.GetLength(1) == 0)
			{
				throw new System.Exception( "ERROR: ImagePlot.ImagePlot: Data null, or zero length" );
			}
#endif

			this.data_ = data;
			this.xStart_ = xStart;
			this.xStep_ = xStep;
			this.yStart_ = yStart;
			this.yStep_ = yStep;
			this.calculateMinMax();
		}
		public ImagePlot( double[,] data )
		{
			this.data_ = data;
			this.calculateMinMax();
		}
		#endregion

		#region Draw
		/// <summary>
		/// TODO: block positions may be off by a pixel or so. maybe. Re-think calculations.
		/// </summary>
		/// <param name="g"></param>
		/// <param name="xAxis"></param>
		/// <param name="yAxis"></param>
		public void Draw( Graphics g, PhysicalAxis xAxis, PhysicalAxis yAxis )
		{
			if ( data_==null || data_.GetLength(0) == 0 || data_.GetLength(1) == 0 )
			{
				return;
			}

			double worldWidth = xAxis.Axis.WorldMax - xAxis.Axis.WorldMin;
			double numBlocksHorizontal = worldWidth / this.xStep_;
			double worldHeight = yAxis.Axis.WorldMax - yAxis.Axis.WorldMin;
			double numBlocksVertical = worldHeight / this.yStep_;

			double physicalWidth = xAxis.PhysicalMax.X - xAxis.PhysicalMin.X;
			double blockWidth = physicalWidth / numBlocksHorizontal;
			bool wPositive = true;
			if (blockWidth < 0.0)
			{
				wPositive = false;
			}
			blockWidth = Math.Abs(blockWidth)+1;

			double physicalHeight = yAxis.PhysicalMax.Y - yAxis.PhysicalMin.Y;
			double blockHeight = physicalHeight / numBlocksVertical;
			bool hPositive = true;
			if (blockHeight < 0.0)
			{
				hPositive = false;
			}
			blockHeight = Math.Abs(blockHeight)+1;

			for (int i=0; i<data_.GetLength(0); ++i)
			{
				for (int j=0; j<data_.GetLength(1); ++j)
				{
					double wX = (double)j*this.xStep_ + xStart_;
					double wY = (double)i*this.yStep_ + yStart_;
					if ( !hPositive )
					{
						wY += yStep_;
					}
					if (!wPositive )
					{
						wX += xStep_;
					}

					if (this.center_)
					{
						wX -= this.xStep_/2.0;
						wY -= this.yStep_/2.0;
					}
					Pen p = new Pen( this.Gradient.GetColor( (data_[i,j]-this.dataMin_)/(this.dataMax_-this.dataMin_) ) );
					int x = (int)xAxis.WorldToPhysical(wX,false).X;
					int y = (int)yAxis.WorldToPhysical(wY,false).Y;
					g.FillRectangle( p.Brush,
						x,
						y, 
						(int)blockWidth,
						(int)blockHeight );
					//g.DrawRectangle(Pens.White,x,y,(int)blockWidth,(int)blockHeight);
				}
			}
		}
		#endregion

		#region Gradient
		public IGradient Gradient
		{
			get
			{
				if (gradient_ == null)
				{
					gradient_ = new LinearGradient( Color.FromArgb(255,255,255), Color.FromArgb(0,0,0) );
				}
				return this.gradient_;
			}
			set
			{
				this.gradient_ = value;
			}
		}
		private IGradient gradient_;
		#endregion

		#region DrawLegendLine
		public void DrawLegendLine( Graphics g, RectangleF startEnd )
		{

		}
		#endregion
		#region Label
		/// <summary>
		/// The trace label.
		/// </summary>
		public string Label
		{
			get
			{
				return label_;
			}
			set
			{
				this.label_ = value;
			}
		}
		private string label_ = "";
		#endregion

		#region SuggestXAxis
		/// <summary>
		/// The method used to set the abscissa axis defaults.
		/// </summary>
		public Axis SuggestXAxis()
		{
			if (this.center_)
			{
				return new LinearAxis( this.xStart_ - this.xStep_/2.0, this.xStart_ + this.xStep_ * data_.GetLength(1) - this.xStep_/2.0 );
			}
			
			return new LinearAxis( this.xStart_, this.xStart_ + this.xStep_ * data_.GetLength(1) );
		}
		#endregion
		#region SuggestYAxis
		/// <summary>
		/// The method used to set the ordinate axis defaults.
		/// </summary>
		public Axis SuggestYAxis()
		{
			if (this.center_)
			{
				return new LinearAxis( this.yStart_ - this.yStep_/2.0, this.yStart_ + this.yStep_ * data_.GetLength(0) - this.yStep_/2.0 );
			}
			
			return new LinearAxis( this.yStart_, this.yStart_ + this.yStep_ * data_.GetLength(0) );
		}
		#endregion

		#region Center
		/// <summary>
		/// Flag for centering the histogram.
		/// </summary>
		public bool Center
		{
			set
			{
				center_ = value;
			}
			get
			{
				return center_;
			}
		}
		private bool center_ = true;
		#endregion

		#region ShowLegendLine
		public bool ShowLegendLine
		{
			get
			{
				return showLegendLine_;
			}
			set
			{
				this.showLegendLine_ = value;
			}
		}
		private bool showLegendLine_ = true;
		#endregion

		#region DataSource
		public object DataSource
		{
			get
			{
				return this.dataSource_;
			}
			set
			{
				this.dataSource_ = value;
			}
		}
		protected object dataSource_ = null;
		#endregion
		#region DataMember
		public string DataMember
		{
			get
			{
				return this.dataMember_;
			}
			set
			{
				this.dataMember_ = value;
			}
		}
		protected string dataMember_ = null;
		#endregion
	}
}
