/*
ScPl - A plotting library for .NET

LabelAxis.cs
Copyright (C) 2003
Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
$Id: LabelAxis.cs,v 1.15 2004/05/07 11:48:55 mhowlett Exp $

*/

using System;
using System.Collections;
using System.Drawing;

namespace scpl
{
	/// <summary>
	/// The class implementing labeled axes.
	/// </summary>
	public class LabelAxis : Axis
	{
		#region Clone implementation
		/// <summary>
		/// Deep copy of LabelAxis.
		/// </summary>
		/// <returns>A copy of the LinearAxis Class.</returns>
		public override object Clone()
		{
			LabelAxis a = new LabelAxis();
			// ensure that this isn't being called on a derived type. If it is, then oh no!
			if (this.GetType() != a.GetType())
			{
				throw new System.Exception( "Clone not defined in derived type. Help!" );
			}
			DoClone( this, a );
			return a;
		}

		/// <summary>
		/// Helper method for Clone.
		/// </summary>
		/// <param name="a">The original object to clone.</param>
		/// <param name="b">The cloned object.</param>
		protected static void DoClone( LabelAxis b, LabelAxis a )
		{
			Axis.DoClone( b, a );

			a.labels_ = (ArrayList)b.labels_.Clone();
			a.numbers_ = (ArrayList)b.numbers_.Clone();

			// a_.property = this.property
		}
		#endregion

		private void init()
		{
			labels_ = new ArrayList();
			numbers_ = new ArrayList();
		}

		#region Constructors
		/// <summary>
		/// Constructor, cloning an existing Axis.
		/// </summary>
		/// <param name="a">The Axis to clone.</param>
		public LabelAxis( Axis a )
			: base( a )
		{
			init();
		}

		/// <summary>
		/// Default parameterless constructor.
		/// </summary>
		public LabelAxis()
			: base()
		{
			init();
		}

		/// <summary>
		/// Constructor that takes only world min and max values.
		/// </summary>
		/// <param name="worldMin">The minimum world coordinate.</param>
		/// <param name="worldMax">The maximum world coordinate.</param>
		public LabelAxis( double worldMin, double worldMax )
			: base( worldMin, worldMax )
		{
			init();
		}
		#endregion

		/// <summary>
		/// Method to add the labels to the axis.
		/// </summary>
		/// <param name="name">The label to display.</param>
		/// <param name="val">A numeric value.</param>
		public void AddLabel( string name, double val )
		{
			labels_.Add( name );
			numbers_.Add( val );
		}
		

		protected override void DrawTicks( 
			Graphics g, 
			PointF physicalMin, 
			PointF physicalMax, 
			out object labelOffset,
			out object boundingBox )
		{

			PointF tLabelOffset;
			RectangleF tBoundingBox;

			this.InitOffsetAndBounds( out labelOffset, out boundingBox );

			for (int i=0; i<labels_.Count; ++i)
			{
				if ((double)numbers_[i] > WorldMin && (double)numbers_[i] < WorldMax)
				{

					this.DrawTick( g, (double)numbers_[i], this.LargeTickSize, 
						(string)labels_[i],
						new Point(0,0), 
						physicalMin, physicalMax,
						out tLabelOffset, out tBoundingBox );
					
					this.UpdateOffsetAndBounds( ref labelOffset, ref boundingBox, 
						tLabelOffset, tBoundingBox );

				}
			}

		}

		#region LargeTickPositions
		/// <summary>
		/// The ArrayList containing the positions of the large ticks.
		/// </summary>
		public override ArrayList LargeTickPositions
		{
			get
			{
				ArrayList toRet = new ArrayList();
				for (int i=0; i<labels_.Count; ++i)
				{
					if ((double)numbers_[i] > WorldMin && (double)numbers_[i] < WorldMax)
					{
						toRet.Add( numbers_[i] );
					}
				}

				return toRet;
			}
		}
		#endregion

		private ArrayList labels_;
		private ArrayList numbers_;
	}
}
