/*
ScPl - A plotting library for .NET

Legend.cs
Copyright (C) 2003
Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: Legend.cs,v 1.17 2004/05/13 12:19:07 mhowlett Exp $

*/

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Collections;

namespace scpl
{

	/// <summary>
	/// Legend functionality specific to drawing legends for a PlotSurface2D.
	/// </summary>
	public class Legend : LegendBase
	{
		#region enum Placement
		/// <summary>
		/// The types of legend placements (enum).
		/// </summary>
		public enum Placement
		{
			/// <summary>
			/// Inside the plot area.
			/// </summary>
			Inside = 0,
			/// <summary>
			/// Outside the plot area.
			/// </summary>
			Outside = 1
		}
		#endregion

		#region private members
		private float xOffset_;
		private float yOffset_;
		private PlotSurface2D.XAxisPosition xAttach_;
		private PlotSurface2D.YAxisPosition yAttach_;
		private Placement horizontalEdgePlacement_;
		private Placement verticalEdgePlacement_;
		private bool neverShiftAxes_;
		#endregion

		#region get/set NeverShiftAxes
		public bool NeverShiftAxes
		{
			get
			{
				return neverShiftAxes_;
			}
			set
			{
				neverShiftAxes_ = value;
			}
		}
		#endregion
		#region get/set XOffset
		/// <summary>
		/// Horizontal offset for the legend.
		/// </summary>
		public float XOffset
		{
			get
			{
				return xOffset_;
			}
			set
			{
				xOffset_ = value;
			}
		}
		#endregion
		#region get/set YOffset
		/// <summary>
		/// Vertical offset for the legend.
		/// </summary>
		public float YOffset
		{
			get
			{
				return yOffset_;
			}
			set
			{
				yOffset_ = value;
			}
		}
		#endregion
		#region get/set VerticalEdgePlacement
		/// <summary>
		/// Defines the vertical placement of the legend (inside or outside).
		/// </summary>
		public Legend.Placement VerticalEdgePlacement
		{
			get
			{
				return verticalEdgePlacement_;
			}
			set
			{
				verticalEdgePlacement_ = value;
			}
		}
		#endregion
		#region get/set HorizontalEdgePlacement
		/// <summary>
		/// Defines the horizontal placement of the legend (inside or outside).
		/// </summary>
		public Legend.Placement HorizontalEdgePlacement
		{
			get
			{
				return horizontalEdgePlacement_;
			}
			set
			{
				horizontalEdgePlacement_ = value;
			}
		}
		#endregion
		#region get/set AttachTo
		/// <summary>
		/// Attach the legend to the specified axes.
		/// </summary>
		/// <param name="xa">The position of the X axis.</param>
		/// <param name="ya">The position of the Y axis.</param>
		public void AttachTo( PlotSurface2D.XAxisPosition xa, PlotSurface2D.YAxisPosition ya )
		{
			xAttach_ = xa;
			yAttach_ = ya; 
		}
		#endregion

		#region Constructor
		public Legend()
		{
			xAttach_ = PlotSurface2D.XAxisPosition.Top;
			yAttach_ = PlotSurface2D.YAxisPosition.Right;
			xOffset_ = 10.0f;
			yOffset_ = 1.0f;
			verticalEdgePlacement_ = Placement.Outside;
			horizontalEdgePlacement_ = Placement.Inside;
			neverShiftAxes_ = false;
		}
		#endregion

		#region method UpdateAxesPositions
		public void UpdateAxesPositions( 
			PhysicalAxis pXAxis1,
			PhysicalAxis pYAxis1,
			PhysicalAxis pXAxis2,
			PhysicalAxis pYAxis2,
			ArrayList plots,
			float scale, float padding, Rectangle bounds,
			out float lXPos, out float lYPos )
		{
		
			float leftIndent = 0.0f;
			float rightIndent = 0.0f;
			float bottomIndent = 0.0f;
			float topIndent = 0.0f;

			// now determine if legend should change any of these (legend should be fully 
			// visible at all times), and draw legend.

			RectangleF legendWidthHeight = this.GetBoundingBox( 0, 0, plots, scale );

			// (1) calculate legend position.

			// y

			lYPos = this.yOffset_;
			
			if ( this.xAttach_ == PlotSurface2D.XAxisPosition.Bottom )
			{
				lYPos += pYAxis1.PhysicalMin.Y;
				if ( this.horizontalEdgePlacement_ == Legend.Placement.Inside )
				{
					lYPos -= legendWidthHeight.Height;
				}
			}
			else
			{
				lYPos += pYAxis1.PhysicalMax.Y;
				if ( this.horizontalEdgePlacement_ == Legend.Placement.Outside )
				{
					lYPos -= legendWidthHeight.Height;
				}
			}
	
			// x

			lXPos = this.xOffset_;
		
			if ( this.yAttach_ == PlotSurface2D.YAxisPosition.Left )
			{
				if ( this.verticalEdgePlacement_ == Legend.Placement.Outside ) 
				{
					lXPos -= legendWidthHeight.Width;
				}
				lXPos += pXAxis1.PhysicalMin.X;
			}
			else
			{
				if ( this.verticalEdgePlacement_ == Legend.Placement.Inside )
				{
					lXPos -= legendWidthHeight.Width;
				}
				lXPos += pXAxis1.PhysicalMax.X;
			}


			// determine update amounts for axes

			if ( !this.neverShiftAxes_ )
			{

				if ( lXPos < padding )
				{
					float changeAmount = -lXPos + padding;
					// only allow axes to move away from bounds.
					if ( changeAmount > 0 )					
					{
						leftIndent = changeAmount;
					}
					lXPos += changeAmount;
				}

				if ( lXPos + legendWidthHeight.Width > bounds.Right - padding )
				{
					float changeAmount = (lXPos - bounds.Right + legendWidthHeight.Width + padding );
					// only allow axes to move away from bounds.
					if ( changeAmount > 0.0f ) 
					{
						rightIndent = changeAmount;
					}
					lXPos -= changeAmount;
				}

				if ( lYPos < padding )
				{
					float changeAmount = -lYPos + padding;
					// only allow axes to move away from bounds.
					if ( changeAmount > 0.0f )					
					{
						topIndent = changeAmount;
					}
					lYPos += changeAmount;
				}

				if ( lYPos + legendWidthHeight.Height > bounds.Bottom - padding )
				{
					float changeAmount = (lYPos - bounds.Bottom + legendWidthHeight.Height + padding );
					// only allow axes to move away from bounds.
					if ( changeAmount > 0.0f ) 
					{
						bottomIndent = changeAmount;
					}
					lYPos -= changeAmount;
				}

				// update axes.

				pXAxis1.PhysicalMin = new PointF( pXAxis1.PhysicalMin.X + leftIndent, pXAxis1.PhysicalMin.Y - bottomIndent );
				pXAxis1.PhysicalMax = new PointF( pXAxis1.PhysicalMax.X - rightIndent, pXAxis1.PhysicalMax.Y - bottomIndent );
				pYAxis1.PhysicalMin = new PointF( pYAxis1.PhysicalMin.X + leftIndent, pYAxis1.PhysicalMin.Y - bottomIndent );
				pYAxis1.PhysicalMax = new PointF( pYAxis1.PhysicalMax.X + leftIndent, pYAxis1.PhysicalMax.Y + topIndent );

				pXAxis2.PhysicalMin = new PointF( pXAxis2.PhysicalMin.X + leftIndent, pXAxis2.PhysicalMin.Y + topIndent );
				pXAxis2.PhysicalMax = new PointF( pXAxis2.PhysicalMax.X - rightIndent, pXAxis2.PhysicalMax.Y + topIndent );
				pYAxis2.PhysicalMin = new PointF( pYAxis2.PhysicalMin.X - rightIndent, pYAxis2.PhysicalMin.Y - bottomIndent );
				pYAxis2.PhysicalMax = new PointF( pYAxis2.PhysicalMax.X - rightIndent, pYAxis2.PhysicalMax.Y + topIndent );
			
			}

		}
		#endregion
	}

}
