/*
ScPl - A plotting library for .NET

MainForm.cs
Copyright (C) 2003
Paolo Pierini, Matt Howlett

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: Marker.cs,v 1.9 2004/03/23 14:13:52 pquadro Exp $

*/
using System;
using System.Drawing;
using System.Drawing.Drawing2D;

namespace scpl
{
	/// <summary>
	/// Enumeration for the Marker Types
	/// </summary>
	public enum MarkerType
	{
		/// <summary>
		/// A simple cross marker (x).
		/// </summary>
		Cross1,
		/// <summary>
		/// Another simple cross marker (+).
		/// </summary>
		Cross2,
		/// <summary>
		/// A circle marker.
		/// </summary>
		Circle,
		/// <summary>
		/// A square marker.
		/// </summary>
		Square,
		/// <summary>
		/// A triangle marker (upwards).
		/// </summary>
        Triangle,
		/// <summary>
		/// A triangle marker (upwards).
		/// </summary>
		TriangleUp,
		/// <summary>
		/// A triangle marker (upwards).
		/// </summary>
		TriangleDown,
		/// <summary>
		/// A diamond,
		/// </summary>
		Diamond,
		/// <summary>
		/// A filled circle
		/// </summary>
		FilledCircle,
		/// <summary>
		/// A filled square
		/// </summary>
		FilledSquare,
		/// <summary>
		/// A filled triangle
		/// </summary>
		FilledTriangle,
		/// <summary>
		/// A small flag (up)
		/// </summary>
		Flag,
		/// <summary>
		/// A small flag (up)
		/// </summary>
		FlagUp,
		/// <summary>
		/// A small flag (down)
		/// </summary>
		FlagDown,
		/// <summary>
		/// No marker
		/// </summary>
		None
	}
	/// <summary>
	/// Summary description for Marker.
	/// </summary>
	public class Marker
	{

		// Private Members of the Marker Class
		private MarkerType markerType_;
		private int size_;
		private int h_;
		private System.Drawing.Color color_;
		private System.Drawing.Pen pen_;
		private bool filled_ = false;
		private bool dropLine_ = false;

		#region get/set properties
		/// <summary>
		/// The Marker Type
		/// </summary>
		public MarkerType Type
		{
			get
			{
				return markerType_;
			}
			set
			{
				markerType_=value;
			}
		}
		/// <summary>
		/// Boolean flag to draw a dropline.
		/// </summary>
		public bool DropLine
		{
			get
			{
				return dropLine_;
			}
			set
			{
				dropLine_ = value;
			}
		}
		/// <summary>
		/// The marker size.
		/// </summary>
		public int Size
		{
			get
			{
				return size_;
			}
			set
			{
				size_=value;
				h_=size_/2;
			}
		}

		/// <summary>
		/// Fill with color.
		/// </summary>
		public bool Filled
		{
			get
			{
				return filled_;
			}
			set
			{
				filled_ = value;
			}
		}

		/// <summary>
		/// The marker color.
		/// </summary>
		public System.Drawing.Color Color
		{
			set
			{
				color_ = value;
			}
			get
			{
				return color_;
			}
		}

		/// <summary>
		/// The Pen used to draw the marker.
		/// </summary>
		public System.Drawing.Pen Pen
		{
			set
			{
				pen_ = value;
			}
			get
			{
				if (pen_== null)
				{
					pen_ = new Pen(color_);
				}
				return pen_;
			}
		}
		#endregion

		#region Constructors
		/// <summary>
		/// Parameterless constructor.
		/// </summary>
		public Marker()
		{
			markerType_=MarkerType.Square;
			Size=4;
			color_=Color.Black;
			pen_=new Pen(Color.Black);
			filled_ = false;
		}

		/// <summary>
		/// Constructor, declaring the marker type.
		/// </summary>
		/// <param name="markertype">The marker type.</param>
		public Marker(MarkerType markertype)
		{
			markerType_=markertype;
			Size=4;
			color_=Color.Black;
			pen_=new Pen(Color.Black);
			filled_ = false;
		}

		/// <summary>
		/// Constructor, declaring the marker type and size.
		/// </summary>
		/// <param name="markertype">The marker type.</param>
		/// <param name="size">The marker size.</param>
		public Marker(MarkerType markertype,int size)
		{
			markerType_=markertype;
			Size=size;
			color_=Color.Black;
			pen_=new Pen(Color.Black);
			filled_ = false;
		}

		/// <summary>
		/// Constructor, declaring the marker type, size and color.
		/// </summary>
		/// <param name="markertype">The marker type.</param>
		/// <param name="size">The marker size.</param>
		/// <param name="color">The marker color.</param>
		public Marker(MarkerType markertype,int size,Color color)
		{
			markerType_=markertype;
			Size=size;
			color_=color;
			pen_=new Pen(color);
			filled_ = false;
		}

		/// <summary>
		/// Constructor, declaring the marker type, size and Pen.
		/// </summary>
		/// <param name="markertype">The marker type.</param>
		/// <param name="size">The marker size.</param>
		/// <param name="pen">The marker Pen.</param>
		public Marker(MarkerType markertype,int size,Pen pen)
		{
			markerType_=markertype;
			Size=size;
			pen_=pen;
			color_=pen.Color;
			filled_ = false;
		}
		/// <summary>
		/// Constructor, declaring the marker type, size and Pen.
		/// </summary>
		/// <param name="markertype">The marker type.</param>
		/// <param name="size">The marker size.</param>
		/// <param name="pen">The marker Pen.</param>
		/// <param name="fill">The fill flag.</param>
		public Marker(MarkerType markertype,int size,Pen pen, bool fill)
		{
			markerType_=markertype;
			Size=size;
			pen_=pen;
			color_=pen.Color;
			filled_ = fill;
		}

		#endregion

		#region Draw
		/// <summary>
		/// Draws the marker at the specified position.
		/// </summary>
		/// <param name="g">The Graphics surface.</param>
		/// <param name="x">The X position (physical).</param>
		/// <param name="y">The Y position (physical).</param>
		internal void Draw( Graphics g, int x, int y)
		{
			bool needToDisposePen = false;
			Pen p;
			if (this.pen_ != null)
			{
				p = this.pen_;
			}
			else
			{
				p =	new	Pen(this.color_);
				needToDisposePen = true;
			}

			// Instance a new SolidBrush for filling
			SolidBrush b = new SolidBrush(p.Color);
			switch(markerType_)
			{
				case MarkerType.Cross1:
					g.DrawLine(p,x-h_,y+h_,x+h_,y-h_);
					g.DrawLine(p,x+h_,y+h_,x-h_,y-h_);
					break;
				case MarkerType.Cross2:
					g.DrawLine(p,x,y-h_,x,y+h_);
					g.DrawLine(p,x-h_,y,x+h_,y);
					break;
				case MarkerType.Circle:
					g.DrawEllipse(p,x-h_,y-h_,size_,size_);
					if ( this.filled_ ) g.FillEllipse(b,x-h_,y-h_,size_,size_);
					break;
				case MarkerType.Square:
					g.DrawRectangle(p,x-h_,y-h_,size_,size_);
					if ( this.filled_ ) g.FillRectangle(b,x-h_,y-h_,size_,size_);
					break;
				case MarkerType.Triangle:
				case MarkerType.TriangleDown:
				{
					Point p1 = new Point(x-h_,y-h_);
					Point p2 = new Point(x,y+h_);
					Point p3 = new Point(x+h_,y-h_);
					Point [] pts = new Point [3] {p1,p2,p3};
					GraphicsPath gp = new GraphicsPath();
					gp.AddPolygon(pts);
					g.DrawPath(p,gp);
					if ( this.filled_ ) g.FillPath(b,gp);
					break;
				}
				case MarkerType.TriangleUp:
				{
					Point p1 = new Point(x-h_,y+h_);
					Point p2 = new Point(x,y-h_);
					Point p3 = new Point(x+h_,y+h_);
					Point [] pts = new Point [3] {p1,p2,p3};
					GraphicsPath gp = new GraphicsPath();
					gp.AddPolygon(pts);
					g.DrawPath(p,gp);
					if ( this.filled_ ) g.FillPath(b,gp);
					break;
				}
				case MarkerType.FilledCircle:
					g.DrawEllipse(p,x-h_,y-h_,size_,size_);
					g.FillEllipse(b,x-h_,y-h_,size_,size_);
					break;
				case MarkerType.FilledSquare:
					g.DrawRectangle(p,x-h_,y-h_,size_,size_);
					g.FillRectangle(b,x-h_,y-h_,size_,size_);
					break;
				case MarkerType.FilledTriangle:
				{
					Point p1 = new Point(x-h_,y-h_);
					Point p2 = new Point(x,y+h_);
					Point p3 = new Point(x+h_,y-h_);
					Point [] pts = new Point [3] {p1,p2,p3};
					GraphicsPath gp = new GraphicsPath();
					gp.AddPolygon(pts);
					g.DrawPath(p,gp);
					g.FillPath(b,gp);
					break;
				}
				case MarkerType.Diamond:
				{
					Point p1 = new Point(x-h_,y);
					Point p2 = new Point(x,y-h_);
					Point p3 = new Point(x+h_,y);
					Point p4 = new Point(x,y+h_);
					Point [] pts = new Point [4] {p1,p2,p3,p4};
					GraphicsPath gp = new GraphicsPath();
					gp.AddPolygon(pts);
					g.DrawPath(p,gp);
					if( this.filled_ ) g.FillPath(b,gp);
					break;
				}
				case MarkerType.Flag:
				case MarkerType.FlagUp:
				{
					Point p1 = new Point(x,y);
					Point p2 = new Point(x,y-size_);
					Point p3 = new Point(x+size_,y-size_+size_/3);
					Point p4 = new Point(x,y-size_+2*size_/3);
					g.DrawLine(p,p1,p2);
					Point [] pts = new Point [3] {p2,p3,p4};
					GraphicsPath gp = new GraphicsPath();
					gp.AddPolygon(pts);
					g.DrawPath(p,gp);
					if( this.filled_ ) g.FillPath(b,gp);
					break;
				}
				case MarkerType.FlagDown:
				{
					Point p1 = new Point(x,y);
					Point p2 = new Point(x,y+size_);
					Point p3 = new Point(x+size_,y+size_-size_/3);
					Point p4 = new Point(x,y+size_-2*size_/3);
					g.DrawLine(p,p1,p2);
					Point [] pts = new Point [3] {p2,p3,p4};
					GraphicsPath gp = new GraphicsPath();
					gp.AddPolygon(pts);
					g.DrawPath(p,gp);
					if( this.filled_ ) g.FillPath(b,gp);
					break;
				}
				case MarkerType.None:
					break;
			}
			if (needToDisposePen) p.Dispose();
			b.Dispose();
		}
		#endregion

	}
}
