/*
ScPl - A plotting library for .NET

Windows.PlotSurface2d.cs
Copyright (C) 2003 
Matt Howlett, Paolo Pierini

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions
are met:

1. Redistributions of source code must retain the above copyright
   notice, this list of conditions and the following disclaimer.
   
2. Redistributions in binary form must reproduce the following text in 
   the documentation and / or other materials provided with the 
   distribution: 
   
   "This product includes software developed as part of 
   the ScPl plotting library project available from: 
   http://www.netcontrols.org/scpl/" 

------------------------------------------------------------------------

THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

$Id: Windows.PlotSurface2D.cs,v 1.38 2004/05/13 12:19:07 mhowlett Exp $

*/

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Windows.Forms;

namespace scpl
{
	namespace Windows 
	{
		/// <summary>
		/// PlotSurface2D Windows Control.
		/// </summary>
		[
		ToolboxBitmapAttribute(typeof(scpl.Windows.PlotSurface2D),
		"Windows.PlotSurface2D.ico")
		]
		public class PlotSurface2D : System.Windows.Forms.UserControl, IPlotSurface2D 
		{
			private System.ComponentModel.IContainer components;
			private System.Windows.Forms.ToolTip showCoordinates_;
			private System.Windows.Forms.ContextMenu rightMenu_;
			private System.Windows.Forms.MenuItem mnuZoomBack;
			private System.Windows.Forms.MenuItem mnuDisplayCoordinates;

			private System.Collections.ArrayList selectedObjects_;
			private bool allowSelection_ = false;
			private scpl.PlotSurface2D ps_;

			private bool mouseActionInitiated_ = false;
			private Point startPoint_ = new Point(-1,-1);
			private Point endPoint_ = new Point(-1,-1);
			// this is the condition for an unset point
			private Point unset_ = new Point(-1,-1);
			private Axis xAxis1Cache_;
			private Axis yAxis1Cache_;
			private Axis xAxis2Cache_;
			private System.Windows.Forms.MenuItem mnuSeparator1;
			private System.Windows.Forms.MenuItem mnuProperties;
			private System.Windows.Forms.PropertyGrid properties;
			private Axis yAxis2Cache_;

			#region Properties for mouse interaction functionality.
			/// <summary>
			/// Flag to allow the control to respond to mouse events.
			/// </summary>
			[
			Category("Plotter"),
			Description("Allows the scpl control to respond to mouse events."),
			Browsable(true)
			]
			public bool AllowSelection
			{
				get
				{
					return allowSelection_;
				}
				set
				{
					allowSelection_ = value;
					// this is probably not needed, but
					// it's probably a good practice to redraw 
					// the control when a property changed
					this.Invalidate();
				}
			}
			/// <summary>
			/// Flag to display a coordinates in a tooltip.
			/// </summary>
			[
			Category("Plotter"),
			Description("Displays the world coordinates at the mouse position."),
			Browsable(true)
			]
			public bool ShowCoordinates
			{
				get
				{
					return this.showCoordinates_.Active;
				}
				set
				{
					this.showCoordinates_.Active = value;
				}
			}
			#endregion

			#region Constructor
			/// <summary>
			/// Parameterless constructor.
			/// </summary>
			public PlotSurface2D()
			{
				// This call is required by the Windows.Forms Form Designer.
				InitializeComponent();

				// double buffer, and update when resize.
				base.SetStyle(ControlStyles.AllPaintingInWmPaint, true);
				base.SetStyle(ControlStyles.DoubleBuffer, true);
				base.SetStyle(ControlStyles.UserPaint, true);
				base.ResizeRedraw = true;

				ps_ = new scpl.PlotSurface2D();

			}
			#endregion

			#region Dispose
			/// <summary> 
			/// Clean up any resources being used.
			/// </summary>
			protected override void Dispose( bool disposing )
			{
				if( disposing )
				{
					if(components != null)
					{
						components.Dispose();
					}
				}
				base.Dispose( disposing );
			}
			#endregion

			#region Component Designer generated code
			/// <summary> 
			/// Required method for Designer support - do not modify 
			/// the contents of this method with the code editor.
			/// </summary>
			private void InitializeComponent()
			{
				this.components = new System.ComponentModel.Container();
				this.showCoordinates_ = new System.Windows.Forms.ToolTip(this.components);
				this.rightMenu_ = new System.Windows.Forms.ContextMenu();
				this.mnuZoomBack = new System.Windows.Forms.MenuItem();
				this.mnuDisplayCoordinates = new System.Windows.Forms.MenuItem();
				this.mnuSeparator1 = new System.Windows.Forms.MenuItem();
				this.mnuProperties = new System.Windows.Forms.MenuItem();
				this.properties = new System.Windows.Forms.PropertyGrid();
				this.SuspendLayout();
				// 
				// showCoordinates_
				// 
				this.showCoordinates_.ShowAlways = true;
				// 
				// rightMenu_
				// 
				this.rightMenu_.MenuItems.AddRange(new System.Windows.Forms.MenuItem[] {
																						   this.mnuZoomBack,
																						   this.mnuDisplayCoordinates,
																						   this.mnuSeparator1,
																						   this.mnuProperties});
				this.rightMenu_.Popup += new System.EventHandler(this.rightMenu__Popup);
				// 
				// mnuZoomBack
				// 
				this.mnuZoomBack.Index = 0;
				this.mnuZoomBack.Text = "Zoom Back";
				this.mnuZoomBack.Click += new System.EventHandler(this.mnuZoomBack_Click);
				// 
				// mnuDisplayCoordinates
				// 
				this.mnuDisplayCoordinates.Index = 1;
				this.mnuDisplayCoordinates.Text = "Show World Coordinates";
				this.mnuDisplayCoordinates.Click += new System.EventHandler(this.mnuDisplayCoordinates_Click);
				// 
				// mnuSeparator1
				// 
				this.mnuSeparator1.Index = 2;
				this.mnuSeparator1.Text = "-";
				// 
				// mnuProperties
				// 
				this.mnuProperties.Index = 3;
				this.mnuProperties.Text = "Properties";
				this.mnuProperties.Click += new System.EventHandler(this.mnuProperties_Click);
				// 
				// properties
				// 
				this.properties.CommandsBackColor = System.Drawing.SystemColors.ControlLightLight;
				this.properties.CommandsVisibleIfAvailable = true;
				this.properties.Enabled = false;
				this.properties.LargeButtons = false;
				this.properties.LineColor = System.Drawing.SystemColors.ScrollBar;
				this.properties.Location = new System.Drawing.Point(72, 24);
				this.properties.Name = "properties";
				this.properties.Size = new System.Drawing.Size(216, 216);
				this.properties.TabIndex = 0;
				this.properties.Text = "propertyGrid1";
				this.properties.ViewBackColor = System.Drawing.SystemColors.Window;
				this.properties.ViewForeColor = System.Drawing.SystemColors.WindowText;
				this.properties.Visible = false;
				// 
				// PlotSurface2D
				// 
				this.BackColor = System.Drawing.SystemColors.ControlLightLight;
				this.Controls.AddRange(new System.Windows.Forms.Control[] {
																			  this.properties});
				this.Name = "PlotSurface2D";
				this.Size = new System.Drawing.Size(328, 272);
				this.ResumeLayout(false);

			}
			#endregion

			#region OnPaint override
			/// <summary>
			/// Override for the OnPaint method.
			/// </summary>
			/// <param name="pe">The Paint event arguments.</param>
			protected override void OnPaint( PaintEventArgs pe )
			{
				Graphics g = pe.Graphics;
				
				Rectangle border = new Rectangle( 0, 0, this.Width, this.Height );
				
				if ( g == null ) 
				{
					throw (new Exception("null graphics context!"));
				}
				
				if ( ps_ == null )
				{
					throw (new Exception("null scpl.PlotSurface2D context!"));
				}
				
				if ( border == Rectangle.Empty )
				{
					throw (new Exception("null border context!"));
				}

				this.Draw( g, border );

				base.OnPaint(pe);
			}
			#endregion

			#region Draw
			/// <summary>
			/// Renders the plot.
			/// </summary>
			/// <param name="g">The Graphics surface.</param>
			/// <param name="bounds">The rectangle storing the bounds for rendering.</param>
			public void Draw( Graphics g, Rectangle bounds )
			{
				ps_.Draw( g, bounds );
			}
			#endregion

			#region Clear
			/// <summary>
			/// Clears the plot and resets to default values.
			/// </summary>
			public void Clear()
			{
				xAxis1Cache_ = null;
				yAxis1Cache_ = null;
				xAxis2Cache_ = null;
				yAxis2Cache_ = null;
				ps_.Clear();
			}
			#endregion

			#region Add methods
			#region Add(IDrawable p) 
			/// <summary>
			/// Adds a trace plot to the plot surface.
			/// </summary>
			/// <param name="p">The trace plot to add.</param>
			public void Add( IDrawable p )
			{
				ps_.Add( p );
			}
			#endregion
			#region Add( IDrawable, XAxisPosition, YAxisPosition ) 
			/// <summary>
			/// Adds a trace plot to the plot surface.
			/// </summary>
			/// <param name="p">The trace plot to add.</param>
			/// <param name="xp">The position of the X axis.</param>
			/// <param name="yp">The position of the Y axis.</param>
			public void Add( IDrawable p, scpl.PlotSurface2D.XAxisPosition xp, scpl.PlotSurface2D.YAxisPosition yp )
			{
				ps_.Add( p, xp, yp );
			}
			#endregion
			#endregion

			#region Property accessors

			#region get/set Legend
			public scpl.Legend Legend
			{
				get
				{
					return ps_.Legend;
				}
				set
				{
					ps_.Legend = value;
				}
			}
			#endregion

			#region get/set Title 
			/// <summary>
			/// The plot title.
			/// </summary>
			[
			Category("Plotter"),
			Description("Set the plot title."),
			Browsable(true)
			]
			public string Title
			{
				get 
				{
					return ps_.Title;
				}
				set 
				{
					ps_.Title = value;
					//helpful in design view. But crap in applications!
					//this.Refresh();
				}
			}
			#endregion
			#region get/set TitleFont 
			/// <summary>
			/// The plot title font.
			/// </summary>
			[
			Category("Plotter"),
			Description("Set the plot title font."),
			Browsable(true)
			]
			public Font TitleFont 
			{
				get 
				{
					return ps_.TitleFont;
				}
				set 
				{
					ps_.TitleFont = value;
				}
			}
			#endregion
			#region get/set Padding 
			/// <summary>
			/// The padding distance.
			/// </summary>
			///
			[
			Category("Plotter"),
			Description("The padding (margins)."),
			Browsable(true)
			]
			public int Padding
			{
				get
				{
					return ps_.Padding;
				}
				set
				{
					ps_.Padding = value;
				}
			}
			#endregion
			#region get Axes
			#region get/set XAxis1 
			/// <summary>
			/// The first abscissa axis.
			/// </summary>
			/// 
			[
			Browsable(false)
			]
			public Axis XAxis1
			{
				get
				{
					return ps_.XAxis1;
				}
				set
				{
					ps_.XAxis1 = value;
				}
			}
			#endregion
			#region get/set YAxis1 
			/// <summary>
			/// The first ordinate axis.
			/// </summary>
			[
			Browsable(false)
			]
			public Axis YAxis1
			{
				get
				{
					return ps_.YAxis1;
				}
				set
				{
					ps_.YAxis1 = value;
				}
			}
			#endregion
			#region get/set XAxis2 
			/// <summary>
			/// The second abscissa axis.
			/// </summary>
			[
			Browsable(false)
			]
			public Axis XAxis2
			{
				get
				{
					return ps_.XAxis2;
				}
				set
				{
					ps_.XAxis2 = value;
				}
			}
			#endregion
			#region get/set YAxis2 
			/// <summary>
			/// The second ordinate axis.
			/// </summary>
			[
			Browsable(false)
			]
			public Axis YAxis2
			{
				get
				{
					return ps_.YAxis2;
				}
				set
				{
					ps_.YAxis2 = value;
				}
			}
			#endregion
			#endregion
			#region get physical Axis caches
			#region get PhysicalXAxis1Cache
			/// <summary>
			/// Returns the physical information of Xaxis1.
			/// </summary>
			[
			Browsable(false)
			]
			public PhysicalAxis PhysicalXAxis1Cache
			{
				get
				{
					return ps_.PhysicalXAxis1Cache;
				}
			}
			#endregion
			#region get PhysicalYAxis1Cache
			/// <summary>
			/// Returns the physical information of Yaxis1.
			/// </summary>
			[
			Browsable(false)
			]
			public PhysicalAxis PhysicalYAxis1Cache
			{
				get
				{
					return ps_.PhysicalYAxis1Cache;
				}
			}
			#endregion
			#region get PhysicalXAxis2Cache
			/// <summary>
			/// Returns the physical information of Xaxis2.
			/// </summary>
			[
			Browsable(false)
			]
			public PhysicalAxis PhysicalXAxis2Cache
			{
				get
				{
					return ps_.PhysicalXAxis2Cache;
				}
			}
			#endregion
			#region get PhysicalYAxis2Cache
			/// <summary>
			/// Returns the physical information of Yaxis2.
			/// </summary>
			[
			Browsable(false)
			]
			public PhysicalAxis PhysicalYAxis2Cache
			{
				get
				{
					return ps_.PhysicalYAxis2Cache;
				}
			}
			#endregion
			#endregion
			#region get/set PlotBackColor
			/// <summary>
			/// The plot background color.
			/// </summary>
			[
			Category("Plotter"),
			Description("Set the plot background color."),
			Browsable(true)
			]
			public System.Drawing.Color PlotBackColor
			{
				get
				{
					return ps_.PlotBackColor;
				}
				set
				{
					ps_.PlotBackColor = value;
				}
			}
			#endregion
			#region get/set SmoothingMode
			public System.Drawing.Drawing2D.SmoothingMode SmoothingMode 
			{ 
				get
				{
					return ps_.SmoothingMode;
				}
				set
				{
					ps_.SmoothingMode = value;
				}
			}
			#endregion
			#endregion

			#region method OnMouseWheel
			protected override void OnMouseWheel(MouseEventArgs e)
			{
				int i=0;
			}
			#endregion

			// Mouse interaction
			#region Mouse Events routines
			/// <summary>
			/// Override method for the MouseDown event.
			/// </summary>
			/// <param name="e">The event arguments.</param>
			protected override void OnMouseDown(MouseEventArgs e)
			{

				if (e.Button==MouseButtons.Left && allowSelection_)
				{
					// keep track of the start point and signal that we
					// have initiated a mouse action
					mouseActionInitiated_ = true;
					startPoint_.X = e.X;
					startPoint_.Y = e.Y;
					// it's a good idea to invalidate the end point
					endPoint_.X = -1;
					endPoint_.Y = -1;
				}
				// don't fail to call the base method!
				base.OnMouseDown(e);
			}
			/// <summary>
			/// Override method for the MouseMove event.
			/// </summary>
			/// <param name="e">The event arguments.</param>
			protected override void OnMouseMove(MouseEventArgs e)
			{
				// we do this only if we can select AND we
				// have initiated a mouse action (pressing the button)
				if ( e.Button==MouseButtons.Left && allowSelection_ && mouseActionInitiated_ )
				{
					// we are here
					Point here = new Point( e.X, e.Y );
					// ok, we can delete the previous box
					if ( endPoint_ != unset_ )
					{
						this.DrawRubberBand( startPoint_, endPoint_ );
					}
					endPoint_ = here;
					// and redraw the last one
					this.DrawRubberBand( startPoint_, here );
					if ( ps_.PlotAreaBoundingBoxCache.Contains(here) )
					{
						showCoordinates_.ShowAlways = true;
						double x = this.PhysicalXAxis1Cache.PhysicalToWorld(here,true);
						double y = this.PhysicalYAxis1Cache.PhysicalToWorld(here,true);
						Console.WriteLine( "start: " + x.ToString() );
						string s="(" + x.ToString("g4") + "," + y.ToString("g4")+")"; 
						showCoordinates_.SetToolTip(this,s);
					}
					else
					{
						showCoordinates_.ShowAlways = false;
					}
				}
				else if ( showCoordinates_.Active )
				{
					// we are here
					Point here = new Point( e.X, e.Y );
					if ( ps_.PlotAreaBoundingBoxCache.Contains(here) )
					{
						showCoordinates_.ShowAlways = true;
						double x = this.PhysicalXAxis1Cache.PhysicalToWorld(here,true);
						double y = this.PhysicalYAxis1Cache.PhysicalToWorld(here,true);
						string s="(" + x.ToString("g4") + "," + y.ToString("g4")+")"; 
						showCoordinates_.SetToolTip(this,s);
					}
					else
					{
						showCoordinates_.ShowAlways = false;
					}
				}
				// don't fail to call the base method!
				base.OnMouseMove( e );
			}
			
			/*
			protected override void OnMouseWheel(MouseEventArgs e)
			{
				//e.Delta
			}
			*/

			/// <summary>
			/// Override method for the MouseUp event.
			/// </summary>
			/// <param name="e">The event arguments.</param>
			protected override void OnMouseUp(MouseEventArgs e)
			{
				if (e.Button==MouseButtons.Left && allowSelection_)
				{
					endPoint_.X=e.X;
					endPoint_.Y=e.Y;
					// terminate mouse action
					mouseActionInitiated_=false;
					if(endPoint_ != unset_)
					{
						this.DrawRubberBand(startPoint_,endPoint_);
					}
					
					Point minPoint = new Point( 0, 0 );
					minPoint.X = Math.Min( startPoint_.X, endPoint_.X );
					minPoint.Y = Math.Min( startPoint_.Y, endPoint_.Y );

					Point maxPoint = new Point( 0, 0 );
					maxPoint.X = Math.Max( startPoint_.X, endPoint_.X );
					maxPoint.Y = Math.Max( startPoint_.Y, endPoint_.Y );

					Rectangle r=this.ps_.PlotAreaBoundingBoxCache;
					if(minPoint != maxPoint && (r.Contains(minPoint) || r.Contains(maxPoint)))
					{
						if ( xAxis1Cache_ == null && xAxis2Cache_ == null &&
							 yAxis1Cache_ == null && yAxis2Cache_ == null )
						{
							if (this.XAxis1 != null)
							{
								xAxis1Cache_ = (Axis)this.XAxis1.Clone();
							}
							if (this.XAxis2 != null)
							{
								xAxis2Cache_ = (Axis)this.XAxis2.Clone();
							}
							if (this.YAxis1 != null)
							{
								yAxis1Cache_ = (Axis)this.YAxis1.Clone();
							}
							if (this.YAxis2 != null)
							{
								yAxis2Cache_ = (Axis)this.YAxis2.Clone();
							}
						}

						// middle wheel will be controlling zoom in future. left mouse may
						// be drag => stack zoom out not long term solution, so just make 
						// zoom out go right out at this stage.
						
						// the commented region below has been moved to a method of PhysicalAxis...
						// the calss below avoids that the user zooms in past the GDI+ 
						// rendering capabilities.
						this.PhysicalXAxis1Cache.SetWorldLimits(minPoint,maxPoint);
						this.PhysicalXAxis2Cache.SetWorldLimits(minPoint,maxPoint);
						this.PhysicalYAxis1Cache.SetWorldLimits(maxPoint,minPoint);
						this.PhysicalYAxis2Cache.SetWorldLimits(maxPoint,minPoint);
						#region Region commented out
//						if (XAxis1 != null)
//						{
//							x1minc = this.XAxis1.WorldMin;
//							x1maxc = this.XAxis1.WorldMax;
//							if ( !XAxis1.Reversed ) 
//							{
//								double tmp = 
//									this.PhysicalXAxis1Cache.PhysicalToWorld(minPoint,true);
//								this.XAxis1.WorldMax = 
//									this.PhysicalXAxis1Cache.PhysicalToWorld(maxPoint,true);
//								this.XAxis1.WorldMin = tmp;
//							}
//							else
//							{
//								double tmp = 
//									this.PhysicalXAxis1Cache.PhysicalToWorld(minPoint,true);
//								this.XAxis1.WorldMin =
//									this.PhysicalXAxis1Cache.PhysicalToWorld(maxPoint,true);
//								this.XAxis1.WorldMax = tmp;
//							}
//						}
//						if (XAxis2 != null)
//						{
//							x2minc = this.XAxis2.WorldMin;
//							x2maxc = this.XAxis2.WorldMax;
//							if ( !XAxis2.Reversed )
//							{
//								double tmp = 
//									this.PhysicalXAxis2Cache.PhysicalToWorld(minPoint,true);
//								this.XAxis2.WorldMax = 
//									this.PhysicalXAxis2Cache.PhysicalToWorld(maxPoint,true);
//								this.XAxis2.WorldMin = tmp; 
//							}
//							else
//							{
//								double tmp = 
//									this.PhysicalXAxis2Cache.PhysicalToWorld(minPoint,true);
//								this.XAxis2.WorldMin = 
//									this.PhysicalXAxis2Cache.PhysicalToWorld(maxPoint,true);
//								this.XAxis2.WorldMax = tmp;
//							}
//						}
//						if (YAxis1 != null)
//						{
//							y1minc = this.YAxis1.WorldMin;
//							y1maxc = this.YAxis1.WorldMax;
//							if ( !YAxis1.Reversed ) 
//							{
//								double tmp = 
//									this.PhysicalYAxis1Cache.PhysicalToWorld(maxPoint,true);
//								this.YAxis1.WorldMax = 
//									this.PhysicalYAxis1Cache.PhysicalToWorld(minPoint,true);
//								this.YAxis1.WorldMin = tmp;
//							}
//							else
//							{
//								double tmp =
//									this.PhysicalYAxis1Cache.PhysicalToWorld(maxPoint,true);
//								this.YAxis1.WorldMin =
//									this.PhysicalYAxis1Cache.PhysicalToWorld(minPoint,true);
//								this.YAxis1.WorldMax = tmp;
//							}
//						}
//						if (YAxis2 != null)
//						{
//							y2minc = this.YAxis2.WorldMin;
//							y2maxc = this.YAxis2.WorldMax;
//							if ( !YAxis2.Reversed ) 
//							{
//								double tmp = 
//									this.PhysicalYAxis2Cache.PhysicalToWorld(maxPoint,true);
//								this.YAxis2.WorldMax = 
//									this.PhysicalYAxis2Cache.PhysicalToWorld(minPoint,true);
//								this.YAxis2.WorldMin = tmp;
//							}
//							else
//							{
//								double tmp = 
//									this.PhysicalYAxis2Cache.PhysicalToWorld(maxPoint,true);
//								this.YAxis2.WorldMin = 
//									this.PhysicalYAxis2Cache.PhysicalToWorld(minPoint,true);
//								this.YAxis2.WorldMax = tmp;
//							}
//						}
						#endregion
						// reset the start/end points
						startPoint_ = unset_;
						endPoint_ = unset_;
						this.Refresh();
					}
				}
				else if (e.Button == MouseButtons.Right)
				{
					Point here = new Point( e.X, e.Y );
					selectedObjects_ = ps_.HitTest(here);
					rightMenu_.Show(this,here);
				}
				// don't fail to call the base method!
				base.OnMouseUp(e);
			}


			private void ZoomBack()
			{
				if ( xAxis1Cache_ != null )
				{
					this.XAxis1 = xAxis1Cache_;
					this.XAxis2 = xAxis2Cache_;
					this.YAxis1 = yAxis1Cache_;
					this.YAxis2 = yAxis2Cache_;

					xAxis1Cache_ = null;
					xAxis2Cache_ = null;
					yAxis1Cache_ = null;
					yAxis2Cache_ = null;
				}					
				this.Refresh();
			}

			private void DrawRubberBand(Point start, Point end)
			{
				Rectangle rect = new Rectangle();
				// the clipping rectangle in screen coordinates
				Rectangle clip = this.RectangleToScreen( ps_.PlotAreaBoundingBoxCache );
				// convert to screen coords
				start = PointToScreen( start );
				end = PointToScreen( end );
				// now, "normalize" the rectangle
				if ( start.X < end.X )
				{
					rect.X = start.X;
					rect.Width = end.X - start.X;
				}
				else
				{
					rect.X = end.X;
					rect.Width = start.X - end.X;
				}
				if ( start.Y < end.Y )
				{
					rect.Y = start.Y;
					rect.Height = end.Y - start.Y;
				}
				else
				{
					rect.Y = end.Y;
					rect.Height = start.Y - end.Y;
				}
				rect = Rectangle.Intersect( rect, clip );
				ControlPaint.DrawReversibleFrame( rect, Color.White, FrameStyle.Dashed );
			}
			#endregion

			#region Context menu
			//HACK: Disabled properties menu
			private void rightMenu__Popup(object sender, System.EventArgs e)
			{
				mnuDisplayCoordinates.Checked = this.ShowCoordinates;
				mnuZoomBack.Enabled = ( xAxis1Cache_ != null);
				bool haveSelection = ( selectedObjects_ != null && selectedObjects_.Count > 0); 
				mnuSeparator1.Visible = haveSelection;
				mnuProperties.Visible = haveSelection;
				if ( haveSelection )
				{
					System.Type t = (System.Type) selectedObjects_[0];
					object sel = selectedObjects_[1];
					if (sel != null)
					{
						mnuProperties.Enabled = true;
						mnuProperties.Visible = true;
						switch (t.ToString())
						{
							case "scpl.Axis":
								mnuProperties.Text="Axis Properties...";
								properties.SelectedObject = sel;
								break;
							case "scpl.Legend":
								mnuProperties.Text="PlotSurface2D Properties...";
								properties.SelectedObject = sel;
								break;
							case "scpl.PlotSurface2D":
                                mnuProperties.Text="PlotSurface2D Properties...";
								properties.SelectedObject = sel;
								break;
							default:
								mnuProperties.Enabled = false;
								mnuProperties.Visible = false;
								break;
						}
					}
					else
					{
						mnuProperties.Visible = false;
						mnuProperties.Enabled = false;
					}
					mnuProperties.Enabled = false; // HACK: delet this to enable properties...
				}
			}
			private void mnuZoomBack_Click(object sender, System.EventArgs e)
			{
				ZoomBack();
			}
			private void mnuDisplayCoordinates_Click(object sender, System.EventArgs e)
			{
				this.ShowCoordinates = !this.ShowCoordinates;
				mnuDisplayCoordinates.Checked = this.ShowCoordinates;
			}
			private void mnuProperties_Click(object sender, System.EventArgs e)
			{
				if (properties.SelectedObject != null)
				{
					this.properties.Visible = true;
					this.properties.Enabled = true;
					// this.properties.
					this.properties.Show();
				}
			}
			#endregion

			#region AddAxesConstraint
			public void AddAxesConstraint( AxesConstraint c )
			{
				ps_.AddAxesConstraint( c );
			}
			#endregion

		}
	}
}
