#include <iostream>
#include <stdexcept>
#include <string>
#include <stdio.h>
#include <vector>
#include <map>
#include <list>

using namespace std;

typedef unsigned int uint32;
typedef unsigned short uint16;
typedef unsigned __int64 uint64;

template <class T>
void ReadData( FILE * f, T& value )
{
	if (1 != fread( &value, sizeof(value), 1, f ))
		throw runtime_error("Unable to read value");
}

void ReadData( FILE * f, string& value )
{
	size_t length;
	ReadData( f, length );
	value.resize(length);
	if (1 != fread( const_cast<char*>(value.data()), length, 1, f ))
		throw runtime_error("Unable to read string");
}

template <class T>
void ReadData( FILE * f, vector<T>& value )
{
	size_t length;
	ReadData( f, length );
	for (size_t i=0; i<length; i++)
	{
		T temp;
		ReadData( f, temp );
		value.push_back(temp);
	}
}

struct SCoding
{
	uint64 nTot;
	uint32 nLow;
	uint32 nSym;
};

void ReadData( FILE * f, SCoding& coding )
{
	ReadData( f, coding.nTot );
	ReadData( f, coding.nLow );
	ReadData( f, coding.nSym );
}

struct SAnnotation
{
	size_t index;
	string annotation;
};

void ReadData( FILE * f, SAnnotation& ann )
{
	ReadData( f, ann.index );
	ReadData( f, ann.annotation );
}

struct SPacket
{
	uint32 uniqueId;
	bool isEncoding;
	vector<string> from;
	vector<string> to;
	vector<SCoding> coding;
	vector<SAnnotation> annotations;

	void Dump(bool encoding = true);
	template <class T>
	void DumpVec( const char * expl, const vector<T>& out )
	{
		cout << expl << ": ";
		switch (out.size())
		{
		case 0:
			cout << "<<unknown>>";
			break;
		case 1:
			cout << out.front();
			break;
		default:
			for (size_t i=0; i<out.size(); i++)
			{
				cout << endl << "  " << out[i];
			}
			break;
		}
		cout << endl;
	}
};

void SPacket::Dump(bool encoding)
{
	DumpVec( "from", from );
	DumpVec( "to", to );
	cout << "id: " << uniqueId << endl;
	cout << "encoding: " << (isEncoding? "yes" : "no") << endl;

	size_t j = 0;
	for (size_t i=0; i<coding.size(); i++)
	{
		while (j < annotations.size() && annotations[j].index <= i)
		{
			cout << annotations[j].annotation << ":" << endl;
			j ++;
		}
		if (encoding)
			cout << "  " << coding[i].nTot << " " << coding[i].nLow << " " << coding[i].nSym << endl;
	}
}

class CData
{
public:
	bool ReadPacket( FILE * f	);
	bool Verify();
	void DumpMessages(bool enc, bool dec, const char * machine);

	size_t NumPackets() const { return m_pkts.size(); }

private:
	typedef list<SPacket> PktList;
	PktList m_pkts;
	typedef list<SPacket*> PPktList;
	PPktList m_enc;
	PPktList m_dec;
};

template <class T>
bool VecContains( const vector<T>& a, const vector<T>& b )
{
	for (int i=0; i<a.size(); i++)
		for (int j=0; j<b.size(); j++)
			if (0 == stricmp(a[i].c_str(), b[j].c_str()))
				return true;
	return false;
}

bool VerifyEncoding( const SPacket& a, const SPacket& b )
{
	if (a.coding.size() != b.coding.size())
		return false;
	for (size_t i=0; i<a.coding.size(); i++)
	{
		if (a.coding[i].nTot != b.coding[i].nTot)
			return false;
		if (a.coding[i].nSym != b.coding[i].nSym)
			return false;
		if (a.coding[i].nLow != b.coding[i].nLow)
			return false;
	}
	return true;
}

bool CData::Verify()
{
	int notFound = 0;
	int butFound = 0;
	for (PPktList::iterator iter = m_enc.begin(); iter != m_enc.end(); ++iter)
	{
		SPacket * pPktEnc = *iter;
		bool found = false;
		for (PPktList::iterator iter2 = m_dec.begin(); iter2 != m_dec.end(); ++iter2)
		{
			SPacket * pPktDec = *iter2;
			if (pPktEnc->uniqueId == pPktDec->uniqueId && VecContains(pPktDec->from, pPktEnc->from) && VecContains(pPktDec->to, pPktEnc->to))
			{
				if (!VerifyEncoding(*pPktDec, *pPktEnc))
				{
					cout << "****************************************" << endl;
					pPktEnc->Dump();
					pPktDec->Dump();
				}
				found = true;
				break;
			}
		}
		butFound += found;
		notFound += !found;
	}

	cout << notFound << " encoded packets not found" << endl;
	cout << butFound << " encoded packets found" << endl;

	return true;
}

void CData::DumpMessages(bool enc, bool dec, const char * machine)
{
	if (enc)
	{
		for (PPktList::iterator iter = m_enc.begin(); iter != m_enc.end(); ++iter)
		{
			cout << "****************************************" << endl;
			SPacket * pPktEnc = *iter;
			bool include = true;
			if (machine)
			{
				include = false;
				for (vector<string>::const_iterator it = pPktEnc->to.begin(); it != pPktEnc->to.end(); ++it)
					include |= (0 == stricmp(it->c_str(), machine));
			}
			pPktEnc->Dump(false);
		}
	}
	if (dec)
	{
		for (PPktList::iterator iter = m_dec.begin(); iter != m_dec.end(); ++iter)
		{
			cout << "****************************************" << endl;
			SPacket * pPktEnc = *iter;
			bool include = true;
			if (machine)
			{
				include = false;
				for (vector<string>::const_iterator it = pPktEnc->from.begin(); it != pPktEnc->from.end(); ++it)
					include |= (0 == stricmp(it->c_str(), machine));
			}
			pPktEnc->Dump(false);
		}
	}
}

bool CData::ReadPacket( FILE * f )
{
	uint32 uniqueId;
	try
	{
		ReadData( f, uniqueId );
	}
	catch (...)
	{
		return false;
	}

	try
	{
		m_pkts.push_back(SPacket());
		SPacket& pkt = m_pkts.back();

		pkt.uniqueId = uniqueId;
		ReadData( f, pkt.isEncoding );
		ReadData( f, pkt.from );
		ReadData( f, pkt.to );
		ReadData( f, pkt.coding );
		ReadData( f, pkt.annotations );

		if (pkt.isEncoding)
			m_enc.push_back(&pkt);
		else
			m_dec.push_back(&pkt);

		//pkt.Dump();
	}
	catch (...)
	{
		m_pkts.pop_back();
		throw;
	}

	return true;
}

int main(int argc, const char ** argv)
{
	int r = 0;

	try
	{
		CData data;

		for (int i=1; i<argc; i++)
		{
			const char * filename = argv[i];
			FILE * f = fopen(filename, "rb");
			if (!f)
				throw runtime_error("Unable to open file: " + string(filename));

			cout << "reading " << filename << endl;

			try
			{
				while (data.ReadPacket(f))
					;
			}
			catch (exception& e)
			{
				cerr << e.what() << endl;
			}

			fclose(f);
		}

		cout << (int)data.NumPackets() << " packets read" << endl;

// here is the ui... choose which operation you need to perform
//		data.Verify();
		data.DumpMessages(true, true, "PC066.INTERN.CRYTEK.DE:64087");
	}
	catch (exception& e)
	{
		cerr << e.what() << endl;
		r = 1;
	}
	catch (...) 
	{
		cerr << "unknown exception" << endl;
		r = 2;
	}

	return r;
}
