/* 
	definitions for SPU Bubble mechanism
*/

#ifndef __SPU_BUBBLES_H
#define __SPU_BUBBLES_H
#pragma once

#if defined(PS3)

//do not change to anything but 4
#define scMaxSPUBubbleCount 4

#include <IJobManSPU.h>

namespace NSPU
{
	//bubble info each SPUDriver retrieves to initializes the bubble logic
	struct SBubbleDirInfo
	{
		uint32 bubbleDirEA;				//effective address of bubble directory
		uint32 bubbleNum;					//number of bubbles
		int32 globalVarBaseAddr;	//global variable base address
		uint32 ppuSyncEA;					//ea of m_SPUJobQueue, spin lock follows, required for SPU job spwaning (only in bubble mode)
	} _ALIGN(16);

	struct SBubbleInfo
	{
		uint32 ea;						//effective address of bubble
		uint32 size;					//transfer size of bubble
	};

#if defined(__SPU__)

	//do not change without adapting in MissHandler_spu.S
	#define RETURN_STACK_MAX_ENTRIES 32
	//entry of the return stack
	//16 byte to load and save quickly
	//written into via vec_ushort8, do not reorder 
	//g_ReturnStackTopIndex is the current offset to its top entry, always incremented by 16 byte == sizeof(SReturnStackEntry)
	struct SReturnStackEntry
	{
		uint16 pad0;					//was: uint16 bubbleTextOff;	//link register offset relative to bubble start
		uint16 bubbleID;			//id of bubble calling the cross bubble function
		uint16 bubbleSlot;		//slot the bubble is currently transferred to and calling from
		uint16 pad [5];
		static const uint32 cIsJobBubble = 0xFFFF;	//identifier for jobs, set into bubbleID as base stack entry
	} _ALIGN(16);

	//possible bubble states
	//do not change values, changed and accessed via asm in MissHandler_spu.S too
	#define BUB_STATE_INACTIVE 0		//initial state
	#define BUB_STATE_STREAMING 1		//bubble is streaming (transfer from main mem to LS in progress)
	#define BUB_STATE_READY 2				//bubble is ready to be used (DMA transfer has been finished)
	
	//bubble state stored for each bubble on SPU
	//do not change order, gets accessed in miss handler asm code
	struct SBubbleState
	{
		uint16 curIndex;				//index of bubble (serves as ID, 0...), duplicated in other directory for fast comp's
		uint16 curState;				//current state of bubble
		uint32 pad0;
		uint32 transDecrEnd;		//decrementer value of bubble transfer end (estimate) (to not stall on it too early), 0 is reset value
		uint32 pad;

		__attribute__((always_inline))
		inline void Reset()
		{
			curIndex				= 0xFFFF;
			curState				= BUB_STATE_INACTIVE;
			transDecrEnd		= 0xFFFFFFFF;//max val so any check with cur dec. value expect it finished
		}
	} _ALIGN(16);

	//macro definitions for bubble intrinsic functions	
	#define ResetBubbleLRUCounter() g_BubbleLRUCounter	= spu_splats((uint32)0)
	#define ResetBubbleLRU() g_SPUBubbleLRUDir	= spu_splats((uint32)0)
	#define IncrBubbleLRU()  g_BubbleLRUCounter = spu_add(g_BubbleLRUCounter, 1)
	#define GetBubbleLRU()	spu_extract(g_BubbleLRUCounter, 0)
	#define UpdateBubbleLRUByIndex(cBubIndex, cLRUVal) g_SPUBubbleLRUDir = spu_insert(cLRUVal, g_SPUBubbleLRUDir, cBubIndex)
	#define UpdateBubbleLRUByID(cBubIDSplat4) g_SPUBubbleLRUDir = spu_sel(g_SPUBubbleLRUDir, g_BubbleLRUCounter, spu_cmpeq(cBubIDSplat4, g_SPUBubbleDir))
	#define ResetBubbleDir() NSPU::g_SPUBubbleDir			= spu_splats(-1)
	#define SetBubbleIndex(cBubIndex, cSlot) g_SPUBubbleDir = spu_insert(cBubIndex, g_SPUBubbleDir, cSlot)
	#define GetBubbleIndex(cSlot) spu_extract(g_SPUBubbleDir, cSlot)
	#define IsBubblePresent(cBubIDSplat4) (spu_extract(spu_orx(spu_cmpeq(g_SPUBubbleDir, cBubIDSplat4)), 0) != 0)
	#define GetBubbleReplIndex() GetReplIndex(g_SPUBubbleLRUDir)
	#define GetEmptyBubbleSlot() spu_extract(spu_add(spu_cntlz(spu_gather(spu_cmpeq(g_SPUBubbleDir, spu_splats(-1)))), -28), 0)
	#define GetBubbleSlot(cBubIDSplat4) spu_extract(spu_add(spu_cntlz(spu_gather(spu_cmpeq(g_SPUBubbleDir, cBubIDSplat4))), -28), 0)
	#define NO_EMPTY_BUBBLE_SLOT 4
#endif//__SPU__
}

#endif //PS3
#endif //__SPU_BUBBLES_H
