

//////////////////////////////////////////////////////////////////////
//
//	Crytek SuperFramework Source code
//	
//	File:PolyBump.cpp
//  Description: Creates a poly bump with 2 models (low and high resolution) as input 
//
//	History: rmal
//	-March 15,2001:Created by Marco Corbetta
//  -July 13,2001: Rewritten from scratch by Marco Corbetta
//  -2/21/2002 modified to work in 3DStudio max as a thread (MM)
//
//////////////////////////////////////////////////////////////////////


#include "stdafx.h"

#include <vector>																			// std::vector
#include <list>																				// std::list
#include <map>																				// std::map
#include "TGA.h"																			// PIX_LoadTGA32
#include "list.h"																			// list2
#include "SimpleIndexedMesh.h"												// CSimpleIndexedMesh	
#include "PolyBump.h"																	// 
#include "PolyBumpWorkerThread.h"											// CPolyBumpWorkerThread
#include "CopyProtection.h"														// ChangeSerialYesNo(),GetSerialFromRegistry(),HowToGetSerialDialog()
#include "KeyGen.h"																		// SimpleCheck() SerialNo
#include "MAC_Address.h"															// NetbiosHelper_CheckCRC
#include "Crytek_Logo.h"															// Crytek Logo
#include "Cry_Math.h"																	//
#include "tools.h"																		// CalcUVArea()
#include "ChooseIntersection.h"												// Choose..Intersection
#include "SRFData.h"																	// CSRFData
#include "RegTools.h"																	// CallResourceCompilerForOneFile()

#include "TriangleInputProxy.h"												// CTriangleInputProxy



typedef unsigned char UBYTE;


struct SSamplePointContext
{
	// constructor
	SSamplePointContext( CPbHighMesh *pHighMesh, CPbHighMesh *pCageMesh ) 
		:m_pTriLow(0), m_pHighMesh(pHighMesh), m_pCageMesh(pCageMesh), m_nAngleMaxCount(0), m_nHorizonTilt(5)
	{
		assert(pHighMesh);

		PerSampleReset(0);
	}

	void PerSampleReset( CPbLowTri *pLowTri )
	{
		m_pTriLow=pLowTri;
		m_pHighPolyTri=0;
		m_vSumHighPolyPos=Vec3(0,0,0);
		m_vSumHighPolyNormal=Vec3(0,0,0);

		for(uint32 dwI=0;dwI<m_nAngleMaxCount;++dwI)
			m_fSumHorizonAngle[dwI]=0;
	}

	//
	void GetHighPolyPosAndNormal( Vec3 &vPos, Vec3 &vNormal ) const
	{
		vPos=m_vSumHighPolyPos;
		vNormal=m_vSumHighPolyNormal;
	}
	//
	void SetHighPolyPosAndNormal( const Vec3 &vPos, const Vec3 &vNormal )
	{
		m_vSumHighPolyPos+=vPos;
		m_vSumHighPolyNormal+=vNormal;
	}
	//
	void SetHorizon( const uint32 dwAngleNo, const float fAngle )
	{
		assert(fAngle>=0 && fAngle<=gf_PI_DIV_2);
		assert(dwAngleNo>=0 && dwAngleNo<=m_nAngleMaxCount);

		m_fSumHorizonAngle[dwAngleNo]+=fAngle;
	}
	//
	float GetHorizon( const uint32 dwAngleNo )
	{
		return m_fSumHorizonAngle[dwAngleNo];
	}

	// in
	CPbLowTri *				m_pTriLow;						//!<
	CPbHighMesh *			m_pHighMesh;					//!<
	CPbHighMesh *			m_pCageMesh;					//!<
	float							m_fLowPolyObjectSize;	//!<
	uint32						m_nAngleMaxCount;			//!< defines the used array elements in m_fHorizon 0=no additonal ray shooting..31
	uint32						m_nHorizonTilt;				//!< Quality of the horizon angle value (count = 1<<m_nHorizonTilt)

	// out
	CPbHighTri *			m_pHighPolyTri;				//!<

private: // ------------------------------------------------------------

	// out
	//	float							m_fSumDiffuseColor[3];		//!< (0..1,0..1,0..1)
	//	float							m_fSumSpecularColor[3];	//!< (0..1,0..1,0..1)
	//	float							m_fSumGloss;							//!< 0..1
	//	float							m_fSumTransparency;			//!< 0..1
	Vec3							m_vSumHighPolyPos;			//!<
	Vec3							m_vSumHighPolyNormal;		//!<
	float							m_fSumHorizonAngle[32];			//!< [0.. m_nAngleMaxCount-1] 0..gf_PI_DIV_2
};









/*
//! /param invVec must not be normalized
static void SetVectorAsRGB( Vec3 invVec, unsigned char *outPtr, const bool inbJitter )
{
	if(inbJitter)				// normal jitter for better quality
	{
		float jitter[3]={ (float)((rand()%1024)-512)/(float)(512*255),			// -1..1 * (1/255)
											(float)((rand()%1024)-512)/(float)(512*255),
											(float)((rand()%1024)-512)/(float)(512*255) };

		invVec*=254/255.0f;																									// 254/255..254/255

		invVec.x+=jitter[0];
		invVec.y+=jitter[1];
		invVec.z+=jitter[2];

		// -1..1
	}

	int iX=(int)((invVec.x+1.0f)*(255.0f/2.0f));		if(iX>255)iX=255;		if(iX<0)iX=0;
	int iY=(int)((invVec.y+1.0f)*(255.0f/2.0f));		if(iY>255)iY=255;		if(iY<0)iY=0;
	int iZ=(int)((invVec.z+1.0f)*(255.0f/2.0f));		if(iZ>255)iZ=255;		if(iZ<0)iZ=0;

	outPtr[0]=(unsigned char)iX;
	outPtr[1]=(unsigned char)iY;
	outPtr[2]=(unsigned char)iZ;
}
*/

CPbHighMesh::CPbHighMesh() 
	:m_pMesh(0), m_nNumTris(0) 
{
}


CPbHighMesh::~CPbHighMesh()
{
#ifdef 	USE_MEMORY_BLOCKS	//win32 memory optimization
	for (unsigned long i = 0; i < m_nBlockCnt; i++)
	{
		if(m_pMesh)
		{
			delete [] m_pMesh[i];
			m_pMesh[i]=NULL;
		}    
	}

	delete [] m_pMesh;
	m_pMesh=NULL;
#else
	if(m_pMesh)
	{
		delete [] m_pMesh;
		m_pMesh=NULL;
	}    
#endif //USE_MEMORY_BLOCKS
}

// calculate the size of the bounding sphere
float CPbLowMesh::CalcBoundingVolume( Vec3 &outMin, Vec3 &outMax )
{
	Vec3 minbox(FLT_MAX,FLT_MAX,FLT_MAX),maxbox(-FLT_MAX,-FLT_MAX,-FLT_MAX);

	// calculate bounding box
	for(int i=0;i!=m_nNumTris;i++)
		m_pMesh[i].ExtendMinMax(minbox,maxbox);

	// calc midpoint
	Vec3 vMidPoint=(minbox+maxbox)*0.5f;
	float Radius2=0.0;

	// calc radius
	for(int i=0;i!=m_nNumTris;i++)
			m_pMesh[i].ExtendSphere(vMidPoint,Radius2);

	outMin=minbox;
	outMax=maxbox;

	return (float)sqrt(Radius2);
}


// calculate the size of the bounding sphere
float CPbHighMesh::CalcBoundingVolume( Vec3 &outMin, Vec3 &outMax )
{
	Vec3 minbox(FLT_MAX,FLT_MAX,FLT_MAX),maxbox(-FLT_MAX,-FLT_MAX,-FLT_MAX);

	CPbHighTri *pTri=NULL;

	// calculate bounding box
	for(int ctTriNum=0;ctTriNum!=m_nNumTris;ctTriNum++)
	{
#ifdef	USE_MEMORY_BLOCKS	//win32 memory optimization
			//fix restCnt ?
			pTri = &m_pMesh[ctTriNum/m_nObjPerBlock][ctTriNum%m_nObjPerBlock];
#else
			pTri = &m_pMesh[ctTriNum];
#endif //USE_MEMORY_BLOCKS
			pTri->ExtendMinMax(minbox,maxbox);
	}

	// calc midpoint
	Vec3 vMidPoint=(minbox+maxbox)*0.5f;
	float Radius2=0.0;

	// calc radius
	for(int ctTriNum=0; ctTriNum!=m_nNumTris;ctTriNum++)
	{
#ifdef 	USE_MEMORY_BLOCKS	//win32 memory optimization
			//fix restCnt ?
			pTri = &m_pMesh[ctTriNum/m_nObjPerBlock][ctTriNum%m_nObjPerBlock];
#else
			pTri = &m_pMesh[ctTriNum];
#endif //USE_MEMORY_BLOCKS
			pTri->ExtendSphere(vMidPoint,Radius2);
	}

	outMin=minbox;
	outMax=maxbox;

	return (float)sqrt(Radius2);
}






















// * Build index for tangent space (tangent vector, binormal vector) and build a new index value for every
//   - vertex index
//   - material id
//   - normal index (build from smoothing groups)
//   - texture index
//
// * Calc tangent and binormal per index, sum indices together
//
// * normalize result



// transform the indexed mesh in a convenient format
//////////////////////////////////////////////////////////////////////
void CPbHighMesh::GetNormalsAndClearBaseVectors( CSimpleIndexedMesh *mesh )
{
	assert(mesh->m_VertCount!=0);			if(mesh->m_VertCount==0)return;


#ifdef 	USE_MEMORY_BLOCKS	//win32 opptimization

	unsigned long  nRestCnt;

	m_nObjPerBlock = MAX_MEMORY_BLOCK / sizeof(CPbHighTri);
	m_nBlockCnt = mesh->m_FaceCount / m_nObjPerBlock;
	nRestCnt = mesh->m_FaceCount % m_nObjPerBlock;

	if (  nRestCnt > 0 )
	{
		m_nBlockCnt++;
		m_pMesh=new CPbHighTri*[m_nBlockCnt];
		if(!m_pMesh)
		{
			MessageBox(0,"","low mem 3",MB_OK);
			return;
		}

		for (unsigned i=0; i<(m_nBlockCnt-1); i++){
			m_pMesh[i] = new CPbHighTri[m_nObjPerBlock];
			if(!m_pMesh[i])
			{
				MessageBox(0,"","low mem for Blocks",MB_OK);
				return;
			}
		}
		m_pMesh[m_nBlockCnt-1] = new CPbHighTri[nRestCnt];
		if(!m_pMesh[m_nBlockCnt-1])
		{
			MessageBox(0,"","low mem for Blocks",MB_OK);
			return;
		}

	}	
	else
	{
		m_pMesh=new CPbHighTri*[m_nBlockCnt];
		if(!m_pMesh)
		{
			MessageBox(0,"","low mem 3",MB_OK);
			return;
		}

		for (unsigned i=0; i<m_nBlockCnt; i++)
		{
			m_pMesh[i] = new CPbHighTri[m_nObjPerBlock];
			if(!m_pMesh[i])
			{
				MessageBox(0,"","low mem for Blocks",MB_OK);
				return;
			}
		}
		m_pMesh[m_nBlockCnt-1] = new CPbHighTri[nRestCnt];

	}	

#else
	m_pMesh=new CPbHighTri[mesh->m_FaceCount];

	if(!m_pMesh)
	{
		MessageBox(0,"","low mem 3",MB_OK);
		return;
	}

#endif //USE_MEMORY_BLOCKS



	m_nNumTris=mesh->m_FaceCount;

	// copy data and build triangle data structures (area,area3d,plane)
	{
		
		CObjNorm *pNorms=0;
		CPbHighTri *pTri=NULL;
		CObjFace *pTface=mesh->m_pFaces;

		// for every traingle
		for(int ctTriNum = 0; ctTriNum<mesh->m_FaceCount; ctTriNum++,pTface++)
		{	
#ifdef 	USE_MEMORY_BLOCKS	//win32 memory optimization

			//fix restCnt ?
			pTri = &m_pMesh[ctTriNum/m_nObjPerBlock][ctTriNum%m_nObjPerBlock];
#else
			pTri = &m_pMesh[ctTriNum];
#endif //USE_MEMORY_BLOCKS
			// for every vertex of the triangle
			for(int j=0;j<3;j++)
			{
				assert(pTface->v[j]>=0);
				
				CObjVert *pTvert = &mesh->m_pVerts[pTface->v[j]];

				memcpy(pTri->m_Verts[j],pTvert,sizeof(Vec3));
/*
				if(mesh->m_pCoors)
				{
					CObjCoor *pTcoord = &mesh->m_pCoors[pTface->t[j]];	// uv mapping supplied
	  			pTri->m_fS[j]=pTcoord->s;
	  			pTri->m_fT[j]=pTcoord->t;		
				}
				else
				{
	  			pTri->m_fS[j]=0.0f;											// no uv mapping supplied
	  			pTri->m_fT[j]=0.0f;		
				}
*/
				if(mesh->m_pNorms)												// normals supplied
				{
					pNorms=&mesh->m_pNorms[pTface->n[j]];  		
					memcpy(pTri->m_VertsNormal[j],pNorms,sizeof(Vec3));
				}
				else																			// no normals supplied
					pTri->m_VertsNormal[j]=Vec3(0,0,0);

//				pTri->m_vTangent[j]=Vec3(0,0,0);					// clear tangent base vectors
//				pTri->m_vBinormal[j]=Vec3(0,0,0);
//				pTri->m_vTNormal[j]=Vec3(0,0,0);
			} //j

			pTri->RefreshInternals();										// this is neccessray to get rayshooting working
		} //k	
	}
}



// transform the indexed mesh in a convenient format
//////////////////////////////////////////////////////////////////////
void CPbLowMesh::GetNormalsAndClearBaseVectors( CSimpleIndexedMesh *mesh )
{
	assert(mesh->m_VertCount!=0);			if(mesh->m_VertCount==0)return;

	m_pMesh=new CPbLowTri[mesh->m_FaceCount];  

	if(!m_pMesh)
	{
		MessageBox(0,"","low mem 4",MB_OK);
		return;
	}

	m_nNumTris=mesh->m_FaceCount;

	// copy data and build triangle data structures (area,area3d,plane)
	{
		CObjNorm *pNorms=0;
		CPbLowTri *pTri=m_pMesh;
		CObjFace *pTface=mesh->m_pFaces;

		// for every traingle
		for(int k=0;k<mesh->m_FaceCount;k++,pTface++,pTri++)
		{	
			// for every vertex of the triangle
			for(int j=0;j<3;j++)
			{
				assert(pTface->v[j]>=0);
				
				CObjVert *pTvert = &mesh->m_pVerts[pTface->v[j]];

				memcpy(pTri->m_Verts[j],pTvert,sizeof(Vec3));

				if(mesh->m_pCoors)
				{
					CObjCoor *pTcoord = &mesh->m_pCoors[pTface->t[j]];	// uv mapping supplied
	  			pTri->m_fS[j]=pTcoord->s;
	  			pTri->m_fT[j]=pTcoord->t;		
				}
				else
				{
	  			pTri->m_fS[j]=0.0f;											// no uv mapping supplied
	  			pTri->m_fT[j]=0.0f;		
				}

				if(mesh->m_pNorms)												// normals supplied
				{
					pNorms=&mesh->m_pNorms[pTface->n[j]];  		
					memcpy(pTri->m_VertsNormal[j],pNorms,sizeof(Vec3));
				}
				else																			// no normals supplied
					pTri->m_VertsNormal[j]=Vec3(0,0,0);

				pTri->m_vTangent[j]=Vec3(0,0,0);					// clear tangent base vectors
				pTri->m_vBinormal[j]=Vec3(0,0,0);
				pTri->m_vTNormal[j]=Vec3(0,0,0);
			} //j

			pTri->RefreshInternals();										// this is neccessray to get rayshooting working
		} //k	
	}
}







void CPbLowMesh::BuildNormalsAndBaseVectors( CSimpleIndexedMesh *mesh )
{
	assert(mesh);		// must not be 0 for CTriangleInputProxy()

	GetNormalsAndClearBaseVectors(mesh);

	CTangentSpaceCalculation<CTriangleInputProxy> tangents;
	CTriangleInputProxy Input(mesh);

	tangents.CalculateTangentSpace(Input);

	// store the calculated data
	{
		CPbLowTri *pTri=m_pMesh;
		CObjFace *pTface=mesh->m_pFaces;

		// for every triangle
		for(DWORD k=0;k<(DWORD)mesh->m_FaceCount;k++,pTri++)
		{	
			DWORD dwBaseIndx[3];

			tangents.GetTriangleBaseIndices(k,dwBaseIndx);

			pTri->m_iTriMaterialID=mesh->m_pFaces[k].shader_id;

			// for every vertex of the triangle
			for(int j=0;j<3;j++)
			{
				float vU[3],vV[3],vN[3];

				tangents.GetBase(dwBaseIndx[j],vU,vV,vN);

				pTri->m_vBinormal[j]=Vec3(vU[0],vU[1],vU[2]);
				pTri->m_vTangent[j]=Vec3(vV[0],vV[1],vV[2]);
				pTri->m_vTNormal[j]=Vec3(vN[0],vN[1],vN[2]);
			}
		}
	}
}














void CalcHorizon( SSamplePointContext &rContext, const Vec3 &vBaseA, const Vec3 &vBaseB, const Vec3 &vBaseN, const Vec3 &vBasePos )
{
	Vec3	vFrom = vBasePos + vBaseN*rContext.m_fLowPolyObjectSize*0.002f;		// slight offset from the surface

	// hemisphere lighting implemented with optimized algorithm (fast, good approximation)
	//
	// algorithm works like this:
	//
	//  To sample the whole hemisphere by dividing it into wedges (nAngleMaxCount).
	//  For every wedge a shadow horizon is searched (binary search, nTiltMaxCount times => resolution:2^nTiltMaxCount) 
	float fSkyArea=0.0f;		
	float walker=0.0f;
	int nTiltMaxCount=rContext.m_nHorizonTilt;
	const float walkerstep=gf_PI_MUL_2/(float)rContext.m_nAngleMaxCount;

	for(uint32 w=0;w<rContext.m_nAngleMaxCount;w++,walker+=walkerstep)
	{
		float s=sinf(walker),c=cosf(walker);
		float tiltanglepart=0.5f*gf_PI_DIV_2;	//
		float deltapart=tiltanglepart*0.5f;		//

		for(int t=0;t<nTiltMaxCount;t++)
		{
			float tilt=sinf(tiltanglepart);
			float rndx,rndy,rndz;

			rndx=s*tilt; rndy=c*tilt; rndz=(float)sqrt(1.0f-tilt);

			Vec3 vDir=vBaseA*rndx + vBaseB*rndy + vBaseN*rndz;

			if(!rContext.m_pHighMesh->CalcIntersection(vFrom,vFrom+vDir*rContext.m_fLowPolyObjectSize,rContext.m_pHighPolyTri))
				tiltanglepart+=deltapart;
				else
				tiltanglepart-=deltapart;

			deltapart*=0.5f;
		}

		rContext.SetHorizon(w,tiltanglepart);
	}


/*
	// hemisphere lighting implemented with simple Monte-Carlo sampling (slow but correct)
	int nCount=0;
	const int nMaxCount=64*6;
	for(int i=0;i<nMaxCount;i++)		// nMaxCount samples
	{
		Vec3 vTo;

		// get ray in a sphere
		do
		{
			vTo=Vec3((rand()%1024)*(1.0f/512.0f)-1.0f,
								(rand()%1024)*(1.0f/512.0f)-1.0f,
								(rand()%1024)*(1.0f/512.0f)-1.0f);

		} while(vTo.Length2()>1.0f);

		if(vTo*highresnorm<0.0f)vTo*=-1.0f;			// wrong side of hermisphere, flip it

		// get ray on a sphere
		vTo.Normalize();

		vTo=vFrom+vTo*infRayScale;

		if(!rContext.m_pHighMesh->CalcIntersection(vFrom,vTo))
			nCount++;
	}
	*outpAccessability=(float)nCount/(float)nMaxCount;
*/
}


// /param inpCageMesh might be 0
// /param outpAccessability 0 or pointer to a float, will be in the range [0..1]
// /param outOccDirection 0 if is outpAccessability 0 as well or pointer to the Vec3
bool GetSamplePoint( SSamplePointContext &rContext, float infRayLength,
										 float infS, float infT, CPolyBumpWorkerThread *inpData, bool inbDoCLipping )
{
	Vec3 lowresnorm,mid,highresnorm,respoint;

	DWORD dwWidth=inpData->m_Properties.GetOutputWidth();
	DWORD dwHeight=inpData->m_Properties.GetOutputHeight();

	float deltamapx=1.0f/dwWidth;
	float deltamapy=1.0f/dwHeight;

	//get interpolated point and normal using barycentric coords from texel's uv
	rContext.m_pTriLow->CalcBarycentricCoordsFromUV( infS,infT,respoint,lowresnorm,inbDoCLipping);

	lowresnorm.Normalize();

	mid=respoint;

	float fMinDist=-infRayLength, fMaxDist=infRayLength;

	// fill in hits with the cage, update fMinDist and fMaxDist
	if(rContext.m_pCageMesh)
	{
		Vec3 end=mid+lowresnorm*fMaxDist;
		Vec3 begin=mid+lowresnorm*fMinDist;
		//
		CIntInfoHighList hitListCage;					hitListCage.reserve(64);

		rContext.m_pCageMesh->GatherRayHits(begin,end,hitListCage,0);

		for(CIntInfoHighIt i=hitListCage.begin();i!=hitListCage.end();i++)
		{
			float fDistFromMid = (*i).m_fDist - infRayLength;

			if(fDistFromMid>0)
			{
				if(fDistFromMid<fMaxDist)
					fMaxDist=fDistFromMid;
			}
			else
			{
				if(fDistFromMid>fMinDist)
					fMinDist=fDistFromMid;
			}
		}
	}


	// m.m. optimization (I shoot only one ray - because both rays have the same direction,
	// the orientation doesnt matter )
	Vec3 end=mid+lowresnorm*fMaxDist;
	Vec3 begin=mid+lowresnorm*fMinDist;
	//
	CIntInfoHighList hitListHighPoly;				hitListHighPoly.reserve(64);
	// fill in hits with the high poly
	rContext.m_pHighMesh->GatherRayHits(begin,end,hitListHighPoly,0);


	CPbHighTri *cb=NULL;

	switch(inpData->m_Properties.m_nHitMode)
	{	
		// nearest hit
		case 0: cb=ChooseNearestIntersection(hitListHighPoly,begin,end,lowresnorm,respoint);break;
		// latest hit
		case 1: cb=ChooseLatestIntersection(hitListHighPoly,begin,end,lowresnorm,respoint);break;

		default:
			assert(0);
	}

	if(!cb) cb=ChooseNearestAcceptableIntersection(hitListHighPoly,begin,end,lowresnorm,respoint);

	if(cb) cb->CalcBarycentricCoordsFromPoint(respoint,highresnorm,false);
		else return false;																														//no tri at this texel

	rContext.m_pHighPolyTri=cb;
	rContext.SetHighPolyPosAndNormal(respoint,highresnorm);
/*
	// bump mapping
	{
		float dfX,dfY;

		if(inpData->GetBumpMapDerivate(infS,1.0f-infT,dfX,dfY))	// is bumpmap supplied?
		{
			Vec3 point12,point21;
			Vec3 vUnused;		// unused

			float stepx=deltamapx*0.1f;
			float stepy=deltamapy*0.1f;

			rContext.m_pTriLow->CalcBarycentricCoordsFromUV(infS+stepx,infT,point21,vUnused,inbDoCLipping);
			rContext.m_pTriLow->CalcBarycentricCoordsFromUV(infS,infT+stepy,point12,vUnused,inbDoCLipping);

			Vec3 dir12=point12-mid;
			Vec3 dir21=point21-mid;

			if(CalcUVArea(rContext.m_pTriLow->m_fS,rContext.m_pTriLow->m_fT)<0.0f)
				{ dir12=-dir12;dir21=-dir21; }

			if((dir12|dir12)>0.00001f && (dir21|dir21)>0.00001f)					// 
			{
				dir12.Normalize();
				dir21.Normalize();

				Matrix33 mat12;		mat12.SetRotationAA33(dfX*6.283f * gf_DEGTORAD,dir12);		// GetRotation should be static
				Matrix33 mat21;		mat21.SetRotationAA33(dfY*6.283f * gf_DEGTORAD,dir21);		// GetRotation should be static

				highresnorm = mat12*highresnorm + mat21*highresnorm;						// trick to make the rotation invariant to its position
				highresnorm.Normalize();
			}
		}
	}
*/

	return true;
}



static bool Difference( const Vec3 inA, const Vec3 inB, const float infTreshold )
{
	float dist2=(inA-inB).len2();

	return dist2>=infTreshold;
}



static bool DoHereAntialias( CSRFData &rSurface, DWORD x, DWORD y, DWORD dwWidth, DWORD dwHeight, CPbLowMesh *pLowMesh, const float fAntialiasTreshold )
{
	Vec3 vNormal,vNext;
	
	vNormal = rSurface.GetWorldHighPolyNormal(x,y);

	if(x!=0)																														// left
	{
		vNext = rSurface.GetWorldHighPolyNormal(x-1,y);
		
		if(Difference(vNormal,vNext,fAntialiasTreshold))
			return true;
	}

	if(y!=0)																														// top
	{
		vNext = rSurface.GetWorldHighPolyNormal(x,y-1);

		if(Difference(vNormal,vNext,fAntialiasTreshold))
			return true;
	}

	if(x!=dwWidth-1)																										// right
	{
		vNext = rSurface.GetWorldHighPolyNormal(x+1,y);

		if(Difference(vNormal,vNext,fAntialiasTreshold))
			return true;
	}

	if(y!=dwHeight-1)																										// down
	{
		vNext = rSurface.GetWorldHighPolyNormal(x,y+1);

		if(Difference(vNormal,vNext,fAntialiasTreshold))
			return true;
	}

	return false;
}

// is this texel on a border? - then do centroid sampling
bool IsBorderPixel( const uint32 dwX, const uint32 dwY, const CSRFData &rSRFData )
{
	if(dwX!=0)if(rSRFData.GetLowPolyTriangleId(dwX-1,dwY)==0xffff)return true;
	if(dwY!=0)if(rSRFData.GetLowPolyTriangleId(dwX,dwY-1)==0xffff)return true;
	if(dwX!=rSRFData.GetWidth()-1)if(rSRFData.GetLowPolyTriangleId(dwX+1,dwY)==0xffff)return true;
	if(dwY!=rSRFData.GetHeight()-1)if(rSRFData.GetLowPolyTriangleId(dwX,dwY+1)==0xffff)return true;

	return false;
}

//create a polybump with 2 meshes as input
//////////////////////////////////////////////////////////////////////
bool CreateBumpMap( CPolyBumpWorkerThread *inpData )
{
	CSRFData SRFDataOutput;	// used to store the .SRF file format in memory

	inpData->DebugLog("CreateBumpMap() started --------------------------------");

	DWORD dwExpandTextureInPixels=(1 << inpData->m_Properties.m_iExpandTextureMode)/2;

	bool bEvaluationVersion=false;

	inpData->m_bUserHasStopped=false;

	// check input
	{
		CSimpleIndexedMesh *m1=inpData->GetLowMesh();		assert(m1);			if(!m1)return false;
		CSimpleIndexedMesh *m2=inpData->GetHighMesh();	assert(m2);			if(!m2)return false;
	}

	DWORD dwHighPolyFaceCount = inpData->GetHighMesh()->m_FaceCount;

	DWORD dwWidth=inpData->m_Properties.GetOutputWidth();
	DWORD dwHeight=inpData->m_Properties.GetOutputHeight();

	SRFDataOutput.Init(dwWidth,dwHeight);

	inpData->DebugLog("Width: %d, Height: %d",dwWidth,dwHeight);

	{
		DWORD m,d,y;
		UBYTE HardwareID[4];

		if(!CCrytekKeyGen::FurtherCheck(g_szSerialNo,"PB",m,d,y,HardwareID)											// Check serial number PB=PolyBump
		|| !NetbiosHelper_CheckCRC(HardwareID))
		{
			char str2[2048];

			sprintf(str2,"Unregistered Version (HardwareID dependant SerialNo is missing or wrong)\n\n"\
					"This trial version is is limited to %dx%d texture size and max. %d triangles\n\n"\
					"Check the AboutBox for information about getting the registered version.",TRIALVERSION_MAXTEXTURESIZE,TRIALVERSION_MAXTEXTURESIZE,TRIALVERSION_MAXHIGHPOLYCOUNT);
			MessageBox(0,str2,"PolyBumpPlugin",MB_OK);

			if(dwWidth>TRIALVERSION_MAXTEXTURESIZE || dwHeight>TRIALVERSION_MAXTEXTURESIZE || dwHighPolyFaceCount>TRIALVERSION_MAXHIGHPOLYCOUNT)
				return false;

			bEvaluationVersion=true;
		}
	}

	SRFDataOutput.AllocSurfaceHighPos();	// allocate space for the displacement map
	
  // allocate space for the normal map
	if(!SRFDataOutput.AllocSurfaceWorldNormals())
	{
		MessageBox(0,"","low mem 8",MB_OK);
		return false;
	}
  
	// -------------------------------------------------------------------------------

  inpData->m_nImageSizeX=dwWidth;
  inpData->m_nImageSizeY=dwHeight;
	float deltamapx=1.0f/dwWidth;
	float deltamapy=1.0f/dwHeight;

	CPbLowMesh *pLowMesh=new CPbLowMesh;

	inpData->DebugLog("build low poly 1");

	// build low poly data structure + accelerator structure
	{
		CSimpleIndexedMesh *m1=inpData->GetLowMesh();		assert(m1);			if(!m1)return false;

		pLowMesh->BuildNormalsAndBaseVectors(m1);

		inpData->DebugLog("build low poly 2 (triangle count: %d)",pLowMesh->m_nNumTris);

		pLowMesh->BuildRayAccelerator(inpData->m_Properties.m_bOutputDebugInfo);

		m1->FreeData();
	}

	// set lowpolymesh into SRFDataOutput
	{
		SRFDataOutput.AllocSourceLowPoly(pLowMesh->m_nNumTris);

		for(uint32 dwTri=0;dwTri<pLowMesh->m_nNumTris;++dwTri)
		{
//			SRFDataOutput.SetLowPolyTriangle(dwTri,);

			CPbLowTri &rTri = pLowMesh->m_pMesh[dwTri];

			for(uint32 dwI=0;dwI<3;++dwI)		// all 3 verices of the triangle
			{	
				SRFDataOutput.SetLowPolyVertex(dwTri,dwI,
					rTri.m_Verts[dwI],																										// pos
					rTri.m_vTangent[dwI],rTri.m_vBinormal[dwI],rTri.m_VertsNormal[dwI],		// tnagents    (todo: check if m_vTangent and m_vBinormal should not swap places)
					rTri.m_fS[dwI],rTri.m_fT[dwI]);																				// UV
			}
		}
	}


	// build cage data structure + accelerator structure
	CPbHighMesh *pCageMesh=0;
	{
		CSimpleIndexedMesh *m3=inpData->GetCageMesh();		assert(m3);			if(!m3)return false;

		if(m3->m_VertCount)
		{
			pCageMesh=new CPbHighMesh();

			inpData->DebugLog("build cage 1");

			pCageMesh->GetNormalsAndClearBaseVectors(m3);

			inpData->DebugLog("build cage poly 2 (triangle count: %d)",pCageMesh->m_nNumTris);

			pCageMesh->BuildRayAccelerator(inpData->m_Properties.m_bOutputDebugInfo);

			m3->FreeData();
		}
	}

  CPbHighMesh *pHighMesh=NULL;

	inpData->DebugLog("build high poly 1");

	// build high poly data structure + accelerator structure
	CSimpleIndexedMesh *m2;
	try
	{
		m2=inpData->GetHighMesh();	assert(m2);			if(!m2)return false;

		pHighMesh=new CPbHighMesh;
    pHighMesh->GetNormalsAndClearBaseVectors(m2);

		inpData->DebugLog("build high poly 2");
		{
			char str[256];

			sprintf(str,"HighRes triangle count: %d (%d bytes)",pHighMesh->m_nNumTris,pHighMesh->m_nNumTris*sizeof(CPbHighTri));

			inpData->DebugLog("build high poly 3 (%s)",str);
		}

		{
			std::string str = pHighMesh->BuildRayAccelerator(inpData->m_Properties.m_bOutputDebugInfo);

			inpData->DebugLog("build high poly 4 (%s)",str);
		}

		m2->FreeData();
  }
	catch (CMemoryException* )
	{
		m2->FreeData();

		SRFDataOutput.FreeData();

	  // clean up
	  delete pLowMesh;
	  delete pHighMesh;
		delete pCageMesh;

		MessageBox(0,"can't allocate memory","High-Poly mesh ",MB_OK);
		return false;
	}

	if(!inpData->ThreadPollMessages(0,"Init")) 
		inpData->m_bUserHasStopped=true;

  // draw tris pointer to keep track of the low mesh triangles
	if(!inpData->m_bUserHasStopped)
		inpData->DrawTrisPointers(SRFDataOutput.GetLowPolyTriangleIdBuffer(),pLowMesh);

	SSamplePointContext SamplePointContext(pHighMesh,pCageMesh);

	Vec3 vMin,vMax;
	
	SamplePointContext.m_fLowPolyObjectSize = pLowMesh->CalcBoundingVolume(vMin,vMax);

	float rayscale=inpData->m_Properties.m_fRayLength*SamplePointContext.m_fLowPolyObjectSize*2.0f;	// *2.0f because diameter not radius should be used

	//  for good results (2^nTiltMaxCount) should be nearly the same as (nAngleMaxCount*3.14)
	switch(inpData->m_Properties.m_iHorizonMode)
	{
		case 0:		break;		// none
		case 1:		SamplePointContext.m_nAngleMaxCount=6;SamplePointContext.m_nHorizonTilt=4;break;
		case 2:		SamplePointContext.m_nAngleMaxCount=12;SamplePointContext.m_nHorizonTilt=5;break;
		case 3:		SamplePointContext.m_nAngleMaxCount=16;SamplePointContext.m_nHorizonTilt=6;break;
		default: assert(0);
	}

	{
		char str[256];

		sprintf(str,"m_nAngleMaxCount=%d, m_nHorizonTilt=%d\n",SamplePointContext.m_nAngleMaxCount,SamplePointContext.m_nHorizonTilt);
		OutputDebugString(str);
	}

	// allocate space for the accessilibity map
	if(SamplePointContext.m_nAngleMaxCount)
		SRFDataOutput.AllocSurfaceHorizon(SamplePointContext.m_nAngleMaxCount);

	{
		DWORD m,d,y;
		UBYTE HardwareID[4];

		if(!CCrytekKeyGen::FurtherCheck(g_szSerialNo,"PB",m,d,y,HardwareID)											// Check serial number PB=PolyBump
		|| !NetbiosHelper_CheckCRC(HardwareID))
		{
			if(dwWidth>TRIALVERSION_MAXTEXTURESIZE || dwHeight>TRIALVERSION_MAXTEXTURESIZE || dwHighPolyFaceCount>TRIALVERSION_MAXHIGHPOLYCOUNT)
				inpData->m_bUserHasStopped=true;
		}
	}


	if(!inpData->m_bUserHasStopped)
		if(!inpData->ThreadPollMessages(1,"Build")) inpData->m_bUserHasStopped=true;

	//for ray-tracing
//	Vec3 start,end;
//	Vec3 inters1,inters2;

	if(inpData->m_Properties.m_iAntiAliasingMode)
		SRFDataOutput.AllocAALevel();

	inpData->DebugLog("build start calculation");

	if(!inpData->m_bUserHasStopped)
	if(inpData->m_Properties.DoNonAAStep())
	{
		DWORD	dwFullProgress = SRFDataOutput.CalcUsedAreaSize();	// to calculate the progress
		DWORD	dwCurrentProgress = 0;

		float mapt=deltamapy*0.5f;											// correct half pixel

		for(DWORD y=0;y<dwHeight;y++,mapt+=deltamapy)
		{    
			const char *szInfoText=	inpData->m_Properties.m_iAntiAliasingMode==0 ? "Pass 1/1" : "Pass 1/2";

			if(!inpData->ThreadPollMessages((dwCurrentProgress*100.0f)/(float)dwFullProgress,szInfoText)){ inpData->m_bUserHasStopped=true;break; }

			float maps=deltamapx*0.5f;																					// correct half pixel
			for(DWORD x=0;x<dwWidth;x++,maps+=deltamapx)
			{    
				uint16 wLowPolyTriangle = SRFDataOutput.GetLowPolyTriangleId(x,y);

				if(wLowPolyTriangle==0xffff)
					continue;													 // no tri at this texel

				dwCurrentProgress++;																									// next texel (for progress)

				SamplePointContext.PerSampleReset(&pLowMesh->m_pMesh[wLowPolyTriangle]);

				bool bDoClipping=IsBorderPixel(x,y,SRFDataOutput);				// is this texel on a border? - then do centroid sampling

				if(!GetSamplePoint(SamplePointContext,rayscale,maps,mapt,inpData,bDoClipping))
					continue;

				{
					Vec3 vPos,vNormal;

					SamplePointContext.GetHighPolyPosAndNormal(vPos,vNormal);

					SRFDataOutput.SetHighPos(x,y,vPos);
					SRFDataOutput.SetWorldHighPolyNormal(x,y,vNormal);
				}

				// then get the interpolated normal on the high poly mesh using barycentric coords
				// calculate one accessibility pixel
				if(SamplePointContext.m_nAngleMaxCount)							// !=0 means horizon should be calculated
				{
					Vec3 vBaseA,vBaseB,vBaseN,vPos;

					SRFDataOutput.CalcHighPolyBase(x,y,vBaseA,vBaseB,vBaseN);
					vPos=SRFDataOutput.GetHighPos(x,y);

					CalcHorizon(SamplePointContext,vBaseA,vBaseB,vBaseN,vPos);
				}

				for(uint32 dwI=0;dwI<SamplePointContext.m_nAngleMaxCount;++dwI)
					SRFDataOutput.SetHorizon(x,y,dwI,SamplePointContext.GetHorizon(dwI));
			} //x
		} //y
	}


	if(!inpData->m_bUserHasStopped)
	if(inpData->m_Properties.m_iAntiAliasingMode)
	{
		float fAntialiasStep,fAntialiasTreshold;

		inpData->m_Properties.GetAntialiasingParam(fAntialiasStep,fAntialiasTreshold);

		for(DWORD y=0;y<dwHeight;y++)
		for(DWORD x=0;x<dwWidth;x++)
		{    
			if(SRFDataOutput.GetLowPolyTriangleId(x,y)==0xffff)
				continue;																					// no tri at this texel

			if(DoHereAntialias(SRFDataOutput,x,y,dwWidth,dwHeight,pLowMesh,fAntialiasTreshold))
				SRFDataOutput.SetAALevel(x,y,1);
		}

		DWORD	dwFullProgress = SRFDataOutput.CalcAALevelSum();	// to calculate the progress
		DWORD	dwCurrentProgress = 0;

		// adaptive Antialiasing
		float mapt=deltamapy*0.5f;																						// correct half pixel

		for(DWORD y=0;y<dwHeight;y++,mapt+=deltamapy)
		{    
			const char *szInfoText=inpData->m_Properties.DoNonAAStep() ? "Pass 2/2" : "Pass 1/1";

			if(!inpData->ThreadPollMessages((dwCurrentProgress*100.0f)/(float)dwFullProgress,szInfoText)){ inpData->m_bUserHasStopped=true;break; }

			float maps=deltamapx*0.5f;																					// correct half pixel
			for(DWORD x=0;x<dwWidth;x++,maps+=deltamapx)
			{    
				uint8 ucAALevel=SRFDataOutput.GetAALevel(x,y);

				if(!ucAALevel)
					continue;																								// no antialiasing at this texel

				uint16 wLowPolyTriangle = SRFDataOutput.GetLowPolyTriangleId(x,y);

				assert(wLowPolyTriangle!=0xffff);													// no tri at this texel

				dwCurrentProgress+=(uint32)ucAALevel;											// next texel (for progress)

				bool bDoClipping=IsBorderPixel(x,y,SRFDataOutput);				// is this texel on a border? - then do centroid sampling

				SamplePointContext.PerSampleReset(&pLowMesh->m_pMesh[wLowPolyTriangle]);

				float stepherex=fAntialiasStep*deltamapx, stepherey=fAntialiasStep*deltamapy;

				float fAntialiasingSampleCount=0.0f;			// in float to avoid int->float cast

				// all non horizon data
				{
					for(float suby=(fAntialiasStep*0.5f-0.5f)*deltamapy;suby<=0.5f*deltamapy;suby+=stepherey)
					for(float subx=(fAntialiasStep*0.5f-0.5f)*deltamapx;subx<=0.5f*deltamapx;subx+=stepherex)
					{
						float jitterx=0.0f,jittery=0.0f;

						if(inpData->m_Properties.DoAAJittering())
						{
							jitterx=((rand()%1024)*(1.0f/1024.0f)-0.5f)*stepherex;
							jittery=((rand()%1024)*(1.0f/1024.0f)-0.5f)*stepherey;
						}

						if(!GetSamplePoint(SamplePointContext,rayscale,maps+subx+jitterx,mapt+suby+jittery,inpData,bDoClipping))
							continue;

						fAntialiasingSampleCount+=1.0f;
					}
				}

				if(fAntialiasingSampleCount)
				{
					float fAntialiasSampleWeight=1.0f/fAntialiasingSampleCount;
					
					{
						Vec3 vPos,vNormal;

						SamplePointContext.GetHighPolyPosAndNormal(vPos,vNormal);

						SRFDataOutput.SetHighPos(x,y,vPos*fAntialiasSampleWeight);
						SRFDataOutput.SetWorldHighPolyNormal(x,y,vNormal*fAntialiasSampleWeight);
					}
				}

/* no AA for the horizon - better to have it in higher resolution 
				// horizon data (afterwards to get better high poly normal as base)
				if(SamplePointContext.m_nAngleMaxCount)							// !=0 means horizon should be calculated
				{
					Vec3 vBaseA,vBaseB,vBaseN,vPos;

					SRFDataOutput.CalcHighPolyBase(x,y,vBaseA,vBaseB,vBaseN);
					vPos=SRFDataOutput.GetHighPos(x,y);

					for(float suby=(fAntialiasStep*0.5f-0.5f)*deltamapy;suby<=0.5f*deltamapy;suby+=stepherey)
					for(float subx=(fAntialiasStep*0.5f-0.5f)*deltamapx;subx<=0.5f*deltamapx;subx+=stepherex)
					{
						float jitterx=0.0f,jittery=0.0f;

						if(inpData->m_Properties.DoAAJittering())
						{
							jitterx=((rand()%1024)*(1.0f/1024.0f)-0.5f)*stepherex;
							jittery=((rand()%1024)*(1.0f/1024.0f)-0.5f)*stepherey;
						}

						CalcHorizon(SamplePointContext,vBaseA,vBaseB,vBaseN,vPos);
					}
				}

				for(uint32 dwI=0;dwI<SamplePointContext.m_nAngleMaxCount;++dwI)
					SRFDataOutput.SetHorizon(x,y,dwI,SamplePointContext.GetHorizon(dwI)*fAntialiasSampleWeight);
*/
			} //x
		} //y
	}

	// output SRF file before expand border to get better compression
	if(inpData->m_Properties.m_bOutputSRFData)
	{
		if(!SRFDataOutput.Save((inpData->m_Properties.m_szOutputFilename+".srf").c_str()))
			MessageBox(0,"file failed","Saving SRF",MB_OK|MB_ICONEXCLAMATION);
		else
		{
			// if possible call rc.exe
			CallResourceCompilerForOneFile(RC_APPNAME,(inpData->m_Properties.m_szOutputFilename+".srf").c_str());
		}
	}


#ifndef RESOURCECOMPILER_ONLY

//		ExpandBorder
	if(!inpData->m_bUserHasStopped)
		SRFDataOutput.ExpandBorder(dwExpandTextureInPixels);

  if(!inpData->m_bUserHasStopped)
	{
		DWORD pitch;

		// save normalmap with MAX help
		if(inpData->m_Properties.m_bOutputHighpolyNormals)
		if(UBYTE *writeMap=inpData->m_BitMapOutput.GetMemoryAndPitch(pitch))
		{
			assert(pitch==dwWidth*3);

			for(DWORD y=0;y<dwHeight;y++)
			for(DWORD x=0;x<dwWidth;x++)
			{
				Vec3 vNormal;
				
				if(inpData->m_Properties.m_bObjectSpace)
					vNormal = SRFDataOutput.GetWorldHighPolyNormal(x,y);
				else
					vNormal = SRFDataOutput.CalcTangentHighPolyNormal(x,y);

				unsigned char r = static_cast<unsigned char>(vNormal.x*127.5f+127.5f);
				unsigned char g = static_cast<unsigned char>(vNormal.y*127.5f+127.5f);
				unsigned char b = static_cast<unsigned char>(vNormal.z*127.5f+127.5f);

				writeMap[3*(x+y*dwWidth)  ]= r;
				writeMap[3*(x+y*dwWidth)+1]= g;
				writeMap[3*(x+y*dwWidth)+2]= b;
			}

			// burn in Crytek logo
			if(bEvaluationVersion)
			if(dwWidth>=256)
			if(dwHeight>=256)
			{
				int left=dwWidth/2-g_RawImageWidth/2,top=dwHeight/2-g_RawImageHeight/2; 

				for(int sy=0;sy<g_RawImageHeight;sy++)
				for(int sx=0;sx<g_RawImageWidth;sx++)
				{
					DWORD col=g_RawImageData[sy*g_RawImageWidth+sx];

					int x=(sx+left)%dwWidth;
					int y=(sy+top)%dwHeight;

					if(col)
					{
						writeMap[3*(x+y*dwWidth)]=(UBYTE)(((int)writeMap[3*(x+y*dwWidth)]-127)/2+127);
						writeMap[3*(x+y*dwWidth)+1]=(UBYTE)(((int)writeMap[3*(x+y*dwWidth)+1]-127)/2+127);
						writeMap[3*(x+y*dwWidth)+2]=(UBYTE)(((int)writeMap[3*(x+y*dwWidth)+2]-127)/2+127);
					}
				}
			}

			inpData->m_BitMapOutput.SaveAndClear(inpData->ExtendFilename(inpData->m_Properties.m_szOutputFilename+inpData->m_Properties.m_szOutputExtension).c_str());
		}
		else MessageBox(0,"output NORMAL file open failed","Error",MB_OK);

		// save accessibilitymap with MAX help
		if(SamplePointContext.m_nAngleMaxCount)
		if(inpData->m_Properties.m_bOutputAccessability)
		if(UBYTE *writeMap=inpData->m_BitMapOutput.GetMemoryAndPitch(pitch))
		{
			assert(pitch==dwWidth*3);

			for(DWORD y=0;y<dwHeight;y++)
			{
				for(DWORD x=0;x<dwWidth;x++)
				{
					unsigned char g = (unsigned char)(SRFDataOutput.CalcAccessibilityValue(x,y)*255.0f);

					writeMap[3*(x+y*dwWidth)]=g;
					writeMap[3*(x+y*dwWidth)+1]=g;
					writeMap[3*(x+y*dwWidth)+2]=g;
				}
			}

			inpData->m_BitMapOutput.SaveAndClear(inpData->ExtendFilename(inpData->m_Properties.m_szOutputFilename+"_AM"+inpData->m_Properties.m_szOutputExtension).c_str());
		}	// end Acessibility

		// **************************
#ifndef USE_REDUCEDFORPUBLIC
		// save occlusiondirection with MAX help
		if(SamplePointContext.m_nAngleMaxCount)
		if(inpData->m_Properties.m_bOutputOccDirection)
		if(UBYTE *writeMap=inpData->m_BitMapOutput.GetMemoryAndPitch(pitch))
		{
			assert(pitch==dwWidth*3);

			for(DWORD y=0;y<dwHeight;y++)
			{
				for(DWORD x=0;x<dwWidth;x++)
				{
					Vec3 vOccDir;

					SRFDataOutput.CalcUnocccludedAreaDirection(x,y,vOccDir,inpData->m_Properties.m_fOccBrighter);

					if(!inpData->m_Properties.m_bObjectSpace)
						SRFDataOutput.TransformDirectionIntoTangentSpace(x,y,vOccDir,vOccDir);

					unsigned char r = static_cast<unsigned char>(vOccDir[0]*127.5f+127.5f);
					unsigned char g = static_cast<unsigned char>(vOccDir[1]*127.5f+127.5f);
					unsigned char b = static_cast<unsigned char>(vOccDir[2]*127.5f+127.5f);

					writeMap[3*(x+y*dwWidth)]=r;
					writeMap[3*(x+y*dwWidth)+1]=g;
					writeMap[3*(x+y*dwWidth)+2]=b;
				}
			}

			{
				char szParams[80]="";

				if(inpData->m_Properties.m_bExtendFilename)
					sprintf(szParams,"_I%.1f",inpData->m_Properties.m_fOccBrighter*100.0f);		// I for interreflections

				inpData->m_BitMapOutput.SaveAndClear(inpData->ExtendFilename(inpData->m_Properties.m_szOutputFilename+"_OD"+szParams+inpData->m_Properties.m_szOutputExtension).c_str());
			}
		}	// end occlusiondirection
#endif // USE_REDUCEDFORPUBLIC

		// **************************

		// save displacementmap with MAX help
		if(inpData->m_Properties.m_bOutputDisplace)
		if(UBYTE *writeMap=inpData->m_BitMapOutput.GetMemoryAndPitch(pitch))
		{
			assert(pitch==dwWidth*3);

			// calculate the min and max of the displacement
			float fAbsBound=0.0f;
			{
				for(DWORD y=0;y<dwHeight;y++)
				{
					for(DWORD x=0;x<dwWidth;x++)
					{
						float fVal=SRFDataOutput.CalcDisplacementDist(x,y,inpData->m_Properties.m_bDisplaceNPatch,inpData->m_Properties.m_bDisplaceQuadratic);

						if(fVal<0.0f)
							if(!inpData->m_Properties.m_bDisplaceSigned)fVal=0.0f;			// clip nagative Values

						if(fVal!=FLT_MAX)
						{
							if(fVal>fAbsBound)fAbsBound=fVal;
							if(fVal<-fAbsBound)fAbsBound=-fVal;
						}
					}
				}
			}

			// scale the result to the output range

			float fDispScale=0.0f;
			
			inpData->m_Properties.m_fReturnDisplaceMag=fAbsBound;

			if(fAbsBound!=0.0f)
				fDispScale=1.0f/fAbsBound;

			float fDispTranslate=1.0f;
			float fDispOutputScale=127.5f;

			if(!inpData->m_Properties.m_bDisplaceSigned)		// unsigned
			{
				fDispTranslate=0.0;
				fDispOutputScale=255.0f;
			}
			else																						// signed
			{
				inpData->m_Properties.m_fReturnDisplaceMag*=2.0f;
			}

			for(DWORD y=0;y<dwHeight;y++)
			for(DWORD x=0;x<dwWidth;x++)
			{
				float fVal=SRFDataOutput.CalcDisplacementDist(x,y,inpData->m_Properties.m_bDisplaceNPatch,inpData->m_Properties.m_bDisplaceQuadratic);

				int g=0;

				if(fVal!=FLT_MAX)
				{
					if(fVal<0.0f)
						if(!inpData->m_Properties.m_bDisplaceSigned)fVal=0.0f;			// clip nagative Values

					g=(int)( (fVal*fDispScale+fDispTranslate)*fDispOutputScale );

					if(g<0)g=0;if(g>255)g=255;															// bound in range [0..255]
				}

				writeMap[3*(x+y*dwWidth)  ]= g;
				writeMap[3*(x+y*dwWidth)+1]= g;
				writeMap[3*(x+y*dwWidth)+2]= g;
			}

			inpData->m_BitMapOutput.SaveAndClear(inpData->ExtendFilename(inpData->m_Properties.m_szOutputFilename+"_DM"+inpData->m_Properties.m_szOutputExtension).c_str());
		}	// end Displace
	}

#endif // !RESOURCECOMPILER_ONLY

	SRFDataOutput.FreeData();

  // clean up
  delete pLowMesh;
  delete pHighMesh;
	delete pCageMesh;

	return true;
}












// raytracing, gather unsorted hits
// call ClearListBefore and GetHitList afterwards
void CPbLowMesh::GatherRayHits( const Vec3 invStart, const Vec3 invEnd, CIntInfoLowList &outIntersections, const CPbLowTri *inpIgnoreObject )
{
	Vec3 vDir=invEnd-invStart;

	float fLength=vDir.Length();
	
	vDir.Normalize();	// Baustelle normalize?

	CEveryObjectOnlyOnceSTLVector<CPbLowTri> sTriangleSink(invStart,vDir,fLength,outIntersections,inpIgnoreObject);			// Baustelle - static may be faster

	m_Raster.GatherRayHitsTo(invStart,invEnd,sTriangleSink);
}



// raytracing, gather unsorted hits
// call ClearListBefore and GetHitList afterwards
void CPbHighMesh::GatherRayHits( const Vec3 invStart, const Vec3 invEnd, CIntInfoHighList &outIntersections, const CPbHighTri *inpIgnoreObject )
{
	Vec3 vDir=invEnd-invStart;

	float fLength=vDir.Length();
	
	vDir/=fLength;									// normalize

	CEveryObjectOnlyOnceSTLVector<CPbHighTri> sTriangleSink(invStart,vDir,fLength,outIntersections,inpIgnoreObject);			// Baustelle - static may be faster

	m_Raster.GatherRayHitsTo(invStart,invEnd,sTriangleSink);
}


// raytracing, test if there is a hit or not
bool CPbHighMesh::CalcIntersection( const Vec3 invStart, const Vec3 invEnd, const CPbHighTri *inpIgnoreObject )
{
	Vec3 vDir=invEnd-invStart;

	float fLength=vDir.Length();

	vDir/=fLength;									// normalize

	CEveryObjectOnlyOnceBool<CPbHighTri> sTriangleSink(invStart,vDir,fLength,inpIgnoreObject);

	m_Raster.GatherRayHitsTo(invStart,invEnd,sTriangleSink);

	return sTriangleSink.GetResult();
}


std::string CPbHighMesh::BuildRayAccelerator( const bool inbDebug )
{
	Vec3 vMin,vMax;

	CalcBoundingVolume(vMin,vMax);

	m_Raster.Init(vMin,vMax,m_nNumTris,USE_RASTERCUBE_ACCELERATOR_MAGNIFIER);

	for(uint32 i=0;i<m_nNumTris;i++)
	{
#ifdef 	USE_MEMORY_BLOCKS	//win32 memory optimization
			CPbHighTri&	pTri = m_pMesh[i/m_nObjPerBlock][i%m_nObjPerBlock];
#else
			CPbHighTri&	pTri = m_pMesh[i];
#endif //USE_MEMORY_BLOCKS
			m_Raster.PutInTriangle(pTri.m_Verts,&pTri); //fix pTri
	}

	m_Raster.PreProcess(inbDebug);

	for(uint32 i=0;i<m_nNumTris;i++)
	{
#ifdef 	USE_MEMORY_BLOCKS	//win32 memory optimization

			CPbHighTri&	pTri = m_pMesh[i/m_nObjPerBlock][i%m_nObjPerBlock];
#else
			CPbHighTri&	pTri = m_pMesh[i];
#endif //USE_MEMORY_BLOCKS

		m_Raster.PutInTriangle(pTri.m_Verts,&pTri);
	}
//	m_Raster.Compress();  deactivated because it seems there is still a bug


	char str[256];
	DWORD iSize[3];
	DWORD byteSize=m_Raster.CalcMemoryConsumption(iSize);

	sprintf(str,"Acceleration scheme: RasterCube %dx%dx%d ~%d bytes",iSize[0],iSize[1],iSize[2],byteSize);

	std::string sRet=str;
/*
	if(inbDebug)
		sRet+=TestRasterCube();
*/
	return sRet;
}



std::string CPbLowMesh::BuildRayAccelerator( const bool inbDebug )
{
	Vec3 vMin,vMax;

	CalcBoundingVolume(vMin,vMax);

	m_Raster.Init(vMin,vMax,m_nNumTris,USE_RASTERCUBE_ACCELERATOR_MAGNIFIER);

	for(uint32 i=0;i<m_nNumTris;i++)
		m_Raster.PutInTriangle(m_pMesh[i].m_Verts,&m_pMesh[i]);

	m_Raster.PreProcess(inbDebug);

	for(uint32 i=0;i<m_nNumTris;i++)
		m_Raster.PutInTriangle(m_pMesh[i].m_Verts,&m_pMesh[i]);

//	m_Raster.Compress();  deactivated because it seems there is still a bug


	char str[256];
	DWORD iSize[3];
	DWORD byteSize=m_Raster.CalcMemoryConsumption(iSize);

	sprintf(str,"Acceleration scheme: RasterCube %dx%dx%d ~%d bytes",iSize[0],iSize[1],iSize[2],byteSize);

	std::string sRet=str;
/*
	if(inbDebug)
		sRet+=TestRasterCube();
*/
	return sRet;
}

