#pragma once

#include <float.h>												// FLT_MAX
#include "RasterCube.h"										// CPossibleIntersectionSink<>

#define ALREADY_TESTED_ARRARY_SIZE	1024



template <class TElement, class TTri>
class CEveryObjectOnlyOnce :public CPossibleIntersectionSink<TTri *>
{
public:

	//! constructor
	//! /param invFrom
	//! /param invDir
	//! /param infRayLength
	//! /param inpIgnore for bias free rayshooting from object surface, may be 0
	CEveryObjectOnlyOnce( const Vec3 &invFrom, const Vec3 &invDir, float infRayLength, const TElement inIgnore )
	{
		m_vFrom=invFrom;
		m_vDir=invDir;
		m_fMaxRayLength=infRayLength;

		if(inIgnore)
		{
			m_dwAlreadyTested=1;
			m_arrAlreadyTested[0]=inIgnore;
		}
		else
		{
			m_dwAlreadyTested=0;
		}
	}

protected:	// --------------------------------------------

	Vec3							m_vFrom;																					//!< position in object space 
	Vec3							m_vDir;																						//!< direction in object space
	TElement					m_arrAlreadyTested[ALREADY_TESTED_ARRARY_SIZE];		//!< [0..ALREADY_TESTED_ARRARY_SIZE-1], this is faster than a set<> at least for typical amounts
	DWORD							m_dwAlreadyTested;																//!< 0..ALREADY_TESTED_ARRARY_SIZE
	float							m_fMaxRayLength;																	//!<

	bool InsertAlreadyTested( const TElement &inObject )
	{
		const TElement *ptr=m_arrAlreadyTested;

		for(uint32 i=0;i<m_dwAlreadyTested;i++)
			if(*ptr++==inObject)
				return true;

		if(m_dwAlreadyTested<ALREADY_TESTED_ARRARY_SIZE-1) 
			m_arrAlreadyTested[m_dwAlreadyTested++]=inObject;
//			else assert(0);		// ALREADY_TESTED_ARRARY_SIZE is not big enougth

		return false;
	}
};



// find all intersection within the range
template <class TProvider, class TElement, class TTri>
class CEveryObjectOnlyOnceSTLVector :public CEveryObjectOnlyOnce<TElement,TTri>
{
public:

	//! constructor
	//! /param invFrom
	//! /param invDir
	//! /param infRayLength
	//! /param inIgnore for bias free rayshooting from object surface, may be 0
	CEveryObjectOnlyOnceSTLVector( TProvider &rProvider, const Vec3 &invFrom, const Vec3 &invDir, float infRayLength,
		std::vector<CIntersInfo<TTri> > &inOutput, const TElement inIgnore )
		:CEveryObjectOnlyOnce<TElement,TTri>(invFrom,invDir,infRayLength,inIgnore), m_FoundIntersections(inOutput), m_rProvider(rProvider)
	{
	}

	//! /param inObject 
	//! /param inoutfRayMaxDist in and out value for max. shooting distance or FLT_MAX, if you found a interesection you may update this because further intersections are no longer interesting
	//! /return true=do further processiung, false=stop I found what I need (e.g. any intersection for shadow testing)
//	bool ReturnElement( TTri * &inObject, float &inoutfRayMaxDist )
	bool ReturnElement( const TElement dwIndex, float &inoutfRayMaxDist )
	{
//		TTri *inObject = m_rProvider.GetTrianglePtr(dwIndex);

		if(!InsertAlreadyTested(dwIndex))
		{
			TTri *pTri;

			float fPos = m_rProvider.CalcTriIntersectionFromDir(dwIndex,m_vFrom,m_vDir,pTri);

			if(fPos!=FLT_MAX)
			{
				assert(fPos>=0);

				if(fPos<m_fMaxRayLength)
				{
					CIntersInfo<TTri> inters;

					inters.m_fDist=fPos;
					inters.m_Point=m_vFrom+inters.m_fDist*m_vDir;
					inters.m_dwObjectIndex=dwIndex;
					inters.m_pTri=pTri;

					m_FoundIntersections.push_back(inters);
				}

/*
				// take the nearest intersection
				if(fPos<m_fMaxRayLength)
				{
					m_FoundIntersection.m_SubObjectNo=inObject;

					inoutfRayMaxDist=m_fMaxRayLength;
				}
*/
			}
		}
		return true;	// do further intersections
	}

private:	// --------------------------------------------

	std::vector<CIntersInfo<TTri> > &			m_FoundIntersections;															//!< reference to the output STL vector - optimizable
	TProvider &														m_rProvider;
};






//! find if there is a intersection in the range
template <class TProvider, class TElement, class TTri>
class CEveryObjectOnlyOnceBool :public CEveryObjectOnlyOnce<TElement,TTri>
{
public:
	//! constructor
	//! /param invFrom start position in worldspace
	//! /param invDir
	//! /param infRayLength
	//! /param inpIgnore for bias free rayshooting from object surface, may be 0
	CEveryObjectOnlyOnceBool( TProvider &rProvider, const Vec3 &invFrom, const Vec3 &invDir, float infRayLength, const TElement inIgnore )
		:CEveryObjectOnlyOnce<TElement,TTri>(invFrom,invDir,infRayLength,inIgnore), m_rProvider(rProvider)
	{
		m_bResult=false;
	}

	//! /param inObject 
	//! /param inoutfRayMaxDist in and out value for max. shooting distance or FLT_MAX, if you found a interesection you may update this because further intersections are no longer interesting
	//! /return true=do further processiung, false=stop I found what I need (e.g. any intersection for shadow testing)
//	bool ReturnElement( TTri * &inObject, float &inoutfRayMaxDist )
	bool ReturnElement( const uint32 dwIndex, float &inoutfRayMaxDist )
	{
//		TTri *inObject = m_rProvider.GetTrianglePtr(dwIndex);

		if(!InsertAlreadyTested(dwIndex))
		{
			TTri *pTri;
			float fPos = m_rProvider.CalcTriIntersectionFromDir(dwIndex,m_vFrom,m_vDir,pTri);

			if(fPos!=FLT_MAX)
			{
				assert(fPos>=0);

				if(fPos<m_fMaxRayLength)
				{
					m_bResult=true;
					return false;		// no further intersection, one hit is enougth
				}
			}
		}
		
		return true;	// try further intersections, this one was no real one
	}

	//! /return reference to the STL list of all found intersections
	bool GetResult()
	{
		return m_bResult;
	}

private: // --------------------------------------------

	bool																	m_bResult;										//!< true=there was a hit, false otherwise
	TProvider &														m_rProvider;
};





/*
template <class TTri>
class CEveryObjectDebug :public CPossibleIntersectionSink<TTri *>
{
public:

	//! constructor
	//! /param invFrom
	//! /param invDir
	//! /param infRayLength
	//! /param inpTri, may not be 0
	CEveryObjectDebug( TTri *inpTri )
	{
		assert(inpTri);

		m_pTriPointer=inpTri;
		m_bFoundInThisBucket=false;
		m_dwBucketsOk=0;
		m_dwBucketsFailed=0;
	}

	//! /param inObject 
	//! /param inoutfRayMaxDist in and out value for max. shooting distance or FLT_MAX, if you found a interesection you may update this because further intersections are no longer interesting
	//! /return true=do further processiung, false=stop I found what I need (e.g. any intersection for shadow testing)
//	bool ReturnElement( TTri * &inObject, float &inoutfRayMaxDist )
	bool ReturnElement( const uint32 dwIndex, float &inoutfRayMaxDist )
	{
		if(inObject==m_pTriPointer)
			m_bFoundInThisBucket=true;

		return true;	// try further intersections, this one was no real one
	}

	void EndReturningBucket()
	{
		if(m_bFoundInThisBucket)
			m_dwBucketsOk++;
		 else 
			m_dwBucketsFailed++;

		m_bFoundInThisBucket=false;
	}

	// --------------------------------------------

	DWORD							m_dwBucketsOk;																		//!<
	DWORD							m_dwBucketsFailed;																//!<

protected: // --------------------------------------------

	TTri *						m_pTriPointer;																		//!< triangle that has to be in every bucket on the way of the line
	bool							m_bFoundInThisBucket;															//!<
};
*/
