#pragma once

#include <float.h>												// FLT_MAX
#include "RasterCube.h"										// CPossibleIntersectionSink<>

#define ALREADY_TESTED_ARRARY_SIZE	1024



class CEveryObjectOnlyOnce :public CPossibleIntersectionSink<CPbTri *>
{
public:

	//! constructor
	//! /param invFrom
	//! /param invDir
	//! /param infRayLength
	//! /param inpIgnore for bias free rayshooting from object surface, may be 0
	CEveryObjectOnlyOnce( const Vec3 &invFrom, const Vec3 &invDir, float infRayLength, const CPbTri *inpIgnore )
	{
		m_vFrom=invFrom;
		m_vDir=invDir;
		m_fMaxRayLength=infRayLength;

		if(inpIgnore)
		{
			m_dwAlreadyTested=1;
			m_arrAlreadyTested[0]=inpIgnore;
		}
		else
		{
			m_dwAlreadyTested=0;
		}
	}

	// --------------------------------------------

protected:
	Vec3							m_vFrom;																					//!< position in object space 
	Vec3							m_vDir;																						//!< direction in object space
	const CPbTri *		m_arrAlreadyTested[ALREADY_TESTED_ARRARY_SIZE];		//!< [0..ALREADY_TESTED_ARRARY_SIZE-1], this is faster than a set<> at least for typical amounts
	DWORD							m_dwAlreadyTested;																//!< 0..ALREADY_TESTED_ARRARY_SIZE
	float							m_fMaxRayLength;																	//!<

	bool InsertAlreadyTested( CPbTri * &inObject )
	{
		const CPbTri * *ptr=m_arrAlreadyTested;

		for(DWORD i=0;i<m_dwAlreadyTested;i++)
			if(*ptr++==inObject)return(true);

		if(m_dwAlreadyTested<ALREADY_TESTED_ARRARY_SIZE-1) m_arrAlreadyTested[m_dwAlreadyTested++]=inObject;
//			else assert(0);		// ALREADY_TESTED_ARRARY_SIZE is not big enougth

		return(false);
	}
};



// find all intersection within the range
class CEveryObjectOnlyOnceSTLList :public CEveryObjectOnlyOnce
{
public:

	//! constructor
	//! /param invFrom
	//! /param invDir
	//! /param infRayLength
	//! /param inpIgnore for bias free rayshooting from object surface, may be 0
	CEveryObjectOnlyOnceSTLList( const Vec3 &invFrom, const Vec3 &invDir, float infRayLength, CIntInfoList &inOutput, const CPbTri *inpIgnore )
		:CEveryObjectOnlyOnce(invFrom,invDir,infRayLength,inpIgnore), m_FoundIntersections(inOutput)
	{
	}

	//! /param inObject 
	//! /param inoutfRayMaxDist in and out value for max. shooting distance or FLT_MAX, if you found a interesection you may update this because further intersections are no longer interesting
	//! /return true=do further processiung, false=stop I found what I need (e.g. any intersection for shadow testing)
	bool ReturnElement( CPbTri * &inObject, float &inoutfRayMaxDist )
	{
		if(!InsertAlreadyTested(inObject))
		{
			float fPos=inObject->CalcIntersectionFromDir(m_vFrom,m_vDir);

			if(fPos!=FLT_MAX)
			{
				assert(fPos>=0);

				if(fPos<m_fMaxRayLength)
				{
					CIntersInfo inters;

					inters.m_fDist=fPos;
					inters.m_Point=m_vFrom+inters.m_fDist*m_vDir;
					inters.m_pTri=inObject;

					m_FoundIntersections.push_back(inters);
				}

/*
				// take the nearest intersection
				if(fPos<m_fMaxRayLength)
				{
					m_FoundIntersection.m_SubObjectNo=inObject;

					inoutfRayMaxDist=m_fMaxRayLength;
				}
*/
			}
		}
		return(true);	// do further intersections
	}

	// --------------------------------------------

private:
	CIntInfoList &			m_FoundIntersections;															//!< reference to the output STL vector - optimizable
};






//! find if there is a intersection in the range
class CEveryObjectOnlyOnceBool :public CEveryObjectOnlyOnce
{
public:
	//! constructor
	//! /param invFrom start position in worldspace
	//! /param invDir
	//! /param infRayLength
	//! /param inpIgnore for bias free rayshooting from object surface, may be 0
	CEveryObjectOnlyOnceBool( const Vec3 &invFrom, const Vec3 &invDir, float infRayLength, const CPbTri *inpIgnore )
		:CEveryObjectOnlyOnce(invFrom,invDir,infRayLength,inpIgnore)
	{
		m_bResult=false;
	}

	//! /param inObject 
	//! /param inoutfRayMaxDist in and out value for max. shooting distance or FLT_MAX, if you found a interesection you may update this because further intersections are no longer interesting
	//! /return true=do further processiung, false=stop I found what I need (e.g. any intersection for shadow testing)
	bool ReturnElement( CPbTri * &inObject, float &inoutfRayMaxDist )
	{
		if(!InsertAlreadyTested(inObject))
		{
			float fPos=inObject->CalcIntersectionFromDir(m_vFrom,m_vDir);

			if(fPos!=FLT_MAX)
			{
				assert(fPos>=0);

				if(fPos<m_fMaxRayLength)
				{
					m_bResult=true;
					return(false);		// no further intersection, one hit is enougth
				}
			}
		}
		
		return(true);	// try further intersections, this one was no real one
	}

	//! /return reference to the STL list of all found intersections
	bool GetResult( void )
	{
		return(m_bResult);
	}


	// --------------------------------------------

private:
	bool							m_bResult;																	//!< true=there was a hit, false otherwise
};






class CEveryObjectDebug :public CPossibleIntersectionSink<CPbTri *>
{
public:

	//! constructor
	//! /param invFrom
	//! /param invDir
	//! /param infRayLength
	//! /param inpTri, may not be 0
	CEveryObjectDebug( CPbTri *inpTri )
	{
		assert(inpTri);

		m_pTriPointer=inpTri;
		m_bFoundInThisBucket=false;
		m_dwBucketsOk=0;
		m_dwBucketsFailed=0;
	}

	//! /param inObject 
	//! /param inoutfRayMaxDist in and out value for max. shooting distance or FLT_MAX, if you found a interesection you may update this because further intersections are no longer interesting
	//! /return true=do further processiung, false=stop I found what I need (e.g. any intersection for shadow testing)
	bool ReturnElement( CPbTri * &inObject, float &inoutfRayMaxDist )
	{
		if(inObject==m_pTriPointer)m_bFoundInThisBucket=true;

		return(true);	// try further intersections, this one was no real one
	}

	void EndReturningBucket( void )
	{
		if(m_bFoundInThisBucket)
			m_dwBucketsOk++;
		 else 
			m_dwBucketsFailed++;

		m_bFoundInThisBucket=false;
	}

	// --------------------------------------------

	DWORD							m_dwBucketsOk;																		//!<
	DWORD							m_dwBucketsFailed;																//!<

protected:
	CPbTri *					m_pTriPointer;																		//!< triangle that has to be in every bucket on the way of the line
	bool							m_bFoundInThisBucket;															//!<
};
