// ResourceCompiler.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"
#include <time.h>
#include <DbgHelp.h>
#include <io.h>
#include "ResourceCompiler.h"
#include "CmdLine.h"
#include "Config.h"
#include "CfgFile.h"
#include "FileUtil.h"
#include "IConvertor.h"
#include "FileUtil.h"
#include "mdump.h"
#include "CpuInfo.h"

#pragma comment( lib, "Version.lib" )

static const char *RC_FILENAME_LOG=							"rc_log.log";
static const char *RC_FILENAME_WARNINGS=				"rc_log_warnings.log";
static const char *RC_FILENAME_ERRORS=					"rc_log_errors.log";
static const char *RC_FILENAME_FILEDEP=					"rc_stats_filedependencies.log";
//static const char *RC_FILENAME_MATDEP=					"rc_stats_materialdependencies.log";
static const char *RC_FILENAME_PRESETUSAGE=			"rc_stats_presetusage.log";
static const char *RC_FILENAME_XLS_FILESIZES=		"rc_stats_filesizes_xls.log";		// in format that can be easily read from Excel



//////////////////////////////////////////////////////////////////////////
// Globals.
//////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
// If not in static library.
//#ifdef CRY_STRING
//int sEmptyStringBuffer[] = { -1, 0, 0, 0 };
//string::StrHeader* string::m_emptyStringData = (string::StrHeader*)&sEmptyStringBuffer;
//wstring::StrHeader* wstring::m_emptyStringData = (wstring::StrHeader*)&sEmptyStringBuffer;
//#endif //CRY_STRING

// Determines whether a path to a file system object such as a file or directory is valid

BOOL RCPathFileExists (const char* szPath)
{
	DWORD dwAttr = GetFileAttributes (szPath);
	return (dwAttr != 0xFFFFFFFF);
}


//////////////////////////////////////////////////////////////////////////
// ResourceCompiler implementation.
//////////////////////////////////////////////////////////////////////////
ResourceCompiler::ResourceCompiler()
{
	m_hLogFile=0;
	m_hErrorLogFile=0;
	m_hWarningLogFile=0;
	m_bWarningHeaderLine=false;
	m_bErrorHeaderLine=false;
	m_bStatistics = false;
	m_bQuiet = false;
	m_iThreads = 1;
	m_threadNumberTLSIndex = -1;
	InitializeLogThreadNumbering();
}

ResourceCompiler::~ResourceCompiler()
{
	// close files if open
	if(m_hLogFile)fclose(m_hLogFile);m_hLogFile=0;
	if(m_hErrorLogFile)fclose(m_hErrorLogFile);m_hErrorLogFile=0;
	if(m_hWarningLogFile)fclose(m_hWarningLogFile);m_hWarningLogFile=0;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::RegisterConvertor( IConvertor *conv )
{
	m_extensionManager.RegisterConvertor( conv, this );
}

//////////////////////////////////////////////////////////////////////////
FILE*	ResourceCompiler::OpenFile( const char *filename,const char *mode )
{
	FILE *file = fopen(filename,mode);
	// check if read only.
	return file;
}

IRCLog *ResourceCompiler::GetIRCLog()
{
	return(this);
}

// returns the file unix time - the latest of modification and creation times
DWORD ResourceCompiler::GetFileUnixTimeMax (const char* filename)
{
	FILETIME ftWrite, ftCreate;
	if (GetFileTime(filename, &ftWrite, &ftCreate))
	{
		return max (FileUtil::FiletimeToUnixTime(ftWrite),FileUtil::FiletimeToUnixTime(ftCreate));
	}
	else
		return 0;
}

// returns the file unix time - the earliest of modification and creation times
DWORD ResourceCompiler::GetFileUnixTimeMin (const char* filename)
{
	FILETIME ftWrite, ftCreate;
	if (GetFileTime(filename, &ftWrite, &ftCreate))
	{
		return min (FileUtil::FiletimeToUnixTime(ftWrite),FileUtil::FiletimeToUnixTime(ftCreate));
	}
	else
		return 0;
}


//////////////////////////////////////////////////////////////////////////
bool ResourceCompiler::GetFileTime( const char *filename,FILETIME *ftimeModify, FILETIME*ftimeCreate )
{
	WIN32_FIND_DATA FindFileData;
	HANDLE hFind;
	hFind = FindFirstFile( filename,&FindFileData );
  if (hFind == INVALID_HANDLE_VALUE)
	{
		return false;
	}
	FindClose(hFind);
	if (ftimeCreate)
	{
		ftimeCreate->dwLowDateTime = FindFileData.ftCreationTime.dwLowDateTime;
		ftimeCreate->dwHighDateTime = FindFileData.ftCreationTime.dwHighDateTime;
	}
	if (ftimeModify)
	{
		ftimeModify->dwLowDateTime = FindFileData.ftLastWriteTime.dwLowDateTime;
		ftimeModify->dwHighDateTime = FindFileData.ftLastWriteTime.dwHighDateTime;
	}
	return true;
}

//////////////////////////////////////////////////////////////////////////
const char* ResourceCompiler::GetSectionName( Platform platform ) const
{
	switch (platform)
	{
	case PLATFORM_PC:				return "PC";
	case PLATFORM_XBOX:			return "XBOX";
	case PLATFORM_PS2:			return "PS2";
	case PLATFORM_GAMECUBE:	return "GAMECUBE";
	default:
		// unknown platform.
		MessageBoxError( _T("Section name requested for unknown platform") );
		assert(0);
	}
	return "";
}

void ResourceCompiler::RemoveOutputFiles()
{
	DeleteFile(m_exePath+RC_FILENAME_LOG);
	DeleteFile(m_exePath+RC_FILENAME_WARNINGS);
	DeleteFile(m_exePath+RC_FILENAME_ERRORS);
	DeleteFile(m_exePath+RC_FILENAME_FILEDEP);
//	DeleteFile(RC_FILENAME_MATDEP);
	DeleteFile(m_exePath+RC_FILENAME_PRESETUSAGE);
	DeleteFile(m_exePath+RC_FILENAME_XLS_FILESIZES);
}


//////////////////////////////////////////////////////////////////////////
// Returns true if successfully converted at least one file
bool ResourceCompiler::Compile( Platform platform,IConfig *config,const char *filespec )
{
	RemoveOutputFiles();		// to remove old files for less confusion

	AutoDetection(); // detect multithreading 

	if (config->HasKey("statistics"))
		m_bStatistics = true;
	else
		m_bStatistics = false;

	m_bWarningsAsErrors = config->HasKey("WX");

	if(!config->HasKey("logfiles"))
	{
		m_hLogFile=fopen(m_exePath+RC_FILENAME_LOG,"wb+");
	}
	{
		m_hWarningLogFile=fopen(m_exePath+RC_FILENAME_WARNINGS,"wb+");
		m_hErrorLogFile=fopen(m_exePath+RC_FILENAME_ERRORS,"wb+");
	}

	DWORD dwFileSpecAttr = GetFileAttributes (filespec);

	std::vector<CString> arrFiles; // files to convert, with relative paths

	bool bRecursive = config->GetAs<bool>("recursive", true);
	
	m_presets = new CfgFile();
	CString presetcfg;

	if(!config->Get("presetcfg", presetcfg))
	{
		char str[512];

		sprintf(str,"No preset configuration defined (e.g. presetcfg=rc_presets_pc.ini)");

		Log("%s",str);

		static bool bFirstTime=true;

		if(bFirstTime)
			MessageBox(0,str,"ResourceCompiler Error",MB_OK|MB_ICONERROR);
	
		bFirstTime=false;

		// it's better to have resource not working without that info
		return false;
	}
	else if(!m_presets->Load( Path::Make(m_exePath.c_str(),presetcfg)) )
	{
		char str[512];

		sprintf(str,"Failed to read preset configuration %s, exiting...", presetcfg.GetString());

		Log("%s",str);

		static bool bFirstTime=true;

		if(bFirstTime)
			MessageBox(0,str,"ResourceCompiler Error",MB_OK|MB_ICONERROR);
	
		bFirstTime=false;

		// it's better to have resource not working without that info
		return false;
	};

	Log("used preset configuration: presetcfg=%s",presetcfg.GetString());

	CString path = Path::GetPath(filespec);
	if (dwFileSpecAttr == 0xFFFFFFFF)
	{
		// there's no such file; so, this is probably a mask:
		// path\*.mask
		
		Log("");
		Log("Scanning drectory ...");
		
		// Scan all files matching filespec.
		FileUtil::ScanDirectory( path,Path::GetFile(filespec),arrFiles, bRecursive);

		Log("");
	}
	else
	if (dwFileSpecAttr & FILE_ATTRIBUTE_DIRECTORY)
	{
		path = Path::AddBackslash(filespec);
		
		// it's a directory; the mask can be found via /file=... option
		FileUtil::ScanDirectory(path, config->GetAs<CString>("file", "*.*"), arrFiles, bRecursive);
	}
	else
		arrFiles.push_back(Path::GetFile(filespec));

	if (arrFiles.empty())
	{
		LogError( "The system cannot find the file specified, 0 file(s) converted" );
		return false;
	}
	
	// determine the target output path (may be a different directory structure)
	// if none is specified, the target is the same as the source, as before.
	CString targetroot;
	if (!config->Get( "targetroot", targetroot ))
		targetroot = path;
	
	targetroot = Path::AddBackslash(targetroot);

	// Determine the number of threads to use.
//	int threadCount = 1;
	//CString threadString;

	//if (config->Get("threads", threadString))   
	//{
	//	m_bUseMultuthreading = false;
	//	m_iThreads = 1;
	//}

	int threadCount = 1;
	CString threadString;
	if (config->Get("threads", threadString))
	{
		char* endptr;
		const char* str = threadString.GetString();
		threadCount = strtol(str, &endptr, 10);
		if (endptr == str) {
			LogWarning("/threads specified, but number of threads not given - try /threads=4 eg. Using %d threads.", m_iThreads);
			threadCount = m_iThreads;
		}
		else if (threadCount < 1)
			LogWarning("%d threads specified, falling back to single-threading.", threadCount);
		if (threadCount < 1)
			threadCount = 1;
		const int maxThreads = m_iThreads;
		if (threadCount > maxThreads)
		{
			LogWarning("%d threads specified, clamping to %d.", threadCount, maxThreads);
			threadCount = maxThreads;
		}
	}

	m_iThreads = threadCount;

	// these are the files that couldn't be converted
	std::vector<CString> arrNonConvertedFiles;

	int nTimer = GetTickCount();

	// Split up the files based on the convertor they are to use.
	typedef std::map<IConvertor*, std::vector<CString> > FileConvertorMap;
	FileConvertorMap fileConvertorMap;
	size_t i, iSize=arrFiles.size();
	for (i = 0; i < iSize; i++)
	{
		CString extension = Path::GetExt(arrFiles[i]);
		{
			CString sOverWriteExtension;
			if(config->Get("overwriteextension",sOverWriteExtension))
				extension=sOverWriteExtension;
		}
		extension.MakeLower();

		IConvertor* convertor = m_extensionManager.FindConvertor(platform, extension.GetString());
		FileConvertorMap::iterator convertorPosition = fileConvertorMap.find(convertor);
		if (convertorPosition == fileConvertorMap.end())
			convertorPosition = fileConvertorMap.insert(std::make_pair(convertor, std::vector<CString>())).first;
		(*convertorPosition).second.push_back(arrFiles[i]);
	}

	// Loop through all the convertors that we need to invoke.
	for (FileConvertorMap::iterator convertorPosition = fileConvertorMap.begin(), convertorEnd = fileConvertorMap.end(); convertorPosition != convertorEnd; ++convertorPosition)
	{
		IConvertor* convertor = (*convertorPosition).first;
		std::vector<CString>& arrConvertorFiles = (*convertorPosition).second;

		if (convertor)
		{
			// Check whether this convertor is thread-safe.
			if (threadCount != 1 && !convertor->SupportsMultithreading())
			{
				LogWarning("/threads specified, but convertor does not support multi-threading. Falling back to single-threading.");
				threadCount = 1;
			}

			// Decide whether to compile files using single- or multi-threading.
			if (threadCount == 1)
				CompileFilesSingleThreaded(path, targetroot, platform, config, arrConvertorFiles, convertor, arrNonConvertedFiles);
			else
				CompileFilesMultiThreaded(threadCount, path, targetroot, platform, config, arrConvertorFiles, convertor, arrNonConvertedFiles);
		}
	}

	int numFilesConverted = arrFiles.size() - arrNonConvertedFiles.size();

	nTimer = GetTickCount() - nTimer;
	char szTimeMsg[128] ;
	szTimeMsg[0] = '\0';
	if (nTimer > 500)
		sprintf (szTimeMsg, " in %.1f sec", nTimer/1000.0f);

	Log("");
	
	if (arrNonConvertedFiles.empty())
		Log ("%d file%s processed%s.", arrFiles.size(), arrFiles.size()>1?"s":"",szTimeMsg);
	else
	{
		Log("");
		Log ( "%d of %d file%s converted%s. Couldn't convert the following file%s:", numFilesConverted, arrFiles.size(), arrFiles.size() > 1 ? "s":"", szTimeMsg,arrNonConvertedFiles.size()>1?"s":"");
		Log("");
		for (i = 0; i < arrNonConvertedFiles.size(); ++i)
			Log ( "   %s", arrNonConvertedFiles[i]);
		Log("");
	}

	delete m_presets;

	return numFilesConverted > 0;
}

bool ResourceCompiler::CompileFilesSingleThreaded(const CString& path, const CString& targetroot, Platform platform,IConfig* config,const std::vector<CString>&files,IConvertor* convertor, std::vector<CString>& nonConvertedFiles)
{
	char szRCPath[1000];
	GetModuleFileName(NULL, szRCPath, sizeof(szRCPath));
	CString exePath = Path::GetPath(szRCPath);

	convertor->Init(config, this, exePath.GetString());

	ICompiler* compiler = convertor->CreateCompiler();

	std::vector<CString> outOfMemoryFiles;
	bool result = CompileFileArray(path, targetroot, platform, config, files, compiler, nonConvertedFiles, outOfMemoryFiles, &m_Files);

	// In single-threaded mode if we didnt have enough memory to compile a file the first time, we won't the second time either.
	nonConvertedFiles.insert(nonConvertedFiles.end(), outOfMemoryFiles.begin(), outOfMemoryFiles.end());

	compiler->Release();

	convertor->DeInit();

	return result;
}

struct ThreadData
{
	ResourceCompiler* rc;
	CString path;
	CString targetroot;
	Platform platform;
	Config config;
	std::vector<CString> files;
	ICompiler* compiler;
	std::vector<CString> nonConvertedFiles;
	std::vector<CString> outOfMemoryFiles;
	int threadIndex;
	std::vector<CFileStats> fileStats;
};

unsigned int WINAPI ResourceCompiler::ThreadFunc(void* threadDataMemory)
{
	ThreadData* data = static_cast<ThreadData*>(threadDataMemory);

	// Initialize the thread local storage, so the log can prepend the thread index to each line.
	if (data->rc->m_threadNumberTLSIndex >= 0)
		TlsSetValue(data->rc->m_threadNumberTLSIndex, &data->threadIndex);

	return data->rc->CompileFileArray(data->path, data->targetroot, data->platform, &data->config, data->files, data->compiler, data->nonConvertedFiles, data->outOfMemoryFiles, &data->fileStats);
}

bool ResourceCompiler::CompileFilesMultiThreaded(int threadCount, const CString& path, const CString& targetroot, Platform platform,IConfig* config,const std::vector<CString>&files,IConvertor* convertor, std::vector<CString>& nonConvertedFiles)
{
	if (threadCount <= 0 || files.empty())
		return true;

	Log("Spawning %d threads.", threadCount);

	char szRCPath[1000];
	GetModuleFileName(NULL, szRCPath, sizeof(szRCPath));
	CString exePath = Path::GetPath(szRCPath);

	// Initialize the convertor once for each thread.
	convertor->Init(config, this, exePath.GetString());

	// Initialize the thread data for each thread.
	std::vector<ThreadData> threadData(threadCount);
	for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex)
	{
		threadData[threadIndex].rc = this;
		threadData[threadIndex].path = path;
		threadData[threadIndex].targetroot = targetroot;
		threadData[threadIndex].platform = platform;
		threadData[threadIndex].config.SetConfigKeyRegistry(this);
		threadData[threadIndex].config.Merge(config);
		threadData[threadIndex].compiler = convertor->CreateCompiler();
		threadData[threadIndex].threadIndex = threadIndex + 1;

		threadData[threadIndex].files.reserve(files.size() / threadCount + 1);
	}

	// Assign files to each thread.
	for (int fileIndex = 0, fileCount = int(files.size()); fileIndex < fileCount; ++fileIndex)
	{
		int threadIndex = fileIndex % threadCount;
		threadData[threadIndex].files.push_back(files[fileIndex]);
	}

	// Spawn the threads.
	std::vector<HANDLE> threads(threadCount);
	for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex)
	{
		threads[threadIndex] = (HANDLE)_beginthreadex( 
			0,														//void *security,
			0,														//unsigned stack_size,
			ThreadFunc,										//unsigned ( *start_address )( void * ),
			&threadData[threadIndex],			//void *arglist,
			0,														//unsigned initflag,
			0);														//unsigned *thrdaddr 
	}

	// Wait until all the threads have exitted.
	WaitForMultipleObjects(threads.size(), &threads[0], TRUE, INFINITE);

	// Release all the compiler objects.
	for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex)
		threadData[threadIndex].compiler->Release();

	// Compile all statistics from all threads.
	for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex)
		m_Files.insert(m_Files.end(), threadData[threadIndex].fileStats.begin(), threadData[threadIndex].fileStats.end());

	// If we ran out of memory when processing files, process them again in one thread (since we may
	// have run out of memory just because we had multiple threads).
	{
		std::vector<CString> retryFiles;
		for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex)
			retryFiles.insert(retryFiles.end(), threadData[threadIndex].outOfMemoryFiles.begin(), threadData[threadIndex].outOfMemoryFiles.end());

		ICompiler* compiler = convertor->CreateCompiler();

		std::vector<CString> outOfMemoryFiles;
		bool result = CompileFileArray(path, targetroot, platform, config, retryFiles, compiler, nonConvertedFiles, outOfMemoryFiles, &m_Files);

		// In single-threaded mode if we didnt have enough memory to compile a file the first time, we won't the second time either.
		nonConvertedFiles.insert(nonConvertedFiles.end(), outOfMemoryFiles.begin(), outOfMemoryFiles.end());

		compiler->Release();
	}

	// Clean up the converter.
	convertor->DeInit();

	// Collate the non-converted files.
	for (int threadIndex = 0; threadIndex < threadCount; ++threadIndex)
		nonConvertedFiles.insert(nonConvertedFiles.end(), threadData[threadIndex].nonConvertedFiles.begin(), threadData[threadIndex].nonConvertedFiles.end());

	return nonConvertedFiles.empty();
}

bool ResourceCompiler::CompileFileArray(const CString& path, const CString& targetroot, Platform platform,IConfig* config,const std::vector<CString>&files,ICompiler* compiler, std::vector<CString>& nonConvertedFiles, std::vector<CString>& outOfMemoryFiles, std::vector<CFileStats>* fileStats)
{
	// Create a copy of the main configuration.
	Config localConfig;
	localConfig.SetConfigKeyRegistry(this);
	localConfig.Merge(config);	// merge main config into local config

	for (int i = 0, count = int(files.size()); i < count; i++)
	{
		// show progress
		{
			int iPercentage=(100*i)/(count);
			char str[0x100];

			_snprintf(str, sizeof(str),"Progress: %3d%% %s",iPercentage,files[i]);

			SetConsoleTitle(str);
		}

		CString strFileName = path + files[i];
		try
		{
			if (!CompileFile(platform, &localConfig, strFileName.GetString(),targetroot.GetString(), Path::GetPath(files[i]).GetString() ,compiler, fileStats))
				nonConvertedFiles.push_back(strFileName);
		}
		catch (std::bad_alloc&)
		{
			outOfMemoryFiles.push_back(strFileName);
		}
	}

	return nonConvertedFiles.empty();
}

void ResourceCompiler::EnsureDirectoriesPresent(const char *path)
{
	DWORD dwFileSpecAttr = GetFileAttributes (path);
	if (dwFileSpecAttr == 0xFFFFFFFF && *path)
	{
		EnsureDirectoriesPresent(Path::GetPath(Path::RemoveBackslash(path)).GetString());
		if(_mkdir(path))
			Log("Creating directory failed: %s", path);

//		Log("Creating directory %s (%s)", path, _mkdir(path) ? "failed" : "ok");
	}
}
	

// makes the relative path out of any
CString NormalizePath(const char* szPath)
{
	char szCurDir[0x800]="";
	GetCurrentDirectory(sizeof(szCurDir),szCurDir);
	strcat(szCurDir, "/");

	char szFullPath[0x800];
	if (!_fullpath(szFullPath, szPath, sizeof(szFullPath)))
		strcpy (szFullPath, szPath);


	char* p, *q;
	CString sRes = szPath;
	for (p = szCurDir, q = szFullPath; *p && *q; ++p, ++q)
	{
		if (tolower(*p)==tolower(*q))
			continue;

    if ((*p=='/'||*p=='\\')&&(*q=='/'||*q=='\\'))
			continue;

		return sRes;
	}

	if (*p)
		return szPath;

	return q; // return whatever has left after truncating the leading path
}

//////////////////////////////////////////////////////////////////////////
bool ResourceCompiler::CompileFile(Platform platform, IConfig* config, const char *filename, const char *outroot, const char *filepath, ICompiler* compiler, std::vector<CFileStats>* fileStats)
{
	CmdLine cmdLine;

	if (!RCPathFileExists(filename))
		return false;

	CString sOutPath = CString(outroot)+filepath;

	// get file extension.
	CString ext = Path::GetExt(filename);

	{
		CString sOverWriteExtension;

		if(config->Get("overwriteextension",sOverWriteExtension))
			ext=sOverWriteExtension;
	}

	ext.MakeLower();

	// get key for special copy/ignore options to certain extensions
	CString extkey = "ext_";
	extkey += ext;
	CString extcommand;
	config->Get(extkey.GetString(), extcommand);
	
	if(extcommand=="ignore")
	{
		Log("Ignoring %s", filename);
		return false;
	}
		
	if(extcommand=="copy")
	{
		CString dest = sOutPath;
		dest += Path::GetFile(filename);
		if(dest!=filename)
		{
			// TODO: can compare filestamps of source and destination to avoid copy, but maybe overkill
			Log("Copying %s to %s", filename, dest.GetString());
			CopyFile(filename, dest.GetString(), false); // overwrites any existing file, same as all converters
		}
		return true;
	}

	CString sourcePath = Path::GetPath(filename);


	CfgFile CfgFile;				// file specific config file

	Config localConfig;

	localConfig.SetConfigKeyRegistry(this);

	localConfig.Merge(config);	// merge main config into local config

	// Setup conversion context.
	ConvertContext cc;

	cc.config										= &localConfig;
	cc.platform									= platform;
	cc.plattformName						= GetSectionName(platform);
	cc.pRC											= this;
	cc.sourceFileIntermediate		= Path::GetFile(filename);			// e.g. ~tmp123.tmp
	cc.sourceFileFinal					= cc.sourceFileIntermediate;				// e.g. dirt.tif
	cc.threads									= m_iThreads;
	cc.fileStats                = fileStats;

	{
		CString sOverWriteFilename;

		if(config->Get("overwritefilename",sOverWriteFilename))
			cc.sourceFileFinal = sOverWriteFilename;
	}


	cc.sourceFileFinalExtension	= ext;
	cc.sourceFolder							= Path::GetPath(filename);
	if (!config->Get("MasterFolder", cc.masterFolder))
		cc.masterFolder						= "";
	if (!cc.masterFolder.IsEmpty() && cc.masterFolder.Right(1)!="/" && cc.masterFolder.Right(1)!="\\")
		cc.masterFolder += '\\';

	cc.outputFile								= "";
	cc.m_sFileFolder						= filepath;
	cc.m_sOutputFolder					= sOutPath;
	cc.pLog											= this;
	cc.pFileSpecificConfig			= &CfgFile;		// should be removed 
	cc.presets									= m_presets;
	cc.bQuiet										= m_bQuiet;

	cc.sourceFolder = NormalizePath(cc.sourceFolder.GetString());
	cc.m_sOutputFolder = NormalizePath(cc.m_sOutputFolder.GetString());

	// Check if output file is valid (have same timestamp as input file).
	compiler->GetOutputFile( cc );
	CString outputFileName = cc.outputFile;
	CString outputFile = cc.getOutputPath();

	const char *sOutFile = cc.outputFile.GetString();
	const char *sOutDir = cc.m_sOutputFolder.GetString();

	if (!cc.masterFolder.IsEmpty())
		EnsureDirectoriesPresent(CString(Path::AddBackslash(cc.masterFolder)+cc.m_sOutputFolder).GetString());
	else
		EnsureDirectoriesPresent(cc.m_sOutputFolder);
	
	// Compare time stamp of output file.
	if (!config->HasKey("refresh"))
	{
		unsigned nTimeSrc = GetFileUnixTimeMax( filename );
		unsigned nTimeTgt = GetFileUnixTimeMax( cc.getOutputPath().GetString() );

		//if (nTimeSrc < nTimeTgt && nFilterTimestamp < nTimeTgt)
		if (nTimeSrc < nTimeTgt)
		{
			// both Source and Filter code are older than target,
			// thus the target is up to date
			Log("Skipping %s: File is up to date", filename);
			return true;
		}												
	}

	Log("");
	Log("-------------------------------------------------------");
	CString destpurefile = Path::GetFile(filename);
	Log("Path='%s'", Path::RemoveBackslash(cc.m_sFileFolder));
	Log("File='%s'", destpurefile);
//	CString destpurepath = Path::GetPath(filename);
//	Log(" %s", destpurepath);
//	Log("");

	OutputDebugString("Current file: '");
	OutputDebugString(filename);
	OutputDebugString("' ... ");

	// file name changed - print new header for warnings and errors
	SetHeaderLine(filename);

	// compile file
	bool bRet=compiler->Process( cc );

	OutputDebugString("processed\n");

	if(!bRet)
		LogError("failed to convert file");

	// Release cloned config.
//	if (config != m_config)
//		config->Release();

	return bRet;
}


void ResourceCompiler::SetHeaderLine( const char *inszLine )
{
	m_bWarningHeaderLine=false;
	m_bErrorHeaderLine=false;
	m_sHeaderLine=inszLine;
}

void ResourceCompiler::InitializeLogThreadNumbering()
{
	if ((m_threadNumberTLSIndex = TlsAlloc()) == TLS_OUT_OF_INDEXES)
		m_threadNumberTLSIndex = -1;
	if (m_threadNumberTLSIndex >= 0)
	{
		static int mainThreadIndex = 0;
		TlsSetValue(m_threadNumberTLSIndex, &mainThreadIndex);
	}
}

int ResourceCompiler::GetThreadNumber()
{
	int* pThreadIndex = (m_threadNumberTLSIndex >= 0 ? static_cast<int*>(TlsGetValue(m_threadNumberTLSIndex)) : 0);
	int threadIndex = (pThreadIndex ? *pThreadIndex : -1);
	return threadIndex;
}

void ResourceCompiler::LogLine( const ELogType ineType, const char* szText )
{
	// Get the index of the thread.
	int threadIndex = GetThreadNumber();

	static volatile int g_LogLock;
	WriteLock lock(g_LogLock);

	if (m_bQuiet)
	{
		if(m_hLogFile)
		{
			fprintf(m_hLogFile,"%s\n",szText);
			fflush(m_hLogFile);
		}
		return;
	}

	char threadString[10];
	threadString[0] = 0;
	if (threadIndex > 0)
		sprintf(threadString, "%d> ", threadIndex);

	switch(ineType)
	{
		case eMessage:
			printf ("%s  ", threadString);							// to make it aligned with E: and W:
			break;
		case eWarning:
			printf ("%sW: ", threadString);							// for Warning
			if(m_hWarningLogFile)
			{
				if(!m_bWarningHeaderLine)
				{
					fprintf(m_hWarningLogFile,"%s\r\n-----------------------------------------------------------------\r\n\r\n", threadString);
					fprintf(m_hWarningLogFile,"%s%s\r\n", threadString, m_sHeaderLine.c_str());
					m_bWarningHeaderLine=true;
				}

				fprintf(m_hWarningLogFile,"%s  %s\r\n",threadString,szText);
				fflush(m_hWarningLogFile);
			}
			break;
		case eError:
			printf ("%sE: ", threadString);							// for Error
			if(m_hErrorLogFile)
			{
				if(!m_bErrorHeaderLine)
				{
					fprintf(m_hErrorLogFile,"%s\r\n-----------------------------------------------------------------\r\n\r\n", threadString);
					fprintf(m_hErrorLogFile,"%s%s\r\n",threadString,m_sHeaderLine.c_str());
					m_bErrorHeaderLine=true;
				}

				fprintf(m_hErrorLogFile,"%s  %s\r\n",threadString,szText);
				fflush(m_hErrorLogFile);
			}
			break;

		default:assert(0);
	}

	if(m_hLogFile)
	{
		fprintf(m_hLogFile,"%s%s\r\n",threadString,szText);
		fflush(m_hLogFile);
	}

	if (m_bWarningsAsErrors && (ineType == eWarning || ineType == eError))
	{
		MessageBox( NULL,szText,"RC Compilation Error",MB_OK|MB_ICONERROR );
		if(m_hLogFile)fclose(m_hLogFile);m_hLogFile=0;
		if(m_hErrorLogFile)fclose(m_hErrorLogFile);m_hErrorLogFile=0;
		if(m_hWarningLogFile)fclose(m_hWarningLogFile);m_hWarningLogFile=0;
		exit( EXIT_FAILURE );
	}

	printf("%s\n",szText);
}


/*
//! Load and parse the Crytek Chunked File into the universal (very big) structure
//! The caller should then call Release on the structure to free the mem
//! @param filename Full filename including path to the file
CryChunkedFile* ResourceCompiler::LoadCryChunkedFile (const char* szFileName)
{
	CChunkFile_AutoPtr pReader = new CChunkFile ();
	if (!pReader->open (szFileName))
		return NULL;

	try
	{
		return new CryChunkedFile(pReader);
	}
	catch (CryChunkedFile::Error& e)
	{
		LogError("%s", e.strDesc.c_str());
		return NULL;
	}
	catch (...)
	{
		LogError("UNEXPECTED ERROR while trying to load Cry Chunked File \"%s\"", szFileName);
		return NULL;
	}
}
*/

//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////
//! Print error message.
void MessageBoxError( const char *format,... )
{
	va_list	ArgList;
	char		szBuffer[1024];

	va_start(ArgList, format);
	vsprintf(szBuffer, format, ArgList);
	va_end(ArgList);

	CString str = "####-ERROR-####: ";
	str += szBuffer;

//	printf( "%s\n",str );
	MessageBox( NULL,szBuffer,_T("Error"),MB_OK|MB_ICONERROR );
}

/*
//////////////////////////////////////////////////////////////////////////
//! Log message.
void Log( const char *format,... )
{
	va_list	ArgList;
	char		szBuffer[1024];

	va_start(ArgList, format);
	vsprintf(szBuffer, format, ArgList);
	va_end(ArgList);

	printf( "%s\n",szBuffer );
}
*/

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::show_help()
{
	Log( "" );
	Log( "Usage: RC filespec /p=<Platform> [/Key1=Value1] [/Key2=Value2] etc..." );
	Log( "" );

	{
		std::map<string,string>::const_iterator it, end=m_KeyHelp.end();

		for(it=m_KeyHelp.begin();it!=end;++it)
		{
			const string &rKey = it->first;
			const string &rHelp = it->second;

			Log( "/%s %s",rKey.c_str(),rHelp.c_str());
		}
	}
}

//////////////////////////////////////////////////////////////////////////
static Platform GetPlatformFromName( const char *sPlatform )
{
	// Platform name to enum mapping.
	struct {
		const char *name;
		Platform platform;
	} platformNames[] =
	{
		{ "PC",PLATFORM_PC },
		{ "XBOX",PLATFORM_XBOX },
		{ "PS2",PLATFORM_PS2 },
		{ "GC",PLATFORM_GAMECUBE },
		{ "GameCube",PLATFORM_GAMECUBE },
	};
	for (int i = 0; i < sizeof(platformNames)/sizeof(platformNames[0]); i++)
	{
		if (stricmp(platformNames[i].name,sPlatform) == 0)
			return platformNames[i].platform;
	}
	return PLATFORM_UNKNOWN;
}




//////////////////////////////////////////////////////////////////////////
void RegisterConvertors( IResourceCompiler *rc )
{
	IRCLog *log=rc->GetIRCLog();					assert(log);

	string strDir;
	{
		char szRCPath[1000];
		if (GetModuleFileName (NULL, szRCPath, sizeof(szRCPath)))
			strDir = PathUtil::GetParentDirectory(szRCPath) + "\\";
	}
	__finddata64_t fd;
	int hSearch = _findfirst64 ((strDir +  "ResourceCompiler*.dll").c_str(), &fd);
	if (hSearch != -1)

	do {
		HMODULE hPlugin = LoadLibrary ((strDir+fd.name).c_str());
		if (!hPlugin)
		{
			log->Log ("Error: Couldn't load plug-in module %s", fd.name);
			continue;
		}
		
		FnRegisterConvertors fnRegister = hPlugin?(FnRegisterConvertors)GetProcAddress(hPlugin, "RegisterConvertors"):NULL;
		if (!fnRegister)
		{
			log->Log ("Error: plug-in module %s doesn't have RegisterConvertors function", fd.name);
			continue;
		}

		time_t nTime = GetTimestampForLoadedLibrary (hPlugin);
		char* szTime = "unknown";
		if (nTime)
		{
			szTime = asctime(localtime(&nTime));
			szTime[strlen(szTime)-1] = '\0';
		}
//		Info ("timestamp %s", szTime);
		log->Log("");
		log->Log("  Loading \"%s\"", fd.name);

		fnRegister (rc);
	}
	while(_findnext64(hSearch, &fd) != -1);

	_findclose(hSearch);
	log->Log("");
}


MiniDumper miniDumper("rcexe");

//////////////////////////////////////////////////////////////////////////
int __cdecl main(int argc, char **argv, char **envp)
{
	/*
	int tmpDbgFlag;
	tmpDbgFlag = _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG);
	// Clear the upper 16 bits and OR in the desired freqency
	tmpDbgFlag = (tmpDbgFlag & 0x0000FFFF) | (32768 << 16);
	tmpDbgFlag |= _CRTDBG_LEAK_CHECK_DF;
	_CrtSetDbgFlag(tmpDbgFlag);

	// Check heap every 
	//_CrtSetBreakAlloc(2031);
	*/


	ResourceCompiler rc;
	Config mainConfig;
	mainConfig.SetConfigKeyRegistry(&rc);
	CmdLine cmdLine;

	rc.QueryVersionInfo();
	SFileVersion fv = rc.GetFileVersion();

	rc.Log("ResourceCompiler Version %d.%d.%d",fv.v[2],fv.v[1] );
	rc.Log("================");

	rc.Log("Copyright(c) Crytek 2001-2005, All Rights Reserved." );
#ifdef _WIN64
	rc.Log("64-bit edition");
#endif

	rc.RegisterKey("wait","wait for key after running the application");
	rc.RegisterKey("WX","treat warning as errors");
	rc.RegisterKey("recursive","traverse input directory with sub folders");
	rc.RegisterKey("refresh","force recompilation of resources with up to date timestamp");
	rc.RegisterKey("p","to specify platform (PC,XBOX,PS2,GC,X360,PS3)");
	rc.RegisterKey("statistics","log statistics to rc_stats_* files");
	rc.RegisterKey("quiet","to suppress all printouts");
	rc.RegisterKey("logfiles","=1 to generate log files: rc_log*.log");
	rc.RegisterKey("presetcfg","to define the path to the presets e.g. \"rc_presets_pc.ini\"");
	rc.RegisterKey("targetroot","to define the destination folder");
	rc.RegisterKey("masterfolder","to define the mastercd folder");
	rc.RegisterKey("threads","=N to use N threads (only supported for TIFF compilation)");

	rc.Log("");
	if(argc>1)
	{
		rc.Log("CommandLine:");
		for(int i=1;i<argc;++i)
			rc.Log("  '%s'",argv[i]);
		rc.Log("");
	}


//	string test = "asd" + string("asdf") + "asdf" + string("df");
//	CString test2 = "asd" + CString("asdf") + "asdf" + CString("df");
	char moduleName[_MAX_PATH];
	GetModuleFileName( NULL, moduleName, _MAX_PATH );//retrieves the PATH for the current module

	bool bConfigFileLoaded = false;
	// Load main config.
	CfgFile cfgFile;
	if (cfgFile.Load( CString(PathUtil::Make(PathUtil::GetPath(moduleName),RC_INI_FILE))) )
	{
		bConfigFileLoaded = true;
		cfgFile.SetConfig( eCP_PriorityPlatform,COMMON_SECTION,&mainConfig );
	}
	else
	{
		char str[512];

		sprintf(str,"Resource compiler ini file ('%s') is missing - check current working folder",RC_INI_FILE);

		rc.Log("%s",str);

		MessageBox(0,str,"ResourceCompiler Error",MB_OK|MB_ICONERROR);
		return 1;
	}

	// Parse command line.
	cmdLine.Parse( argc,argv,&mainConfig );

	bool bWork=true;

	if(argc<=1)
	{
		bWork=false;
	}


	CString platformStr;
	if (!((IConfig &)mainConfig).Get( "p",platformStr ))
	{
		// Platform switch not specified.
		rc.Log("Platform (/p) not specified, defaulting to PC.");
		rc.Log("");
		platformStr = "PC";
		mainConfig.Set(eCP_PriorityCmdline,"p",platformStr.GetString());
	}
	
	// Detect platform.
	Platform platform = GetPlatformFromName(platformStr.GetString());
	if (platform == PLATFORM_UNKNOWN)
	{
		char str[512];

		sprintf(str,"Unknown platform %s specified",(const char*)platformStr.GetString());

		rc.Log("%s",str);

		MessageBox(0,str,"ResourceCompiler Error",MB_OK|MB_ICONERROR);
		return 1;
	}


	rc.GetHWnd();

	rc.Log("Registering sub compilers (ResourceCompiler*.dll)");


	RegisterConvertors( &rc );
	if (bConfigFileLoaded)
	{
		// Load configuration from per platform section. 
		cfgFile.SetConfig( eCP_PriorityFile, rc.GetSectionName(platform),&mainConfig );
	}

	rc.m_bQuiet = mainConfig.HasKey("quiet");

	if(bWork)
	{
		//compile all files into ccgf
		rc.Compile( platform,&mainConfig,cmdLine.m_fileSpec.GetString() );

		rc.PostBuild();		// e.g. print material dependencies
	}

	rc.DeInit();		// to clean up before waiting for user

	//rc.SortLogFiles();

	if(!rc.m_bQuiet)
	if(!bWork)
		rc.show_help();

  if(mainConfig.HasKey("wait"))
  {
		rc.Log("");		
		rc.Log("                                              <RETURN>  (/wait was specified)");			// right aligned on 80 char screen
		getchar();
	};

	return 0;
}

//! Returns the main application window
HWND ResourceCompiler::GetHWnd()
{
	HMODULE hKernel32 = LoadLibrary ("kernel32.dll");
	HWND hResult = GetDesktopWindow();
	if (hKernel32)
	{
		//typedef WINBASEAPI  HWND APIENTRY (*FnGetConsoleWindow )(VOID);
		typedef HWND (APIENTRY*FnGetConsoleWindow )(VOID);
		FnGetConsoleWindow GetConsoleWindow = (FnGetConsoleWindow)GetProcAddress (hKernel32, "GetConsoleWindow");
		if (GetConsoleWindow)
		{
			hResult = GetConsoleWindow();
		}
		FreeLibrary (hKernel32);
	}
	return hResult;
}

HWND ResourceCompiler::GetEmptyWindow()
{
	if (!m_hEmptyWindow)
	{
		const char szClassName[] = "DirectXWnd";
		WNDCLASS wc;
		memset (&wc, 0, sizeof(wc));
		wc.style = CS_OWNDC;
		wc.lpfnWndProc = DefWindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance  = (HINSTANCE)GetModuleHandle (NULL);
		wc.lpszClassName = szClassName;

		ATOM atomWndClass = RegisterClass (&wc);
		m_hEmptyWindow = CreateWindow (szClassName, "DirectXEmpty", WS_OVERLAPPEDWINDOW, CW_USEDEFAULT,CW_USEDEFAULT,256,256,NULL, NULL, wc.hInstance, 0);
	}
	return m_hEmptyWindow;
}

// ------------------------------------------------------
void ResourceCompiler::AddDependencyMaterial( const char *inszSrcFilename, const char *inszMatName, const char *inszScriptName )
{
	if (!m_bStatistics)
		return;
	CMatDep dep;

	dep.m_sMatName=inszMatName;
	dep.m_sScriptName=inszScriptName;

	m_MaterialDependencies.insert( CMatDepPair(dep,inszSrcFilename) );

//	Log("  DepMat: <%s> <%s>",inszMatName,inszScriptName);
}




// ------------------------------------------------------
void ResourceCompiler::AddDependencyFile( const char *inszSrcFilename, const char *inszPathFileName )
{
	if (!m_bStatistics)
		return;
	m_FileDependencies.insert( CFileDepPair(inszPathFileName,inszSrcFilename) );

//	Log("  DepFile: <%s>",inszPathFileName);
}

// ------------------------------------------------------
void ResourceCompiler::ShowFileDependencies()
{
	const char *szFileName=RC_FILENAME_FILEDEP;

	FILE *out=fopen(m_exePath+szFileName,"wb");

	if(!out)
	{
		LogError("unable to open %s - file it not updated",szFileName);
		return;
	}

	CFileDepMap::iterator it;

	string sLastDep="";		// for a nice printout
	bool bFirst=true;

	for(it=m_FileDependencies.begin(); it!=m_FileDependencies.end(); ++it)
	{
		const string &rsDepFile = it->first;
		const string &rsSrcFile = it->second;

		if(bFirst || rsDepFile!=sLastDep)
		{
			fprintf(out,"\r\n");
			fprintf(out,"'%s'\r\n",rsDepFile.c_str());
			sLastDep=rsDepFile;
			bFirst=false;
		}

		fprintf(out,"    used by: '%s'\r\n",rsSrcFile.c_str());
	}

	fprintf(out,"\r\n");
	fprintf(out,"------------------------------------------------------------------------------\r\n");
	fprintf(out,"\r\n");
	fprintf(out,"all used files:\r\n");
	fprintf(out,"\r\n");

	bFirst=true;
	for(it=m_FileDependencies.begin(); it!=m_FileDependencies.end(); ++it)
	{
		const string &rsDepFile = it->first;
		const string &rsSrcFile = it->second;

		if(bFirst || rsDepFile!=sLastDep)
		{
			fprintf(out,"  '%s'\r\n",rsDepFile.c_str());
			sLastDep=rsDepFile;
			bFirst=false;
		}
	}

	fclose(out);
}


// ------------------------------------------------------
void ResourceCompiler::ShowPresetUsage()
{
	const char *szFileName=RC_FILENAME_PRESETUSAGE;

	FILE *out=fopen(m_exePath+szFileName,"wb");

	if(!out)
	{
		LogError("unable to open %s - file it not updated",szFileName);
		return;
	}

	std::vector<std::pair<string, int> > presetSortedMappingTable(m_Files.size());
	for (int fileIndex = 0, fileCount = int(m_Files.size()); fileIndex < fileCount; ++fileIndex)
		presetSortedMappingTable[fileIndex] = std::make_pair(m_Files[fileIndex].m_sPreset, fileIndex);
	std::sort(presetSortedMappingTable.begin(), presetSortedMappingTable.end());

	string sLastDep="";		// for a nice printout
	bool bFirst=true;

	fprintf(out,"preset usage:\r\n");
	fprintf(out,"\r\n");

	//for(it=m_PresetUsage.begin(); it!=m_PresetUsage.end(); ++it)
	for (int mappingTableIndex = 0, mappingTableSize = int(presetSortedMappingTable.size()); mappingTableIndex < mappingTableSize; ++mappingTableIndex)
	{
		const uint32 dwFileStatsIndex = presetSortedMappingTable[mappingTableIndex].second;

		CFileStats &stats = m_Files[dwFileStatsIndex];

		if(bFirst || stats.m_sPreset!=sLastDep)
		{
			fprintf(out,"\r\n  PRESET %s:\r\n",stats.m_sPreset.c_str());
			sLastDep=stats.m_sPreset;
			bFirst=false;
		}

		fprintf(out,"    %3d%% '%s' %d KB -> %d KB \r\n",(stats.m_DstFileSizeKB*100)/stats.m_SrcFileSizeKB,stats.m_sDestFilename.c_str(),
			stats.m_SrcFileSizeKB,stats.m_DstFileSizeKB);
	}

	fclose(out);
}






// ------------------------------------------------------
void ResourceCompiler::ShowXLSFileSizes()
{
	const char *szFileName=RC_FILENAME_XLS_FILESIZES;

	FILE *out=fopen(m_exePath+szFileName,"wb");

	if(!out)
	{
		LogError("unable to open %s - file it not updated",szFileName);
		return;
	}

	fprintf(out,"DestFileSizeInKB"
							"\tDestFileName"
							"\tPreset"
							"\tWidth"
							"\tHeight"
							"\tAlpha"
							"\tMips"
							"\tMemInKB(in Memory)"
							"\tFormat"
							"\tReduce"
							"\tAttachedMemInKB(e.g. Alpha of 3dC)"
							"\tDestFilePath"
							"\r\n");
	fprintf(out,"----------------------------------------------------------------------------------------------<in Excel friendly format>\r\n");

	// Produce a mapping table into the file stats array that is sorted by file size.
	std::vector<std::pair<uint32, uint32> > fileSizeSortedMappingTable(m_Files.size());
	for (int fileIndex = 0, fileCount = int(m_Files.size()); fileIndex < fileCount; ++fileIndex)
		fileSizeSortedMappingTable[fileIndex] = std::make_pair(m_Files[fileIndex].m_DstFileSizeKB, fileIndex);
	std::sort(fileSizeSortedMappingTable.begin(), fileSizeSortedMappingTable.end());

	for (int mappingTableIndex = 0, mappingTableSize = fileSizeSortedMappingTable.size(); mappingTableIndex < mappingTableSize; ++mappingTableIndex)
	{
		const uint32 dwFileStatsIndex = fileSizeSortedMappingTable[mappingTableIndex].second;

		CFileStats &stats = m_Files[dwFileStatsIndex];

		fprintf(out,"%d"					// DestFileSizeInKB
								"\t%s"				// DestFileName
								"\t%s"				// preset
								"\t%s"				// info
								"\t%s\r\n",		// DestPath
			stats.m_DstFileSizeKB,
			Path::GetFile(stats.m_sDestFilename.c_str()),
			stats.m_sPreset,
			stats.m_sInfo,
			Path::GetPath(stats.m_sDestFilename.c_str()));
	}

	fclose(out);
}

/*
// ------------------------------------------------------
void ResourceCompiler::ShowMaterialDependencies()
{
	const char *szFileName=RC_FILENAME_MATDEP;

	FILE *out=fopen(m_exePath+szFileName,"wb");

	if(!out)
	{
		LogError("unable to open %s - file it not updated",szFileName);
		return;
	}

	CMatDepMap::iterator it;

	CMatDep LastDep;		// for a nice printout
	bool bFirst=true;

	// max info
	for(it=m_MaterialDependencies.begin(); it!=m_MaterialDependencies.end(); ++it)
	{
		const CMatDep &rsMatDep = it->first;
		const string &rsSrcFile = it->second;

		if(bFirst || !(rsMatDep==LastDep))
		{
			fprintf(out,"\r\n");
			fprintf(out,"scriptmaterial='%s' materialname='%s'\r\n",rsMatDep.m_sScriptName.c_str(),rsMatDep.m_sMatName.c_str());
			LastDep=rsMatDep; 
			bFirst=false;
		}

		fprintf(out,"    used by: '%s'\r\n",rsSrcFile.c_str());
	}

	fprintf(out,"\r\n");
	fprintf(out,"------------------------------------------------------------------------------\r\n");
	fprintf(out,"\r\n");
	fprintf(out,"all used scriptmaterials:\r\n");
	fprintf(out,"\r\n");

	// only the used scripsmaterials
	bFirst=true;
	for(it=m_MaterialDependencies.begin(); it!=m_MaterialDependencies.end(); ++it)
	{
		const CMatDep &rsMatDep = it->first;
		const string &rsSrcFile = it->second;

		if(bFirst || !(rsMatDep.m_sScriptName==LastDep.m_sScriptName))
		{
			fprintf(out,"  '%s'\r\n",rsMatDep.m_sScriptName.c_str());
			LastDep=rsMatDep; 
			bFirst=false;
		}
	}

	fclose(out);
}
*/

// ------------------------------------------------------
void ResourceCompiler::PostBuild()
{
	if (m_bStatistics)
	{
		Log("writing statistics files (rc_stats_...) ");
		Log("");

		ShowFileDependencies();
//		ShowMaterialDependencies();
		ShowPresetUsage();
		ShowXLSFileSizes();
	}
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::QueryVersionInfo()
{
	char moduleName[_MAX_PATH];
	DWORD dwHandle;
	UINT len;

	char ver[1024*8];

	GetModuleFileName( NULL, moduleName, _MAX_PATH );//retrieves the PATH for the current module
	m_exePath = Path::AddBackslash( Path::GetPath(moduleName) );

	int verSize = GetFileVersionInfoSize( moduleName,&dwHandle );
	if (verSize > 0)
	{
		GetFileVersionInfo( moduleName,dwHandle,1024*8,ver );
		VS_FIXEDFILEINFO *vinfo;
		VerQueryValue( ver,"\\",(void**)&vinfo,&len );

		m_fileVersion.v[0] = vinfo->dwFileVersionLS & 0xFFFF;
		m_fileVersion.v[1] = vinfo->dwFileVersionLS >> 16;
		m_fileVersion.v[2] = vinfo->dwFileVersionMS & 0xFFFF;
		m_fileVersion.v[3] = vinfo->dwFileVersionMS >> 16;

		m_productVersion.v[0] = vinfo->dwProductVersionLS & 0xFFFF;
		m_productVersion.v[1] = vinfo->dwProductVersionLS >> 16;
		m_productVersion.v[2] = vinfo->dwProductVersionMS & 0xFFFF;
		m_productVersion.v[3] = vinfo->dwProductVersionMS >> 16;
	}
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::DeInit()
{
	m_extensionManager.UnregisterAll();
}

//////////////////////////////////////////////////////////////////////////
ICfgFile *ResourceCompiler::CreateCfgFile()
{
	return new CfgFile;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::VerifyKeyRegistration( const char *szKey )
{
	assert(szKey);

	string sKey = szKey;

	sKey.MakeLower();

	if(m_KeyHelp.count(sKey)==0)
		Log("INFO: key '%s' was not registered, call RegisterKey() before using the key",szKey);
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::RegisterKey( const char *key, const char *helptxt )
{
	string sKey = key;

	sKey.MakeLower();

	assert(m_KeyHelp.count(sKey)==0);		// registered twice

	m_KeyHelp[sKey] = helptxt;
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::AutoDetection()
{
	unsigned int system;
	GetNumCPUCores(system, m_iThreads);
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::SortLogFileByThread(FILE* file)
{
	fseek(file, 0, SEEK_SET);

	// Sort all the lines into a vector for each thread.
	typedef std::vector<std::vector<string> > StringsByThread;
	StringsByThread stringsByThread;
	while (!feof(file))
	{
		char lineBuffer[1024];
		fgets(lineBuffer, sizeof(lineBuffer), file);

		// Check the start of the line to see whether it is prefixed with the thread index.
		// Threads are written in this form: 3> (message text).
		int threadIndex = 0, threadIndexLength = 0;
		{
			int pos;
			const int MAX_THREAD_LENGTH = 5;
			for (pos = 0; lineBuffer[pos] >= '0' && lineBuffer[pos] <= '9' && pos < MAX_THREAD_LENGTH; ++pos);
			threadIndexLength = (lineBuffer[pos] == '>' ? pos : 0);
			if (threadIndexLength > 0)
			{
				threadIndex = strtol(lineBuffer, 0, 10);
				threadIndexLength += 2; // Skip the '> '
			}
		}

		// Sanity check for thread index.
		const int THREAD_COUNT_SANITY_CHECK = 128;
		if (threadIndex > THREAD_COUNT_SANITY_CHECK)
			threadIndex = THREAD_COUNT_SANITY_CHECK;

		if (int(stringsByThread.size()) < threadIndex + 1)
			stringsByThread.resize(threadIndex + 1);

		stringsByThread[threadIndex].push_back(lineBuffer + threadIndexLength);
	}

	// Output all the lines.
	fseek(file, 0, SEEK_SET);
	for (int threadIndex = 0, threadCount = int(stringsByThread.size()); threadIndex < threadCount; ++threadIndex)
	{
		std::vector<string>& lines = stringsByThread[threadIndex];
		for (int lineIndex = 0, lineCount = int(lines.size()); lineIndex < lineCount; ++lineIndex)
			fprintf(file, "%s", lines[lineIndex].c_str());
	}
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::SortLogFiles()
{
	if (m_hLogFile)
		SortLogFileByThread(m_hLogFile);
	if (m_hWarningLogFile)
		SortLogFileByThread(m_hWarningLogFile);
	if (m_hErrorLogFile)
		SortLogFileByThread(m_hErrorLogFile);
}

//////////////////////////////////////////////////////////////////////////
void ResourceCompiler::AddFileStats(ConvertContext* pCC, const char *inszPathFileName, const size_t SrcFileSize, const size_t DstFileSize, const char *inszPreset, const char *inszInfo)
{
	CFileStats value;

	value.m_SrcFileSizeKB=(SrcFileSize+1023)/1024;
	value.m_DstFileSizeKB=(DstFileSize+1023)/1024;
	value.m_sDestFilename=inszPathFileName;

	if(inszPreset)
		value.m_sPreset=inszPreset;

	if(inszInfo)
		value.m_sInfo=inszInfo;

	uint32 dwFileStatsIndex = (uint32)pCC->fileStats->size();

	pCC->fileStats->push_back(value);
}
