#pragma once

#include "Cry_Math.h"									// 
#include "PixelFormats.h"							// CPixelFormats


// needs to be in the same order as in IDC_PREVIEWMODE
enum EPreviewMode
{
	ePM_RGB=0,						// normal
	ePM_AAA=1,						// replicate the alpha channel as greyscale value
	ePM_RGBA=2,						// alpha to fade between RGB and background
	ePM_NormalLength=3,		// replicate length of the normal as greyscale value
	ePM_RGBmulA=4,				// RGB * Alpha (good to fake HDR)
	ePM_AlphaTest=5,			// Shows the Alpha Channel visualized with an AlpheTest of 0.5
	ePM_mCIE2RGB=6,				// mCIE
};


class CBumpProperties
{
public:

	CBumpProperties()
	{
		SetToDefault();
	}

	void SetToDefault()
	{
		m_iBumpToNormalFilter=0;
		m_fBumpStrength=5.0f;			// default like NVidia plugin
		m_fBumpBlurAmount=0.0f;
		m_sBumpmapName="";
	}

	string				m_sBumpmapName;					// filename without path and special characters like space, qutotes, .. (dot is ok)
	int						m_iBumpToNormalFilter;	// 0=none, 1=Gauss, 2=GaussAlpha
	float					m_fBumpStrength;				// -1000.0 .. 1000.0
	float					m_fBumpBlurAmount;			// 0 .. 10.0
};



class CImageProperties
{
public:
	enum {NUM_CONTROLLED_MIP_MAPS = 6};

	CImageProperties()
	{
		SetToDefault();
	}

	void SetToDefault()
	{
		m_bMipmaps=true;
		//m_bMipMirror=false;
    m_bMipGammaCorrected=false;
		m_bAutoOptimizeFile=true;
		m_bSupressEngineReduce=false;
		m_bDontStreamTexture=false;
		m_bMipRenormalize=false;
		m_bUserDialog=false;
		m_bUserDialogCustom=true;
		m_iDestPixelFormat=0;
		m_ePreviewMode=ePM_RGB;
		m_iReduceResolutionGlobal=0;
		m_iReduceResolutionFile=0;
		m_bPreviewFiltered=false;
		m_bPreviewTiled=true;
		m_bApplyRangeAdjustment=true;
		m_bAutoDetectLuminanceOnly=false;
		m_iMinAlpha=0;
		m_bPowOf2=true;
		m_eGlobalCompressor=eCGDirectX;
		m_bAverageColor=true;
		m_bMaintainAlphaCoverage=false;
		std::fill(m_iMIPAlpha, m_iMIPAlpha + NUM_CONTROLLED_MIP_MAPS, 50);
		m_iMipSharpen=35;				// default is already sharpening a bit
		m_iColorModel=0;

		m_Bump2Normal.SetToDefault();
		m_AlphaAsBump.SetToDefault();
		m_sPreset="";
	}

	char *GetPreviewModeText() const
	{
		switch(m_ePreviewMode)
		{
			case ePM_RGB:								return "Normal RGB preview mode\n(no gamma correction)";
			case ePM_AAA:								return "Preview Alpha channel in greyscale\n(no gamma correction)";
			case ePM_RGBA:							return "RGB Alpha blended with the background\n(no gamma correction)";
			case ePM_NormalLength:			return "Normalmap debug: The brighter a pixel is the better the normal,\nblue indicates very short normals, red impossible normals (>1)";
			case ePM_RGBmulA:						return "RGB multiplied with Alpha, useful to get cheap HDR textures\n(no gamma correction)";
			case ePM_AlphaTest:					return "Shows the Alpha Channel vizualized with an AlpheTest of 0.5";
			case ePM_mCIE2RGB:					return "mCIE to RGB conversion (for internal tests)";
			default:	assert(0);				return "";
		}
	}

	// Return:
	//   index in the g_pixelformats table [0..GetPixelFormatCount()-1] or -1 if no conversion should happen
	TPixelFormatID GetDestPixelFormat( const bool bAlphaChannelUsed ) const
	{
		return CPixelFormats::FindFinalTextureFormat(m_iDestPixelFormat,bAlphaChannelUsed);
	}

	// Arguments:
	//   iDestPixelFormat - [0..GetPixelFormatCount()-1], -1 if not used
	void SetDestPixelFormat( const TPixelFormatID iDestPixelFormat )
	{
		m_iDestPixelFormat=iDestPixelFormat;
	}

	uint32 GetReduceResolution() const
	{
		return (uint32)max(0,(m_iReduceResolutionGlobal+m_iReduceResolutionFile));
	}

	// keep number assignment as it's also used for file storage
	enum EGlobalCompressor
	{
		eCGDirectX=0,			// DirectX, might used hardware - fastest
		eCGNVDXT=1,				// NVDXT lib fast and good quality, also supports 3Dc
		eCGATI=2,					// good quality, also supports 3Dc, all formats not supported are redirected to other compressor
		eCGTIF=3,					// .TIF file format
		eCGSimple=4				// experimental DXT compressor - simple because of the reduced source code (CSimpleDXTCompressor)
	};

	static float lerp( float a, float b, float t ) { return a*(1-t)+b*t; }

	// Arguments:
	//   iMip - 0..iMaxMip-1
	//   iMaxMip - >1
	// Returns:
	//   0..255
	float ComputeMIPAlphaOffset( const int iMip, const int iMaxMip )
	{
		float fVal = m_iMIPAlpha[NUM_CONTROLLED_MIP_MAPS - 1];
		if (iMip < NUM_CONTROLLED_MIP_MAPS * 2)
		{
			float fInterpolationSlider1 = m_iMIPAlpha[iMip / 2];
			float fInterpolationSlider2 = m_iMIPAlpha[iMip / 2 + 1];
			fVal = fInterpolationSlider1 + (fInterpolationSlider2 - fInterpolationSlider1) * (iMip & 1) * 0.5f;
		}

		return 0.5f-fVal/100.0f;
	}

	// -------------------------------------------------------------------------

	CBumpProperties		m_Bump2Normal;										//
	CBumpProperties		m_AlphaAsBump;										//
	bool							m_bMipmaps;												// about +1/3 more memory
	//bool							m_bMipMirror;											// for tiled texture
	bool							m_bMipGammaCorrected;							// mipmap generation and reduce resolution with gamma in mind (good for diffuse textures, bad for normalmaps)
	bool							m_bAutoOptimizeFile;							// 
	bool							m_bSupressEngineReduce;						// flag propagated to engine to affect loading (reduce texture resolution during loading)
	bool							m_bDontStreamTexture;							// 
	bool							m_bMipRenormalize;								// only good for normalmaps
	bool							m_bUserDialog;										// show the user dialog for interactive tweaking
	bool							m_bUserDialogCustom;							// true=show extended userdialog with the ability to set custom settings
	bool							m_bPowOf2;												// true=image needs to be power of 2 in width and height 
	int								m_iReduceResolutionGlobal;				// [0..[ to remove the top mipmap levels  /globalreduce
	int								m_iReduceResolutionFile;					// -2..3 to remove the top mipmap levels  /reduce
	bool							m_bApplyRangeAdjustment;					// true = -1..1 => 0..1 adjustment (needed for normalmaps)
	bool							m_bAutoDetectLuminanceOnly;				// if R=G=B and no alpha then use L8 (luminance)
	bool							m_bAverageColor;									//
	bool							m_bMaintainAlphaCoverage;					// maintain alpha coverage in mipmaps
	int								m_iMIPAlpha[NUM_CONTROLLED_MIP_MAPS];	// 0..50=normal..100
	int								m_iMipSharpen;										// 0=no sharpening .. 100=full sharpening
	int								m_iMinAlpha;											// 0..255 to limit per pixel pow factor adjustment - prevents NAN in shader
	int								m_iColorModel;										// 0=RGB, 1=CIE

	// global properties
	EGlobalCompressor	m_eGlobalCompressor;							// 0=DirectX, 1=NVDXT, 2=ATI, 3=TIF, 4=Simple

	CString						m_sPreset;												//

	// preview properties
	EPreviewMode			m_ePreviewMode;										// e.g. 0:RGB, 1:AAA replicate the alpha channel as greyscale value
	bool							m_bPreviewFiltered;								// activate the bilinear filter in the preview
	bool							m_bPreviewTiled;									// 



private: // ---------------------------------------------------------------------

	TPixelFormatID		m_iDestPixelFormat;							// index in the g_pixelformats table [0..GetPixelFormatCount()-1], -1 if not used
};
