#include "StdAfx.h"

#include "AnimationCompiler.h"
#include "AnimationInfoLoader.h"
#include "CGF\CGFLoader.h"
#include "CGF\CGFSaver.h"
#include "SkeletonInfo.h"
#include "AnimationLoader.h"
#include "IConfig.h"
#include "TrackStorage.h"


CAnimationCompiler * g_pAnimationCompiler;

static bool ScanDirectoryRecursive( const CString &root,const CString &path,const CString &file,std::vector<CString> &files, bool recursive , std::vector<uint32>& sizes);


class CAnimationLoaderListener : public IAnimationLoaderListener
{
public:
	CAnimationLoaderListener(ConvertContext &cc)
		:	cc(cc)
	{
	}

	virtual void OnAnimationLoaderMessage(MessageType type, const char* szMessage)
	{
		switch (type)
		{
		case MESSAGE_INFO:
			cc.pLog->Log("%s", szMessage);
			break;
		case MESSAGE_WARNING:
			cc.pLog->LogWarning("%s", szMessage);
			break;
		case MESSAGE_ERROR:
			cc.pLog->LogError("%s", szMessage);
			break;
		}
	}

private:
	ConvertContext &cc;
};


CAnimationCompiler::CAnimationCompiler(ICryXML * pXML) : m_pXMLParser(pXML)
{
	m_TotalShared = 0;
	m_TotalMemoryShared = 0;
	m_TotalMemory = 0;
	m_TotalTracks = 0;
	m_pTrackStorage.reset(0);
	m_refCount = 1;
	m_iCount = 0;
	m_fAverage = 0;
	m_fTotal = 0;
	//	m_iThreads = 0;
}

CAnimationCompiler::~CAnimationCompiler(void)
{
}

void CAnimationCompiler::Release()
{
	if (--m_refCount <= 0)
		delete this;
}

bool /*CAnimationCompiler::*/GetFileTime( const char *filename,FILETIME *ftimeModify, FILETIME*ftimeCreate )
{
	WIN32_FIND_DATA FindFileData;
	HANDLE hFind;
	hFind = FindFirstFile( filename,&FindFileData );
	if (hFind == INVALID_HANDLE_VALUE)
	{
		return false;
	}
	FindClose(hFind);
	if (ftimeCreate)
	{
		ftimeCreate->dwLowDateTime = FindFileData.ftCreationTime.dwLowDateTime;
		ftimeCreate->dwHighDateTime = FindFileData.ftCreationTime.dwHighDateTime;
	}
	if (ftimeModify)
	{
		ftimeModify->dwLowDateTime = FindFileData.ftLastWriteTime.dwLowDateTime;
		ftimeModify->dwHighDateTime = FindFileData.ftLastWriteTime.dwHighDateTime;
	}
	return true;
}


// special mode for counting shared tracks information
void CAnimationCompiler::ReportMode(ConvertContext &cc, int iReportMode, const char * singleFile, ILoaderCGFListener * listener)
{

	std::vector<CString> fileList;
	std::string dirPath(cc.sourceFolder);
	std::string destPath;
	std::vector<uint32> sizes;
	ScanDirectoryRecursive(dirPath.c_str(),CString(""), CString("*.caf"), fileList, true, sizes);

	bool bSingleFile(false);

	for (uint32  anim = 0; anim < fileList.size(); ++anim)
	{

		std::string animFile(dirPath);
		animFile += fileList[anim];

		if ( (!bSingleFile) || (bSingleFile && (strstr(fileList[anim], singleFile) || strstr(singleFile, fileList[anim]))))
		{

			if (bSingleFile && strlen(singleFile) == 0)
				continue;
			if (bSingleFile && strlen(fileList[anim]) == 0)
				continue;

			CAnimation currentAnimation(0);

			SAnimationDesc desc;
			if (currentAnimation.LoadAnimationFromFile(animFile.c_str(), animFile.c_str(), listener, desc, false,m_Behavior.m_bNeedCalculate))
			{

				//if ( m_Behavior.m_bCheckLocomotion)
				{
					std::string res;
					if (currentAnimation.CheckLoco(res))
						cc.pLog->Log("File %s. %s", animFile.c_str(), res.c_str());
				}

				if (iReportMode == 1)
				{
					if (currentAnimation.HasOldFormat() && !currentAnimation.HasNewFormat())
					{
						cc.pLog->Log("File %s has only uncompressed chunks", animFile.c_str());
					}
				}
				else
					if (iReportMode == 2)
					{
						if (currentAnimation.HasOldFormat() && currentAnimation.HasNewFormat())
						{
							cc.pLog->Log("File %s has both uncompressed and compressed data", animFile.c_str());
						}
					}

					if (bSingleFile)
						return;
			}
			else
			{
				cc.pLog->Log("Failed processed file %s", animFile.c_str());
			}
		}
	}

}
static const __int64 SECS_BETWEEN_EPOCHS = 11644473600;
static const __int64 SECS_TO_100NS = 10000000; /* 10^7 */


DWORD FiletimeToUnixTime(const FILETIME& ft)
{
	return (DWORD)((((__int64&)ft) / SECS_TO_100NS) - SECS_BETWEEN_EPOCHS);
}


void RunThread(void * pData) 
{
	ThreadCompiler * pCompiler = (ThreadCompiler *)pData;
	pCompiler->Run();

}


int  CAnimationCompiler::ProcessSingleAnimation(CSkeletonInfo* currentSkeleton, const std::string& animFile, const std::string& destFile, 
																								ILoaderCGFListener * listener, ConvertContext &cc, SAnimationDesc& pDesc, int oldsize)
{
	static volatile int g_lockMemMan;
	WriteLock lock(g_lockMemMan);

	if (m_Behavior.m_bUseMultiThreading)
	{
		// wait
		while(m_vThreads.empty()) {
			Sleep(0);

		}
	}

	ThreadCompiler * pCurrThread = m_vThreads.back();
	m_vThreads.pop_back();

	pCurrThread->currentSkeleton = currentSkeleton;
	pCurrThread->animFile = animFile;
	pCurrThread->destFile = destFile; 
	pCurrThread->listener = listener;
	pCurrThread->cc = cc;
	pCurrThread->pDesc = pDesc;
	pCurrThread->oldsize = oldsize;
	pCurrThread->pCompiler = this;

	if (!m_Behavior.m_bUseMultiThreading)
		//(pCurrThread->Run();
		RunThread((void *)pCurrThread);
	else {
		cc.pLog->Log("Threads available %i. Started %i %s", m_vThreads.size(), pCurrThread->id, pCurrThread->destFile );
		_beginthread(RunThread, 0, (void *)pCurrThread);
	}

	return pCurrThread->resSize;
}

void ThreadCompiler::Run()
{
	//	int resSize = -1;
	// check for update
	FILETIME  updated, created;
	GetFileTime(animFile.c_str(), &updated, &created);
	DWORD latest = max(FiletimeToUnixTime(updated), FiletimeToUnixTime(created));
	static volatile int g_lockMemMan;
	if (pCompiler->m_Behavior.m_bNeedUpdateOnlyNew)
	{
		if (pCompiler->m_pTrackStorage.get())
		{
			//m_pTrackStorage
			DWORD stored;
			if (pCompiler->m_pTrackStorage->GetAnimFileTime(animFile, &stored))
			{
				// dont need update
				if (stored >= latest)
				{
					WriteLock lock(g_lockMemMan);

					pCompiler->m_iCount++;
					pCompiler->m_fAverage = pCompiler->m_fTotal/ (float)pCompiler->m_iCount;
					pCompiler->m_vThreads.push_back(this);

					return;
				}
			}
		}
	}

	//	cc.pLog->Log("Started %s", destFile.c_str());
	std::auto_ptr<CAnimation> currentAnimation;
	currentAnimation.reset(new CAnimation(currentSkeleton));
	bool bNeedCalculate(pCompiler->m_Behavior.m_bNeedCalculate);
	if (currentAnimation->LoadAnimationFromFile(animFile.c_str(), destFile.c_str(), listener, pDesc, true, pCompiler->m_Behavior.m_bNeedCalculate))
	{

		if (pCompiler->m_Behavior.m_bNeedCalculate && !currentAnimation->HasOldFormat())
		{
			// we don't have an uncompressed data. File not processed
			//			cc.pLog->LogError("File %s has only compressed data. File deleted", animFile.c_str());
			cc.pLog->Log("File %s has only compressed data. Stored", animFile.c_str());
			// free chunk file
			bNeedCalculate = false;
			//			currentAnimation.reset(0);
			//			DeleteFile(animFile.c_str());
			//			return 0;
		}

		if (!pCompiler->m_Behavior.m_bNeedCalculate && !currentAnimation->HasNewFormat())
		{
			// we don't have a compressed data. need calculate this
			cc.pLog->Log("File %s has only uncompressed data. Compiled", animFile.c_str());
			bNeedCalculate = true;
		}

		CAnimation::EDeleteMethod delFlag;


		bool bNeedProcessing(true);

		if (bNeedCalculate)
		{
			delFlag = CAnimation::eNew;

			if (pCompiler->m_Behavior.m_bNeedDeleteUncompressedData || pCompiler->m_Behavior.m_bNeedDeleteAllData)
				delFlag = CAnimation::eAll;
		}
		else
		{
			delFlag = CAnimation::eOld;

			if (currentAnimation->HasNewFormat())
			{
				bNeedProcessing = false;
			}
		}

		currentAnimation->DeleteOldChunk(delFlag, bNeedProcessing);

		if(currentAnimation->ProcessAnimation(listener, pCompiler->m_pTrackStorage.get(), bNeedCalculate, pCompiler->m_Behavior.m_bNeedUpdateOnlyNew))
		{
			resSize = currentAnimation->SaveAnimationToFile(destFile.c_str(), listener, cc, bNeedProcessing);

			GetFileTime(animFile.c_str(), &updated, &created);
			latest = max(FiletimeToUnixTime(updated), FiletimeToUnixTime(created));
			currentAnimation->m_dwTimestamp = latest;
			if (!pDesc.m_SkipSaveToDatabase)
				currentAnimation->SaveToDB(pCompiler->m_pTrackStorage.get(), pCompiler->m_Behavior.m_bNeedUpdateOnlyNew);
		}
		else
		{
			cc.pLog->LogError("Error in processing %s", animFile.c_str());
		}

		if (pCompiler->m_Behavior.m_bNeedSaveDBA && !pDesc.m_SkipSaveToDatabase)
		{
			currentAnimation.reset(0);
			DeleteFile(destFile.c_str());
		}
	}

	if (oldsize > 0)
	{
		float currentRatio = 100.0f - (float)resSize * 100.0f / (float)oldsize;
		pCompiler->m_fTotal += currentRatio;
		cc.pLog->Log("Processed %s. Old size %d. New size %d. Compress ratio %.1f%%", destFile.c_str(), oldsize, resSize, currentRatio );
	}

	WriteLock lock(g_lockMemMan);

	pCompiler->m_iCount++;
	pCompiler->m_fAverage = pCompiler->m_fTotal/ (float)pCompiler->m_iCount;
	pCompiler->m_vThreads.push_back(this);


	return ;
}


void CAnimationCompiler::PrepareThreads()
{
	int count = 1;

	if (m_Behavior.m_bUseMultiThreading)
		count = std::max(1, m_Behavior.m_iSupportedThreads);

	for (int i = 0; i < count; ++i) {
		m_vThreads.push_back(new ThreadCompiler);
		m_vThreads.back()->id = i;

	}
}

//bool 


bool CAnimationCompiler::Process( ConvertContext &cc ) 
{

	std::string descfile;

	if (stricmp(cc.sourceFileFinalExtension,"cba") == 0)
	{
		descfile = cc.getSourcePath();
	}

	if (cc.platform == PLATFORM_XBOX)
	{
		m_Behavior.m_bNeedSwapEndian = true;
	}

	CString singleFile;
	bool bSingleFile = cc.config->Get("file", singleFile);
	std::string fix(singleFile);
	FixString(fix);
	singleFile = fix.c_str();

	CString destFolder;
	bool bDestFolder = cc.config->Get("dest", destFolder);

	int iReportMode(false);
	bool bReport = cc.config->Get("report", iReportMode);

	int iCheckloco(false);
	bool bCheckLoco = cc.config->Get("checkloco", iCheckloco);

	m_Behavior.m_bCheckLocomotion = bCheckLoco && iCheckloco > 0;


	{

		bool bCleanup(false);
		bool bF =	cc.config->Get("cleanupfast", bCleanup);

		bCleanup = bCleanup && bF;

		bool bOnlynew(false);
		bF =	cc.config->Get("cleanup", bOnlynew);

		bOnlynew = bOnlynew && bF;

		//int iCountTracks(0);
		//bool bCountTracks = cc.config->Get("statistics", iCountTracks);

		int iBuildTracks(0);
		bool bBuildTracks = cc.config->Get("build", iBuildTracks);

		int iUpdateTracks(0);

		bool bUpdateTracks = cc.config->Get("update", iUpdateTracks);

		//int iThreadsNumber(0);
		//bool bUseMultiThreading = cc.config->Get("threads", iThreadsNumber);

		m_Behavior.m_bUseMultiThreading = !bSingleFile && cc.threads > 1;//bUseMultiThreading;
		m_Behavior.m_iSupportedThreads = cc.threads;

		if (m_Behavior.m_bNeedSwapEndian)
		{
			bUpdateTracks = false;
			bBuildTracks = true;
		}



		// set vars for behavior
		if (!bCleanup)//bCleanup || iCompress)
			m_Behavior.m_bNeedCalculate = true;

		if (iUpdateTracks)	
		{
			m_Behavior.m_bNeedLoadDBA = true; 
			m_Behavior.m_bNeedUpdateOnlyNew = true;
		}

		if (iUpdateTracks || iBuildTracks)
			m_Behavior.m_bNeedSaveDBA = true; 

		if (bCleanup)
			m_Behavior.m_bNeedDeleteUncompressedData = true;

		if (bOnlynew)
			m_Behavior.m_bNeedDeleteAllData = true;
	}

	if (m_Behavior.m_bNeedSaveDBA)
	{
		m_pTrackStorage.reset(new CTrackStorage);
	}

	class Listener : public ILoaderCGFListener
	{
	public:
		Listener(ConvertContext& cc): cc(cc) {}
		virtual void Warning( const char *format ) {cc.pLog->LogWarning("%s", format);}
		virtual void Error( const char *format ) {cc.pLog->LogError("%s", format);}

	private:
		ConvertContext& cc;
	};
	Listener listener(cc);


	if (bReport && iReportMode > 0)
	{
		ReportMode(cc, iReportMode, singleFile, &listener);
		return true;
	}

	PrepareThreads();

	if (bSingleFile)
	{
		m_pTrackStorage.reset(0);
	}

	CAnimationInfoLoader loader(m_pXMLParser);

	CAnimationLoaderListener listenerAnim(cc);
	if (!loader.LoadDescription(descfile, &listenerAnim))
		return false;


	bool bProcessed(false);

	for (uint32 i = 0; i < loader.m_Definitions.size(); ++i)
	{
		// Load model and process job
		std::string filename(cc.sourceFolder);
		//		if (bDestFolder)
		//			filename =  destFolder;
		filename += loader.m_Definitions[i].m_Model;

		CSkeletonInfo currentSkeleton;
		if (currentSkeleton.LoadCHRModel(filename.c_str(), &listener))
		{
			//fill list with filenames
			std::vector<CString> fileList;
			std::string dirPath(cc.sourceFolder);
			std::string destPath;
			dirPath += loader.m_Definitions[i].GetAnimationPathWithoutSlash() + "\\";
			destPath = dirPath;
			std::vector<uint32> sizes;
			ScanDirectoryRecursive(dirPath.c_str(),CString(""), CString("*.caf"), fileList, true, sizes);

			if (bDestFolder)
			{
				destPath = destFolder;
				destPath += "\\" + loader.m_Definitions[i].GetAnimationPathWithoutSlash() + "\\";
			}

			// clear trackstorage
			if (m_pTrackStorage.get())
			{
				m_pTrackStorage->Clear();

			}


			std::string dbFile;

			bool yep = true;
			for (uint32 anim = 0; anim < fileList.size(); ++anim)
			{

				std::string animFile(dirPath);
				animFile += fileList[anim];
				FixString(animFile);
				std::string destFile(destPath);
				destFile += fileList[anim];

				const SAnimationDefinition * pDefinition = loader.GetAnimationDefinition(animFile);
				if (!pDefinition)
				{
					//FIXME: Add message - definition not found!
					continue;
				}


				if (bDestFolder)
					dbFile = destFolder + "\\";
				else
					dbFile = cc.sourceFolder;

				dbFile += pDefinition->m_DBName;

				if (m_pTrackStorage.get() && m_Behavior.m_bNeedLoadDBA && yep)
				{
					yep = false;
					m_pTrackStorage->LoadDataBase(dbFile.c_str());
				}

				SAnimationDesc pDesc;

				if ( (!bSingleFile) || (bSingleFile && (strstr(/*fileList[anim]*/animFile.c_str(), singleFile) || strstr(singleFile, /*fileList[anim]*/animFile.c_str()))))
				{
					pDesc =  pDefinition->GetAnimationDesc(std::string(fileList[anim]));

					if (bSingleFile && strlen(singleFile) == 0)
						continue;
					if (bSingleFile && strlen(fileList[anim]) == 0)
						continue;

					int resSize = ProcessSingleAnimation(&currentSkeleton, animFile, destFile, &listener, cc, pDesc, sizes[anim]);//, bCleanup, bOnlynew );

					if(resSize)
						bProcessed = true;

					//if (sizes[anim] > 0)
					//{
					//	float currentRatio = 100.0f - (float)resSize * 100.0f / (float)sizes[anim];
					//	cc.pLog->Log("Processed %s. Old size %d. New size %d. Compress ratio %.1f%%", destFile.c_str(), sizes[anim], resSize, currentRatio );
					//	total += currentRatio;
					//}
					//count++;
					//average = total/ (float)count;

					if (bSingleFile)
					{
						cc.pLog->Log("Processed %d files. Average compress ratio %f%%", m_iCount, m_fAverage);
						return true;
					}
				}
			}

			if (m_Behavior.m_bUseMultiThreading)
			{
				// wait
				while(m_vThreads.size() != m_Behavior.m_iSupportedThreads) {
					Sleep(0);

				}
			}


			if (m_pTrackStorage.get())
			{
				//				m_pTrackStorage->SaveDataBase900(dbFile.c_str());
				if (m_Behavior.m_bNeedSwapEndian)
					m_pTrackStorage->SaveDataBase904(dbFile.c_str());
				else
					m_pTrackStorage->SaveDataBase903(dbFile.c_str());
			}
		}
	} 


	if (!bProcessed && bSingleFile)
	{
		//	// that mean we find no model or something
		//	// try to process with default param
		//	SAnimationDesc pDesc;
		//	int resSize = ProcessSingleAnimation(0, std::string(singleFile), std::string(singleFile), &listener, cc, pDesc, bCleanup, bOnlynew );
		cc.pLog->LogError("Probably the animation(%s) was exported to an external path.\nTo export compressed data please save your file to:\nJ:\\Game02\\Game\\Animations", singleFile);
		DeleteFile(singleFile);
	}

	cc.pLog->Log("Processed %d files. Average compress ratio %.1f%%.  ", m_iCount, m_fAverage  );

	if (m_pTrackStorage.get())
	{
		cc.pLog->Log("All models shared %d tracks. Total tracks %d. Shared memory %d. Total Memory %d", m_TotalShared, m_TotalTracks, m_TotalMemoryShared , m_TotalMemory  );
	}

	return true;
}

bool CAnimationCompiler::GetOutputFile(ConvertContext &cc)
{
	cc.outputFile = cc.sourceFileFinal;
	if (!cc.masterFolder.IsEmpty())
		cc.m_sOutputFolder = Path::AddBackslash(cc.masterFolder) + cc.m_sOutputFolder;
	return true;
}

ICompiler* CAnimationCompiler::CreateCompiler()
{
	// Only ever return one compiler, since we don't support multithreading. Since
	// the compiler is just this object, we can tell whether we have already returned
	// a compiler by checking the ref count.
	if (m_refCount >= 2)
		return 0;

	// Until we support multithreading for this convertor, the compiler and the
	// convertor may as well just be the same object.
	++m_refCount;
	return this;
}

bool CAnimationCompiler::SupportsMultithreading() const
{
	return false;
}

//////////////////////////////////////////////////////////////////////////
// the paths must have trailing slash
static bool ScanDirectoryRecursive( const CString &root,const CString &path,const CString &file,std::vector<CString> &files, bool recursive, std::vector<uint32>& sizes )
{
	__finddata64_t c_file;
	intptr_t hFile;

	bool anyFound = false;

	CString fullPath = root + path + file;
	if ( (hFile = _findfirst64( fullPath.GetString(), &c_file )) != -1L )
	{
		// Find the rest of the files.
		do {
			if (!(c_file.attrib & _A_SUBDIR))
			{
				anyFound = true;
				files.push_back( path + c_file.name );
				sizes.push_back(c_file.size);
			}
		}	while (_findnext64( hFile, &c_file ) == 0);
		_findclose( hFile );
	}

	if (recursive)
	{
		fullPath = root + path + "*.*";
		if( (hFile = _findfirst64( fullPath.GetString(), &c_file )) != -1L )
		{
			// Find directories.
			do {
				if (c_file.attrib & _A_SUBDIR)
				{
					// If recursive.
					if (c_file.name[0] != '.')
					{
						if (ScanDirectoryRecursive( root,path + c_file.name + "\\",file,files,recursive, sizes ))
							anyFound = true;
					}
				}
			}	while (_findnext64( hFile, &c_file ) == 0);
			_findclose( hFile );
		}
	}

	return anyFound;
}
