////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2006.
// -------------------------------------------------------------------------
//  File name:   ColladaCompiler.cpp
//  Version:     v1.00
//  Created:     3/4/2006 by Michael Smith
//  Compilers:   Visual Studio.NET 2005
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ColladaCompiler.h"
#include "ColladaLoader.h"
#include "CryVersion.h"
#include "CGF\CGFSaver.h"
#include "StaticObjectCompiler.h"
#include "StatCGFPhysicalize.h"
#include "cgf/CGFNodeMerger.h"

void GenerateDefaultUVs(CMesh& mesh);

ColladaCompiler::ColladaCompiler(ICryXML* pCryXML)
:	pCryXML(pCryXML),
	m_refCount(1)
{
	this->pCryXML->AddRef();
	this->pPhysicsInterface = 0;
}

ColladaCompiler::~ColladaCompiler()
{
	delete this->pPhysicsInterface;
	this->pCryXML->Release();
}

void ColladaCompiler::Release()
{
	if (--m_refCount <= 0)
		delete this;
}

class ColladaLoaderListener : public IColladaLoaderListener
{
public:
	ColladaLoaderListener(ConvertContext &cc)
		:	cc(cc)
	{
	}

	virtual void OnColladaLoaderMessage(MessageType type, const char* szMessage)
	{
		switch (type)
		{
		case MESSAGE_INFO:
			cc.pLog->Log("%s", szMessage);
			break;
		case MESSAGE_WARNING:
			cc.pLog->LogWarning("%s", szMessage);
			break;
		case MESSAGE_ERROR:
			cc.pLog->LogError("%s", szMessage);
			break;
		}
	}

private:
	ConvertContext &cc;
};

bool ColladaCompiler::Process(ConvertContext &cc)
{
	CString inputFile = cc.getSourcePath();
	CString outputFile = cc.getSourcePath();
	outputFile = Path::ReplaceExtension(outputFile, "cgf");

	ColladaLoaderListener listener(cc);
	CContentCGF* pCGF = ColladaLoader().Load(inputFile, this->pCryXML, &listener);
	if (!pCGF)
	{
		return false;
	}

	if (!this->PrepareCGF(pCGF))
	{
		delete pCGF;
		return false;
	}

	// Make an object to compile the nodes.
	if (this->pPhysicsInterface == 0)
		this->pPhysicsInterface = new CPhysicsInterface;
	CStaticObjectCompiler compiler(this->pPhysicsInterface);

	// Merge all the nodes together, if required.
	if (pCGF->GetExportInfo()->bMergeAllNodes)
	{
		// Create a list of the used material ids.
		std::vector<int> usedMaterialIDs;
		FindUsedMaterialIDs(usedMaterialIDs, pCGF);

		string errorMessage;
		if (!CCGFNodeMerger().MergeNodes(pCGF, usedMaterialIDs, errorMessage))
		{
			cc.pLog->LogError("Error merging nodes: %s", errorMessage.c_str());
			return false;
		}
	}

	// Compile the cgf.
	CContentCGF* pCompiledCGF = compiler.MakeCompiledCGF(pCGF);

	// Set export data.
	bool success = false;
	if (pCompiledCGF)
	{
		SFileVersion fv = cc.pRC->GetFileVersion();
		pCompiledCGF->GetExportInfo()->rc_version[0] = fv.v[0];
		pCompiledCGF->GetExportInfo()->rc_version[1] = fv.v[1];
		pCompiledCGF->GetExportInfo()->rc_version[2] = fv.v[2];
		pCompiledCGF->GetExportInfo()->rc_version[3] = fv.v[3];
		sprintf(pCompiledCGF->GetExportInfo()->rc_version_string," RCVer:%d.%d ",fv.v[2],fv.v[1]);

		// Save the processed data to a chunkfile.
		CChunkFile chunkFile;
		CSaverCGF cgfSaver(outputFile, chunkFile);

		cgfSaver.SetContent(pCompiledCGF);

		// Only store 
		cgfSaver.SaveExportFlags();
		cgfSaver.SaveMaterials();
		cgfSaver.SaveNodes();
		cgfSaver.SaveBreakablePhysics();

		// Force remove of the read only flag and after write restore previous file flags.
		SetFileAttributes(outputFile,FILE_ATTRIBUTE_ARCHIVE);
		chunkFile.Write(outputFile);
		success = true;
	}

	delete pCGF;
	delete pCompiledCGF;
	return success;
}

bool ColladaCompiler::GetOutputFile(ConvertContext &cc)
{
	cc.outputFile = cc.sourceFileFinal;
	if (!cc.masterFolder.IsEmpty())
		cc.m_sOutputFolder = Path::AddBackslash(cc.masterFolder) + cc.m_sOutputFolder;
	return true;
}

ICompiler* ColladaCompiler::CreateCompiler()
{
	// Only ever return one compiler, since we don't support multithreading. Since
	// the compiler is just this object, we can tell whether we have already returned
	// a compiler by checking the ref count.
	if (m_refCount >= 2)
		return 0;

	// Until we support multithreading for this convertor, the compiler and the
	// convertor may as well just be the same object.
	++m_refCount;
	return this;
}

bool ColladaCompiler::SupportsMultithreading() const
{
	return false;
}

int ColladaCompiler::GetNumPlatforms() const
{
	return 1;
}

Platform ColladaCompiler::GetPlatform(int index) const
{
	switch (index)
	{
	case 0:	return PLATFORM_PC;
	//case 1:	return PLATFORM_XBOX;
	//case 2:	return PLATFORM_PS2;
	//case 3:	return PLATFORM_GAMECUBE;
	};
	return PLATFORM_UNKNOWN;
}

int ColladaCompiler::GetNumExt() const
{
	return 1;
}

const char* ColladaCompiler::GetExt(int index) const
{
	return "dae";
}

DWORD ColladaCompiler::GetTimestamp() const
{
	return GetTimestampForLoadedLibrary(g_hInst);
}

bool ColladaCompiler::PrepareCGF(CContentCGF* pCGF)
{
	// Loop through all the meshes.
	for (int i = 0; i < pCGF->GetNodeCount(); i++) 
	{
		CNodeCGF *pNodeCGF = pCGF->GetNode(i);
		CMesh* mesh = pNodeCGF->pMesh;
		if (mesh)
		{
			// If the mesh has no UVs, generate them here.
			if (mesh->m_nCoorCount == 0)
				GenerateDefaultUVs(*mesh);
		}
	}

	return true;
}

void ColladaCompiler::FindUsedMaterialIDs(std::vector<int>& usedMaterialIDs, CContentCGF* pCGF)
{
	std::set<int> materialIDSet;
	for (int nodeIndex = 0; nodeIndex < pCGF->GetNodeCount(); ++nodeIndex)
	{
		CNodeCGF* node = pCGF->GetNode(nodeIndex);
		CMesh* mesh = 0;
		if (node && node->pMesh && !node->bPhysicsProxy && node->type == CNodeCGF::NODE_MESH)
			mesh = node->pMesh;
		if (mesh)
		{
			for (int subsetIndex = 0; subsetIndex < mesh->m_subsets.size(); ++subsetIndex)
			{
				SMeshSubset& subset = mesh->m_subsets[subsetIndex];
				materialIDSet.insert(subset.nMatID);
			}
		}
	}

	usedMaterialIDs.clear();
	std::copy(materialIDSet.begin(), materialIDSet.end(), back_inserter(usedMaterialIDs));
}

// Copied from MaxCryExport
void GenerateDefaultUVs(CMesh& mesh)
{
	unsigned numVerts = mesh.m_numVertices;
	mesh.SetTexCoordsCount(numVerts);

	// make up the cubic coordinates
	for (unsigned nVert = 0; nVert < numVerts; ++nVert)
	{
		Vec3& v = mesh.m_pPositions[nVert];
		Vec3 r(fabs(v.x), fabs(v.y), fabs(v.z));
		SMeshTexCoord uv = {0, 0};
		if (r.x > 1e-3 || r.y > 1e-3 || r.z > 1e-3)
		{
			if (r.x > r.y)
			{
				if (r.x > r.z)
				{
					// X rules
					uv.s = v.y/r.x;
					uv.t = v.z/r.x;
				}
				else
				{
					// Z rules
					uv.s = v.x / r.z;
					uv.t = v.y / r.z;
				}
			}
			else
			{
				// r.x < r.y
				if (r.y > r.z)
				{
					// Y rules
					uv.s = v.x / r.y;
					uv.t = v.z / r.y;
				}
				else
				{
					// Z rules
					uv.s = v.x / r.z;
					uv.t = v.y / r.z;
				}
			}
		}
		// now the texture coordinates are in the range [-1,1]
		// we want normalized to 0..1 texture coordinates
		uv.s = (uv.s + 1)/2;
		uv.t = (uv.t + 1)/2;

		mesh.m_pTexCoord[nVert] = uv;
	}

	// Store the texture vertex indices in the face.
	for (int faceIndex = 0; faceIndex < mesh.m_numFaces; ++faceIndex)
	{
		mesh.m_pFaces[faceIndex].t[0] = mesh.m_pFaces[faceIndex].v[0];
		mesh.m_pFaces[faceIndex].t[1] = mesh.m_pFaces[faceIndex].v[1];
		mesh.m_pFaces[faceIndex].t[2] = mesh.m_pFaces[faceIndex].v[2];
	}
}
