////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2006.
// -------------------------------------------------------------------------
//  File name:   ColladaLoader.cpp
//  Version:     v1.00
//  Created:     3/4/2006 by Michael Smith
//  Compilers:   Visual Studio.NET 2005
//  Description: 
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ColladaLoader.h"
#include "IXml.h"
#include "IXMLSerializer.h"
#include "MeshCompiler\MeshCompiler.h"
#include <cstdarg>
#include <vector>
#include <map>
#include <cctype>
#include <Cry_Vector3.h>

enum ColladaArrayType
{
	TYPE_FLOAT,
	TYPE_INT,
	TYPE_NAME,
	TYPE_BOOL,
	TYPE_ID
};

void ReportWarning(IColladaLoaderListener* pListener, const char* szFormat, ...);
void ReportInfo(IColladaLoaderListener* pListener, const char* szFormat, ...);
void ReportError(IColladaLoaderListener* pListener, const char* szFormat, ...);

template <typename T> class ResourceMap : public std::map<std::string, T*>
{
public:
	~ResourceMap()
	{
		for (iterator itEntry = this->begin(); itEntry != this->end(); ++itEntry)
			delete (*itEntry).second;
	}
};

template <typename T> class ResourceVector : public std::vector<T*>
{
public:
	~ResourceVector()
	{
		for (iterator itEntry = this->begin(); itEntry != this->end(); ++itEntry)
			delete (*itEntry);
	}
};

#pragma region Array
class IColladaArray
{
public:
	virtual ~IColladaArray() {}

	virtual bool ReadAsFloat(std::vector<float>& values, int stride, int offset, IColladaLoaderListener* pListener) = 0;
	virtual bool ReadAsInt(std::vector<int>& values, int stride, int offset, IColladaLoaderListener* pListener) = 0;
	virtual bool ReadAsName(std::vector<std::string>& values, int stride, int offset, IColladaLoaderListener* pListener) = 0;
	virtual bool ReadAsBool(std::vector<bool>& values, int stride, int offset, IColladaLoaderListener* pListener) = 0;
	virtual bool ReadAsID(std::vector<std::string>& values, int stride, int offset, IColladaLoaderListener* pListener) = 0;

	virtual void Reserve(int size) = 0;
	virtual bool Parse(const char* buf, IColladaLoaderListener* pListener) = 0;
	virtual int GetSize() = 0;
};

template <ColladaArrayType TypeCode> class ArrayType {};
template <> class ArrayType<TYPE_FLOAT>
{
public: 
	typedef float Type;
	static const char* GetName() {return "FLOAT";}
	static float Parse(const char*& buf, bool& success)
	{
		char* end;
		float value = float(std::strtod(buf, &end));
		if (end == buf)
			success = false;
		else
			buf = end;
		return value;
	}
};

template <> class ArrayType<TYPE_INT>
{
public: 
	typedef int Type; 
	static const char* GetName() {return "INT";}
	static int Parse(const char*& buf, bool& success)
	{
		char* end;
		int value = int(std::strtol(buf, &end, 0));
		if (end == buf)
			success = false;
		else
			buf = end;
		return value;
	}
};

template <> class ArrayType<TYPE_NAME>
{
public:
	typedef std::string Type;
	static const char* GetName() {return "NAME";}
	static std::string Parse(const char*& buf, bool& success)
	{
		const char* p = buf;
		while (std::isspace(*p))
		{
			++p;
			if (*p == 0)
			{
				success = false;
				return 0;
			}
		}

		std::string s;
		while (!std::isspace(*p) && *p != 0)
			s += *p++;

		buf = p;
		return s;
	}
};

template <> class ArrayType<TYPE_BOOL>
{
public: 
	typedef bool Type;
	static const char* GetName() {return "BOOL";}
	static bool Parse(const char*& buf, bool& success)
	{
		const char* p = buf;
		while (std::isspace(*p))
		{
			++p;
			if (*p == 0)
			{
				success = false;
				return 0;
			}
		}

		std::string s;
		while (!std::isspace(*p) && *p != 0)
			s += tolower(*p++);

		buf = p;
		return s != "false";
	}
};

template <> class ArrayType<TYPE_ID>
{
public:
	typedef std::string Type;
	static const char* GetName() {return "ID";}
	static std::string Parse(const char*& buf, bool& success)
	{
		const char* p = buf;
		while (std::isspace(*p))
		{
			++p;
			if (*p == 0)
			{
				success = false;
				return 0;
			}
		}

		std::string s;
		while (!std::isspace(*p) && *p != 0)
			s += *p++;

		buf = p;
		return s;
	}
};

template <ColladaArrayType TypeCode1, ColladaArrayType TypeCode2> class Converter
{
public:
	typedef typename ArrayType<TypeCode1>::Type DataType1;
	typedef typename ArrayType<TypeCode2>::Type DataType2;

	static void Convert(DataType1& out, DataType2 in) {assert(0);}
	static bool CheckConversion(IColladaLoaderListener* pListener)
	{
		ReportError(pListener, "Cannot convert values of type %s to type %s.", ArrayType<TypeCode2>::GetName(), ArrayType<TypeCode1>::GetName());
		return false;
	}
};

template <ColladaArrayType TypeCode1, ColladaArrayType TypeCode2> class ValidConverter
{
public:
	static bool CheckConversion(IColladaLoaderListener* pListener)
	{
		ReportInfo(pListener, "Converting values of type %s to %s.", ArrayType<TypeCode2>::GetName(), ArrayType<TypeCode1>::GetName());
		return true;
	}
};

template <ColladaArrayType TypeCode> class Converter<TypeCode, TypeCode>
{
public:
	typedef typename ArrayType<TypeCode>::Type DataType;

	static void Convert(DataType& out, DataType in)
	{
		out = in;
	}

	static bool CheckConversion(IColladaLoaderListener* pListener)
	{
		return true;
	}
};

template <> class Converter<TYPE_BOOL, TYPE_INT> : public ValidConverter<TYPE_BOOL, TYPE_INT>
{
public:
	static void Convert(bool& out, int in) {out = (in != 0);}
};

template <> class Converter<TYPE_INT, TYPE_BOOL> : public ValidConverter<TYPE_INT, TYPE_BOOL>
{
public:
	static void Convert(int& out, bool in) {out = int(in);}
};

template <> class Converter<TYPE_INT, TYPE_FLOAT> : public ValidConverter<TYPE_INT, TYPE_FLOAT>
{
public:
	static void Convert(int& out, float in) {out = int(in);}
};

template <> class Converter<TYPE_FLOAT, TYPE_INT> : public ValidConverter<TYPE_FLOAT, TYPE_INT>
{
public:
	static void Convert(float& out, int in) {out = float(in);}
};

template <ColladaArrayType TypeCode1, ColladaArrayType TypeCode2> class ArrayReader
{
public:
	typedef typename ArrayType<TypeCode1>::Type OutputType;
	typedef typename ArrayType<TypeCode2>::Type InputType;

	static bool Read(std::vector<OutputType>& values, int stride, int offset, IColladaLoaderListener* pListener, const std::vector<InputType> data)
	{
		if (!Converter<TypeCode1, TypeCode2>::CheckConversion(pListener))
			return false;
		values.reserve(data.size() / stride + 1);
		for (int index = offset; index < int(data.size()); index += stride)
		{
			OutputType value = OutputType();
			Converter<TypeCode1, TypeCode2>::Convert(value, data[index]);
			values.push_back(value);
		}
		return true;
	}
};

template <ColladaArrayType TypeCode> class ColladaArray : public IColladaArray
{
public:
	typedef typename ArrayType<TypeCode>::Type DataType;

	virtual bool ReadAsFloat(std::vector<float>& values, int stride, int offset, IColladaLoaderListener* pListener)
	{
		return ArrayReader<TYPE_FLOAT, TypeCode>::Read(values, stride, offset, pListener, this->data);
	}

	virtual bool ReadAsInt(std::vector<int>& values, int stride, int offset, IColladaLoaderListener* pListener)
	{
		return ArrayReader<TYPE_INT, TypeCode>::Read(values, stride, offset, pListener, this->data);
	}

	virtual bool ReadAsName(std::vector<std::string>& values, int stride, int offset, IColladaLoaderListener* pListener)
	{
		return ArrayReader<TYPE_NAME, TypeCode>::Read(values, stride, offset, pListener, this->data);
	}

	virtual bool ReadAsBool(std::vector<bool>& values, int stride, int offset, IColladaLoaderListener* pListener)
	{
		return ArrayReader<TYPE_BOOL, TypeCode>::Read(values, stride, offset, pListener, this->data);
	}

	virtual bool ReadAsID(std::vector<std::string>& values, int stride, int offset, IColladaLoaderListener* pListener)
	{
		return ArrayReader<TYPE_ID, TypeCode>::Read(values, stride, offset, pListener, this->data);
	}

	virtual void Reserve(int size)
	{
		this->data.reserve(size);
	}

	virtual bool Parse(const char* buf, IColladaLoaderListener* pListener)
	{
		this->data.clear();
		this->data.resize(100);

		// Loop until we have consumed the whole buffer.
		const char* p = buf;
		bool success = true;
		int index;
		for (index = 0; ; ++index)
		{
			// Consume any whitespace.
			while (isspace(*p))
			{
				++p;
			}
			if (*p == 0)
				break;

			if (size_t(index) >= this->data.size())
				this->data.resize(this->data.size() * 2);

			// Parse the next value.
			DataType value = ArrayType<TypeCode>::Parse(p, success);
			this->data[index] = value;
			if (!success)
				return false;
		}
		this->data.resize(index);
		return true;
	}

	virtual int GetSize()
	{
		return int(this->data.size());
	}

private:
	std::vector<DataType> data;
};

template class ColladaArray<TYPE_FLOAT>;
typedef ColladaArray<TYPE_FLOAT> ColladaFloatArray;
template class ColladaArray<TYPE_INT>;
typedef ColladaArray<TYPE_INT> ColladaIntArray;
template class ColladaArray<TYPE_NAME>;
typedef ColladaArray<TYPE_NAME> ColladaNameArray;
template class ColladaArray<TYPE_BOOL>;
typedef ColladaArray<TYPE_BOOL> ColladaBoolArray;
template class ColladaArray<TYPE_ID>;
typedef ColladaArray<TYPE_ID> ColladaIDArray;

IColladaArray* CreateColladaArray(ColladaArrayType type)
{
	switch (type)
	{
	case TYPE_FLOAT: return new ColladaFloatArray();
	case TYPE_INT: return new ColladaIntArray();
	case TYPE_NAME: return new ColladaNameArray();
	case TYPE_BOOL: return new ColladaBoolArray();
	case TYPE_ID: return new ColladaIDArray();
	default:
		return 0;
	}
}
#pragma endregion Array
class ColladaAssetMetaData
{
public:
	ColladaAssetMetaData(const Vec3& upAxis) : upAxis(upAxis) {}
	const Vec3& GetUpAxis() const {return this->upAxis;}

private:
	Vec3 upAxis;
};

class ColladaDataAccessor
{
public:
	ColladaDataAccessor(IColladaArray* array, int stride, int offset, int size);
	void AddParameter(const std::string& name, ColladaArrayType type, int offset);

	int GetSize(const std::string& param);
	bool ReadAsFloat(const std::string& param, std::vector<float>& values, IColladaLoaderListener* pListener);
	bool ReadAsInt(const std::string& param, std::vector<int>& values, IColladaLoaderListener* pListener);
	bool ReadAsName(const std::string& param, std::vector<std::string>& values, IColladaLoaderListener* pListener);
	bool ReadAsBool(const std::string& param, std::vector<bool>& values, IColladaLoaderListener* pListener);
	bool ReadAsID(const std::string& param, std::vector<std::string>& values, IColladaLoaderListener* pListener);

private:
	class Parameter
	{
	public:
		std::string name;
		ColladaArrayType type;
		int offset;
	};

	IColladaArray* array;
	typedef std::map<std::string, Parameter> ParameterMap;
	ParameterMap parameters;
	int stride;
	int offset;
	int size;
};

class ColladaDataSource
{
public:
	ColladaDataSource(ColladaDataAccessor* pDataAccessor);
	virtual ~ColladaDataSource();

	ColladaDataAccessor* GetDataAccessor() {return this->pDataAccessor;}

private:
	ColladaDataAccessor* pDataAccessor;
};


typedef ResourceMap<ColladaDataSource> DataSourceMap;

class ColladaVertexStream
{
public:
	ColladaVertexStream(ColladaDataSource* pPositionDataSource);
	bool ReadVertexPositions(Vec3* positions, int size, IColladaLoaderListener* pListener);
	int GetSize();

private:
	ColladaDataSource* pPositionDataSource;
};
typedef ResourceMap<ColladaVertexStream> VertexStreamMap;

class ColladaMaterial
{
public:
	ColladaMaterial(const std::string& name);
	const std::string& GetName();

private:
	std::string name;
};

class CGFMaterialMapEntry
{
public:
	CGFMaterialMapEntry(int id, CMaterialCGF* material) : id(id), material(material) {}
	int id;
	CMaterialCGF* material;
};

typedef std::map<ColladaMaterial*, CGFMaterialMapEntry> CGFMaterialMap;
typedef ResourceMap<ColladaMaterial> MaterialMap;

#pragma region Primitives
class IPrimitiveStream
{
public:
	virtual ~IPrimitiveStream() {}

	virtual bool ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener) = 0;
	virtual bool AddToMesh(CMesh* mesh, CGFMaterialMap& cgfMaterials, int subsetIndex, IColladaLoaderListener* pListener) = 0;
	virtual ColladaMaterial* GetMaterial() = 0;
};

class IndexGroup
{
public:
	enum {MAX_INDICES = 100};

	IndexGroup(const std::vector<int>& indices, int indexOffset, int indexCount);

	bool operator<(const IndexGroup& other) const;
	int operator[](int i) {return (*indices)[this->indexOffset + i];}

private:
	const std::vector<int>* indices;
	int indexOffset;
	int indexCount;
};

class BasePrimitiveStream : public IPrimitiveStream
{
	friend class IndexGroup;
public:
	BasePrimitiveStream();

	virtual bool ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener);
	virtual ColladaMaterial* GetMaterial();
	virtual bool AddToMesh(CMesh* mesh, CGFMaterialMap& cgfMaterials, int subsetIndex, IColladaLoaderListener* pListener);

protected:
	class IntStream
	{
	public:
		IntStream(const char* text);
		int Read();

	private:
		const char* position;
	};

	int ReadIndexGroup(IntStream& stream);
	void SetVertexStream(ColladaVertexStream* vertexStream);
	void SetNormalSource(ColladaDataSource* normalSource);
	void SetColourSource(ColladaDataSource* colourSource);
	void SetTexCoordSource(ColladaDataSource* colourSource);

	int indexGroupSize;
	std::vector<int> indexStore;
	std::vector<IndexGroup> mergedIndexToIndexGroupMap;
	std::map<IndexGroup, int> indexGroupToMergedIndexMap;
	ColladaVertexStream* vertexStream;
	int positionIndexOffset;
	ColladaDataSource* normalSource;
	int normalIndexOffset;
	ColladaDataSource* colourSource;
	int colourIndexOffset;
	ColladaDataSource* texCoordSource;
	int texCoordIndexOffset;
	ColladaMaterial* material;

private:
	int AddIndexGroup(const std::vector<int>& indices);
};

class BasePolygonPrimitiveStream : public BasePrimitiveStream
{
public:
	virtual bool AddToMesh(CMesh* mesh, CGFMaterialMap& cgfMaterials, int subsetIndex, IColladaLoaderListener* pListener);

protected:
	std::vector<int> polygonSizes;
	std::vector<int> indices;
};

class PolygonsPrimitiveStream : public BasePolygonPrimitiveStream
{
public:
	virtual bool ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener);
};

class TrifansPrimitiveStream : public BasePolygonPrimitiveStream
{
public:
	virtual bool ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener);
};

class PolyStripPrimitiveStream : public BasePolygonPrimitiveStream
{
public:
	virtual bool ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener);
};

class TrianglesPrimitiveStream : public BasePrimitiveStream
{
public:
	virtual bool ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener);
	virtual bool AddToMesh(CMesh* mesh, CGFMaterialMap& cgfMaterials, int subsetIndex, IColladaLoaderListener* pListener);

protected:
	std::vector<int> indices;
};

class TriStripsPrimitiveStream : public BasePrimitiveStream
{
public:
	virtual bool ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener);
	virtual bool AddToMesh(CMesh* mesh, CGFMaterialMap& cgfMaterials, int subsetIndex, IColladaLoaderListener* pListener);

protected:
	std::vector<int> stripSizes;
	std::vector<int> indices;
};

#pragma endregion Primitives

typedef ResourceMap<IColladaArray> ArrayMap;

class ColladaMesh
{
public:
	ColladaMesh(std::vector<IPrimitiveStream*>& streams);
	~ColladaMesh();

	CMesh* CreateMesh(CGFMaterialMap& cgfMaterials, IColladaLoaderListener* pListener);

private:
	std::vector<IPrimitiveStream*> streams;
};

class ColladaGeometry
{
public:
	ColladaGeometry(ColladaMesh* pMesh);
	CMesh* CreateMesh(CGFMaterialMap& cgfMaterials, IColladaLoaderListener* pListener);

private:
	ColladaMesh* pMesh;
};

typedef ResourceMap<ColladaGeometry> GeometryMap;

class ColladaNode
{
public:
	explicit ColladaNode(ColladaGeometry* pGeometry);
	Matrix34& GetTransform();
	void SetName(const std::string& name);
	const std::string& GetName() const {return this->name;}

	CNodeCGF* CreateNodeCGF(CMaterialCGF* rootMaterial, CGFMaterialMap& cgfMaterials, ColladaAssetMetaData* metaData, IColladaLoaderListener* pListener);

private:
	std::string name;
	ColladaGeometry* pGeometry;
	Matrix34 transform;
};

typedef ResourceVector<ColladaNode> NodeVector;

class ColladaScene
{
public:
	ColladaScene();
	~ColladaScene();
	void AddNode(ColladaNode* pNode);

	CContentCGF* CreateContentCGF(const std::string& filename, MaterialMap& materials, ColladaAssetMetaData* metaData, IColladaLoaderListener* pListener);
	CMaterialCGF* CreateMaterial(CGFMaterialMap& cgfMaterials, MaterialMap& materials, IColladaLoaderListener* pListener);

	std::vector<ColladaNode*> nodes;
};

typedef ResourceMap<ColladaScene> SceneMap;

class ColladaLoaderImpl
{
public:
	CContentCGF* Load(const char* szFileName, ICryXML* pCryXML, IColladaLoaderListener* pListener);

private:
	XmlNodeRef LoadXML(const char* szFileName, ICryXML* pCryXML, IColladaLoaderListener* pListener);
	ColladaAssetMetaData* ParseMetaData(XmlNodeRef node);
	bool ParseElementIDs(XmlNodeRef node, std::map<std::string, XmlNodeRef>& idMap, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener);
	bool ParseArrays(ArrayMap& arrays, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener);
	IColladaArray* ParseArray(XmlNodeRef node, ColladaArrayType type, IColladaLoaderListener* pListener);
	bool ParseMaterials(MaterialMap& materials, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener);
	ColladaMaterial* ParseMaterial(XmlNodeRef node, IColladaLoaderListener* pListener);
	bool ParseDataSources(DataSourceMap& sources, ArrayMap& arrays, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener);
	ColladaDataSource* ParseDataSource(XmlNodeRef node, ArrayMap& arrays, IColladaLoaderListener* pListener);
	bool ParseVertexStreams(VertexStreamMap& vertexStreams, DataSourceMap& dataSources, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener);
	ColladaVertexStream* ParseVertexStream(XmlNodeRef node, DataSourceMap& dataSources, IColladaLoaderListener* pListener);
	bool ParseGeometries(GeometryMap& geometries, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener);
	ColladaGeometry* ParseGeometry(XmlNodeRef node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener);
	ColladaScene* ParseScene(XmlNodeRef sceneNode, NodeVector& nodes, SceneMap& scenes, GeometryMap& geometries, IColladaLoaderListener* pListener);
	bool ParseSceneDefinitions(SceneMap& scenes, NodeVector& nodes, GeometryMap& geometries, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener);
	ColladaScene* ParseSceneDefinition(XmlNodeRef sceneNode, NodeVector& nodes, GeometryMap& geometries, IColladaLoaderListener* pListener);
	ColladaScene* ParseSceneNode(XmlNodeRef sceneNode, NodeVector& nodes, GeometryMap& geometries, IColladaLoaderListener* pListener);
};

CContentCGF* ColladaLoader::Load(const char* szFileName, ICryXML* pCryXML, IColladaLoaderListener* pListener)
{
	pCryXML->AddRef();
	CContentCGF* pCGF = ColladaLoaderImpl().Load(szFileName, pCryXML, pListener);
	pCryXML->Release();
	return pCGF;
}

CContentCGF* ColladaLoaderImpl::Load(const char* szFileName, ICryXML* pCryXML, IColladaLoaderListener* pListener)
{
	
	
	
	
	
	XmlNodeRef root = this->LoadXML(szFileName, pCryXML, pListener);
	if (root == 0)
		return 0;

	std::map<std::string, XmlNodeRef> idMap;
	std::multimap<std::string, XmlNodeRef> tagMap;
	if (!this->ParseElementIDs(root, idMap, tagMap, pListener))
		return 0;

	std::auto_ptr<ColladaAssetMetaData> metaData((tagMap.find("asset") != tagMap.end()) ? this->ParseMetaData((*tagMap.find("asset")).second) : 0);

	MaterialMap materials;
	if (!this->ParseMaterials(materials, tagMap, pListener))
		return 0;

	ArrayMap arrays;
	if (!this->ParseArrays(arrays, tagMap, pListener))
		return 0;

	DataSourceMap sources;
	if (!this->ParseDataSources(sources, arrays, tagMap, pListener))
		return 0;

	VertexStreamMap vertexStreams;
	if (!this->ParseVertexStreams(vertexStreams, sources, tagMap, pListener))
		return 0;

	GeometryMap geometries;
	if (!this->ParseGeometries(geometries, materials, vertexStreams, sources, tagMap, pListener))
		return 0;

	SceneMap scenes;
	NodeVector nodes;
	if (!this->ParseSceneDefinitions(scenes, nodes, geometries, tagMap, pListener))
		return 0;

	XmlNodeRef sceneNode = root->findChild("scene");
	if (!sceneNode)
		ReportError(pListener, "Document contains no \"scene\" node.");
	std::auto_ptr<ColladaScene> scene(sceneNode ? this->ParseScene(sceneNode, nodes, scenes, geometries, pListener) : 0);

	CContentCGF* cgf = 0;
	if (scene.get())
		cgf = scene->CreateContentCGF(szFileName, materials, metaData.get(), pListener);

	ReportInfo(pListener, "Finished loading COLLADA.");

	return cgf;
}

XmlNodeRef ColladaLoaderImpl::LoadXML(const char* szFileName, ICryXML* pCryXML, IColladaLoaderListener* pListener)
{
	// Get the xml serializer.
	IXMLSerializer* pSerializer = pCryXML->GetXMLSerializer();

	// Read in the input file.
	XmlNodeRef root;
	{
		char szErrorBuffer[1024];
		root = pSerializer->Read(szFileName, sizeof(szErrorBuffer), szErrorBuffer);
		if (!root)
		{
			ReportError(pListener, "Cannot read file \"%s\": %s\n", szFileName, szErrorBuffer);
			return 0;
		}
	}

	ReportInfo(pListener, "Finished loading xml.");

	return root;
}

ColladaAssetMetaData* ColladaLoaderImpl::ParseMetaData(XmlNodeRef node)
{
	// Try to read the world up axis.
	bool readUpAxis = false;
	Vec3 upAxis;
	XmlNodeRef upAxisNode = node->findChild("up_axis");
	if (upAxisNode)
	{
		enum Axis {X_AXIS, Y_AXIS, Z_AXIS};
		Axis axis = X_AXIS;
		bool chosenAxis = false;
		for (const char* pChar = upAxisNode->getContent(); *pChar; ++pChar)
		{
			switch (*pChar)
			{
			case 'x': case 'X': axis = X_AXIS; chosenAxis = true; break;
			case 'y': case 'Y': axis = Y_AXIS; chosenAxis = true; break;
			case 'z': case 'Z': axis = Z_AXIS; chosenAxis = true; break;
			}
		}

		if (chosenAxis)
		{
			switch (axis)
			{
			case X_AXIS: upAxis = Vec3(1.0f, 0.0f, 0.0f); break;
			case Y_AXIS: upAxis = Vec3(0.0f, 1.0f, 0.0f); break;
			case Z_AXIS: upAxis = Vec3(0.0f, 0.0f, 1.0f); break;
			}
			readUpAxis = true;
		}
	}

	if (!readUpAxis)
	{
		upAxis = Vec3(0.0f, 1.0f, 0.0f);
		readUpAxis = true;
	}

	return new ColladaAssetMetaData(upAxis);
}

bool ColladaLoaderImpl::ParseElementIDs(XmlNodeRef node, std::map<std::string, XmlNodeRef>& idMap, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener)
{
	// Check whether the node has an ID attribute.
	if (node->haveAttr("id"))
		idMap.insert(std::make_pair(node->getAttr("id"), node));

	// Add the node to the tag map.
	std::string tag = node->getTag();
	std::transform(tag.begin(), tag.end(), tag.begin(), std::tolower);
	tagMap.insert(std::make_pair(tag, node));

	// Recurse to the children.
	for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
	{
		XmlNodeRef child = node->getChild(childIndex);
		if (!this->ParseElementIDs(child, idMap, tagMap, pListener))
			return false;
	}

	return true;
}


bool ColladaLoaderImpl::ParseArrays(ArrayMap& arrays, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener)
{
	// Create a list of node tags that indicate arrays.
	std::pair<std::string, ColladaArrayType> typeList[] = {
		std::make_pair("float_array", TYPE_FLOAT),
		std::make_pair("int_array", TYPE_INT),
		std::make_pair("name_array", TYPE_NAME),
		std::make_pair("bool_array", TYPE_BOOL),
		std::make_pair("idref_array", TYPE_ID)};

		// Loop through all the types of array.
		for (int typeIndex = 0; typeIndex < sizeof(typeList) / sizeof(typeList[0]); ++typeIndex)
		{
			const std::string& tag = typeList[typeIndex].first;
			ColladaArrayType type = typeList[typeIndex].second;

			// Loop through all the elements with this tag.
			typedef std::multimap<std::string, XmlNodeRef> TagMap;
			std::pair<TagMap::iterator, TagMap::iterator> range = tagMap.equal_range(tag);
			for (TagMap::iterator itNodeEntry = range.first; itNodeEntry != range.second; ++itNodeEntry)
			{
				XmlNodeRef node = (*itNodeEntry).second;
				if (node->haveAttr("id"))
				{
					IColladaArray* pArray = this->ParseArray(node, type, pListener);
					if (pArray)
						arrays.insert(std::make_pair(node->getAttr("id"), pArray));
					else
						ReportWarning(pListener, "Failed to load %s node as array.", node->getTag());
				}
				else
				{
					ReportWarning(pListener, "%s node contains no id attribute.", node->getTag());
				}
			}
		}

		return true;
}

IColladaArray* ColladaLoaderImpl::ParseArray(XmlNodeRef node, ColladaArrayType type, IColladaLoaderListener* pListener)
{
	IColladaArray* pArray = CreateColladaArray(type);
	if (!pArray->Parse(node->getContent(), pListener))
	{
		delete pArray;
		ReportError(pListener, "%s contains invalid data.", node->getTag());
		return 0;
	}
	return pArray;
}

bool ColladaLoaderImpl::ParseDataSources(DataSourceMap& sources, ArrayMap& arrays, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener)
{
	typedef std::multimap<std::string, XmlNodeRef> TagMap;
	std::pair<TagMap::iterator, TagMap::iterator> range = tagMap.equal_range("source");
	for (TagMap::iterator itNodeEntry = range.first; itNodeEntry != range.second; ++itNodeEntry)
	{
		XmlNodeRef node = (*itNodeEntry).second;
		if (node->haveAttr("id"))
		{
			ColladaDataSource* pDataSource = this->ParseDataSource(node, arrays, pListener);
			if (pDataSource)
				sources.insert(std::make_pair(node->getAttr("id"), pDataSource));
			else
				ReportWarning(pListener, "Failed to load %s node as data source.", node->getTag());
		}
		else
		{
			ReportWarning(pListener, "%s node contains no id attribute.", node->getTag());
		}
	}

	return true;
}

ColladaArrayType AttrToColladaArrayType(std::string attribute)
{
	std::transform(attribute.begin(), attribute.end(), attribute.begin(), tolower);
	if (attribute == "float")
		return TYPE_FLOAT;
	if (attribute == "int")
		return TYPE_INT;
	if (attribute == "name")
		return TYPE_NAME;
	if (attribute == "bool")
		return TYPE_BOOL;
	if (attribute == "idref")
		return TYPE_ID;
	return ColladaArrayType(-1);
}


ColladaDataSource* ColladaLoaderImpl::ParseDataSource(XmlNodeRef node, ArrayMap& arrays, IColladaLoaderListener* pListener)
{
	ColladaDataAccessor* pAccessor = 0;

	// Look for the common technique tag - either a tag called technique with an
	// attribute of profile="COMMON" or technique_common tag (Collada v1.3/1.4).
	XmlNodeRef techniqueCommonNode = 0;
	for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
	{
		XmlNodeRef child = node->getChild(childIndex);
		std::string childTag = child->getTag();
		std::transform(childTag.begin(), childTag.end(), childTag.begin(), std::tolower);

		if (childTag == "technique_common")
		{
			techniqueCommonNode = child;
			break;
		}
		else if (childTag == "technique")
		{
			if (child->haveAttr("profile") && std::string("COMMON") == child->getAttr("profile"))
				techniqueCommonNode = child;
		}
	}

	if (techniqueCommonNode)
	{
		// Look inside the technique node for an accessor node.
		XmlNodeRef accessorNode = techniqueCommonNode->findChild("accessor");
		if (accessorNode)
		{
			if (accessorNode->haveAttr("source") && accessorNode->haveAttr("stride"))
			{
				int stride;
				accessorNode->getAttr("stride", stride);
				int size = 0;
				if (accessorNode->haveAttr("count"))
					accessorNode->getAttr("count", size);
				int offset = 0;
				if (accessorNode->haveAttr("offset"))
					accessorNode->getAttr("offset", offset);

				bool foundSource = false;
				std::string sourceID = accessorNode->getAttr("source");

				// Try to find the source reference id.
				size_t hashPos = sourceID.rfind('#');
				if (hashPos != string::npos && hashPos > 0)
				{
					ReportWarning(pListener, "\"accessor\" node uses non-local data URI - cannot load data source (\"%s\").", sourceID.c_str());
				}
				else if (hashPos != string::npos)
				{
					sourceID = sourceID.substr(hashPos + 1, string::npos);
					foundSource = true;
				}
				else
				{
					foundSource = true;
				}

				if (foundSource)
				{
					// Look up the data source.
					ArrayMap::iterator itArrayEntry = arrays.find(sourceID);
					if (itArrayEntry != arrays.end())
					{
						IColladaArray* array = (*itArrayEntry).second;
						pAccessor = new ColladaDataAccessor(array, stride, offset, size);

						// Look for parameter entries.
						int offset = 0;
						for (int childIndex = 0; childIndex < accessorNode->getChildCount(); ++childIndex)
						{
							XmlNodeRef child = accessorNode->getChild(childIndex);
							std::string childTag = child->getTag();
							std::transform(childTag.begin(), childTag.end(), childTag.begin(), tolower);
							if (childTag == "param")
							{
								if (child->haveAttr("type"))
								{
									// Read the type attribute and look up the type.
									ColladaArrayType type = AttrToColladaArrayType(child->getAttr("type"));
									if (type >= 0)
									{
										if (child->haveAttr("name"))
										{
											// Add the parameter to the accessor.
											std::string name = child->getAttr("name");
											std::transform(name.begin(), name.end(), name.begin(), tolower);
											pAccessor->AddParameter(name, type, offset);
										}
										else
										{
											ReportInfo(pListener, "\"param\" node has no \"name\" attribute - treating as dummy, skipping param.");
										}
									}
									else
									{
										ReportWarning(pListener, "\"param\" node has unknown \"type\" attribute (\"%s\" - skipping param.", child->getAttr("type"));
									}
								}
								else
								{
									ReportWarning(pListener, "\"param\" node needs to specify a \"type\" attribute - skipping param.");
								}
								++offset;
							}
						}
					}
					else
					{
						ReportWarning(pListener, "\"accessor\" node refers to non-existent data array (\"%s\") - cannot load data source.", sourceID.c_str());
						return 0;
					}
				}
				else
				{
					return 0;
				}
			}
			else
			{
				ReportWarning(pListener, "\"accessor\" node requires \"source\" and \"stride\" attributes.");
				return 0;
			}
		}
		else
		{
			ReportWarning(pListener, "Data source contains no accessor (cannot understand format without it).");
			return 0;
		}
	}
	else
	{
		ReportWarning(pListener, "Data source contains no common technique (cannot understand format without it).");
		return 0;
	}

	if (pAccessor != 0)
		return new ColladaDataSource(pAccessor);
	return 0;
}

bool ColladaLoaderImpl::ParseMaterials(MaterialMap& materials, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener)
{
	typedef std::multimap<std::string, XmlNodeRef> TagMap;
	std::pair<TagMap::iterator, TagMap::iterator> range = tagMap.equal_range("material");
	for (TagMap::iterator itNodeEntry = range.first; itNodeEntry != range.second; ++itNodeEntry)
	{
		XmlNodeRef node = (*itNodeEntry).second;
		if (node->haveAttr("id"))
		{
			ColladaMaterial* pMaterial = this->ParseMaterial(node, pListener);
			if (pMaterial)
				materials.insert(std::make_pair(node->getAttr("id"), pMaterial));
			else
				ReportWarning(pListener, "Failed to load %s node as material.", node->getTag());
		}
		else
		{
			ReportWarning(pListener, "%s node contains no id attribute.", node->getTag());
		}
	}

	return true;
}

ColladaMaterial* ColladaLoaderImpl::ParseMaterial(XmlNodeRef node, IColladaLoaderListener* pListener)
{
	bool success = true;
	std::string name;

	// Read the name of the material.
	bool foundName = false;
	if (node->haveAttr("name"))
	{
		name = node->getAttr("name");
		foundName = true;
	}

	// If there is no name attribute, fall back to using the id as the name.
	if (!foundName && node->haveAttr("id"))
	{
		name = node->getAttr("id");
		foundName = true;
	}

	if (!foundName)
	{
		ReportWarning(pListener, "\"%s\" node has no name attribute.", node->getTag());
		success = false;
	}

	// If we were successful reading attributes, create the material.
	ColladaMaterial* material = 0;
	if (success)
		material = new ColladaMaterial(name);
	return material;
}

bool ColladaLoaderImpl::ParseVertexStreams(VertexStreamMap& vertexStreams, DataSourceMap& dataSources, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener)
{
	typedef std::multimap<std::string, XmlNodeRef> TagMap;
	std::pair<TagMap::iterator, TagMap::iterator> range = tagMap.equal_range("vertices");
	for (TagMap::iterator itNodeEntry = range.first; itNodeEntry != range.second; ++itNodeEntry)
	{
		XmlNodeRef node = (*itNodeEntry).second;
		if (node->haveAttr("id"))
		{
			ColladaVertexStream* pVertexStream = this->ParseVertexStream(node, dataSources, pListener);
			if (pVertexStream)
				vertexStreams.insert(std::make_pair(node->getAttr("id"), pVertexStream));
			else
				ReportWarning(pListener, "Failed to load %s node as vertex stream.", node->getTag());
		}
		else
		{
			ReportWarning(pListener, "%s node contains no id attribute.", node->getTag());
		}
	}

	return true;
}

ColladaVertexStream* ColladaLoaderImpl::ParseVertexStream(XmlNodeRef node, DataSourceMap& dataSources, IColladaLoaderListener* pListener)
{
	ColladaVertexStream* pVertexStream = 0;
	ColladaDataSource* pPositionDataSource = 0;

	// Loop through the children, looking for input elements.
	for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
	{
		XmlNodeRef inputNode = node->getChild(childIndex);
		std::string inputTag = inputNode->getTag();
		std::transform(inputTag.begin(), inputTag.end(), inputTag.begin(), tolower);
		if (inputTag == "input")
		{
			// Check whether this input specifies a semantic.
			if (inputNode->haveAttr("semantic"))
			{
				std::string semantic = inputNode->getAttr("semantic");
				std::transform(semantic.begin(), semantic.end(), semantic.begin(), tolower);
				if (semantic == "position")
				{
					if (inputNode->haveAttr("source"))
					{
						std::string sourceID = inputNode->getAttr("source");
						if (sourceID[0] == '#')
						{
							sourceID = sourceID.substr(1, string::npos);

							// Look up the data source.
							DataSourceMap::iterator itDataSourceEntry = dataSources.find(sourceID);
							if (itDataSourceEntry != dataSources.end())
							{
								pPositionDataSource = (*itDataSourceEntry).second;
							}
							else
							{
								ReportWarning(pListener, "\"input\" node refers to non-existent data array (\"%s\") - skipping input.", sourceID.c_str());
								return 0;
							}
						}
						else
						{
							ReportWarning(pListener, "\"input\" node uses non-local data URI - skipping input (\"%s\").", sourceID.c_str());
						}
					}
					else
					{
						ReportWarning(pListener, "\"input\" node does not specify data source - skipping input.");
					}
				}
			}
			else
			{
				ReportWarning(pListener, "\"input\" node does not specify semantic - skipping input.");
			}
		}
	}

	// If we found a source for position data, then create the vertex stream.
	if (pPositionDataSource != 0)
		pVertexStream = new ColladaVertexStream(pPositionDataSource);
	return pVertexStream;
}

bool ColladaLoaderImpl::ParseGeometries(GeometryMap& geometries, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener)
{
	typedef std::multimap<std::string, XmlNodeRef> TagMap;
	std::pair<TagMap::iterator, TagMap::iterator> range = tagMap.equal_range("geometry");
	for (TagMap::iterator itNodeEntry = range.first; itNodeEntry != range.second; ++itNodeEntry)
	{
		XmlNodeRef node = (*itNodeEntry).second;
		if (node->haveAttr("id"))
		{
			ColladaGeometry* pGeometry = this->ParseGeometry(node, materials, vertexStreams, dataSources, pListener);
			if (pGeometry)
				geometries.insert(std::make_pair(node->getAttr("id"), pGeometry));
			else
				ReportWarning(pListener, "Failed to load %s node as geometry.", node->getTag());
		}
		else
		{
			ReportWarning(pListener, "%s node contains no id attribute.", node->getTag());
		}
	}

	return true;
}

ColladaGeometry* ColladaLoaderImpl::ParseGeometry(XmlNodeRef node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener)
{
	ColladaMesh* pMesh = 0;

	// Look for a mesh element.
	XmlNodeRef meshNode = node->findChild("mesh");
	if (meshNode)
	{
		// Look for a polygon element.
		enum PolygonStreamType
		{
			POLY_SOURCE_LINES,
			POLY_SOURCE_LINESTRIPS,
			POLY_SOURCE_POLYGONS,
			POLY_SOURCE_POLYLIST,
			POLY_SOURCE_TRIANGLES,
			POLY_SOURCE_TRIFANS,
			POLY_SOURCE_TRISTRIPS
		};

		typedef std::map<std::string, PolygonStreamType> StreamTypeMap;
		StreamTypeMap streamTypeMap;
		streamTypeMap.insert(std::make_pair("lines", POLY_SOURCE_LINES));
		streamTypeMap.insert(std::make_pair("linestrips", POLY_SOURCE_LINESTRIPS));
		streamTypeMap.insert(std::make_pair("polygons", POLY_SOURCE_POLYGONS));
		streamTypeMap.insert(std::make_pair("polylist", POLY_SOURCE_POLYLIST));
		streamTypeMap.insert(std::make_pair("triangles", POLY_SOURCE_TRIANGLES));
		streamTypeMap.insert(std::make_pair("trifans", POLY_SOURCE_TRIFANS));
		streamTypeMap.insert(std::make_pair("tristrips", POLY_SOURCE_TRISTRIPS));

		// Loop through the children, looking for polygon sources.
		XmlNodeRef sourceNode = 0;
		std::vector<IPrimitiveStream*> streams;
		for (int i = 0; i < meshNode->getChildCount(); ++i)
		{
			sourceNode = meshNode->getChild(i);
			std::string sourceTag = sourceNode->getTag();
			std::transform(sourceTag.begin(), sourceTag.end(), sourceTag.begin(), tolower);
			StreamTypeMap::iterator itStreamTypeEntry = streamTypeMap.find(sourceTag);
			IPrimitiveStream* stream = 0;
			if (itStreamTypeEntry != streamTypeMap.end())
			{
				PolygonStreamType sourceType = (*itStreamTypeEntry).second;
				switch (sourceType)
				{
					// Unsupported source types.
				case POLY_SOURCE_LINES:
				case POLY_SOURCE_LINESTRIPS:
					ReportWarning(pListener, "\"%s\" node contains unsupported primitive source \"%s\".", meshNode->getTag(), sourceNode->getTag());
					break;

					// Supported source types.
				case POLY_SOURCE_POLYGONS:
					stream = new PolygonsPrimitiveStream();
					break;

				case POLY_SOURCE_TRIFANS:
					stream = new TrifansPrimitiveStream();
					break;

				case POLY_SOURCE_POLYLIST:
					stream = new PolyStripPrimitiveStream();
					break;

				case POLY_SOURCE_TRIANGLES:
					stream = new TrianglesPrimitiveStream();
					break;

				case POLY_SOURCE_TRISTRIPS:
					stream = new TriStripsPrimitiveStream();
					break;

				default:
					assert(0);
					break;
				}
			}

			// Check whether we found a primitive stream.
			if (stream != 0)
			{
				// Initialise the source object with data from the node.
				if (!stream->ParseNode(sourceNode, materials, vertexStreams, dataSources, pListener))
				{
					delete stream;
					stream = 0;
				}
				else
				{
					streams.push_back(stream);
				}
			}
		}

		// If we succeeded in finding a primitive stream, create the mesh object.
		if (!streams.empty())
		{
			pMesh = new ColladaMesh(streams);
		}
		else
		{
			ReportWarning(pListener, "\"%s\" node contains no supported primitive sources.", meshNode->getTag());
		}
	}
	else
	{
		ReportWarning(pListener, "\"geometry\" node contains no \"mesh\" node - only mesh geometry is supported.");
	}

	if (pMesh)
		return new ColladaGeometry(pMesh);
	return 0;
}

void ReadXmlNodeTransform(Matrix34& transform, XmlNodeRef node, IColladaLoaderListener* pListener)
{
	// Transforms are described in COLLADA as a series of elementary operations, like OpenGL. It
	// is assumed that vectors are stored in columns, and that transform matrices are post-multiplied.
	// This effectively means that transforms are given in the reverse order to which they are applied
	// to the node.

	// Initialise the transform.
	transform.SetIdentity();

	// Loop through all the children looking for transformations.
	for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
	{
		XmlNodeRef child = node->getChild(childIndex);
		std::string childTag = child->getTag();
		std::transform(childTag.begin(), childTag.end(), childTag.begin(), tolower);

		// Check whether this child represents a translation.
		if (childTag == "translate")
		{
			Vec3 translation(0.0f, 0.0f, 0.0f);
			const char* pos = child->getContent();
			for (int i = 0; i < 3; ++i)
			{
				char* end;
				double value = strtod(pos, &end);
				if (pos == end)
					break;
				pos = end;
				translation[i] = value;
			}

			// Apply the translation - make sure to post-multiply. This means that the new
			// transformation is applied *before* the previous transformations (COLLADA
			// specifies transforms in reverse order).
			transform = transform * Matrix34::CreateTranslationMat(translation);
		}
		// Check whether this child represents a rotation.
		else if (childTag == "rotate")
		{
			float values[4] = {0.0f};
			const char* pos = child->getContent();
			for (int i = 0; i < 4; ++i)
			{
				char* end;
				double value = strtod(pos, &end);
				if (pos == end)
					break;
				pos = end;
				values[i] = value;
			}

			// Apply the rotation - make sure to post-multiply. This means that the new
			// transformation is applied *before* the previous transformations (COLLADA
			// specifies transforms in reverse order).
			Vec3 axis(values[0], values[1], values[2]);
			float angleInRadians = DEG2RAD(values[3]);
			transform = transform * Matrix34::CreateRotationAA(angleInRadians, axis);
		}
		// Check whether this child represents a scale.
		else if (childTag == "scale")
		{
			Vec3 scale(0.0f, 0.0f, 0.0f);
			const char* pos = child->getContent();
			for (int i = 0; i < 3; ++i)
			{
				char* end;
				double value = strtod(pos, &end);
				if (pos == end)
					break;
				pos = end;
				scale[i] = value;
			}

			// Apply the scale - make sure to post-multiply. This means that the new
			// transformation is applied *before* the previous transformations (COLLADA
			// specifies transforms in reverse order).
			transform = transform * Matrix34::CreateScale(scale);
		}
		// Check whether this child represents a skew.
		else if (childTag == "skew")
		{
			ReportWarning(pListener, "\"node\" node contains \"skew\" transform, which is unsupported - ignoring skew.");
		}
	}
}

ColladaScene* ColladaLoaderImpl::ParseScene(XmlNodeRef sceneNode, NodeVector& nodes, SceneMap& scenes, GeometryMap& geometries, IColladaLoaderListener* pListener)
{
	// The nodes in the scene can either be listed as children of the scene node, or in visual_scene nodes.
	// First parse the scene node itself for nodes.
	ColladaScene* rootScene = this->ParseSceneNode(sceneNode, nodes, geometries, pListener);
	if (rootScene == 0)
		rootScene = new ColladaScene();

	// Make a list of all the scenes, to be merged together at the end.
	std::vector<ColladaScene*> sceneList;

	// Now look for instance_visual_scene nodes that refer to other scenes.
	for (int childIndex = 0; childIndex < sceneNode->getChildCount(); ++childIndex)
	{
		XmlNodeRef childNode = sceneNode->getChild(childIndex);
		std::string childTag = childNode->getTag();
		std::transform(childTag.begin(), childTag.end(), childTag.begin(), tolower);
		if (childTag == "instance_visual_scene")
		{
			// Check whether the node has a url attribute.
			if (childNode->haveAttr("url"))
			{
				// Try to find the visual_scene from the url attribute.
				bool foundVisualScene = true;
				std::string visualSceneID = childNode->getAttr("url");
				size_t hashPos = visualSceneID.rfind("#");
				if (hashPos != std::string::npos && hashPos > 0)
					foundVisualScene = false;
				else if (hashPos != std::string::npos)
					visualSceneID = visualSceneID.substr(1, std::string::npos);

				if (foundVisualScene)
				{
					// Look up the scene in the scene map.
					SceneMap::iterator itScene = scenes.find(visualSceneID);
					if (itScene != scenes.end())
						sceneList.push_back((*itScene).second);
					else
						ReportWarning(pListener, "\"%s\" node refers to non-existent scene - skipping.", childNode->getTag());
				}
				else
				{
					ReportWarning(pListener, "\"%s\" node refers to non-local scene - skipping.", childNode->getTag());
				}
			}
			else
			{
				ReportWarning(pListener, "\"%s\" node has no \"url\" attribute - skipping.", childNode->getTag());
			}
		}
	}

	// Now merge together all the nodes from all the scenes.
	for (std::vector<ColladaScene*>::iterator itScene = sceneList.begin(); itScene != sceneList.end(); ++itScene)
	{
		ColladaScene* scene = *itScene;
		std::copy(scene->nodes.begin(), scene->nodes.end(), std::back_inserter(rootScene->nodes));
	}

	return rootScene;
}

bool ColladaLoaderImpl::ParseSceneDefinitions(SceneMap& scenes, NodeVector& nodes, GeometryMap& geometries, std::multimap<std::string, XmlNodeRef>& tagMap, IColladaLoaderListener* pListener)
{
	typedef std::multimap<std::string, XmlNodeRef> TagMap;
	std::pair<TagMap::iterator, TagMap::iterator> range = tagMap.equal_range("visual_scene");
	for (TagMap::iterator itNodeEntry = range.first; itNodeEntry != range.second; ++itNodeEntry)
	{
		XmlNodeRef node = (*itNodeEntry).second;
		if (node->haveAttr("id"))
		{
			ColladaScene* pScene = this->ParseSceneDefinition(node, nodes, geometries, pListener);
			if (pScene)
				scenes.insert(std::make_pair(node->getAttr("id"), pScene));
			else
				ReportWarning(pListener, "Failed to load \"%s\" node as scene.", node->getTag());
		}
		else
		{
			ReportWarning(pListener, "%s node contains no id attribute.", node->getTag());
		}
	}

	return true;
}

ColladaScene* ColladaLoaderImpl::ParseSceneDefinition(XmlNodeRef sceneNode, NodeVector& nodes, GeometryMap& geometries, IColladaLoaderListener* pListener)
{
	return this->ParseSceneNode(sceneNode, nodes, geometries, pListener);
}

// Declare a stack to handle the recursion.
class NodeStackEntry
{
public:
	NodeStackEntry(XmlNodeRef parentNode) : parentNode(parentNode), childIndex(0) {}
	NodeStackEntry(const NodeStackEntry& other) {parentNode = other.parentNode; childIndex = other.childIndex;}
	~NodeStackEntry() {}
	NodeStackEntry& operator=(const NodeStackEntry& other) {parentNode = other.parentNode; childIndex = other.childIndex; return *this;}
	XmlNodeRef parentNode;
	int childIndex;
};

ColladaScene* ColladaLoaderImpl::ParseSceneNode(XmlNodeRef sceneNode, NodeVector& nodes, GeometryMap& geometries, IColladaLoaderListener* pListener)
{
	std::vector<NodeStackEntry> nodeStack;

	// Create a scene node.
	ColladaScene* scene = new ColladaScene();

	// Add the root node to the scene.
	nodeStack.push_back(NodeStackEntry(sceneNode));

	// Look for nodes in the scene.
	int unknownNodeCount = 0;
	while (!nodeStack.empty())
	{
		NodeStackEntry& stackTop = nodeStack.back();
		int nodeIndex = stackTop.childIndex++;
		XmlNodeRef parentNode = stackTop.parentNode;

		if (nodeIndex >= parentNode->getChildCount())
		{
			// All the children of this node have been processed - pop it off the stack.
			nodeStack.pop_back();
		}
		else
		{
			XmlNodeRef nodeNode = parentNode->getChild(nodeIndex);
			std::string nodeTag = nodeNode->getTag();
			std::transform(nodeTag.begin(), nodeTag.end(), nodeTag.begin(), tolower);
			if (nodeTag == "node")
			{
				// Push this node onto the stack.
				nodeStack.push_back(NodeStackEntry(nodeNode));

				// Check whether the node has an instance
				XmlNodeRef instanceNode = nodeNode->findChild("instance");
				if (!instanceNode)
					instanceNode = nodeNode->findChild("instance_geometry");
				if (instanceNode)
				{
					if (instanceNode->haveAttr("url"))
					{
						std::string instanceURL = instanceNode->getAttr("url");
						if (instanceURL[0] == '#')
						{
							instanceURL = instanceURL.substr(1, std::string::npos);

							// Look up the geometry in the geometry map.
							GeometryMap::iterator itGeometryEntry = geometries.find(instanceURL);
							if (itGeometryEntry != geometries.end())
							{
								ColladaGeometry* geometry = (*itGeometryEntry).second;

								// Create the node.
								ColladaNode* node = new ColladaNode(geometry);

								// Set the transform.
								ReadXmlNodeTransform(node->GetTransform(), nodeNode, pListener);

								// Set the name of the node.
								std::string name;
								if (nodeNode->haveAttr("name"))
								{
									name = nodeNode->getAttr("name");
								}
								else if (nodeNode->haveAttr("id"))
								{
									name = nodeNode->getAttr("id");
								}
								else
								{
									char str[128];
									sprintf( str,"unknownNode%d",(++unknownNodeCount) );
									name = str;
								}
								node->SetName(name);

								// Add the node to the scene.
								scene->AddNode(node);
								nodes.push_back(node);
							}
							else
							{
								ReportWarning(pListener, "\"instance\" node refers to non-existent geometry (%s).", instanceURL.c_str());
							}
						}
						else
						{
							ReportWarning(pListener, "\"instance\" node refers to external geometry data (%s) - this is not supported.", instanceURL.c_str());
						}
					}
					else
					{
						ReportWarning(pListener, "\"instance\" node has no \"url\" attribute.");
					}
				}
				else
				{
					ReportWarning(pListener, "Node \"%s\" contains no geometry instance.", nodeNode->getTag());
				}
			}
		}
	}

	// Return the created scene.
	return scene;
}

enum SplitMaterialNameResult
{
	MATERIAL_NAME_INVALID,
	MATERIAL_NAME_VALID,
	MATERIAL_NAME_NUMBERONLY
};

class CCustomMaterialSettings
{
public:
	int physicalizeType;
	bool sh;
	int shSides;
	bool shAmbient;
	float shOpacity;
};

bool ReadCustomMaterialSettingsFromMaterialName(const std::string& name, CCustomMaterialSettings& settings)
{
	bool settingsReadSuccessfully = true;

	enum {MAX_SETTINGS_STRING_LENGTH = 100};
	if (name.size() >= MAX_SETTINGS_STRING_LENGTH)
		settingsReadSuccessfully = false;

	// Keep track of what values we have read.
	bool readPhysicalizeType = false;
	int physicalizeType = 0;
	bool readSHSides = false;
	int shSides = 2;
	bool readSHAmbient = false;
	bool shAmbient = true;
	bool readSHOpacity = false;
	float shOpacity = 1.0f;

	// Loop through all the characters in the string.
	const char* position = name.c_str();
	string token;
	token.reserve(100);
	while (settingsReadSuccessfully && *position != 0)
	{
		// Skip any underscores.
		for (; *position == '_'; ++position);

		// Read until the next non-alpha character.
		token.clear();
		for (; ((*position) >= 'A' && (*position) <= 'Z') || ((*position) >= 'a' && (*position) <= 'z'); ++position)
			token.push_back(char(tolower(*position)));

		// Read the integer argument.
		char* end;
		int argument = strtol(position, &end, 10);
		bool argumentReadSuccessfully = true;
		if (position == end)
		{
			argumentReadSuccessfully = false;
			settingsReadSuccessfully = false;
		}
		else
		{
			position = end;
		}

		if (argumentReadSuccessfully)
		{
			// Check what token this is.
			if (token == "phys")
			{
				readPhysicalizeType = true;
				physicalizeType = argument;
			}
			else if (token == "shsides")
			{
				readSHSides = true;
				shSides = argument;
			}
			else if (token == "shambient")
			{
				readSHAmbient = true;
				shAmbient = (argument != 0);
			}
			else if (token == "shopacity")
			{
				readSHOpacity = true;
				shOpacity = float(argument) / 100.0f;
			}
			else
			{
				settingsReadSuccessfully = false;
			}
		}
	}

	// Apply the values we found.
	settings.physicalizeType = physicalizeType;
	bool sh = readSHSides || readSHAmbient || readSHOpacity;
	if (sh)
	{
		settings.sh = true;
		settings.shSides = shSides;
		settings.shAmbient = shAmbient;
		settings.shOpacity = shOpacity;
	}
	else
	{
		settings.sh = false;
		settings.shSides = 1;
		settings.shAmbient = false;
		settings.shOpacity = 0.0f;
	}

	return settingsReadSuccessfully;
}

SplitMaterialNameResult SplitMaterialName(const std::string& name, int& id, std::string& actualName, CCustomMaterialSettings& settings)
{
	// Name defaults to the whole string.
	actualName = name;

	// Ignore everything up to and including the last '.' in the string, if there is one.
	const char* position = name.c_str();
	const char* lastPeriod = strrchr(position, '.');
	if (lastPeriod)
		position = lastPeriod + 1;

	// Check whether the string starts with a number. There may be underscores in front of the number.
	// If so, we want to skip the underscores, but not if there is no number.
	bool startsWithNumber = true;
	int numberPosition;
	for (numberPosition = 0; position[numberPosition] == '_'; ++numberPosition);
	if (position[numberPosition] == 0)
		startsWithNumber = false;
	char* end;
	id = strtol(position + numberPosition, &end, 10);
	if (end == position + numberPosition)
		startsWithNumber = false;
	else
		position = end;

	// Check whether there is an _ between the number and the name, and if so skip it.
	if (startsWithNumber && *position == '_')
		++position;

	// Check whether there is a name after the id. There may also be material
	// directives, such as physicalisation flags.
	string nameBuffer;
	nameBuffer.reserve(200);
	while (*position != 0)
	{
		// Check whether the rest of the string specifies custom settings.
		if (ReadCustomMaterialSettingsFromMaterialName(position, settings))
			break;

		// Add all the underscores to the name.
		for (; *position == '_'; ++position)
			nameBuffer.push_back(*position);

		// Add the string up until the next _ to the name.
		for (; *position != '_' && *position != 0; ++position)
			nameBuffer.push_back(*position);
	}

	// Set the actual name if appropriate.
	bool hasName = !nameBuffer.empty();
	if (hasName)
		actualName = nameBuffer;

	// Return a value indicating how we went.
	if (startsWithNumber && hasName)
		return MATERIAL_NAME_VALID;
	else if (startsWithNumber)
		return MATERIAL_NAME_NUMBERONLY;
	else
		return MATERIAL_NAME_INVALID;
}

void ReportWarning(IColladaLoaderListener* pListener, const char* szFormat, ...)
{
	va_list args;
	va_start(args, szFormat);
	char szBuffer[1024];
	vsprintf(szBuffer, szFormat, args);
	pListener->OnColladaLoaderMessage(IColladaLoaderListener::MESSAGE_WARNING, szBuffer);
	va_end(args);
}

void ReportInfo(IColladaLoaderListener* pListener, const char* szFormat, ...)
{
	va_list args;
	va_start(args, szFormat);
	char szBuffer[1024];
	vsprintf(szBuffer, szFormat, args);
	pListener->OnColladaLoaderMessage(IColladaLoaderListener::MESSAGE_INFO, szBuffer);
	va_end(args);
}

void ReportError(IColladaLoaderListener* pListener, const char* szFormat, ...)
{
	va_list args;
	va_start(args, szFormat);
	char szBuffer[1024];
	vsprintf(szBuffer, szFormat, args);
	pListener->OnColladaLoaderMessage(IColladaLoaderListener::MESSAGE_ERROR, szBuffer);
	va_end(args);
}

ColladaDataAccessor::ColladaDataAccessor(IColladaArray* array, int stride, int offset, int size)
:	array(array),
stride(stride),
offset(offset),
size(size)
{
}

void ColladaDataAccessor::AddParameter(const std::string& name, ColladaArrayType type, int offset)
{
	Parameter param;
	param.name = name;
	param.offset = offset;
	param.type = type;
	this->parameters.insert(std::make_pair(name, param));
}

int ColladaDataAccessor::GetSize(const std::string& param)
{
	// Calculate the size based on the size of the array.
	int size = this->array->GetSize() / this->stride;

	// Return the smaller of that and the claimed size.
	if (size > this->size && this->size > 0)
		size = this->size;
	return size;
}

bool ColladaDataAccessor::ReadAsFloat(const std::string& param, std::vector<float>& values, IColladaLoaderListener* pListener)
{
	// Look up the parameter.
	ParameterMap::iterator itParameter = this->parameters.find(param);
	if (itParameter == this->parameters.end())
		return false;
	const Parameter& paramDef = (*itParameter).second;

	// Compute the offset based on the accessor offset and the parameter offset.
	int offset = this->offset + paramDef.offset;

	// Read the data from the array.
	return this->array->ReadAsFloat(values, this->stride, offset, pListener);
}

bool ColladaDataAccessor::ReadAsInt(const std::string& param, std::vector<int>& values, IColladaLoaderListener* pListener)
{
	// Look up the parameter.
	ParameterMap::iterator itParameter = this->parameters.find(param);
	if (itParameter == this->parameters.end())
		return false;
	const Parameter& paramDef = (*itParameter).second;

	// Compute the offset based on the accessor offset and the parameter offset.
	int offset = this->offset + paramDef.offset;

	// Read the data from the array.
	return this->array->ReadAsInt(values, this->stride, offset, pListener);
}

bool ColladaDataAccessor::ReadAsName(const std::string& param, std::vector<std::string>& values, IColladaLoaderListener* pListener)
{
	// Look up the parameter.
	ParameterMap::iterator itParameter = this->parameters.find(param);
	if (itParameter == this->parameters.end())
		return false;
	const Parameter& paramDef = (*itParameter).second;

	// Compute the offset based on the accessor offset and the parameter offset.
	int offset = this->offset + paramDef.offset;

	// Read the data from the array.
	return this->array->ReadAsName(values, this->stride, offset, pListener);
}

bool ColladaDataAccessor::ReadAsBool(const std::string& param, std::vector<bool>& values, IColladaLoaderListener* pListener)
{
	// Look up the parameter.
	ParameterMap::iterator itParameter = this->parameters.find(param);
	if (itParameter == this->parameters.end())
		return false;
	const Parameter& paramDef = (*itParameter).second;

	// Compute the offset based on the accessor offset and the parameter offset.
	int offset = this->offset + paramDef.offset;

	// Read the data from the array.
	return this->array->ReadAsBool(values, this->stride, offset, pListener);
}

bool ColladaDataAccessor::ReadAsID(const std::string& param, std::vector<std::string>& values, IColladaLoaderListener* pListener)
{
	// Look up the parameter.
	ParameterMap::iterator itParameter = this->parameters.find(param);
	if (itParameter == this->parameters.end())
		return false;
	const Parameter& paramDef = (*itParameter).second;

	// Compute the offset based on the accessor offset and the parameter offset.
	int offset = this->offset + paramDef.offset;

	// Read the data from the array.
	return this->array->ReadAsID(values, this->stride, offset, pListener);
}

ColladaDataSource::ColladaDataSource(ColladaDataAccessor* pDataAccessor)
:	pDataAccessor(pDataAccessor)
{
}

ColladaDataSource::~ColladaDataSource()
{
	delete pDataAccessor;
}

ColladaVertexStream::ColladaVertexStream(ColladaDataSource* pPositionDataSource)
:	pPositionDataSource(pPositionDataSource)
{
}


bool ColladaVertexStream::ReadVertexPositions(Vec3* positions, int size, IColladaLoaderListener* pListener)
{
	std::vector<float> xs;
	std::vector<float> ys;
	std::vector<float> zs;
	bool hasCorrectParams = true;
	hasCorrectParams = hasCorrectParams && this->pPositionDataSource->GetDataAccessor()->ReadAsFloat("x", xs, pListener);
	hasCorrectParams = hasCorrectParams && this->pPositionDataSource->GetDataAccessor()->ReadAsFloat("y", ys, pListener);
	hasCorrectParams = hasCorrectParams && this->pPositionDataSource->GetDataAccessor()->ReadAsFloat("z", zs, pListener);
	if (!hasCorrectParams)
	{
		ReportWarning(pListener, "Vertex stream requires XYZ parameters.");
		return false;
	}
	for (int i = 0; i < size; ++i)
	{
		if (i < int(xs.size()))
			positions[i].x = xs[i];
		else
			positions[i].x = 0.0f;
		if (i < int(ys.size()))
			positions[i].y = ys[i];
		else
			positions[i].y = 0.0f;
		if (i < int(zs.size()))
			positions[i].z = zs[i];
		else
			positions[i].z = 0.0f;
	}

	return true;
}

int ColladaVertexStream::GetSize()
{
	return this->pPositionDataSource->GetDataAccessor()->GetSize("x");
}

ColladaMesh::ColladaMesh(std::vector<IPrimitiveStream*>& streams)
:	streams(streams)
{
}

ColladaMesh::~ColladaMesh()
{
	for (std::vector<IPrimitiveStream*>::iterator itStream = this->streams.begin(); itStream != this->streams.end(); ++itStream)
		delete *itStream;
}

CMesh* ColladaMesh::CreateMesh(CGFMaterialMap& cgfMaterials, IColladaLoaderListener* pListener)
{
	CMesh* mesh = new CMesh();
	mesh->m_bbox.Reset();

	// Loop through all the primitive streams, adding them to the mesh.
	int subsetIndex = 0;
	for (std::vector<IPrimitiveStream*>::iterator itStream = this->streams.begin(); itStream != this->streams.end(); ++itStream)
	{
		// Add the geometry from this primitive stream to the mesh.
		IPrimitiveStream* stream = *itStream;
		if (!stream->AddToMesh(mesh, cgfMaterials, subsetIndex, pListener))
		{
			ReportWarning(pListener, "Failed to add stream to mesh.");
			delete mesh;
			return 0;
		}

		// Look up the material ID for the material associated with this primitive stream.
		CGFMaterialMap::iterator itMaterialEntry = cgfMaterials.find(stream->GetMaterial());
		int materialID = -1;
		if (itMaterialEntry != cgfMaterials.end())
			materialID = (*itMaterialEntry).second.id;
		else
			ReportWarning(pListener, "Polygon stream specifies material (\"%s\") with no ID.", stream->GetMaterial()->GetName().c_str());

		// Add a mesh subset for this geometry.
		SMeshSubset meshSubset;
		meshSubset.nMatID = materialID;
		meshSubset.nPhysicalizeType = PHYS_GEOM_TYPE_NONE;
		mesh->m_subsets.push_back(meshSubset);

		++subsetIndex;
	}

	// If there are texcoords, then there may have been some missing texcoords, as it is possible
	// that there was a primitive stream that omitted them. In this case, make sure we pad out the uvs
	// to match up with the vertex stream.
	if (mesh->GetTexCoordsCount() > 0)
	{
		int oldUVCount = mesh->GetTexCoordsCount();
		if (oldUVCount < mesh->GetVertexCount())
		{
			mesh->SetTexCoordsCount(mesh->GetVertexCount());
			SMeshTexCoord uv = {0.0f, 0.0f};
			std::fill(mesh->m_pTexCoord + oldUVCount, mesh->m_pTexCoord + mesh->GetTexCoordsCount(), uv);
		}
	}

	// If there are vertex colours, then there may have been some missing vertex colours, as it is possible
	// that there was a primitive stream that omitted them. In this case, make sure we pad out the vertex
	// colours to match up with the vertex stream.
	if (mesh->m_streamSize[CMesh::COLORS_0] > 0)
	{
		int oldVertexColourCount = mesh->m_streamSize[CMesh::COLORS_0];
		if (oldVertexColourCount < mesh->GetVertexCount())
		{
			mesh->ReallocStream(CMesh::COLORS_0, mesh->GetVertexCount());
			SMeshColor color = {0, 0, 0, 0xFF};
			std::fill(mesh->m_pColor0 + oldVertexColourCount, mesh->m_pColor0 + mesh->m_streamSize[CMesh::COLORS_0], color);
		}
	}

	return mesh;
}

ColladaGeometry::ColladaGeometry(ColladaMesh* pMesh)
:	pMesh(pMesh)
{
}

CMesh* ColladaGeometry::CreateMesh(CGFMaterialMap& cgfMaterials, IColladaLoaderListener* pListener)
{
	return this->pMesh->CreateMesh(cgfMaterials, pListener);
}

BasePrimitiveStream::BasePrimitiveStream()
:	indexGroupSize(-1),
	vertexStream(0),
	normalSource(0),
	colourSource(0),
	texCoordSource(0),
	positionIndexOffset(-1),
	normalIndexOffset(-1),
	colourIndexOffset(-1),
	texCoordIndexOffset(-1)
{
}

bool BasePrimitiveStream::ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener)
{
	// Check whether the node specifies a material.
	if (node->haveAttr("material"))
	{
		// Look for the material the object calls for.
		bool foundMaterial = true;
		std::string materialID = node->getAttr("material");
		size_t hashPos = materialID.rfind('#');
		if (hashPos != std::string::npos && hashPos == 0)
			materialID = materialID.substr(1, std::string::npos);
		else if (hashPos != std::string::npos)
			foundMaterial = false;

		if (foundMaterial)
		{
			// Look up the data source.
			MaterialMap::iterator itMaterialEntry = materials.find(materialID);
			if (itMaterialEntry != materials.end())
			{
				this->material = (*itMaterialEntry).second;
			}
			else
			{
				ReportWarning(pListener, "\"%s\" node refers to non-existent material (\"%s\").", node->getTag(), materialID.c_str());
				return 0;
			}
		}
		else
		{
			ReportWarning(pListener, "\"%s\" node uses non-local material URI (\"%s\").", node->getTag(), materialID.c_str());
		}
	}
	else
	{
		ReportInfo(pListener, "\"%s\" node does not specify a material.", node->getTag());
	}

	// Create a list of semantic types that map to data sources. Note that this does not involve vertex semantics, since
	// they must be handled specially, as they point to a <vertex> tag rather than a <source> tag.
	enum DataStreamType
	{
		DATA_STREAM_NORMAL,
		DATA_STREAM_COLOUR,
		DATA_STREAM_TEXCOORD
	};
	std::map<string, DataStreamType> semanticMap;
	semanticMap["normal"] = DATA_STREAM_NORMAL;
	semanticMap["color"] = DATA_STREAM_COLOUR;
	semanticMap["texcoord"] = DATA_STREAM_TEXCOORD;

	// Look through the child nodes for input nodes.
	int offset = 0;
	int numIndices = 0;
	for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
	{
		XmlNodeRef inputNode = node->getChild(childIndex);
		std::string inputTag = inputNode->getTag();
		std::transform(inputTag.begin(), inputTag.end(), inputTag.begin(), tolower);
		if (inputTag == "input")
		{
			// Check whether this input overrides the offset.
			const char* offsetAttr = 0;
			if (inputNode->haveAttr("idx"))
				offsetAttr = "idx";
			if (inputNode->haveAttr("offset"))
				offsetAttr = "offset";
			if (offsetAttr)
			{
				char* end;
				const char* buf = inputNode->getAttr(offsetAttr);
				int newOffset = strtol(buf, &end, 0);
				if (buf != end)
					offset = newOffset;
			}

			// Check whether tag has both a semantic and a source.
			if (inputNode->haveAttr("semantic") && inputNode->haveAttr("source"))
			{
				// Get the semantic of the input.
				string semantic = inputNode->getAttr("semantic");
				std::transform(semantic.begin(), semantic.end(), semantic.begin(), tolower);
				if (semantic == "vertex")
				{
					// Look for a vertex stream of this id.
					std::string vertexStreamID = inputNode->getAttr("source");
					if (vertexStreamID[0] == '#')
					{
						vertexStreamID = vertexStreamID.substr(1, std::string::npos);
						VertexStreamMap::iterator itVertexStreamEntry = vertexStreams.find(vertexStreamID);
						if (itVertexStreamEntry != vertexStreams.end())
						{
							ColladaVertexStream* vertexStream = (*itVertexStreamEntry).second;
							this->SetVertexStream(vertexStream);
							this->positionIndexOffset = offset;
						}
						else
						{
							ReportWarning(pListener, "\"input\" tag specifies non-existent vertex stream (%s) - skipping input.", vertexStreamID.c_str());
						}
					}
					else
					{
						ReportWarning(pListener, "\"input\" tag specifies external vertex stream (%s) - skipping input.", vertexStreamID.c_str());
					}
				}
				// Check whether the input refers to another data stream.
				else if (semanticMap.find(semantic) != semanticMap.end())
				{
					DataStreamType streamType = semanticMap[semantic];

					// Look for a data source with this id.
					std::string dataSourceID = inputNode->getAttr("source");
					if (dataSourceID[0] == '#')
					{
						dataSourceID = dataSourceID.substr(1, std::string::npos);
						DataSourceMap::iterator itDataSourceEntry = dataSources.find(dataSourceID);
						if (itDataSourceEntry != dataSources.end())
						{
							ColladaDataSource* dataSource = (*itDataSourceEntry).second;
							switch (streamType)
							{
							case DATA_STREAM_NORMAL:
								this->SetNormalSource(dataSource);
								this->normalIndexOffset = offset;
								break;
							case DATA_STREAM_COLOUR:
								this->SetColourSource(dataSource);
								this->colourIndexOffset = offset;
								break;
							case DATA_STREAM_TEXCOORD:
								this->SetTexCoordSource(dataSource);
								this->texCoordIndexOffset = offset;
								break;
							}
						}
						else
						{
							ReportWarning(pListener, "\"input\" tag (semantic \"%s\") specifies non-existent data source (%s) - skipping input.", inputNode->getAttr("semantic"), dataSourceID.c_str());
						}
					}
					else
					{
						ReportWarning(pListener, "\"input\" tag (semantic \"%s\") specifies external data source (%s) - skipping input.", inputNode->getAttr("semantic"), dataSourceID.c_str());
					}
				}
			}
			else
			{
				ReportWarning(pListener, "\"input\" tag requires both \"semantic\" and \"source\" attributes.");
			}
			++offset;
			if (offset > numIndices)
				numIndices = offset;
		}
	}

	// The offset at this point is the total number of indices per polygon.
	this->indexGroupSize = numIndices;
	if (this->indexGroupSize == 0)
	{
		ReportWarning(pListener, "\"%s\" node contains no input tags.", node->getTag());
		return false;
	}
	if (this->indexGroupSize > IndexGroup::MAX_INDICES)
	{
		ReportWarning(pListener, "\"%s\" node contains too many (%d) input tags.", node->getTag(), this->indexGroupSize);
		return false;
	}

	return true;
}

ColladaMaterial* BasePrimitiveStream::GetMaterial()
{
	return this->material;
}

bool BasePrimitiveStream::AddToMesh(CMesh* mesh, CGFMaterialMap& cgfMaterials, int subsetIndex, IColladaLoaderListener* pListener)
{
	// Set the position data.
	std::vector<Vec3> positions;
	if (this->vertexStream)
	{
		positions.resize(this->vertexStream->GetSize());
		if (!this->vertexStream->ReadVertexPositions(&positions[0], this->vertexStream->GetSize(), pListener))
		{
			ReportError(pListener, "Failed to load positions for mesh.");
			return false;
		}
	}
	else
	{
		ReportError(pListener, "Mesh has no vertex stream.");
		return false;
	}

	// Read in the normal positions.
	std::vector<Vec3> normals;
	bool readNormalsSuccessfully = false;
	if (this->normalSource)
	{
		normals.resize(this->normalSource->GetDataAccessor()->GetSize("x"));
		std::vector<float> xs;
		std::vector<float> ys;
		std::vector<float> zs;
		bool hasCorrectParams = true;
		hasCorrectParams = hasCorrectParams && this->normalSource->GetDataAccessor()->ReadAsFloat("x", xs, pListener);
		hasCorrectParams = hasCorrectParams && this->normalSource->GetDataAccessor()->ReadAsFloat("y", ys, pListener);
		hasCorrectParams = hasCorrectParams && this->normalSource->GetDataAccessor()->ReadAsFloat("z", zs, pListener);
		if (hasCorrectParams)
		{
			//for (int i = 0; i < int(this->mergedIndexToIndexGroupMap.size()); ++i)
			for (int i = 0; i < int(normals.size()); ++i)
			{
				if (i < int(xs.size()))
					normals[i].x = xs[i];
				else
					normals[i].x = 0.0f;
				if (i < int(ys.size()))
					normals[i].y = ys[i];
				else
					normals[i].y = 0.0f;
				if (i < int(zs.size()))
					normals[i].z = zs[i];
				else
					normals[i].z = 1.0f;
			}
			readNormalsSuccessfully = true;
		}
		else
		{
			ReportWarning(pListener, "Normal stream requires XYZ parameters.");
		}
	}

	if (!readNormalsSuccessfully)
	{
		// TODO: generate reasonable normals.
		ReportInfo(pListener, "Mesh has no normals - creating dummy normals - TODO: generate reasonable normals.");
		normals.resize(1);
		normals[0] = Vec3(0.0f, 0.0f, 1.0f);
	}

	// Create the position and normal stream.
	int oldVertexCount = mesh->GetVertexCount();
	mesh->SetVertexCount(oldVertexCount + int(this->mergedIndexToIndexGroupMap.size()));

	AABB bbox = mesh->m_bbox;
	//bbox.Reset();
	for (int i = 0; i < int(this->mergedIndexToIndexGroupMap.size()); ++i)
	{
		int normalIndex = (this->normalIndexOffset >= 0 ? this->mergedIndexToIndexGroupMap[i][this->normalIndexOffset] : 0);
		if (this->mergedIndexToIndexGroupMap[i][this->positionIndexOffset] >= int(positions.size()) ||
			 normalIndex >= int(normals.size()))
		{
			ReportWarning(pListener, "Index out of range.");
			return false;
		}
		mesh->m_pPositions[oldVertexCount + i] = positions[this->mergedIndexToIndexGroupMap[i][this->positionIndexOffset]];
		mesh->m_pNorms[oldVertexCount + i] = normals[normalIndex];
		bbox.Add(mesh->m_pPositions[oldVertexCount + i]);
	}
	mesh->m_bbox = bbox;

	// Read in the texture coordinates.
	bool readTextureCoordinatesSuccessfully = false;
	std::vector<SMeshTexCoord> uvs;
	if (this->texCoordSource)
	{
		// Read in the st values.
		std::vector<float> ss;
		std::vector<float> ts;
		bool hasSTParams = true;
		hasSTParams = hasSTParams && this->texCoordSource->GetDataAccessor()->ReadAsFloat("s", ss, pListener);
		hasSTParams = hasSTParams && this->texCoordSource->GetDataAccessor()->ReadAsFloat("t", ts, pListener);

		// If the st values are not present, then we cannot load texcoords.
		if (hasSTParams)
		{
			// Merge the attributes into a single vector of coordinates.
			uvs.resize(ss.size());
			for (int i = 0; i < int(uvs.size()); ++i)
			{
				if (i < int(ss.size()))
					uvs[i].s = ss[i];
				else
					uvs[i].s = 0.0f;
				if (i < int(ts.size()))
					uvs[i].t = ts[i];
				else
					uvs[i].t = 0.0f;
			}

			readTextureCoordinatesSuccessfully = true;
		}
		else
		{
			ReportWarning(pListener, "Texcoord stream requires at least \"S\" and \"T\" coordinates - ignoring uvs.");
		}
	}

	// Add the uvs to the mesh.
	if (readTextureCoordinatesSuccessfully)
	{
		// Resize the uvs stream.
		int oldUVCount = mesh->GetTexCoordsCount();
		mesh->SetTexCoordsCount(oldUVCount + int(this->mergedIndexToIndexGroupMap.size()));

		// It is possible that there are uvs for this stream, but not for previous ones.
		// We should make sure that the uvs for the missing section are set to an appropriate default.
		if (oldUVCount < oldVertexCount)
		{
			SMeshTexCoord uv = {0.0f, 0.0f};
			std::fill(mesh->m_pTexCoord + oldUVCount, mesh->m_pTexCoord + oldVertexCount, uv);
		}

		// Copy the uvs. There seems to be a -1 scale for texture coords (ie v=0 coordinates are along the bottom of the texture).
		for (int i = 0; i < int(this->mergedIndexToIndexGroupMap.size()); ++i)
		{
			if (this->mergedIndexToIndexGroupMap[i][this->texCoordIndexOffset] >= int(uvs.size()))
			{
				ReportWarning(pListener, "Index out of range.");
				return false;
			}
			mesh->m_pTexCoord[oldVertexCount + i] = uvs[this->mergedIndexToIndexGroupMap[i][this->texCoordIndexOffset]];
			mesh->m_pTexCoord[oldVertexCount + i].t = 1.0f - mesh->m_pTexCoord[oldVertexCount + i].t;
		}
	}

	// Read in the vertex colour positions.
	bool readVertexColoursSuccessfully = false;
	std::vector<SMeshColor> vertexColours;
	if (this->colourSource)
	{
		// Read in the rgb values.
		std::vector<float> rs;
		std::vector<float> gs;
		std::vector<float> bs;
		bool hasRGBParams = true;
		hasRGBParams = hasRGBParams && this->colourSource->GetDataAccessor()->ReadAsFloat("r", rs, pListener);
		hasRGBParams = hasRGBParams && this->colourSource->GetDataAccessor()->ReadAsFloat("g", gs, pListener);
		hasRGBParams = hasRGBParams && this->colourSource->GetDataAccessor()->ReadAsFloat("b", bs, pListener);

		// Read in the alpha values, if present.
		std::vector<float> as;
		bool hasAlphaParam = this->colourSource->GetDataAccessor()->ReadAsFloat("a", as, pListener);
		if (!hasAlphaParam)
			as.clear();

		// If rgb values are not present, then we cannot load vertex colours.
		if (hasRGBParams)
		{
			// Merge the attributes into a single vector of colours.
			vertexColours.resize(rs.size());
			for (int i = 0; i < int(vertexColours.size()); ++i)
			{
				if (i < int(rs.size()))
					vertexColours[i].r = unsigned(255 * rs[i]);
				else
					vertexColours[i].r = 0;
				if (i < int(gs.size()))
					vertexColours[i].g = unsigned(255 * gs[i]);
				else
					vertexColours[i].g = 0;
				if (i < int(bs.size()))
					vertexColours[i].b = unsigned(255 * bs[i]);
				else
					vertexColours[i].b = 0;
				if (i < int(as.size()))
					vertexColours[i].a = unsigned(255 * as[i]);
				else
					vertexColours[i].a = 255;
			}

			readVertexColoursSuccessfully = true;
		}
		else
		{
			ReportWarning(pListener, "Vertex colour stream requires at least RGB parameters - ignoring vertex colours.");
		}
	}

	// Add the vertex colours to the mesh.
	if (readVertexColoursSuccessfully)
	{
		// Resize the vertex colours stream.
		int oldVertexColourCount = mesh->m_streamSize[CMesh::COLORS_0];
		mesh->ReallocStream(CMesh::COLORS_0, oldVertexCount + int(this->mergedIndexToIndexGroupMap.size()));

		// It is possible that there are vertex colours for this stream, but none for previous ones.
		// We should make sure that the vertex colours for the missing section are set to an appropriate default.
		if (oldVertexColourCount < oldVertexCount)
		{
			SMeshColor colour = {0, 0, 0, 0xFF};
			std::fill(mesh->m_pColor0 + oldVertexColourCount, mesh->m_pColor0 + oldVertexCount, colour);
		}

		// Copy the vertex colours.
		for (int i = 0; i < int(this->mergedIndexToIndexGroupMap.size()); ++i)
		{
			if (this->mergedIndexToIndexGroupMap[i][this->colourIndexOffset] >= int(vertexColours.size()))
			{
				ReportWarning(pListener, "Index out of range.");
				return false;
			}
			mesh->m_pColor0[oldVertexCount + i] = vertexColours[this->mergedIndexToIndexGroupMap[i][this->colourIndexOffset]];
		}
	}

	return true;
}

int BasePrimitiveStream::AddIndexGroup(const std::vector<int>& indices)
{
	// Look for the indices in the map.
	std::map<IndexGroup, int>::iterator itIndexGroup = this->indexGroupToMergedIndexMap.find(IndexGroup(indices, 0, this->indexGroupSize));
	if (itIndexGroup == this->indexGroupToMergedIndexMap.end())
	{
		// Store the indices.
		int oldStoreSize = int(this->indexStore.size());
		for (int i = 0; i < this->indexGroupSize; ++i)
			this->indexStore.push_back(indices[i]);
		//int* storedIndices = &this->indexStore[oldStoreSize];

		// Add the index group to the maps.
		IndexGroup indexGroup(this->indexStore, oldStoreSize, this->indexGroupSize);
		int mergedIndex = int(this->mergedIndexToIndexGroupMap.size());
		this->mergedIndexToIndexGroupMap.push_back(indexGroup);
		itIndexGroup = this->indexGroupToMergedIndexMap.insert(std::make_pair(indexGroup, mergedIndex)).first;
	}

	// Return the index of the merged index.
	return (*itIndexGroup).second;
}

BasePrimitiveStream::IntStream::IntStream(const char* text)
:	position(text)
{
}

int BasePrimitiveStream::IntStream::Read()
{
	char* end;
	int value = int(std::strtol(this->position, &end, 0));
	if (end == this->position)
		return -1;
	else
		this->position = end;
	return value;
}

int BasePrimitiveStream::ReadIndexGroup(IntStream& stream)
{
	// Parse all the indices in the group.
	static std::vector<int> indices;
	indices.resize(IndexGroup::MAX_INDICES);
	for (int i = 0; i < this->indexGroupSize; ++i)
	{
		int index = stream.Read();
		if (index < 0)
			return -1;
		indices[i] = index;
	}

	// Merge the indices into one.
	int mergedIndex = this->AddIndexGroup(indices);

	return mergedIndex;
}

void BasePrimitiveStream::SetVertexStream(ColladaVertexStream* vertexStream)
{
	this->vertexStream = vertexStream;
}

void BasePrimitiveStream::SetNormalSource(ColladaDataSource* normalSource)
{
	this->normalSource = normalSource;
}

void BasePrimitiveStream::SetTexCoordSource(ColladaDataSource* texCoordSource)
{
	this->texCoordSource = texCoordSource;
}

void BasePrimitiveStream::SetColourSource(ColladaDataSource* colourSource)
{
	this->colourSource = colourSource;
}

bool BasePolygonPrimitiveStream::AddToMesh(CMesh* mesh, CGFMaterialMap& cgfMaterials, int subsetIndex, IColladaLoaderListener* pListener)
{
	// Remember how many vertices there used to be.
	int oldVertexCount = mesh->GetVertexCount();

	// Call the base class version.
	if (!BasePrimitiveStream::AddToMesh(mesh, cgfMaterials, subsetIndex, pListener))
		return false;

	// Compute the number of faces we will have.
	int numFaces = 0;
	for (int i = 0; i < int(this->polygonSizes.size()); ++i)
	{
		numFaces += this->polygonSizes[i] - 2;
	}

	// Allocate the new faces.
	int oldFaceCount = mesh->GetFacesCount();
	mesh->SetFacesCount(oldFaceCount + numFaces);

	// Loop through all the polygons, generating triangular faces.
	int indexIndex = 0;
	int numVertices = int(this->mergedIndexToIndexGroupMap.size());
	int faceIndex = 0;
	for (int polyIndex = 0; polyIndex < int(this->polygonSizes.size()); ++polyIndex)
	{
		int firstIndex = this->indices[indexIndex++];
		int lastIndex = this->indices[indexIndex++];

		for (int vertexIndex = 2; vertexIndex < this->polygonSizes[polyIndex]; ++vertexIndex)
		{
			int index = this->indices[indexIndex++];

			SMeshFace* pFace = &mesh->m_pFaces[oldFaceCount + faceIndex++];
			memset(pFace, 0, sizeof(SMeshFace));
			if (firstIndex < 0 || firstIndex >= numVertices
				|| lastIndex < 0 || lastIndex >= numVertices
				|| index < 0 || index >= numVertices)
			{
				ReportError(pListener, "Mesh sub-set contains out-of-range indices.");
				return false;
			}
			pFace->v[0] = pFace->t[0] = oldVertexCount + firstIndex;
			pFace->v[1] = pFace->t[1] = oldVertexCount + lastIndex;
			pFace->v[2] = pFace->t[2] = oldVertexCount + index;
			pFace->nSubset = subsetIndex;
			lastIndex = index;
		}
	}

	return true;
}

bool PolygonsPrimitiveStream::ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener)
{
	// Call the base class implementation. This will compute the value of this->indexGroupSize().
	if (!BasePrimitiveStream::ParseNode(node, materials, vertexStreams, dataSources, pListener))
		return false;

	// Find out how many polygons there are and reserve space for them.
	this->polygonSizes.clear();
	this->indices.clear();
	if (node->haveAttr("count"))
		this->polygonSizes.reserve(atol(node->getAttr("count")));

	// Read all the polygons.
	for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
	{
		XmlNodeRef primNode = node->getChild(childIndex);
		std::string primTag = primNode->getTag();
		std::transform(primTag.begin(), primTag.end(), primTag.begin(), tolower);
		if (primTag == "p")
		{
			int indexCount = 0;
			IntStream stream(primNode->getContent());
			for (;;)
			{
				int mergedVertexIndex = this->ReadIndexGroup(stream);
				if (mergedVertexIndex == -1)
					break;
				this->indices.push_back(mergedVertexIndex);
				++indexCount;
			}

			this->polygonSizes.push_back(indexCount);
		}
	}

	return true;
}

bool TrifansPrimitiveStream::ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener)
{
	// Call the base class implementation. This will compute the value of this->indexGroupSize().
	if (!BasePrimitiveStream::ParseNode(node, materials, vertexStreams, dataSources, pListener))
		return false;

	// Find out how many polygons there are and reserve space for them.
	this->polygonSizes.clear();
	this->indices.clear();
	if (node->haveAttr("count"))
		this->polygonSizes.reserve(atol(node->getAttr("count")));

	// Read all the trifans.
	for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
	{
		XmlNodeRef primNode = node->getChild(childIndex);
		std::string primTag = primNode->getTag();
		std::transform(primTag.begin(), primTag.end(), primTag.begin(), tolower);
		if (primTag == "p")
		{
			int indexCount = 0;
			IntStream stream(primNode->getContent());
			for (;;)
			{
				int mergedVertexIndex = this->ReadIndexGroup(stream);
				if (mergedVertexIndex == -1)
					break;
				this->indices.push_back(mergedVertexIndex);
				++indexCount;
			}

			this->polygonSizes.push_back(indexCount);
		}
	}

	return true;
}

bool PolyStripPrimitiveStream::ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener)
{
	// Call the base class implementation. This will compute the value of this->indexGroupSize().
	if (!BasePrimitiveStream::ParseNode(node, materials, vertexStreams, dataSources, pListener))
		return false;

	// Find out how many polygons there are and reserve space for them.
	this->polygonSizes.clear();
	if (node->haveAttr("count"))
		this->polygonSizes.reserve(atol(node->getAttr("count")));

	// Check whether there is a vcount element.
	XmlNodeRef vcountNode = node->findChild("vcount");
	if (vcountNode)
	{
		// Read the polygon sizes from the vcount node.
		IntStream stream(vcountNode->getContent());
		int polygonSize;
		int numIndices = 0;
		do 
		{
			polygonSize = stream.Read();
			if (polygonSize >= 0)
			{
				this->polygonSizes.push_back(polygonSize);
				numIndices += polygonSize;
			}
		}
		while(polygonSize >= 0);

		// Reserve memory for the indices.
		this->indices.clear();
		this->indices.reserve(numIndices);
	
		// Read all the polygons.
		for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
		{
			XmlNodeRef primNode = node->getChild(childIndex);
			std::string primTag = primNode->getTag();
			std::transform(primTag.begin(), primTag.end(), primTag.begin(), tolower);
			if (primTag == "p")
			{
				IntStream stream(primNode->getContent());
				for (;;)
				{
					int mergedVertexIndex = this->ReadIndexGroup(stream);
					if (mergedVertexIndex == -1)
						break;
					this->indices.push_back(mergedVertexIndex);
				}
			}
		}
	}
	else
	{
		ReportWarning(pListener, "\"%s\" node has no \"vcount\" element - skipping geometry.");
		return false;
	}

	return true;
}

bool TrianglesPrimitiveStream::ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener)
{
	// Call the base class implementation. This will compute the value of this->indexGroupSize().
	if (!BasePrimitiveStream::ParseNode(node, materials, vertexStreams, dataSources, pListener))
		return false;

	// Find out how many triangles there are and reserve space for them.
	this->indices.clear();
	if (node->haveAttr("count"))
		this->indices.reserve(atol(node->getAttr("count")) * 3);

	// Read all the triangles.
	for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
	{
		XmlNodeRef primNode = node->getChild(childIndex);
		std::string primTag = primNode->getTag();
		std::transform(primTag.begin(), primTag.end(), primTag.begin(), tolower);
		if (primTag == "p")
		{
			int indexCount = 0;
			IntStream stream(primNode->getContent());
			for (;;)
			{
				int mergedVertexIndex = this->ReadIndexGroup(stream);
				if (mergedVertexIndex == -1)
					break;
				this->indices.push_back(mergedVertexIndex);
				++indexCount;
			}
		}
	}

	return true;
}

bool TrianglesPrimitiveStream::AddToMesh(CMesh* mesh, CGFMaterialMap& cgfMaterials, int subsetIndex, IColladaLoaderListener* pListener)
{
	// Remember how many vertices there used to be.
	int oldVertexCount = mesh->GetVertexCount();

	// Call the base class version.
	if (!BasePrimitiveStream::AddToMesh(mesh, cgfMaterials, subsetIndex, pListener))
		return false;

	// Compute the number of faces we will have.
	int numFaces = int(this->indices.size()) / 3;

	// Allocate the new faces.
	int oldFaceCount = mesh->GetFacesCount();
	mesh->SetFacesCount(oldFaceCount + numFaces);

	// Loop through all the polygons, generating triangular faces.
	int indexIndex = 0;
	int numVertices = int(this->mergedIndexToIndexGroupMap.size());
	int faceIndex = 0;
	int numDegenerateFaces = 0;
	for (int triangleIndex = 0; triangleIndex < numFaces; ++triangleIndex)
	{
		int firstIndex = this->indices[indexIndex++];
		int secondIndex = this->indices[indexIndex++];
		int thirdIndex = this->indices[indexIndex++];

		if (firstIndex < 0 || firstIndex >= numVertices
			|| secondIndex < 0 || secondIndex >= numVertices
			|| thirdIndex < 0 || thirdIndex >= numVertices)
		{
			ReportError(pListener, "Mesh sub-set contains out-of-range indices.");
			return false;
		}

		// Check for degenerate faces.
		if (firstIndex == secondIndex || secondIndex == thirdIndex || firstIndex == thirdIndex)
		{
			++numDegenerateFaces;
		}
		else
		{
			// Add the face to the list.
			SMeshFace* pFace = &mesh->m_pFaces[oldFaceCount + faceIndex++];
			memset(pFace, 0, sizeof(SMeshFace));
			pFace->v[0] = oldVertexCount + firstIndex;
			pFace->v[1] = oldVertexCount + secondIndex;
			pFace->v[2] = oldVertexCount + thirdIndex;
			pFace->nSubset = subsetIndex;
		}
	}

	// Check whether there were any degenerate faces - if so, shrink the face array to remove the null faces at the end.
	if (numDegenerateFaces > 0)
	{
		ReportWarning(pListener, "Mesh subset contains %d degenerate faces - deleting faces.", numDegenerateFaces);
		mesh->SetFacesCount(mesh->GetFacesCount() - numDegenerateFaces);
	}

	return true;
}

bool TriStripsPrimitiveStream::ParseNode(XmlNodeRef& node, MaterialMap& materials, VertexStreamMap& vertexStreams, DataSourceMap& dataSources, IColladaLoaderListener* pListener)
{
	// Call the base class implementation. This will compute the value of this->indexGroupSize().
	if (!BasePrimitiveStream::ParseNode(node, materials, vertexStreams, dataSources, pListener))
		return false;

	// Find out how many polygons there are and reserve space for them.
	this->stripSizes.clear();
	this->indices.clear();
	if (node->haveAttr("count"))
		this->stripSizes.reserve(atol(node->getAttr("count")));

	// Read all the strips.
	for (int childIndex = 0; childIndex < node->getChildCount(); ++childIndex)
	{
		XmlNodeRef primNode = node->getChild(childIndex);
		std::string primTag = primNode->getTag();
		std::transform(primTag.begin(), primTag.end(), primTag.begin(), tolower);
		if (primTag == "p")
		{
			int indexCount = 0;
			IntStream stream(primNode->getContent());
			for (;;)
			{
				int mergedVertexIndex = this->ReadIndexGroup(stream);
				if (mergedVertexIndex == -1)
					break;
				this->indices.push_back(mergedVertexIndex);
				++indexCount;
			}

			this->stripSizes.push_back(indexCount);
		}
	}

	return true;
}

bool TriStripsPrimitiveStream::AddToMesh(CMesh* mesh, CGFMaterialMap& cgfMaterials, int subsetIndex, IColladaLoaderListener* pListener)
{
	// Remember how many vertices there used to be.
	int oldVertexCount = mesh->GetVertexCount();

	// Call the base class version.
	if (!BasePrimitiveStream::AddToMesh(mesh, cgfMaterials, subsetIndex, pListener))
		return false;

	// Compute the number of faces we will have.
	int numFaces = 0;
	for (int i = 0; i < int(this->stripSizes.size()); ++i)
	{
		numFaces += this->stripSizes[i] - 2;
	}

	// Allocate the new faces.
	int oldFaceCount = mesh->GetFacesCount();
	mesh->SetFacesCount(oldFaceCount + numFaces);

	// Loop through all the polygons, generating triangular faces.
	int indexIndex = 0;
	int numVertices = int(this->mergedIndexToIndexGroupMap.size());
	int faceIndex = 0;
	int numDegenerateFaces = 0;
	for (int polyIndex = 0; polyIndex < int(this->stripSizes.size()); ++polyIndex)
	{
		int indices[2];
		indices[0] = this->indices[indexIndex++];
		indices[1] = this->indices[indexIndex++];

		for (int vertexIndex = 2; vertexIndex < this->stripSizes[polyIndex]; ++vertexIndex)
		{
			int index = this->indices[indexIndex];

			// Check that the indices are within range.
			if (indices[0] < 0 || indices[0] >= numVertices
				|| indices[1] < 0 || indices[1] >= numVertices
				|| index < 0 || index >= numVertices)
			{
				ReportError(pListener, "Mesh sub-set contains out-of-range indices.");
				return false;
			}

			// Check for degenerate faces.
			if (indices[0] == indices[1] || indices[1] == index || indices[0] == index)
			{
				++numDegenerateFaces;
			}
			else
			{
				// Add the face to the list.
				SMeshFace* pFace = &mesh->m_pFaces[oldFaceCount + faceIndex++];
				memset(pFace, 0, sizeof(SMeshFace));
				pFace->v[0] = oldVertexCount + indices[0];
				pFace->v[1] = oldVertexCount + indices[1];
				pFace->v[2] = oldVertexCount + index;
				pFace->nSubset = subsetIndex;
			}

			// Overwrite the index with the same parity - this trick ensures that winding order of
			// triangles is constant.
			indices[indexIndex & 1] = index;
			++indexIndex;
		}
	}

	// Check whether there were any degenerate faces - if so, shrink the face array to remove the null faces at the end.
	if (numDegenerateFaces > 0)
	{
		ReportWarning(pListener, "Mesh subset contains %d degenerate faces - deleting faces.", numDegenerateFaces);
		mesh->SetFacesCount(mesh->GetFacesCount() - numDegenerateFaces);
	}

	return true;
}

IndexGroup::IndexGroup(const std::vector<int>& indices, int indexOffset, int indexCount)
:	indices(&indices),
	indexOffset(indexOffset),
	indexCount(indexCount)
{
}

bool IndexGroup::operator<(const IndexGroup& other) const
{
	for (int i = 0; i < this->indexCount; ++i)
	{
		if ((*this->indices)[this->indexOffset + i] < (*other.indices)[other.indexOffset + i])
			return true;
		if ((*this->indices)[this->indexOffset + i] > (*other.indices)[other.indexOffset + i])
			return false;
	}
	return false;
}

ColladaNode::ColladaNode(ColladaGeometry* pGeometry)
:	pGeometry(pGeometry)
{
	this->transform.SetIdentity();
}

Matrix34& ColladaNode::GetTransform()
{
	return this->transform;
}

void ColladaNode::SetName(const std::string& name)
{
	this->name = name;
}

void DecomposeTransform(Vec3& translation, CryQuat& rotation, Vec3& scale, const Matrix34& transform)
{
	translation = transform.GetTranslation();
	Matrix33 orientation(transform);
	scale = Vec3(orientation.m00, orientation.m11, orientation.m22);
	orientation.OrthonormalizeFast();
	rotation = CryQuat(orientation);
}

CNodeCGF* ColladaNode::CreateNodeCGF(CMaterialCGF* rootMaterial, CGFMaterialMap& cgfMaterials, ColladaAssetMetaData* metaData, IColladaLoaderListener* pListener)
{
	CNodeCGF* node = 0;
	CMesh* mesh = this->pGeometry->CreateMesh(cgfMaterials, pListener);
	if (mesh)
	{
		node = new CNodeCGF();
		node->pos_cont_id = 0;
		node->rot_cont_id = 0;
		node->scl_cont_id = 0;
		node->pMesh = mesh;
		node->type = CNodeCGF::NODE_MESH;
		node->name = this->name.c_str();
		node->pMaterial = rootMaterial;

		// We need to adjust for the differences in coordinate systems. The meta data of the file
		// specifies the up-axis of the scene, so we need to apply a rotation that will turn that up-axis
		// to point down the z-axis (the cryengine axis).
		Matrix34 transform = this->transform;
		if (metaData)
		{
			// Create the rotation.
			CryQuat coordinateRotation = CryQuat::CreateRotationV0V1(metaData->GetUpAxis(), Vec3(0.0f, 0.0f, 1.0f));

			// We want to rotate the mesh around the mesh origin, and then rotate the translation of the node to
			// move the centrepoint to the new position. We don't want to change the orientation of the node in
			// worldspace.
			transform.SetTranslation(coordinateRotation * transform.GetTranslation());
			for (int i = 0; i < mesh->GetVertexCount(); ++i)
			{
				mesh->m_pPositions[i] = coordinateRotation * mesh->m_pPositions[i];
				mesh->m_pNorms[i] = coordinateRotation * mesh->m_pNorms[i];
			}
		}

		node->localTM = transform;
		node->worldTM = transform;
		node->bIdentityMatrix = false;
		DecomposeTransform(node->pos, node->rot, node->scl, node->worldTM);
	}

	return node;
}

ColladaScene::ColladaScene()
{
}

ColladaScene::~ColladaScene()
{
	//for (std::vector<ColladaNode*>::iterator itNode = this->nodes.begin(); itNode != this->nodes.end(); ++itNode)
	//	delete *itNode;
}

void ColladaScene::AddNode(ColladaNode* pNode)
{
	this->nodes.push_back(pNode);
}

CContentCGF* ColladaScene::CreateContentCGF(const std::string& filename, MaterialMap& materials, ColladaAssetMetaData* metaData, IColladaLoaderListener* pListener)
{
	// Check whether there are any nodes in the scene.
	CContentCGF* cgf = 0;

	if (!this->nodes.empty())
	{
		// Create the content cgf object.
		cgf = new CContentCGF(filename.c_str());
	}
	else
	{
		ReportError(pListener, "Unable to load any nodes from the scene.");
	}

	// Create the root multi-material for the scene.
	int failedNodeCount = 0;
	if (cgf)
	{
		CGFMaterialMap cgfMaterials;
		CMaterialCGF* material = this->CreateMaterial(cgfMaterials, materials, pListener);

		// Add the material to the scene.
		cgf->AddMaterial(material);
		cgf->SetCommonMaterial(material);

		// Add the sub-materials to the scene.
		for (int subMaterialIndex = 0; subMaterialIndex < int(material->subMaterials.size()); ++subMaterialIndex)
			cgf->AddMaterial(material);

		// Add nodes to the cgf.
		for (std::vector<ColladaNode*>::iterator itNode = this->nodes.begin(); itNode != this->nodes.end(); ++itNode)
		{
			CNodeCGF* node = (*itNode)->CreateNodeCGF(material, cgfMaterials, metaData, pListener);
			if (node != 0)
			{
				cgf->AddNode(node);
			}
			else
			{
				ReportError(pListener, "Failed to load node \"%s\".", (*itNode)->GetName().c_str());
				++failedNodeCount;
			}
		}
	}

	if (failedNodeCount > 0)
	{
		delete cgf;
		cgf = 0;
	}

	return cgf;
}

CMaterialCGF* ColladaScene::CreateMaterial(CGFMaterialMap& cgfMaterials, MaterialMap& materials, IColladaLoaderListener* pListener)
{
	// We need to select the name of the parent material (ie the name of the *.MTL file). The convention is to take this name from
	// the name of the material library that is used. The material library name is given as the first part of the material name when
	// exported from XSI (ie a name is of the form <library>.<material>). It might be possible to have recursive libraries inside
	// the main one - for now we will use the name of the top-level library. It is possible for one object to contain materials from
	// different libraries - this is an error. We should also check whether the library name is DefaultLib - this is the name given
	// to the default library by XSI, and we should warn the user that he probably hasn't set the material library up properly.
	std::set<std::string> libraryNames;
	for (MaterialMap::iterator itMaterial = materials.begin(); itMaterial != materials.end(); ++itMaterial)
	{
		// Get the material.
		ColladaMaterial* colladaMaterial = (*itMaterial).second;

		// Get the name of the material.
		std::string name = colladaMaterial->GetName();

		// Examine the name to see whether it has a library name in the XSI format (ie <library>.<material>).
		size_t periodPos = name.find('.');
		if (periodPos != std::string::npos)
		{
			// Get the library name.
			std::string library = name.substr(0, periodPos);

			// Add the library name to the set of names.
			libraryNames.insert(library);
		}
	}

	// If there are multiple libs, try to remove DefaultLib from the list.
	if (libraryNames.size() > 1 && libraryNames.find("DefaultLib") != libraryNames.end())
			libraryNames.erase("DefaultLib");

	// Now that we have a list of the material names, we have three possibilities:
	// 1) There are no library names set - flag a warning, but use a default library name.
	// 2) There is one library name set - if the library is DefaultLib, report a warning.
	// 3) There are multiple names set - flag a warning, but just choose one.
	std::string parentMaterialName;
	switch (libraryNames.size())
	{
	case 0:
		{
			ReportWarning(pListener, "Cannot determine *.MTL filename - if using XSI, check that materials are in a library whose name is that of the MTL file.");
			parentMaterialName = "parent";
		}
		break;

	case 1:
		{
			std::string library = *libraryNames.begin();
			if (library == "DefaultLib")
				ReportWarning(pListener, "*.MTL filename is DefaultLib - if using XSI, make sure materials are in a library whose name is that of the MTL file.");
			parentMaterialName = library;
		}
		break;

	default:
		{
			ReportWarning(pListener, "Multiple possible *.MTL filenames found - if using XSI, make sure ALL materials are in a library whose name is that of the MTL file.");
			std::string library = *libraryNames.begin();
			parentMaterialName = library;
		}
		break;
	}

	// Create a multi-material to hold the sub-materials.
	CMaterialCGF* material = new CMaterialCGF();
	material->name = parentMaterialName.c_str();
	material->nPhysicalizeType = PHYS_GEOM_TYPE_NONE;

	// There is a naming convention for materials that specifies the material ID that the material should
	// receive in the engine. Names should be of the form <id>[_]<name>. To cater for materials that are
	// not of this form, we shall perform two passes over the material list. In the first pass, any materials
	// that fit the convention shall be assigned the materials they specify. In the second pass, any other
	// materials shall be given available ids.

	class MaterialEntry
	{
	public:
		MaterialEntry(int id, std::string name, ColladaMaterial* colladaMaterial, const CCustomMaterialSettings& settings) : id(id), name(name), colladaMaterial(colladaMaterial), settings(settings) {}

		int id;
		std::string name;
		ColladaMaterial* colladaMaterial;
		CCustomMaterialSettings settings;
	};

	typedef std::map<ColladaMaterial*, MaterialEntry> MaterialCreationMap;
	MaterialCreationMap materialCreationMap;
	std::set<int> assignedIDs;

	// Pass 1.
	for (MaterialMap::iterator itMaterial = materials.begin(); itMaterial != materials.end(); ++itMaterial)
	{
		// Get the material.
		ColladaMaterial* colladaMaterial = (*itMaterial).second;

		// Check whether the name fits the naming convention.
		int id;
		std::string name;
		bool addToList = true;
		CCustomMaterialSettings settings;
		switch (SplitMaterialName(colladaMaterial->GetName(), id, name, settings))
		{
		case MATERIAL_NAME_INVALID:
				ReportWarning(pListener, "Material \"%s\" does not fit naming convention (<id>_<name>) - auto-assigning ID.", colladaMaterial->GetName().c_str());
				addToList = false;
			break;

		case MATERIAL_NAME_VALID:
			// Check whether the material ID is invalid.
			if (id > MAX_SUB_MATERIALS || id < 1)
			{
				ReportWarning(pListener, "Material \"%s\" specifies invalid id (%d).", colladaMaterial->GetName().c_str(), id);
				addToList = false;
			}
			break;

		case MATERIAL_NAME_NUMBERONLY:
			ReportWarning(pListener, "Material \"%s\" specifies an id but no name (naming convention is \"<id>_<name>\")", colladaMaterial->GetName().c_str());
			break;

		default:
			assert(0);
		}

		if (addToList)
		{
			// Check whether the requested ID has already been assigned.
			if (assignedIDs.find(id) == assignedIDs.end())
			{
				materialCreationMap.insert(std::make_pair(colladaMaterial, MaterialEntry(id, name, colladaMaterial, settings)));
				assignedIDs.insert(id);
			}
			else
			{
				ReportWarning(pListener, "Material \"%s\" specifies duplicate ID - auto-assigning ID.", colladaMaterial->GetName().c_str());
			}
		}
	}

	// Pass 2. Start assigning ids starting from 1, since material ids in the collada file are 1 based.
	int nextID = 1;
	for (MaterialMap::iterator itMaterial = materials.begin(); itMaterial != materials.end(); ++itMaterial)
	{
		// Get the material.
		ColladaMaterial* colladaMaterial = (*itMaterial).second;

		// Check whether the material has already been assigned an ID.
		if (materialCreationMap.find(colladaMaterial) == materialCreationMap.end())
		{
			// The material has not been assigned an ID - do so now. Choose the ID to assign.
			while (assignedIDs.find(nextID) != assignedIDs.end())
				++nextID;

			// Extract the material name.
			int id;
			std::string name;
			CCustomMaterialSettings settings;
			SplitMaterialName(colladaMaterial->GetName(), id, name, settings);

			// Assign the ID.
			id = nextID++;
			ReportInfo(pListener, "Assigning ID %d to material \"%s\".", id, colladaMaterial->GetName().c_str());
			materialCreationMap.insert(std::make_pair(colladaMaterial, MaterialEntry(id, name, colladaMaterial, settings)));
			assignedIDs.insert(id);
		}
	}

	// Subtract 1 from all IDs to convert from 1 based (UI) to 0 based (engine).
	for (MaterialCreationMap::iterator itMaterialCreationEntry = materialCreationMap.begin(); itMaterialCreationEntry != materialCreationMap.end(); ++itMaterialCreationEntry)
	{
		MaterialEntry& entry = (*itMaterialCreationEntry).second;
		entry.id -= 1;
	}

	// Now that all materials have been assigned IDs, create sub-materials for each of them and add them to the parent material.
	int maxMaterialID = 0;
	for (MaterialCreationMap::iterator itMaterialCreationEntry = materialCreationMap.begin(); itMaterialCreationEntry != materialCreationMap.end(); ++itMaterialCreationEntry)
	{
		MaterialEntry& entry = (*itMaterialCreationEntry).second;

		if (maxMaterialID < entry.id)
			maxMaterialID = entry.id;
	}
	material->subMaterials.resize(maxMaterialID + 1);
	std::fill(material->subMaterials.begin(), material->subMaterials.end(), (CMaterialCGF*)0);
	for (MaterialCreationMap::iterator itMaterialCreationEntry = materialCreationMap.begin(); itMaterialCreationEntry != materialCreationMap.end(); ++itMaterialCreationEntry)
	{
		MaterialEntry& entry = (*itMaterialCreationEntry).second;

		// Create the material.
		CMaterialCGF* subMaterial = new CMaterialCGF;
		subMaterial->name = entry.name.c_str();
		subMaterial->nPhysicalizeType = PHYS_GEOM_TYPE_NONE;

		// Apply the custom material settings.
		const CCustomMaterialSettings& settings = entry.settings;

		// Read the physicalisation data.
		EPhysicsGeomType physicaliseType = PHYS_GEOM_TYPE_NONE;
		switch (settings.physicalizeType)
		{
		case 0: physicaliseType = PHYS_GEOM_TYPE_NONE; break;
		case 1: physicaliseType = PHYS_GEOM_TYPE_DEFAULT; break;
		case 2: physicaliseType = PHYS_GEOM_TYPE_NO_COLLIDE; break;
		case 3: physicaliseType = PHYS_GEOM_TYPE_OBSTRUCT; break;
		default:
			ReportWarning(pListener, "Invalid physicalization type \"%d\" in material name \"%s\" - must be 0-3.", settings.physicalizeType, entry.colladaMaterial->GetName().c_str());
		}
		subMaterial->nPhysicalizeType = physicaliseType;

		// Read the SH settings.
		if (settings.sh)
			subMaterial->nFlags |= MTL_NAME_CHUNK_DESC::FLAG_SH_COEFFS;
		bool sh2Sides = false;
		if (settings.shSides < 1 || settings.shSides > 2)
			ReportWarning(pListener, "Invalid number of sh sides (%d) in material name \"%s\" - must be 0/1.", settings.shSides, entry.colladaMaterial->GetName().c_str());
		else
			sh2Sides = settings.shSides == 2;
		if (sh2Sides)
			subMaterial->nFlags |= MTL_NAME_CHUNK_DESC::FLAG_SH_2SIDED;
		if (settings.shAmbient)
			subMaterial->nFlags |= MTL_NAME_CHUNK_DESC::FLAG_SH_AMBIENT;
		float shOpacity = settings.shOpacity;
		if (shOpacity < 0.0f || shOpacity > 1.0f)
		{
			ReportWarning(pListener, "SH Opacity invalid (%f) in material name \"%s\" - must be 0-100.", settings.shOpacity, entry.colladaMaterial->GetName().c_str());
			if (shOpacity < 0.0f)
				shOpacity = 0.0f;
			if (shOpacity > 1.0f)
				shOpacity = 1.0f;
		}
		subMaterial->shOpacity = settings.shOpacity;

		// Add the sub material to its parent.
		material->subMaterials[entry.id] = subMaterial;

		// Register the ID of this material, so we can look it up later.
		cgfMaterials.insert(std::make_pair(entry.colladaMaterial, CGFMaterialMapEntry(entry.id, subMaterial)));
	}

	return material;
}

ColladaMaterial::ColladaMaterial(const std::string& name)
:	name(name)
{
}

const std::string& ColladaMaterial::GetName()
{
	return this->name;
}
