//////////////////////////////////////////////////////////////////////
//
//  Crytek (C) 2001 
//
//  File: CryColor3.h
//  Description: 3D Color template.
//
//  History:
//  - August 12, 2001: Created by Alberto Demichelis
//
//////////////////////////////////////////////////////////////////////

#ifndef CRYTEK_CRYCOLOR3_H
#define CRYTEK_CRYCOLOR3_H

//////////////////////////////////////////////////////////////////////////////////////////////
namespace cry
{

template <class T> struct color3;
typedef color3<float>			color3f; // [0.0, 1.0]
typedef color3<double>			color3d; // [0.0, 1.0]
typedef color3<unsigned char>	color3b; // [0, 255]
typedef color3<unsigned short>	color3w; // [0, 65535]		

//////////////////////////////////////////////////////////////////////////////////////////////
// RGBA Color structure.
template <class T> struct color3
{
public:
	union {
		struct {
			T	r, g, b;
		};
		T v[3];
	};

public:
	inline color3();
	inline color3(const T *p_elts);
	inline color3(const color3 & v);
	inline color3(T _x, T _y = 0, T _z = 0);
	
public:
	inline void set(T _x, T _y = 0, T _z = 0);

	inline color3 operator + () const;
	inline color3 operator - () const;
	
	inline color3 & operator += (const color3 & v);
	inline color3 & operator -= (const color3 & v);
	inline color3 & operator *= (const color3 & v);
	inline color3 & operator /= (const color3 & v);
	inline color3 & operator *= (T s);
	inline color3 & operator /= (T s);
	
	inline color3 operator + (const color3 & v) const;
	inline color3 operator - (const color3 & v) const;
	inline color3 operator * (const color3 & v) const;
	inline color3 operator / (const color3 & v) const;
	inline color3 operator * (T s) const;
	inline color3 operator / (T s) const;
	
	inline bool operator == (const color3 & v) const;
	inline bool operator != (const color3 & v) const;

	inline unsigned char pack_rgb332();
	inline unsigned short pack_argb4444();
	inline unsigned short pack_rgb555();
	inline unsigned short pack_rgb565();
	inline unsigned long pack_rgb888();
	inline unsigned long pack_argb8888();

	inline unsigned long pack8() { return pack_rgb332(); }
	inline unsigned long pack12() { return pack_argb4444(); }
	inline unsigned long pack15() { return pack_rgb555(); }
	inline unsigned long pack16() { return pack_rgb565(); }
	inline unsigned long pack24() { return pack_rgb888(); }
	inline unsigned long pack32() { return pack_argb8888(); }

	inline void clamp(T bottom = 0.0f, T top = 1.0f);

	inline void maximum(const color3<T> &ca, const color3<T> &cb);
	inline void minimum(const color3<T> &ca, const color3<T> &cb);
	inline void abs();
	
	inline void adjust_contrast(T c);
	inline void adjust_saturation(T s);

	inline void lerp(const color3<T> &ca, const color3<T> &cb, T s);
	inline void negative(const color3<T> &c);
	inline void grey(const color3<T> &c);
	inline void black_white(const color3<T> &c, T s);

public:
	inline friend color3 operator * (T s, const color3 & v);
};

//////////////////////////////////////////////////////////////////////////////////////////////
// functions implementation

///////////////////////////////////////////////
template <class T>
inline color3<T>::color3<T>()
{
}

///////////////////////////////////////////////
template <class T>
inline color3<T>::color3<T>(const T *p_elts)
{
	r = p_elts[0]; g = p_elts[1]; b = p_elts[2];
}

///////////////////////////////////////////////
template <class T>
inline color3<T>::color3<T>(const color3<T> & v)
{
	r = v.r; g = v.g; b = v.b;
}

///////////////////////////////////////////////
template <class T>
inline color3<T>::color3<T>(T _x, T _y, T _z)
{
	r = _x; g = _y; b = _z;
}

///////////////////////////////////////////////
template <class T>
inline void color3<T>::set(T _x, T _y, T _z)
{
	r = _x; g = _y; b = _z;
}

///////////////////////////////////////////////
template <class T>
inline color3<T> color3<T>::operator + () const
{
	return *this;
}

///////////////////////////////////////////////
template <class T>
inline color3<T> color3<T>::operator - () const
{
	return color3<T>(-r, -g, -b);
}

///////////////////////////////////////////////
template <class T>
inline color3<T> & color3<T>::operator += (const color3<T> & v)
{
	r += v.r; g += v.g; b += v.b;
	return *this;
}

///////////////////////////////////////////////
template <class T>
inline color3<T> & color3<T>::operator -= (const color3<T> & v)
{
	r -= v.r; g -= v.g; b -= v.b;
	return *this;
}

///////////////////////////////////////////////
template <class T>
inline color3<T> & color3<T>::operator *= (const color3<T> & v)
{
	r *= v.r; g *= v.g; b *= v.b;
	return *this;
}

///////////////////////////////////////////////
template <class T>
inline color3<T> & color3<T>::operator /= (const color3<T> & v)
{
	r /= v.r; g /= v.g; b /= v.b;
	return *this;
}

///////////////////////////////////////////////
template <class T>
inline color3<T> & color3<T>::operator *= (T s)
{
	r *= s; g *= s; b *= s;
	return *this;
}

///////////////////////////////////////////////
template <class T>
inline color3<T> & color3<T>::operator /= (T s)
{
	s = 1.0f / s;
	r *= s; g *= s; b *= s;
	return *this;
}

///////////////////////////////////////////////
template <class T>
inline color3<T> color3<T>::operator + (const color3<T> & v) const
{
	return color3<T>(r + v.r, g + v.g, b + v.b, a + v.a);
}

///////////////////////////////////////////////
template <class T>
inline color3<T> color3<T>::operator - (const color3<T> & v) const
{
	return color3<T>(r - v.r, g - v.g, b - v.b);
}

///////////////////////////////////////////////
template <class T>
inline color3<T> color3<T>::operator * (T s) const
{
	return color3<T>(r * s, g * s, b * s);
}

///////////////////////////////////////////////
template <class T>
inline color3<T> color3<T>::operator / (T s) const
{
	s = 1.0f / s;
	return color3<T>(r * s, g * s, b * s);
}

///////////////////////////////////////////////
template <class T>
inline bool color3<T>::operator == (const color3<T> & v) const
{
	return (r == v.r) && (g == v.g) && (b == v.b);
}

///////////////////////////////////////////////
template <class T>
inline bool color3<T>::operator != (const color3<T> & v) const
{
	return (r != v.r) || (g != v.g) || (b != v.b);
}

///////////////////////////////////////////////
template <class T>
inline color3<T> operator * (T s, const color3<T> & v)
{
	return color3<T>(v.r * s, v.g * s, v.b * s);
}

///////////////////////////////////////////////
template <class T>
inline unsigned char color3<T>::pack_rgb332()
{
	unsigned char cr;
	unsigned char cg;
	unsigned char cb;
	if(sizeof(r) == 1) // char and unsigned char
	{
		cr = r;
		cg = g;
		cb = b;
	}
	else if(sizeof(r) == 2) // short and unsigned short
	{
		cr = unsigned short(r)>>8;
		cg = unsigned short(g)>>8;
		cb = unsigned short(b)>>8;
	}
	else // float or double
	{
		cr = unsigned char(r * 255.0f);
		cg = unsigned char(g * 255.0f);
		cb = unsigned char(b * 255.0f);
	}
	return ((cr >> 5) << 5) | ((cg >> 5) << 2) | (cb >> 5);
}

///////////////////////////////////////////////
template <class T>
inline unsigned short color3<T>::pack_argb4444()
{
	unsigned char cr;
	unsigned char cg;
	unsigned char cb;
	if(sizeof(r) == 1) // char and unsigned char
	{
		cr = r;
		cg = g;
		cb = b;
	}
	else if(sizeof(r) == 2) // short and unsigned short
	{
		cr = unsigned short(r)>>8;
		cg = unsigned short(g)>>8;
		cb = unsigned short(b)>>8;
	}
	else // float or double
	{
		cr = unsigned char(r * 255.0f);
		cg = unsigned char(g * 255.0f);
		cb = unsigned char(b * 255.0f);
	}
	return ((cr >> 4) << 8) | ((cg >> 4) << 4) | (cb >> 4);
}

///////////////////////////////////////////////
template <class T>
inline unsigned short color3<T>::pack_rgb555()
{
	unsigned char cr;
	unsigned char cg;
	unsigned char cb;
	if(sizeof(r) == 1) // char and unsigned char
	{
		cr = r;
		cg = g;
		cb = b;
	}
	else if(sizeof(r) == 2) // short and unsigned short
	{
		cr = unsigned short(r)>>8;
		cg = unsigned short(g)>>8;
		cb = unsigned short(b)>>8;
	}
	else // float or double
	{
		cr = unsigned char(r * 255.0f);
		cg = unsigned char(g * 255.0f);
		cb = unsigned char(b * 255.0f);
	}
	return ((cr >> 3) << 10) | ((cg >> 3) << 5) | (cb >> 3);
}

///////////////////////////////////////////////
template <class T>
inline unsigned short color3<T>::pack_rgb565()
{
	unsigned char cr;
	unsigned char cg;
	unsigned char cb;
	if(sizeof(r) == 1) // char and unsigned char
	{
		cr = r;
		cg = g;
		cb = b;
	}
	else if(sizeof(r) == 2) // short and unsigned short
	{
		cr = unsigned short(r)>>8;
		cg = unsigned short(g)>>8;
		cb = unsigned short(b)>>8;
	}
	else // float or double
	{
		cr = unsigned char(r * 255.0f);
		cg = unsigned char(g * 255.0f);
		cb = unsigned char(b * 255.0f);
	}
	return ((cr >> 3) << 11) |	((cg >> 2) << 5) | (cb >> 3);
}

///////////////////////////////////////////////
template <class T>
inline unsigned long color3<T>::pack_rgb888()
{
	unsigned char cr;
	unsigned char cg;
	unsigned char cb;
	if(sizeof(r) == 1) // char and unsigned char
	{
		cr = r;
		cg = g;
		cb = b;
	}
	else if(sizeof(r) == 2) // short and unsigned short
	{
		cr = unsigned short(r)>>8;
		cg = unsigned short(g)>>8;
		cb = unsigned short(b)>>8;
	}
	else // float or double
	{
		cr = unsigned char(r * 255.0f);
		cg = unsigned char(g * 255.0f);
		cb = unsigned char(b * 255.0f);
	}
	return (cr << 16) | (cg << 8) | cb;
}

///////////////////////////////////////////////
template <class T>
inline unsigned long color3<T>::pack_argb8888()
{
	unsigned char cr;
	unsigned char cg;
	unsigned char cb;
	if(sizeof(r) == 1) // char and unsigned char
	{
		cr = r;
		cg = g;
		cb = b;
	}
	else if(sizeof(r) == 2) // short and unsigned short
	{
		cr = unsigned short(r)>>8;
		cg = unsigned short(g)>>8;
		cb = unsigned short(b)>>8;
	}
	else // float or double
	{
		cr = unsigned char(r * 255.0f);
		cg = unsigned char(g * 255.0f);
		cb = unsigned char(b * 255.0f);
	}
	return (cr << 16) | (cg << 8) | cb;
}

///////////////////////////////////////////////
template <class T>
inline void color3<T>::clamp(T bottom, T top)
{
	     if(r < bottom)	r = bottom;
	else if(r > top)	r = top;
	     if(g < bottom)	g = bottom;
	else if(g > top)	g = top;
	     if(b < bottom)	b = bottom;
	else if(b > top)	b = top;
}

///////////////////////////////////////////////
template <class T>
void color3<T>::maximum(const color3<T> &ca, const color3<T> &cb)
{
	r = (ca.r > cb.r) ? ca.r : cb.r;
	g = (ca.g > cb.g) ? ca.g : cb.g;
	b = (ca.b > cb.b) ? ca.b : cb.b;
}

///////////////////////////////////////////////
template <class T>
void color3<T>::minimum(const color3<T> &ca, const color3<T> &cb)
{
	r = (ca.r < cb.r) ? ca.r : cb.r;
	g = (ca.g < cb.g) ? ca.g : cb.g;
	b = (ca.b < cb.b) ? ca.b : cb.b;
}

///////////////////////////////////////////////
template <class T>
void color3<T>::abs()
{
	r = (r < 0) ? -r : r;
	g = (g < 0) ? -g : g;
	b = (b < 0) ? -b : b;
}

///////////////////////////////////////////////
template <class T>
void color3<T>::adjust_contrast(T c)
{
	r = 0.5f + c * (r - 0.5f);
	g = 0.5f + c * (g - 0.5f);
	b = 0.5f + c * (b - 0.5f);
}

///////////////////////////////////////////////
template <class T>
void color3<T>::adjust_saturation(T s)
{
	// Approximate values for each component's contribution to luminance.
    // Based upon the NTSC standard described in ITU-R Recommendation BT.709.
    T grey = r * 0.2125f + g * 0.7154f + b * 0.0721f;    
    r = grey + s * (r - grey);
	g = grey + s * (g - grey);
	b = grey + s * (b - grey);
}

///////////////////////////////////////////////
template <class T>
void color3<T>::lerp(const color3<T> &ca, const color3<T> &cb, T s)
{
	r = ca.r + s * (cb.r - ca.r);
	g = ca.g + s * (cb.g - ca.g);
	b = ca.b + s * (cb.b - ca.b);
}

///////////////////////////////////////////////
template <class T>
void color3<T>::negative(const color3<T> &c)
{
	r = T(1.0f) - c.r;
	g = T(1.0f) - c.g;
	b = T(1.0f) - c.b;
}

///////////////////////////////////////////////
template <class T>
void color3<T>::grey(const color3<T> &c)
{
	T m = (c.r + c.g + c.b) / T(3);

	r = m;
	g = m;
	b = m;
	
	return pOut;
}

///////////////////////////////////////////////
template <class T>
void color3<T>::black_white(const color3<T> &c, T s)
{
	T add = c.r + c.g + c.b;
	if(add <= s)
	{
		r = T(0.0f);
		g = T(0.0f);
		b = T(0.0f);
	}
	else
	{
		r = T(1.0f);
		g = T(1.0f);
		b = T(1.0f);
	}
}

//////////////////////////////////////////////////////////////////////////////////////////////
} // end of namespace cry

#endif // CRYTEK_CRYCOLOR3_H
