////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   i3dengine.h
//  Version:     v1.00
//  Created:     28/5/2001 by Vladimir Kajalin
//  Compilers:   Visual Studio.NET
//  Description: 3dengine interface
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef CRY3DENGINEINTERFACE_H
#define CRY3DENGINEINTERFACE_H

#ifdef WIN32
	#ifdef CRY3DENGINEENGINE_EXPORTS
		#define CRY3DENGINEENGINE_API __declspec(dllexport)
	#else
		#define CRY3DENGINEENGINE_API __declspec(dllimport)
	#endif
#else
	#define CRY3DENGINEENGINE_API
#endif

// !!! Do not add any headers here !!!
#include <IProcess.h>
#include <CryEngineDecalInfo.h>
#include "IStatobj.h"
// !!! Do not add any headers here !!!

struct ISystem;
struct ICryCharInstance;
struct CVars;
struct pe_params_particle;
struct IMatInfo;
struct RenderLMData;
struct AnimTexInfo;
template <class T> class list2;

/*! SVariationValue used to specify value, which can have random variance.
		Used by particle system parameters.
 */
template <class T>
struct SVariationValue
{
	T value;
	// Random variation value.
	float variation;

	SVariationValue() { variation = 0; }

	//! Cast to holded type.
	operator T() const { return value; }
	//! Assign operator for variable.
	void operator=( const T& val ) { value = val; }

	T GetVariantValue() const { return value*(1 + variation*GenRand()); }

private:
	//! Generate random value in [-1,+1].
	float GenRand() const { return 2.0f*((float)rand()/RAND_MAX) - 1.0f; }
};

// Specialation for vectors.
Vec3 SVariationValue<Vec3>::GetVariantValue() const
{
	Vec3 v;
	v.x = value.x*(1 + variation*GenRand());
	v.y = value.y*(1 + variation*GenRand());
	v.z = value.z*(1 + variation*GenRand());
	return v;
}

//////////////////////////////////////////////////////////////////////////
typedef SVariationValue<float> FloatVariant;
typedef SVariationValue<Vec3> Vec3Variant;


//! Particle Blend Type
enum ParticleBlendType
{
  ParticleBlendType_AlphaBased,
  ParticleBlendType_ColorBased,
  ParticleBlendType_Additive,
  ParticleBlendType_None
};

//! Particle system parameters
struct ParticleParams
{
  ParticleParams() { memset(this,0,sizeof(*this)); }
  Vec3 vPosition; // spawn position
  Vec3 vDirection; // initial direction  (normalization not important)
  float fFocus; // if 0 - particles go in all directions, if more than 20 - particles go mostly in vDirection
  Vec3 vColorStart; // initial color
  Vec3 vColorEnd; // final color
  FloatVariant fSpeed; // initial speed ( +- 25% random factor applyed, m/sec )
	float fSpeedFadeOut; // Time in which before end of life time speed decreases from normal to 0.
	float fSpeedAccel;	// Constant speed acceleration along particle heading.
  Vec3Variant vRotation; // rotation speed (degree/sec)
  Vec3Variant vInitAngles; // initial rotation
  int   nCount; // number of particles to spawn
  FloatVariant fSize; // initial size of particles
  float fSizeSpeed; // particles will grow with this speed
	float fSizeFadeIn; // Time in which at the begning of life time size goes from 0 to fSize.
	float fSizeFadeOut; // Time in which at the end of life time size goes from fSize to 0.
	float fThickness;	// lying thickness - for physicalized particles only
  FloatVariant fLifeTime; // time of life of particle
  float fFadeInTime; // particle will fade in slowly during this time
  INT_PTR nTexId; // texture id for body and trail (if used) ( if 0 - will be used default ball/glow texture )	//AMD Port
  int   nTexAnimFramesCount; // number of frames in animated texture ( 0 if no animation )
  ParticleBlendType eBlendType; // see ParticleBlendType
  float fTailLenght; // delay of tail ( 0 - no tail, 1 meter if speed is 1 meter/sec )
  int   nParticleFlags; // see particle system flags
  bool  bRealPhysics; // use physics engine to control particles
  IStatObj * pStatObj; // if not 0 - this object will be used instead sprite
  ParticleParams * pChild; // child process definition
  float fChildSpawnPeriod; // if more than 0 - run child process every x seconds, if 0 - run it at collision
	float fChildSpawnTime; // if more then 0, Spawn child process for max this ammount of time.
  int   nDrawLast; // add this element into second list and draw this list last
  float fBouncenes; // if 0 - particle will not bounce from the ground, 0.5 is good in most cases
	float  fTurbulenceSize; // radius of turbulence
	float  fTurbulenceSpeed; // speed of rotation
	float fDirVecScale; //the game need to store this(Alberto)
	struct IEntityRender * pEntity; // spawner entity
	struct IShader * pShader; // shader for partice
	float fPosRandomOffset; // maximum distance of random offset from original position
	IMatInfo *pMaterial; // Override material.
	// Used internally.
	AnimTexInfo *pAnimTex;

	//////////////////////////////////////////////////////////////////////////
	// New parameters, used by Particle effects.
	//////////////////////////////////////////////////////////////////////////
	//! Spawn Position offset from effect spawn position.
	Vec3 vPositionOffset;
	//! Random offset of particle relative to spawn position.
	Vec3 vRandomPositionOffset;
	//! Delay actual spawn time by this ammount.
	FloatVariant fSpawnDelay;
	//! Life time of emitter.
	FloatVariant fEmitterLifeTime;
	//! When using emitter, spawn time between between 2 particle bursts.
	float fSpawnPeriod;

	//! Global effect scale. (0 ignored)
	float fScale;
	//! Object scale, multiplied with fSize to give scale adjustment between object and texture.
	//! 0 not affect fSize.
	float fObjectScale;

	Vec3 vNormal; // lying normal - for physicalized particles only
	int iPhysMat; // material for physicalized particles
  Vec3 vGravity; // gravity(wind) vector
};

/*!
IParticleEffect interface.
This Object with this interface is created by CreateParticleEffect method of 3d engine.
*/
struct IParticleEffect : public _i_reference_target_t
{
	//! Number of child processes of this effect.
	enum { NUM_PARTICLE_PROCESSES = 2 };
	//! Sound effect associated with particles effect.
	struct SoundParams
	{
		const char *szSound;
		float volume;
		float minRadius;
		float maxRadius;
		bool bLoop;
		SoundParams()
		{
			szSound = "";
			volume = 0;
			minRadius = 1;
			maxRadius = 10;
			bLoop = true;
		}
	};
	/*
	//! Additional effect parameters.
	struct EffectParams
	{
		//! Multiplies spawn period by this ammount.
		float spawnPeriodMultiplier;

		EffectParams() { spawnPeriodMultiplier = 1; }
	};
	*/

	//////////////////////////////////////////////////////////////////////////
	// Spawn this effect.
	virtual void Spawn( const Vec3 &pos,const Vec3 &dir,float fScale=1.0f ) = 0;

	//! Assign name to this particle effect.
	virtual void SetName( const char *sName ) = 0;
	//! Returns name of this particle effect.
	virtual const char* GetName() = 0;

	//! Enable or disable this effect.
	virtual void SetEnabled( bool bEnabled ) = 0;
	virtual bool IsEnabled() const = 0;

	//! Return ParticleParams or specified process.
	//! @param process 0=Primary Process, 1=Child Process.//zdes bil maks
	virtual ParticleParams& GetParticleParams( int process ) = 0;

	//////////////////////////////////////////////////////////////////////////
	// Texture and geometry.
	//////////////////////////////////////////////////////////////////////////
	virtual const char* GetTexture( int process ) const  = 0;
	virtual const char* GetGeometry( int process ) const = 0;
	virtual void SetTexture( int process,const char *s ) = 0;
	virtual void SetGeometry( int process,const char *s ) = 0;

	//////////////////////////////////////////////////////////////////////////
	// Sound parameters.
	//! Set Sound parameters for this particle effect.
	virtual void SetSoundParams( const SoundParams &params ) = 0;
	//! Get Sound parameters for this particle effect.
	virtual void GetSoundParams( SoundParams &params ) const = 0;
	//////////////////////////////////////////////////////////////////////////

	//////////////////////////////////////////////////////////////////////////
	// Spawn rate.
	//virtual EffectParams& GetEffectParams() = 0;

	//////////////////////////////////////////////////////////////////////////
	// Child particle systems.
	//////////////////////////////////////////////////////////////////////////
	//! Get number of sub Particles childs.
	virtual int GetChildCount() const = 0;
	//! Get sub Particles child by index.
	virtual IParticleEffect* GetChild( int index ) const = 0;
	//! Adds a new sub Particles.
	virtual void AddChild( IParticleEffect *pEffect ) = 0;
	//! Remove specific sub Particles
	virtual void RemoveChild( IParticleEffect *pEffect ) = 0;
	//! Remove all sub Particles.
	virtual void ClearChilds() = 0;
	//! Insert sub particles in between other child particles.
	virtual void InsertChild( int slot,IParticleEffect *pEffect ) = 0;
	//! Find slot where sub Particles stored.
	//! @retun slot index if Particles found, -1 if Particles not found.
	virtual int FindChild( IParticleEffect *pEffect ) const = 0;
};
TYPEDEF_AUTOPTR(IParticleEffect);

//! Particle emitter interface
struct IParticleEmitter : public _i_reference_target_t
{
	//! Set raw emitter particle parameters (Do not use if you use SetEffect).
	virtual void SetParams( const ParticleParams &params ) = 0;
	//! Set particle effect to spawn at this emitter (Do not use if you use SetParams).
	virtual void SetEffect( IParticleEffect *pEffect ) = 0;
	//! Get current particle params.
	virtual const ParticleParams& GetParams() const = 0;
	//! Set emitter position and direction and scale.
  virtual void SetPos( const Vec3 &vPos,const Vec3 &vDir,float fScale ) = 0;
	//! Override spawn period (Call after SetParams or SetEffect).
	virtual void SetSpawnPeriod( float fSpawnPeriod ) = 0;
	//! Override emitter life time (Call after SetParams or SetEffect).
	//! @param fLifeTime Number of seconds emitter is active (if negative unlimited life time).
	virtual void SetLifeTime( float fLifeTime ) = 0;
	//! Associate with spawner entity.
	virtual void SetEntity( IEntityRender *pEntity ) = 0;
	//! Associate with material.
	virtual void SetMaterial( IMatInfo *pMaterial ) = 0;
};
TYPEDEF_AUTOPTR(IParticleEmitter);

//! Particle system flags
#define PART_FLAG_BILLBOARD     0 // usual particle
#define PART_FLAG_HORIZONTAL    1 // flat horisontal rounds on the water
#define PART_FLAG_UNDERWATER    2 // particle will be removed if go out from outdoor water
#define PART_FLAG_LINEPARTICLE  4 // draw billboarded line from vPosition to vPosition+vDirection
#define PART_FLAG_SWAP_XY       8 // alternative order of rotation (zxy)
#define PART_SIZE_LINEAR       16 // change size liner with time
#define PART_FLAG_NO_OFFSET    32 // disable centering of static objects
#define PART_FLAG_DRAW_NEAR    64 // render particle in near (weapon) space
#define PART_FLAG_FOCUS_PLANE  128 // Focus will spread partices along normal plane instead of direction axis.
#define PART_FLAG_NO_DRAW_UNDERWATER 256 // particle will be not visible when it position is under water

//! Physics material enumerator, allows for 3dengine to get material id from game code
struct IPhysMaterialEnumerator
{
  virtual int EnumPhysMaterial(const char * szPhysMatName) = 0;
	virtual bool IsCollidable(int nMatId) = 0;
};

//! flags for DrawLowDetail
#define DLD_RESET_RENDER_OBJECTS  1
#define DLD_ADD_LIGHTSOURCES      2
#define DLD_INDOORS               4
#define DLD_DETAIL_OBJECTS        8
#define DLD_DETAIL_TEXTURES      16
#define DLD_TERRAIN_WATER        32
#define DLD_FAR_SPRITES          64
#define DLD_STATIC_OBJECTS      128
#define DLD_ENTITIES            256
#define DLD_PARTICLES           512
#define DLD_TERRAIN_FULLRES    1024
#define DLD_TERRAIN_LIGHT      2048
#define DLD_NEAR_OBJECTS			 4096

// phys foreign data flags
#define PFF_HIDABLE 1
#define PFF_EXCLUDE_FROM_STATIC 2

// duplicated definition to avoid including irenderer
#define STRIPTYPE_DEFAULT					4

//! contains current state of oclusion test
/*struct OcclusionTestClient
{
  OcclusionTestClient() { memset(this,0,sizeof(OcclusionTestClient)); bLastResult = true; }
  unsigned char ucOcclusionByTerrainFrames;
  unsigned char ucOcclusionByObjectsFrames;
  bool  bLastResult;
  int   nLastVisibleFrameID;
//	class CREOcclusionQuery * arrREOcclusionQuery[2];
};*/

//! structure to pass statobj group properites
struct IStatInstGroup
{
	IStatInstGroup() 
	{ 
		pStatObj = 0;
		bHideability = 0;
		fBending = 0;
		bCastShadow = 0;
//		bRecvShadow = 0;
		bPrecShadow = true;
		bUseAlphaBlending = 0;
//		bTakeBrightnessFromLightBit = 0;
		fSpriteDistRatio = 1.f;
		fShadowDistRatio = 1.f;
		fMaxViewDistRatio= 1.f;
		fBrightness = 1.f;
		bUpdateShadowEveryFrame = 0;
//		fAmbScale = 1.f;
		nSpriteTexRes = 0;
		pMaterial = 0;
    fBackSideLevel = 1.f;
    bCalcLighting = true;
    bUseSprites = true;
		bFadeSize = true;
	}

	struct IStatObj * pStatObj;
	bool	bHideability;
	float fBending;
	bool	bCastShadow;
//	bool	bRecvShadow;
	bool	bPrecShadow;
	bool	bUseAlphaBlending;
//	bool	bTakeBrightnessFromLightBit;
	float fSpriteDistRatio;
	float fShadowDistRatio;
	float fMaxViewDistRatio;
	float	fBrightness;
	bool  bUpdateShadowEveryFrame;
//	float fAmbScale;
	int		nSpriteTexRes;
  float fBackSideLevel;
  bool  bCalcLighting;
  bool  bUseSprites;
	bool  bFadeSize;

	//! Override material for this instance group.
	IMatInfo *pMaterial;

	//! flags similar to entity render flags
	int m_dwRndFlags;
};

//! Interface for water volumes editing from editor
struct IWaterVolume
{
	virtual void UpdatePoints(const Vec3 * pPoints, int nCount, float fHeight) = 0;
	virtual void SetFlowSpeed(float fSpeed) = 0;
	virtual void SetTriSizeLimits(float fTriMinSize, float fTriMaxSize) = 0;
	virtual void SetShader(const char * szShaderName) = 0;
	virtual void SetMaterial( IMatInfo *pMaterial ) = 0;
	virtual IMatInfo * GetMaterial() = 0;
	virtual void SetName(const char * szName) = 0;

	//! move water level and asssign new Z value to all vertices of water geometry
	virtual void SetPositionOffset(const Vec3d & vNewOffset) = 0;

  // !tiago: added, set refraction and reflection parameters for water shader
  virtual void SetRefractionParams(const Vec3 &pColor, float fTileX, float fTileY, float fBumpAmountX, float fBumpAmountY, float fSpeedX, float fSpeedY)=0;
  virtual void SetReflectionParams(const Vec3 &pColor, float fAmount, float fTileX, float fTileY, float fBumpAmountX, float fBumpAmountY, float fSpeedX, float fSpeedY)=0;
  virtual void SetWaveAmplitude(float fAmplitude) = 0;  
};

//! Interface for visarea volumes 
struct IVisArea
{
	//! return last rendered frame id
	virtual int GetVisFrameId() = 0;

	/*! return the list of conections for specified sector
    If bSkipDisabledPortals is true blocked areas will be not retuned 
		return value is limited to nMaxConnNum
		if return value is equal to nMaxConnNum - probobly not all connections was returned*/
	virtual	int GetVisAreaConnections(IVisArea ** pAreas, int nMaxConnNum, bool bSkipDisabledPortals = false) = 0;

	//! return true if has portal to outside
	virtual bool IsConnectedToOutdoor() = 0;

	//! return name set in editor in lowercase
	virtual const char * GetName() = 0;

	//! return true is area is portal area
	virtual bool IsPortal() = 0;
};


enum EVertsSharing
{
  evs_NoSharing=0,
  evs_ShareAndSortForCache=1,
};

// water level unknown
#define WATER_LEVEL_UNKNOWN -1000000

//! Interface to the 3d engine dll
struct I3DEngine : public IProcess
{
  //! Initialize 3dengine (call once, after creations)
	virtual bool Init() = 0;
  //! Load level from specified folder (precompiles some info if called from editor, will delete all old resources)
  virtual bool LoadLevel(const char * szFolderName, const char * szMissionName, bool bEditorMode = false) = 0;
  //! Update 3dengine (should be called every frame)
	virtual void Update() = 0;
  //! Set camera for rendering
	virtual void SetCamera(const CCamera &cam, bool bToTheScreen=true) = 0;	// set before calling Draw()
  //! Draw world
	virtual void Draw() = 0;
  //! Draw world for rendering into texture, DrawFlags will define what to draw
  virtual void DrawLowDetail(const int & DrawFlags) = 0;

  //! Shut down 3dengine
	virtual void ShutDown(bool bEditorMode=false) = 0;
  //! Delete everything
  virtual void Release() = 0;

  //! Activates lighting for indoors (for debug only)
	virtual void ActivateLight(const char *szName,bool bActivate)=0;

  /*! Load cgf file and create non animated object.
      Returns pointer to already loaded object with same name if found.
      Reference counting used */
	virtual IStatObj * MakeObject(const char * szFileName, const char * szGeomName = 0, 
		EVertsSharing eVertsSharing = evs_ShareAndSortForCache, 
		bool bLoadAdditinalInfo = true, 
    bool bKeepInLocalSpace = false) = 0;
	//! Reduces reference counter for object and deletes object if counter is 0
	virtual bool ReleaseObject(IStatObj * pObject)=0;

	//! Creates empty static object.
	virtual IStatObj* MakeObject() = 0;

	//! Get number of loaded CGFs
	virtual int GetLoadedObjectCount() { return 0; }

  //! Register entities in sectors for rendering
  virtual void RegisterEntity( IEntityRender * pEntity )=0;
  //! Unregister entities in sectors 
  virtual bool UnRegisterEntity( IEntityRender * pEntity )=0;
  /*! Unregister all entities in all sectors (or only one if specified)
      Returns true if specified entity was found */
  virtual bool UnRegisterInAllSectors(IEntityRender * pEntity = NULL) = 0;

  //! Get water level in specified point (taking into account global water level and water volumes)
	virtual float GetWaterLevel(const Vec3 * pvPos = NULL, Vec3 * pvFlowDir = NULL) = 0;
  /*! Get water level in position of specified object taking into account global water level 
      and water volumes. For indoor objects global water level is ignored.*/
  virtual float GetWaterLevel(IEntityRender * pEntityRender, Vec3 * pvFlowDir = NULL) = 0;

  //! Spawn particles 
  virtual void SpawnParticles( const ParticleParams & SpawnParticleParams ) = 0;

  //! Removes all particles and decals
  virtual void ResetParticlesAndDecals( ) = 0;

  //! Create particle emitter
  virtual IParticleEmitter* CreateParticleEmitter() = 0;
  //! Delete particle emitter
  virtual void DeleteParticleEmitter(IParticleEmitter * pPartEmitter) = 0;

	//////////////////////////////////////////////////////////////////////////
	// ParticleEffects
	//////////////////////////////////////////////////////////////////////////
	//! Creates new particle effect.
	virtual IParticleEffect* CreateParticleEffect() = 0;
	//! Deletes existing particle effect.
	virtual void DeleteParticleEffect( IParticleEffect* pEffect ) = 0;
	//! Find existing particle effect by name.
	virtual IParticleEffect* FindParticleEffect( const char *sEffectName ) = 0;
	//////////////////////////////////////////////////////////////////////////


	//! Spawn decal on the walls, static objects, terrain and entities
  virtual void CreateDecal( CryEngineDecalInfo & Decal )=0;

  /*! Call back for renderer. 
      Renders detail textures on terrain. Will be removed from here.*/
	virtual void DrawTerrainDetailTextureLayers() = 0; // used by renderer
  /*! Call back for renderer.
      Renders far trees/object as sprites. Will be removed from here.*/
	virtual void DrawFarTrees() = 0; // used by renderer
  /*! Call back for renderer.
      Renders decals, particles, bflyes. Need to remove from here.*/
	virtual void DrawTerrainParticles(struct IShader * pShader) = 0;// used by renderer  
  
  /*! Set render call back to make possible to 
    render user defined objects from outside of 3dengine (used by editor)*/
  virtual void SetRenderCallback(void (*pFunc)(void *pParams), void *pParams) = 0;

  /*! Load cgf and caf files and creates animated object.
      Returns pointer to already loaded object with same name if found.
      Reference counting used */
  virtual ICryCharInstance * MakeCharacter(const char * cid_file_name, unsigned int dwFlags = 0)=0;
	
	/*! Scan the file and determine if it's a valid for animation cgf.
	    Returns true if the file is a valid character file. */
	virtual bool IsCharacterFile (const char* szFileName) = 0;

  //! Reduces reference counter for object and deletes object if counter is 0
  virtual void RemoveCharacter(ICryCharInstance * pCryCharInstance)=0;  

  //! Get current world color 
  virtual Vec3 GetWorldColor(bool bScaled=true)=0; // for example red at evening
  //! Set current world color 
  virtual void SetWorldColor(Vec3 vColor)=0;  
  //! Set world color ratio
  virtual void SetWorldColorRatio(float fWorldColorRatio) = 0;
  //! Get world color ratio
  virtual float GetWorldColorRatio() = 0;

  //! Switch skybox
  virtual void SetSkyBox(const char * szShaderName) = 0;

  //! Set/Get view distance
  virtual void SetMaxViewDistance(float fMaxViewDistance)=0;
  virtual float GetMaxViewDistance()=0;
  
  //! Set/Get fog params
	virtual void SetFogColor(const Vec3& vFogColor)=0;
	virtual void SetFogStart(const float fFogStart)=0;
	virtual void SetFogEnd(const float fFogEnd)=0;
	virtual Vec3 GetFogColor( )=0;
	virtual float GetFogStart( )=0;
	virtual float GetFogEnd( )=0;

  //! Returns interpolated terrain elevation ( all x,y values are valid )
  virtual float GetTerrainElevation(float x, float y) = 0;
  //! Returns terrain elevation ( only values between 0 and WORLD_SIZE are valid )
  virtual float GetTerrainZ(int x, int y)=0;
  //! Returns current size of terrain unit in meters ( currently is 2 )
  virtual int   GetHeightMapUnitSize()=0;
  //! Returns size of terrain in meters ( currently is 2048 )
  virtual int   GetTerrainSize()=0;
  //! Returns size of terrain sector in meters ( currently is 64 )
  virtual int   GetTerrainSectorSize()=0;
  //! Maximum view distance in meters ( usualy 512 )
  virtual float GetMaxViewDist()=0;

  //! Places object at specified position (for editor)
  virtual bool AddStaticObject(int nObjectID, const Vec3 & vPos, const float fScale, unsigned char ucBright=255) = 0;
  //! Removes static object from specified position (for editor)
	virtual bool RemoveStaticObject(int nObjectID, const Vec3 & vPos) = 0;
	//! On-demand physicalization of a static object
	virtual bool PhysicalizeStaticObject(void *pForeignData,int iForeignData,int iForeignFlags) = 0;
	//! Removes all static objects on the map (for editor)
  virtual void RemoveAllStaticObjects() = 0;
  //! Allows to set terrain surface type id for specified point in the map (for editor)
  virtual void SetTerrainSurfaceType(int x, int y, int nType)=0; // from 0 to 6 - sur type ( 7 = hole )
  /*! Return terrain surface type id for specified point on the map 
      Return -1 if point is outside of the map or if there is hole in terrain here ) */
  virtual int  GetTerrainSurfaceType(int x, int y)=0; // from 0 to 6 - sur type ( 7 = hole )
  
  //! Updates part of hight map (in terrain units, by default update only elevation)
  virtual void SetTerainHightMapBlock(int x1, int y1, int nSizeX, int nSizeY, unsigned short * TerrainBlock, unsigned short nUpdateMask = (((unsigned short)-1) & (~31))) = 0;
  //! Returns true if game modified terrain hight map since last update by editor
  virtual bool IsTerainHightMapModifiedByGame() = 0;
  
	//! returns terrain sector texture id, texture dimensions and disable streaming on this sector
	//! returns 0 in case of error (wrong SectorOrogin)
	virtual int LockTerrainSectorTexture(int nSectorOroginX, int nSectorOroginY, int & nTexDim) = 0;

	//! Set group parameters
	virtual bool SetStatInstGroup(int nGroupId, const IStatInstGroup & siGroup) = 0;

	//! Get group parameters
	virtual bool GetStatInstGroup(int nGroupId,       IStatInstGroup & siGroup) = 0;

	//! returns dimensions of entire terrain texture
	virtual int GetTerrainTextureDim() = 0;

  //! Set burbed out flag
  virtual void SetTerrainBurnedOut(int x, int y, bool bBurnedOut) = 0;
  //! Get burbed out flag
  virtual bool IsTerrainBurnedOut(int x, int y) = 0;

	//! recalculate shore geometry, save it to disk and reload into engine
	virtual void RecompileBeaches() = 0;

  //! Makes hole in terrain if there is no static objects near
  virtual void OnExplosion(Vec3 vPos, float fRadius, int nTexID, bool bDeformTerrain = true) = 0;

  //! Makes 3d waves on the water surface
  virtual void AddWaterSplash (Vec3 vPos, enum eSplashType eST, float fForce, int Id=-1) = 0;

  //! Force to draw quad on entire screen with specified shader (night vision) ( use szShaderName="" to disable drawing )
  virtual void SetScreenShader( const char * szShaderName ) = 0;

  //! Set physics material enumerator
  virtual void SetPhysMaterialEnumerator(IPhysMaterialEnumerator * pPhysMaterialEnumerator) = 0;
  virtual IPhysMaterialEnumerator * GetPhysMaterialEnumerator() = 0;

  //! Allows to enable fog in editor
  virtual void SetupDistanceFog() = 0;

  //! Load environment settings for specified mission
  virtual void LoadEnvironmentSettingsFromXML(const char * szMissionName, bool bEditorMode, const char * szMissionXMLString = 0) = 0;

  //! Load detail texture and detail object settings from XML doc (load from current LevelData.xml if pDoc is 0)
	virtual void	LoadTerrainSurfacesFromXML(void * pDoc = NULL) = 0;

  //! Recalculate shore geometry
  virtual void UpdateBeaches() = 0;

  //! Returns true if point is in water
  virtual bool IsPointInWater(Vec3 vPos) = 0;

  //! Returns true if point is in the building
  //virtual bool IsPointInsideIndoors(const Vec3 & vPos) = 0;

  //! Creates new light source in the world to be used during this frame (or longer)
  virtual void AddDynamicLightSource(const class CDLight & LSource, IEntityRender * pEnt, int nEntityLightId=-1, const Matrix44* pMatrix=NULL) = 0;

  //! Make move/bend vegetations in specified area (not implemented yet)
  virtual void ApplyForceToEnvironment(Vec3 vPos, float fRadius, float fAmountOfForce) = 0;

  //! Return sun position (if bMoveUp=true sun will be 30% higher, it makes shadows from objects not so long)
  virtual Vec3 GetSunPosition(bool bMoveUp = true) = 0;
  
  //! Returns light mask for this point (valid only during rendering stage)
  virtual unsigned int GetLightMaskFromPosition(const Vec3 & vPos, float fRadius=1.f) = 0;

  //! Returns lighting level for this point
	virtual Vec3 GetAmbientColorFromPosition(const Vec3 & vPos, float fRadius=1.f) = 0;

  //! Returns fog volume id
	virtual int GetFogVolumeIdFromBBox(const Vec3 & vBoxMin, const Vec3 & vBoxMax) = 0;

  //! Return surface normal at specified position
  //virtual Vec3 GetTerrainSurfaceNormal(Vec3 vPos) = 0;

  //! Render shadows of objects into frame buffer and read this picture
  virtual bool MakeSectorLightMap(int nSectorOriginX, int nSectorOriginY, unsigned char * pImage, int nImageSize) = 0;

  //! get distance to the sector containig ocean water
  virtual float GetDistanceToSectorWithWater() = 0;

  //! allows to slowly replace skybox with bg(fog) color
  virtual void SetSkyBoxAlpha(float fAlpha /* from 0 to 1 */) = 0;

  //! set/get current number of butterflies and other live particles
  virtual void SetBFCount(int nCount/* from 0 to MAX_BF_COUNT */) = 0;
  virtual int  GetBFCount() = 0;
  virtual void SetGrasshopperCount(int nCount/* from 0 to MAX_BF_COUNT */) = 0;
  virtual int  GetGrasshopperCount() = 0;
  virtual void SetGrasshopperCGF( int nSlot, IStatObj * pStatObj ) = 0;

  //! get environment ambient color specified in editor
  virtual Vec3 GetOutdoorAmbientColor() = 0;

  //! get environment sun color specified in editor
  virtual Vec3 GetSunColor() = 0;

  //! check object visibility taking into account portals and terrain occlusion test
//  virtual bool IsBoxVisibleOnTheScreen(const Vec3 & vBoxMin, const Vec3 & vBoxMax, OcclusionTestClient * pOcclusionTestClient = NULL)=0;
  //! check object visibility taking into account portals and terrain occlusion test
//  virtual bool IsSphereVisibleOnTheScreen(const Vec3 & vPos, const float fRadius, OcclusionTestClient * pOcclusionTestClient = NULL)=0;

  //! Clears all rendering resources, should be called before LoadLevel() and before loading of any textures from script
  virtual void ClearRenderResources()=0;

  //! Alloc entity render info
  virtual struct IEntityRenderState * MakeEntityRenderState() = 0;
  //! Free entity render info
  virtual void FreeEntityRenderState(IEntityRender * pEntity) = 0;

  //! Return pointer to full file name of file in current level folder
  virtual const char * GetLevelFilePath(const char * szFileName) = 0;

  //! make smoothed terrain for ocean calculations
  virtual void MakeUnderWaterSmoothHMap(int nWaterUnitSize) = 0;

  //! returns ptr to smoothed terrain for ocean calculations
  virtual unsigned short * GetUnderWaterSmoothHMap(int & nDimensions) = 0;

  //! refresh detail objects
  virtual void UpdateDetailObjects() = 0;

  //! display 3dengine stats, takes and updates current cursor position
  virtual void DisplayInfo(float & fTextPosX, float & fTextPosY, float & fTextStepY) = 0;

	//!  added by M.M. (don't copy the interface pointer and don't forget to call Release)
	//! used for precalculation in ShadowVolumes
	virtual class IEdgeConnectivityBuilder *GetNewConnectivityBuilder( void )=0;

	//! creates a connectivity object that can be used to deserialize the connectivity data
	virtual class IStencilShadowConnectivity *NewConnectivity() = 0;

	//! Returns the connectivity builder that's building the connectivity for static objects
	virtual class IEdgeConnectivityBuilder *GetNewStaticConnectivityBuilder (void) = 0;

	//!  added by M.M. (don't delete or Release the interface pointer)
	//! used for precalculation in ShadowVolumes
	virtual class IEdgeDetector *GetEdgeDetector( void )=0;

	//! Enable/disable heat vision mode
	virtual void EnableHeatVision(bool bEnable) = 0;

	//! Enable/Disable portal at specified position
	virtual void	ActivatePortal(const Vec3 &vPos, bool bActivate, IEntityRender *pEntity) = 0;

	//! Count memory usage
	virtual void GetMemoryUsage(class ICrySizer * pSizer)=0;

	//! Create water volume
	virtual IWaterVolume * CreateWaterVolume() = 0;

	//! Delete water volume
	virtual void DeleteWaterVolume(IWaterVolume * pWaterVolume) = 0;

	//! Find water volume by name
	virtual IWaterVolume * FindWaterVolumeByName(const char * szName) = 0;

	//! Create visarea
	virtual IVisArea * CreateVisArea() = 0;

	//! Delete visarea
	virtual void DeleteVisArea(IVisArea * pVisArea) = 0;

	//! Update visarea
	virtual void UpdateVisArea(IVisArea * pArea, const Vec3 * pPoints, int nCount, const char * szName, float fHeight, const Vec3 & vAmbientColor, bool bAfectedByOutLights, bool bSkyOnly, const Vec3 & vDynAmbientColor) = 0;

	/*! decides if a sound is potentially hearable between vis areas (different sectors, a door block the sounds)
		@param pArea1	  (the sector of one of the source)
		@param pArea2	  (the sector of one of the source)
		@return	true if sound is hearable, false otherwise
	*/
	virtual bool IsVisAreasConnected(IVisArea * pArea1, IVisArea * pArea2, int nMaxReqursion = 1, bool bSkipDisabledPortals = true) = 0;

	//! Create EntityRender object
	virtual IEntityRender * CreateEntityRender() = 0;

	//! Delete EntityRender object
	virtual void DeleteEntityRender(IEntityRender * pEntityRender) = 0;

	//! draw rain for renderer to render into texture
	virtual void DrawRain() = 0;

	//! set density of rain (0 - no rain, 1.f - max rain)
	virtual void SetRainAmount( float fAmount ) = 0;

	//! set wind direction and force Vec3(0,0,0) = no wind
	virtual void SetWindForce( const Vec3 & vWindForce ) = 0;

	//! returns amount of light (form 0.f to 1.f, valid only during entity render)
  //! if bOnlyVisibleLights is true - only sources with origin in the camera frustum will be taken into account
	virtual float GetLightAmountForEntity(IEntityRender * pEntity, bool bOnlyVisibleLights = false) = 0;

  //! returns amount of ambient light (form 0.f to 1.f)
  virtual float GetAmbientLightAmountForEntity(IEntityRender * pEntity) = 0;

	//! return indoor visibility area containing this position
	virtual	IVisArea * GetVisAreaFromPos(const Vec3 &vPos) = 0;	

	//! enable/disable outdoor water and beaches rendering
	virtual void EnableOceanRendering(bool bOcean, bool bShore) = 0;

	//! Allocates new material object, and add it to the material manager.
	virtual IMatInfo* CreateMatInfo() = 0;
	//! Deletes material object from material manager.
	//! @param pMatInfo Must be a valid pointer to existing material object.
	virtual void DeleteMatInfo(IMatInfo * pMatInfo) = 0;
	//! Rename material object (You should not use IMatInfo::SetName directly).
	//! @param pMtl Must be a valid pointer to existing material object.
	virtual void RenameMatInfo( IMatInfo *pMtl,const char *sNewName ) = 0;
	//! Find loaded material from the library.
	virtual IMatInfo* FindMaterial( const char *sMaterialName ) = 0;

	//! \brief Create an instance of a lightmap serialization manager
	virtual struct ILMSerializationManager * CreateLMSerializationManager() = 0;

	//! return true if object can be visible soon
	virtual bool IsPotentiallyVisible(IEntityRender * pEntityRender, float fAdditionRadius=0) = 0;

	//! create new static lsource, returns source id or -1 if it fails
	virtual INT_PTR AddStaticLightSource(const class CDLight & LSource, IEntityRender *pCreator, ICryCharInstance * pCryCharInstance=NULL, const char * szBoneName=NULL) = 0;	//AMD Port
	//! delete static lsource (return false if not found)
	virtual bool DeleteStaticLightSource(INT_PTR nLightId) = 0;		//AMD Port
	//! gives access to static lsources list (for lmap generator)
	virtual const list2<CDLight*> * GetStaticLightSources() = 0;
  //! Reload heightmap and reset decals and particles, in future will restore deleted vegetations
  virtual void RestoreTerrainFromDisk() = 0;

  // tmp
  virtual const char * GetFilePath(const char * szFileName) { return GetLevelFilePath(szFileName); }

  // tiago: added

  //! set blur mask texture
  virtual void SetBlurMask(ITexPic *pMask)=0;
  //! set screen mask texture
  virtual void SetScreenMask(ITexPic *pMask)=0;
  //! set screen fx type
  virtual void SetScreenFx(const char *pEffectName, int iActive)=0;
  //! set screen fx paramater
  virtual void SetScreenFxParam(const char *pEffectName, const char *pEffectParam, void *pValue)=0;

  //! physicalize area if not physicalized yet
  virtual void CheckPhysicalized(const Vec3 & vBoxMin, const Vec3 & vBoxMax) = 0;

	//! in debug mode check memory heap and makes assert, do nothing in release
	virtual void CheckMemoryHeap() = 0;

	//! return value of e_obj_lod_ratio cvar(about 10.f), used for entities
	virtual float GetObjectsLODRatio() = 0;

	//! return value of e_obj_view_dist_ratio cvar(about 55.f), used for entities
	virtual float GetObjectsViewDistRatio() = 0;

	/*!
	Set material parameter
	@param szMatName materal name
	@param nTexSlot text slot id, see EFTT_DIFFUSE for example
	@param nSubMatId submaterial id, -1 means use root material
	@param szParamName can be one of:
		
		m_eTGType
		m_eRotType
		m_eUMoveType
		m_eVMoveType
		m_bTexGenProjected

		m_Tiling[0]
		m_Tiling[1]
    m_Tiling[2]
		m_Offs[0]
		m_Offs[1]
		m_Offs[2]

		m_Rot[0]
		m_Rot[1]
		m_Rot[2]
		m_RotOscRate[0]
		m_RotOscRate[1]
		m_RotOscRate[2]
		m_RotOscAmplitude[0]
		m_RotOscAmplitude[1]
		m_RotOscAmplitude[2]
		m_RotOscPhase[0]
		m_RotOscPhase[1]
		m_RotOscPhase[2]

		m_UOscRate
		m_VOscRate
		m_UOscAmplitude
		m_VOscAmplitude
		m_UOscPhase
		m_VOscPhase

	@param fValue new value
*/
	virtual bool SetMaterialFloat( char * szMatName, int nSubMatId, int nTexSlot, char * szParamName, float fValue ) = 0;

	//! close terrain texture file handle and allows to replace/update it
	virtual void CloseTerrainTextureFile() = 0;
	
	//! remove all decals attached to specified entity
	virtual void DeleteEntityDecals(IEntityRender * pEntity) = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////
#ifdef __cplusplus
extern "C" {
#endif

#if (defined(GAMECUBE) || defined (PS2))
#define __TIMESTAMP__ "Ver1.0"
#endif

// expirimental way to track interface version 
// this value will be compared with value passed from system module
const char g3deInterfaceVersion[64] = __TIMESTAMP__;

// CreateCry3DEngine function type definition
typedef I3DEngine * (*PFNCREATECRY3DENGINE)(ISystem	* pSystem,const char * szInterfaceVersion);

//! Creates 3dengine instance
CRY3DENGINEENGINE_API I3DEngine * CreateCry3DEngine(ISystem	* pSystem,const char * szInterfaceVersion=g3deInterfaceVersion);

#ifdef __cplusplus
}
#endif




#endif //CRY3DENGINEINTERFACE_H

