////////////////////////////////////////////////////////////////////////////
//
//  Crytek Engine Source File.
//  Copyright (C), Crytek Studios, 2002.
// -------------------------------------------------------------------------
//  File name:   iCryAnimation.h
//  Version:     v1.00
//  Created:     28/5/2001 by Vladimir Kajalin
//  Compilers:   Visual Studio.NET
//  Description: CryAnimation interface
// -------------------------------------------------------------------------
//  History:
//
////////////////////////////////////////////////////////////////////////////

#ifndef ICRY_ANIMATION
#define ICRY_ANIMATION

#ifndef _XBOX
	#ifdef CRYANIMATION_EXPORTS
		#define CRYANIMATION_API __declspec(dllexport)
	#else
		#define CRYANIMATION_API __declspec(dllimport)
	#endif
#else
	#define CRYANIMATION_API
#endif

#include "smartptr.h"
#include "Cry_Math.h"

//! Forward declarations
#include "IBindable.h"
struct	IShader;
class   CryMatrix;
struct  SRendParams;
template <class T> class list2;
struct CryEngineDecalInfo;
struct ParticleParams;
struct CryCharMorphParams;
struct IMatInfo;

//#include "CryParticleSpawnInfo.h"

//////////////////////////////////////////////////////////////////////
#include <IPhysics.h>

//! flags used by game
enum
{
	CS_FLAG_DRAW_MODEL = 1 << 0,
	CS_FLAG_DRAW_NEAR  = 1 << 1,
	CS_FLAG_UPDATE     = 1 << 2
	/*
	CS_FLAG_MIRROR_X   = 1 << 3,// PARTIALLY supported (only for 1stperson weapons)
	CS_FLAG_MIRROR_Y   = 1 << 4,// NOT supported
	CS_FLAG_MIRROR_Z   = 1 << 5 // NOT supported
	*/
};

//! character limb indentifiers used for IK
enum limb_identifiers { LIMB_LEFT_ARM=0,LIMB_RIGHT_ARM=1, LIMB_LEFT_LEG=2,LIMB_RIGHT_LEG=3 };
enum limb_ik_flags { ik_leg=1, ik_arm=2, ik_avoid_stretching=4 };
#define IK_NOT_USED 1E9f

//! Interface to bone matrices for procedural animations
struct ICryBone
{
	// sets the flag. If the flag is set to true, the relative to default pose matrix
	// is not calculated (it must be set externally via the GetRelativeToDefPoseMatrixPointer())
	// used by LipSync
	virtual void DoNotCalculateBoneRelativeMatrix(bool bDoNotCalculate) = 0;

	// unfixes the bone matrix - the effect relaxes all the previous calls to FixBoneMatrix
	void UnfixBoneMatrix()
	{
		DoNotCalculateBoneRelativeMatrix(false);
	}
  
	// remembers the given matrix as the bone's matrix (relative to parent) and from now on
	// doesn't change it. To reset this effect, call DoNotCalculateBoneRelativeMatrix(false)
	virtual void FixBoneMatrix (const Matrix44& mtxBone) = 0;

	// fixes the bone matrix to the given position in world coordinates,
	// assuming the character position and orientation are given by the vCharPos and vCharAngles
	// vCharAngles are the same as in the entity and in the Draw call to ICryCharInstance
	virtual void FixBoneOriginInWorld (const Vec3& vCharPos, const Vec3& vCharAngles, const Vec3& vTargetOrigin) = 0;

	// Sets the bone matrix to the given position in world coordinates only for this frame (until update)
	// assuming the character position and orientation are given by the vCharPos and vCharAngles
	// vCharAngles are the same as in the entity and in the Draw call to ICryCharInstance
	virtual void SetBoneOriginInWorld (const Vec3& vCharPos, const Vec3& vCharAngles, const Vec3& vTargetOrigin) = 0;
	
	// sets the plus-matrix rotation components.
	// the plus-matrix is used to rotate the upper body in response to small rotation of the head
	// (in order to avoid to frequent rotation of the whole body) and is set by the game code
	// NOTE: the angles are in degrees
	virtual void SetPlusRotation(float x, float y, float z) = 0;
	virtual void SetPlusRotation(const CryQuat& qRotation) = 0;

	// resets the plus-matrix to identity, so that the body is not rotated additionally to the currently played animation.
	virtual void ResetPlusRotation() = 0;

	virtual Vec3 GetBonePosition() = 0;
	virtual Vec3 GetBoneAxis(char cAxis) = 0;

	//virtual int GetId() = 0;
	// returns the parent world coordinate system rotation as a quaternion
	virtual CryQuat GetParentWQuat () = 0;

	virtual ICryBone* GetParent() = 0;
	// returns the matrix relative to parent
	virtual const Matrix44& GetRelativeMatrix() = 0;
	// returns the matrix in object coordinates
	virtual const Matrix44& GetAbsoluteMatrix() = 0;
};

struct AnimSinkEventData
{
	AnimSinkEventData (void* _p = NULL):
		p(_p),
		n((INT_PTR)_p)
	{
	}
	
	void* p;
	INT_PTR   n;

	operator void* () {return p;}
	operator const void* ()const {return p;}
	bool operator == (const AnimSinkEventData& that) const {return p == that.p && n == that.n;}
};


//! CallBack functions for game code
struct ICharInstanceSink
{
  virtual void OnStartAnimation(const char *sAnimation) = 0;
  virtual void OnAnimationEvent(const char *sAnimation, AnimSinkEventData data) = 0;
  virtual void OnEndAnimation(const char *sAnimation) = 0;
};

//! Type of blending between animation layers
enum AnimTwinMode
{
  AnimTwinMode_Replace,
  AnimTwinMode_Add
};

//! Interface describing a set of animations. Each animation is described as a set of properties
//! This interface is free from concepts specific for CryEngine
struct IAnimationSet
{
	//! Returns the number of animations in this set
	virtual int Count() = 0;

	//! Returns the number of morph targets in the set
	virtual int CountMorphTargets() {return 0;}

	//! Returns the index of the animation in the set, -1 if there's no such animation
	virtual int Find (const char* szAnimationName) = 0;

	//! Loads the animation data in memory. fWhenRequired is the timeout in seconds from current moment when
	//! the animation data will actually be required
	virtual void StartLoadAnimation (int nAnimId, float fWhenRequired) {}

	//! Loads the animation data in memory. fWhenRequired is the timeout in seconds from current moment when
	//! the animation data will actually be required
	void StartLoadAnimation (const char* szAnimationName, float fWhenRequired)
	{
		StartLoadAnimation(Find (szAnimationName),fWhenRequired);
	}

	//! Unloads animation from memory
	//! The client must take into account that the animation can be shared and, if unloaded, the other
	//! character models (animation sets) will have to load it back to use.
	virtual void UnloadAnimation (int nAnimId) {}
	
	//! Unloads animation from memory
	//! The client must take into account that the animation can be shared and, if unloaded, the other
	//! character models (animation sets) will have to load it back to use.
	void UnloadAnimation (const char* szAnimationName)
	{
		UnloadAnimation(Find(szAnimationName));
	}

	//! Returns the index of the morph target in the set, -1 if there's no such morph target
	virtual int FindMorphTarget (const char* szMorphTarget) {return -1;}

	//! Returns the given animation length, in seconds
	virtual float GetLength (int nAnimationId) = 0;

	//! Returns the given animation length, in seconds
	virtual float GetLength (const char* szAnimationName)
	{
		return GetLength(Find(szAnimationName));
	}

	//! Returns the given animation's start, in seconds; 0 if the id is invalid
	virtual float GetStart (int nAnimationId) {return 0;}// default implementation

	//! Returns the given animation's start, in seconds
	virtual float GetStart (const char* szAnimationName)
	{
		return GetStart (Find(szAnimationName));
	}

	//! Returns the given animation name
	virtual const char* GetName (int nAnimationId) = 0;

	//! Returns the name of the morph target
	virtual const char* GetNameMorphTarget (int nMorphTargetId) {return "!NOT IMPLEMENTED!";}

	//! Retrieves the animation loop flag
	virtual bool IsLoop (int nAnimationId) = 0;

	//! Retrieves the animation loop flag
	virtual bool IsLoop (const char* szAnimationName)
	{
		return IsLoop(Find(szAnimationName));
	}
};

//! Interface that describes and manipulates a set of animations
//! Cry... because it's specific to the needs of the CryEngine
struct ICryAnimationSet: public IAnimationSet
{
	//! Modifies the animation loop flag
	virtual void SetLoop (int nAnimationId, bool bIsLooped) = 0;
};

//! Interface to the character instance model
//! A set of characters created out of the same CGF share many properties, and
//! interface to those properties is through the ICryCharModel.
//! The instance of the ICryCharModel is guaranteed to be alive at least as long as
//! all its ICryCharInstance are alive
struct ICryCharModel: public _i_reference_target_t
{
	//! Returns the interface for animations applicable to this model
	virtual ICryAnimationSet* GetAnimationSet () = 0;
	
	//! Return name of bone from bone table, return zero if the nId is out of range (the game gets this id from physics)
	virtual const char * GetBoneName(int nId) const = 0;

	//! Returns the index of the bone by its name or -1 if no such bone exists; this is Case-Sensitive
	virtual int GetBoneByName (const char* szName) {return -1;}

	//! Returns the number of bones; all bone ids are in the range from 0 to this number exclusive; 0th bone is the root
	virtual int NumBones() const = 0;

	//! Returns the scale of the model - not used now
	virtual float GetScale() const = 0;

	enum ClassEnum
	{
		CLASS_UNKNOWN,
		CLASS_ANIMOBJECT,
		CLASS_CRYCHARBODY
	};

	virtual ClassEnum GetClass() {return CLASS_UNKNOWN;}

	//! returns the file name of the character model
	virtual const char* GetFileName() {return "";}

	// returns the extra data attached to the character by the artist during exporting
	virtual const char* GetProperty(const char* szName) {return "";}
};


//! This is interface to an object that should be skinned
//! It's used to let the renderer know that an object callback must be called before rendering because
//! the object vertices can be deformed
struct IDeformableRenderMesh
{
	//! Renderer calls this function to allow update the video vertex buffers right before the rendering
	virtual void ProcessSkinning(const Matrix44& mtxModel, int nTemplate, int nLod=-1, bool bForceUpdate=false) = 0;
};

struct ICryCharVisualElement: public _reference_target_t
{
	// returns true if the given submesh is visible
	virtual bool IsVisible() = 0;

	// depending on bVisible, either makes the submesh visible or invisible
	virtual void SetVisible(bool bVisible = true) = 0;
};

// this is submesh of the character instance: a piece of geometry that can be
// turned visible/invisible by wish
struct ICryCharSubmesh: public ICryCharVisualElement
{
	// returns the model of the submesh, or NULL in case of failure
	virtual ICryCharModel* GetModel () = 0;

	//! Start the specified by parameters morph target
	virtual void StartMorph (const char* szMorphTarget, const CryCharMorphParams& params) = 0;

	//! Start the specified by parameters morph target
	virtual void StartMorph (int nMorphTargetId, const CryCharMorphParams& params) = 0;

	//! Finds the morph with the given id and sets its relative time.
	//! Returns false if the operation can't be performed (no morph)
	//! The default implementation for objects that don't implement morph targets is to always fail
	virtual bool SetMorphTime (int nMorphTargetId, float fTime) = 0;

	//! Set morph speed scale
	//! Finds the morph target with the given id, sets its morphing speed and returns true;
	//! if there's no such morph target currently playing, returns false
	virtual bool SetMorphSpeed (int nMorphTargetId, float fSpeed) = 0;

	//! Stops morph by target id
	virtual bool StopMorph (int nMorphTargetId) = 0;

	//! Stops all morphs
	virtual void StopAllMorphs() = 0;

	//! freezes all currently playing morphs at the point they're at
	virtual void FreezeAllMorphs(){}

	// Adds a decal to the character
	virtual void AddDecal (CryEngineDecalInfo& Decal) = 0;

	// cleans up the decals on this body part
	virtual void ClearDecals() = 0;

	//! returns the leaf buffer materials in this character (as they are used in the renderer)
	virtual const list2<struct CMatInfo>*getLeafBufferMaterials() = 0;

	virtual bool SetShaderTemplateName (const char *TemplName, int Id, const char *ShaderName=0,IMatInfo *pCustomMaterial=0, unsigned nFlags = 0) = 0;
};

// describes sword trail effect
struct ICryCharFxTrail: public ICryCharVisualElement
{

};

//! Interface to character animation
//! This interface contains methods for manipulating and querying an animated character
//! Instance. The methods allow modify the animated instance to the certain way,
//! animate it, render, retrieve BBox/etc, control physics, particles and skinning, transform.
//! TODO:
//! Split this interface up into a few logical interfaces, starting with the ICryCharModel
struct ICryCharInstance: public IBindable
{
	virtual ~ICryCharInstance() {};

	// Releases character instance.
	virtual void Release() = 0;

  // Set rendering flags like (draw/notdraw) and position offset
  virtual void SetFlags(int nFlags)=0;
  virtual int  GetFlags()=0;


	//Executes a per-character script command
	virtual bool ExecScriptCommand (int nCommand, void* pParams, void* pResult) {return false;}

	//! Set shader template to be used with character
  virtual bool SetShaderTemplateName(const char *TemplName, int Id, const char *ShaderName=0,IMatInfo *pCustomMaterial=0,unsigned nFlags = 0)=0;
  //! Get shader template 
  virtual const char * GetShaderTemplateName() = 0;
  //! Set refract coef. for refractive shader
  virtual void SetShaderFloat(const char *Name, float fVal, const char *ShaderName=NULL) = 0;
  //! Sets color parameter
  virtual void SetColor(const char *Name, float fR, float fG, float fB, float fA) = 0;

	//! Draw the character using a set of specified rendering parameters ( for outdoors )
  virtual void Draw(const SRendParams & RendParams)=0;	

	//! marks all LODs as needed to be reskinned
	virtual void ForceReskin () {}

	//! returns the leaf buffer materials in this character (as they are used in the renderer)
	virtual const list2<struct CMatInfo>*getLeafBufferMaterials() {return NULL;}

	//! Interface for the renderer - returns the CDLight describing the light in this character;
	//! returns NULL if there's no light with such index
	//! ICryCharInstance owns this light. This light may be destructed without notice upon next call to 
	//! any other function but GetLight(). Nobody may every modify it.
	virtual const class CDLight* GetBoundLight (int nIndex) = 0;

	//! Draw the character shadow volumes into the stencil buffer using a set of specified 
	//! rendering parameters ( for indoors )
	//! nLimitLOD is the LOD to use (if available), if the current character LOD is higher (<)
	virtual void RenderShadowVolumes(const SRendParams *rParams, int nLimitLOD = 0)=0;

	//! Draw the character without shaders for shadow mapping
  //virtual void DrawForShadow(const Vec3 & vTranslationPlus = Vec3(0,0,0))=0;
  
  //! Return dynamic bbox of object
  virtual void GetBBox(Vec3& Mins, Vec3& Maxs)=0;

	//! Return dynamic center of object
  virtual const Vec3 GetCenter()=0;
  //! Return dynamic radius of object
  virtual const float GetRadius()=0;

	// this is the handle that determines the object binding to a bone in the character
	typedef ULONG_PTR ObjectBindingHandle;
	// this is the invalid object binding
	enum {nInvalidObjectBindingHandle = 0};

  //! Attach object to bone (Return invalid handle if bone not found)
	//! Detaches all objects beforehand; if the object is NULL, just detaches all objects from the bone
	//! If the bone name is invalid, detaches everything to avoid using dangle pointers (in case the bone name was misspelled)
  virtual ObjectBindingHandle AttachObjectToBone(IBindable * pWeaponModel, const char * szBoneName, bool bUseRelativeToDefPoseMatrix = true, unsigned nFlags = 0)=0;   

	// detaches all objects from bones; returns the nubmer of bindings deleted
	virtual unsigned DetachAll() {return 0;}

	// detach all bindings to the given bone; returns the nubmer of bindings deleted
	virtual unsigned DetachAllFromBone(unsigned nBone) {return 0;}

	// attaches the object to the given bone. The returned value is the handle of the binding,
	// that can be used to examine the binding and delete it
	enum 
	{
		FLAGS_ATTACH_ZOFFSET = 1
	};
	virtual ObjectBindingHandle AttachToBone (IBindable*pObj, unsigned nBone, unsigned nFlags = 0) {return nInvalidObjectBindingHandle;}

	// detaches the given binding; returns true upon successful detach; if it returns false, the binding handle is invalid
	// the binding becomes invalid immediately after detach
	virtual bool Detach (ObjectBindingHandle nHandle) {return false;}

	// checks if the given binding is valid
	virtual bool IsBindingValid (ObjectBindingHandle nHandle) {return false;}

	// returns the number of bindings; valid until the next attach/detach operation
	virtual size_t GetBindingCount() {return 0;}

	// fills the given array with  GetBindingCount() pointers to IBindable
	virtual void EnumBindables(IBindable** pResult) {}

	typedef ULONG_PTR LightHandle;
	enum {InvalidLightHandle = 0};
	//! attach a light to a bone
	//! If bCopyLight is true, then this light will be copied and the copy will be managed by the CryCharInstance
	//! You can always get the pointer to it via GetLight, but don't you dare to delete it!
	virtual LightHandle AttachLight (CDLight* pDLight, unsigned nBone, bool bCopyLight = false) {return InvalidLightHandle;}
	//! detach the light from the bone
	virtual void DetachLight (CDLight* pDLight){}

	//! Attach a light (copying the light actually) to the bone
	//! Returns the handle identifying the light. With this handle, you can either
	//! Retrieve the light information or detach it.
	virtual LightHandle AttachLight (const CDLight& rDLight, const char* szBoneName){return InvalidLightHandle;}
	//! Detaches the light by the handle retuned by AttachLight
	virtual void DetachLight (LightHandle nHandle) {}
	//! Returns the light by the light handle; returns NULL if no such light found
	virtual CDLight* GetLight(LightHandle nHandle) {return NULL;}
	//! Returns the light handle if the light is attached; returns invalid handle, if this light is not attached
	//! NOTE: if your light was attached with copying, then you won't get the handle for the original light pointer
	//! because the original light might have been attached several times and have several pointers in this case
	virtual LightHandle GetLightHandle (CDLight* pLight) {return InvalidLightHandle;}

	//! Enables/Disables the Default Idle Animation restart.
	//! If this feature is enabled, then the last looped animation will be played back after the current (non-loop) animation is finished.
	//! Only those animations started with the flag bTreatAsDefaultIdleAnimation == true will be taken into account
	virtual void EnableLastIdleAnimationRestart (unsigned nLayer, bool bEnable = true) = 0;

	//! Start the specified animation with the given parameters
	virtual bool StartAnimation (const char* szAnimName, const struct CryCharAnimationParams& Params) {return false;}

	// FOR TEST ONLY enables/disables StartAnimation* calls; puts warning into the log if StartAnimation* is called while disabled
	virtual void EnableStartAnimation (bool bEnable) {}

	//! Start the specified by parameters morph target
	virtual void StartMorph (const char* szMorphTarget, const CryCharMorphParams& params) {}

	//! Start the specified by parameters morph target
	virtual void StartMorph (int nMorphTargetId, const CryCharMorphParams& params) {}

	//! Finds the morph with the given id and sets its relative time.
	//! Returns false if the operation can't be performed (no morph)
	//! The default implementation for objects that don't implement morph targets is to always fail
	virtual bool SetMorphTime (int nMorphTargetId, float fTime) {return false;}

	//! freezes all currently playing morphs at the point they're at
	virtual void FreezeAllMorphs() {}

	//! Stops the animation at the specified layer. Returns true if there was some animation on that layer, and false otherwise
	virtual bool StopAnimation (int nLayer) {return false;}

  //! Return current animation name ( Return 0 if animations stoped )
  virtual const char * GetCurAnimation() = 0;

	//! Returns the current animation in the layer or -1 if no animation is being played 
	//! in this layer (or if there's no such layer)
	virtual int GetCurrentAnimation (unsigned nLayer) = 0;
  
	//! Resets all animation layers ( stops all animations )
  virtual void ResetAnimations()=0;
  
	//! Set animations speed scale
	//! This is the scale factor that affects the animation speed of the character.
	//! All the animations are played with the constant real-time speed multiplied by this factor.
	//! So, 0 means still animations (stuck at some frame), 1 - normal, 2 - twice as fast, 0.5 - twice slower than normal.
  virtual void SetAnimationSpeed(float fSpeed) = 0;

  virtual float GetAnimationSpeed() = 0;

	//! Set morph speed scale
	//! Finds the morph target with the given id, sets its morphing speed and returns true;
	//! if there's no such morph target currently playing, returns false
	virtual bool SetMorphSpeed (int nMorphTargetId, float fSpeed) {return false;}

	//! Stops morph by target id
	virtual bool StopMorph (int nMorphTargetId) {return false;}

	//! Stops all morphs
	virtual void StopAllMorphs() {}

	//! This is the same as SetAnimationSpeed, but it sets the speed for layers
	//! NOTE: It does NOT override the overall animation speed, but it's multiplies it
	virtual void SetAnimationSpeed(int nLayer, float fSpeed) {}

	//! Enable object animation time update. If the bUpdate flag is false, subsequent calls to Update will not animate the character
	virtual void EnableTimeUpdate (bool bUpdate) = 0;

	//! Set the current time of the given layer, in seconds
	virtual void SetLayerTime (int nLayer, float fTimeSeconds) = 0;
	//! Return the current time of the given layer, in seconds
	virtual float GetLayerTime (int nLayer) = 0;

	enum UpdateEnum
	{
		flagDontUpdateBones = 1,
		flagDontUpdateAttachments = 1 << 1
	};

  //! Processes skining (call this function every frame to animate character)
	//! dwFlags	is a bitwise OR of one of the flags specified in the UpdateEnum enumeration
	virtual void Update (Vec3 vPos = Vec3(0,0,0), float fRadius=0, unsigned uFlags = 0) {}

	//! Updates the bones and the bounding box. Should be called if animation update
	//! cycle in EntityUpdate has already passed but you need the result of new animatmions
	//! started after Update right now.
	virtual void ForceUpdate() {}

	//! Synchronizes state with character physical animation; should be called after all updates (such as animation, game bones updates, etc.)
	virtual void UpdatePhysics( float fScale=1.0f )=0;

  //! IK (Used by physics engine)
	virtual void BuildPhysicalEntity(IPhysicalEntity *pent,float mass,int surface_idx,float stiffness_scale=1.0f,int nLod=0) = 0;
	virtual IPhysicalEntity *CreateCharacterPhysics(IPhysicalEntity *pHost, float mass,int surface_idx,float stiffness_scale, int nLod=0) = 0;
	virtual int CreateAuxilaryPhysics(IPhysicalEntity *pHost, int nLod=0) = 0;
	virtual IPhysicalEntity *GetCharacterPhysics() = 0;
	virtual IPhysicalEntity *GetCharacterPhysics(const char *pRootBoneName) = 0;
	virtual IPhysicalEntity *GetCharacterPhysics(int iAuxPhys) = 0;
  virtual void SynchronizeWithPhysicalEntity(IPhysicalEntity *pent, const Vec3& posMaster=Vec3(zero),const Quat& qMaster=Quat(1,0,0,0)) = 0;
	virtual IPhysicalEntity *RelinquishCharacterPhysics() = 0;
	virtual void DestroyCharacterPhysics(int iMode=0) = 0;
	virtual void SetCharacterPhysParams(float mass,int surface_idx) = 0;
  virtual void SetLimbIKGoal(int limbid, vectorf ptgoal=vectorf(1E10f,0,0), int ik_flags=0, float addlen=0, vectorf goal_normal=vectorf(zero)) = 0;
  virtual vectorf GetLimbEndPos(int limbid) = 0;
  virtual void AddImpact(int partid, vectorf point,vectorf impact) = 0;
	virtual int TranslatePartIdToDeadBody(int partid) = 0;
	virtual vectorf GetOffset() = 0;
	virtual void SetOffset(vectorf offset) = 0;

  //! Direct access to the specified  bone
  virtual ICryBone * GetBoneByName(const char * szName) = 0;

  //! Pose character bones
  virtual bool SetAnimationFrame(const char * szString, int nFrame)=0;

	//! Callback interface; <<TODO::>> THese should be in a separate interface
  
	
	//! Enables receiving OnStart/OnEnd of all animations from this character instance
	//! THe specified sink also receives the additional animation events specified through AddAnimationEvent interface
  virtual void AddAnimationEventSink(ICharInstanceSink * pCharInstanceSink) = 0;

	//! Counterpart to AddAnimationEventSink
	virtual void RemoveAnimationEventSink(ICharInstanceSink * pCharInstanceSink) = 0;

  //! Enables receiving OnStart/OnEnd of specified animation from this character instance
	//! The specified sink also receives the additional animation events specified through AddAnimationEvent interface for this animation
  virtual void AddAnimationEventSink(const char* szAnimName, ICharInstanceSink * pCharInstanceSink) = 0;

	//! Counterpart to the AddAnimationEventSink
  virtual void RemoveAnimationEventSink(const char* szAnimName, ICharInstanceSink * pCharInstanceSink) = 0;

	//! Adds an animation event; whenever the character plays the specified frame of the specified animation,
	//! it calls back the animation event sinkreceiving OnEvent notification of specified animation for all instances using this model
  virtual bool AddAnimationEvent(const char * szAnimName, int nFrameID, AnimSinkEventData pUserData) = 0;

  //! Deletes the animation event; from now on the sink won't be receiving the animation this event
	virtual bool RemoveAnimationEvent (const char* szAnimName, int nFrameID, AnimSinkEventData pUserData) = 0;

	//! Deletes all animation events
	virtual void RemoveAllAnimationEvents(){}

	//! Returns the model interface
	virtual ICryCharModel* GetModel() = 0;

  //! Return position of helper of the static object which is attached to animated object
  virtual Vec3 GetTPVWeaponHelper(const char * szHelperName, ObjectBindingHandle nHandle) = 0;

	//! Returns position of specified helper ( exported into cgf file )
	//! Actually returns the given bone's position
	//! Default implementation: 000
	virtual Vec3 GetHelperPos(const char * szHelperName) {return Vec3(0,0,0);}
	//! Returns the matrix of the specified helper ( exported into cgf file )
	//! Actually returns the given bone's matrix
	virtual const Matrix44 * GetHelperMatrixByName(const char * szHelperName) {return NULL;}

	//! Returns the matrix of the helper object (in matOut), like GetTPVWeaponHelper returns the position.
	//! When the return value is true, matOut contains the matrix of the helper object
	//! When the return value is false, operation failed - assume the matOut to be undefined
	virtual bool GetTPVWeaponHelperMatrix(const char * szHelperName, ObjectBindingHandle nHandle, Matrix44& matOut) {return false;}

  //! Set the twining type. If replace - animations of second layer will overwrite first, otherwise it will sum
  virtual void SetTwiningMode(AnimTwinMode eTwinMode = AnimTwinMode_Replace) = 0;

  //! Return damage zone id for specified bone id
  virtual int GetDamageTableValue (int nId) = 0;

  //! returns true if the character is playing any animation now (so bbox is changing)
  virtual bool IsCharacterActive() = 0;

  // temporary hack - this is used in XPlayer in CryGame only
	/*
	Vec3 m_vAngles;
	void		SetAngles( const Vec3& angles ) {m_vAngles = angles;}
	Vec3&	GetAngles( ) {return m_vAngles;}
	*/

  //! Spawn decal on the walls, static objects, terrain and entities
	//! The decal hit direction and source point are in the local coordinates of the chracter.
	virtual void CreateDecal(CryEngineDecalInfo& DecalLCS)=0;

	//! cleans up all decals in this character
	virtual void ClearDecals() {}

	//! Returns true if this character was created from the file the path refers to.
	//! If this is true, then there's no need to reload the character if you need to change its model to this one.
	virtual bool IsModelFileEqual (const char* szFileName) = 0;


	//int AddParticleEmitter(ParticleParams& rInfo, float fSpawnRate, bool bFacesUp)
	//{
	//	return AddParticleEmitter(rInfo, CryParticleSpawnInfo(fSpawnRate, bFacesUp?CryParticleSpawnInfo::FLAGS_RAIN_MODE:0));
	//}

	//! Sets up particle spawning. After this funtion is called, every subsequenc frame,
	//! During the character deformation, particles will be spawned in the given characteristics.
	//! The returned handle is to be used to stop particle spawning
	//! -1 means invalid handle value (couldn't add the particle spawn task, or not implemented)
	virtual int AddParticleEmitter(struct ParticleParams& rInfo, const struct CryParticleSpawnInfo& rSpawnInfo) {return -1;}

	//! Stops particle spawning started with StartParticleSpawn that returned the parameter
	//! Returns true if the particle spawn was stopped, or false if the handle is invalid
	//! -1 means remove all particle emitters
	virtual bool RemoveParticleEmitter (int nHandle) {return false;}

	//! sets the scale to the given vector (1,1,1 is the default)
	//! isn't compatible with physicalized objects, use only with weapons
	virtual void SetScale (const Vec3d& vScale) {}

	//! Pushes the underlying tree of objects into the given Sizer object for statistics gathering
	virtual void GetMemoryUsage(class ICrySizer* pSizer) const = 0;

	//! sets the given aniimation to the given layer as the default
	virtual void SetDefaultIdleAnimation(unsigned nLayer, const char* szAnimName) = 0;

	//! notifies the renderer that the character will soon be rendered
	virtual void PreloadResources ( float fDistance, float fTime, int nFlags) {}


	//! adds a submesh, returns handle to it which can be used to delete the submesh
	//! submesh is created either visible or invisible
	//! submesh creation/destruction is heavy operations, so the clients must use they rarely,
	//! and set visible/invisible when they need to turn them on/off
	//! But creating many submeshes is memory-consuming so the number of them must be kept low at all times
	virtual ICryCharSubmesh* NewSubmesh (ICryCharModel* pModel, bool bVisible = false) {return NULL;}
	// adds submesh to the specified slot; replaces submesh if there's some there
	virtual ICryCharSubmesh* NewSubmesh (unsigned nSlot, ICryCharModel* pModel, bool bVisible = false) {return NULL;}
	
	//! removes submesh from the character
	//! does not change slot assignments. That is, removing a submesh from slot 1, you don't shift models from upper slots
	//! you may NOT remove submesh from slot 0
	virtual void RemoveSubmesh (ICryCharSubmesh* pSubmesh){}
	virtual void RemoveSubmesh (unsigned nSlot) {}

	//! enumeration of submeshes
	//! returns the number of currently allocated submesh slots
	virtual size_t NumSubmeshes() {return 0;}
	//! returns submesh from the i-th slot. Some submesh slots may be empty (NULL)
	virtual ICryCharSubmesh* GetSubmesh(unsigned i){return NULL;}


	virtual ICryCharFxTrail* NewFxTrail (unsigned nSlot, const struct CryCharFxTrailParams&) {return NULL;}
	virtual void RemoveFxTrail(unsigned nSlot) {}


	// returns the pointer to this character instance, if it's a character instance
	virtual struct ICryCharInstance* GetICryCharInstance() {return this;}
};

struct ICryCharManager
{
	// these are the flags used by this interface methods
	enum
	{
		// Model keep in memory hints.
		// The model of a character can be locked in memory. This means, that even if the 
		// number of characters using this model drops to zero, it'll be kept in memory.
		// Such model is called Persistent.
		// A model that's deleted as soon as there are no characters referencing it is called Transient.
		// The default (when there's no hint) is defined by ca_KeepModels console variable.
		// If there are both hints, it's up to the character animation system to decide whether to keep it or not.
		nHintModelTransient  = 1, // unlock the body if it's locked
		nHintModelPersistent = 2, // lock the body

		// Temporary flag for AnimObject testing.
		nMakeAnimObject = 0x1000,
	};

	// this structure contains statistics about CryCharManager
	struct Statistics
	{
		// number of character instances
		unsigned numCharacters;
		// number of character models (CGF)
		unsigned numCharModels;
		// number of animobjects
		unsigned numAnimObjects;
		// number of animobject models
		unsigned numAnimObjectModels;
	};

	// returns statistics about this instance of character animation manager
	// don't call this too frequently
	virtual void GetStatistics(Statistics& rStats) = 0;

	// puts the size of the whole subsystem into this sizer object, classified,
	// according to the flags set in the sizer
	virtual void GetMemoryUsage(class ICrySizer* pSizer) const = 0;

	//! Load cgf and caf file and creates animated object.
  //! Returns pointer to already loaded object with same name if found.
  //! Reference counting used.
	//! Use the flags to give the manager hints about object persistense
  virtual ICryCharInstance * MakeCharacter(const char * cid_file_name, unsigned nFlags = 0)=0;

	// loads the character model (which is ref-counted, so you must assign it to an autopointer)
	// NOTE: may be not implemented in some code branches
	// Game01: not implemented
	// Game04: implemented fully
	virtual ICryCharModel* LoadModel(const char* szFileName, unsigned nFlags = 0) = 0;

  //! Reduces reference counter for object and deletes object if counter is 0
	//! Also accepts hints
  virtual void RemoveCharacter(ICryCharInstance * pCryCharInstance, unsigned nFlags = 0)=0;  

	//! cleans up all decals on all characters; equivalent to ExecScriptCommand(CASCMD_CLEAR_DECALS)
	virtual void ClearDecals() = 0;

	//! Cleans up all resources - currently deletes all bodies and characters (even if there are references on them)
	//! if bDependantOnly is false, even resources that aren't dependant on the other susystems will be cleared
	virtual void ClearResources(bool bDependantOnly = true) = 0;

	//! Executes a script command
	//! Returns true if the command was executed, or false if not
	//! All the possible values for nCommand are in the CryAnimationScriptCommands.h
	//! file in the CryAnimationScriptCommandEnum enumeration. All the parameter/result
	//! structures are also there.
	virtual bool ExecScriptCommand (int nCommand, void* pParams = NULL, void* pResult = NULL) {return false;}

	//! This is normally very fast routine, should be called every frame.
	virtual void Update() = 0;

	//! The specified animation will be unloaded from memory; it will be loaded back upon the first invokation (via StartAnimation())
	virtual void UnloadAnimation(const char* szFileName) = 0;


	//! Starts loading the specified animation. fWhenRequired is the timeout, in seconds, from the current moment,
	//! when the animation data will actually be needed
	virtual void StartLoadAnimation (const char* szFileName, float fWhenRequired) = 0;

		//! Delete itself
  virtual void Release()=0;

	//! Locks all models in memory
	virtual void LockResources() = 0;

	//! Unlocks all models in memory
	virtual void UnlockResources() = 0;
};

///////////////////////////////////////////////////////////////////////////////////////////////////////////////
#ifdef __cplusplus
extern "C" {
#endif

#if (defined(GAMECUBE) || defined (PS2))
#define __TIMESTAMP__ "Ver1.0"
#endif

//#ifdef PS2
//#define __TIMESTAMP__ "Ver1.0"
//#endif

// expirimental way to track interface version 
// this value will be compared with value passed from system module
const char gAnimInterfaceVersion[64] = __TIMESTAMP__;

// CreateCryAnimation function type definition
typedef ICryCharManager * (*PFNCREATECRYANIMATION)(ISystem	* pSystem,const char * szInterfaceVersion);

//! Creates 3dengine instance
CRYANIMATION_API ICryCharManager * CreateCharManager(ISystem	* pSystem,const char * szInterfaceVersion=gAnimInterfaceVersion);

#ifdef __cplusplus
}
#endif
///////////////////////////////////////////////////////////////////////////////////////////////////////////////

#endif // ICRY_ANIMATION
