#ifndef _CRY_SYSTEM_H_
#define _CRY_SYSTEM_H_

#ifdef WIN32
	#ifdef CRYSYSTEM_EXPORTS
		#define CRYSYSTEM_API __declspec(dllexport)
	#else
		#define CRYSYSTEM_API __declspec(dllimport)
	#endif
#else
	#define CRYSYSTEM_API
#endif

#include "platform.h" // Needed for LARGE_INTEGER (for consoles).

////////////////////////////////////////////////////////////////////////////////////////////////
// Forward declarations
////////////////////////////////////////////////////////////////////////////////////////////////
#include <IXMLDOM.h>
#include <IXml.h>

struct ILog;
struct IEntitySystem;
struct IEntity;
struct ICryPak;
struct IKeyboard;
struct IMouse;
struct IConsole;
struct IInput;
struct IRenderer;
struct IConsole;
struct IProcess;
struct I3DEngine;
struct ITimer;
struct IGame;
struct IScriptSystem;
struct IAISystem;
struct IFlash;
struct INetwork;
struct ICryFont;
struct IMovieSystem;
class IPhysicalWorld;
struct IMemoryManager;
struct ISoundSystem;
struct IMusicSystem;
struct XDOM::IXMLDOMDocument;
struct FrameProfileSystem;
struct FrameProfiler;
class IStreamEngine;


#define PROC_MENU		1
#define PROC_3DENGINE	2

//ID for script userdata typing (maybe they should be moved into the game.dll)
#define USER_DATA_SOUND			1
#define USER_DATA_TEXTURE		2
#define USER_DATA_OBJECT		3
#define USER_DATA_LIGHT			4

enum ESystemUpdateFlags
{
	ESYSUPDATE_IGNORE_AI			= 0x0001,
	ESYSUPDATE_IGNORE_PHYSICS = 0x0002,
	//! Special update mode for editor.
	ESYSUPDATE_EDITOR					=	0x0004,
};

//! User defined callback, which can be passed to ISystem.
struct ISystemUserCallback
{
	/** Signals to User that engine error occured.
			@return true to Halt execution or false to ignore this error.
	*/
	virtual bool OnError( const char *szErrorString ) = 0;
	/** If working in Editor environment notify user that engine want to Save current document.
			This happens if critical error have occured and engine gives a user way to save data and not lose it
			due to crash.
	*/
	virtual void OnSaveDocument() = 0;
	
	/** Notify user that system wants to switch out of current process.
			(For ex. Called when pressing ESC in game mode to go to Menu).
	*/
	virtual void OnProcessSwitch() = 0;
};

//! Structure passed to Init method of ISystem interface.
struct SSystemInitParams
{
	void *hInstance;
	void *hWnd;
	const char *sCmdLine;
	ISystemUserCallback *pUserCallback;
	//! You can specify your own ILog to be used by System.
	ILog *pLog;
	//! File name to use for log.
	const char* sLogFileName;
	//! When runing in Editor mode.
	bool bEditor;
	//! When runing in Preview mode (Minimal initialization).
	bool bPreview;
	//! When runing in Automated testing mode.
	bool bTestMode;

	//! Initialization defaults.
	SSystemInitParams()
	{
		hInstance = 0;
		hWnd = 0;
		sCmdLine = 0;
		pLog = 0;
		pUserCallback = 0;
		sLogFileName = 0;
		bEditor = false;
		bPreview = false;
		bTestMode = false;
	}
};

//! Structure passed to CreateGame method of ISystem interface.
struct SGameInitParams
{
	//! Name of Game DLL. (Win32 Only)
	const char *sGameDLL;
	//! Skip auto loading of level and mission.
	bool bSkipAutoLoad;
	//! Pointer to already created game interface.
	IGame* pGame;

	SGameInitParams()
	{
		sGameDLL = NULL;
		pGame = NULL;
		bSkipAutoLoad = false;
	}
};


////////////////////////////////////////////////////////////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////////////////////
//! Main Engine Interface
//! initialize and dispatch all engine's subsystems 
struct ISystem
{ 
	//! Loads GameDLL and creates game instance.
	virtual bool CreateGame( const SGameInitParams &params ) = 0;
	
	//! Release ISystem.
	virtual void Release() = 0;

	//! Upadate all subsystems
	//! @param flags one or more flags from ESystemUpdateFlags sructure.
	virtual bool Update( int updateFlags=0,bool bMultiplayer=false ) = 0;

	//! Begin rendering frame.
	virtual void	RenderBegin() = 0;
	//! Render subsystems.
	virtual void	Render(bool bMultiplayer) = 0;
	//! End rendering frame and swap back buffer.
	virtual void	RenderEnd() = 0;

	//! Renders the statistics; this is called from RenderEnd, but if the 
	//! Host application (Editor) doesn't employ the Render cycle in ISystem,
	//! it may call this method to render the essencial statistics
	virtual void RenderStatistics () = 0;

	//! dumps the memory usage statistics to the log
	virtual void DumpMemoryUsageStatistics() = 0;

	//! Quit the appliacation
	virtual void	Quit() = 0;
	//! return true if the application is in the shutdown phase
	virtual bool	IsQuitting() = 0;

	//! Display error message.
	//! Logs it to console and file and error message box.
	//! Then terminates execution.
	virtual void Error( const char *sFormat,... ) = 0;
	//! Display warning message.
	//! Logs it to console and file and display a warning message box.
	//! Not terminates execution.
	virtual void Warning( const char *sFormat,... ) = 0;

	//! return the related subsystem interface
	//@{
	virtual IGame						*GetIGame() = 0;
	virtual INetwork				*GetINetwork() = 0;
	virtual IRenderer				*GetIRenderer() = 0;
	virtual IInput					*GetIInput() = 0;
	virtual ITimer					*GetITimer() = 0;
	virtual IConsole				*GetIConsole() = 0;
	virtual IScriptSystem		*GetIScriptSystem() = 0;
	virtual I3DEngine				*GetI3DEngine() = 0;
	virtual ISoundSystem		*GetISoundSystem() = 0;
	virtual IMusicSystem		*GetIMusicSystem() = 0;
  virtual IPhysicalWorld	*GetIPhysicalWorld() = 0;
	virtual IMovieSystem		*GetIMovieSystem() = 0;
	virtual IAISystem				*GetAISystem() = 0;
	virtual IMemoryManager	*GetIMemoryManager() = 0;
	virtual IEntitySystem		*GetIEntitySystem() = 0;
	virtual ICryFont				*GetICryFont()	= 0;
	virtual ICryPak				  *GetIPak()	= 0;
	virtual ILog						*GetILog() = 0;
	virtual IStreamEngine   *GetStreamEngine() = 0;
	//@}

	virtual void DebugStats(bool checkpoint, bool leaks) = 0;
	virtual void DumpWinHeaps() = 0;
	virtual int DumpMMStats(bool log) = 0;
	
	virtual XDOM::IXMLDOMDocument *CreateXMLDocument() = 0;

	//////////////////////////////////////////////////////////////////////////
	// IXmlNode interface.
	//////////////////////////////////////////////////////////////////////////
	//! Creates new xml node.
	virtual XmlNodeRef CreateXmlNode( const char *sNodeName="" ) = 0;
	//! Load xml file, return 0 if load failed.
	virtual XmlNodeRef LoadXmlFile( const char *sFilename ) = 0;

	//! config file read write
	//@{
  virtual char *GetIniVar( const char * szVarName, const char * szFileName, const char * def_val )=0;
  virtual float GetIniVar( const char * szVarName, const char * szFileName, float def_val )=0;
  virtual float*GetIniVar( const char * szVarName, const char * szFileName, float def_val[4] )=0;
  virtual void StoreIniVar( const char * pValue,    const char * szVarName, const char * szFileName )=0;
  virtual void StoreIniVar( float fValue,           const char * szVarName, const char * szFileName )=0;
  virtual void StoreIniVar( int nValue,             const char * szVarName, const char * szFileName )=0;
	//@}

	virtual void SetViewCamera(class CCamera &Camera) = 0;
	virtual CCamera& GetViewCamera() = 0;

	virtual void CreateEntityScriptBinding(IEntity *pEntity)=0;
	//! When ignore update sets to true, system will ignore and updates and render calls.
	virtual void IgnoreUpdates( bool bIgnore ) = 0;

	//! Set rate of Garbage Collection for script system.
	virtual void SetGCFrequency( int rate ) = 0;

	//! Load music-definitions from script and pass it to the music-system.
	virtual bool LoadMusicDataFromLUA(const char *pszFilename) = 0;

	/*! Set the active process
		@param process a pointer to a class that implement the IProcess interface
	*/
	virtual void SetIProcess(IProcess *process) = 0;
	/*! Get the active process
		@return a pointer to the current active process
	*/
	virtual IProcess* GetIProcess() = 0;

#if defined (WIN32) || defined (PS2)
	virtual IRenderer* CreateRenderer(bool fullscreen, void* hinst, void* hWndAttach = 0) = 0;
#endif	

	//! Returns true if system running in Test mode.
	virtual bool IsTestMode() const = 0;

	virtual void ShowDebugger(const char *pszSourceFile, int iLine, const char *pszReason) = 0;
	
	//! Frame profiler functions
	virtual void SetFrameProfiler(bool on) = 0;
	virtual FrameProfiler *StartTime( LARGE_INTEGER &time, FrameProfiler *pThis ) = 0;
	virtual void ContributeTime( LARGE_INTEGER &time, char *name, FrameProfiler *pParent) = 0;

	//! Compressed file read & write
	virtual bool WriteCompressedFile(char *filename, void *data, unsigned int bitlen) = 0;
	virtual unsigned int ReadCompressedFile(char *filename, void *data, unsigned int maxbitlen) = 0;
	virtual unsigned int GetCompressedFileSize(char *filename)=0;
};

//////////////////////////////////////////////////////////////////////////
// CrySystem DLL Exports.
//////////////////////////////////////////////////////////////////////////
typedef ISystem* (*PFNCREATESYSTEMINTERFACE)( SSystemInitParams &initParams );

// interface of the DLL
extern "C"
{
	CRYSYSTEM_API ISystem* CreateSystemInterface( SSystemInitParams &initParams );

	//! Display error message.
	//! Logs it to console and file and error message box.
	//! Then terminates execution.
	CRYSYSTEM_API void CryError( const char *sFormat,... );
	//! Display warning message.
	//! Logs it to console and file and display a warning message box.
	//! Not terminates execution.
	CRYSYSTEM_API void CryWarning( const char *sFormat,... );

	//! Simple log of data.
	CRYSYSTEM_API void CryLog( const char *sFormat,... );
}

#endif _CRY_SYSTEM_H_