#include "stdafx.h"
#include "FarCryAutoCDApp.h"
#include "BackBuffer.h"
#include "Jukebox.h"
#include "MenuData.h"
#include "Utilities.h"
#include "FarCryRegistry.h"
#include "resource.h"

#include <process.h>



#define FRAMELESS_WND



const unsigned int c_uiFadeScreenTimerID( 0 );
const unsigned int c_uiFadeTimerScreenInterval( 50 );

const unsigned char c_ucDefTransparency( 220 );



using namespace std;



static LRESULT CALLBACK
GlobalWndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	// delegate to wnd class
	return( CFarCryAutoCDApp::GetInstance().WndProc( hWnd, message, wParam, lParam ) );
}



CFarCryAutoCDApp::CFarCryAutoCDApp()
: m_hInstance( 0 )
, m_hWnd( 0 )
, m_pkBackBuffer( 0 )
, m_pkJukebox( 0 )
, m_pkMenuData( 0 )
, m_ucCurButtonID( BID_NOBUTTON )
, m_iLastDragMouseScreenX( 0 )
, m_iLastDragMouseScreenY( 0 )
, m_bLeftMouseButtonPressed( false )
, m_bDragWindow( false )
, m_bScreenIsFading( false )
, m_ucFadeScreenCounter( 0 )
{	
}



CFarCryAutoCDApp::~CFarCryAutoCDApp()
{
	delete m_pkBackBuffer;
	delete m_pkJukebox;
	delete m_pkMenuData;

	// dump leaks here... be aware of stl container members and the existance of other static object 
	// instances (their destruction order can't be determined) which will be reported as "pseudo" leaks
	//_CrtDumpMemoryLeaks();
}



CFarCryAutoCDApp& 
CFarCryAutoCDApp::GetInstance()
{
	// returns the only instance of this class
	static CFarCryAutoCDApp s_kInstance;
	return( s_kInstance );
}



ATOM 
CFarCryAutoCDApp::RegisterClass()  const
{
	assert( 0 != m_hInstance );

	// get class name for resource
	tstring strClassName( GetStringFromResource( m_hInstance, IDC_FARCRYAUTOCD ) );
		
	// initialize class structure
	WNDCLASSEX sWndClassEx;
	sWndClassEx.cbSize			= sizeof( WNDCLASSEX ); 
	sWndClassEx.style			= CS_HREDRAW | CS_VREDRAW;
	sWndClassEx.lpfnWndProc		= (WNDPROC) GlobalWndProc;
	sWndClassEx.cbClsExtra		= 0;
	sWndClassEx.cbWndExtra		= 0;
	sWndClassEx.hInstance		= m_hInstance;
	sWndClassEx.hIcon			= LoadIcon( m_hInstance, (LPCTSTR) IDI_FARCRYAUTOCD );
	sWndClassEx.hCursor			= LoadCursor( 0, IDC_ARROW );
	sWndClassEx.hbrBackground	= (HBRUSH) ( COLOR_WINDOW + 1 );
	sWndClassEx.lpszMenuName	= (LPCTSTR) IDC_FARCRYAUTOCD;
	sWndClassEx.lpszClassName	= strClassName.c_str();
	sWndClassEx.hIconSm			= LoadIcon( sWndClassEx.hInstance, (LPCTSTR) IDI_FARCRYAUTOCD );

	// register class
	return( RegisterClassEx( &sWndClassEx ) );
}



void
CFarCryAutoCDApp::PrepareTransparency()
{
	// get access to "SetLayeredWindowAttributes" function
	HMODULE hUser32( LoadLibrary( _T( "user32.dll" ) ) );
	if( 0 != hUser32 )
	{
		typedef DWORD (WINAPI* PFSetLayeredWindowAttributes) ( HWND, DWORD, BYTE, DWORD );
		PFSetLayeredWindowAttributes pFunc( (PFSetLayeredWindowAttributes) GetProcAddress( hUser32, "SetLayeredWindowAttributes" ) );

		// set default transparency if functionality is available
		if( 0 != pFunc ) 
		{
			SetWindowLong( m_hWnd, GWL_EXSTYLE , GetWindowLong( m_hWnd , GWL_EXSTYLE ) | WS_EX_LAYERED );
			pFunc( m_hWnd, RGB( 255, 255, 255 ), c_ucDefTransparency, LWA_COLORKEY | LWA_ALPHA );
		}

		FreeLibrary( hUser32 );
	}
}



void
CFarCryAutoCDApp::InitInstance( const HINSTANCE& hInstance, int nCmdShow )
{
	// set instance handle
	m_hInstance = hInstance;

#ifdef FRAMELESS_WND
    DWORD dwWndStyle( WS_POPUP | /*WS_SYSMENU |*/ WS_MINIMIZEBOX );
#else
    DWORD dwWndStyle( WS_OVERLAPPEDWINDOW & ~WS_MAXIMIZEBOX );
#endif // #ifdef FRAMELESS_WND

	// try to create window
	if( 0 == ( m_hWnd = CreateWindow( (LPCTSTR) RegisterClass(), 
									  GetStringFromResource( m_hInstance, IDS_APP_TITLE ).c_str(), 
                                      dwWndStyle,
									  ( GetSystemMetrics( SM_CXSCREEN ) - c_iWndWidth ) / 2, 
									  ( GetSystemMetrics( SM_CYSCREEN ) - c_iWndHeight ) / 2, 
									  c_iWndWidth, c_iWndHeight, 0, 0, m_hInstance, 0 ) ) )
	{
		throw( texception( GetStringFromResource( m_hInstance, IDS_FAILEDTOCREATEWINDOW ) ) );
	}

#ifndef FRAMELESS_WND
	// adjust window size and position
	RECT rectWnd;
	GetWindowRect( m_hWnd, &rectWnd );

	RECT rectClient;
	GetClientRect( m_hWnd, &rectClient );

	int iNewWidth( c_iWndWidth + ( rectWnd.right - rectWnd.left ) - ( rectClient.right - rectClient.left ) );
	int iNewHeight( c_iWndHeight + ( rectWnd.bottom - rectWnd.top ) - ( rectClient.bottom - rectClient.top ) );

	MoveWindow( m_hWnd,
				( GetSystemMetrics( SM_CXSCREEN ) - iNewWidth  ) >> 1, 
				( GetSystemMetrics( SM_CYSCREEN ) - iNewHeight ) >> 1, 
				iNewWidth, iNewHeight, TRUE );
#endif // #ifndef FRAMELESS_WND

	// initialize back buffer
	m_pkBackBuffer = new CBackBuffer( m_hInstance, m_hWnd );

	// initialize sound system
	m_pkJukebox = new CJukebox( m_hInstance );

	// initialize menu data
	m_pkMenuData = new CMenuData( m_hInstance, m_pkBackBuffer );

	// draw inital state of window
	OnDraw();

	// present window
	//PrepareTransparency();
	ShowWindow( m_hWnd, nCmdShow );
	UpdateWindow( m_hWnd );

	// start playing background music
	m_pkJukebox->PlaySound( CJukebox::SND_BACKGROUND_MUSIC );
}



void 
CFarCryAutoCDApp::SetCursorSafe( const TCHAR* pcCursorName )
{
	HCURSOR hCursor( LoadCursor( 0, pcCursorName ) );
	if( 0 == hCursor )
	{
		// safety net... IDC_ARROW should always be available
		hCursor = LoadCursor( 0, IDC_ARROW );
		assert( 0 != hCursor );
	}
	SetCursor( hCursor );
}



int 
CFarCryAutoCDApp::Run()
{	
	int iRet;

	// standard message loop
	MSG sMessage;
	while( 0 != ( iRet = GetMessage( &sMessage, 0, 0, 0 ) ) )
	{ 
		if( -1 == iRet )
		{
			return( -1 );
		}
		else
		{
			if( FALSE != PeekMessage( &sMessage, 0, 0, 0, PM_REMOVE ) )
			{
				if( WM_QUIT == sMessage.message ) 
				{
					break;
				}
			}

			TranslateMessage( &sMessage ); 
			DispatchMessage( &sMessage ); 
		}
	}
	return( (int) sMessage.wParam );
}



void 
CFarCryAutoCDApp::HandleSysMenu()
{
	// handle system menu button clicks
	switch( m_ucCurButtonID )
	{
    case BID_SYSMENU_MINIMIZE:
        {
            ShowWindow( m_hWnd, SW_MINIMIZE );
            break;
        }
    case BID_SYSMENU_QUIT:
        {
			m_ucCurButtonID = BID_NOBUTTON;
            SendMessage( m_hWnd, WM_CLOSE, 0, 0 );
            break;
        }
    default:
        {
            assert( 0 );
            break;
        }
    }
}



void 
CFarCryAutoCDApp::ClientToScreen(  int iClientX, int iClientY, int& iScreenX, int& iScreenY  ) const
{
    POINT p;
    p.x = iClientX;
    p.y = iClientY;
    ::ClientToScreen( m_hWnd, &p );
    iScreenX = p.x;
    iScreenY = p.y;
}



void 
CFarCryAutoCDApp::FadeScreen()
{
	assert( 0 != m_pkBackBuffer );
	
	// cheap fade down effect using bit shifts and masks
	unsigned int* puiBackBufferBits( m_pkBackBuffer->GetPtr() );
	for( unsigned int uiIndex( 0 ); uiIndex < c_iWndWidth * c_iWndHeight; ++uiIndex )
	{
		puiBackBufferBits[ uiIndex ] = ( puiBackBufferBits[ uiIndex ] >> 1 ) & 0x7F7F7F7F;
	}

	InvalidateRect( m_hWnd, 0, FALSE );
}



void 
CFarCryAutoCDApp::OnPaint()
{
	// prepare paint struct
	PAINTSTRUCT sPS;	
	HDC hDC( BeginPaint( m_hWnd, &sPS ) );
	assert( 0 != hDC );

	// blit from back buffer to window
	BOOL bRet( BitBlt( hDC, 0, 0, c_iWndWidth, c_iWndHeight, m_pkBackBuffer->GetDC(), 0, 0, SRCCOPY ) );
	assert( FALSE != bRet );

	// mark end of window paining
	EndPaint( m_hWnd, &sPS );
}



void 
CFarCryAutoCDApp::OnDraw()
{
	assert( 0 != m_pkBackBuffer );

	// get access to back buffer
	HDC hBackBufferDC( m_pkBackBuffer->GetDC() );
	unsigned int* puiBackBufferBits( m_pkBackBuffer->GetPtr() );

	// draw window content
	const unsigned int* const pImageData[ 2 ] =
	{
		&m_pkMenuData->GetMenuBase()[ 0 ],
		( false == m_bLeftMouseButtonPressed ) ? ( &m_pkMenuData->GetMenuHighlighted()[ 0 ] ) : ( &m_pkMenuData->GetMenuSelected()[ 0 ] )
	};

	const Raw8BitImage& imgMenuMask( m_pkMenuData->GetMenuMask() );
	for( unsigned int uiIndex( 0 ); uiIndex < c_iWndWidth * c_iWndHeight; ++uiIndex )
	{
		// draw pixel from either base image or highligh/selected image (in case button ID at current mouse point == pixel's mask)
		puiBackBufferBits[ uiIndex ] = pImageData[ imgMenuMask[ uiIndex ] == m_ucCurButtonID ][ uiIndex ];
	}

	// draw frame using GDI functions
	MoveToEx( hBackBufferDC, 0, 0, 0 );
	LineTo( hBackBufferDC, c_iWndWidth - 1, 0 );
	LineTo( hBackBufferDC, c_iWndWidth - 1, c_iWndHeight - 1 );
	LineTo( hBackBufferDC, 0, c_iWndHeight - 1 );
	LineTo( hBackBufferDC, 0, 0 );

	// enforce WM_PAINT
	InvalidateRect( m_hWnd, 0, FALSE );
}




void 
CFarCryAutoCDApp::OnKeyDown( unsigned int uiChar, unsigned int uiRepCnt, unsigned int uiFlags )
{
	switch( uiChar )
	{
	case VK_ESCAPE:
		{
			m_ucCurButtonID = BID_NOBUTTON;
			SendMessage( m_hWnd, WM_CLOSE, 0, 0 );
		}
	default:
		{
		}
	}
}



void 
CFarCryAutoCDApp::OnMouseMove( int iNewX, int iNewY, unsigned int uiFlags )
{
	if( false == m_bScreenIsFading )
	{	
		if( false != m_bLeftMouseButtonPressed && 0 == ( uiFlags & MK_LBUTTON ) )
		{
			// mouse button was pressed but got release w/o notice of OnLButtonUp( ... )
			m_bLeftMouseButtonPressed = false;
			OnDraw();
		}

        if( false != m_bLeftMouseButtonPressed && false != m_bDragWindow )
        {
			// handle dragging of window
            RECT rectWnd;
            GetWindowRect( m_hWnd, &rectWnd );

            int iCurDragMouseScreenX( 0 );
            int iCurDragMouseScreenY( 0 );
            ClientToScreen( iNewX, iNewY, iCurDragMouseScreenX, iCurDragMouseScreenY );

            MoveWindow( m_hWnd, 
                        rectWnd.left + ( iCurDragMouseScreenX - m_iLastDragMouseScreenX ), 
                        rectWnd.top + ( iCurDragMouseScreenY - m_iLastDragMouseScreenY ), 
                        rectWnd.right - rectWnd.left,
                        rectWnd.bottom - rectWnd.top, TRUE );

            m_iLastDragMouseScreenX = iCurDragMouseScreenX;
	        m_iLastDragMouseScreenY = iCurDragMouseScreenY;

            SetCursorSafe( IDC_SIZEALL );
        }
        else
        {
			// update current button ID as a result of the mouse move
    		unsigned char ucNewButtonID( m_pkMenuData->GetMenuMask()[ GetOffset( iNewX, iNewY ) ] );			
		    if( m_ucCurButtonID != ucNewButtonID )
		    {
			    m_ucCurButtonID = ucNewButtonID;

			    if( BID_NOBUTTON == m_ucCurButtonID  )
			    {
				    //m_pkJukebox->PlaySound( CJukebox::SND_BUTTON_LEFTUNTOUCHED );
				    m_bLeftMouseButtonPressed = false;
			    }
			    else
			    {
					if( m_ucCurButtonID < __SYS_MENU_START )
					{
						m_pkJukebox->PlaySound( CJukebox::SND_BUTTON_HOVERING );
					}			
			    }

			    OnDraw();
		    }
            
			// set mouse cursor accordingly
		    if( BID_NOBUTTON != m_ucCurButtonID )
		    {
			    SetCursorSafe( IDC_HAND );
		    }
		    else
		    {
			    SetCursorSafe( IDC_ARROW );
		    }
        }
	}
}



void
CFarCryAutoCDApp::OnLButtonDown( int iNewX, int iNewY )
{
	if( false == m_bScreenIsFading )
	{
		if( BID_NOBUTTON == m_ucCurButtonID )
		{
			// initiate window dragging
            SetCursorSafe( IDC_SIZEALL );
            ClientToScreen( iNewX, iNewY, m_iLastDragMouseScreenX, m_iLastDragMouseScreenY );
            SetCapture( m_hWnd );
            m_bDragWindow = true;
		}
        else
        {
			if( m_ucCurButtonID < __SYS_MENU_START )
			{
				m_pkJukebox->PlaySound( CJukebox::SND_BUTTON_PRESSED );
			}			
        }

		// mark LMB as pressed and redraw window content
		m_bLeftMouseButtonPressed = true;
        OnDraw();
	}
}



void
CFarCryAutoCDApp::OnLButtonUp()
{
	if( false == m_bScreenIsFading )
	{
		const TCHAR* strCursorID( IDC_ARROW );

		if( BID_NOBUTTON != m_ucCurButtonID )
		{
			// handle button events
            if( m_ucCurButtonID < __SYS_MENU_START )
            {
				switch( m_ucCurButtonID )
				{
				case BID_INSTALL:
				case BID_UNINSTALL:
				case BID_PLAY:
				case BID_QUIT:
					{
						SendMessage( m_hWnd, WM_CLOSE, 0, 0 );
						break;
					}
				default:
					{
						RunButtonEvent();
						strCursorID = IDC_HAND;
						break;
					}
				}								
			}
            else
            {
                HandleSysMenu();
            }
            //m_pkJukebox->PlaySound( CJukebox::SND_BUTTON_RELEASED );
		}

        if( false != m_bDragWindow )
        {
			// leave window dragging mode
            ReleaseCapture();
            m_bDragWindow = false;
        }        
        
		// mark LMB as not-pressed and redraw window content
		m_bLeftMouseButtonPressed = false;
		SetCursorSafe( strCursorID );
		OnDraw();
	}
}



void
CFarCryAutoCDApp::OnTimer( unsigned int uiTimerID )
{
	switch( uiTimerID )
	{
	case c_uiFadeScreenTimerID:
		{
			if( m_ucFadeScreenCounter < 8 )
			{
				// still need to update screen fade effect
				FadeScreen();
				++m_ucFadeScreenCounter;

				// fade out sound as well
				//m_pkJukebox->SetVolume( ( 8 - (int) m_ucFadeScreenCounter ) * 255 / 8 );
			}
			else
			{
				// screen was faded down. kill timer execute button event and quit
				KillTimer( m_hWnd, c_uiFadeScreenTimerID );
				RunButtonEvent();
				PostQuitMessage( 0 );
			}
			break;
		}
	default:
		{
			break;
		}
	}
}



void
CFarCryAutoCDApp::OnClose()
{
	m_bScreenIsFading = true;
	SetTimer( m_hWnd, c_uiFadeScreenTimerID, c_uiFadeTimerScreenInterval, 0 );
}



void
CFarCryAutoCDApp::RunButtonEvent()
{
	// perform action assigned to clicked button
	switch( m_ucCurButtonID )
	{
	case BID_INSTALL:
		{
			tstring strInstallCmd;
			if( false != FarCryRegistry::GetInstallCmd( strInstallCmd ) )
			{
				try
				{
#ifdef UNICODE
					_wspawnl( _P_NOWAIT, strInstallCmd.c_str(), strInstallCmd.c_str(), 0 );					
#else
					_spawnl( _P_NOWAIT, strInstallCmd.c_str(), strInstallCmd.c_str(), 0 );					
#endif
				}
				catch( ... )
				{
				}			
			}
			break;
		}
	case BID_PLAY:
		{
			tstring strLaunchCmd;
			tstring strLaunchWorkingDir;
			if( false != FarCryRegistry::GetLaunchCmd( strLaunchCmd, strLaunchWorkingDir ) )
			{
				try
				{
					SetCurrentDirectory( strLaunchWorkingDir.c_str() );
#ifdef UNICODE
					_wspawnl( _P_NOWAIT, strLaunchCmd.c_str(), strLaunchCmd.c_str(), 0 );					
#else
					_spawnl( _P_NOWAIT, strLaunchCmd.c_str(), strLaunchCmd.c_str(), 0 );					
#endif
				}
				catch( ... )
				{
				}			
			}
			break;
		}
	case BID_UNINSTALL:
		{
			tstring strUninstallCmd; 
			tstring strUninstallParam1;
			if( false != FarCryRegistry::GetUninstallCmd( strUninstallCmd, strUninstallParam1 ) )
			{
				try
				{
#ifdef UNICODE
					_wspawnl( _P_NOWAIT, strUninstallCmd.c_str(), strUninstallCmd.c_str(), strUninstallParam1.c_str(), 0 );					
#else
					_spawnl( _P_NOWAIT, strUninstallCmd.c_str(), strUninstallCmd.c_str(), strUninstallParam1.c_str(), 0 );					
#endif
				}
				catch( ... )
				{
				}			
			}
			break;
		}
	case BID_README:
		{
			tstring strReadmePath;
			if( false != FarCryRegistry::GetReadmePath( strReadmePath ) )
			{
				ShellExecute( 0, 0, strReadmePath.c_str(), 0, 0, SW_SHOW );
			}
			break;
		}
	case BID_MANUAL:
		{
			tstring strManualPath;
			if( false != FarCryRegistry::GetManualPath( strManualPath ) )
			{
				ShellExecute( 0, 0, strManualPath.c_str(), 0, 0, SW_SHOW );
			}
			break;
		}
	case BID_REGISTER:
		{
			tstring strRegisterCmd; 
			tstring strRegisterParam1; 
			tstring strRegisterParam2;
			tstring strRegisterWorkingDir;
			if( false != FarCryRegistry::GetRegisterCmd( strRegisterCmd, strRegisterParam1, strRegisterParam2, strRegisterWorkingDir ) )
			{
				try
				{
					SetCurrentDirectory( strRegisterWorkingDir.c_str() );
#ifdef UNICODE
					_wspawnl( _P_NOWAIT, strRegisterCmd.c_str(), strRegisterCmd.c_str(), strRegisterParam1.c_str(), strRegisterParam2.c_str(), 0 );					
#else
					_spawnl( _P_NOWAIT, strRegisterCmd.c_str(), strRegisterCmd.c_str(), strRegisterParam1.c_str(), strRegisterParam2.c_str(), 0 );
#endif
				}
				catch( ... )
				{
				}			
			}
			break;
		}
	case BID_WEB:
		{
			tstring strWeblinkPath;
			if( false != FarCryRegistry::GetWebLinkPath( strWeblinkPath ) )
			{
				ShellExecute( 0, 0, strWeblinkPath.c_str(), 0, 0, SW_SHOW );
			}
			break;
		}
	case BID_QUIT:
	default:
		{
			break;
		}
	}
}



void 
CFarCryAutoCDApp::OnSize( unsigned int uiType, unsigned short usWidth, unsigned short usHeight )
{
	if( SIZE_RESTORED == uiType && 0 != m_pkBackBuffer )
	{
		// reset internal states and redraw window content when window is restored from minimized state
		m_bLeftMouseButtonPressed = false;
		m_ucCurButtonID = BID_NOBUTTON;
		OnDraw();
	}
}



LRESULT 
CFarCryAutoCDApp::OnMouseActivate()
{
	// update LBM state when window is reactivated and redraw window content
	m_bLeftMouseButtonPressed = true;
	OnDraw();

	return( MA_ACTIVATE );
}



LRESULT 
CFarCryAutoCDApp::WndProc( HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam )
{
	// our window proc
	switch( message ) 
	{
	case WM_KEYDOWN:
		{
			OnKeyDown( (unsigned int) wParam, (unsigned int) LOWORD( lParam ), (unsigned int) HIWORD( lParam ) );
			break;
		}
	case WM_MOUSEMOVE:
		{
			OnMouseMove( (short) LOWORD( lParam ), (short) HIWORD( lParam ), (unsigned int) wParam );
			break;
		}
	case WM_LBUTTONDOWN:
		{
			OnLButtonDown( (short) LOWORD( lParam ), (short) HIWORD( lParam ) );
			break;
		}
	case WM_LBUTTONUP:
		{
			OnLButtonUp();
			break;
		}
	case WM_ERASEBKGND:
		{			
			break;
		}
	case WM_PAINT:
		{
			OnPaint();
			break;
		}
	case WM_TIMER:
		{
			OnTimer( (unsigned int) wParam );
			break;
		}
	case WM_CLOSE:
		{
			OnClose();
			break;
		}
	case WM_SETCURSOR:
		{		
			return( 1 );
		}
 	case WM_SIZE:
		{
			OnSize( (unsigned int) wParam, (unsigned short) LOWORD( lParam ), (unsigned short) HIWORD( lParam ) );
			break;
		}
	case WM_MOUSEACTIVATE:
		{
			return( OnMouseActivate() );
		}
	default:
		{
			return( DefWindowProc( hWnd, message, wParam, lParam ) );
		}
	}
	return( 0 );
}