#include "stdafx.h"
#include "MenuData.h"
#include "Utilities.h"
#include "LanguageID.h"
#include "BackBuffer.h"
#include "LocalizationTable.h"
#include "LanguageID.h"
#include "FarCryRegistry.h"
#include "resource.h"

#ifdef UNICODE
#include <vector>
#endif

extern "C"
{
#   include "jpeglib.h"
}



const TCHAR c_pcResMenuBackground[]( _T( "IDR_MENU_BACKGROUND" ) );
const TCHAR c_pcResSysMenuButtons[]( _T( "IDR_MENU_SYSMENUBUTTONS" ) );

const unsigned int c_uiMenuColorInactive( RGB( 170, 232, 248 ) );
const unsigned int c_uiMenuColorBase( RGB( 170, 232, 248 ) );
const unsigned int c_uiMenuColorHighlighted( RGB( 225, 223, 216 ) );
const unsigned int c_uiMenuColorSelected( RGB( 255, 255, 255 ) );
const unsigned int c_uiMenuColorShadow( RGB( 0, 0, 0 ) );
const unsigned int c_uiSysMenuColorBase( RGB( 0, 0, 0 ) );
const unsigned int c_uiSysMenuColorHighlighted( RGB( 225, 223, 216 ) );
const unsigned int c_uiSysMenuColorSelected( RGB( 255, 255, 255 ) );




const int c_iMenuFontSize( 25 );
const int c_iMenuFontSizeJapanese( 25 );

const int c_menuButtonPos[ NUM_MENUBUTTONS ][ 2 ] =
{
	{ c_iWndWidth / 2, c_iWndHeight - 25 - 8 * ( c_iMenuFontSize + 3 ) },
	{ c_iWndWidth / 2, c_iWndHeight - 25 - 7 * ( c_iMenuFontSize + 3 ) },
	{ c_iWndWidth / 2, c_iWndHeight - 25 - 6 * ( c_iMenuFontSize + 3 ) },
	{ c_iWndWidth / 2, c_iWndHeight - 25 - 5 * ( c_iMenuFontSize + 3 ) },
	{ c_iWndWidth / 2, c_iWndHeight - 25 - 4 * ( c_iMenuFontSize + 3 ) },
	{ c_iWndWidth / 2, c_iWndHeight - 25 - 3 * ( c_iMenuFontSize + 3 ) },
	{ c_iWndWidth / 2, c_iWndHeight - 25 - 2 * ( c_iMenuFontSize + 3 ) },
  { c_iWndWidth / 2, c_iWndHeight - 25 - 1 * ( c_iMenuFontSize + 3 ) }
};

const int c_iSysMenuButtonWidth( 16 );
const int c_iSysMenuButtonHeight( 16 );

const int c_sysMenuButtonPos[ NUM_SYSMENUBUTTONS ][ 2 ] =
{
    { c_iWndWidth - ( 2 * c_iSysMenuButtonWidth + 7 + 1 ), 6 },
    { c_iWndWidth - ( c_iSysMenuButtonWidth + 7 ) , 6 },
};



using namespace std;



static inline unsigned int
RGBToBGR( unsigned int uiColor )
{
    return( ( ( uiColor >> 16 ) & 0x0000FF ) | 
            ( ( uiColor       ) & 0x00FF00 ) | 
            ( ( uiColor << 16 ) & 0xFF0000 ) );
}



static inline unsigned char
GetR( unsigned int color )
{
    return( ( color >> 16 ) & 0xFF );
}



static inline unsigned char
GetG( unsigned int color )
{
    return( ( color >> 8 ) & 0xFF );
}



static inline unsigned char
GetB( unsigned int color )
{
    return( ( color ) & 0xFF );
}



static inline unsigned char
Modulate( unsigned char a, unsigned char b )
{
    return(  a * b  >> 8 );
}



static inline unsigned char
Lerp( unsigned char a, unsigned char b, unsigned char f )
{
    return( a + ( ( b - a ) * f >> 8 ) );
}



static void
JPEGErrorExit( j_common_ptr cinfo )
{
	char cBuffer[ JMSG_LENGTH_MAX ];
	(*cinfo->err->format_message) ( cinfo, cBuffer );
#ifdef UNICODE
	size_t length( strlen( cBuffer ) );
	vector< wchar_t > wStr( length + 1 );
	mbstowcs( &wStr[ 0 ], cBuffer, length );
	wStr[ length ] = 0;
	throw( texception( &wStr[ 0 ] ) );
#else
	throw( texception( cBuffer ) );
#endif
}



static inline void
JPEGCleanUp( jpeg_decompress_struct& cinfo )
{
	try
	{
		jpeg_finish_decompress( &cinfo );
	}
	catch( ... )
	{
	}

	try
	{
		jpeg_destroy_decompress( &cinfo );
	}
	catch( ... )
	{
	}
}



CMenuData::CMenuData( const HINSTANCE& hInstance, CBackBuffer* pkBackBuffer )
: m_pkBackBuffer( pkBackBuffer )
, m_imgMenuBase()
, m_imgMenuHighlighted()
, m_imgMenuSelected()
, m_imgMenuMask()
{
	// enable all buttons by default
	for( int i( 0 ); i < NUM_MENUBUTTONS; ++i )
	{
		m_pbButtonInactive[ i ] = false;
	}

	// disable various buttons depending on whether FarCry is installed or not
	if( false == FarCryRegistry::IsInstalled() )
	{
		m_pbButtonInactive[ BID_PLAY ] = true;
		m_pbButtonInactive[ BID_UNINSTALL ] = true;
		m_pbButtonInactive[ BID_REGISTER ] = true;
	}
	else
	{
		m_pbButtonInactive[ BID_INSTALL ] = true;
	}

	// create menu mask
	CreateMenuMask();

	// create menu images 
	CreateBaseMenu( hInstance );
	CreateHighlightedMenu( hInstance );
	CreateSelectedMenu( hInstance );
}



CMenuData::~CMenuData()
{
}



void
CMenuData::LoadJpegFromMemory( Raw32BitImage& image, unsigned int& uiWidth, unsigned int& uiHeight, const unsigned char* pucJpegData, unsigned int uiJpegSize ) const
{    
	jpeg_decompress_struct cinfo;

	try
	{
		// setup jpeg error routines and overwrite error_exit with our own
		jpeg_error_mgr jerr;
		cinfo.err       = jpeg_std_error( &jerr );
		jerr.error_exit = JPEGErrorExit;

		// prepare decompression
		jpeg_create_decompress( &cinfo );
		jpeg_stdio_src( &cinfo, 0 );

		// load jpeg data from memory
		cinfo.src->bytes_in_buffer = uiJpegSize;
		cinfo.src->next_input_byte = (JOCTET *) pucJpegData;

		// read header
		jpeg_read_header( &cinfo, TRUE );

		// set to high quality decompression
		cinfo.dct_method          = JDCT_FLOAT;
		cinfo.do_fancy_upsampling = TRUE;
		cinfo.do_block_smoothing  = TRUE;

		// start decompression
		jpeg_start_decompress( &cinfo );

		// allocate memory for one scanline
		JSAMPARRAY jpegBuf( (*cinfo.mem->alloc_sarray) ( (j_common_ptr) &cinfo, JPOOL_IMAGE,
			cinfo.output_width * cinfo.out_color_components, 1 ) );

		// set image dimensions
		uiWidth = cinfo.output_width;
		uiHeight = cinfo.output_height;

		// reserve enough space for image
		image.clear();
		image.reserve( uiWidth * uiHeight );

		// image is rgb
		if( 3 == cinfo.out_color_components )
		{
			const unsigned int uiAlphaValue( 0xFF000000 );
			while( cinfo.output_scanline < uiHeight )
			{
				unsigned int uiPixelOffset( 0 );

				unsigned char* pucScanline( jpegBuf[ 0 ] );
				jpeg_read_scanlines( &cinfo, jpegBuf, 1 );

				for( unsigned int i( 0 ); i < uiWidth; ++i, uiPixelOffset += 3 )
				{
					image.push_back( ( *( pucScanline + uiPixelOffset + 0 ) << 16 ) +
						             ( *( pucScanline + uiPixelOffset + 1 ) <<  8 ) +
						             ( *( pucScanline + uiPixelOffset + 2 )       ) + uiAlphaValue );
				}
			}
		}
		// image is gray scale
		else
		{
			const unsigned int uiAlphaValue( 0xFF000000 );
			while( cinfo.output_scanline < uiHeight )
			{
				unsigned int uiAddrRGB( cinfo.output_scanline * uiWidth );

				unsigned char* pucScanline( jpegBuf[ 0 ] );
				jpeg_read_scanlines( &cinfo, jpegBuf, 1 );

				for( unsigned int i( 0 ); i < uiWidth; ++i )
				{
					image.push_back( ( *( pucScanline + i ) << 16 ) +
						             ( *( pucScanline + i ) <<  8 ) +
						             ( *( pucScanline + i )       ) + uiAlphaValue );
				}
			}
		}
	}
	catch( ... )
	{
		JPEGCleanUp( cinfo );
		throw;
	}

	JPEGCleanUp( cinfo );
}



void
CMenuData::LoadJpegFromResource( Raw32BitImage& image, unsigned int& uiWidth, unsigned int& uiHeight, const HINSTANCE& hInstance, const TCHAR* pcResourceName ) const
{
	// get access to menu image in resource 
	const unsigned char* pucData( 0 );
	unsigned int uiDataSize( 0 );
	GetAccessToResource( pucData, uiDataSize, pcResourceName );

	// menu image not found in resource
	if( 0 == pucData )
	{
		throw( texception( GetStringFromResource( hInstance, IDS_FAILEDTOLOADDATA ) ) );
	}

	// load menu image
	LoadJpegFromMemory( image, uiWidth, uiHeight, pucData, uiDataSize );
}



void 
CMenuData::LoadMenuBackground( Raw32BitImage& imgMenuBackground, const HINSTANCE& hInstance ) const
{
    unsigned int uiWidth( 0 );
    unsigned int uiHeight( 0 );
    LoadJpegFromResource( imgMenuBackground, uiWidth, uiHeight, hInstance, c_pcResMenuBackground );
    assert( c_iWndWidth == uiWidth && c_iWndHeight == uiHeight );
}



void 
CMenuData::LoadSysMenuButtons( Raw32BitImage& imgSysMenuButtons, const HINSTANCE& hInstance ) const
{
    unsigned int uiWidth( 0 );
    unsigned int uiHeight( 0 );
    LoadJpegFromResource( imgSysMenuButtons, uiWidth, uiHeight, hInstance, c_pcResSysMenuButtons );
    assert( NUM_SYSMENUBUTTONS * c_iSysMenuButtonWidth == uiWidth && c_iSysMenuButtonHeight == uiHeight );
}



void 
CMenuData::CreateBaseMenu( const HINSTANCE& hInstance )
{
	CreateMenu( hInstance, m_imgMenuBase, c_uiMenuColorBase, false );
    CreateSysMenu( hInstance, m_imgMenuBase, RGBToBGR( c_uiSysMenuColorBase ) );
}



void 
CMenuData::CreateHighlightedMenu( const HINSTANCE& hInstance )
{
	CreateMenu( hInstance, m_imgMenuHighlighted, c_uiMenuColorHighlighted, false );
    CreateSysMenu( hInstance, m_imgMenuHighlighted, RGBToBGR( c_uiSysMenuColorHighlighted ) );
}



void 
CMenuData::CreateSelectedMenu( const HINSTANCE& hInstance )
{
	CreateMenu( hInstance, m_imgMenuSelected, c_uiMenuColorSelected, true );
    CreateSysMenu( hInstance, m_imgMenuSelected, RGBToBGR( c_uiSysMenuColorSelected ) );
}



static HFONT
CreateMenuFont()
{
	if( 0x11 == GetPrimaryLanguageID() )
	{
		// use special font containing japanese glyphs
		return( CreateFont( c_iMenuFontSizeJapanese, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, 
							DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, ANTIALIASED_QUALITY, 
							FF_DONTCARE, _T( "MS Gothic" ) ) );
	}
	else
	{
		return( CreateFont( c_iMenuFontSize, 0, 0, 0, FW_NORMAL, FALSE, FALSE, FALSE, 
							DEFAULT_CHARSET, OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, ANTIALIASED_QUALITY, 
							FF_DONTCARE, _T( "Arial Narrow" ) ) );
	}
}


void
CMenuData::GetMenuButtonTextRect( HDC hBackBufferDC, int iButtonIndex, const tstring& strButtonText, RECT& rect ) const
{
	ZeroMemory( &rect, sizeof( rect ) );
	DrawText( hBackBufferDC, strButtonText.c_str(), (int) strButtonText.size(), &rect, DT_CALCRECT );

	int iWidth( rect.right - rect.left );
	int iHeight( rect.bottom - rect.top );

	rect.left += c_menuButtonPos[ iButtonIndex ][ 0 ] - iWidth / 2; 
	rect.right += c_menuButtonPos[ iButtonIndex ][ 0 ] - iWidth / 2; 
	rect.top += c_menuButtonPos[ iButtonIndex ][ 1 ] - iHeight / 2; 
	rect.bottom += c_menuButtonPos[ iButtonIndex ][ 1 ] - iHeight / 2;
}


void 
CMenuData::CreateMenu( const HINSTANCE& hInstance, Raw32BitImage& imgMenu, unsigned int uiMenuFontColor, bool bBold )
{
	Raw32BitImage imgTemp;
	LoadMenuBackground( imgTemp, hInstance );

	// get access to back buffer
	HDC hBackBufferDC( m_pkBackBuffer->GetDC() );
	unsigned int* puiBackBufferBits( m_pkBackBuffer->GetPtr() );

	// copy image to back buffer
	for( unsigned int uiIndex( 0 ); uiIndex < c_iWndWidth * c_iWndHeight; ++uiIndex )
	{
		puiBackBufferBits[ uiIndex ] = imgTemp[ uiIndex ];		
	}

	// create font
	HFONT hFont( CreateMenuFont() );
	HFONT hOldFont( (HFONT) SelectObject( hBackBufferDC, hFont ) );

	// render text
	SetMapMode( hBackBufferDC, MM_TEXT );
	SetBkMode( hBackBufferDC, TRANSPARENT );

	// render inactive text first
	for( unsigned int i( 0 ); i < NUM_MENUBUTTONS; ++i )
	{
		if( false != m_pbButtonInactive[ c_buttonIDs[ i ] ] )
		{
			if( 0x11 != GetPrimarySystemLanguageID() || c_buttonIDs[ i ] != BID_REGISTER )
			{
				tstring strButtonText( CLocalizationTable::GetButtonText( c_buttonIDs[ i ] ) );
				
				RECT rect;
				GetMenuButtonTextRect( hBackBufferDC, i, strButtonText, rect );

				SetTextColor( hBackBufferDC, c_uiMenuColorShadow );
				DrawText( hBackBufferDC, strButtonText.c_str(), (int) strButtonText.size(), &rect, DT_CENTER );

				rect.left -= 1; rect.right -= 1; rect.top -= 1; rect.bottom -= 1;			
				SetTextColor( hBackBufferDC, c_uiMenuColorInactive );
				DrawText( hBackBufferDC, strButtonText.c_str(), (int) strButtonText.size(), &rect, DT_CENTER );
			}
		}
	}
	
	// blend 1:1 with background image to get transparency effect
	for( unsigned int uiIndex( 0 ); uiIndex < c_iWndWidth * c_iWndHeight; ++uiIndex )
	{
		puiBackBufferBits[ uiIndex ] = ( ( puiBackBufferBits[ uiIndex ] >> 1 ) & 0x7F7F7F7F ) +
									   ( ( imgTemp[ uiIndex ]           >> 1 ) & 0x7F7F7F7F );		
	}

	// now render active text
	for( unsigned int i( 0 ); i < NUM_MENUBUTTONS; ++i )
	{
		if( false == m_pbButtonInactive[ c_buttonIDs[ i ] ] )
		{
			if( 0x11 != GetPrimarySystemLanguageID() || c_buttonIDs[ i ] != BID_REGISTER )
			{
				tstring strButtonText( CLocalizationTable::GetButtonText( c_buttonIDs[ i ] ) );

				RECT rect;
				GetMenuButtonTextRect( hBackBufferDC, i, strButtonText, rect );

				SetTextColor( hBackBufferDC, c_uiMenuColorShadow );
				DrawText( hBackBufferDC, strButtonText.c_str(), (int) strButtonText.size(), &rect, DT_CENTER );

				rect.left -= 1; rect.right -= 1; rect.top -= 1; rect.bottom -= 1;			
				SetTextColor( hBackBufferDC, uiMenuFontColor );
				DrawText( hBackBufferDC, strButtonText.c_str(), (int) strButtonText.size(), &rect, DT_CENTER );
			}
		}
	}

	// init final menu image
	imgMenu.clear();
	imgMenu.reserve( c_iWndWidth * c_iWndHeight );

	// copy back buffer to final menu image
	for( unsigned int uiIndex( 0 ); uiIndex < c_iWndWidth * c_iWndHeight; ++uiIndex )
	{
		imgMenu.push_back( puiBackBufferBits[ uiIndex ] );
	}

	// deselect font
	SelectObject( hBackBufferDC, hOldFont );
}



void 
CMenuData::CreateMenuMask()
{
	// get access to back buffer
	HDC hBackBufferDC( m_pkBackBuffer->GetDC() );
	unsigned int* puiBackBufferBits( m_pkBackBuffer->GetPtr() );

	// init mask image
	m_imgMenuMask.clear();
	m_imgMenuMask.reserve( c_iWndWidth * c_iWndHeight );
	for( unsigned int uiIndex( 0 ); uiIndex < c_iWndWidth * c_iWndHeight; ++uiIndex )
	{
		m_imgMenuMask.push_back( BID_NOBUTTON );
	}

	// init back buffer
	memset( puiBackBufferBits, 0, sizeof( unsigned int ) * c_iWndWidth * c_iWndHeight );

	// create font
	HFONT hFont( CreateMenuFont() );
	HFONT hOldFont( (HFONT) SelectObject( hBackBufferDC, hFont ) );

	// render text
	SetMapMode( hBackBufferDC, MM_TEXT );
	SetBkMode( hBackBufferDC, OPAQUE );
	SetTextColor( hBackBufferDC, c_uiMenuColorBase );

	for( unsigned int i( 0 ); i < NUM_MENUBUTTONS; ++i )
	{
		if( false == m_pbButtonInactive[ c_buttonIDs[ i ] ] )
		{
			if( 0x11 != GetPrimarySystemLanguageID() || c_buttonIDs[ i ] != BID_REGISTER )
			{
				tstring strButtonText( CLocalizationTable::GetButtonText( c_buttonIDs[ i ] ) );

				RECT rect;
				GetMenuButtonTextRect( hBackBufferDC, i, strButtonText, rect );
				rect.left -= 1; rect.right += 1; rect.top -= 1; rect.bottom += 1;			

				for( int y( rect.top ); y < rect.bottom; ++y )
				{
					if( y >= 0 && y < c_iWndHeight )
					{
						for( int x( rect.left ); x < rect.right; ++x )
						{
							if( x >= 0 && x < c_iWndWidth )
							{
								m_imgMenuMask[ GetOffset( x, y ) ] = c_buttonIDs[ i ];
							}
						}
					}
				}
			}
		}
	}

	// create system menu mask
	CreateSysMenuMask();
}



void 
CMenuData::CreateSysMenu( const HINSTANCE& hInstance, Raw32BitImage& imgMenu, unsigned int uiSysMenuColor )
{
    // assume that imgMenu is already filled with image data so system menu buttons can be placed
    assert( c_iWndWidth * c_iWndHeight == imgMenu.size() );

    // load button image of system menu
	Raw32BitImage imgTemp;
    LoadSysMenuButtons( imgTemp, hInstance );

    // alpha blend buttons over menu background
	for( unsigned int i( 0 ); i < NUM_SYSMENUBUTTONS; ++i )
	{
        for( int y( c_sysMenuButtonPos[ i ][ 1 ] ), py( 0 ); y < c_sysMenuButtonPos[ i ][ 1 ] + c_iSysMenuButtonHeight; ++y, ++py )
		{
            assert( y >= 0 && y < c_iWndHeight );
            unsigned int uiOffset( NUM_SYSMENUBUTTONS * c_iSysMenuButtonWidth * py + i * c_iSysMenuButtonWidth );
			for( int x( c_sysMenuButtonPos[ i ][ 0 ] ); x < c_sysMenuButtonPos[ i ][ 0 ] + c_iSysMenuButtonWidth; ++x, ++uiOffset )
			{
                assert( x >= 0 && x < c_iWndWidth );
                imgMenu[ GetOffset( x, y ) ] = 
                    ( Lerp( GetR( imgMenu[ GetOffset( x, y ) ] ), Modulate( GetR( imgTemp[ uiOffset ] ), GetR( uiSysMenuColor ) ), GetR( imgTemp[ uiOffset ] ) ) << 16 ) |
                    ( Lerp( GetG( imgMenu[ GetOffset( x, y ) ] ), Modulate( GetG( imgTemp[ uiOffset ] ), GetG( uiSysMenuColor ) ), GetG( imgTemp[ uiOffset ] ) ) <<  8 ) |
                    ( Lerp( GetB( imgMenu[ GetOffset( x, y ) ] ), Modulate( GetB( imgTemp[ uiOffset ] ), GetB( uiSysMenuColor ) ), GetB( imgTemp[ uiOffset ] ) )       );
			}
		}
	}
}



void 
CMenuData::CreateSysMenuMask()
{
	// set mask for system menu buttons
	for( unsigned int i( 0 ); i < NUM_SYSMENUBUTTONS; ++i )
	{
		for( int y( c_sysMenuButtonPos[ i ][ 1 ] ); y < c_sysMenuButtonPos[ i ][ 1 ] + c_iSysMenuButtonHeight; ++y )
		{
			assert( y >= 0 && y < c_iWndHeight );
			for( int x( c_sysMenuButtonPos[ i ][ 0 ] ); x < c_sysMenuButtonPos[ i ][ 0 ] + c_iSysMenuButtonWidth; ++x )
			{
				assert( x >= 0 && x < c_iWndWidth );
				m_imgMenuMask[ GetOffset( x, y ) ] = c_sysButtonIDs[ i ];
			}
		}
	}
}
