#include "stdafx.h"
#include "GameOptionsPage.h"
#include "SystemConfiguration.h"
#include "SystemAnalysis.h"
#include "FarCryRegistry.h"
#include "Helpers.h"

// include resource IDs of english version, for all other languages the IDs MUST BE THE SAME!!!
#include "..\FarCryConfigEng\resource.h"

#include <set>
#include <string>



struct SLanguage
{
	const TCHAR* pcName;
	unsigned int uiLocalizedID;
};

const SLanguage c_supportedLanguages[] =
{
	{ _T( "english" ), IDS_ENGLISH },
	{ _T( "german" ), IDS_GERMAN },
	{ _T( "french" ), IDS_FRENCH },
	{ _T( "italian" ), IDS_ITALIAN },
	{ _T( "spanish" ), IDS_SPANISH },
	{ _T( "japanese" ), IDS_JAPANESE },
	{ _T( "polish" ), IDS_POLISH }
};



using namespace std;



IMPLEMENT_DYNAMIC( CGameOptionsPage, CPropertyPage )



CGameOptionsPage::CGameOptionsPage( CSystemConfiguration& kSysConfig )
: CPropertyPage( IDD_GAME )
, m_kSysConfig( kSysConfig )
, m_kLanguageCtrl()
, m_kGoreCtrl()
, m_kLazyWeaponCtrl()
, m_dDeathTime( 0.0 )
, m_kSysSpecCtrl()
, m_kAutoDetectSysSpecCtrl()
{
}



CGameOptionsPage::~CGameOptionsPage()
{
}


// adaption of MFC's AfxTextFloatFormat
static void 
MyTextFloatFormat( CDataExchange* pDX, int nIDC, void* pData, double value, int nSizeGcvt )
{
	ASSERT( 0 != pData );
	pDX->PrepareEditCtrl( nIDC );
	
	HWND hWndCtrl;
	pDX->m_pDlgWnd->GetDlgItem( nIDC, &hWndCtrl );
	
	TCHAR szBuffer[ 400 ];
	if( FALSE != pDX->m_bSaveAndValidate )
	{
		::GetWindowText( hWndCtrl, szBuffer, sizeof( szBuffer ) / sizeof( szBuffer[ 0 ] ) );
		double d;
		if( 1 != _stscanf( szBuffer, _T( "%lf" ), &d ) )
		{
			MessageBox( AfxGetMainWnd()->m_hWnd, LoadString( IDS_INCORRECT_VALUE ), LoadString( IDS_APP_TITLE ), MB_OK | MB_ICONHAND );
			pDX->Fail();
		}
		if( FLT_DIG == nSizeGcvt )
		{
			*( (float*) pData ) = (float) d;
		}
		else
		{
			*( (double*) pData)  = d;
		}
	}
	else
	{
		_stprintf( szBuffer,  _T( "%.*g" ), nSizeGcvt, value );
		AfxSetWindowText( hWndCtrl, szBuffer );
	}
}



//static void 
//AFXAPI DDX_Text(CDataExchange* pDX, int nIDC, float& value)
//{
//	MyTextFloatFormat( pDX, nIDC, &value, value, FLT_DIG );
//}



static void 
My_DDX_Text( CDataExchange* pDX, int nIDC, double& value )
{
	MyTextFloatFormat( pDX, nIDC, &value, value, DBL_DIG );
}



void 
CGameOptionsPage::DoDataExchange( CDataExchange* pDX )
{
	CPropertyPage::DoDataExchange( pDX );

	DDX_Control( pDX, IDC_LANGUAGE, m_kLanguageCtrl );
	DDX_Control( pDX, IDC_GORE, m_kGoreCtrl );
	DDX_Control( pDX, IDC_LAZYWEAPON, m_kLazyWeaponCtrl );
	My_DDX_Text( pDX, IDC_DEATHTIME, m_dDeathTime );
	if( m_dDeathTime < 0.0 )
	{
		AfxMessageBox( IDS_INCORRECT_VALUE );
		pDX->Fail();
	}	
	DDX_Control( pDX, IDC_SYSSPEC, m_kSysSpecCtrl );
	DDX_Control( pDX, IDC_AD_SYSSPEC, m_kAutoDetectSysSpecCtrl );
}



BEGIN_MESSAGE_MAP( CGameOptionsPage, CPropertyPage )
	ON_BN_CLICKED( IDC_AD_SYSSPEC, OnBnClickedAutoDetectSysSpec )
END_MESSAGE_MAP()



BOOL 
CGameOptionsPage::OnInitDialog()
{
	CPropertyPage::OnInitDialog();

	// setup initial language value
	SysConfigValue g_language( m_kSysConfig.GetValue( "g_language" ) );
	
	int index( 0 );
	for( int i( 0 ); i < sizeof( c_supportedLanguages ) / sizeof( c_supportedLanguages[ 0 ] ); ++i )
	{
		tstring strFarCryInstallationFolder;
		FarCryRegistry::GetFarCryInstallationFolder( strFarCryInstallationFolder );

		tstring strLanguagePak( strFarCryInstallationFolder + _T( "FCData\\Localized\\" ) );
		strLanguagePak += c_supportedLanguages[ i ].pcName;
		strLanguagePak += _T( ".pak" );

		// search for language pak
#ifdef UNICODE
		FILE* f( _wfopen( strLanguagePak.c_str(), _T( "rb" ) ) );
#else
		FILE* f( fopen( strLanguagePak.c_str(), _T( "rb" ) ) );
#endif
		if( 0 !=  f )
		{
			fclose( f );

			// add language to combo box
			m_kLanguageCtrl.AddString( LoadString( c_supportedLanguages[ i ].uiLocalizedID ) );
			m_kLanguageCtrl.SetItemDataPtr( m_kLanguageCtrl.GetCount() - 1, (void*) c_supportedLanguages[ i ].pcName );
#ifdef UNICODE
			SysConfigValue::size_type length( g_language.size() );
			vector< wchar_t > wStr( length + 1 );
			mbstowcs( &wStr[ 0 ], g_language.c_str(), length );	
			wStr[ length ] = 0;
			tstring g_language_wc( &wStr[ 0 ] );
			if( g_language_wc == c_supportedLanguages[ i ].pcName )
#else
			if( g_language == c_supportedLanguages[ i ].pcName )
#endif
			{
				index = m_kLanguageCtrl.GetCount() - 1; // we found our current language
			}
		}
	}
	// set currently used language, if "index" has been overwritten in loop above 
	// it just defaults to the first language which was made availabe in the combo box
	if( m_kLanguageCtrl.GetCount() > 0 )
	{
		m_kLanguageCtrl.SetCurSel( index );
	}

#ifdef CAN_ADJUST_GORE
	// setup initial gore value
	SysConfigValue g_gore( m_kSysConfig.GetValue( "g_gore" ) );
	m_kGoreCtrl.SetCheck( false != IsBigger( g_gore, 0 ) ? 1 : 0 );
	m_kGoreCtrl.ShowWindow( SW_SHOW );
#endif // #ifdef CAN_ADJUST_GORE

	// setup initial lazy weapon value
	m_kLazyWeaponCtrl.SetRange( 0, 100 );
	m_kLazyWeaponCtrl.SetTicFreq( 10 );

	SysConfigValue cl_LazyWeapon( m_kSysConfig.GetValue( "cl_lazy_weapon" ) );
	int iLazyWeaponValue( (int) ( atof( cl_LazyWeapon.c_str() ) * 100.0 ) );
	m_kLazyWeaponCtrl.SetPos( iLazyWeaponValue );

	// setup initial death time value
	SysConfigValue p_deathtime( m_kSysConfig.GetValue( "p_deathtime" ) );
	m_dDeathTime = atof( p_deathtime.c_str() );

	// setup initial sys spec value
	m_kSysSpecCtrl.AddString( LoadString( IDS_LOW ) );
	m_kSysSpecCtrl.AddString( LoadString( IDS_MEDIUM ) );
	m_kSysSpecCtrl.AddString( LoadString( IDS_HIGH ) );
	m_kSysSpecCtrl.AddString( LoadString( IDS_VERYHIGH ) );

	SysConfigValue sys_spec( m_kSysConfig.GetValue( "sys_spec" ) );
	if( false != IsEqual( sys_spec, "0" ) )
	{
		m_kSysSpecCtrl.SetCurSel( 0 );
	}
	else if( false != IsEqual( sys_spec, "1" ) )
	{
		m_kSysSpecCtrl.SetCurSel( 1 );
	}
	else if( false != IsEqual( sys_spec, "2" ) )
	{
		m_kSysSpecCtrl.SetCurSel( 2 );
	}
	else if( false != IsEqual( sys_spec, "3" ) )
	{
		m_kSysSpecCtrl.SetCurSel( 3 );
	}

	// if needed, perform auto detection of optimal settings
	if( false != m_kSysConfig.IsInitialConfig() )
	{
		OnBnClickedAutoDetectSysSpec();
	}

	UpdateData( FALSE );
	return( TRUE );  
}



void 
CGameOptionsPage::OnBnClickedAutoDetectSysSpec()
{
	m_kSysSpecCtrl.SetCurSel( GetCPUQuality() );
}



void 
CGameOptionsPage::UpdateSystemConfig()
{
	// write new language value
	int index( m_kLanguageCtrl.GetCurSel() );
	if( CB_ERR != index )
	{
		TCHAR* pcLanguage( (TCHAR*) m_kLanguageCtrl.GetItemDataPtr( index ) );
#ifdef UNICODE
		size_t length( wcslen( pcLanguage ) );
		vector< char > cStr( length + 1 );
		wcstombs( &cStr[ 0 ], pcLanguage, length );	
		cStr[ length ] = 0;
		m_kSysConfig.SetValue( "g_language", SysConfigKey( &cStr[ 0 ] ) );
#else
		m_kSysConfig.SetValue( "g_language", SysConfigKey( pcLanguage ) );
#endif
	}

#ifdef CAN_ADJUST_GORE
	// write new gore value
	m_kSysConfig.SetValue( "g_gore", ( FALSE != m_kGoreCtrl.GetCheck() ) ? "2" : "0" );
#endif // #ifdef CAN_ADJUST_GORE

	// write new lazy weapon value
	char szData[ 32 ];
	sprintf( szData, "%1.4lf", m_kLazyWeaponCtrl.GetPos() / 100.0 );
	m_kSysConfig.SetValue( "cl_lazy_weapon", SysConfigValue( szData ) );

	// write new death time value
	sprintf( szData, "%1.4lf", m_dDeathTime );
	m_kSysConfig.SetValue( "p_deathtime", SysConfigValue( szData ) );

	// write new sys spec value
	index = m_kSysSpecCtrl.GetCurSel();
	if( CB_ERR != index )
	{	
		sprintf( szData, "%d", index );
		m_kSysConfig.SetValue( "sys_spec", SysConfigKey( szData ) );
	}
}



void 
CGameOptionsPage::OnOK()
{
	CPropertyPage::OnOK();
	UpdateSystemConfig();
	SetDefaultKeyBindings();
}



void
CGameOptionsPage::SetDefaultKeyBindings()
{
	// if game is installed try to setup the key bindings for the default player (if not existent already)
	tstring strFarCryInstallationFolder;	
	if( false != FarCryRegistry::GetFarCryInstallationFolder( strFarCryInstallationFolder ) )
	{
		// build path to key bindings of default profile
		tstring strDefKeysFilePath( strFarCryInstallationFolder + _T( "Profiles\\player\\default\\game.cfg" ) );

		// build path to localized key bindings
		tstring strLocalizedKeyFilePath( strFarCryInstallationFolder );
		strLocalizedKeyFilePath += _T( "Profiles\\defaults\\" );

		SysConfigValue g_language( m_kSysConfig.GetValue( "g_language" ) );
#ifdef UNICODE
		SysConfigValue::size_type length( g_language.size() );
		vector< wchar_t > wStr( length + 1 );
		mbstowcs( &wStr[ 0 ], g_language.c_str(), length );	
		wStr[ length ] = 0;
		strLocalizedKeyFilePath += (TCHAR*) &wStr[ 0 ];
#else
		strLocalizedKeyFilePath += g_language.c_str();
#endif
		strLocalizedKeyFilePath += _T( "\\game.cfg" );

		// copy file if not already existent
		CopyFile( strLocalizedKeyFilePath.c_str(), strDefKeysFilePath.c_str(), TRUE );	
	}
}