#include "StdAfx.h"
#include "SystemConfiguration.h"

#include <vector>
#include <fstream>
#include <sstream>
#include <algorithm>



using namespace std;



const char c_pcSystemCfgComment[] = "-- This file has been written by FarCryConfigurator.exe";



CSystemConfiguration::CSystemConfiguration( const string& strSysConfigFilePath )
: m_strSysConfigFilePath( strSysConfigFilePath )
, m_colCCVars()
, m_bIsInitialConfig( false )
{
	InitDefaultValues();
	ParseSystemConfig();
}



CSystemConfiguration::~CSystemConfiguration()
{
}



static inline string
RemoveWhiteSpaces( string& s )
{
	return( string( s.begin(), remove( s.begin(), s.end(), ' ' ) ) );	
}



void
CSystemConfiguration::Parse( istream& kStream )
{
	while( false != kStream.good() )
	{
		string strLine;
		getline( kStream, strLine, '\n' );

		// skip comments
		if( 0 < strLine.find( "--" ) )
		{
			// extract key
			string::size_type posEq( strLine.find( "=", 0 ) );
			if( string::npos != posEq )
			{
				string strKey( RemoveWhiteSpaces( string( strLine, 0, posEq ) ) );
				if( false == strKey.empty() )
				{
					// extract value
					string::size_type posValueStart( strLine.find( "\"", posEq + 1 ) + 1 );
					string::size_type posValueEnd( strLine.find( "\"", posValueStart ) );

					if( string::npos != posValueStart && string::npos != posValueEnd )
					{
						string strValue( strLine, posValueStart, posValueEnd - posValueStart );						
						//m_colCCVars.insert( SysConfigMap::value_type( strKey, strValue ) );
						m_colCCVars[ strKey ] = strValue;
					}
				}					
			}
		}
	}
}



static const unsigned char*
GetAccessToResource( unsigned int& uiDataSize, const TCHAR* pcResourceName )
{
	HRSRC  hInfoBlock;
	HANDLE hRes;

	// find the requested resource in our module
	if( 0 == ( hInfoBlock = FindResource( GetModuleHandle( 0 ), pcResourceName, RT_RCDATA ) ) )
	{
		return( 0 );
	}

	// get size
	uiDataSize = SizeofResource( 0, hInfoBlock );
	if( 0 == uiDataSize )
	{
		return( 0 );
	}

	// load resource
	if( 0 == ( hRes = LoadResource( 0, hInfoBlock ) ) )
	{
		return( 0 );
	}

	// get pointer to resource data
	return( ( const unsigned char *) LockResource( hRes ) );
}



void
CSystemConfiguration::SetSystemCfgFileAttributes( const std::string& strSysConfigFilePath )
{
#ifdef UNICODE
	size_t length( strSysConfigFilePath.size() );
	vector< wchar_t > wStr( length + 1 );
	mbstowcs( &wStr[ 0 ], strSysConfigFilePath.c_str(), length );	
	wStr[ length ] = 0;
	SetFileAttributes( &wStr[ 0 ], FILE_ATTRIBUTE_ARCHIVE );
#else 
	SetFileAttributes( strSysConfigFilePath.c_str(), FILE_ATTRIBUTE_ARCHIVE );
#endif
}



void
CSystemConfiguration::InitDefaultValues()
{
	const unsigned char* pucData( 0 );
	unsigned uiDataSize( 0 );
	if( 0 != ( pucData = GetAccessToResource( uiDataSize, _T( "IDR_SYSTEM_CFG" ) ) ) )
	{
		stringstream kStream( (const char*) pucData );
		Parse( kStream );
	}
}



void 
CSystemConfiguration::ParseSystemConfig()
{
	SetSystemCfgFileAttributes( m_strSysConfigFilePath );

	ifstream kStream( m_strSysConfigFilePath.c_str() );
	if( false != kStream.good() )
	{
		Parse( kStream );
	}

	// should be called after system.cfg was parsed
	DetermineInitialConfigStatus();
}



void 
CSystemConfiguration::SerializeSystemConfig()
{
	SetSystemCfgFileAttributes( m_strSysConfigFilePath );

	ofstream kStream( m_strSysConfigFilePath.c_str() );
	if( false != kStream.good() )
	{
		// write comment
		kStream << c_pcSystemCfgComment << endl << endl;

		// write cvars
		SysConfigMap::iterator itEnd( m_colCCVars.end() );
		for( SysConfigMap::iterator it( m_colCCVars.begin() ); it != itEnd; ++it )
		{
			kStream << (*it).first << " = " << "\"" << (*it).second << "\"" << endl;
		}
	}
}



void 
CSystemConfiguration::PatchLanguageFromSetup( const SysConfigValue& newLanguage )
{
	const char c_pcSystemCfg[]( "system.cfg" );

	SetSystemCfgFileAttributes( c_pcSystemCfg );

	stringstream strTemp;
	{
		ifstream kIfStream( c_pcSystemCfg );
		while( false != kIfStream.good() )
		{		
			string strLine;
			getline( kIfStream, strLine, '\n' );

			if( 0 == strLine.find( "g_language" ) )
			{
				strTemp << "g_language = \"" << newLanguage << "\"" << endl;
			}
			else
			{
				strTemp << strLine << endl;
			}
		}
	}
	{
		ofstream kOfStream( c_pcSystemCfg );
		if( false != kOfStream.good() )
		{		
			kOfStream << strTemp.str();
		}
	}
}



void 
CSystemConfiguration::DetermineInitialConfigStatus()
{
	ifstream kStream( m_strSysConfigFilePath.c_str() );	
	if( false != kStream.good() )
	{
		string strLine;
		getline( kStream, strLine, '\n' );

		if( c_pcSystemCfgComment == strLine )
		{
			// first line in system.cfg contains a comment written only by FarCryConfigurator,
			// i.e. this config file has been written by us before
			m_bIsInitialConfig = false;
		}
		else
		{
			if( false != IsEqual( GetValue( "sys_firstlaunch" ), "1" ) )
			{
				// the system.cfg seems to be the one which intially ships and gets installed with the game
				m_bIsInitialConfig = true;
			}
			else
			{
				// the system.cfg has been modified by the game before
				m_bIsInitialConfig = false;
			}
		}
	}
	else
	{
		// system.cfg wasn't found, this is the first time we attempt to create/adjust the config
		m_bIsInitialConfig = true;
	}
}



bool 
CSystemConfiguration::IsInitialConfig() const
{
	return( m_bIsInitialConfig );
}



void 
CSystemConfiguration::SetValue( const SysConfigKey& key, const SysConfigValue& value )
{
	m_colCCVars[ key ] = value;
}



SysConfigValue 
CSystemConfiguration::GetValue( const SysConfigKey& key ) const
{
	SysConfigMap::const_iterator it( m_colCCVars.find( key ) );
	if( m_colCCVars.end() != it )
	{
		return( (*it).second );
	}

	return( SysConfigValue( "" ) );
}
