#include "StdAfx.h"
#include "ChunkFile.h"
#include <algorithm>

ChunkFile::ChunkFile()
{
	m_file = 0;
	m_fileHeader.FileType				= FileType_Anim;
	m_fileHeader.ChunkTableOffset		= -1;
	m_fileHeader.Version				= AnimFileVersion;
		
	strcpy(m_fileHeader.Signature,FILE_SIGNATURE);
}

ChunkFile::~ChunkFile()
{
	Close();
}

//////////////////////////////////////////////////////////////////////////
bool ChunkFile::Read( const char *file  )
{
	m_file = fopen( file,"rb" );
	if (!m_file)
		return false;

	fseek( m_file,0,SEEK_END );
	m_fileSize = ftell(m_file);
	fseek( m_file,0,SEEK_SET );

	int res = fread( &m_fileHeader,sizeof(m_fileHeader),1,m_file );
	if (res != 1)
	{
		Close();
		return false;
	}

	if (!ReadChunkTable( m_fileHeader.ChunkTableOffset ))
	{
		Close();
		return false;
	}
	
	LoadTimeChunk();

	Close();
	return true;
}

//////////////////////////////////////////////////////////////////////////
void ChunkFile::Close()
{
	if (m_file)
	{
		fclose(m_file);
	}
	m_file = 0;
}

inline bool ChunkLess( ChunkDesc *d1,ChunkDesc *d2 )
{
	return d1->hdr.FileOffset < d2->hdr.FileOffset;
}

//////////////////////////////////////////////////////////////////////////
bool ChunkFile::ReadChunkTable( int ofs )
{
	int res;
	unsigned i;
	
  fseek(m_file,ofs,SEEK_SET);
  
	int n_chunks;
  res=fread(&n_chunks,sizeof(n_chunks),1,m_file);
  if(res!=1)
  {
    return false;
  }

	std::vector<ChunkDesc*> m_sortedChunks;

	m_chunks.clear();
	m_chunks.resize(n_chunks);
	m_sortedChunks.resize(n_chunks);

	CHUNK_HEADER *chunks = new CHUNK_HEADER[n_chunks];
	assert( chunks );
  res = fread( chunks,sizeof(CHUNK_HEADER),n_chunks,m_file );
  if(res != n_chunks)
  {
    return false;
  }

	int nextFileOffset,chunkSize;
	for (i = 0; i < (int)m_chunks.size(); i++)
	{
		m_chunks[i].hdr = chunks[i];
		m_sortedChunks[i] = &m_chunks[i];
	}

	std::sort( m_sortedChunks.begin(),m_sortedChunks.end(),ChunkLess );
		
	for (i = 0; i < m_sortedChunks.size(); i++)
	{
		if (i+1 < m_sortedChunks.size())
			nextFileOffset = m_sortedChunks[i+1]->hdr.FileOffset;
		else
			nextFileOffset = m_fileSize;

		ChunkDesc &cd = *m_sortedChunks[i];
		chunkSize = nextFileOffset - cd.hdr.FileOffset;
		cd.data.resize( chunkSize );
		fseek( m_file,cd.hdr.FileOffset,SEEK_SET );
		cd.size = chunkSize;
		int res = fread( &(cd.data[0]),chunkSize,1,m_file );
		if(res != 1)
		{
			return false;
		}
	}

	delete []chunks;
	return true;
}

bool ChunkFile::Write( const char *file )
{
	m_file = fopen( file,"wb" );
	if (!m_file)
		return false;

	unsigned i;

	fseek( m_file,sizeof(m_fileHeader),SEEK_SET );

	for(i=0;i<m_chunks.size();i++)
	{
		int ofs = ftell(m_file);
		CHUNK_HEADER &hdr = m_chunks[i].hdr;
		hdr.FileOffset = ofs;
		// write header.
		memcpy( &m_chunks[i].data[0],&hdr,sizeof(hdr) );
		// write data.
		fwrite( &m_chunks[i].data[0],m_chunks[i].data.size(),1,m_file );
	}

	int chunkTableOffset = ftell(m_file);

	//=======================
	//Write # of Chunks
	//=======================
	unsigned nch = m_chunks.size();
	if (fwrite(&nch,sizeof(nch),1,m_file) !=1)
		return false;

	//=======================
	//Write Chunk List
	//=======================
	for(i=0;i<nch;i++)
	{
		CHUNK_HEADER &hdr = m_chunks[i].hdr;
		if (fwrite(&hdr,sizeof(hdr),1,m_file)!=1)
			return false;
	}

	//update Header for chunk list offset
	m_fileHeader.ChunkTableOffset	= chunkTableOffset;
	fseek(m_file,0,SEEK_SET);
	if (fwrite(&m_fileHeader,sizeof(m_fileHeader),1,m_file) != 1)
		return false;

	Close();

	return true;
}

ChunkDesc* ChunkFile::FindChunkByType( int type )
{
	for(unsigned i=0; i<m_chunks.size();i++)
	{
		if (m_chunks[i].hdr.ChunkType == type) 
		{
			return &m_chunks[i];
		}
	}
	return 0;
}

ChunkDesc* ChunkFile::FindChunkById( int id )
{
	for(unsigned i=0; i<m_chunks.size();i++)
	{
		if (m_chunks[i].hdr.ChunkID == id) 
		{
			return &m_chunks[i];
		}
	}
	return 0;
}

void ChunkFile::LoadTimeChunk()
{
	ChunkDesc *cd;

	cd = FindChunkByType( ChunkType_Timing );
	if (!cd)
		return;
	
	timeChunk.chdr = cd->hdr;
	memcpy( &timeChunk,&cd->data[0],sizeof(timeChunk) );
	int ofs = sizeof(timeChunk);
	for (int i = 0; i < timeChunk.nSubRanges; i++)
	{
		RANGE_ENTITY ent;
		ent = *((RANGE_ENTITY*)&cd->data[ofs]);

		m_ranges.push_back( ent );
		ofs += sizeof(RANGE_ENTITY);
	}
}

void ChunkFile::SaveTimeChunk()
{
	ChunkDesc *cd;

	cd = FindChunkByType( ChunkType_Timing );
	if (!cd)
		return;
	
	timeChunk.chdr = cd->hdr;
	timeChunk.nSubRanges = m_ranges.size();
	
	cd->data.resize( sizeof(timeChunk) + timeChunk.nSubRanges*sizeof(RANGE_ENTITY) );
	memcpy( &cd->data[0],&timeChunk,sizeof(timeChunk) );
	int ofs = sizeof(timeChunk);
	
	for (int i = 0; i < timeChunk.nSubRanges; i++)
	{
		*((RANGE_ENTITY*)&cd->data[ofs]) = m_ranges[i];

		ofs += sizeof(RANGE_ENTITY);
	}
}

void ChunkFile::LoadControllerKeys( ChunkDesc* cd,std::vector<CryBoneKey> &keys )
{
	CONTROLLER_CHUNK_DESC_0826 chunk;

	memcpy( &chunk,&cd->data[0],sizeof(chunk) );
	int ofs = sizeof(chunk);
	int nKeys	= chunk.nKeys;
	
	keys.clear();
	for (int i = 0; i < nKeys; i++)
	{
		CryBoneKey key;
		key = *((CryBoneKey*)&cd->data[ofs]);
		keys.push_back( key );
		ofs += sizeof(CryBoneKey);
	}
}

void ChunkFile::SaveControllerKeys( ChunkDesc* cd,std::vector<CryBoneKey> &keys )
{
	CONTROLLER_CHUNK_DESC_0826 chunk;

	memcpy( &chunk,&cd->data[0],sizeof(chunk) );
	chunk.nKeys = keys.size();

	cd->data.resize( sizeof(CONTROLLER_CHUNK_DESC_0826) + chunk.nKeys*sizeof(CryBoneKey) );
	memcpy( &cd->data[0],&chunk,sizeof(chunk) );
	int ofs = sizeof(chunk);
	int nKeys	= keys.size();
	for (int i = 0; i < nKeys; i++)
	{
		*((CryBoneKey*)&cd->data[ofs]) = keys[i];
		ofs += sizeof(CryBoneKey);
	}
}

void ChunkFile::MergeKeys( std::vector<CryBoneKey> &keys1,AnimRange range1,std::vector<CryBoneKey> &keys2,AnimRange range2 )
{
	int timeShift = (range2.end - range2.start) - (range1.end - range1.start);

	int k1 = keys1.size();
	int k2 = keys1.size();
	// convert 
	unsigned i;
	for (i = 0; i < keys1.size(); i++)
	{
		if (keys1[i].time >= range1.start) {
			k1 = i;
			break;
		}
	}
	for (i = k1; i < keys1.size(); i++)
	{
		if (keys1[i].time > range1.end) {
			k2 = i;
			break;
		}
	}

	std::vector<CryBoneKey> nkeys;
	nkeys.insert( nkeys.end(),keys1.begin(),keys1.begin()+k1 );

	for (i = 0; i < keys2.size(); i++)
	{
		if (keys2[i].time >= range2.start && keys2[i].time <= range2.end)
		{
			keys2[i].time += range1.start - range2.start;
			nkeys.push_back( keys2[i] );
		}
	}

	// Shift time, at keys after needed range.
	for (i = k2; i < keys1.size(); i++)
	{
		keys1[i].time += timeShift;
		nkeys.push_back( keys1[i] );
	}

	keys1 = nkeys;
}

void ChunkFile::MergeAnim( const char *cafFile,const char *animFile,const char *animName,bool bExportNew )
{
	ChunkFile acf;
	int AnimOffset = 10;

	unsigned i;
	Read( cafFile );
	
	acf.Read( animFile );

	RANGE_ENTITY *range = 0;
	int rangeId = -1;

	int startFrame = timeChunk.global_range.end;
	int endFrame = timeChunk.global_range.end;

	for (i = 0; i < m_ranges.size(); i++)
	{
		if (stricmp(m_ranges[i].name,animName)==0)
		{
			range = &m_ranges[i];
			rangeId = i;
			startFrame = range->start;
			endFrame = range->end;
		}
	}

	if (!range)
	{
		RANGE_ENTITY re;
		strcpy( re.name,animName );
		if (m_ranges.empty())
		{
			timeChunk.global_range.start = acf.timeChunk.global_range.start;
			timeChunk.global_range.end = acf.timeChunk.global_range.start;
		}
		re.start = timeChunk.global_range.end + AnimOffset;
		re.end = re.start + (acf.timeChunk.global_range.end - acf.timeChunk.global_range.start);
		m_ranges.push_back( re );
		//timeChunk.global_range.end += acf.timeChunk.global_range.end - acf.timeChunk.global_range.start;
		timeChunk.global_range.end = re.end;

		startFrame = re.start;
		endFrame = re.end;
	}
	else
	{
		int prevDt = range->end - range->start;
		//range->start = timeChunk.global_range.end;
		range->end = range->start + (acf.timeChunk.global_range.end - acf.timeChunk.global_range.start);
		int dt = (range->end - range->start) - prevDt;
		timeChunk.global_range.end += dt;
		// shift time.
		for (i = rangeId+1; i < m_ranges.size(); i++)
		{
			m_ranges[i].start += dt;
			m_ranges[i].end += dt;
		}
	}

	//////////////////////////////////////////////////////////////////////////
	// Marger animations.
	AnimRange r1,r2;
	r1.start = startFrame * timeChunk.TicksPerFrame;
	r1.end = endFrame * timeChunk.TicksPerFrame;
	r2.start = acf.timeChunk.global_range.start * acf.timeChunk.TicksPerFrame;
	r2.end = acf.timeChunk.global_range.end * acf.timeChunk.TicksPerFrame;
	ChunkFile &basefile = bExportNew ? acf : *this;

	std::vector<CryBoneKey> keys;
	std::vector<CryBoneKey> keys2;
	for(i=0;i<m_chunks.size();i++)
	{
		if (m_chunks[i].hdr.ChunkType == ChunkType_Controller) 
		{
			// Insert.
			LoadControllerKeys( &m_chunks[i],keys );
			
			ChunkDesc *cd = acf.FindChunkById( m_chunks[i].hdr.ChunkID );
			if (cd->hdr.ChunkType == ChunkType_Controller) {
				LoadControllerKeys( cd,keys2 );
				MergeKeys( keys,r1,keys2,r2 );
			} else if (bExportNew)
				continue;
			SaveControllerKeys(bExportNew ? cd:&m_chunks[i], keys );
		}
	}

	basefile.timeChunk = timeChunk;
	basefile.SaveTimeChunk();

	basefile.Write( cafFile );
}

void ChunkFile::SaveCalFile( const char *cafFile )
{
	// Save .cal file.
	char calFile[_MAX_PATH];
	char drive[_MAX_DRIVE];
	char dir[_MAX_DIR];
	char fname[_MAX_FNAME];
	char fext[_MAX_EXT];

	_splitpath( cafFile,drive,dir,fname,fext );
	_makepath( calFile,drive,dir,fname,".cal" );

	FILE *f = fopen( calFile,"wt" );
	if (!f)
		return;

	fprintf( f,"0-0 s 1 DoNothing\n" );
	fprintf( f,"0-0 s 1 DoNothingLoop\n" );
	for (unsigned i = 0; i < m_ranges.size(); i++)
	{
		fprintf( f,"%d-%d s 1 %s\n",m_ranges[i].start,m_ranges[i].end,m_ranges[i].name );
	}
	fclose(f);
}