//===========================================================================*\
//	Part of Crytek Character Studio and Object Export Plug-in
//
//  Copyright:  Cuneyt Ozdas 2000, 2001
//				 cuneyt@cuneytozdas.com
//===========================================================================*/

#include "stdafx.h"

#include "Utility.h"
#include "interpik.h"
#include <iskin.h>
#include <iiksys.h>
#include <Shlobj.h>
#include <set>

#include "Crc32.h"
#include "ChunkFile.h"

#include "CryKeyInterpolation.h"
 
#include "CryMeshCompact.h"
#include "ModifierUtils.h"

#if ( MAX_PRODUCT_VERSION_MAJOR == 6 )
#	include "Morpher6/Include/wm3.h"
#elif ( MAX_PRODUCT_VERSION_MAJOR == 5 )
#	include "Morpher5\wm3.h"
#elif( MAX_PRODUCT_VERSION_MAJOR == 4 )
#	include "MorpherAPI\Include\wm3.h"
#else
#	error Unsupported Max version (not 4, 5 or 6)
#endif

#include "CryShader\CryShader.h"
#include "AdjustRotLog.h"

//#define EXPORT_INIT_POSE

#ifndef M_PI
#define M_PI 3.1415926535897932384626433832795
#endif

#define max(a,b) (((a) > (b)) ? (a) : (b))
#define min(a,b) (((a) < (b)) ? (a) : (b))

extern char* BuildNo;	
typedef int (* CryCallbackFunc)(HWND, char *);
static Class_ID BipedClassID(0x9125,0x0);
static Class_ID stdUVClassID(STDUV_CLASS_ID,0);

const int ID_SKIP_NODE = 0x1001;

//from uvgen.cpp
//==============
const int PB_UOFFS		= 0;
const int PB_VOFFS		= 1;
const int PB_USCL		= 2;
const int PB_VSCL		= 3;
const int PB_UANGLE		= 4;
const int PB_VANGLE		= 5;
const int PB_WANGLE		= 6;
const int PB_BLUR		= 7;
const int PB_NSAMT		= 8;
const int PB_NSSIZ		= 9;
const int PB_NSLEV		= 10;
const int PB_NSPHS		= 11;
const int PB_BLUROFFS	= 12;

StdUVGen *GetTextureStdUVGen(Texmap* tex)
{
	if(!tex) return NULL;

	//get UVGen
	UVGen		*uvgen	= tex->GetTheUVGen();
	if(!uvgen)							return NULL;
	if(uvgen->ClassID()!=stdUVClassID)	return NULL;
	
	//Get UV pblock
	return (StdUVGen *)uvgen;
};

char s[2048];	//general purpose string buffer
bool bPreviewMode = false;

//flags for DLL Extension Calls
#define DLL_FLAG_GEOM 1
#define DLL_FLAG_ANIM 2

/*
//====================================================================
//====================================================================
//====================================================================
#define TIME_BOMB_DAY	20
#define TIME_BOMB_MONTH	4
#define TIME_BOMB_YEAR	2001

void GetMySystemTime(LPSYSTEMTIME time)
{
	GetSystemTime(time);
}

HWND AddMyRollupPage(Interface *ip, HINSTANCE hInst, TCHAR *dlgTemplate, DLGPROC dlgProc, TCHAR *title, LPARAM param=0,DWORD flags=0 )
{
	return ip->AddRollupPage(hInst, dlgTemplate,dlgProc,title,param,flags);
}

BYTE *secure_start;
BYTE *secure_stop;
bool CheckIntegrity()
{
	int xor=0;
	int tot=0;
	for(BYTE *i=secure_start;i<=secure_stop;i++)
	{
		xor ^= (*i);
		tot += (*i);
	}

	DebugPrint("\nx=%02x y=%04x",xor,tot);

	return ((xor==0x72) && (tot==0x2c18));
}
*/

int CryMessageBox( HWND hWnd,LPCTSTR lpText,LPCTSTR lpCaption,UINT uType)
{
	if (bPreviewMode)
		return -1;
	return MessageBox( hWnd,lpText,lpCaption,uType );
}

// finds the node in the array, and returns the index of the given node within the array.
// returns -1 if the node was not found
int FindNode (const INodeTab& tabNodes, INode* pNode)
{
	for (int i = 0; i < tabNodes.Count(); ++i)
	{
		if (tabNodes[i] == pNode)
			return i;
	}
	return -1;
}

//====================================================================
//====================================================================
//====================================================================
//! Doxygen text by cuneyt
void Matrix3_to_CryMatrix(Matrix44 &cm, Matrix3 &m3)
{
	MRow *arow=m3.GetAddr();
	cm[0][0] = arow[0][0];
	cm[0][1] = arow[0][1];
	cm[0][2] = arow[0][2];
	cm[0][3] = 0.0f;
	cm[1][0] = arow[1][0];
	cm[1][1] = arow[1][1];
	cm[1][2] = arow[1][2];
	cm[1][3] = 0.0f;
	cm[2][0] = arow[2][0];
	cm[2][1] = arow[2][1];
	cm[2][2] = arow[2][2];
	cm[2][3] = 0.0f;
	cm[3][0] = arow[3][0];
	cm[3][1] = arow[3][1];
	cm[3][2] = arow[3][2];
	cm[3][3] = 1.0f;
};

void CryMatrix_to_Matrix3(Matrix3 &m3, Matrix44 &cm)
{
	MRow *arow=m3.GetAddr();
	arow[0][0]=cm[0][0]; 
	arow[0][1]=cm[0][1]; 
	arow[0][2]=cm[0][2]; 
	arow[1][0]=cm[1][0]; 
	arow[1][1]=cm[1][1]; 
	arow[1][2]=cm[1][2]; 
	arow[2][0]=cm[2][0]; 
	arow[2][1]=cm[2][1]; 
	arow[2][2]=cm[2][2]; 
	arow[3][0]=cm[3][0]; 
	arow[3][1]=cm[3][1]; 
	arow[3][2]=cm[3][2]; 
};


/*===========================================================================*\
 |	Class Descriptor
\*===========================================================================*/
static	CSExportUtilClassDesc	CSExportUtilCD;
		CSExportUtility			theCSExportUtility;

ClassDesc*	GetCSExportUtilDesc()							{return &CSExportUtilCD;}
void *		CSExportUtilClassDesc::Create( BOOL loading )	{return &theCSExportUtility;}


// Reset all the utility values on File/Reset
/*
void CSExportUtilClassDesc::ResetClassParams (BOOL fileReset) 
{
	DebugPrint("\n\n*************\nReset is called\n****************\n");
}
*/

// -------------------------------------------
// parameter validator - class declaration
// -------------------------------------------
class NodeValidator : public PBValidator
{
public:
	CSExportUtility *util;
	BOOL Validate(PB2Value &v)
	{
		Interval ivalid;

		//Check if it is of one of the supported types
		INode *node=(INode*)(v.r);
		ObjectState os = node->EvalWorldState(0);
		SClass_ID sid=os.obj->SuperClassID();
		Class_ID  cid=os.obj->ClassID();

		switch(sid)
		{
			case GEOMOBJECT_CLASS_ID:
				{
					//check if this is a biped object: if so reject it.	
					if(cid == BipedClassID ) return false;

					//reject light and camera targets
					if(node->IsTarget())
					{
						INode *parent=node->GetLookatNode();
						if(parent)
						{
							ObjectState pos = parent->EvalWorldState(0);
							if(pos.obj)
							{
								SClass_ID psid=pos.obj->SuperClassID();
								if((psid==LIGHT_CLASS_ID) || (psid==CAMERA_CLASS_ID)) return false;
							}
						}
					}
				}
				//fall through
			case LIGHT_CLASS_ID:
			case CAMERA_CLASS_ID:
				break;

			case HELPER_CLASS_ID:
				{
					if(	cid==Class_ID(DUMMY_CLASS_ID, 0) || 
						cid==Class_ID(POINTHELP_CLASS_ID, 0)) 
						break;
				}
				//fall through

			default:
				return false;		
		}

		//check for dublicate entry
		bool nodeexists=false;
		int nNodes=util->pb_nodes->Count(PB_NODES);
		for(int i=0;i<nNodes;i++)
		{
			INode *node = util->pb_nodes->GetINode(PB_NODES,0,i);
			if(!node) continue;
			//DebugPrint("\n%s",node->GetName());
			if(v.r == node) nodeexists=true;
		}
		
		return !nodeexists;
	}
};

class BoneValidator : public PBValidator
{
public:
	CSExportUtility *util;
	INode *LastValidatedNode;
	char LastValidatedName[512];
	BoneValidator() {LastValidatedNode=NULL;}
	BOOL Validate(PB2Value &v)
	{
		LastValidatedNode=NULL;

		//check for dublicate entry (if one of our parents is in the list, we will not be accepted)
		bool nodeexists=false;
		int nNodes=util->pb_bones->Count(PB_BONES);
		
		INode *node=(INode *)v.r;
		if(!node) return false;

		//DebugPrint("Validating \"%s\" ...",node->GetName());
		while(node && !nodeexists)
		{
			for(int i=0;i<nNodes;i++)
			{
				INode *check = util->pb_bones->GetINode(PB_BONES,0,i);
				if(!check) continue;
				//DebugPrint("\n%s",node->GetName());
				if(node == check)
				{
					nodeexists=true;
					//DebugPrint("x\n");
					break;
				}
			}
			node=node->GetParentNode();
		}
		
		if(!nodeexists)
		{	
			LastValidatedNode=(INode *)v.r;
			if(LastValidatedNode)
			{
				strcpy(LastValidatedName,LastValidatedNode->GetName());
			}
			else LastValidatedName[0]=0;
			//DebugPrint("OK\n");
		}
		return !nodeexists;
	}
};

NodeValidator node_validator;
BoneValidator bone_validator;

static ParamBlockDesc2 nodes_param_blk ( pbid_Nodes, _T("CSNodesParams"),  0, &CSExportUtilCD, P_AUTO_CONSTRUCT + P_AUTO_UI, 0, 
	//rollout
	IDD_NODES, IDS_OBJECTS, 0, 0, NULL, 
	// params
	PB_NODES,			_T("Nodes"),		TYPE_INODE_TAB, 0, P_AUTO_UI ,	IDS_NONAME,
		p_ui,			TYPE_NODELISTBOX,	IDC_OBJ_LIST,IDC_NODE_ADD,0,IDC_NODE_REMOVE,
		p_validator,	&node_validator,
		end,
	PB_ADD_RELATED,		_T("AddRelatedBones"),	TYPE_BOOL, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX, 	IDC_ADD_RELATED_BONES, 
		p_enable_ctrls,	1,PB_ADD_WHOLE_SKEL,
		end, 
	PB_ADD_WHOLE_SKEL,	_T("AddWholeSkeleton"),	TYPE_BOOL, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX, 	IDC_ADD_WHOLE_SKEL, 
		end, 
	PB_EXPORT_MAT,		_T("ExportMaterial"),	TYPE_BOOL, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX, 	IDC_WRITE_MATERIALS, 
		end, 
	PB_ALLOW_MULTIUV,	_T("AllowMultiUV"),		TYPE_BOOL, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX, 	IDC_ALLOW_MULTIUV, 
		end, 
	PB_WRITE_VCOL,		_T("WriteVertexColors"),TYPE_BOOL, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX, 	IDC_WRITE_VCOL, 
		end, 
	PB_WRITE_WEIGHTS,	_T("WriteWeights"),		TYPE_BOOL, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX, 	IDC_WRITE_WEIGHTS, 
		p_enable_ctrls,	1,PB_LINK_TYPE,
		end, 
	PB_LINK_TYPE,		_T("LinkType"),			TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_RADIO, 2, 	IDC_1LINK, IDC_NLINKS,
		end, 
	PB_GENERATE_DEFAULT_UVS, _T("GenerateDefaultUVs"), TYPE_BOOL, 0, IDS_NONAME,
		p_default,    1,
		p_ui,       TYPE_SINGLECHEKBOX, IDC_GENERATE_DEFAULT_UVS,
		end,
	PB_INDIVIDUAL_FILES, _T("ExportToIndividualFiles"), TYPE_BOOL, 0, IDS_NONAME,
		p_default, 0,
		p_ui, TYPE_SINGLECHEKBOX, IDC_INDIVIDUALFILES,
		end,
	PB_WRITE_VA,		_T("WriteVertexAnim"),	TYPE_BOOL, 		0,	IDS_NONAME,
		p_default, 		0, 
		p_ui, 			TYPE_SINGLECHEKBOX, 	IDC_WRITE_VERT_ANIM, 
		//p_enabled,		FALSE,
		p_enable_ctrls,	1, PB_VERT_ANIM_RATE,
		end,
	PB_VERT_ANIM_RATE,	_T("VertexAnimRate"),		TYPE_INT,		0,	IDS_NONAME,
		p_default,		1,
		p_range, 		1, 10000, 
		p_ui,			TYPE_SPINNER, EDITTYPE_INT, IDC_VERT_ANIM_FR_EDIT, IDC_VERT_ANIM_FR_SPIN, SPIN_AUTOSCALE,
		end,
	PB_BUILDING,		_T("Building"),	TYPE_BOOL, 		0,	IDS_NONAME,
		p_default, 		0, 
		p_ui, 			TYPE_SINGLECHEKBOX, 	IDC_BUILDING, 
		end,
	PB_EXPORT_GEOM,	_T("ExportGeometry"),	TYPE_BOOL, 		0,	IDS_NONAME,
		p_default, 		TRUE, 
		p_ui, 			TYPE_SINGLECHEKBOX, 	IDC_WRITE_GEOMETRY,
		end, 
	PB_MORPH_MIN_OFFSET,	_T("MorphMinOffset"),		TYPE_FLOAT,		0,	IDS_NONAME,
		p_default,		0.1f,
		p_range, 		0.0f, 100.0f, 
		p_ui,			TYPE_SPINNER, EDITTYPE_FLOAT, IDC_MORPH_MIN_OFFSET_EDIT, IDC_MORPH_MIN_OFFSET_SPIN, SPIN_AUTOSCALE,
		end,
	end
	);

static ParamBlockDesc2 bones_param_blk ( pbid_Bones, _T("CSBonesParams"),  0, &CSExportUtilCD, P_AUTO_CONSTRUCT + P_AUTO_UI, 1, 
	//rollout
	IDD_BONES, IDS_EXPORT, 0, 0, NULL, 
	// params
	PB_BONES,			_T("Bones"),			TYPE_INODE_TAB, 0, P_AUTO_UI,	IDS_NONAME,
		p_ui,			TYPE_NODELISTBOX,		IDC_BONE_LIST,IDC_BONE_ADD,0,IDC_BONE_REMOVE,
		p_validator,	&bone_validator,
		end,
	PB_BONE_ANIM_ATKEYS,_T("BoneAnimAtKeys"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_BONE_ANIM_KEYS,
		p_enable_ctrls,	3, PB_CHILD_TIMES,PB_PARENT_TIMES,PB_STABILIZE_FEET,
		end, 
	PB_CHILD_TIMES,		_T("ChildKeyTimes"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		0, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_CHILD_TIMES,
		end, 
	PB_PARENT_TIMES,	_T("PArentKeyTimes"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		0, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_PARENT_TIMES,
		end, 
	PB_STABILIZE_FEET,	_T("StabilizeFeet"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_STABILIZE_FEET,
		end, 
	PB_BONE_ANIM_EVERY, _T("BoneAnimEvery"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		0, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_BONE_ANIM_EVERY,
		p_enable_ctrls,	1, PB_BONE_ANIM_RATE, 
		end, 
	PB_BONE_ANIM_RATE,	_T("BoneAnimRate"),		TYPE_INT,		0,	IDS_NONAME,
		p_default,		5,
		p_range, 		1, 10000, 
		p_ui,			TYPE_SPINNER, EDITTYPE_INT, IDC_BONE_ANIM_FR_EDIT, IDC_BONE_ANIM_FR_SPIN, SPIN_AUTOSCALE,
		end,
	PB_IGNORE_DUMMY,	_T("IgnoreDummyBones"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_IGNORE_MESH_BONES,
		end,
	PB_KEY_CLEANUP_ENABLE, _T("EnableKeyCleanup"), TYPE_INT, 0, IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_ENABLE_KEY_CLEANUP,
		p_enable_ctrls,	2, PB_KEY_CLEANUP_ROTATION, PB_KEY_CLEANUP_POSITION,
		end,
	PB_KEY_CLEANUP_ROTATION, _T("KeyCleanupRotation"), TYPE_FLOAT, 0, IDS_NONAME,
		p_default,    5.0f,
		p_range,      1.0f, 20.0f,
		p_ui,         TYPE_SPINNER, EDITTYPE_FLOAT, IDC_KEY_CLEANUP_ROTATION, IDC_KEY_CLEANUP_ROTATION_SPIN, 0.1f,
		end,
	PB_KEY_CLEANUP_POSITION, _T("KeyCleanupPosition"), TYPE_FLOAT, 0, IDS_NONAME,
		p_default,    3.5f,
		p_range,      1.0f, 20.0f,
		p_ui,         TYPE_SPINNER, EDITTYPE_FLOAT, IDC_KEY_CLEANUP_POSITION, IDC_KEY_CLEANUP_POSITION_SPIN, 0.1f,
		end,
	end
	);

static ParamBlockDesc2 timing_param_blk ( pbid_Timing, _T("CSTiming"),  0, &CSExportUtilCD, P_AUTO_CONSTRUCT + P_AUTO_UI, 3, 
	//rollout
	IDD_TIMING, IDS_TIMING, 0, 0, NULL, 
	// params
	PB_MAN_RANGE,		_T("ManualRange"),		TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		0, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_MANUAL_RANGE,
		p_enable_ctrls,	2, PB_RANGE_START, PB_RANGE_END,
		end, 
	PB_RANGE_START,		_T("RangeStart"),		TYPE_INT,		0,	IDS_NONAME,
		p_default,		0,
		p_range, 		-10000, 10000, 
		p_ui,			TYPE_SPINNER, EDITTYPE_INT, IDC_ANIM_START_EDIT, IDC_ANIM_START_SPIN, SPIN_AUTOSCALE,
		end,
	PB_RANGE_END,		_T("RangeEnd"),		TYPE_INT,			0,	IDS_NONAME,
		p_default,		100,
		p_range, 		-10000, 10000, 
		p_ui,			TYPE_SPINNER, EDITTYPE_INT, IDC_ANIM_END_EDIT, IDC_ANIM_END_SPIN, SPIN_AUTOSCALE,
		end,
	end
	);

static ParamBlockDesc2 options_param_blk ( pbid_Options, _T("CSOptions"),  0, &CSExportUtilCD, P_AUTO_CONSTRUCT + P_AUTO_UI, 2, 
	//rollout
	IDD_OPTIONS, IDS_OPTIONS, 0, APPENDROLL_CLOSED, NULL, 
	// params
	PB_AUTO_SAVE,		_T("AutoSaveConfig"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		0, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_AUTO_SAVE,
		end, 
	PB_WARN_NONRIGID,	_T("WarnNonRigid"),		TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_WARN_NONRGID,
		end, 
	PB_WARN_ORPHAN_TV,	_T("WarnOrphanTV"),		TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_WARN_ORPHAN_TV,
		end, 
	PB_WARN_DLL,		_T("WarnDLLErrors"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_WARN_DLL,
		end, 
	PB_WARN_MULTI_VCOL,	_T("WarnMultiVColor"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_WARN_MULTI_VCOL,
		end, 
	PB_WARN_OFFAXIS_SCL,_T("OffAxisScaling"),	TYPE_INT, 		0,	IDS_NONAME,
		p_default, 		1, 
		p_ui, 			TYPE_SINGLECHEKBOX,		IDC_WARN_OFFAXIS_SCL,
		end, 
	end
	);

//ProgressDlgProc  ___________________________________________________________________________________________
BOOL CALLBACK ProgressDlgProc(HWND hWnd,UINT msg,WPARAM wParam,LPARAM lParam) 
{
	switch (msg) 
	{
		case WM_INITDIALOG:
			break;

		case WM_LBUTTONDOWN: case WM_LBUTTONUP: case WM_MOUSEMOVE:
			return FALSE;

		default: 
			return FALSE;
	}
	return TRUE;
}

void ResetReadOnlyFlag (const char* filename)
{
	DWORD fileflags = GetFileAttributes(filename);
	
	if ((fileflags != 0xffffffff)  && (fileflags & FILE_ATTRIBUTE_READONLY))
	{
		if (MessageBox(NULL,"This file is marked read-only. Do you still want to overwrite it?","Read Only Warning",MB_YESNO) == IDYES)
		{
			fileflags &= ~FILE_ATTRIBUTE_READONLY;
			SetFileAttributes(filename,fileflags);
		}
	}
}

void CreateDirectoryForFile (const char* szFileName)
{
	int nLen = strlen (szFileName);
	char* szTmp = new char[nLen];
	for (int i = 0; i < nLen; ++i)
	{
		if (szFileName[i] == '\\' || szFileName [i] == '/')
		{
			memcpy (szTmp, szFileName, i);
			szTmp[i] = '\0';
			if (GetFileAttributes (szTmp) == INVALID_FILE_ATTRIBUTES)
				CreateDirectory (szTmp, NULL);
		}
	}
	delete []szTmp;
}

void WarningFileNotSaved (HWND hParent, const TSTR& strFileName)
{
	MessageBox (hParent, (const char*)(strFileName + " file was not saved"), "Warning", MB_OK|MB_ICONASTERISK);
}

//OptionsDlgProc  ___________________________________________________________________________________________
BOOL OptionsDlgProc::DlgProc(TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg) 
	{
		case WM_INITDIALOG:
			sprintf(s,"Build No: %s",BuildNo);
			SetWindowText(GetDlgItem(hWnd,IDC_BUILD_NO),s );
			break;

		case WM_COMMAND:
			switch (LOWORD(wParam)) 
			{
				case IDC_LOAD:
					tsu->ConfigLoad(false);
					break;

				case IDC_SAVE:
					tsu->ConfigSave(false);
					break;

				case IDC_RESET:
					if(IDYES==MessageBox(hWnd,"This command will set all of the parameters to the factory defaults\nAre you sure ?","Are you Sure?",MB_YESNO | MB_ICONQUESTION))
					{
						CSExportUtilCD.Reset(tsu);
					}
					break;

				case IDC_SAVE_DEFAULT:
					int old=tsu->pb_options->GetInt(PB_AUTO_SAVE);
					tsu->pb_options->SetValue(PB_AUTO_SAVE,0,1);
					tsu->ConfigSave(true);
					tsu->pb_options->SetValue(PB_AUTO_SAVE,0,old);
					break;
			}
			break;

		default: 
			return FALSE;
	}
	return TRUE;
}



BOOL CALLBACK NamedRangesDlgProc(HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);

//TimingDlgProc  ___________________________________________________________________________________________
BOOL TimingDlgProc::DlgProc(TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static HWND dlgwnd=NULL;
	switch (msg) 
	{
	case WM_INITDIALOG:
		//SetDlgItemText( hWnd,IDC_ANIM_NAME,tsu->animationName );
		break;

 	case  WM_DESTROY:
		if(dlgwnd) DestroyWindow(dlgwnd);
		dlgwnd=NULL;
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) 
		{
			case IDC_EDIT_NAMED_SEG:
				if(!dlgwnd)
				{
					HWND parent = tsu->ip->GetMAXHWnd();
					//int res=DialogBoxParam(hInstance,  MAKEINTRESOURCE(IDD_NAMED_RANGES),parent, NamedRangesDlgProc, LPARAM (tsu));
					//if(res==-1) res=GetLastError();
					dlgwnd=CreateDialogParam(hInstance,  MAKEINTRESOURCE(IDD_NAMED_RANGES),parent, NamedRangesDlgProc, LPARAM (tsu));
				}
				ShowWindow(dlgwnd,SW_SHOW);
				break;
			case IDC_EXPORTNEW:
				//Timur[2/5/2002] tsu->bExportNew = IsDlgButtonChecked( hWnd,IDC_EXPORTNEW )!=0;
				break;
			case IDC_ANIM_NAME:
				/*
				if (HIWORD(wParam) == EN_CHANGE)
				{
					char animName[1024];
					GetDlgItemText( hWnd,IDC_ANIM_NAME,animName,sizeof(animName) );
					tsu->animationName = animName;
				}
				if (HIWORD(wParam) == EN_SETFOCUS)
				{
					DisableAccelerators();
				}
				if (HIWORD(wParam) == EN_KILLFOCUS)
				{
					EnableAccelerators();
				}
				*/
				break;
		}

	default: 
		return FALSE;
	}
	return TRUE;
}

//NodesDlgProc  ___________________________________________________________________________________________
BOOL NodesDlgProc::DlgProc(TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	int id = LOWORD(wParam);
	switch (msg) 
	{
	case WM_INITDIALOG:
		{
			//tsu->Init(hWnd);
			ICustButton *but=GetICustButton(GetDlgItem(hWnd,IDC_EXPORT_NODES));
			but->SetImage(tsu->ButtonImages,0,0,0,0,125,20);
			ReleaseICustButton(but);
		}
		break;

	case WM_DESTROY:
		//tsu->Destroy(hWnd);
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) 
		{
		case IDC_EXPORT_NODES:
			{
			tsu->hProgressWindow= CreateDialog(hInstance, MAKEINTRESOURCE(IDD_PROGRESS), tsu->hwnd_nodes, ProgressDlgProc);
			tsu->hProgressBar	= GetDlgItem(tsu->hProgressWindow, IDC_PROGRESS);
			RECT maxrect, myrect;
			GetWindowRect(tsu->ip->GetMAXHWnd(), &maxrect);
			GetWindowRect(tsu->hProgressWindow, &myrect);
			SetWindowPos(tsu->hProgressWindow, HWND_TOP, (maxrect.right+maxrect.left-myrect.right+myrect.left)>>1, (maxrect.bottom+maxrect.top-myrect.bottom+myrect.top)>>1,0,0,SWP_NOSIZE);
			ShowWindow(tsu->hProgressWindow, SW_SHOW);
			
			int nSaved = tsu->SaveGeomFile("");

			DestroyWindow(tsu->hProgressWindow);

			tsu->hProgressBar = tsu->hProgressWindow = NULL;

				if (nSaved == IDNO)
					WarningFileNotSaved (tsu->ip->GetMAXHWnd(), "");
			}

			break;

		case IDC_PREVIEW_NODES:
			{
				tsu->hProgressWindow= CreateDialog(hInstance, MAKEINTRESOURCE(IDD_PROGRESS), tsu->hwnd_nodes, ProgressDlgProc);
				tsu->hProgressBar	= GetDlgItem(tsu->hProgressWindow, IDC_PROGRESS);
				RECT maxrect, myrect;
				GetWindowRect(tsu->ip->GetMAXHWnd(), &maxrect);
				GetWindowRect(tsu->hProgressWindow, &myrect);
				SetWindowPos(tsu->hProgressWindow, HWND_TOP, (maxrect.right+maxrect.left-myrect.right+myrect.left)>>1, (maxrect.bottom+maxrect.top-myrect.bottom+myrect.top)>>1,0,0,SWP_NOSIZE);
				ShowWindow(tsu->hProgressWindow, SW_SHOW);
				
				bPreviewMode = true;
				int nSaved = tsu->SaveGeomFile( tsu->lastSavedFile );
				bPreviewMode = false;
				
				DestroyWindow(tsu->hProgressWindow);
				
				tsu->hProgressBar = tsu->hProgressWindow = NULL;
				
				switch (nSaved)
				{
				case IDYES:
					tsu->PreviewGeomFile( tsu->lastSavedFile );
					break;
				case IDCANCEL:
					MessageBox (tsu->ip->GetMAXHWnd(),
						(const char*)(tsu->lastSavedFile + " file was not saved because the user cancelled the action"), "Warning", MB_OK|MB_ICONASTERISK);
					break;
					break;
				case IDNO:
					WarningFileNotSaved (tsu->ip->GetMAXHWnd(), tsu->lastSavedFile);
					break;
				default:
					{
						char szError[100];
						sprintf (szError, "%d", nSaved);
					MessageBox (tsu->ip->GetMAXHWnd(),
						(const char*)(tsu->lastSavedFile + " file was not saved because of unknown error " + szError), "Warning", MB_OK|MB_ICONASTERISK);
					}
					break;
				}
			}
			break;

		case IDC_PREVIEW_GAME:
			{
				tsu->hProgressWindow= CreateDialog(hInstance, MAKEINTRESOURCE(IDD_PROGRESS), tsu->hwnd_nodes, ProgressDlgProc);
				tsu->hProgressBar	= GetDlgItem(tsu->hProgressWindow, IDC_PROGRESS);
				RECT maxrect, myrect;
				GetWindowRect(tsu->ip->GetMAXHWnd(), &maxrect);
				GetWindowRect(tsu->hProgressWindow, &myrect);
				SetWindowPos(tsu->hProgressWindow, HWND_TOP, (maxrect.right+maxrect.left-myrect.right+myrect.left)>>1, (maxrect.bottom+maxrect.top-myrect.bottom+myrect.top)>>1,0,0,SWP_NOSIZE);
				ShowWindow(tsu->hProgressWindow, SW_SHOW);
				
				bPreviewMode = true;
				int nSaved = tsu->SaveGeomFile( tsu->lastSavedFile );
				bPreviewMode = false;
				
				DestroyWindow(tsu->hProgressWindow);
				
				tsu->hProgressBar = tsu->hProgressWindow = NULL;
				
				switch (nSaved)
				{
				case IDYES:
					tsu->PreviewGame( tsu->lastSavedFile );
					break;
				case IDNO:
					WarningFileNotSaved (tsu->ip->GetMAXHWnd(), tsu->lastSavedFile);
					break;
				}
			}
			break;
		
		case IDC_NODE_CLEAR:
			tsu->pb_nodes->ZeroCount(PB_NODES);
			break;

		case IDC_NODE_ADD_SEL:
			{
				int n=tsu->ip->GetSelNodeCount();
				for(int i=0;i<n;i++)
				{
					INode *node=tsu->ip->GetSelNode(i);
					PB2Value v;
					v.r=node;
					if(node_validator.Validate(v))
					{
						tsu->pb_nodes->Append(PB_NODES,1,&node);
					}
				}
			}
			break;
		}
		break;
	}
	return FALSE;
}

//BonesDlgProc  ___________________________________________________________________________________________
BOOL BonesDlgProc::DlgProc(TimeValue t, IParamMap2 *map, HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	int id = LOWORD(wParam);
	switch (msg) 
	{
	case WM_INITDIALOG:
		{
			ICustButton *but=GetICustButton(GetDlgItem(hWnd,IDC_EXPORT_BONES));
			but->SetImage(tsu->ButtonImages,1,1,1,1,125,20);
			ReleaseICustButton(but);
		}
		break;

	case WM_DESTROY:
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) 
		{
		case IDC_EXPORT_BONES:
		{
			tsu->hProgressWindow= CreateDialog(hInstance, MAKEINTRESOURCE(IDD_PROGRESS), tsu->hwnd_bones, ProgressDlgProc);
			tsu->hProgressBar	= GetDlgItem(tsu->hProgressWindow, IDC_PROGRESS);
			RECT maxrect, myrect;
			GetWindowRect(tsu->ip->GetMAXHWnd(), &maxrect);
			GetWindowRect(tsu->hProgressWindow, &myrect);
			SetWindowPos(tsu->hProgressWindow, HWND_TOP, (maxrect.right+maxrect.left-myrect.right+myrect.left)>>1, (maxrect.bottom+maxrect.top-myrect.bottom+myrect.top)>>1,0,0,SWP_NOSIZE);
			ShowWindow(tsu->hProgressWindow, SW_SHOW);
			
			int nResult = tsu->SaveBoneAnimFile();

			DestroyWindow(tsu->hProgressWindow);

			tsu->hProgressBar = tsu->hProgressWindow = NULL;

			switch (nResult)
			{
			case IDNO:
				WarningFileNotSaved (tsu->ip->GetMAXHWnd(), "");
				break;
			}
		}

		break;
		
		case IDC_BONE_CLEAR:
			tsu->pb_bones->ZeroCount(PB_BONES);
			break;

		case IDC_BONE_ADD_SEL:
			{
				int n=tsu->ip->GetSelNodeCount();
				for(int i=0;i<n;i++)
				{
					INode *node=tsu->ip->GetSelNode(i);
					PB2Value v;
					v.r=node;
					if(bone_validator.Validate(v))
					{
						tsu->pb_bones->Append(PB_BONES,1,&node);
					}
				}
			}
			break;
		}
		break;

	}
	return FALSE;
}

/*===========================================================================*\
 |  Utility implimentations
\*===========================================================================*/

static int once = 1;

CSExportUtility::CSExportUtility()
{
	m_nDefaultMtlIndex = -1;
	m_nDefaultMtlChunkId = -1;
	node_validator.util=this;
	bone_validator.util=this;
	
	iu			=NULL;
	ip			=NULL;
	pb_nodes	=NULL;
	pb_bones	=NULL;
	pmap_nodes	=NULL;
	pmap_bones	=NULL;
	hwnd_nodes	=NULL;
	hwnd_bones	=NULL;
	hwnd_about	=NULL;
	ChunkList.Reset();

	hProgressWindow	=NULL;
	hProgressBar	=NULL;
	bExportNew = false;
	//ActiveNode		=NULL;

	m_pidlLast = NULL;
}

CSExportUtility::~CSExportUtility()
{
	node_validator.util=NULL;
	bone_validator.util=NULL;
}

int	CSExportUtility::GetTime()
{
	// Always Export at 0 time
	return 0;

	//return ip->GetTime();
}

//  ______________________________________________________________________________________
void CSExportUtility::BeginEditParams(Interface *ip,IUtil *iu) 
{
	ButtonImages=ImageList_Create(125,20, ILC_COLOR24 | ILC_MASK  , 2, 2);
	assert(ButtonImages);
	HBITMAP hBitmap = LoadBitmap(hInstance, MAKEINTRESOURCE(IDB_BUTTONS));
	HBITMAP hMask	= LoadBitmap(hInstance, MAKEINTRESOURCE(IDB_BUTTONS_MASK));
	ImageList_Add(ButtonImages, hBitmap, hMask);
	DeleteObject(hBitmap);
	DeleteObject(hMask);

	//CSExportUtilCD.MakeAutoParamBlocks(this);

	if (once)
		CSExportUtilCD.MakeAutoParamBlocks(this);
	once = 0;

	this->iu = iu;
	this->ip = ip;

	//CSExportUtilCD.BeginEditParams((IObjParam*)ip, this, 0, this);
	CSExportUtilCD.BeginEditParams((IObjParam*)ip, this, 0, 0);

	bones_param_blk.SetUserDlgProc(new BonesDlgProc(this));
	nodes_param_blk.SetUserDlgProc(new NodesDlgProc(this));
	options_param_blk.SetUserDlgProc(new OptionsDlgProc(this));
	timing_param_blk.SetUserDlgProc(new TimingDlgProc(this));

	pmap_nodes	=pb_nodes->GetMap();
	pmap_bones	=pb_bones->GetMap();

	hwnd_nodes	=pmap_nodes->GetHWnd();
	hwnd_bones	=pmap_bones->GetHWnd();
	//hwnd_about	=ip->AddRollupPage(hInstance, MAKEINTRESOURCE(IDD_ABOUT), AboutDlgProc, "About");

	EnableDisableControls();

/*
#ifdef PROTECTION_ENABLED
#pragma message ("===========Protection in BeginEditParams Enabled===========")
	if(time_Bomb_fired || !CheckIntegrity())
	{
		ip->RegisterTimeChangeCallback((TimeChangeCallback *)13051972);
	}
#endif
*/

	ConfigLoad(true);
}
	
//  ______________________________________________________________________________________
void CSExportUtility::EndEditParams(Interface *ip,IUtil *iu) 
{
	//====================
	//Correct the MAX BUG: 
	//====================
	//Nuke the command mode if we are in Pick... state.
	//This is a MAX BUG: MAX fails to remove that command mode and it crashes.
	CommandMode* cmdMode = GetCOREInterface()->GetCommandMode();
	if((!cmdMode) || cmdMode->ID()==CID_STDPICK) GetCOREInterface()->SetStdCommandMode(CID_OBJSELECT);
		
	ConfigSave(true);

	if(hwnd_about) ip->DeleteRollupPage(hwnd_about);

	CSExportUtilCD.EndEditParams((IObjParam*)ip, this, END_EDIT_REMOVEUI, this);

	this->iu = NULL;
	this->ip = NULL;

	pmap_nodes=pmap_bones=NULL;
	hwnd_about=hwnd_nodes=hwnd_bones=NULL;

	ImageList_Destroy(ButtonImages);
}

//  ______________________________________________________________________________________
IParamBlock2* CSExportUtility::GetParamBlock(int i)
{ 
	switch(i)
	{
	case 0: return pb_nodes;
	case 1: return pb_bones;
	case 2: return pb_options;
	case 3: return pb_timing;
	}
	return NULL;
} // return i'th ParamBlock


//  ______________________________________________________________________________________
IParamBlock2* CSExportUtility::GetParamBlockByID(short id) 
{
	switch(id)
	{
	case pbid_Nodes:	return pb_nodes;
	case pbid_Bones:	return pb_bones;
	case pbid_Options:	return pb_options;
	case pbid_Timing:	return pb_timing;
	}
	return NULL; 
} // return ParamBlock given ID


/*===========================================================================*\
 |	Subanim & References support
\*===========================================================================*/

RefTargetHandle CSExportUtility::GetReference(int i)
{
	switch (i) 
	{
		case 0: return pb_nodes;
		case 1: return pb_bones;
		case 2: return pb_options;
		case 3: return pb_timing;
		default: return NULL;
	}
}

//  ______________________________________________________________________________________
void CSExportUtility::SetReference(int i, RefTargetHandle rtarg)
{
	switch (i) 
	{
		case 0:
			pb_nodes	= (IParamBlock2*)rtarg;
			break;

		case 1:
			pb_bones	= (IParamBlock2*)rtarg;
			break;

		case 2: pb_options	= (IParamBlock2*)rtarg; break;
		case 3: pb_timing	= (IParamBlock2*)rtarg; break;
	}
}

//  ______________________________________________________________________________________
RefResult CSExportUtility::NotifyRefChanged( Interval changeInt, RefTargetHandle hTarget,PartID& partID,  RefMessage message) 
{
	static bool avoid_recurse=false;

	switch (message) 
	{
		case REFMSG_CHANGE:
			nodes_param_blk.InvalidateUI();
			bones_param_blk.InvalidateUI();

			EnableDisableControls();

			if(hTarget == pb_nodes)
			{
				int last_id=pb_nodes->LastNotifyParamID();
				if((last_id == PB_NODES) || (last_id == PB_ADD_WHOLE_SKEL) || (last_id == PB_ADD_RELATED))
				{
					if(pb_nodes->GetInt(PB_ADD_RELATED))
					{
						AddRelatedBones();
					}
				}
			}
			else if(hTarget == pb_bones)
			{
				int last_id=pb_bones->LastNotifyParamID();
				if(last_id == PB_BONES) 
				{
					if(int n=pb_bones->Count(PB_BONES))
					{
						INode *LastNode=pb_bones->GetINode(PB_BONES,0,n-1);
						if(LastNode)
						{
							char *s=LastNode->GetName();
							if(!avoid_recurse)
							{
								avoid_recurse=true;
								RemoveChildBones(LastNode);
								avoid_recurse=false;
							}
						}
					}
				}
			}

			break;
	}
	return REF_SUCCEED;
}

//  ______________________________________________________________________________________
// IDYES - continue export
// IDNO - stop export, warn the user
// IDCANCEL - stop export, the user doesn't need to be warned
// ID_SKIP_NODE - continue the export, but skip the node
int CSExportUtility::CheckNodeProblems(INode *node)
{
	TimeValue time=GetTime();

	if (!node) return IDYES;
												 
	//get necessary node properties
	char		name[256];
	AffineParts ap;
	Point3		decomposed_p;
	Quat		decomposed_r;
	Point3		decomposed_s;
	Matrix3		tm				= node->GetObjTMAfterWSM(time);
	decomp_affine(tm,&ap);
	DecomposeMatrix(tm, decomposed_p, decomposed_r, decomposed_s);
	strcpy(name,node->GetName());
	
	//==================================
	//Check non-uniform off-axis scaling
	//==================================
	float EPSILON=1e-8f;
	float l=max(EPSILON ,Length(decomposed_s));
	if((Length(ap.k-decomposed_s)/l)>0.001f && pb_options->GetInt(PB_WARN_OFFAXIS_SCL))
	{
		if (!bPreviewMode)
		{
			int res=CryMessageBox(hwnd_nodes,"Node has off-axis non-uniform scaling\nDo you still want to export this node",name,MB_OKCANCEL);
			if(res==IDCANCEL) return IDNO;
		}
	}


	/*
	//======================
	//Check negative scaling
	//======================
	if(ap.f < 0.0f) 
	{
		ms->negative_scaling = true;
		DebugPrint("node %s has negative determinant: reversing the vertex order",name);
	}
	else
	{
		ms->negative_scaling = false;
	}
	*/

	//get mesh if this is a geom object
	Object		*der_obj= node->GetObjectRef();
	Object		*obj	= der_obj->Eval(time).obj;
	SClass_ID	sid		= obj->SuperClassID();
	Class_ID	cid		= obj->ClassID();

	if(sid != GEOMOBJECT_CLASS_ID) return IDYES;

	if(!obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0)))
	{
		if (bPreviewMode)
			return IDYES;
		int res=MessageBox(hwnd_nodes,"Node contains a geometric object but cannot be converted to mesh\nDo you still want to export this node",name,MB_YESNOCANCEL);
		switch (res)
		{
		case IDYES:
			return IDYES;
		case IDNO:
			return ID_SKIP_NODE;
		case IDCANCEL:
			return IDCANCEL;
		}
	}

	TriObject	*tri	= (TriObject *) obj->ConvertToType(time, Class_ID(TRIOBJ_CLASS_ID, 0));
	BOOL		needDel = (obj != tri);
	Mesh		*mesh	= &tri->mesh;
	if (!mesh)
	{
		if (bPreviewMode)
			return IDYES;
		int res=MessageBox(hwnd_nodes,"Serious Warning: Node's triangle mesh could not be obtained\nDo you still want to export this node",name,MB_YESNOCANCEL);
		switch (res)
		{
		case IDYES:
			return IDYES;
		case IDNO:
			return ID_SKIP_NODE;
		case IDCANCEL:
			return IDCANCEL;
		}
	}

	int nVerts = mesh->getNumVerts();
	int nFaces = mesh->getNumFaces();
	int nTVerts= mesh->getNumTVerts();
	int nCVerts= mesh->getNumVertCol();

	Face		*Faces	  = mesh->faces;
	Point3		*Verts	  = mesh->verts;
	TVFace		*TexFaces = mesh->tvFace;
	UVVert		*TexVerts = mesh->tVerts;
	VertColor	*ColVerts = mesh->vertCol;
	TVFace		*ColFaces = mesh->vcFace;

	//======================
	//Check non-rigid links
	//======================
	Modifier *phy_mod=FindPhysiqueModifier(der_obj);
	//ms->NonRigid_Links =false;
	if(phy_mod)
	{
		IPhysiqueExport *phyInterface = (IPhysiqueExport *)( phy_mod->GetInterface(I_PHYINTERFACE));
		if(phyInterface)
		{

			IPhyContextExport *phyContex = phyInterface->GetContextInterface(node);
			if(phyContex)
			{
				//phyContex->ConvertToRigid(false);
				//phyContex->AllowBlending(true);

				Matrix3 tm = node->GetObjTMAfterWSM(time);

				int n_vert=phyContex->GetNumberVertices();
				for(int i=0;i<n_vert;i++)
				{
					UpdateProgress(100*i/n_vert,"\"%s\" checking bone weights", name);

					Point3 vertex_at_world=tm*Verts[i];

					IPhyVertexExport *vert=phyContex->GetVertexInterface(i);
					int vert_type=vert->GetVertexType();
					phyContex->ReleaseVertexInterface(vert);
					
					if((vert_type != RIGID_NON_BLENDED_TYPE) && (vert_type != RIGID_BLENDED_TYPE) && pb_options->GetInt(PB_WARN_NONRIGID))
					{
						//ms->NonRigid_Links=true;
						sprintf(s,"Some of the vertices of \"%s\" are using DEFORMABLE Physique links."
							"\nOnly RIGID links can be exported. Although the exporter can convert these DEFORMABLE vertices to RIGID ones,"
							"\nthis process may cause loss of precision and the resulting mesh deformation might be different from the one seen in MAX"
							"\nWe STRONGLY RECOMMEND you to re-initialize the Physique so that only RIGID links are used."
							"\nyou can see what the exported animation will look like by setting the Pysique \"Viewport Skin Update\" parameter to \"Rigid\""
							"\n\nDo you still want to export this node?",name);
						int res=CryMessageBox(hwnd_nodes, s, "IMPORTANT WARNING!!!", MB_OKCANCEL | MB_ICONHAND);
						if(res==IDCANCEL)
						{
							phyInterface->ReleaseContextInterface(phyContex);
							phy_mod->ReleaseInterface(I_PHYINTERFACE, phyInterface);
							return IDCANCEL;
						}
						break;
					}
				}			
				phyInterface->ReleaseContextInterface(phyContex);
			}
			phy_mod->ReleaseInterface(I_PHYINTERFACE, phyInterface);
		}							
	}

	//===================
	//Check orphan tverts
	//===================
	//ms->misshared_tverts=false;
	//ms->orphan_tverts	=false;
	bool misshared_tverts	= false;
	bool orphan_tverts		= false;
	if(nTVerts)
	{
		int *tvert_used=(int *)calloc(nTVerts, sizeof(int));
		int *tvert_vert=(int *)calloc(nTVerts, sizeof(int));
		assert(tvert_used);
		assert(tvert_vert);

		IntTab Degen_Faces;
		IntTab Degen_VertIdx;
		for(int i=0;i<nFaces;i++)
		{
			for(int j=0;j<3;j++)
			{
				int na=tvert_used[TexFaces[i].t[j]]++;
				if(na)
				{
					if(tvert_vert[TexFaces[i].t[j]] != Faces[i].v[j])
					{
						//ms->misshared_tverts=true;
						misshared_tverts=true;
						DebugPrint("tvert %d is used by vert %d and vert %d (face %d)\n",TexFaces[i].t[j], tvert_vert[TexFaces[i].t[j]], Faces[i].v[j],i);
						Degen_Faces.Append(1,&i);
						Degen_VertIdx.Append(1,&j);
					}
				}
				tvert_vert[TexFaces[i].t[j]] = Faces[i].v[j];
			}
		}

		IntTab orphan_tverts_tab;
		bool orphan_tverts =false;
		for(i=0;i<nTVerts;i++)
		{
			if(!tvert_used[i]) orphan_tverts_tab.Append(1,&i);
		}

		if(orphan_tverts_tab.Count())
		{
			//ms->orphan_tverts	=true;
			orphan_tverts	=true;
			DebugPrint("%d tverts are orphaned\n",orphan_tverts_tab.Count());
		}

		free(tvert_used);
		tvert_used=NULL;
	}

	if((misshared_tverts || orphan_tverts) && pb_options->GetInt(PB_WARN_ORPHAN_TV))
	{
		int res=CryMessageBox(hwnd_nodes, "Node's mesh has some degenerate texture coordinates\n"
			"Do you still want to export this node?", name, MB_YESNOCANCEL);
		switch (res)
		{
		case IDNO:
			return ID_SKIP_NODE;
		case IDCANCEL:
			return IDCANCEL;
		}
	}

	//============================
	//Check multiple Vertex Colors
	//============================
	if(pb_nodes->GetInt(PB_WRITE_VCOL))
	{
		if(pb_options->GetInt(PB_WARN_MULTI_VCOL))
		{
			if(nCVerts && (nVerts != nCVerts))
			{
				int res=CryMessageBox(hwnd_nodes, "Node has multiple vertex colors per vertex.\n"
					"If you continue exporting, average of these vertex colors will be used for such vertices.\n\n"
					"Do you still want to continue exporting this node", name, MB_YESNOCANCEL);
				switch (res)
				{
				case IDNO:
					return ID_SKIP_NODE;
				case IDCANCEL:
					return IDCANCEL;
				}
			}
		}
	}

	return false;
}

//  ______________________________________________________________________________________
void CSExportUtility::CalcVertexNormals(Tab<VNormal> &vnorms, Mesh *mesh, BOOL ConsiderSmoothing, BOOL negate, Point3* pVertexSubstitute)
{
	int nVerts = mesh->getNumVerts();
	int nFaces = mesh->getNumFaces();
	Face		*Faces	  = mesh->faces;
	Point3		*Verts	  = pVertexSubstitute?pVertexSubstitute:mesh->verts;
	
	//calculate vertex normals
	vnorms.SetCount(nVerts);
	memset (&vnorms[0],0, nVerts*sizeof(VNormal));
	// Compute face and vertex surface normals
	for (int i = 0; i < nVerts; i++) vnorms[i] = VNormal();	//init

	for (i = 0; i < nFaces; i++) 
	{
		// Calculate the surface normal
		Point3 v0, v1, v2;
		v0 = Verts[Faces[i].v[0]];
		v1 = Verts[Faces[i].v[1]];
		v2 = Verts[Faces[i].v[2]];
		Point3 fnorm = negate ? (v2-v1)^(v1-v0) : (v1-v0)^(v2-v1);
		for (int j=0; j<3; j++) 
		{		
			vnorms[Faces[i].v[j]].AddNormal(fnorm,ConsiderSmoothing?Faces[i].smGroup:1);
		}
	}

	for (i=0; i < nVerts; i++) 
	{
		vnorms[i].Normalize();
	}
}

struct VVertex
{
	Point3 pos;
	Point3 norm;
	DWORD smooth;
	int index;
	bool init;
	VVertex *next;

	VVertex() { smooth=0; index=0; next=NULL; init=false; pos=norm=Point3(0,0,0);}
	~VVertex() { delete next; }

	VVertex( int i,Point3 &p,Point3 &n,DWORD sm )
	{
		pos = p;
		norm = n;
		smooth = sm;
		index = i;
		init = true;
		next = 0;
	}
	
	void AddVertex( int i,Point3 &p,Point3 &n,DWORD sm )
	{
		if (!(sm&smooth) && init)
		{
			if (next)
				next->AddVertex(i,p,n,sm);
			else {
				next = new VVertex(i,p,n,sm);
			}
		} 
		else {
			pos = p;
			norm = n;
			smooth = sm;
			index = i;
			init = true;
		}
	}
	VVertex& GetVertex( DWORD sm )
	{
		if (smooth&sm || !next) return *this;
		else return next->GetVertex(sm);	
	}
};

void CSExportUtility::SplitSmoothGroups( Mesh &mesh,Mesh &outMesh,std::vector<int> &arrVertexMap,BOOL negate, Point3* pActualVerts )
{
	if (!pActualVerts)
		pActualVerts = mesh.verts;

	Tab<VNormal> vnorms;
	CalcVertexNormals( vnorms,&mesh,true,negate, pActualVerts);

	int i,j;
	int nVerts = mesh.getNumVerts();
	int nFaces = mesh.getNumFaces();
	Face		*Faces	  = mesh.faces;
	Point3		*Verts	  = pActualVerts;

	std::vector<Face> newFaces;
	std::vector<Point3> newVerts;
	std::vector<Point3> newNormals;
	std::vector<VVertex> vverts;

	newFaces.resize(nFaces);
	newVerts.resize(nVerts);
	newNormals.resize(nVerts);
	arrVertexMap.resize(nVerts);

	// Inistialize vertices.
	vverts.resize(nVerts);
	for (i = 0; i < nVerts; ++i)
	{
		newVerts[i] = Verts[i];
		arrVertexMap[i] = i;
		vverts[i] = VVertex();
		newNormals[i] = vnorms[i].norm;
	}

	for (i = 0; i < nFaces; i++) 
	{
		newFaces[i] = Faces[i];
		for (j = 0; j < 3; j++)
		{
			int v = Faces[i].v[j];
			Point3 vn = vnorms[v].GetNormal( Faces[i].smGroup );
			vverts[v].AddVertex( v,Verts[v],vn,Faces[i].smGroup );
		}
	}

	for (i = 0; i < nVerts; i++)
	{
		newNormals[vverts[i].index] = vverts[i].norm;
		for (VVertex *vv = vverts[i].next; vv != 0; vv = vv->next)
		{
			int index = vv->index;
			int newIndex = newVerts.size();
			vv->index = newIndex;
			newVerts.push_back(Verts[index]);
			newNormals.push_back(vv->norm);

			arrVertexMap.push_back(index);
		}
	}

	for (i = 0; i < nFaces; i++) 
	{
		for (j = 0; j < 3; j++)
		{
			int v = newFaces[i].v[j];
			int newIndex = vverts[v].GetVertex( Faces[i].smGroup ).index;
			newFaces[i].v[j] = newIndex;
		}
	}

	outMesh.DeepCopy( &mesh,ALL_CHANNELS );
	
	// Copy new verts.
	outMesh.setNumVerts( newVerts.size() );
	for (i = 0; i < (int)newVerts.size(); i++)
	{
		outMesh.setVert( i,newVerts[i] );
		outMesh.setNormal( i,newNormals[i] );
	}

	// Copy new faces
	outMesh.setNumFaces( newFaces.size(), TRUE);
	for (i = 0; i < (int)newFaces.size(); i++)
	{
		outMesh.faces[i] = newFaces[i];
	}
}

// Generate the default texture coordinates for the mesh
void CSExportUtility::GenerateDefaultUVs (Mesh& mesh)
{
	unsigned numVerts = mesh.getNumVerts();
	mesh.setNumTVerts(numVerts);
	// make up the cubic coordinates
	for (unsigned nVert = 0; nVert < numVerts; ++nVert)
	{
		Point3& v = mesh.getVert(nVert);
		Point3	r (fabs(v.x), fabs(v.y), fabs(v.z));
		UVVert uv (0,0,0);
		if (r.x > 1e-3 || r.y > 1e-3 || r.z > 1e-3)
		{
			if (r.x > r.y)
			{
				if (r.x > r.z)
				{
					// X rules
					uv.x = v.y/r.x;
					uv.y = v.z/r.x;
					uv.z = v.x;
				}
				else
				{
					// Z rules
					uv.x = v.x / r.z;
					uv.y = v.y / r.z;
					uv.z = v.z;
				}
			}
			else
			{
				// r.x < r.y
				if (r.y > r.z)
				{
					// Y rules
					uv.x = v.x / r.y;
					uv.y = v.z / r.y;
					uv.z = v.y;
				}
				else
				{
					// Z rules
					uv.x = v.x / r.z;
					uv.y = v.y / r.z;
					uv.z = v.z;
				}
			}
		}
		// now the texture coordinates are in the range [-1,1]
		// we want normalized to 0..1 texture coordinates
		uv.x = (uv.x + 1)/2;
		uv.y = (uv.y + 1)/2;

		mesh.setTVert(nVert, uv);
	}

	unsigned numFaces = mesh.getNumFaces();
	mesh.setNumTVFaces (numFaces);
	TVFace* pTVFaces = mesh.tvFace;
	Face* pFaces = mesh.faces;

	for (unsigned nTVFace = 0; nTVFace < numFaces; ++nTVFace)
	{
		for (unsigned i = 0; i < 3; ++i)
			pTVFaces[nTVFace].t[i] = pFaces[nTVFace].v[i];
	}

	// clean up the face material indices: they have no meaning as long as we don't have UVs in Max
	for (unsigned nFace = 0; nFace < numFaces; ++nFace)
		mesh.setFaceMtlIndex(nFace, 0);
}

//  ______________________________________________________________________________________
bool CSExportUtility::SavePhysiqueInfo(FILE* f, INode* node, Point3 *Verts,std::vector<int> &arrVertexMap)
{
	DebugPrint("\nsaving physique info for \"%s\"",node->GetName()); 
	Interval ivalid;
	TimeValue time=GetTime();
	Matrix3 tm = node->GetObjTMAfterWSM(time);

	int LinkType = pb_nodes->GetInt(PB_LINK_TYPE, time);

	//PrintDebugModifiers (node);

	Modifier *phy_mod=FindPhysiqueModifier(node->GetObjectRef());
	if(!phy_mod) return true;

	IPhysiqueExport *phyInterface = (IPhysiqueExport *)( phy_mod->GetInterface(I_PHYINTERFACE));
	if(!phyInterface) return true;

	IPhyContextExport *phyContex = phyInterface->GetContextInterface(node);
	if(!phyContex) return true;
	
	phyContex->ConvertToRigid(true);
	phyContex->AllowBlending(LinkType);

	//int n_vert=phyContex->GetNumberVertices();
	size_t n_vert = arrVertexMap.size();
	for(size_t k=0;k<n_vert;k++)
	{
		int i = arrVertexMap[k];
		UpdateProgress(100*k/n_vert,"\"%s\" Writing bone weights", node->GetName());

		Point3 vertex_at_world=
#ifdef EXPORT_INIT_POSE
#else
			tm*
#endif
			Verts[k];

		static bool bStandardExtract = false;

		IPhyVertexExport *vert=phyContex->GetVertexInterface(i);
		int vert_type=vert ? vert->GetVertexType() : -1;
		switch(vert_type)
		{
		case RIGID_NON_BLENDED_TYPE:
			{
				IPhyRigidVertex *RigVert=(IPhyRigidVertex *)vert;

				CryLink link;
				link.BoneID		= BoneList.GetID(RigVert->GetNode());
				link.Blending	= 1.0f;

				Point3 off;
				if (bStandardExtract)
					off=RigVert->GetOffsetVector();
				else
					off=vertex_at_world*BoneList.invTM[link.BoneID];
				
				link.offset.x	= off.x;
				link.offset.y	= off.y;
				link.offset.z	= off.z;
				
				//=================
				// Write nLinks
				//=================
				//DebugPrint("\nnlinks for vert %4d = %4d", i, nLinks);
				int nLinks = 1;
				int res=fwrite(&nLinks, sizeof(nLinks), 1, f);
				if(res !=1) return true;
				//=================
				// Write Link
				//=================
				//DebugPrint(" blend=%f",link.Blending);
				res=fwrite(&link, sizeof(link), 1, f);
				if(res !=1) return true;
			}
			break;

		case RIGID_BLENDED_TYPE:
			{
				IPhyBlendedRigidVertex *BlndVert=(IPhyBlendedRigidVertex *)vert;
				int j, nLinks=BlndVert->GetNumberNodes();
				
				//=================
				// Write nLinks
				//=================
				//DebugPrint("\nnlinks for vert %4d = %4d", i, nLinks);
				std::vector<CryLink> arrLinks;

				// the minimal weight that the link must have to be exported
				const float fMinWeight = 0.01f;
				
				// this is the max number of weights per vertex - the extra ewights are clamped
				const unsigned nMaxLinksPerVertex = 5;

				// the total weight of the links that are exported (for renormalization)
				float fTotalWeight = 0;

				for(j=0;j<nLinks;j++)
				{
					CryLink link;
					INode* pBoneNode = BlndVert->GetNode(j);
					TCHAR* szBoneName = pBoneNode->GetName();
					link.BoneID		= BoneList.GetID (pBoneNode);
					link.Blending	= BlndVert->GetWeight(j);
					if (link.Blending < fMinWeight)
						continue;
					
					Point3 off;
					if (bStandardExtract)
						off=BlndVert->GetOffsetVector(j);
					else
						off=vertex_at_world*BoneList.invTM[link.BoneID];

					link.offset.x	= off.x;
					link.offset.y	= off.y;
					link.offset.z	= off.z;
					
					fTotalWeight += link.Blending;
					arrLinks.push_back(link);
				}

				if (fTotalWeight < 0.5)
				{
					char szDebug [1024];
					sprintf (szDebug, "%d links {", nLinks);
					unsigned i;
					for (i = 0; i < arrLinks.size(); ++i)
					{
						if (i)
							strcat (szDebug, ";");
						CryLink& l = arrLinks[i];
						const char *szBoneName = "#OUT OF RANGE#";
						if (l.BoneID >= 0 && l.BoneID < BoneList.arrNames.size())
							szBoneName = BoneList.arrNames[l.BoneID].c_str();
						sprintf (szDebug+strlen(szDebug), "v=(%g,%g,%g) w=%g b=%s", l.offset.x, l.offset.y, l.offset.z, l.Blending,szBoneName);
					}
					sprintf (szDebug+strlen(szDebug), "} vertex={%g,%g,%g}", Verts[k].x, Verts[k].y, Verts[k].z);

					char szMessage[1024+400];
					sprintf (szMessage, "Cannot export vertex %d because all link weights sum up to only %.2f. They should sum up to 1. Please renormalize and/or reassign the vertices.\nDebug info: %s", k, fTotalWeight, szDebug);
					MessageBox(ip->GetMAXHWnd(), szMessage, "Error", MB_ICONSTOP|MB_OK);
					return true;
				}

				for (j = 0; j < (int)arrLinks.size(); ++j)
					arrLinks[j].Blending /= fTotalWeight;

				std::sort (arrLinks.begin(), arrLinks.end(), CryLinkOrderByBlending());

				if (arrLinks.size() > nMaxLinksPerVertex)
					arrLinks.resize (nMaxLinksPerVertex);

				// write out the links: save the number of links, and the links themselves then
				nLinks = arrLinks.size();
				int res=fwrite(&nLinks, sizeof(nLinks), 1, f);
				if(res !=1) return true;
				res = fwrite(&arrLinks[0], sizeof(arrLinks[0]), arrLinks.size(), f);
				if (res != arrLinks.size())
					return true;
			}
			break;

		default:
			{
				CryMessageBox(hwnd_bones,"There are unsupported Physique links\nCheck \"Force Rigid\" checkbox","Error",MB_OK);

				phyContex->ReleaseVertexInterface(vert);
				phyInterface->ReleaseContextInterface(phyContex);
				phy_mod->ReleaseInterface(I_PHYINTERFACE, phyInterface);

				return true;
			}
		}
		phyContex->ReleaseVertexInterface(vert);
	}

	phyInterface->ReleaseContextInterface(phyContex);
	phy_mod->ReleaseInterface(I_PHYINTERFACE, phyInterface);

	DebugPrint("\ndone saving physique info for \"%s\"",node->GetName()); 
	return false;
}

// SaveVertAnim ______________________________________________________________________________________
bool CSExportUtility::SaveVertAnim(FILE* f, INode *node)
{
	Object *der_obj = node->GetObjectRef();
	if(!der_obj) return true;

	TimeValue	time	= GetTime();
	Object		*obj	= der_obj->Eval(time).obj;
	SClass_ID	sid		= obj->SuperClassID();
	if(!obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0)))  return false;

	TriObject	*tri	= (TriObject *) obj->ConvertToType(time, Class_ID(TRIOBJ_CLASS_ID, 0));
	BOOL		needDel = (obj != tri);
	Mesh		*mesh	= &tri->mesh;
	if (!mesh) return true;
	
	//check if mesh is suitable
	int nVerts = mesh->getNumVerts();
	int nFaces = mesh->getNumFaces();
	Face		*Faces	  = mesh->faces;
	Point3		*Verts	  = mesh->verts;

	//Get Parameters
	int	StartFrame	=	pb_timing->GetInt(PB_RANGE_START);
	int EndFrame	=	pb_timing->GetInt(PB_RANGE_END);
	int	SampleRate	=	pb_nodes->GetInt(PB_VERT_ANIM_RATE);

	//Calc # of keys
	int nKeys = 1+((EndFrame-StartFrame)/SampleRate);

	//==================
	//Write chunk header
	//==================
	fpos_t fpos;
	fgetpos(f,&fpos);
	int chunk_id=ChunkList.GetID(node,ChunkType_VertAnim);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_VertAnim at pos %d)",ch_ent.FileOffset);

	VERTANIM_CHUNK_DESC chunk;
	chunk.chdr				= ch_ent;
	chunk.nFaces			= nFaces;
	chunk.nVerts			= nVerts;
	chunk.nKeys				= nKeys;
	chunk.GeomID			= ChunkList.GetID(node, ChunkType_Mesh);
	int res=fwrite(&chunk,sizeof(chunk),1,f);
	if(res!=1) return true;

	if(needDel) tri->DeleteThis();
	tri=NULL; mesh=NULL;

	for(int i=0;i<nKeys;i++)
	{
		UpdateProgress(100*i/nKeys,"Writing vertex animation keys", "");
		time = (StartFrame + i*SampleRate)*GetTicksPerFrame();

		//UpdateProgress(100*i/nKeys,"\"%s\" Writing vertex animation key %d", ActiveNode->GetName(),i);
		obj	= der_obj->Eval(time).obj;
		sid		= obj->SuperClassID();
		if(!obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0)))  return false;
		tri		= (TriObject *) obj->ConvertToType(time, Class_ID(TRIOBJ_CLASS_ID, 0));
		needDel = (obj != tri);
		mesh	= &tri->mesh;
		if (!mesh) return true;

		nVerts = mesh->getNumVerts();
		nFaces = mesh->getNumFaces();
		Faces  = mesh->faces;
		Verts  = mesh->verts;

		if(nVerts!=chunk.nVerts)
		{
			CryMessageBox(hwnd_nodes,"number of vertices of node \"%s\" changes in time\nVertex animation for objects with dynamic topologies can not be exported","Error",MB_OK);
			return true;
		}

		//=====================
		// Write Key Header
		//=====================
		KEY_HEADER kh;
		kh.KeyTime = time;
		int res=fwrite(&kh,sizeof(kh),1,f);
		if(res!=1) return true;

		//=====================
		// Write Vertices
		//=====================
		//calc vertex normals
		BOOL ConsiderSmooth = true;
		Tab<VNormal> vnorms;
		CalcVertexNormals(vnorms, mesh, ConsiderSmooth);

		for(int j=0;j<nVerts;j++)
		{
			Point3 v = Verts[j];
			Point3 n = vnorms[j].GetNormal(1);

			CryVertex vert;
			vert.p.x=v.x; 
			vert.p.y=v.y; 
			vert.p.z=v.z;

			vert.n.x=n.x;
			vert.n.y=n.y;
			vert.n.z=n.z;
			res=fwrite(&vert,sizeof(vert),1,f);
			if(res!=1) return true;
		}

		if(needDel) tri->DeleteThis();
		tri=NULL; mesh=NULL;
	}

	return false;
}
// SaveHelperNode ______________________________________________________________________________________
bool CSExportUtility::SaveHelperObject(FILE* f, INode *node)
{
	Object *der_obj = node->GetObjectRef();
	if(!der_obj) return true;

	TimeValue time=GetTime();
	Object *obj = der_obj->Eval(time).obj;
	
	HelperTypes type;
	Vec3 size(0.0f, 0.0f, 0.0f);
	Class_ID cid=obj->ClassID();

	if (der_obj->SuperClassID()==SYSTEM_CLASS_ID && der_obj->ClassID()==Class_ID(XREFOBJ_CLASS_ID,0))
	{
		type = HP_XREF;
		//IXRefObject *ix = (IXRefObject *)obj;
		//ix->BrowseFile(FALSE);
		Box3 box;
		obj->GetLocalBoundBox(time,node,0,box );
		Point3 p=box.Width();
		size.x = p.x;	size.y = p.y;	size.z = p.z;
	}
	else if (cid==Class_ID(POINTHELP_CLASS_ID, 0))	
	{
		type=HP_POINT;
	}
	else if(cid==Class_ID(DUMMY_CLASS_ID, 0))
	{
		type=HP_DUMMY;
		Box3 box=((DummyObject *)obj)->GetBox();
		Point3 p=box.Width();
		size.x = p.x;	size.y = p.y;	size.z = p.z;
	}
	else if(cid==Class_ID(CAMERA_CLASS_ID, 0))
	{
		type=HP_CAMERA;
	}
	else assert(false);

	//==================
	//write chunk header
	//==================
	UpdateProgress(100,"Writing helper Object");
	fpos_t fpos;
	fgetpos(f,&fpos);
	int chunk_id=ChunkList.GetID(node,ChunkType_Helper);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_Helper at pos %d)",ch_ent.FileOffset);
	
	HELPER_CHUNK_DESC hcd;
	hcd.chdr = ch_ent;
	hcd.type = type;
	hcd.size = size;
	int res=fwrite(&hcd,sizeof(hcd),1,f);
	if(res!=1) return true;

	return false;
}

// SaveLightNode ______________________________________________________________________________________
bool CSExportUtility::SaveLightObject(FILE* f, INode *node)
{
	Object *der_obj = node->GetObjectRef();
	if(!der_obj) return true;

	TimeValue	time	= GetTime();
	Object		*obj	= der_obj->Eval(time).obj;
	LightObject *lt_obj	= (LightObject*)obj;

	//==================
	//Write chunk header
	//==================
	UpdateProgress(100,"Writing light object");
	fpos_t fpos;
	fgetpos(f,&fpos);
	int chunk_id=ChunkList.GetID(node,ChunkType_Light);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_Light at pos %d)",ch_ent.FileOffset);

	LightState ls;
	Interval ivalid;
	lt_obj->EvalLightState(time,ivalid,&ls);
	
	LIGHT_CHUNK_DESC lcd;
	lcd.chdr				= ch_ent;
	lcd.type				= (LightTypes)ls.type;
	lcd.on					= ls.on?true:false;
	lcd.color.r				= (BYTE)min(255,max(0, 255*ls.color.r));
	lcd.color.g				= (BYTE)min(255,max(0, 255*ls.color.g));
	lcd.color.b				= (BYTE)min(255,max(0, 255*ls.color.b));
	lcd.intens				= ls.intens;
	lcd.fallsize			= ls.fallsize;
	lcd.hotsize				= ls.hotsize;
	lcd.useNearAtten		= ls.useNearAtten?true:false;
	lcd.nearAttenEnd		= ls.nearAttenEnd;
	lcd.nearAttenStart		= ls.nearAttenStart;
	lcd.useAtten			= ls.useAtten?true:false;
	lcd.attenEnd			= ls.attenEnd;
	lcd.attenStart			= ls.attenStart;
	lcd.shadow				= ls.shadow?true:false;

	Matrix3 tm = node->GetNodeTM(0);
	/*
	float yaw,pitch,roll;
	tm.GetYawPitchRoll( &yaw,&pitch,&roll ); 
	lcd.vDirection.x = yaw;
	lcd.vDirection.y = pitch;
	lcd.vDirection.z = roll;
	*/
	float ang[3];
	MatrixToEuler( tm,ang,EULERTYPE_XYZ );
	lcd.vDirection.x = ang[0];
	lcd.vDirection.y = ang[1];
	lcd.vDirection.z = ang[2];


	if (lt_obj->GetProjector())
	{
		Texmap *tex = lt_obj->GetProjMap();

		//name
		if (tex->ClassID() == Class_ID(BMTEX_CLASS_ID, 0))
		{
			BitmapTex *bmt = (BitmapTex*)tex;
			TSTR sf;
			sf = GetRelativePath( bmt->GetMapName() );
			strcpy( lcd.szLightImage,sf );
		}
    else
      lcd.szLightImage[0] = 0;
	}
  else
    lcd.szLightImage[0] = 0;
	
	int res=fwrite(&lcd,sizeof(lcd),1,f);
	if(res!=1) return true;

	return false;
}

// SaveLightNode ______________________________________________________________________________________

bool CSExportUtility::SaveController(FILE* f, Control* cont)
{
	if(!cont) 
		return true;
	IKeyControl *ikey=GetKeyControlInterface(cont);
	if(!ikey) 
		return true;

	if(!ikey->GetNumKeys()) 
	{
	//	char lala[255];
	//	sprintf(lala,"object %s has a controller with no keys. Aborting!",m_pCurrentNode->GetName());
	//	MessageBox(NULL,lala,"ANDREW!",MB_OK);
		return false;
	}
		

	TimeValue	time	= GetTime();
	int res;

	//==================
	//Write chunk header
	//==================
	UpdateProgress(100,"Writing controller object");
	fpos_t fpos;
	fgetpos(f,&fpos);
	int chunk_id=ChunkList.GetID(cont ,ChunkType_Controller);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_Controller at pos %d)",ch_ent.FileOffset);


	Class_ID cid=cont->ClassID();
	CtrlTypes type=CTRL_NONE;

	if     (cid==Class_ID(LININTERP_FLOAT_CLASS_ID,0))		type=CTRL_LINEER1;
	else if(cid==Class_ID(LININTERP_POSITION_CLASS_ID,0))	type=CTRL_LINEER3;
	else if(cid==Class_ID(LININTERP_SCALE_CLASS_ID,0))		type=CTRL_LINEER3;
	else if(cid==Class_ID(LININTERP_ROTATION_CLASS_ID,0))	type=CTRL_LINEERQ;

	else if(cid==Class_ID(HYBRIDINTERP_FLOAT_CLASS_ID,0))	type=CTRL_BEZIER1;
	else if(cid==Class_ID(HYBRIDINTERP_POSITION_CLASS_ID,0))type=CTRL_BEZIER3; 
	else if(cid==Class_ID(HYBRIDINTERP_POINT3_CLASS_ID,0))	type=CTRL_BEZIER3;   
	else if(cid==Class_ID(HYBRIDINTERP_SCALE_CLASS_ID,0))	type=CTRL_BEZIER3;
	else if(cid==Class_ID(HYBRIDINTERP_ROTATION_CLASS_ID,0))type=CTRL_BEZIERQ;

	else if(cid==Class_ID(TCBINTERP_FLOAT_CLASS_ID,0))		type=CTRL_TCB1;
	else if(cid==Class_ID(TCBINTERP_POSITION_CLASS_ID,0))	type=CTRL_TCB3; 
	else if(cid==Class_ID(TCBINTERP_POINT3_CLASS_ID,0))		type=CTRL_TCB3;   
	else if(cid==Class_ID(TCBINTERP_SCALE_CLASS_ID,0))		type=CTRL_TCB3;
	else if(cid==Class_ID(TCBINTERP_ROTATION_CLASS_ID,0))	type=CTRL_TCBQ;

	if(type==CTRL_NONE) 
		return true;
	
	CONTROLLER_CHUNK_DESC_0826 chunk;
	chunk.chdr		= ch_ent;
	chunk.nKeys		= ikey->GetNumKeys();
	chunk.type		= type;
	chunk.nFlags	= 0;
	chunk.nControllerId = chunk.chdr.ChunkID;

	int ort = cont->GetORT(ORT_AFTER);
	if (ort == ORT_CYCLE)
		chunk.nFlags |= CTRL_ORT_CYCLE;
	else if (ort == ORT_LOOP)
		chunk.nFlags |= CTRL_ORT_LOOP;

	res=fwrite(&chunk,sizeof(chunk),1,f); 

	if(res!=1) 
		return true;
	
	for(int i=0;i<chunk.nKeys;i++)
	{
		//====================== LINEER ====================================
		if(cid==Class_ID(LININTERP_FLOAT_CLASS_ID,0))
		{
			ILinFloatKey key;		ikey->GetKey(i, &key); 
			CryLin1Key ck; 
			ck.time	= key.time;
			ck.val	= key.val;
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		else if(cid==Class_ID(LININTERP_POSITION_CLASS_ID,0))
		{
			ILinPoint3Key key;		ikey->GetKey(i, &key); 
			CryLin3Key ck;
			ck.time	= key.time;
			ck.val	= Vec3(key.val.x, key.val.y, key.val.z);
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		else if(cid==Class_ID(LININTERP_SCALE_CLASS_ID,0))
		{
			ILinScaleKey key;		ikey->GetKey(i, &key); 
			CryLin3Key ck;
			ck.time	= key.time;
			ck.val	= Vec3(key.val.s.x, key.val.s.y, key.val.s.z);
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		else if(cid==Class_ID(LININTERP_ROTATION_CLASS_ID,0))
		{
			ILinRotKey key;			ikey->GetKey(i, &key); 
			CryLinQKey ck;
			ck.time	= key.time;
			ck.val	= CryQuat(key.val.w,key.val.x, key.val.y, key.val.z);
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		//======================= HYBRID ===================================
		else if(cid==Class_ID(HYBRIDINTERP_FLOAT_CLASS_ID,0))
		{
			IBezFloatKey key;		ikey->GetKey(i, &key); 
			CryBez1Key ck; 
			ck.time		= key.time;
			ck.val		= key.val;
			ck.intan	= key.intan;
			ck.outtan	= key.outtan;
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		else if(cid==Class_ID(HYBRIDINTERP_POSITION_CLASS_ID,0) || 
				cid==Class_ID(HYBRIDINTERP_POINT3_CLASS_ID,0))
		{
			IBezPoint3Key key;		ikey->GetKey(i, &key); 
			CryBez3Key ck; 
			ck.time		= key.time;
			ck.val		= Vec3(key.val.x, key.val.y, key.val.z);
			ck.intan	= Vec3(key.intan.x, key.intan.y, key.intan.z);
			ck.outtan	= Vec3(key.outtan.x, key.outtan.y, key.outtan.z);
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;

		}
		else if(cid==Class_ID(HYBRIDINTERP_SCALE_CLASS_ID,0))
		{
			IBezScaleKey key;		ikey->GetKey(i, &key); 
			CryBez3Key ck; 
			ck.time		= key.time;
			ck.val		= Vec3(key.val.s.x, key.val.s.y, key.val.s.z);
			ck.intan	= Vec3(key.intan.x, key.intan.y, key.intan.z);
			ck.outtan	= Vec3(key.outtan.x, key.outtan.y, key.outtan.z);
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		else if(cid==Class_ID(HYBRIDINTERP_ROTATION_CLASS_ID,0))
		{
			IBezQuatKey key;		ikey->GetKey(i, &key); 
			CryBezQKey ck; 
			ck.time		= key.time;
			ck.val		= CryQuat(key.val.w,key.val.x, key.val.y, key.val.z);
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		//======================== TCB ==================================
		else if(cid==Class_ID(TCBINTERP_FLOAT_CLASS_ID,0))
		{
			ITCBFloatKey key;		ikey->GetKey(i, &key); 
			CryTCB1Key ck; 
			ck.time		= key.time;
			ck.val		= key.val;
			ck.t		= key.tens;
			ck.c		= key.cont;
			ck.b		= key.bias;
			ck.ein		= key.easeIn;
			ck.eout		= key.easeOut;
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		else if(cid==Class_ID(TCBINTERP_POSITION_CLASS_ID,0) ||
				cid==Class_ID(TCBINTERP_POINT3_CLASS_ID,0))
		{
			ITCBPoint3Key key;		ikey->GetKey(i, &key); 
			CryTCB3Key ck; 
			ck.time		= key.time;
			ck.val		= Vec3(key.val.x, key.val.y, key.val.z);
			ck.t		= key.tens;
			ck.c		= key.cont;
			ck.b		= key.bias;
			ck.ein		= key.easeIn;
			ck.eout		= key.easeOut;
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		else if(cid==Class_ID(TCBINTERP_SCALE_CLASS_ID,0))
		{
			ITCBScaleKey key;		ikey->GetKey(i, &key); 
			CryTCB3Key ck; 
			ck.time		= key.time;
			ck.val		= Vec3(key.val.s.x, key.val.s.y, key.val.s.z);
			ck.t		= key.tens;
			ck.c		= key.cont;
			ck.b		= key.bias;
			ck.ein		= key.easeIn;
			ck.eout		= key.easeOut;
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		else if(cid==Class_ID(TCBINTERP_ROTATION_CLASS_ID,0))
		{
			ITCBRotKey key;		ikey->GetKey(i, &key); 
			// Export relative Angle Axis.
			//Quat q(key.val);
			CryTCBQKey ck; 
			ck.time		= key.time;
			ck.val	= CryQuat(key.val.angle,key.val.axis.x,key.val.axis.y,key.val.axis.z);
			ck.t		= key.tens;
			ck.c		= key.cont;
			ck.b		= key.bias;
			ck.ein		= key.easeIn;
			ck.eout		= key.easeOut;
			if(fwrite(&ck,sizeof(ck),1,f)!=1) 
				return true;
		}
		else assert(false);
	}

	return false;
}

int CSExportUtility::PrepareVertexColors(CryIRGB *col, Mesh *mesh)
{
	int nVerts = mesh->getNumVerts();
	int nFaces = mesh->getNumFaces();
	int nCVerts= mesh->getNumVertCol();

	VertColor	*ColVerts = mesh->vertCol;
	TVFace		*ColFaces = mesh->vcFace;
	Face		*Faces	  = mesh->faces;

	if(! (nCVerts && nFaces && nVerts)) return TRUE;

	int			*count=(int *)calloc(nVerts,sizeof(int));
	VertColor	*fcols=(VertColor *) calloc(nVerts,sizeof(VertColor));
		
	if(!count) return TRUE;
	if(!fcols) return TRUE;

	for(int i=0;i<nFaces;i++)
	{
		int v0=Faces[i].v[0];
		int v1=Faces[i].v[1];
		int v2=Faces[i].v[2];

		VertColor c0=ColVerts[ColFaces[i].t[0]];
		VertColor c1=ColVerts[ColFaces[i].t[1]];
		VertColor c2=ColVerts[ColFaces[i].t[2]];

		fcols[v0]+=c0; count[v0]++;
		fcols[v1]+=c1; count[v1]++;
		fcols[v2]+=c2; count[v2]++;
	}

	for(i=0;i<nVerts;i++)
	{
		if(count[i])
		{
			col[i].r= int(fcols[i].x*255/count[i]);
			col[i].g= int(fcols[i].y*255/count[i]);
			col[i].b= int(fcols[i].z*255/count[i]);
		}
		else
		{
			col[i].r=col[i].g=col[i].b=0;
		}
	}

	free(count);
	free(fcols);
	return FALSE;
}

// SaveGeomNode ______________________________________________________________________________________
bool CSExportUtility::SavePatchObject(FILE* f, Object *obj)
{
/*
	Interval ivalid;
	TimeValue time=GetTime();

	ObjectState os		= node->EvalWorldState(time);
	Object		*obj	=os.obj;
	
	//convert to PatchObject
	if(!obj->CanConvertToType(patchObjectClassID)) return true;
	PatchObject *pobj	= (PatchObject*)obj->ConvertToType(time,patchObjectClassID);
	if(!pobj) return true;

	PatchMesh	*pmesh	= &(pobj->patch);
	//BOOL		needDel;
	//NullView	nullView;
	//PatchMesh *pmesh = ((GeomObject*)os.obj)->GetRenderPatchMesh(time,node,nullView,needDel);
	if (!pmesh) return true;

	int nVerts	= pmesh->numVerts;
	int nVects	= pmesh->numVecs;
	int nPatches= pmesh->numPatches;
	int nTVerts = pmesh->numTVerts[1];
	int nCVerts = pmesh->numTVerts[0];

	PatchVert	*verts		= pmesh->verts;
	UVVert 		*tverts		= pmesh->tVerts[1];
	Patch		*patches	= pmesh->patches;
	TVPatch		*tpatches	= pmesh->tvPatches[1];
	PatchVec	*vects		= pmesh->vecs;

	BOOL		WorldSpace		= pb_nodes->GetInt(PB_GEOM_WS);
	BOOL		HasBoneInfo		= pb_nodes->GetInt(PB_WRITE_WEIGHTS);
	BOOL		WriteVCol		= pb_nodes->GetInt(PB_WRITE_VCOL);
	BOOL		AllowMultiUV	= pb_nodes->GetInt(PB_ALLOW_MULTIUV);
	BOOL		ExportMat		= pb_nodes->GetInt(PB_EXPORT_MAT);

	//Turn off if object has no Physique on it.
	if(HasBoneInfo)
	{
		Modifier *phy_mod=FindPhysiqueModifier(node);
		if(!phy_mod) HasBoneInfo=FALSE;
	}

	//turn off vertex color export if mesh do not have them.
	if(!nCVerts) WriteVCol=FALSE;

	//Check if MultiUV per vertex
	if((!AllowMultiUV) && nTVerts && (nTVerts!=nVerts))
	{
		MessageBox(hwnd_nodes,"There are multiple UV coordinates per vertex\nEither check your mapping or turn on 'Allow Multi UV/vert' option", node->GetName(), MB_OK);
		return true;
	}

	//get material
	Mtl		*mat		= NULL;
	int		MatID		= -1;
	bool	Mat_multi	= false;
	int		n_sub_mtl	= 0;
	if(ExportMat)
	{
		mat=node->GetMtl();
		if(mat)
		{
			MatID=matlist.GetID(mat);
			if(mat->IsMultiMtl())
			{
				Mat_multi	=true;
				n_sub_mtl	=mat->NumSubMtls();
			}
		}
	}

	TSTR PropStr; 
	node->GetUserPropBuffer(PropStr);

	//==================
	//Write chunk header
	//==================
	UpdateProgress(100,"\"%s\" Writing chunk header", node->GetName());

	fpos_t fpos;
	fgetpos(f,&fpos);
	//append to chunk list
	CHUNK_HEADER ch_ent;
	ch_ent.ChunkType		= ChunkType_PatchMesh;
	ch_ent.ChunkVersion		= PATCHMESH_CHUNK_DESC_VERSION;
	ch_ent.FileOffset		= int(fpos);
	ChunkList.Append(1,&ch_ent);
	
	DebugPrint(" (ChunkType_PatchMesh at pos %d)",ch_ent.FileOffset);
	PATCHMESH_CHUNK_DESC chunk;
	chunk.ChunkType			= ChunkType_PatchMesh;
	chunk.ChunkVersion		= PATCHMESH_CHUNK_DESC_VERSION;
	chunk.GeomID			= GeomList.GetID(node);
	chunk.HasBoneInfo		= HasBoneInfo?true:false;
	chunk.nPatches			= nPatches;
	chunk.nTVerts			= nTVerts;
	chunk.nVerts			= nVerts + nVects;
	chunk.InWorldSpace		= WorldSpace?true:false;
	chunk.MatID				= MatID;
	chunk.PropStrLen		= PropStr.Length();
	int res=fwrite(&chunk,sizeof(chunk),1,f);
	if(res!=1) return true;
	
	//==================
	//Write PropString
	//==================
	UpdateProgress(100,"\"%s\" Writing properties", node->GetName());
	if(chunk.PropStrLen)
	{
		res=fwrite(PropStr.data(),chunk.PropStrLen,1,f);
		if(res!=1) return true;
	}

	//=======================
	//Write vertex positions
	//=======================
	Matrix3 tm = node->GetObjTMAfterWSM(time);
	for(int i=0;i<nVerts;i++)
	{
		UpdateProgress(100*i/(nVerts+nVects),"\"%s\" Writing vertices", node->GetName());

		Point3 v = verts[i].p;
		//Point3 n = vnorms[i].GetNormal(1);
		if(WorldSpace)
		{
			v=tm*v; 
			//n=Normalize(tm.VectorTransform(n));
		}

		Vec3 vv;
		vv.x=v.x;vv.y=v.y; vv.z=v.z;

		res=fwrite(&vv,sizeof(vv),1,f);
		if(res!=1) return true;
	}

	for(i=0;i<nVects;i++)
	{
		UpdateProgress(100*(i+nVerts)/(nVerts+nVects),"\"%s\" Writing vertices", node->GetName());

		Point3 v = vects[i].p;
		//Point3 n = vnorms[i].GetNormal(1);
		if(WorldSpace)
		{
			v=tm*v; 
			//n=Normalize(tm.VectorTransform(n));
		}

		Vec3 vv;
		vv.x=v.x;vv.y=v.y; vv.z=v.z;

		res=fwrite(&vv,sizeof(vv),1,f);
		if(res!=1) return true;
	}

	//=======================
	//Write Patches
	//=======================
	for(i=0;i<nPatches;i++)
	{
		Patch &patch=patches[i];

		int patchtype = (patch.type == PATCH_TRI) ? TRI_PATCH : QUAD_PATCH;
		res=fwrite(&patchtype,sizeof(patchtype),1,f);
		if(res!=1) return true;

		//write the vertex indices: we convert edge and internal vectors to vertices
		if(patchtype == TRI_PATCH)
		{
			CryTriPatch p;

			p.v[ 0] = patch.v[0];
			p.v[ 1] = patch.vec[0]+nVerts;
			p.v[ 2] = patch.vec[1]+nVerts;
			p.v[ 3] = patch.v[1];
			p.v[ 4] = patch.vec[2]+nVerts;
			p.v[ 5] = patch.vec[3]+nVerts;
			p.v[ 6] = patch.v[2];
			p.v[ 7] = patch.vec[4]+nVerts;
			p.v[ 8] = patch.vec[5]+nVerts;

			p.v[ 9] = patch.interior[0]+nVerts;
			p.v[10] = patch.interior[1]+nVerts;
			p.v[11] = patch.interior[2]+nVerts;

			res=fwrite(&p,sizeof(p),1,f);
			if(res!=1) return true;
		}
		else
		{
			CryQuadPatch p;

			p.v[ 0] = patch.v[0];
			p.v[ 1] = patch.vec[0]+nVerts;
			p.v[ 2] = patch.vec[1]+nVerts;
			p.v[ 3] = patch.v[1];
			p.v[ 4] = patch.vec[2]+nVerts;
			p.v[ 5] = patch.vec[3]+nVerts;
			p.v[ 6] = patch.v[2];
			p.v[ 7] = patch.vec[4]+nVerts;
			p.v[ 8] = patch.vec[5]+nVerts;
			p.v[ 9] = patch.v[3];
			p.v[10] = patch.vec[6]+nVerts;
			p.v[11] = patch.vec[7]+nVerts;

			p.v[12] = patch.interior[0]+nVerts;
			p.v[13] = patch.interior[1]+nVerts;
			p.v[14] = patch.interior[2]+nVerts;
			p.v[15] = patch.interior[3]+nVerts;

			res=fwrite(&p,sizeof(p),1,f);
			if(res!=1) return true;
		}
	}

	if(nTVerts)
	{
		//==================
		//Write Texture Vertices
		//==================
		for (i=0; i<nTVerts; i++) 
		{
			UpdateProgress(100*i/nTVerts,"\"%s\" Writing texture vertices", node->GetName());

			CryUV tv;
			tv.u = tverts[i].x;
			tv.v = tverts[i].y;

			res=fwrite(&tv, sizeof(tv), 1, f);
			if(res!=1) return true;
		}
	
		//=====================
		//Write Texture Patches
		//=====================
		for (i=0; i<nPatches; i++) 
		{
			UpdateProgress(100*i/nPatches,"\"%s\" Writing texture patches", node->GetName());
			
			
			//if(meshstatus.negative_scaling)
			//{
			//	mf.t0			= TexFaces[i].t[0];
			//	mf.t1			= TexFaces[i].t[2]; //reverse the order
			//	mf.t2			= TexFaces[i].t[1];
			//}
			//else
			//{
			//	mf.t0			= TexFaces[i].t[0];
			//	mf.t1			= TexFaces[i].t[1];
			//	mf.t2			= TexFaces[i].t[2];
			//}

			CryTexPatch tp;
			tp.t0=tpatches[i].tv[0];
			tp.t1=tpatches[i].tv[1];
			tp.t2=tpatches[i].tv[2];
			tp.t3=tpatches[i].tv[3];

			int res=fwrite(&tp, sizeof(tp), 1, f);
			if(res!=1) return true;
		}
	}

	if(HasBoneInfo)
	{
		Point3 *temp_vertices= (Point3 *)calloc(nVerts+nVects,sizeof(Point3));
		if(!temp_vertices) return true;
		for(int i=0;i<nVerts;i++) temp_vertices[i]=verts[i].p;
		for(i=0;i<nVects;i++) temp_vertices[i+nVerts]=vects[i].p;

		int res=SavePhysiqueInfo(f, node, temp_vertices);

		free(temp_vertices); temp_vertices=NULL;
		if(res) return true;
	}
	*/

	return false;
}

void CSExportUtility::SaveSourceInfo (FILE*f, int ChunkId)
{
	fpos_t fpos;
	fgetpos(f,&fpos);
	int chunk_id = ChunkList.GetID(NULL, ChunkId);
	if(chunk_id == -1)
		return;
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	TSTR& strFilePath = ip->GetCurFilePath();
	fwrite(strFilePath.data(), strFilePath.length()+1, 1, f);
	
	time_t curTime;
	time(&curTime);
	const char* szTime = asctime(localtime(&curTime));
	fwrite (szTime, strlen(szTime)+1, 1, f);

	char szName[0x100];
	DWORD dwSize = sizeof(szName);
  if(!GetUserName (szName, &dwSize))
	{
		dwSize = 0;
		*szName = '\0';
	}
	fwrite (szName, strlen(szName), 1, f);
	fwrite ("@", 1,1,f);

	if (!GetComputerName (szName, &dwSize))
	{
		dwSize = 0;
		*szName = '\0';
	}
	fwrite (szName, strlen(szName)+1, 1, f);

	fgetpos(f,&fpos);
	// pad to have 4-byte-aligned data
	if (fpos&3)
	{
		long nPad = 0;
		fwrite (&nPad, 1, 4-(fpos&3), f);
	}
}

bool CSExportUtility::SaveMeshObject(FILE* f, INode* node, int ChunkId, INodeTab& arrBones, unsigned nFlags)
{
	
	/*
	Modifier* pPhysiqueModifier = FindPhysiqueModifier(node->GetObjectRef());
	
	if (pPhysiqueModifier)
	{
		IPhysiqueExport* pPhysique = (IPhysiqueExport*)pPhysiqueModifier->GetInterface(I_PHYINTERFACE);
		pPhysique->SetInitialPose (true);
		pPhysiqueModifier->ReleaseInterface(I_PHYINTERFACE, pPhysique);
	}
	*/
	
	/*
	Tab<Modifier*> arrSkinModifiers;
	GetSkinModifiers (node, arrSkinModifiers);
	EnableModifiers (arrSkinModifiers, false);
	*/

	bool bResult = SaveMeshObject_1(f, node, ChunkId, arrBones, nFlags);
	//EnableModifiers (arrSkinModifiers, true);
	return bResult;
}

// SaveGeomNode ______________________________________________________________________________________
// bIsBoneMeshChunk - true if the bone chunk is being saved
bool CSExportUtility::SaveMeshObject_1(FILE* f, INode* node, int ChinkId, INodeTab& arrBones, unsigned nFlags)
{
	BOOL		HasBoneInfo		= pb_nodes->GetInt(PB_WRITE_WEIGHTS) && ((nFlags&(nSMF_DontSaveBoneInfo|nSMF_IsBoneMeshChunk))==0);
	if (HasBoneInfo && !IsHaveBoneInfo(node))
	{
		HasBoneInfo = false;
	}

	/*if (HasBoneInfo && arrBones.Count() > 0)
	{
		std::vector<Matrix3>arrMatrices;
		arrMatrices.resize (arrBones.Count());

		if (CalculateBoneInitPos(node, arrBones, &arrMatrices[0]) > 0)
		{
			for (unsigned i = 0; i < arrBones.Count(); ++i)
			{
				arrBones[i]->SetNodeTM(0,arrMatrices[i]);
				arrBones[i]->InvalidateTreeTM();
			}
		}

		node->InvalidateWS();
	}*/

	Object *der_obj;
	if(!node || !(der_obj = 
		node->
		EvalWorldState(0).obj
		//GetObjectRef ()
		)) 
		return true;

	Object* pBaseObject = GetObjectBeforeSkin(node);

	TimeValue	time	= GetTime();
	Object		*obj	= der_obj->Eval(time).obj;
	if(!obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID, 0)))  return false;
	TriObject	*tri	= (TriObject *) obj->ConvertToType(time, Class_ID(TRIOBJ_CLASS_ID, 0));
	Mesh		*mesh	= &tri->mesh;

	if (!mesh) 
		return true;

	//Watch out: there may be multiple nodes for this object
	char name[256];
	strcpy(name,node->GetName());

	//MeshStatus meshStatus;
	//int res=CheckDegenerateMesh(der_obj, mesh, time, &meshStatus);
	//if(res) return true;
	
	BOOL		WriteVCol		= pb_nodes->GetInt(PB_WRITE_VCOL);
	BOOL		AllowMultiUV	= pb_nodes->GetInt(PB_ALLOW_MULTIUV);
	BOOL		ExportMat		= pb_nodes->GetInt(PB_EXPORT_MAT);
	// we don't want to generate default UVs for bones
	BOOL    bGenDefUVs  = pb_nodes->GetInt (PB_GENERATE_DEFAULT_UVS) && ((nFlags&nSMF_IsBoneMeshChunk) == 0);

	BOOL		SplitSmooth = TRUE;

	Tab<Point3> arrInitPosVerts;
#ifdef EXPORT_INIT_POSE
	if (0 && HasBoneInfo)
	{
		arrInitPosVerts.SetCount(mesh->numVerts);
		CalculatePhysiqueInitialPose (node, &arrInitPosVerts[0]);
	}
#endif

	bool negateNormals = false;

	// maps split smooth groups mesh vertex indices --> original mesh indices
	std::vector<int> arrVertexMapSmooth;
	Mesh outMesh;
	SplitSmoothGroups ( *mesh,outMesh,arrVertexMapSmooth,negateNormals,arrInitPosVerts.Count()?&arrInitPosVerts[0]:mesh->verts);
	mesh = &outMesh;

	if (bGenDefUVs && mesh->getNumTVerts()==0)
	{
		GenerateDefaultUVs(*mesh);
	}

	//check if mesh is suitable
	int nVerts = mesh->getNumVerts();
	int nFaces = mesh->getNumFaces();
	int nTVerts= mesh->getNumTVerts();
	int nCVerts= mesh->getNumVertCol();
	int i;

	Face		*Faces	  = mesh->faces;
	Point3		*Verts	  = mesh->verts;
	TVFace		*TexFaces = mesh->tvFace;
	UVVert		*TexVerts = mesh->tVerts;
	VertColor	*ColVerts = mesh->vertCol;
	TVFace		*ColFaces = mesh->vcFace;

	//Turn off if object has no Physique on it.
/*	if(HasBoneInfo)
	{
		Modifier *phy_mod=FindPhysiqueModifier(der_obj);
		if(!phy_mod) HasBoneInfo=FALSE;
	}
*/
	//turn off vertex color export if mesh do not have them.
	if((!mesh->mapSupport(0)) || (!nCVerts)) WriteVCol=FALSE;

	//Check if MultiUV per vertex
	if((!AllowMultiUV) && nTVerts && (nTVerts!=nVerts))
	{
		CryMessageBox(hwnd_nodes,"There are multiple UV coordinates per vertex\nEither check your mapping or turn on 'Allow Multi UV/vert' option", name, MB_OK);
		return true;
	}

		//calc vertex normals
	/*
	BOOL ConsiderSmooth = false;
	Tab<VNormal> vnorms;
	CalcVertexNormals(vnorms, mesh, ConsiderSmooth);
	*/


	//==================
	//Write chunk header
	//==================
	UpdateProgress(100,"\"%s\" Writing chunk header", name);
	fpos_t fpos;
	fgetpos(f,&fpos);
	int chunk_id=ChunkList.GetID(node, ChinkId);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_Mesh at pos %d)",ch_ent.FileOffset);

	MESH_CHUNK_DESC chunk;
	chunk.chdr				= ch_ent;
	chunk.HasBoneInfo		= HasBoneInfo?true:false;
	chunk.HasVertexCol		= WriteVCol?true:false;
	chunk.nFaces			= nFaces;
	chunk.nTVerts			= nTVerts;
	chunk.nVerts			= nVerts;
	chunk.VertAnimID		= ChunkList.GetID(node, ChunkType_VertAnim);

	std::vector<CryUV> arrUVs;
	std::vector<CryTexFace> arrTexFaces;
	if(nTVerts)
	{
		UpdateProgress(0,"\"%s\" Preparing texture vertices", name);

		// copy all texture vertices into this array arrUVs
		arrUVs.resize (nTVerts);

		for (i=0; i<nTVerts; i++) 
		{
			CryUV& tv = arrUVs[i];
			tv.u = TexVerts[i].x;
			tv.v = TexVerts[i].y;
		}

		UpdateProgress(10,"\"%s\" Preparing texture faces", name);

		// copy all texture faces into this array arrTexFaces
		arrTexFaces.resize (nFaces);
		for (i=0; i<nFaces; i++) 
		{
			CryTexFace& mf = arrTexFaces[i];
			mf.t0			= TexFaces[i].t[0];
			mf.t1			= TexFaces[i].t[1];
			mf.t2			= TexFaces[i].t[2];
		}

		// validate and optimize the info
		UpdateProgress(20, "\"%s\" Cleaning up texture coordinates", name);
		if (!ValidateCryUVs(&arrTexFaces[0], arrTexFaces.size(), arrUVs.size()))
			return true;

		arrUVs.resize (CompactCryUVs(&arrTexFaces[0], arrTexFaces.size(), &arrUVs[0], arrUVs.size()));
	}
	nTVerts = chunk.nTVerts	= arrUVs.size();

	Matrix3 tmNode = node->GetNodeTM (GetTime());

	Matrix3 tm(1);

#ifdef EXPORT_INIT_POSE
	if (HasBoneInfo)
	{
		tm = tmNode;
		tm.Invert();
	}
	else
#endif
	{
		if ((nFlags & nSMF_DontApplyObjOffset)==0) {
			Point3 pos = node->GetObjOffsetPos();
			tm.PreTranslate(pos);
			Quat quat = node->GetObjOffsetRot();
			PreRotateMatrix(tm, quat);
			ScaleValue scaleValue = node->GetObjOffsetScale();
			ApplyScaling(tm, scaleValue);
		}
	}

	UpdateProgress(30, "\"%s\" Preparing vertices", name);
	std::vector<CryVertex> arrVerts;
	std::vector<CryFace> arrFaces;
	// fill the vertices in
	arrVerts.resize (nVerts);
	for(i=0;i<nVerts;i++)
	{
		Point3 v = tm.PointTransform(Verts[i]);
		Point3 n = mesh->getNormal(i);

		CryVertex& vert = arrVerts[i];
		vert.p.x=v.x; 
		vert.p.y=v.y; 
		vert.p.z=v.z;

		vert.n.x=n.x;
		vert.n.y=n.y;
		vert.n.z=n.z;
	}
	
	UpdateProgress(40, "\"%s\" Preparing faces", name);
	arrFaces.resize (nFaces);
	
	for (i=0; i<nFaces; i++) 
	{
		CryFace& mf = arrFaces[i];
		mf.v0			= Faces[i].v[0];
		mf.v1			= Faces[i].v[1];	
		mf.v2			= Faces[i].v[2];

		mf.SmGroup		= Faces[i].smGroup;
		mf.MatID = -1;
		int matid = Faces[i].getMatID();
		int matcnt=-1;
		Mtl *dest_material;

		//get the material used for this

		if (node && node->GetMtl())
		{
			Mtl *mtl = node->GetMtl();

			if (mtl->IsMultiMtl())
				dest_material = mtl->GetSubMtl(matid);
			else
				dest_material = mtl;

			// find it in the chunk list and count all standard materials before it;
			for (int ctr=0;ctr<ChunkList.ptr_list.Count();ctr++)
				if (ChunkList.ch_list[ctr].ChunkType == ChunkType_Mtl)
				{
					Mtl *current = (Mtl *) ChunkList.ptr_list[ctr];
					if (!current || !current->IsMultiMtl())  matcnt++;

					if (current==dest_material) 
					{
						mf.MatID = matcnt;
						break;
					}
				}

		}
		else
		{
			mf.MatID = getDefaultMtlIndex ();
		}
	}

	UpdateProgress(50, "\"%s\" Cleaning up coordinates", name);
	if (!ValidateCryVerts (&arrFaces[0], arrFaces.size(), arrVerts.size()))
		return true;

	// maps splitted vertices to compacted vertices
	std::vector<int> arrVertMap;
	// compacts (detects and throws away the unreferred vertices)
	CompactCryVertMap (&arrFaces[0], arrFaces.size(), nVerts,  arrVertMap);

	// construct the vertex map and vertex array for passing to bone binding saving routines
	// arrVertexMap maps compacted vertices -> original Max vertices (through the split vertices)
	std::vector<int> arrVertexMap;
	// vertices corresponding to the compacted vertex array
	std::vector<Point3> arrVertsForBoneSavingRoutines;
	{
		arrVertexMap.resize(arrVertMap.size());
		arrVertsForBoneSavingRoutines.resize(arrVertMap.size());
		for (unsigned i = 0; i < arrVertMap.size(); ++i)
		{
			arrVertsForBoneSavingRoutines[i] = Verts[arrVertMap[i]];
			arrVertexMap[i] = arrVertexMapSmooth[arrVertMap[i]];
		}
	}

	chunk.nFaces = nFaces;
	chunk.nVerts = arrVertMap.size();//nVerts;

	int res=fwrite(&chunk,sizeof(chunk),1,f);
	if(res!=1) 
		return true;

	//======================
	//Write vertex positions
	//======================

	UpdateProgress(60, "\"%s\" Writing optimized coordinates", name);
	for (i = 0; i < (int)arrVertMap.size(); ++i)
	{
		res = fwrite (&arrVerts[arrVertMap[i]], sizeof(CryVertex), 1, f);
		if (res != 1)
			return true;
	}

	UpdateProgress(65, "\"%s\" Writing optimized faces", name);
	res = fwrite(&arrFaces[0], sizeof(arrFaces[0]), arrFaces.size(), f);
	if (res != arrFaces.size())
		return true;

	if(nTVerts)
	{
		// write to the disk
		UpdateProgress (75, "\"%s\" Writing texture vertices", name);
		res = fwrite (&arrUVs[0], sizeof(arrUVs[0]), arrUVs.size(), f);
		if(res!=arrUVs.size()) 
			return true;

		UpdateProgress (85, "\"%s\" Writing texture faces", name);
		res = fwrite (&arrTexFaces[0], sizeof(arrTexFaces[0]), arrTexFaces.size(), f);
		if (res != arrTexFaces.size())
			return true;
	}

	if(HasBoneInfo)
	{
		UpdateProgress (90, "\"%s\" Writing bone info", name);
		//we are sure that only "one" node uses this derived_object
		if (SaveBoneInfo(f,node,&arrVertsForBoneSavingRoutines[0],arrVertexMap))
			return true;
		//int res=SavePhysiqueInfo(f, node, Verts);
		//if(res) 
		//	return true;
	}

	if(WriteVCol)
	{
		UpdateProgress(100,"\"%s\" Writing Vertex Colors", name);
		CryIRGB *colors=(CryIRGB*) calloc(nVerts, sizeof(CryIRGB));
		if(!colors) 
			return true;
		if(PrepareVertexColors(colors, mesh)) 
			return true;

		//===================
		//Write Vertex Colors
		//===================
		for (i = 0; i < (int)arrVertMap.size(); ++i)
		{
			res = fwrite (colors+arrVertMap[i], sizeof(CryIRGB), 1, f);
			if (res != 1)
				return true;
		}

		free(colors);
	}
	
	if(obj != tri) 
	{
		DebugPrint("Deleting intermediate tri object of %s",name);
		tri->DeleteThis();
		DebugPrint(" OK");
	}

	if (nFlags & nSMF_SaveMorphTargets)
	{
		Matrix3 tm(1);
		Point3 pos = node->GetObjOffsetPos();
		tm.PreTranslate(pos);
		Quat quat = node->GetObjOffsetRot();
		PreRotateMatrix(tm, quat);
		ScaleValue scaleValue = node->GetObjOffsetScale();
		ApplyScaling(tm, scaleValue);
#ifdef _DEBUG
		AffineParts parts;
		decomp_affine(tmNode, &parts);
		Matrix3 tmNodeNS; // node without scaling
		parts.q.MakeMatrix (tmNodeNS);
		tmNodeNS.SetRow (3, parts.t);
#endif
		if (SaveMorphTargets(f, node, tm, arrVertexMap, ch_ent.ChunkID))
			return true; // error
	}

	if ((nFlags & nSMF_SaveInitBonePos) && HasBoneInfo)
	{
		if (SaveBoneInitialPos (f,node,arrBones,ch_ent.ChunkID))
			return true; // error
	}

	return false; // success, no error
} 

class TexmapNameEnumCallback : public NameEnumCallback {
public:
	TSTR file;
	virtual void RecordName(TCHAR *name)
	{
		if (file.Length() == 0)
			file = name;
	}
};

/*
class MyMtlBase: public ReferenceTarget, public ISubMap {
public:
	friend class Texmap;
	TSTR name;
	ULONG mtlFlags;
	int defaultSlotType;
	int activeCount;
};
*/

void CSExportUtility::FillDefaultTextureInfo (TextureMap3 &dst, int id)
{
	memset (&dst, 0, sizeof(dst));

	dst.uoff_ctrlID = dst.urot_ctrlID = dst.uscl_ctrlID = dst.voff_ctrlID = dst.vrot_ctrlID = dst.vscl_ctrlID = dst.wrot_ctrlID = -1;
	dst.uscl_val = dst.vscl_val = 1.0f;

	dst.Amount = 90;

	strncpy( dst.name, "Textures\\Defaults\\DefaultNoUVs.dds",sizeof(dst.name) );
	dst.type = TEXMAP_ENVIRONMENT;

	//dst.flags |= TEXMAP_NOMIPMAP;
}

void CSExportUtility::FillTextureInfo(TextureMap3 &dst, StdMat *mtl,int id)
{
	//initialize
	memset(&dst,0,sizeof(dst));
	dst.uoff_ctrlID = dst.urot_ctrlID = dst.uscl_ctrlID = dst.voff_ctrlID = dst.vrot_ctrlID = dst.vscl_ctrlID = dst.wrot_ctrlID = -1;
	dst.uscl_val = dst.vscl_val = 1.0f;

	if (!mtl->MapEnabled(id))
		return;

	TimeValue	time	= GetTime();

	// Fill texture map ammount.
	dst.Amount = (int)(100.0f*mtl->GetTexmapAmt( id,time ));
	
	memset( dst.Reserved,0,sizeof(dst.Reserved) );

	Texmap *src = mtl->GetSubTexmap(id);
	if(!src)
		return;

	//name
	if(src->ClassID() == Class_ID(BMTEX_CLASS_ID, 0))
	{
		BitmapTex *bmt = (BitmapTex*)src;
		TSTR mapFile,path;
		mapFile = bmt->GetMapName();
		path = GetRelativePath(mapFile);
		if (path.Length() >= sizeof(dst.name))
		{
			char str[1024];
			sprintf( str,"Ttexture name %s longer then %d characters",path.data(),sizeof(dst.name) );
			MessageBox( NULL,str,"Warning!!!",MB_OK );
		}
		strncpy( dst.name,path,sizeof(dst.name) );

		if (bmt->GetFilterType() == FILTER_NADA)
		{
			dst.flags |= TEXMAP_NOMIPMAP;
		}
	}

	if(src->ClassID() == Class_ID(ACUBIC_CLASS_ID, 0))
	{
		StdCubic *cbc = (StdCubic*)src;
		if (cbc->GetDoNth())
		{
			dst.nthFrame = cbc->GetNth();
		}
		dst.refSize = cbc->GetSize(0);
		dst.refBlur = cbc->GetBlur(0);
		int reqFlags = cbc->Requirements(0);
		if (reqFlags&MTLREQ_AUTOREFLECT || reqFlags&MTLREQ_AUTOMIRROR)
		{
			dst.type = TEXMAP_AUTOCUBIC;
		}
		else
		{
			dst.type = TEXMAP_CUBIC;
			TSTR sn,sp,sf;
	
			TexmapNameEnumCallback nameEnum;
			cbc->EnumAuxFiles( nameEnum,FILE_ENUM_ALL );

			TSTR path = GetRelativePath(nameEnum.file);
			if (path.Length() >= sizeof(dst.name))
			{
				char str[1024];
				sprintf( str,"Ttexture name %s longer then %d characters",path.data(),sizeof(dst.name) );
				MessageBox( NULL,str,"Warning!!!",MB_OK );
			}
			strncpy( dst.name,path,sizeof(dst.name) );
		}
	}

	UVGen *a1 = src->GetTheUVGen();
	XYZGen *a2 = src->GetTheXYZGen();

	//get UVGen
	UVGen *uvgen=src->GetTheUVGen();
	if(!uvgen) return;
	if(uvgen->ClassID()!=stdUVClassID) return;
	StdUVGen	*stduv	= (StdUVGen *)uvgen;
	
	//Get Tiling Flags
	dst.utile	= stduv->GetFlag(U_WRAP)?true:false;
	dst.vtile	= stduv->GetFlag(V_WRAP)?true:false;
	dst.umirror = stduv->GetFlag(U_MIRROR)?true:false;
	dst.vmirror = stduv->GetFlag(V_MIRROR)?true:false;

	int coordMap = stduv->GetCoordMapping(0);

	/*
	stduv->InitSlotType( MAPSLOT_ENVIRON );
	src->InitSlotType( MAPSLOT_ENVIRON );
	int type = src->MapSlotType(0);
	type = stduv->MapSlotType(0);


	type = ((MyMtlBase*)stduv)->defaultSlotType;
	*/

	//if (coordMap == UVMAP_EXPLICIT){}
	if (coordMap == UVMAP_SPHERE_ENV || coordMap == UVMAP_CYL_ENV || UVMAP_SHRINK_ENV)
	{
		dst.type = TEXMAP_ENVIRONMENT;
	}
	if (coordMap == UVMAP_SCREEN_ENV)
	{
		//dst.type = TEXMAP_SCREENENVIRONMENT;
	}
	
	//Get UV values
	dst.uoff_val = stduv->GetUOffs(time);
	dst.uscl_val = stduv->GetUScl(time);
	dst.urot_val = stduv->GetUAng(time);
	dst.voff_val = stduv->GetVOffs(time);
	dst.vscl_val = stduv->GetVScl(time);
	dst.vrot_val = stduv->GetVAng(time);
	dst.wrot_val = stduv->GetWAng(time);

	//Get UV Controllers
	IParamBlock *pb		= (IParamBlock *)stduv->GetReference(0);
	if(!pb) return;

	Control *cont=NULL;
	if(cont=pb->GetController(PB_UOFFS))	dst.uoff_ctrlID = ChunkList.GetID(cont,ChunkType_Controller);
	if(cont=pb->GetController(PB_USCL))		dst.uscl_ctrlID = ChunkList.GetID(cont,ChunkType_Controller);
	if(cont=pb->GetController(PB_UANGLE))	dst.urot_ctrlID = ChunkList.GetID(cont,ChunkType_Controller);
	if(cont=pb->GetController(PB_VOFFS))	dst.voff_ctrlID = ChunkList.GetID(cont,ChunkType_Controller);
	if(cont=pb->GetController(PB_VSCL))		dst.vscl_ctrlID = ChunkList.GetID(cont,ChunkType_Controller);
	if(cont=pb->GetController(PB_VANGLE))	dst.vrot_ctrlID = ChunkList.GetID(cont,ChunkType_Controller);
	if(cont=pb->GetController(PB_WANGLE))	dst.wrot_ctrlID = ChunkList.GetID(cont,ChunkType_Controller);
}


// saves the default material chunk, if needed
bool CSExportUtility::SaveDefaultMaterial (FILE* f, bool bBuilding)
{
	if (m_nDefaultMtlIndex == -1)
		return true; // there's no default material (it's not required)

	fpos_t fpos;
	fgetpos(f, &fpos);
	CHUNK_HEADER& ch = *ChunkList.ch_list.Addr(m_nDefaultMtlChunkId);
	ch.FileOffset = (int)fpos;
	DebugPrint ("\nDefault Material at pos %d", ch.FileOffset);

	MTL_CHUNK_DESC_EXPORTER chunk;
	memset (&chunk, 0, sizeof(chunk));
	strncpy (chunk.name, "s_nouvmap(TBD)/mat_concrete", sizeof(chunk.name));
	chunk.MtlType = MTL_STANDARD;
	chunk.chdr = ch;

	Color ca (1, 1, 1);
	Color cd (0.5882f, 0.5882f, 0.5882f);
	Color cs (0.8f, 0.5f, 0.5f);

	chunk.col_a.r=(unsigned char)(255*ca.r);
	chunk.col_a.g=(unsigned char)(255*ca.g);
	chunk.col_a.b=(unsigned char)(255*ca.b);
	chunk.col_d.r=(unsigned char)(255*cd.r);
	chunk.col_d.g=(unsigned char)(255*cd.g);
	chunk.col_d.b=(unsigned char)(255*cd.b);
	chunk.col_s.r=(unsigned char)(255*cs.r);
	chunk.col_s.g=(unsigned char)(255*cs.g);
	chunk.col_s.b=(unsigned char)(255*cs.b);

	chunk.Dyn_Bounce			= 1;
	chunk.Dyn_StaticFriction	= 0;
	chunk.Dyn_SlidingFriction	= 0;

	chunk.specLevel = 0.5;
	chunk.specShininess = 0.3f;
	chunk.opacity = 1;
	chunk.selfIllum = bBuilding ? 0 : 0.5f;
	int trType = 0;
	if (trType == TRANSP_SUBTRACTIVE)
		chunk.flags |= MTLFLAG_SUBTRACTIVE;
	if (trType == TRANSP_ADDITIVE)
		chunk.flags |= MTLFLAG_ADDITIVE;
	//chunk.flags |= MTLFLAG_FACEMAP;
	//chunk.flags |= MTLFLAG_WIRE;
	//chunk.flags |= MTLFLAG_2SIDED;

	FillDefaultTextureInfo(chunk.tex_d, ID_DI );
	int res=fwrite(&chunk,sizeof(chunk),1,f);
	if(res!=1) return false;
	return true;
}

bool CSExportUtility::SaveMaterial(FILE* f, Mtl* mat)
{
	TimeValue	time	= GetTime();
	
	if(!mat) return false;
	
	//==================
	//Write chunk header
	//==================
	fpos_t fpos;
	fgetpos(f,&fpos);
	//append to chunk list
	int chunk_id=ChunkList.GetID(mat,ChunkType_Mtl);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_Mtl at pos %d)",ch_ent.FileOffset);

	MTL_CHUNK_DESC_EXPORTER chunk;
	memset(&chunk,0,sizeof(chunk));
	strcpy(chunk.name ,mat->GetName());

	if(mat->ClassID() == Class_ID(DMTL_CLASS_ID, 0) || mat->ClassID() == Class_ID(DMTL2_CLASS_ID, 0) ) 
		chunk.MtlType = MTL_STANDARD;
	else if(mat->ClassID() == Class_ID(MULTI_CLASS_ID, 0))
		chunk.MtlType = MTL_MULTI;
	else if(mat->ClassID() == Class_ID(DOUBLESIDED_CLASS_ID, 0))
		chunk.MtlType = MTL_2SIDED;
	else 
		chunk.MtlType = MTL_UNKNOWN;
	chunk.chdr		= ch_ent;

	if(chunk.MtlType == MTL_STANDARD)
	{
		StdMat* stdmat = (StdMat *)mat;

		bool bCrytekShader = false;
		if (stdmat->SupportsShaders())
		{
			StdMat2* stdmat2 = (StdMat2*)stdmat;
			Shader *shader = stdmat2->GetShader();
			if (shader && (shader->ClassID() == CrytekShaderClassID))
			{
				// This material uses Crytek Shader.
				bCrytekShader = true;
				FillCrytekMaterial( stdmat2,chunk );
			}
		}
		if (!bCrytekShader)
			FillStdMaterial( stdmat,chunk );
	}
	else if(chunk.MtlType == MTL_MULTI || chunk.MtlType == MTL_2SIDED)
	{
		chunk.nChildren= mat->NumSubMtls();
	}
			
	int res=fwrite(&chunk,sizeof(chunk),1,f);
	if(res!=1) return true;

	//=======================
	//Write sub Material ID's
	//=======================
	if(chunk.MtlType == MTL_MULTI || chunk.MtlType == MTL_2SIDED)
	{
		for(int i=0;i<chunk.nChildren;i++)
		{
			Mtl *submtl=mat->GetSubMtl(i);
			int childID = ChunkList.GetID(submtl,ChunkType_Mtl);
			
			int res=fwrite(&childID,sizeof(childID),1,f);
			if(res!=1) return true;
		}
	}

	return false;
}

void CSExportUtility::FillCrytekMaterial( StdMat2* mat,MTL_CHUNK_DESC_EXPORTER &chunk )
{
	TimeValue	time	= GetTime();

	//chunk.MtlType = MTL_CRYTEKSHADER;
	chunk.flags |= MTLFLAG_CRYSHADER;

	//set colors
	Color ca = mat->GetAmbient(time);
	Color cd = mat->GetDiffuse(time);
	Color cs = mat->GetSpecular(time);
	chunk.col_a.r=(unsigned char)(255*ca.r);
	chunk.col_a.g=(unsigned char)(255*ca.g);
	chunk.col_a.b=(unsigned char)(255*ca.b);
	chunk.col_d.r=(unsigned char)(255*cd.r);
	chunk.col_d.g=(unsigned char)(255*cd.g);
	chunk.col_d.b=(unsigned char)(255*cd.b);
	chunk.col_s.r=(unsigned char)(255*cs.r);
	chunk.col_s.g=(unsigned char)(255*cs.g);
	chunk.col_s.b=(unsigned char)(255*cs.b);

	chunk.Dyn_Bounce			= mat->GetDynamicsProperty(time, 0, DYN_BOUNCE);
	chunk.Dyn_StaticFriction	= mat->GetDynamicsProperty(time, 0, DYN_STATIC_FRICTION);
	chunk.Dyn_SlidingFriction	= mat->GetDynamicsProperty(time, 0, DYN_SLIDING_FRICTION);

	chunk.specLevel = mat->GetShinStr(0);
	chunk.specShininess = mat->GetShininess(0);
	chunk.opacity = mat->GetOpacity(0);
	chunk.selfIllum = mat->GetSelfIllum(0);

	CrytekShader *shader = (CrytekShader*)mat->GetShader();
	
	int trType = 0;

	/*
	if (trType == TRANSP_SUBTRACTIVE)
		chunk.flags |= MTLFLAG_SUBTRACTIVE;
	if (trType == TRANSP_ADDITIVE)
		chunk.flags |= MTLFLAG_ADDITIVE;
		*/

	if (shader->IsAlphaBlend())
	{
		if (chunk.opacity == 1)
		{
			// Some opacity.
			chunk.opacity = 0.99f;
		}
		if (shader->IsAdditive())
		{
			chunk.flags |= MTLFLAG_ADDITIVE;
		}
	}
	else
	{
		// No alpha blending.
		chunk.opacity = 1;
	}

	if (shader->IsAdditiveDecal())
	{
		chunk.flags |= MTLFLAG_ADDITIVEDECAL;
	}

	if (shader->IsUseGlossiness())
	{
		chunk.flags |= MTLFLAG_USEGLOSSINESS;
	}

	if (shader->IsPhysicalize())
	{
		chunk.flags |= MTLFLAG_PHYSICALIZE;
		chunk.Dyn_Bounce = 1;
	}
	else
	{
		chunk.Dyn_Bounce = 0;
	}

	if (shader->IsNoShadow())
	{
		chunk.Dyn_StaticFriction = 1;
	}
	else
	{
		chunk.Dyn_StaticFriction = 0;
	}

	if (mat->GetFaceMap())
		chunk.flags |= MTLFLAG_FACEMAP;
	if (mat->GetWire())
		chunk.flags |= MTLFLAG_WIRE;
	if (mat->GetTwoSided())
		chunk.flags |= MTLFLAG_2SIDED;

	float alphaTest = shader->GetAlphaTest();
	chunk.alphaTest = alphaTest;

	const char* shaderName = shader->GetShaderName();
	const char* surfaceName = shader->GetSurfaceName();

	std::string strFullName = chunk.name;
	if (*shaderName)
	{
		strFullName += "(";
		strFullName += shaderName;
		strFullName += ")";
	}
	if (*surfaceName)
	{
		strFullName += "/";
		strFullName += surfaceName;
	}

	if (strFullName.length() > sizeof(chunk.name)-1)
	{
		// Warning surface name too long.
		char str[1024];
		_snprintf( str, sizeof(str), "Surface material name %s is longer than %d character\nIt will be truncated, the results may be unpredictable.", strFullName.c_str(), sizeof(chunk.name)-1);
		MessageBox( NULL,str,_T("Warning"), MB_OK|MB_ICONWARNING|MB_TOPMOST );
	}

	strncpy (chunk.name, strFullName.c_str(), sizeof(chunk.name));
	chunk.name[sizeof(chunk.name)-1] = '\0';

	// Not used.
	FillTextureInfo(chunk.tex_a, mat,CH_PR );
	FillTextureInfo(chunk.tex_d, mat,CH_DI );
	FillTextureInfo(chunk.tex_s, mat,CH_GL ); // tex_s This Gloss (shininess) Specular Map.
	FillTextureInfo(chunk.tex_o, mat,CH_DC );
	FillTextureInfo(chunk.tex_b, mat,CH_BN );
	FillTextureInfo(chunk.tex_g, mat,CH_SP ); // tex_g This Color Specular Map.
	FillTextureInfo(chunk.tex_fl, mat,CH_DT );
	FillTextureInfo(chunk.tex_rl, mat,CH_RL );
	FillTextureInfo(chunk.tex_subsurf, mat,CH_SSS );
	FillTextureInfo(chunk.tex_det, mat,CH_BU );

}

//////////////////////////////////////////////////////////////////////////
void CSExportUtility::FillStdMaterial( StdMat* mat,MTL_CHUNK_DESC_EXPORTER &chunk )
{
	TimeValue	time	= GetTime();
	//set colors
	Color ca = mat->GetAmbient(time);
	Color cd = mat->GetDiffuse(time);
	Color cs = mat->GetSpecular(time);
	chunk.col_a.r=(unsigned char)(255*ca.r);
	chunk.col_a.g=(unsigned char)(255*ca.g);
	chunk.col_a.b=(unsigned char)(255*ca.b);
	chunk.col_d.r=(unsigned char)(255*cd.r);
	chunk.col_d.g=(unsigned char)(255*cd.g);
	chunk.col_d.b=(unsigned char)(255*cd.b);
	chunk.col_s.r=(unsigned char)(255*cs.r);
	chunk.col_s.g=(unsigned char)(255*cs.g);
	chunk.col_s.b=(unsigned char)(255*cs.b);

	chunk.Dyn_Bounce			= mat->GetDynamicsProperty(time, 0, DYN_BOUNCE);
	chunk.Dyn_StaticFriction	= mat->GetDynamicsProperty(time, 0, DYN_STATIC_FRICTION);
	chunk.Dyn_SlidingFriction	= mat->GetDynamicsProperty(time, 0, DYN_SLIDING_FRICTION);

	chunk.specLevel = mat->GetShinStr(0);
	chunk.specShininess = mat->GetShininess(0);
	chunk.opacity = mat->GetOpacity(0);
	if (!mat->GetSelfIllumColorOn())
	{
		chunk.selfIllum = mat->GetSelfIllum(0);
	}

	chunk.flags = 0;

	int trType = mat->GetTransparencyType();
	if (trType == TRANSP_SUBTRACTIVE)
		chunk.flags |= MTLFLAG_SUBTRACTIVE;
	if (trType == TRANSP_ADDITIVE)
		chunk.flags |= MTLFLAG_ADDITIVE;
	if (mat->GetFaceMap())
		chunk.flags |= MTLFLAG_FACEMAP;
	if (mat->GetWire())
		chunk.flags |= MTLFLAG_WIRE;
	if (mat->GetTwoSided())
		chunk.flags |= MTLFLAG_2SIDED;

	{	
		int gloss = -1;
		for (int i = 0; i < 50; i++)
		{
			if (mat->MapEnabled(i))
			{
				gloss = i;
			}
		}

		FillTextureInfo(chunk.tex_a, mat,ID_AM );
		FillTextureInfo(chunk.tex_d, mat,ID_DI );
		FillTextureInfo(chunk.tex_s, mat,ID_SP );
		FillTextureInfo(chunk.tex_o, mat,ID_OP );
		FillTextureInfo(chunk.tex_b, mat,ID_BU );
		FillTextureInfo(chunk.tex_g, mat,ID_SS );
		FillTextureInfo(chunk.tex_fl,mat,ID_FI );
		FillTextureInfo(chunk.tex_rl, mat,ID_RL );
		FillTextureInfo(chunk.tex_subsurf, mat,ID_RR );
		FillTextureInfo(chunk.tex_det, mat,ID_DP );
	}
}

//////////////////////////////////////////////////////////////////////////
bool CSExportUtility::SaveSceneProps(FILE* f)
{
	//=======================
	//Write SceneProperties
	//=======================
	int numProps = ip->GetNumProperties(PROPSET_USERDEFINED);
	if(numProps)
	{
		//==================
		//Write chunk header
		//==================
		fpos_t fpos;
		fgetpos(f,&fpos);
		//append to chunk list
		int chunk_id=ChunkList.GetID(NULL,ChunkType_SceneProps);
		assert(chunk_id != -1);
		CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
		ch_ent.FileOffset	 = int(fpos);
		DebugPrint("\n(ChunkType_SceneProps at pos %d)",ch_ent.FileOffset);


		SCENEPROPS_CHUNK_DESC chunk;
		chunk.chdr				= ch_ent;
		chunk.nProps			= numProps;
		int res=fwrite(&chunk,sizeof(chunk),1,f);
		if(res!=1) return true;

		//==================
		//Write bone names
		//==================
		for (int i=0; i<numProps; i++) 
		{
			UpdateProgress(100*i/numProps,"Writing SeceneProps...");
			const PROPSPEC*		pPropSpec	= ip->GetPropertySpec(PROPSET_USERDEFINED, i);
			const PROPVARIANT*	pPropVar	= ip->GetPropertyVariant(PROPSET_USERDEFINED, i);

			SCENEPROP_ENTITY item;
			if (pPropSpec->ulKind == PRSPEC_LPWSTR) //we support only the string
			{
				_tcscpy(item.name, TSTR(pPropSpec->lpwstr));
				VariantToString(pPropVar, item.value, sizeof(item.value));
			}
			else
			{
				_tcscpy(item.name ,"<<ERROR>>");
				_tcscpy(item.value, "Invalid Property Type");
			}
			int res=fwrite(&item,sizeof(item),1,f);
			if(res!=1) return true;
		}
	}

	return false;
}

bool CSExportUtility::SaveTiming(FILE* f)
{
	//=======================
	//Write TimeRanges
	//=======================
	int anim_start;
	int anim_end;
	int	manualrange = pb_timing->GetInt(PB_MAN_RANGE);

	if(manualrange)
	{
		anim_start	= pb_timing->GetInt(PB_RANGE_START);
		anim_end	= pb_timing->GetInt(PB_RANGE_END);
	}
	else
	{
		Interval ii =ip->GetAnimRange();
		anim_start	=ii.Start() / GetTicksPerFrame();
		anim_end	=ii.End()   / GetTicksPerFrame();
	}

	//==================
	//Write chunk header
	//==================
	fpos_t fpos;
	fgetpos(f,&fpos);
	//append to chunk list
	int chunk_id=ChunkList.GetID(NULL,ChunkType_Timing);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_Timing at pos %d)",ch_ent.FileOffset);

	TIMING_CHUNK_DESC chunk;
	chunk.chdr					= ch_ent;
	chunk.global_range.start	= anim_start;
	chunk.global_range.end		= anim_end;
	chunk.SecsPerTick			= TicksToSec(1);
	chunk.TicksPerFrame			= GetTicksPerFrame();
	strcpy(chunk.global_range.name,"GlobalRange");
	chunk.nSubRanges = 0;

	chunk.nSubRanges			= Ranges.Count();
	chunk.nSubRanges			= 0;

	if (animationName.Length() > 0)
		chunk.nSubRanges = 1;

	int res=fwrite(&chunk,sizeof(chunk),1,f);
	if(res!=1) return true;

	if (animationName.Length() > 0)
	{
		RANGE_ENTITY ent;
		strcpy( ent.name,animationName );
		ent.start = anim_start;
		ent.end = anim_end;
		int res=fwrite(&ent,sizeof(ent),1,f);
		if(res!=1) return true;
	}

	/*
	for(int i=0;i<chunk.nSubRanges;i++)
	{
		RANGE_ENTITY ent=Ranges[i];
		int res=fwrite(&ent,sizeof(ent),1,f);
		if(res!=1) return true;
	}
	*/

	return false;
}


//////////////////////////////////////////////////////////////////////////
// Saves the chunk containing initial poses of each bone
bool CSExportUtility::SaveBoneInitialPos (FILE* f, INode* pSkin, INodeTab &arrBones, int nChunkIdMesh)
{	
	if (arrBones.Count() == 0)
		return true;
	std::vector<Matrix3> arrBoneInitPos;
	arrBoneInitPos.resize (arrBones.Count());
	if (CalculateBoneInitPos(pSkin, arrBones,&arrBoneInitPos[0]) == 0)
		return true;

	fpos_t fpos;
	fgetpos (f, &fpos);

	BONEINITIALPOS_CHUNK_DESC_0001 ChunkHeader;
	ChunkHeader.nChunkIdMesh = nChunkIdMesh;
	ChunkHeader.numBones = arrBones.Count();

	// append
	CHUNK_HEADER ch_ent;
	ch_ent.FileOffset = (int)fpos;
	ch_ent.ChunkType  = ChunkType_BoneInitialPos;
	ch_ent.ChunkVersion = ChunkHeader.VERSION;
	ChunkList.Append(NULL, &ch_ent, true);

	if (1 != fwrite (&ChunkHeader,sizeof(ChunkHeader),1,f))
		return true; // error

	// convert Matrix3 into 4x3 Cry matrices to put into the chunk
	for (unsigned i = 0; i < arrBoneInitPos.size(); ++i)
	{
		SBoneInitPosMatrix mtx;
		for (int x = 0; x < 4; ++x)
		{
			Point3 ptRow = arrBoneInitPos[i].GetRow (x);
			for (int y = 0; y < 3; ++y)
				mtx[x][y]= ptRow[y];
		}
		if (1 != fwrite (&mtx, sizeof(mtx), 1, f))
			return true;
	}

	return false;
}


bool CSExportUtility::SaveBoneNameList(FILE* f)
{
	//==================
	//Write chunk header
	//==================
	fpos_t fpos;
	fgetpos(f,&fpos);
	//append to chunk list
	int chunk_id=ChunkList.GetID(NULL,ChunkType_BoneNameList);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_BoneNameList at pos %d)",ch_ent.FileOffset);


	UpdateProgress(100,"Writing bone name list");

	BONENAMELIST_CHUNK_DESC_0745 chunk;
	chunk.numEntities = BoneList.arrNames.size();
	int res=fwrite(&chunk,sizeof(chunk),1,f);
	if(res!=1) return true;

	//==================
	//Write bone names
	//==================
	for (int i = 0; i < chunk.numEntities; ++i)
	{
		const std::string& strName = BoneList.arrNames[i];
		int res = fwrite (strName.c_str(),strName.length()+1,1,f);
		if(res!=1) return true;
	}
	// write the terminating double-\0
	res = fwrite ("\0", 1, 1, f);
	if (res != 1) return true;

	// move the file pointer position so that it's 4-byte aligned
	fgetpos(f,&fpos);
	unsigned long nPad = 0; // pad bytes

	if (fpos & 3)
	{
		// unaligned pos
		fwrite (&nPad, 1, size_t(4- (fpos&3)), f);
	}

	return false;
}

inline Vec3d& operator << (Vec3d& left, Point3& right)
{
	left.x = right.x;
	left.y = right.y;
	left.z = right.z;
	return left;
}


// for each bone, find the attached light(s) and  add them to the light list, 
// and to the chunk list to export
void CSExportUtility::PrepareBoneLightBinding()
{
	m_arrLightBinding.clear();
	// scan each bone for presence of light children
	for (int nBoneId = 0; nBoneId < BoneList.nodes.Count(); ++nBoneId)
	{
		INode* pBone = BoneList.nodes[nBoneId];
		int numChildren = pBone->NumberOfChildren();
		for (int nChild = 0; nChild < numChildren;++nChild)
		{
			INode* pChild = pBone->GetChildNode(nChild);
			TimeValue time = GetTime();
			Object* pObj = pChild->GetObjectRef()->Eval(time).obj;

			if (pObj->SuperClassID() == LIGHT_CLASS_ID && ChunkList.GetID(pChild,ChunkType_Light) == -1)
			{
				CHUNK_HEADER ch;

				ch.ChunkType = ChunkType_Node;
				ch.ChunkVersion = NODE_CHUNK_DESC_VERSION;
				ChunkList.Append(pChild,&ch,true);

				ch.ChunkType = ChunkType_Light;
				ch.ChunkVersion = LIGHT_CHUNK_DESC_VERSION;
				ch.FileOffset = -1;
				int nLightChunkId = ChunkList.Append(pChild, &ch, true);

				SBoneLightBind bind;
				bind.nLightChunkId = nLightChunkId;
				bind.nBoneId = nBoneId;
				// local light TM (relative to the parent)
				Matrix3 tmLocalLight = pChild->GetNodeTM(time) * Inverse (pChild->GetParentTM(time));

				bind.vLightOffset << tmLocalLight.GetTrans();
				Quat qRotLogN = Quat (tmLocalLight).LogN();
        bind.vRotLightOrientation.x = qRotLogN.x;
        bind.vRotLightOrientation.y = qRotLogN.y;
        bind.vRotLightOrientation.z = qRotLogN.z;

				m_arrLightBinding.push_back (bind);
			}
		}
	}
}

// saves the light-bone binding info
bool CSExportUtility::SaveBoneLightBinding (FILE* f)
{
	if (m_arrLightBinding.empty())
		return false;

	fpos_t fpos;
	fgetpos(f,&fpos);
	//append to chunk list
	int chunk_id = ChunkList.GetID(NULL,ChunkType_BoneLightBinding);
	assert (chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_BoneLightBinding at pos %d)",ch_ent.FileOffset);

	BONELIGHTBINDING_CHUNK_DESC_0001 Desc;
	Desc.numBindings = m_arrLightBinding.size();
	int res = fwrite (&Desc, sizeof(Desc), 1, f);
	if (res != 1) return true;

	res = fwrite(&m_arrLightBinding[0], sizeof(m_arrLightBinding[0]), m_arrLightBinding.size(), f);
	if (res != m_arrLightBinding.size())
		return true;
	return false;
}

bool CSExportUtility::SaveNode (FILE* f, INode* node )
{
	if(!node) return true;
	char nodeName[1024];
	strcpy( nodeName,node->GetName() );

	TimeValue time=GetTime();

	{
		Object *obj = node->GetObjectRef();
		if (obj)
		{
			//obj = obj->Eval(time).obj;
			if (obj->SuperClassID()==SYSTEM_CLASS_ID && obj->ClassID()==Class_ID(XREFOBJ_CLASS_ID,0))
			{
				// Copy external xref object name to helpers node name.
				IXRefObject *ix = (IXRefObject *)obj;
				strcat( nodeName,"/" );
				strcat( nodeName,ix->GetObjName() );
			}
		}
	}

	// PETAR: Pivot matrix ignored ??
	Matrix3 pivot;
	Point3 pivotpos = node->GetObjOffsetPos();
	Quat pivotrot = node->GetObjOffsetRot();
	ScaleValue pivotscale = node->GetObjOffsetScale();

	pivot.IdentityMatrix();
	pivot.PreTranslate(pivotpos);
	PreRotateMatrix(pivot,pivotrot);
	ApplyScaling(pivot,pivotscale);

	// END PETAR ----------------------

	// determine whether we export the parent of this node; if we don't, then export it in the world
	// coordinate frame (if the corresponding fixup option is set)
	int nParentChunkId = ChunkList.GetID(node->GetParentNode(),ChunkType_Node);

	Matrix3 rtm; // relative to parent
	/*
	//Get Transform Matrix44
	Matrix3 atm=node->GetNodeTM(time);  // world TM
	atm=pivot*atm; // PETAR - pivot ignored
	Matrix3 ptm=node->GetParentNode()->GetNodeTM(time); // parent world tm

	rtm = atm*Inverse(ptm);
	*/

		// Node Local TM.
	rtm = node->GetNodeTM(time) * Inverse(node->GetParentTM(time));

	if (pb_nodes->GetInt (PB_INDIVIDUAL_FILES) && nParentChunkId == -1)
	{
		// the top of the hierarchy is always exported non-displaced
		rtm.SetRow(3,Point3(0,0,0));
	}
/*
	if (pb_nodes->GetInt (PB_ROTATE_TO_WORLD_CS))
	{
		if (nParentChunkId == -1)
			// rotate the matrix so that it's the same as in world coordinates
			for (int i = 0; i < 3; ++i)
				rtm.SetRow(i, atm.GetRow(i));
	}
*/
	Point3	decomposed_p;
	Quat	decomposed_r;
	Point3	decomposed_s;
	DecomposeMatrix(rtm, decomposed_p, decomposed_r, decomposed_s);
	//MRow *rrow =atm.GetAddr();
	//AffineParts rap;
	//decomp_affine(rtm,&rap);

	//CHECK CONTROLLERS
	int pos_cont_id=-1, rot_cont_id=-1, scl_cont_id=-1;
	Control *c=node->GetTMController();
	if(c && c->ClassID() == Class_ID(PRS_CONTROL_CLASS_ID,0))
	{
		pos_cont_id = ChunkList.GetID(c->GetPositionController(),ChunkType_Controller);
		rot_cont_id = ChunkList.GetID(c->GetRotationController(),ChunkType_Controller);
		scl_cont_id = ChunkList.GetID(c->GetScaleController()   ,ChunkType_Controller);
	}

	//==================
	//Write chunk header
	//==================
	fpos_t fpos;
	fgetpos(f,&fpos);
	//append to chunk list
	int chunk_id=ChunkList.GetID(node,ChunkType_Node);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	DebugPrint("\n(ChunkType_Node at pos %d for \"%s\")",ch_ent.FileOffset, node->GetName());


	UpdateProgress(100,"\"%s\" Writing node chunk", nodeName);

	//Get # of children that will be saved
	IntTab children;
	for(int i=0;i<node->NumberOfChildren();i++)
	{
		INode *child_node=node->GetChildNode(i);
		int childchunkID=ChunkList.GetID(child_node,ChunkType_Node);
		if(childchunkID !=-1) children.Append(1,&childchunkID);
	}

	TSTR PropStr; 
	node->GetUserPropBuffer(PropStr);

	NODE_CHUNK_DESC ncd;
	ncd.chdr				= ch_ent;
	strncpy(ncd.name,nodeName,64);
	ncd.MatID				= ChunkList.GetID(node->GetMtl(), ChunkType_Mtl);
	ncd.nChildren			= children.Count();
	ncd.ObjectID			= ChunkList.GetID_Xept_type(node, ChunkType_Node);
	ncd.ParentID			= nParentChunkId;

	ncd.IsGroupHead			= node->IsGroupHead()?true:false;
	ncd.IsGroupMember		= node->IsGroupMember()?true:false;

	Matrix3_to_CryMatrix(ncd.tm, rtm);
	ncd.pos_cont_id			= pos_cont_id;
	ncd.rot_cont_id			= rot_cont_id;
	ncd.scl_cont_id			= scl_cont_id;
	ncd.pos					= Vec3(decomposed_p.x, decomposed_p.y, decomposed_p.z);
	ncd.rot					= CryQuat(decomposed_r.w,decomposed_r.x, decomposed_r.y, decomposed_r.z);
	ncd.scl					= Vec3(decomposed_s.x, decomposed_s.y, decomposed_s.z);
	ncd.PropStrLen			= PropStr.Length();

	if (fabs((ncd.rot|ncd.rot)-1) > 0.01)
		ncd.rot = CryQuat (Matrix33(ncd.tm));

	int res=fwrite(&ncd,sizeof(ncd),1,f);
	if(res!=1) return true;
	
	//================
	//Write PropString
	//================
	UpdateProgress(100,"\"%s\" Writing properties", nodeName);
	if(ncd.PropStrLen)
	{
		int res=fwrite(PropStr.data(),PropStr.Length(),1,f);
		if(res!=1) return true;
		int nPad = 0;
		// align
		if (PropStr.Length() & 3)
		{
			res = fwrite (&nPad, 4 - (PropStr.Length() & 3),1,f);
			if (res != 1) return true;
		}
	}

	//======================
	//Write Child Chunk ID's
	//======================
	UpdateProgress(100,"\"%s\" Writing children", nodeName);
	if(ncd.nChildren)
	{
		for(int i=0;i<ncd.nChildren;i++)
		{
			int childchunkid=children[i];
			int res=fwrite(&childchunkid,sizeof(childchunkid),1,f);
			if(res!=1) return true;					
		}
	}

	return false;
};

// GetBoneNodeKeyTimes ______________________________________________________________________________________
bool CSExportUtility::GetBoneNodeKeyTimes(INode *node, IntTab &keytimes, INodeTab *children)
{
	Interval ivalid;

	int startframe	= pb_timing->GetInt(PB_RANGE_START);
	int endframe	= pb_timing->GetInt(PB_RANGE_END);
	int step		= pb_bones->GetInt(PB_BONE_ANIM_RATE);
	int atkeys		= pb_bones->GetInt(PB_BONE_ANIM_ATKEYS);
	int atfixed		= pb_bones->GetInt(PB_BONE_ANIM_EVERY);
	int manualrange = pb_timing->GetInt(PB_MAN_RANGE);


	//==================================================
	// BUG_WORKAROUND: for root resample every frame
	//==================================================
	/*
	if(atkeys && node->GetParentNode() && node->GetParentNode()->IsRootNode())
	{
		atfixed	= true;
		step	= 2;
	}
	*/
	//==================================================
	// END BUG_WORKAROUND: for root resample every frame
	//==================================================

	int starttime, endtime;
	if(manualrange)
	{
		starttime	= startframe * GetTicksPerFrame();
		endtime		= endframe * GetTicksPerFrame();
	}
	else
	{
		Interval ii=ip->GetAnimRange();
		starttime	=ii.Start();
		endtime		=ii.End();
		startframe  =starttime / GetTicksPerFrame();
		endframe    =endtime / GetTicksPerFrame();
	}

	//sample at the starttime & end time
	keytimes.Append(1,&starttime);
	keytimes.Append(1,&endtime);
	//sample in fixed rate
	if(atfixed)
	{
		int i=0;
		int n=max(1,(endframe-startframe)/step);
		for(int frame=startframe;frame<=endframe;frame+=step,i++) 
		{ 
			//UpdateProgress(100*i/n,"\"%s\" Calculating bone keys at fixed interval", node->GetName());
			int tt=frame*GetTicksPerFrame(); keytimes.Append(1,&tt); 
		}
	}

	//sample at keys
	if(atkeys)
	{
		Control *pC = node->GetTMController();
		if(pC)
		{
			
			if(pC->ClassID() == BIPBODY_CONTROL_CLASS_ID)
			{
				Animatable *vert=pC->SubAnim(0);
				Animatable *horz=pC->SubAnim(1);
				Animatable *rot =pC->SubAnim(2);
				
				TimeValue ttt=starttime;
				while(vert->GetNextKeyTime(ttt, NEXTKEY_RIGHT | NEXTKEY_POS|NEXTKEY_ROT|NEXTKEY_SCALE, ttt) && (ttt>starttime) && (ttt<endtime)) 	
					keytimes.Append(1,&ttt);
				
				ttt=starttime;
				while(horz->GetNextKeyTime(ttt, NEXTKEY_RIGHT | NEXTKEY_POS|NEXTKEY_ROT|NEXTKEY_SCALE, ttt) && (ttt>starttime) && (ttt<endtime)) 	
					keytimes.Append(1,&ttt);
				
				ttt=starttime;
				while(rot->GetNextKeyTime(ttt, NEXTKEY_RIGHT | NEXTKEY_POS|NEXTKEY_ROT|NEXTKEY_SCALE, ttt) && (ttt>starttime) && (ttt<endtime)) 	
					keytimes.Append(1,&ttt);

				bool cs_cont=true;
			}
			else if(pC->ClassID() == FOOTPRINT_CLASS_ID)
			{
				Animatable *vert=pC->SubAnim(0);
				Animatable *horz=pC->SubAnim(1);
				Animatable *rot =pC->SubAnim(2);
				bool cs_cont=true;
			}
			else if ((pC->ClassID() == BIPSLAVE_CONTROL_CLASS_ID))
			{
				Animatable *vert=pC->SubAnim(0);
				Animatable *horz=pC->SubAnim(1);
				Animatable *rot =pC->SubAnim(2);
				bool cs_cont=true;
			}

			IKeyControl *ikeys = GetKeyControlInterface(pC);
			int nNodekeys=node->NumKeys();

			Tab<TimeValue>	TP,TR,TS;

			Interval interval(starttime, endtime);
			pC->GetKeyTimes(TR, interval, KEYAT_ROTATION );
			pC->GetKeyTimes(TP, interval, KEYAT_POSITION );
			pC->GetKeyTimes(TS, interval, KEYAT_SCALE );
			int nkeys=TR.Count();
		
			for(int i=0;i<nkeys;i++)
			{
				//UpdateProgress(100*i/nkeys,"\"%s\" Calculating bone keys at keys", node->GetName());
				int kt=TR[i];
				keytimes.Append(1, &kt);
			}
		}
	}

	//sort and aliminate the duplicates
	CleanUpIntTab(keytimes);

	return false;
}

////////////////////////////////////////////////////////////////////////////////////////////
// CSExportUtility::GetBoneNodeKeys
//
// Given the list of key times, generate linear CryBoneKey's for all those times.
// The generated linear keys go to the keys table.
// The keys may be optimized, in which case the keys table will contain less keys than there are positions in the keytime table.
// The optimization is a simple linear reduction, no tricks.
//
// PARAEMTERS:
//   pNode         - the node for which to generate the keys
//   tabKeyTimes   - array of times at which to generate keys.
//   tabOutKeys [OUTPUT] - upon output contains an array of keys defining the animation
//              NOTE: the keys may be optimized
// ASSUMES:
//   tabKeyTimes must be ordered and must not contain any duplicates
////////////////////////////////////////////////////////////////////////////////////////////
void CSExportUtility::GetBoneNodeKeys(INode *pNode, const IntTab &tabKeyTimes, Tab<CryBoneKey> &tabOutKeys)
{
	tabOutKeys.SetCount(tabKeyTimes.Count());

	for(int i=0; i<tabKeyTimes.Count(); i++)
	{
		assert (!i || tabKeyTimes[i] > tabKeyTimes[i-1]);

		UpdateProgress(100*i/tabKeyTimes.Count(),"\"%s\" Preparing bone keys %d/%d", pNode->GetName(), i, tabKeyTimes.Count());

		// the time of the key
		TimeValue time = tabKeyTimes[i];

		Matrix3 atm = pNode->GetNodeTM(time);
		Matrix3 ptm = pNode->GetParentNode()->GetNodeTM(time);
		
		atm.NoScale();
		ptm.NoScale();
		Matrix3 rtm = atm*Inverse(ptm);

		CryBoneKey* pKey = tabOutKeys.Addr(i);

		pKey->time = time;

		// Decompose and inner access to abs and rel matrices
		AffineParts aap,rap;
		decomp_affine (atm,&aap);
		decomp_affine (rtm,&rap);

		pKey->abspos.x = aap.t.x;
		pKey->abspos.y = aap.t.y;
		pKey->abspos.z = aap.t.z;

		pKey->relpos.x = rap.t.x;
		pKey->relpos.y = rap.t.y;
		pKey->relpos.z = rap.t.z;

		pKey->relquat.v.x = rap.q.x;
		pKey->relquat.v.y = rap.q.y;
		pKey->relquat.v.z = rap.q.z;
		pKey->relquat.w   = rap.q.w;
	}

	if (pb_bones->GetInt (PB_KEY_CLEANUP_ENABLE))
	{
		// perform the key cleanup, since the user wants it, with the specified thresholds

		float fThreshRotation = pb_bones->GetFloat (PB_KEY_CLEANUP_ROTATION);
		float fThreshPosition = pb_bones->GetFloat (PB_KEY_CLEANUP_POSITION);

		// 1 digit    => 10^-1 = 0.1
		// 2 digits   => 10^-2 = 0.01
		// 3.5 digits => 10^-3.5 ~= 0.0031
		// we multiply by two because the error is the error of square length, not the length itself
		float fPosError = float (pow (10.0f, -2.0f*fThreshPosition));

		float fQuatError = float (1.0f - pow (10.0f, -fThreshRotation));

		unsigned nOptimizedKeys = OptimizeKeys (tabOutKeys.Addr(0), tabOutKeys.Count(), fPosError, fQuatError);
		tabOutKeys.SetCount(nOptimizedKeys);
	}
}

// GetValidChildrenList ______________________________________________________________________________________
bool CSExportUtility::GetValidChildrenList(INode *node, INodeTab &children, bool recurse)
{
	Interval ivalid;
	TimeValue time=GetTime();
	BOOL ignore_dummy= pb_bones->GetInt(PB_IGNORE_DUMMY);

	for(int i=0;i<node->NumChildren();i++)
	{
		INode *child	=node->GetChildNode(i);
		if(!child) continue;

		char* szName = child->GetName();
		OutputDebugString (szName);
		OutputDebugString("\n");

		//do not export footprint node
		if(IsFootPrint(child)) continue;
		
		//don't export deformed mesh as a bone
		if(FindPhysiqueModifier(child->GetObjectRef())) continue;

		// hmmm
		//0000000000000000000000000000000000000000000000000000000000000000
		if(FindSkinModifier(child->GetObjectRef())) continue;

		//Ignore dummy objects if the checkbox is set
		if(ignore_dummy)
		{
			ObjectState os		=child->EvalWorldState(time);
			Class_ID	cid		=os.obj->ClassID();
			if(cid == Class_ID(DUMMY_CLASS_ID,0))
			{
				/*
				if (!children.Count() && recurse)
				{
					char szBuf[0x200];
					_snprintf (szBuf, sizeof(szBuf), "The root object %s is a dummy. You ignore dummies. Skeleton can not be constructed. File cannot be exported.", szName);
					CryMessageBox(hwnd_nodes, szBuf, "Error", MB_OK);
					return true;
				}
				*/
				continue;
			}
		}
		//if(ignore_mesh && (cid != BipedClassID) && (cid != Class_ID(BONE_CLASS_ID,0)) ) continue;

		children.Append(1,&child);

		if(recurse)
			if (GetValidChildrenList(child, children, recurse))
				return true;
	}
	return false;
}

void AddKeyTimes(IntTab &a, IntTab &b)
{
	for(int i=0;i<b.Count();i++)
	{
		int val=b[i];
		a.Append(1,&val);
	}
}

float DotProd (const CryQuat& q, const CryQuat& p)
{
	return q.w*p.w + q.v*p.v;
}


// SaveBoneController ______________________________________________________________________________________
bool CSExportUtility::SaveBoneController (FILE *f, INode *node, IntTab *parentkeytimes)
{
	Interval ivalid;
	TimeValue time=GetTime();
	char *name = node->GetName();

	//---------------------
	// Determine the flags
	//---------------------
	//stabilize
	bool stabilize_leg=false;
	if(pb_bones->GetInt(PB_BONE_ANIM_ATKEYS) && pb_bones->GetInt(PB_STABILIZE_FEET) && IsLegBone(node)) stabilize_leg=true;
	//use parent times
	bool use_parent_time = stabilize_leg || (pb_bones->GetInt(PB_BONE_ANIM_ATKEYS) && pb_bones->GetInt(PB_PARENT_TIMES));
	//use child times
	bool use_child_time  = pb_bones->GetInt(PB_BONE_ANIM_ATKEYS) && pb_bones->GetInt(PB_CHILD_TIMES);

	//----------------
	//Get Own Key times
	//----------------
	IntTab mykeytimes;
	mykeytimes.ZeroCount();
	GetBoneNodeKeyTimes(node, mykeytimes);

	//-----------------------
	//Get all child key times
	//-----------------------
	//int use_child_time = pb_bones->GetInt(PB_BONE_ANIM_ATKEYS) && pb_bones->GetInt(PB_CHILD_TIMES);
	INodeTab allchildren;
	allchildren.ZeroCount();
	if (GetValidChildrenList(node, allchildren, true))
		return true;
	IntTab childkeytimes;
	childkeytimes.ZeroCount();
	for(int i=0;i<allchildren.Count();i++)
	{
		GetBoneNodeKeyTimes(allchildren[i], childkeytimes); 
	}

	//--------------------
	//Get parent key times
	//--------------------
	IntTab ownANDparentkeytimes;
	ownANDparentkeytimes.ZeroCount();
	//combine own and parent key times
	AddKeyTimes(ownANDparentkeytimes, mykeytimes);
	if(parentkeytimes) 
	{
		AddKeyTimes(ownANDparentkeytimes, *parentkeytimes);
	}
	CleanUpIntTab(ownANDparentkeytimes); //sort and aliminate the duplicates

	//-----------------------
	//Get leg child key times
	//-----------------------
	IntTab legchildkeytimes;
	legchildkeytimes.ZeroCount();
	if(IsBipedRoot(node))
	{
		INode *Lthigh=FindLThigh(node,true);
		INode *Rthigh=FindRThigh(node,true);

		INodeTab chtab;
		if(Lthigh) if (GetValidChildrenList(Lthigh, chtab, true))return true;
		if(Rthigh) if(GetValidChildrenList(Rthigh, chtab, true))return true;
		
		if(Lthigh && Lthigh->GetParentNode()) 
		{
			INode *tmp=Lthigh->GetParentNode();
			if(tmp)
			{
				chtab.Append(1,&tmp);			//append Spine
				tmp=tmp->GetParentNode();	
				if(tmp) chtab.Append(1,&tmp);	//append Pelvis
			}
		}
		
		int n=chtab.Count();
		for(int i=0;i<n;i++)
		{
			GetBoneNodeKeyTimes(chtab[i], legchildkeytimes); 
		}
	}

	//------------------
	// COMBINE THE TIMES
	//------------------
	IntTab combined_keytimes;
	combined_keytimes.ZeroCount();
	if(use_parent_time)						AddKeyTimes(combined_keytimes,ownANDparentkeytimes);
	else									AddKeyTimes(combined_keytimes,mykeytimes);
	if(use_child_time)						AddKeyTimes(combined_keytimes,childkeytimes);
	if(stabilize_leg)						AddKeyTimes(combined_keytimes,legchildkeytimes);
	CleanUpIntTab(combined_keytimes);				//sort and aliminate the duplicates

	//------------
	//Get the keys
	//------------
	Tab<CryBoneKey> keys;
	GetBoneNodeKeys (node, combined_keytimes, keys);

	//------------------------
	//Get 1st level child list
	//------------------------
	INodeTab children;
	children.ZeroCount();
	if (GetValidChildrenList(node, children, false))
		return true;

	//==================
	//Write chunk header
	//==================
	UpdateProgress(100,"\"%s\" Writing chunk header", name);
	fpos_t fpos;
	fgetpos(f,&fpos);
	int chunk_id=ChunkList.GetID(node,ChunkType_Controller);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	
	DebugPrint(" (ChunkType_Controller at pos %d)",ch_ent.FileOffset);
	CONTROLLER_CHUNK_DESC_0827 chunk;
	chunk.numKeys				= keys.Count();
	chunk.nControllerId = Crc32Gen::GetCRC32( name );
	int	res=fwrite(&chunk,sizeof(chunk),1,f);
	if(res!=1) return true;	
	/*
	//=======================
	//Write the bone entity
	//=======================
	UpdateProgress(100,"\"%s\" Writing bone ", node->GetName());
	TSTR PropStr; 
	node->GetUserPropBuffer(PropStr);
	BONE_ENTITY bone;
	bone.ParentID	= BoneList.GetID(node->GetParentNode());
	bone.BoneID		= BoneList.GetID(node);
	bone.nChildren	= children.Count();
	bone.nKeys		= keys.Count();
	strncpy(bone.prop,PropStr,32);
	int res=fwrite(&bone,sizeof(bone),1,f);
	if(res!=1) return true;
	*/

	//==============
	//Write the keys
	//==============
	CryQuat qLast;
	qLast.SetIdentity();
	CryKeyPQLog keyLast;
	keyLast.nTime = 0;
	keyLast.vPos = Vec3d(0,0,0);
	keyLast.vRotLog = Vec3d(0,0,0);
	for(i=0;i<keys.Count();i++)
	{
		//DebugPrint(" [%d]", keys[i].time/GetTicksPerFrame());
		if (DotProd(qLast,keys[i].relquat) >= 0)
			qLast =  keys[i].relquat;
		else
			qLast = -keys[i].relquat;

		CryKeyPQLog key;
		key.nTime = keys[i].time;
		key.vPos = keys[i].relpos;

		key.vRotLog = log (qLast).v;
		AdjustRotLog (key.vRotLog, keyLast.vRotLog);

		keyLast = key;

		int res=fwrite(&key,sizeof(key),1,f);
		if(res!=1) return true;
	}
	
	//-------
	//recurse
	//-------
	for(i=0;i<children.Count();i++)
	{
		int res;
		if(stabilize_leg && IsLegBone(children[i]))
			res=SaveBoneController(f,children[i],&combined_keytimes);
		else
			res=SaveBoneController(f,children[i],&ownANDparentkeytimes);
		
		if(res) return res?true:false;
	}
	return false;
}

bool CSExportUtility::SaveBoneHierarchy(FILE *f, INodeTab &allnodes)
{
	if(!allnodes.Count()) return false;

	UpdateProgress(100,"Writing bone anim header");

	//===========================
	//Write BoneAnim chunk header
	//===========================
	fpos_t fpos;
	fgetpos(f,&fpos);
	int chunk_id=ChunkList.GetID(allnodes[0],ChunkType_BoneAnim);
	assert(chunk_id != -1);
	CHUNK_HEADER &ch_ent = *(ChunkList.ch_list.Addr(chunk_id));
	ch_ent.FileOffset	 = int(fpos);
	
	BONEANIM_CHUNK_DESC chunk;
	chunk.chdr				= ch_ent;
	chunk.nBones			= allnodes.Count();
	int res=fwrite(&chunk,sizeof(chunk),1,f);
	if(res!=1) return true;

	int n=allnodes.Count(),i;
	for(i=0;i<n;i++) {
		char physname[64]; sprintf(physname,"%s Phys",allnodes[i]->GetName());
		INode *node = GetCOREInterface()->GetINodeByName(physname);
		if (node)
			break; // at least one bone has a " Phys" counterpart 
	}
	bool bAlwaysExportMesh = i==n; // if " Phys" conterparts are not used, assume that all bones have geometry

	std::set<unsigned int> controllerIds;

	//===============
	//Write The Bones
	//===============
	for(i=0;i<n;i++)
	{
		INode *node=allnodes[i];
		if(!node) continue;

		INodeTab children;
		if (GetValidChildrenList(node, children, false))
			return true;

		BONE_ENTITY bone;
		bone.BoneID			= BoneList.GetID(node);
    if (node->GetParentNode() != ip->GetRootNode() && !node->IsRootNode())
		  bone.ParentID = BoneList.GetID(node->GetParentNode(), false);
    else
      bone.ParentID = -1;

		//bone.ControllerID	= ChunkList.GetID(node,	ChunkType_Controller);
		int ctrlId = Crc32Gen::GetCRC32( node->GetName() );
		
		if (controllerIds.find(ctrlId) != controllerIds.end())
		{
			// Same Controller ID generated for different bones.
			MessageBox( NULL,"Same Controller ID generated for different bones\r\nChange name of colliding bones.","Warning!!!",MB_OK );
		}
		controllerIds.insert(ctrlId);
		bone.ControllerID	= ctrlId;
		bone.nChildren		= children.Count();
		bone.prop[0] = 0;

		bool bExportMesh = false;
		char physname[64]; sprintf(physname,"%s Phys",node->GetName());
		INode *physnode; Control *c; JointParams *jp;
		//changed by [MG]
		physnode=GetCOREInterface()->GetINodeByName(physname);
		if ( physnode == NULL)
			bone.phys.flags = -1;
		else
		if (((c=physnode->GetTMController()) && (c=c->GetRotationController()) ||
				(c=node->GetTMController()) && (c=c->GetRotationController())) &&
				(jp=(JointParams*)c->GetProperty(PROPID_JOINTPARAMS)))
		{
			//changed by [MG]
			if ( !physnode )
			{
				char buf[128];
				sprintf(buf,"Can not find %s node",physname);
				CryMessageBox(hwnd_nodes, buf , "Error", MB_OK);
				return false;
			}

			bone.phys.flags = (~jp->flags & JNT_XACTIVE*7);
			int j,k,l;
			for(j=0;j<jp->dofs;j++) 
			{
				if (jp->flags & JNT_XLIMITED<<j) {
					bone.phys.min[j]=jp->min[j]; bone.phys.max[j]=jp->max[j];
				} else {
					bone.phys.min[j]=-1E10; bone.phys.max[j]=1E10;
				}
				bone.phys.spring_angle[j] = jp->spring[j];
				bone.phys.spring_tension[j] = jp->stens[j]*50.0f;
				bone.phys.damping[j] = jp->damping[j];
			}
			INode *framenode,*frameparent;
			sprintf(physname,"%s Phys ParentFrame",node->GetName());
			if ((framenode=GetCOREInterface()->GetINodeByName(physname)) && (frameparent=framenode->GetParentNode())) {
				Matrix3 frametm=framenode->GetNodeTM(0), parenttm=frameparent->GetNodeTM(0);
				for(j=0;j<3;j++) for(k=0;k<3;k++) for(l=0,bone.phys.framemtx[j][k]=0;l<3;l++)
					bone.phys.framemtx[j][k] += parenttm.GetRow(j)[l]*frametm.GetRow(k)[l];
			} else
				bone.phys.framemtx[0][0] = 100;

			TSTR PropStr;
			physnode->GetUserPropBuffer(PropStr);
			strncpy(bone.prop,PropStr,32);
			bone.prop[31] = 0;
			bExportMesh = true;
		} 
		TSTR PropStr; 
		node->GetUserPropBuffer(PropStr);
		if (bAlwaysExportMesh && !bExportMesh) {
			if (!strstr(PropStr,"nonphysical"))
				bExportMesh = true;
		}

		if (bExportMesh) {
			CHUNK_HEADER new_ch_ent;
			new_ch_ent.ChunkType = ChunkType_BoneMesh;
			new_ch_ent.ChunkVersion	= MESH_CHUNK_DESC_VERSION;
			new_ch_ent.FileOffset = -1;
			ChunkList.Append(node,&new_ch_ent,true);
			bone.phys.nPhysGeom = ChunkList.GetID(node, ChunkType_BoneMesh);
		} else
			bone.phys.nPhysGeom = -1;

		if (bone.prop[0]==0) {
			strncpy(bone.prop,PropStr,32);
			bone.prop[sizeof(bone.prop)-1] = 0;
		}
		int res=fwrite(&bone,sizeof(bone)/*-sizeof(BONE_PHYSICS)*/,1,f);
		if(res!=1) return true;
	}

	fflush(f);

	return false;
}

// SaveBoneAnimFile ______________________________________________________________________________________
int CSExportUtility::SaveBoneAnimFile()
{
/*
	#ifdef PROTECTION_ENABLED
		#pragma message ("===========Protection in SaveBoneAnimFile Enabled ===========")
		if(!CheckIntegrity())
		{
			ip->RegisterCommandModeChangedCallback((CommandModeChangedCallback *)0x13051972);
		}
	#endif
*/
	#define HALT { fclose(f); DeleteFile(filename); return IDNO; }
	Interval ivalid;
	TimeValue time=GetTime();
	//===================================
	//check if there are bones to export
	//===================================
	int nobjects=pb_bones->Count(PB_BONES);
	//discount "deleted" and "invalid" nodes
	for(int i=nobjects-1;i>=0;i--)	if(!pb_bones->GetINode(PB_BONES,time,i)) nobjects--;
	if(!nobjects)
	{
		CryMessageBox(hwnd_nodes,"Bone list is Empty\nPlease Add one or more bone to the list which will be exported","No bones to Export",MB_OK | MB_ICONEXCLAMATION);
		return IDCANCEL;
	}
	FILE *f = 0;

	for(int k=0;k<nobjects;k++)
	{
		INode *rootbone=pb_bones->GetINode(PB_BONES,0,k);
		if(!rootbone) continue;
	
		//=================
		//create anim file
		//=================
		char def_fname[512];
		char *max_name=GetCOREInterface()->GetCurFileName();
		if(strlen(max_name)>=4)
		{
			strcpy(def_fname,max_name);
			def_fname[strlen(def_fname)-4]='_';
			def_fname[strlen(def_fname)-3]=0;
			strcat(def_fname,rootbone->GetName());
		}
		else strcpy(def_fname,rootbone->GetName());

		int	 res=0;
		char filename[1024];
		//char filenameOrg[1024];
		char str[512];
		sprintf(str,"Save skeleton starting at bone \"%s\" As...",rootbone->GetName());
		int bSave = SaveFileDlg(FileType_Anim,filename, str, def_fname);
		if(!bSave) return IDCANCEL;

    //char tempPath[1024];
		//strcpy( filenameOrg,filename );
    //GetTempPath( sizeof(tempPath),tempPath );
    //strcpy( filename,tempPath );
    //strcat( filename,"CryExport.caf" );
		f = fopen( filename,"wb" );
		if (!f) return IDNO; // error: cannot open the file
		
		ChunkList.Reset();

		//==================
		//Write File Header
		//==================
		FILE_HEADER header;
		header.FileType				= FileType_Anim;
		header.ChunkTableOffset		= -1;
		header.Version				= AnimFileVersion;

		strcpy(header.Signature,FILE_SIGNATURE);
		res=fwrite(&header,sizeof(header),1,f);
		res=res!=1;
		if(res) HALT;

		CHUNK_HEADER ch_ent;
		ChunkList.Reset();
		BoneList.Reset();
		ch_ent.FileOffset		= -1;
		//=======================
		//Prepare SceneProperties
		//=======================
		int numProps = ip->GetNumProperties(PROPSET_USERDEFINED);
		if(numProps)
		{
			ch_ent.ChunkType		= ChunkType_SceneProps;
			ch_ent.ChunkVersion		= SCENEPROPS_CHUNK_DESC_VERSION;
			ChunkList.Append(NULL,&ch_ent,true);
		}
		//=======================
		//Prepare Timing
		//=======================
		ch_ent.ChunkType		= ChunkType_Timing;
		ch_ent.ChunkVersion		= TIMING_CHUNK_DESC_VERSION;
		ChunkList.Append(NULL,&ch_ent,true);

		//---------------------------------------------------
		//Get all the bones and prepare the controller chunks
		//---------------------------------------------------
		INodeTab allnodes;
		allnodes.Append(1,&rootbone);
		if (GetValidChildrenList(rootbone, allnodes, true))
			return true;
		
		//add controller chunks
		for(int i=0;i<allnodes.Count();i++)
		{
			INode *node = allnodes[i];

			//============================
			//Prepare Controller chunk(s)
			//============================
			ch_ent.ChunkType		= ChunkType_Controller;
			ch_ent.ChunkVersion		= CONTROLLER_CHUNK_DESC_0827::VERSION;
			ChunkList.Append(node,&ch_ent,true);

			//also fill the Bonelist
			BoneList.GetID(node);
		}
		
		/*
		//============================
		//Prepare bone hierarchy chunk
		//============================
		ch_ent.ChunkType		= ChunkType_BoneAnim;
		ch_ent.ChunkVersion		= BONEANIM_CHUNK_DESC_VERSION;
		ChunkList.Append(rootbone,&ch_ent,true);
		*/
				
		//===========================
		//prepare bone namelist chunk
		//===========================
		ch_ent.ChunkType		= ChunkType_BoneNameList;
		ch_ent.ChunkVersion		= BONENAMELIST_CHUNK_DESC_0745::VERSION;
		ChunkList.Append(NULL,&ch_ent,true);

		//---------------------------------------------------
		//Every chunk is prepared.. Now write them
		//---------------------------------------------------
		//==================================================
		//Treverse the Bone Tree and save Controller Chunks
		//==================================================
		res=SaveBoneController(f, rootbone);
		if(res) HALT;

		//==================
		//Save other chunks
		//==================
		for(i=0;i<ChunkList.ch_list.Count();i++)
		{
			int	ChunkType = ChunkList.ch_list[i].ChunkType;
			ReferenceTarget* ChunkPtr = ChunkList.ptr_list[i];
			int res=0;

			switch(ChunkType)
			{
			case ChunkType_Controller:
				//skip: we expoerted hierarchically with SaveBoneController
				break;

			case ChunkType_BoneAnim:
				//res=SaveBoneHierarchy(f, allnodes);
				break;

			case ChunkType_BoneNameList:
				res=SaveBoneNameList(f);
				break;

			case ChunkType_Timing:
				res=SaveTiming(f);
				break;

			case ChunkType_SceneProps:
				res=SaveSceneProps(f);
				break;

			case ChunkType_BoneMesh:
				//res=SaveMeshObject(f,(INode*)ChunkPtr,true,false);
				break;

			default:
					CryMessageBox(hwnd_nodes, "unsuported chunk type encountered", "Internal Error", MB_OK | MB_ICONINFORMATION);
			}

			if(res) HALT;
		}
		
		//=======================
		//Write Chunk List
		//=======================
		res=WriteChunkList(f, header);
		if(res) HALT;

		//==========================================
		//CLOSE File and Call DLL Callback function
		//==========================================
		fclose(f);
		CallExtensionDLLs(filename, DLL_FLAG_ANIM);

		/*
		// Check if original filenam exist.
		FILE *pFile = fopen(filenameOrg,"rb");
		if (!pFile)
		{
			MoveFile( filename,filenameOrg );
			ChunkFile cf;
			cf.Read( filenameOrg );
			cf.SaveCalFile( filenameOrg );
			return false;
		}
		fclose(pFile);

		if (animationName.Length() == 0)
		{
			MessageBox(hwnd_nodes, "Fill Animation Name, Animation name is Empty!", "Warning", MB_OK | MB_ICONINFORMATION);
			return false;
		}

		ChunkFile cf;
		cf.MergeAnim( filenameOrg,filename,animationName, bExportNew );
		cf.SaveCalFile( filenameOrg );
		*/
		
		/*
		ChunkFile cf;
		cf.Read( filename );

		char filename2[_MAX_PATH];
		char drive[_MAX_PATH];
		char path[_MAX_PATH];
		char fname[_MAX_PATH];
		char fext[_MAX_PATH];
		_splitpath( filename,drive,path,fname,fext );
		_makepath( filename2,drive,path,fname,".ca2" );
		
		cf.Write( filename2 );
		*/
	}

#undef HALT	
	return IDYES;
}


void CSExportUtility::VariantToString(const PROPVARIANT* pProp, TCHAR* szString, int bufSize)
{
	switch (pProp->vt) 
	{
		case VT_LPWSTR:
			_tcscpy(szString, TSTR(pProp->pwszVal));
			break;
		case VT_LPSTR:
			_tcscpy(szString, TSTR(pProp->pszVal));
			break;
		case VT_I4:
			_stprintf(szString, "%ld", pProp->lVal);
			break;
		case VT_R4:
			_stprintf(szString, "%f", pProp->fltVal);
			break;
		case VT_R8:
			_stprintf(szString, "%lf", pProp->dblVal);
			break;
		case VT_BOOL:
			_stprintf(szString, "%d", pProp->boolVal ? 1  : 0);
			break;
		case VT_FILETIME:
			SYSTEMTIME sysTime;
			FileTimeToSystemTime(&pProp->filetime, &sysTime);
			GetDateFormat(LOCALE_SYSTEM_DEFAULT, DATE_SHORTDATE, &sysTime, NULL, szString, bufSize);
			break;
		default:
			_tcscpy(szString, "");	
			break;
	}
}

// generates a default material chunk if needed, and returns its index (in the numbering of standard materials)
int CSExportUtility::getDefaultMtlIndex ()
{
	if (m_nDefaultMtlIndex != -1)
		return m_nDefaultMtlIndex;

  // add the default material
	CHUNK_HEADER ch;
	ch.FileOffset = -1;
	ch.ChunkType  = ChunkType_Mtl;
	ch.ChunkVersion = MTL_CHUNK_DESC_EXPORTER::VERSION;
	ChunkList.Append(NULL, &ch, true);
	m_nDefaultMtlChunkId = ch.ChunkID;

	// find it in the chunk list and count all standard materials before it;
	m_nDefaultMtlIndex = -1;
	for (int ctr=0;ctr<ChunkList.ptr_list.Count();ctr++)
		if (ChunkList.ch_list[ctr].ChunkType == ChunkType_Mtl)
		{
			Mtl *current = (Mtl *) ChunkList.ptr_list[ctr];
			if (!current || !current->IsMultiMtl())  m_nDefaultMtlIndex++;

			if (ChunkList.ch_list[ctr].ChunkID == m_nDefaultMtlChunkId) 
				break;
		}

	return m_nDefaultMtlIndex;
}

void CSExportUtility::RecursePrepareMtlChunks(Mtl* mat)
{
	if(!mat) return;

	CHUNK_HEADER ch_ent;
	ch_ent.FileOffset		= -1;
	ch_ent.ChunkType		= ChunkType_Mtl;
	ch_ent.ChunkVersion		= MTL_CHUNK_DESC_EXPORTER::VERSION;
	ChunkList.Append(mat, &ch_ent,true);
	
	if(mat->ClassID() == Class_ID(DMTL_CLASS_ID, 0) || mat->ClassID() == Class_ID(DMTL2_CLASS_ID, 0) ) 
	{
		StdMat* std = (StdMat *)mat;

		//get maps
		Texmap *tex[3];
		tex[0] = std->GetSubTexmap(ID_DI);
		tex[1] = std->GetSubTexmap(ID_OP);
		tex[2] = std->GetSubTexmap(ID_BU);
		
		ch_ent.ChunkType		= ChunkType_Controller;
		ch_ent.ChunkVersion		= CONTROLLER_CHUNK_DESC_0826::VERSION;
		
		for(int i=0;i<3;i++)
		{
			StdUVGen *stduv = GetTextureStdUVGen(tex[i]);
			if(!stduv) continue;
			IParamBlock *pb=(IParamBlock *)stduv->GetReference(0);
			if(!pb) continue;

			Control *cont=NULL;
			if(cont=pb->GetController(PB_UOFFS))	ChunkList.Append(cont, &ch_ent,true);
			if(cont=pb->GetController(PB_USCL))		ChunkList.Append(cont, &ch_ent,true);
			if(cont=pb->GetController(PB_UANGLE))	ChunkList.Append(cont, &ch_ent,true);
			if(cont=pb->GetController(PB_VOFFS))	ChunkList.Append(cont, &ch_ent,true);
			if(cont=pb->GetController(PB_VSCL))		ChunkList.Append(cont, &ch_ent,true);
			if(cont=pb->GetController(PB_VANGLE))	ChunkList.Append(cont, &ch_ent,true);
			if(cont=pb->GetController(PB_WANGLE))	ChunkList.Append(cont, &ch_ent,true);
						
		}
	}
	else if(mat->ClassID() == Class_ID(MULTI_CLASS_ID, 0) || mat->ClassID() == Class_ID(DOUBLESIDED_CLASS_ID, 0))
	{
		int n_sub=mat->NumSubMtls();
		for(int i=0;i<n_sub;i++)
		{
			Mtl *submtl=mat->GetSubMtl(i);
			RecursePrepareMtlChunks(submtl);
		}
	}
}


// SaveGeomFile ______________________________________________________________________________________
int CSExportUtility::SaveGeomFile( TSTR fileName )
{
#define HALT(saved) { fclose(f); DeleteFile(filename); return saved; }

	Interval ivalid;
	TimeValue time=GetTime();
	//lastSavedFile = "";

	//===================================
	//check if there are nodes to export
	//===================================
	int nobjects=pb_nodes->Count(PB_NODES);
	//discount "deleted" and "invalid" nodes
	for(int i=nobjects-1;i>=0;i--)	if(!pb_nodes->GetINode(PB_NODES,time,i)) nobjects--;

	if(!nobjects)
	{
		MessageBox(hwnd_nodes,"Node list is Empty\nPlease Add one or more nodes to the list which will be exported","No nodes to Export",MB_OK | MB_ICONEXCLAMATION);
		return IDCANCEL;
	}

	bBuilding = pb_nodes->GetInt(PB_BUILDING) != 0;
	bool bExportGeom = pb_nodes->GetInt(PB_EXPORT_GEOM) != 0;

	bool bIndividualFiles = pb_nodes->GetInt (PB_INDIVIDUAL_FILES) != 0;

	//=================
	//create cgf file
	//=================
	FILE *f = 0;
	int	 res=0;
	char filename[1024];

	if (fileName.Length() == 0)
	{
		char def_fname[512];
		strcpy (def_fname,GetCOREInterface()->GetCurFileName());
		
		if (strlen(def_fname)>=4)
			def_fname[strlen(def_fname)-4]=0;
		else
			def_fname[0] = 0;
		
		int bSave = bIndividualFiles?
			SaveDirDlg(FileType_Geom,filename,"Save the OBJECTs in ...",def_fname)
			:
			SaveFileDlg(FileType_Geom,filename,"Save the OBJECTS as ...",def_fname);;
		if (!bSave)
			filename[0] = 0;
	}
	else
	{
		strcpy( filename,fileName.data() );
	}

	lastSavedFile = "";

	if(!filename[0]) 
	{
		return IDNO;
	}

	int nFileNameLength = strlen(filename);

	if (!bIndividualFiles)
	{
		char fdrive[_MAX_DRIVE];
		char fdir[_MAX_DIR];
		char fname[_MAX_FNAME];
		char fext[_MAX_EXT];
		_splitpath( filename,fdrive,fdir,fname,fext );
		if (stricmp(fext,"anm")==0 || stricmp(fext,".anm")==0)
		{
			// No geometry export.
			bExportGeom = false;
		}
	}

	for (int nFile = 0; nFile < (bIndividualFiles? pb_nodes->Count(PB_NODES): 1); ++nFile)
	{
		if (bIndividualFiles)
		{
			INode* pNode = pb_nodes->GetINode (PB_NODES,time,nFile);
			if (!pNode)
				continue;
			if (filename[nFileNameLength-1] != '/' && filename[nFileNameLength-1] != '\\')
			{
				filename[nFileNameLength++] = '\\';
				filename[nFileNameLength] = '\0';
			}
			strcpy (filename + nFileNameLength, pNode->GetName());
			strcat (filename, bBuilding?".bld":".cgf");
			CreateDirectoryForFile (filename);
			ResetReadOnlyFlag(filename);
		}
		
		f = fopen (filename, "wb");
		if (!f)
		{
			if (bIndividualFiles)
			{
				strcat (filename, ":\n Cannot Export this file. Continue?");
				if (MessageBox(hwnd_nodes,filename,"Warning", MB_OKCANCEL | MB_ICONEXCLAMATION) == IDCANCEL)
					return IDCANCEL;
			}
			else
				return IDNO;
		}

	//==================
	//Write File Header
	//==================
	FILE_HEADER header;
	header.FileType				= FileType_Geom;
	header.ChunkTableOffset		= -1;
	header.Version				= GeomFileVersion;
	strcpy(header.Signature,FILE_SIGNATURE);
	res=fwrite(&header,sizeof(header),1,f);
	res=res!=1;
	if(res) HALT(IDNO);

	//=======================
	//pre-prepare all chunks
	//=======================
	BoneList.Reset();
	ChunkList.Reset();
	CHUNK_HEADER ch_ent;
	ch_ent.FileOffset		= -1;
	//=======================
	//Prepare the info chunk
	//=======================
	ch_ent.ChunkType = ChunkType_SourceInfo;
	ch_ent.ChunkID = 0;
	ch_ent.ChunkVersion = 0;
	ChunkList.Append(NULL, &ch_ent, true);
	//=======================
	//Prepare SceneProperties
	//=======================
	int numProps = ip->GetNumProperties(PROPSET_USERDEFINED);
	if(numProps)
	{
		ch_ent.ChunkType		= ChunkType_SceneProps;
		ch_ent.ChunkVersion		= SCENEPROPS_CHUNK_DESC_VERSION;
		ChunkList.Append(NULL,&ch_ent,true);
	}
	//=======================
	//Prepare Timing
	//=======================
	ch_ent.ChunkType		= ChunkType_Timing;
	ch_ent.ChunkVersion		= TIMING_CHUNK_DESC_VERSION;
	ChunkList.Append(NULL,&ch_ent,true);

	for(i=0;i<nobjects;i++)
	{
		INode *node = pb_nodes->GetINode(PB_NODES,time, i);
		if(!node) continue;
		if (bIndividualFiles && i != nFile)
			continue;

		//Check for the possible problems that this node may have
		int res=CheckNodeProblems(node);
		if(res==IDCANCEL) HALT(IDCANCEL);
		if(res==IDNO) HALT(IDNO);
		if(res==ID_SKIP_NODE) continue;
		
		//=======================
		//Prepare nodes
		//=======================
		ch_ent.ChunkType		= ChunkType_Node;
		ch_ent.ChunkVersion		= NODE_CHUNK_DESC_VERSION;
		ChunkList.Append(node,&ch_ent,true);
		/*
		Modifier* pPhysiqueModifier = FindPhysiqueModifier (node->GetObjectRef());
		if (pPhysiqueModifier)
		{
			IPhysiqueExport* pPhysique = (IPhysiqueExport*)pPhysiqueModifier->GetInterface(I_PHYINTERFACE);
			pPhysique->SetInitialPose (true);
		}
		*/

		//=======================
		//Prepare Controllers
		//=======================
		Control *c=node->GetTMController();
		if(c && c->ClassID() == Class_ID(PRS_CONTROL_CLASS_ID,0))
		{
			Control *pc=NULL, *rc=NULL, *sc=NULL;
			IKeyControl *pk=GetKeyControlInterface((pc=c->GetPositionController()));
			IKeyControl *rk=GetKeyControlInterface((rc=c->GetRotationController()));
			IKeyControl *sk=GetKeyControlInterface((sc=c->GetScaleController()));

			ch_ent.ChunkType		= ChunkType_Controller;
			ch_ent.ChunkVersion		= CONTROLLER_CHUNK_DESC_0826::VERSION;
			if(pk && pk->GetNumKeys()) ChunkList.Append(pc,&ch_ent,true);
			if(rk && rk->GetNumKeys()) ChunkList.Append(rc,&ch_ent,true);
			if(sk && sk->GetNumKeys()) ChunkList.Append(sc,&ch_ent,true);
		}

		if (!bExportGeom)
		{
			continue;
		}

		//=======================
		//Prepare Materials
		//=======================
		if(node->GetMtl())
		{
			RecursePrepareMtlChunks(node->GetMtl());
		};

		//=======================
		//Prepare objects
		//=======================
		Object *der_obj	= node->GetObjectRef();

		if (!der_obj)
			continue;
		
		// Check for XRef.
		if (der_obj->SuperClassID()==SYSTEM_CLASS_ID && der_obj->ClassID()==Class_ID(XREFOBJ_CLASS_ID,0))
		{
			// This node is XRef Object.
			ch_ent.ChunkType	= ChunkType_Helper;
			ch_ent.ChunkVersion	= HELPER_CHUNK_DESC_VERSION;
		}
		else
		{
			// Not XRef object.
			Object *obj		= der_obj->Eval(time).obj;
			SClass_ID sid	= obj->SuperClassID();
			Class_ID  cid	= obj->ClassID();
			switch(sid)
			{
			case GEOMOBJECT_CLASS_ID:
			/*if(obj->IsSubClassOf(patchObjectClassID) || cid==Class_ID(PATCHOBJ_CLASS_ID,0) || cid==Class_ID(PATCHGRID_CLASS_ID,0) || cid==Class_ID(0x1fff,0))	//tri-patch
			res=SavePatchNode(f,node);
				else*/
				{
					ch_ent.ChunkType		= ChunkType_Mesh;
					ch_ent.ChunkVersion		= MESH_CHUNK_DESC_VERSION;
				}
				break;
				
			case LIGHT_CLASS_ID:
				ch_ent.ChunkType		= ChunkType_Light;
				ch_ent.ChunkVersion		= LIGHT_CHUNK_DESC_VERSION;
				break;
			
			case CAMERA_CLASS_ID:
			case HELPER_CLASS_ID:
				ch_ent.ChunkType		= ChunkType_Helper;
				ch_ent.ChunkVersion		= HELPER_CHUNK_DESC_VERSION;
				break;
				
			default:
				sprintf(s,"Object \"%s\" can not be exported since it is not of the supported types",node->GetName());
				CryMessageBox(hwnd_nodes, s, "unsuported object type", MB_OK | MB_ICONINFORMATION);
				continue;
			}
		}


		ch_ent.FileOffset		= -1;
		res = ChunkList.Append(node, &ch_ent,true);

		if(res!=-1)
		{
			//=======================
			//Prepare Vertex Anim
			//=======================
			if(ch_ent.ChunkType==ChunkType_Mesh && pb_nodes->GetInt(PB_WRITE_VA))
			{
				ch_ent.ChunkType		= ChunkType_VertAnim;
				ch_ent.ChunkVersion		= VERTANIM_CHUNK_DESC_VERSION;
				ChunkList.Append(node, &ch_ent,true);
			}
		}
		else	//this object has alredy been saved in another chunk
		{
			//check if multiple nodes use the instanced-physique-assigned objects
			if(pb_nodes->GetInt(PB_WRITE_WEIGHTS) && FindPhysiqueModifier(der_obj))
			{
				sprintf(s,"Node \"%s\" is an instance of another node.\nInstanced nodes with Physique modifiers on them are not allowed therefore this node will NOT be saved."
					"\n\nDo you want skip this node and continue exporting?",node->GetName());
				if (bPreviewMode)
					return IDNO;
				res=CryMessageBox(hwnd_nodes, s, "Instanced bone-animated nodes found!", MB_YESNO | MB_ICONINFORMATION | MB_DEFBUTTON2);
				if(res==IDNO) return IDCANCEL;
			}
		}
	}

	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	// Prepare Bones hierarchy.
	//////////////////////////////////////////////////////////////////////////
	INodeTab allBones;
	for (i = 0; i < pb_bones->Count(PB_BONES); i++)
	{
		INode *rootbone = pb_bones->GetINode(PB_BONES,0,i);
		if(!rootbone)
			continue;

		allBones.Append(1,&rootbone);
		if (GetValidChildrenList(rootbone, allBones, true))
			return true;

		// Fill the Bonelist
		for(int j=0;j<allBones.Count();j++)
		{
			INode *node = allBones[j];
			BoneList.GetID( node );
			if(node->GetMtl())
			{
				RecursePrepareMtlChunks(node->GetMtl());
			};
		}

		//============================
		//Prepare bone hierarchy chunk
		//============================
		ch_ent.FileOffset		= -1;
		ch_ent.ChunkType		= ChunkType_BoneAnim;
		ch_ent.ChunkVersion		= BONEANIM_CHUNK_DESC_VERSION;
		ChunkList.Append(rootbone,&ch_ent,true);
	}

	//=======================
	//Prepare Bone Names
	//=======================
	if (BoneList.nodes.Count() > 0 /*&& pb_nodes->GetInt(PB_WRITE_WEIGHTS)*/)
	{
		ch_ent.ChunkType	= ChunkType_BoneNameList;
		ch_ent.ChunkVersion	= BONENAMELIST_CHUNK_DESC_0745::VERSION;
		ChunkList.Append(NULL,&ch_ent,true);

		//ch_ent.ChunkType = ChunkType_BoneInitialPos;
		//ch_ent.ChunkVersion = 0;
		//ChunkList.Append(NULL, &ch_ent, true);

		ch_ent.ChunkType = ChunkType_BoneLightBinding;
		ch_ent.ChunkVersion = BONELIGHTBINDING_CHUNK_DESC_0001::VERSION;
		ChunkList.Append(NULL, &ch_ent, true);

		// for each bone, find the attached light(s) and  add them to the light list, 
		// and to the chunk list to export
		PrepareBoneLightBinding();
	}
	//////////////////////////////////////////////////////////////////////////
	//////////////////////////////////////////////////////////////////////////
	



	//=======================
	//WRITE THE CHUNKS
	//=======================
	//Every chunk is prepared.. Now write them
	m_numSkippedMorphTargets = 0;
	
	int lajno=0;
	m_nDefaultMtlIndex = -1;
	m_nDefaultMtlChunkId = -1;
	for(i=0;i < ChunkList.ch_list.Count();i++)
	{
		int					ChunkType = ChunkList.ch_list[i].ChunkType;
		ReferenceTarget*	ChunkPtr  = ChunkList.ptr_list[i];
		int res=0;

		switch(ChunkType)
		{
		case ChunkType_SourceInfo:
			SaveSourceInfo(f, ChunkType);
			break;
		case ChunkType_Mesh:
			res=SaveMeshObject(f,(INode *)ChunkPtr,ChunkType,allBones, nSMF_SaveMorphTargets|nSMF_SaveInitBonePos);
			break;

		case ChunkType_BoneMesh:
			res=SaveMeshObject(f,(INode *)ChunkPtr,ChunkType,allBones, nSMF_IsBoneMeshChunk|nSMF_DontSaveBoneInfo);
			break;

		case ChunkType_Helper:
			res=SaveHelperObject(f,(INode *)ChunkPtr);
			break;

		case ChunkType_VertAnim:
			res=SaveVertAnim(f,(INode *)ChunkPtr);
			break;

		case ChunkType_Node:
			{
				INode *node = (INode *)ChunkPtr;
				res=SaveNode(f,node );
			}
			break;

		case ChunkType_Controller:
			res=SaveController(f,(Control *)ChunkPtr);
			break;

		case ChunkType_Mtl:
			lajno++;
			res=SaveMaterial(f,(Mtl *)ChunkPtr);
			break;

		case ChunkType_BoneAnim:
			res=SaveBoneHierarchy(f, allBones);
			break;

		case ChunkType_BoneNameList:
			res = SaveBoneNameList(f);
			break;

		case ChunkType_BoneLightBinding:
			res = SaveBoneLightBinding (f);
			break;

		case ChunkType_Light:
			res=SaveLightObject(f,(INode*)ChunkPtr);
			break;

		case ChunkType_SceneProps:
			res=SaveSceneProps(f);
			break;

		case ChunkType_Timing:
			res=SaveTiming(f);
			break;

		case ChunkType_MeshMorphTarget:
			break;

		case ChunkType_BoneInitialPos:
			break;

		default:
			CryMessageBox(hwnd_nodes, "unsuported chunk type encountered", "Internal Error", MB_OK | MB_ICONINFORMATION);
			break;
		}

		if(res) 
			HALT(IDNO);
	}

	SaveDefaultMaterial (f,bBuilding);

	//=======================
	//Write Chunk List
	//=======================
	res=WriteChunkList(f, header);
	if(res) HALT(IDNO);

	//==========================================
	//CLOSE File and Call DLL Callback function
	//==========================================
	fclose(f);

	if (m_numSkippedMorphTargets)
	{
		char szBuf[400];
		sprintf (szBuf, "%d Morph targets were not saved because they don't have vertices with the specified (%g) minimal offset",
			m_numSkippedMorphTargets, pb_nodes->GetFloat (PB_MORPH_MIN_OFFSET));
		MessageBox (ip->GetMAXHWnd(), szBuf, "Warning", MB_OK);
	}

	CallExtensionDLLs(filename,DLL_FLAG_GEOM);
	filename[nFileNameLength] = '\0';
	lastSavedFile = filename;
	m_bLastSavedIndividualFiles = bIndividualFiles;
	
#undef HALT
	}
	return IDYES;
}

// WriteChunkList ______________________________________________________________________________________
bool CSExportUtility::WriteChunkList(FILE *f, FILE_HEADER &header)
{
	//get chunk list pos
	fpos_t chunklistpos;
	fgetpos(f,&chunklistpos);

	//remove the chunks that are not saved
	for(int i=ChunkList.ch_list.Count()-1;i>=0;i--) 
	{
		if(ChunkList.ch_list[i].FileOffset == -1)
		{
			ChunkList.ch_list.Delete(i,1);
			ChunkList.ptr_list.Delete(i,1);
		}
	}

	//=======================
	//Write # of Chunks
	//=======================
	int nch=ChunkList.ch_list.Count();
	int res=fwrite(&nch,sizeof(nch),1,f);
	if(res!=1) return true;

	//=======================
	//Write Chunk List
	//=======================
	for(i=0;i<nch;i++)
	{
		CHUNK_HEADER ch_ent=ChunkList.ch_list[i];
		int res=fwrite(&ch_ent,sizeof(ch_ent),1,f);
		if(res!=1) return true;
	}

	//update Header for chunk list offset
	header.ChunkTableOffset		= int(chunklistpos);
	res=fseek(f,0,SEEK_SET);
	if(res) return true;
	res=fwrite(&header,sizeof(header),1,f);
	if(res!=1) return true;

	return false;
}


int CSExportUtility::SaveDirDlgCallback (HWND hwnd, UINT uMsg, LPARAM lParam)
{
	switch (uMsg)
	{
	case BFFM_INITIALIZED:
		if (lastSavedFile.Length() && m_bLastSavedIndividualFiles)
		{
			const char* szLastFile = lastSavedFile.data();
			SendMessage (hwnd, BFFM_SETSELECTION, (WPARAM)szLastFile, (LPARAM)szLastFile);
		}/*
		else
		{
			TSTR strMaxFile = GetCOREInterface()->GetCurFileName();
			const char* szMaxFile = strMaxFile.data();
			for (int i = strMaxFile.Length()-1; i > 0; --i)
				if(strMaxFile[i] == '\\' || strMaxFile[i] == '/')
				{
					strMaxFile.Resize (i);
					break;
				}
			if (strMaxFile.Length())
				SendMessage (hwnd, BFFM_SETSELECTION, (WPARAM)strMaxFile.data(), TRUE);
		}*/
		break;
	}
	return 0;
}



int CSExportUtility::SaveDirDlg(int FileType,char *filename, char *title, char *def_file_name)
{
	BROWSEINFO bi;
	memset (&bi, 0, sizeof(bi));
	bi.hwndOwner      = hwnd_nodes;
	bi.pszDisplayName = filename;
	bi.lpszTitle      = title;
	bi.ulFlags        = BIF_RETURNONLYFSDIRS|BIF_NEWDIALOGSTYLE;

	bi.lpfn   = SaveDirDlg_BrowseCallbackProc;
	bi.lParam = (LPARAM)this;

	LPITEMIDLIST pIIL = SHBrowseForFolder(&bi);
	if (!pIIL)
		return 0;

	if (!SHGetPathFromIDList (pIIL, filename))
		filename[0] = '\0';

	LPMALLOC pMalloc;
	if (SUCCEEDED(SHGetMalloc (&pMalloc)))
	{
		pMalloc->Free(m_pidlLast);
		pMalloc->Release();
	}

	m_pidlLast = pIIL;

	return filename[0];
}


// SaveFileDlg ______________________________________________________________________________________
// bDirectory  - if it's true, then the directory choose dialog is displayed
int CSExportUtility::SaveFileDlg(int FileType,char *filename, char *title, char *def_file_name)
{
	static char		FileName[256];

	OPENFILENAME	ofn;	
	FilterList		filter;

	char def_ext[16];

	switch(FileType)
	{
	case FileType_Geom:
		if (bBuilding)
		{
			// Save as a building.
			filter.Append(_T("Crytek Building File (*.bld)"));
			filter.Append(_T("*.bld"));
			strcpy(def_ext,"bld");
		}
		else
		{
			// Save as normal.
			filter.Append(_T("Crytek Geometry File (*.cgf)"));
			filter.Append(_T("*.cgf"));
			filter.Append(_T("Crytek Animated Geometry File (*.cga)"));
			filter.Append(_T("*.cga"));
			filter.Append(_T("Animation File (*.anm)"));
			filter.Append(_T("*.anm"));
			strcpy(def_ext,"cgf");
		}
		break;

	case FileType_Anim:
		filter.Append(_T("Crytek Bone Animation File (*.caf)"));
		filter.Append(_T("*.caf"));
		strcpy(def_ext,"caf");
		break;
	
	default: 
		assert(false);
	}

	FileName[0]=0;
	if(def_file_name) strcpy(FileName,def_file_name);

	//Set dialog box options
	ofn.lStructSize       = sizeof(OPENFILENAME);
	ofn.hwndOwner         = hwnd_nodes;
	ofn.hInstance         = NULL;	
	ofn.lpstrTitle        = title;
	ofn.lpstrFilter       = filter;
	ofn.lpstrCustomFilter = NULL;
	ofn.nMaxCustFilter    = 0;
	ofn.nFilterIndex      = 1;
	ofn.lpstrFile         = FileName;
	ofn.nMaxFile          = sizeof(FileName)-1;
	ofn.lpstrFileTitle    = NULL;
	ofn.nMaxFileTitle     = 0;
	ofn.lpstrInitialDir   = m_strLastChosenSaveFile.Length() ? (const char*)m_strLastChosenSaveFile : (const char*)NULL;
	ofn.Flags             = OFN_OVERWRITEPROMPT|OFN_PATHMUSTEXIST|OFN_HIDEREADONLY;
	ofn.nFileOffset       = 0;
	ofn.nFileExtension    = 0;
	ofn.lpstrDefExt       = def_ext;
	ofn.lCustData         = 0;
	ofn.lpfnHook          = NULL;
	ofn.lpTemplateName    = NULL;

	if(!GetSaveFileName(&ofn))
	{
			return NULL; //cancelled (no error though, it is ok)
	}

	if(filename) strcpy(filename,FileName);

	// find the directory separator and remember the directory for subsequent references
	{
		for (int i = strlen(FileName)-1; i >= 0; --i)
		{
			if (FileName[i] == '\\' || FileName[i] == '/')
			{
				FileName[i] = '\0';
				break;
			}
		}
		m_strLastChosenSaveFile = FileName;
	}

	ResetReadOnlyFlag (filename);

	return TRUE;
}



INode *CSExportUtility::FindRootBone(INode *node)
{
	INode *n=node;
	if(!n) return NULL;

	while(n->GetParentNode())
	{
		Control *pc=n->GetTMController();
		if(pc && (pc->ClassID() == BIPBODY_CONTROL_CLASS_ID)) 
		{
			while (n->GetParentNode()
				&& n->EvalWorldState(0).obj->ClassID() == Class_ID(DUMMY_CLASS_ID,0))
			return n;
		}

		if(n->GetParentNode()->IsRootNode()) return n;
		n=n->GetParentNode();
	}
	
	return NULL;
}

void CSExportUtility::UpdateProgress(int percent, char *format,  ...)
{
	static int  old_percent=-1234;
	static char old_format[64];

	va_list list;
	va_start(list, format);

	if(!hProgressBar) return;
	if((percent==old_percent) && (!strcmp(format,old_format))) return;

	old_percent=percent;
	strcpy(old_format,format);

	vsprintf(s, format, list);
	SetWindowText(hProgressWindow, s);
	SendMessage(hProgressBar, PBM_SETPOS, percent, 0);
}

void CSExportUtility::EnableDisableControls()
{
	if(!pmap_bones) return;
	if(!pmap_nodes) return;

	Interval ivalid;
	//check for dublicate entry
	bool RootFound=false;
	/*
	int nNodes=pb_nodes->Count(PB_NODES);
	for(int i=0;i<nNodes;i++)
	{
		INode *node;
		pb_nodes->GetValue(PB_NODES,0,node,ivalid, i);

		if(FindPhysiqueModifier(node) && FindRootBone(node)) 
		{
			RootFound=true;
			break;
		}
	}	

	pmap_export->Enable(PB_WRITE_BONE_ANIM, RootFound);
	*/
}

void CSExportUtility::RemoveChildBones(INode *parent)
{
	if(!pb_nodes)	return;
	if(!pb_bones)	return;
	if(!pmap_bones) return;
	if(!pmap_nodes) return;

	for(int i=pb_bones->Count(PB_BONES)-1;i>=0;i--)
	{
		INode *bone=pb_bones->GetINode(PB_BONES,0,i);
		if(!bone) continue;

		bone=bone->GetParentNode();
		while(bone)
		{
			if(bone==parent)
			{
				DebugPrint("%s deletes the child bone %s\n",parent->GetName(),pb_bones->GetINode(PB_BONES,0,i)->GetName());
				DebugPrint("before: %d",pb_bones->Count(PB_BONES));
				pb_bones->Delete(PB_BONES,i,1);
				DebugPrint("after: %d\n",pb_bones->Count(PB_BONES));
				break;
			}

			bone=bone->GetParentNode();
		}
	}
}

void CSExportUtility::AddRelatedBones()
{
	if(!pb_nodes)	return;
	if(!pb_bones)	return;
	if(!pmap_bones) return;
	if(!pmap_nodes) return;

	
	pb_bones->SetCount(PB_BONES,0);

	int nobj=pb_nodes->Count(PB_NODES);
	for(int k=0;k<nobj;k++)
	{
		INode *nodek=pb_nodes->GetINode(PB_NODES,0,k);
		if(!nodek) continue;

		
		

		Modifier *phy_mod=FindPhysiqueModifier(nodek->GetObjectRef());
		if(!phy_mod) 
		{
			Modifier *skin_mod=FindSkinModifier(nodek->GetObjectRef());
			if (!skin_mod)  continue;	
			AddSkinRelatedBones(skin_mod,nodek);
			continue;
		}
		
	

		IPhysiqueExport *phyInterface = (IPhysiqueExport *)( phy_mod->GetInterface(I_PHYINTERFACE));
		if(!phyInterface) continue;

		IPhyContextExport *phyContex = phyInterface->GetContextInterface(nodek);
		if(!phyContex) continue;
		
		phyContex->ConvertToRigid(true);
		phyContex->AllowBlending(true);

		int n_vert=phyContex->GetNumberVertices();
		for(int i=0;i<n_vert;i++)
		{
			IPhyVertexExport *vert=phyContex->GetVertexInterface(i);
			
			int vert_type=vert->GetVertexType();
			switch(vert_type)
			{
			case RIGID_NON_BLENDED_TYPE:
				{
					IPhyRigidVertex *RigVert=(IPhyRigidVertex *)vert;
					INode *bone=RigVert->GetNode();

					PB2Value v;
					v.r=bone;

					if(pb_nodes->GetInt(PB_ADD_WHOLE_SKEL))
					{
						INode *root_bone=FindRootBone(bone);
						if(bone_validator.Validate(v))
						{
							pb_bones->Append(PB_BONES,1,&bone);
							DebugPrint("\"%s\" added....\n",bone->GetName());
							//phyContex->ReleaseVertexInterface(vert);
							break;
						}
					}

					if(bone_validator.Validate(v))
					{
						pb_bones->Append(PB_BONES,1,&bone);
						DebugPrint("\"%s\" added....\n",bone->GetName());
						//RemoveChildBones(bone);
					}
				}
				break;

			case RIGID_BLENDED_TYPE:
				{
					IPhyBlendedRigidVertex *BlndVert=(IPhyBlendedRigidVertex *)vert;
					int nLinks=BlndVert->GetNumberNodes();
					for(int j=0;j<nLinks;j++)
					{
						INode *bone=BlndVert->GetNode(j);
						
						PB2Value v; v.r=bone;

						if(pb_nodes->GetInt(PB_ADD_WHOLE_SKEL))
						{
							INode *root_bone=FindRootBone(bone);
							//DebugPrint("Root of \"%s\" is \"%s\"\n",bone->GetName(), root_bone->GetName());

							v.r=root_bone;
							if(bone_validator.Validate(v))
							{
								pb_bones->Append(PB_BONES,1,&root_bone);
								DebugPrint("\"%s\" added....\n",root_bone->GetName());
								i=n_vert+1; //break the outer loop (i) also
								break;
							}
						}

						if(bone_validator.Validate(v))
						{
							pb_bones->Append(PB_BONES,1,&bone);
							DebugPrint("\"%s\" added....\n",bone->GetName());
							//RemoveChildBones(bone);
						}
					}
				}
				break;
			}
		
			phyContex->ReleaseVertexInterface(vert);
		} //for vertex
		
		phyInterface->ReleaseContextInterface(phyContex);
		phy_mod->ReleaseInterface(I_PHYINTERFACE, phyInterface);
	} //for obj
}

INode *CSExportUtility::DerObj2Node(Object *obj)
{
	int n=pb_nodes->Count(PB_NODES);
	for(int i=0;i<n;i++)
	{
		INode *node=pb_nodes->GetINode(PB_NODES,0,i);
		if(node && node->GetObjectRef()==obj ) return node;
	}

	return NULL;
}

//sorts the tab and removes duplicate elements
static int CompTable( const void *elem1, const void *elem2 ) 
{
	int a = *((int *)elem1);
	int b = *((int *)elem2);
	
	return a-b;
}

void CSExportUtility::CleanUpIntTab(IntTab &tab)
{
	tab.Sort(CompTable);
	for(int i=1;i<tab.Count();i++)
	{
		if(tab[i] == tab[i-1])
		{	
			tab.Delete(i,1);
			i--;
		}
	}
}

bool CSExportUtility::IsBipedBone(INode *node)
{
	if(!node) return false;
	Control *pc=node->GetTMController();
	return (pc && (
		(pc->ClassID() == BIPBODY_CONTROL_CLASS_ID) || 
		(pc->ClassID() == FOOTPRINT_CLASS_ID) || 
		(pc->ClassID() == BIPSLAVE_CONTROL_CLASS_ID)));
}

bool CSExportUtility::IsBipedRoot(INode *node)
{
	if(!node) return false;
	Control *pc=node->GetTMController();
	return (pc && (pc->ClassID() == BIPBODY_CONTROL_CLASS_ID));
}

bool CSExportUtility::IsFootPrint(INode *node)
{
	if(!node) return false;
	Control *pc=node->GetTMController();
	return (pc && (pc->ClassID() == FOOTPRINT_CLASS_ID));
}

bool CSExportUtility::IsLegBone(INode *node)
{
	if(!node) return false;
	if(FindRThigh(node,false))	return true;
	if(FindLThigh(node,false))	return true;
	if(FindRThigh(node,true))	return true;
	if(FindLThigh(node,true))	return true;
	return false;
}

INode * CSExportUtility::FindLThigh(INode *root, bool forward)
{
	char *name=root->GetName();
	char *pos=strstr(name," L Thigh");
	if(pos)
	{
		if( (pos-name+strlen(pos)) == strlen(name))
		{
			return root;
		}
	}

	if(forward)
	{
		for(int i=0;i<root->NumberOfChildren();i++)
		{
			if(INode *found=FindLThigh(root->GetChildNode(i),forward)) return found;
		}
	}
	else if(root->GetParentNode())
	{
		if(INode *found=FindLThigh(root->GetParentNode(),forward)) return found;
	}


	return NULL;
}

INode * CSExportUtility::FindRThigh(INode *root, bool forward)
{
	char *name=root->GetName();
	char *pos=strstr(name," R Thigh");
	if(pos)
	{
		if( (pos-name+strlen(pos)) == strlen(name))
		{
			return root;
		}
	}

	if(forward)
	{
		for(int i=0;i<root->NumberOfChildren();i++)
		{
			if(INode *found=FindRThigh(root->GetChildNode(i),forward)) return found;
		}
	}
	else if(root->GetParentNode())
	{
		if(INode *found=FindRThigh(root->GetParentNode(),forward)) return found;
	}
	return NULL;
}

void CSExportUtility::CallExtensionDLLs(char *fname, int Flags)
{
	//Try to load CryPlug.dll
	char extension_name[256];
	sprintf(extension_name,"%s\\%s",ip->GetDir(APP_PLUGCFG_DIR),"Crytek*.dll");

	WIN32_FIND_DATA fd;
	HANDLE hSearch=FindFirstFile(extension_name, &fd);
	if(hSearch != INVALID_HANDLE_VALUE)
	{
		do
		{
			char dllname[256];
			sprintf(dllname,"%s\\%s",ip->GetDir(APP_PLUGCFG_DIR), fd.cFileName);
			UpdateProgress(25,"Loading Extension DLL: \"%s\"",fd.cFileName);
			HINSTANCE hDLL=LoadLibrary(dllname);
			CryCallbackFunc geomCallBack=NULL;
			CryCallbackFunc animCallBack=NULL;
			if(hDLL)
			{
				geomCallBack=(CryCallbackFunc)GetProcAddress(hDLL,_T("GeomCallback"));
				animCallBack=(CryCallbackFunc)GetProcAddress(hDLL,_T("AnimCallback"));
				
				if((Flags & DLL_FLAG_GEOM) && geomCallBack)
				{
					UpdateProgress(50,"Executing Extension DLL: \"%s\"",fd.cFileName);
					int res=geomCallBack(hwnd_nodes,fname);
					if(res && pb_options->GetInt(PB_WARN_DLL))
					{
						sprintf(s,"Extension DLL file \"%s\" returned an error value of %d",dllname,res);
						MessageBox(hwnd_nodes,s,"Extension DLL Error",MB_OK);
					}
				}

				if((Flags & DLL_FLAG_ANIM) && animCallBack)
				{
					UpdateProgress(75,"Executing Extension DLL: \"%s\"",fd.cFileName);
					int res=animCallBack(hwnd_nodes,fname);
					if(res && pb_options->GetInt(PB_WARN_DLL))
					{
						sprintf(s,"Extension DLL file \"%s\" returned an error value of %d",dllname,res);
						MessageBox(hwnd_nodes,s,"Extension DLL Error",MB_OK);
					}
				}

				FreeLibrary(hDLL);      
				hDLL=NULL;

				UpdateProgress(100,"Executing Extension DLL: \"%s\"",fd.cFileName);
			}
			else if(pb_options->GetInt(PB_WARN_DLL))
			{
				sprintf(s,"Can not load the Extension DLL file \"%s\"\nDo you want to continue executing other Extension DLLs?",dllname);
				int res=MessageBox(hwnd_nodes,s,"Extension DLL Error",MB_YESNO);
				if(res==IDNO) break;
			}

		} while (FindNextFile(hSearch, &fd));
	}
}

//DEL UVVert CSExportUtility::CropFilter(int uvindex,int numFaces,UVVert *pUVerts, TVFace *pTFaces, Face *pFaces, INode *node)
//DEL {
//DEL 
//DEL 	UVVert temp;
//DEL 	temp.Set(pUVerts[uvindex].x,pUVerts[uvindex].y,pUVerts[uvindex].z);
//DEL 
//DEL 	int faceindex=0;
//DEL 
//DEL 	// first find to which face this uv pair belongs
//DEL 	for (int i=0;i<numFaces;i++)
//DEL 		for (int j=0;j<3;j++)
//DEL 			if (pTFaces[i].t[j] == uvindex) faceindex=i;
//DEL 
//DEL 	// get the material of the face
//DEL 	MtlID matid = pFaces[faceindex].getMatID();
//DEL 	Mtl *material = node->GetMtl();
//DEL 	if (material->IsMultiMtl()) material = material->GetSubMtl(matid);
//DEL 	
//DEL 	// get the diffuse TextureMap and get it down to bitmapinfo
//DEL 	Texmap *diffuse = material->GetSubTexmap(ID_DI);
//DEL 	if(diffuse->ClassID() == Class_ID(BMTEX_CLASS_ID, 0))
//DEL 	{
//DEL 		BitmapTex *diffusebmp = (BitmapTex*)diffuse;
//DEL 		BitmapInfo bmpinfo = diffusebmp->GetBitmap(0)->Storage()->bi;
//DEL 	
//DEL 		if (bmpinfo.CustWidth() != bmpinfo.Width()) 
//DEL 			MessageBox(NULL,"Yadda!","KKD",MB_OK);
//DEL 		
//DEL 	}
//DEL 	
//DEL 	return temp;
//DEL }


// arrVertexMap maps compacted vertices -> original Max vertices (through the split vertices)
bool CSExportUtility::SaveBoneInfo(FILE *f, INode *node, Point3 *Vertices,std::vector<int> &arrVertexMap)
{

	// try to find physique or skin

	Modifier *mod=FindPhysiqueModifier(node->GetObjectRef());

	if(mod) 
		return SavePhysiqueInfo(f,node,Vertices,arrVertexMap);
	else
	{
		mod = FindSkinModifier(node->GetObjectRef());
		if (mod) 
			return SaveSkinInfo(f,node,mod,Vertices,arrVertexMap);
		else
			return true;  // no modifier found
	}

	return false;
		
}

// returns false if it's ok
// arrVertexMap maps compacted vertices -> original Max vertices (through the split vertices)
// nChunkIdMesh - the chunk id of the mesh of pNode that is to be exported or already has been exported
// tm - with this matrix, all the target vertices will be transformed (the original Max vertices were transformed with this matrix before being exported)
bool CSExportUtility::SaveMorphTargets (FILE* f, INode* pNode, const Matrix3& tm, std::vector<int> &arrVertexMap, int nChunkIdMesh)
{
	MorphR3* pMorpher = FindMorpherModifier (pNode);
	if (!pMorpher)
		return false; // it's ok

	for (int nChannel = 0; nChannel < g_nMaxMorphChannels; ++nChannel)
	{
		morphChannel& rChannel = pMorpher->chanBank[nChannel];
		if (rChannel.mActive)
		{
			rChannel.rebuildChannel();
			if (SaveMorphTarget (f, tm, rChannel, arrVertexMap, nChunkIdMesh))
				return true;
		}
	}
	return false;
}

// arrVertexMap maps compacted vertices -> original Max vertices (through the split vertices)
// returns error bit
bool CSExportUtility::SaveMorphTarget (FILE* f, const Matrix3& tm, morphChannel& rChannel, std::vector<int> &arrVertexMap, int nChunkIdMesh)
{
	float fMinOffsetSqr = pb_nodes->GetFloat(PB_MORPH_MIN_OFFSET);
	fMinOffsetSqr *= fMinOffsetSqr;

	int nModifiedVertex;
	unsigned numMorphVertices = 0;

	// the bit set of modified vertices, in Max indexation
#if ( MAX_PRODUCT_VERSION_MAJOR == 4 )

	// Code specific for 3ds max 4

	BitArray arrModifiedMaxVerts (rChannel.nPts);
	arrModifiedMaxVerts.ClearAll();

	// set the bits for the vertices modified in Max indexation
	for (nModifiedVertex = 0; nModifiedVertex < rChannel.nmPts; ++nModifiedVertex)
	{
		int nMaxVertex = rChannel.mOptdata[nModifiedVertex];
		float fDeltaLen = rChannel.mDeltas[nMaxVertex].LengthSquared();
		if (fDeltaLen >= fMinOffsetSqr)
		{
			arrModifiedMaxVerts.Set(nMaxVertex);
			++numMorphVertices;
		}
	}

#else

	// Code specific for 3ds max 5 and 6

	BitArray arrModifiedMaxVerts (rChannel.mNumPoints);
	arrModifiedMaxVerts.ClearAll();

	// set the bits for the vertices modified in Max indexation
	for (nModifiedVertex = 0; nModifiedVertex < rChannel.mNumPoints; ++nModifiedVertex)
	{
		float fDeltaLen = rChannel.mDeltas[nModifiedVertex].LengthSquared();
		if (fDeltaLen >= fMinOffsetSqr)
		{
			arrModifiedMaxVerts.Set(nModifiedVertex);
			++numMorphVertices;
		}
	}

#endif


	if (!numMorphVertices)
	{
		++m_numSkippedMorphTargets;
		return false;
	}

	// add the chunk to the chunk list, already with the file offset info
	fpos_t fpos;
	fgetpos(f,&fpos);
	CHUNK_HEADER ch_ent;
	ch_ent.FileOffset = (int)fpos;
	ch_ent.ChunkType  = ChunkType_MeshMorphTarget;
	ch_ent.ChunkVersion = MESHMORPHTARGET_CHUNK_DESC_0001::VERSION;
	ChunkList.Append(NULL, &ch_ent, true);

	// write the header of the chunk directly into the file
	MESHMORPHTARGET_CHUNK_DESC_0001 ChunkHeader;
	ChunkHeader.nChunkIdMesh = nChunkIdMesh;
	ChunkHeader.numMorphVertices = 0;

	// calculate the number of the vertices modified in Cry indexation (some of Max-indexed vertices may not be present,
	// or some may be present twice)
	for (nModifiedVertex = 0; nModifiedVertex < (int)arrVertexMap.size(); ++nModifiedVertex)
	{
		if (arrModifiedMaxVerts[arrVertexMap[nModifiedVertex]])
			++ChunkHeader.numMorphVertices;			
	}

	// write the header finally
	if (1 != fwrite (&ChunkHeader, sizeof(ChunkHeader), 1, f))
		return true;

	// for each modified vertex (in Cry indexation) write the corresponding structure
	for (nModifiedVertex = 0; nModifiedVertex < (int)arrVertexMap.size(); ++nModifiedVertex)
	{
		if (arrModifiedMaxVerts[arrVertexMap[nModifiedVertex]])
		{
			SMeshMorphTargetVertex MorphVertex;
			MorphVertex.nVertexId = nModifiedVertex;
			MorphVertex.ptVertex  << tm.PointTransform(rChannel.mPoints[arrVertexMap[nModifiedVertex]]);
			if (1 != fwrite (&MorphVertex, sizeof(MorphVertex), 1, f))
				return true;
		}
	}
	// now write the name of the morph target; in the future, padding may be needed for proper alignment
	if (1 != fwrite (rChannel.mName.data(), rChannel.mName.length()+1, 1, f))
		return true;

	return false; // all OK
}

bool CSExportUtility::IsHaveBoneInfo( INode *node )
{
	// try to find physique or skin
	Modifier *mod=FindPhysiqueModifier(node->GetObjectRef());
	if(mod) 
		return true;
	else
	{
		mod = FindSkinModifier(node->GetObjectRef());
		if (mod) 
			return true;
	}
	// Doesnt have bone info.
	return false;		
}

bool CSExportUtility::SaveSkinInfo(FILE *f, INode *node, Modifier *m, Point3 *Vertices,std::vector<int> &arrVertexMap)
{

	DebugPrint("\nSaving skin info for \"%s\"", node->GetName());
	Interval ivalid;
	TimeValue time = GetTime();
	Matrix3 tm = node->GetObjTMAfterWSM(time);

	ISkin *skinInterface = (ISkin *) m->GetInterface(I_SKIN);
	if (!skinInterface) return true;

	ISkinContextData *skinContext = skinInterface->GetContextInterface(node);
	if (!skinContext) return true;

//	int nVerts = skinContext->GetNumPoints();
	int nVerts = arrVertexMap.size();

	for (int k=0;k<nVerts;k++)
	{
		int i = arrVertexMap[k];
		UpdateProgress(100*i/nVerts,"\"%s\" writing bone weigths", node->GetName());
		
		Point3 worldvertex = tm*Vertices[k];

		int nBones = skinContext->GetNumAssignedBones(i);
		
		// write number of assigned links
		int res = fwrite(&nBones, sizeof(nBones), 1, f);
		if (res!=1) return true;
		float tot_blend=0;

		for (int j=0;j<nBones;j++)
		{
			CryLink link;

			INode *pBone = skinInterface->GetBone(skinContext->GetAssignedBone(i,j));
			link.BoneID = BoneList.GetID(pBone);
			tot_blend += link.Blending = skinContext->GetBoneWeight(i,j);
			Point3 off = worldvertex*BoneList.invTM[link.BoneID];

			link.offset.x = off.x;
			link.offset.y = off.y;
			link.offset.z = off.z;

			res = fwrite(&link,sizeof(link),1,f);
			if (res!=1) return true;
		}

		
	}

	m->ReleaseInterface(I_SKIN,skinInterface);

	DebugPrint("Done exporting skin for \"%s\"",node->GetName());
	return false;
}



bool CSExportUtility::AddSkinRelatedBones(Modifier *skin, INode *node)
{
	ISkin *skinInterface = (ISkin *) skin->GetInterface(I_SKIN);
	if (!skinInterface) return true;

	ISkinContextData *skinContext = skinInterface->GetContextInterface(node);

	int nVerts = skinContext->GetNumPoints();

	for (int i=0;i<nVerts;i++)
	{
		 int nLinks = skinContext->GetNumAssignedBones(i);
		 for (int j=0;j<nLinks;j++)
		 {
			 INode *bone = skinInterface->GetBone(skinContext->GetAssignedBone(i,j));
			 PB2Value v; 
			 v.r = bone;

			 if (pb_nodes->GetInt(PB_ADD_WHOLE_SKEL))
			 {
				 INode *root_bone = FindRootBone(bone);
				 v.r = root_bone;

				 if (bone_validator.Validate(v))
				 {
					 pb_bones->Append(PB_BONES,1,&root_bone);
					 i=nVerts; // break outer loop
					 break;
				 }
			 } 

			 if (bone_validator.Validate(v))
			 {
				 pb_bones->Append(PB_BONES,1,&bone);
			 }
		 }
	}

	skin->ReleaseInterface(I_SKIN, skinInterface);
	return false;
}


bool CSExportUtility::PreviewGeomFile( TSTR fileName )
{
	if (fileName.length() == 0)
		return true;

	/*
	char cidpath[_MAX_PATH];
  char drive[_MAX_DRIVE];
  char dir[_MAX_DIR];
  char fname[_MAX_FNAME];
  char ext[_MAX_EXT];


	// Find .caf file with same name in same directory.
	_splitpath( fileName.data(),drive,dir,fname,ext );
	_makepath( cidpath,drive,dir,fname,".caf" );

	FILE *f = fopen( cidpath,"rb" );
	if (f)
	{
		// If .caf file found load it instead of cgf.
		fclose(f);
		fileName = cidpath;
	}
	*/
	HWND h = FindWindow( NULL,"CryEngine Sandbox" );		//changed by [MG]
	if (!h)
	{
		// Open new instance of CryEdit.
		char cmd[1024];
		
		HKEY hKey;
		char szMasterCD[_MAX_PATH];
		DWORD dwBufLen;
		
		RegOpenKeyEx( HKEY_CURRENT_USER,"Software\\Crytek",0,KEY_QUERY_VALUE,&hKey );
		if (RegQueryValueEx( hKey,"MasterCD", NULL, NULL,(LPBYTE)szMasterCD,&dwBufLen) != ERROR_SUCCESS)
		{
			strcpy( szMasterCD,"C:\\MasterCD" );
		}
		RegCloseKey( hKey );
	
		// Remove backslash if present.
		if (szMasterCD[strlen(szMasterCD)-1] == '\\')
			szMasterCD[strlen(szMasterCD)-1] = 0;
		
		sprintf( cmd,"%s\\Editor.exe %s",szMasterCD,(char*)fileName.data() );

		PROCESS_INFORMATION pi;
		STARTUPINFO si;
		memset( &si,0,sizeof(si) );
		si.cb = sizeof(si);
		CreateProcess( NULL,cmd,NULL,NULL,FALSE,0,NULL,szMasterCD,&si,&pi );
		Sleep(100);
	}
	else
	{
		char szData[1024];
		strcpy( szData,fileName );
		COPYDATASTRUCT data;
		data.dwData = 1024;
		data.cbData = strlen(szData);
		data.lpData = szData;
		SendMessage( h,WM_COPYDATA,(WPARAM)NULL,(LPARAM)&data );
		SetActiveWindow( h );
		SetForegroundWindow( h );
	}
	return false;
}

bool CSExportUtility::PreviewGame( TSTR fileName )
{
	if (fileName.length() == 0)
		return true;

	// Open new instance of Game.
	char cmd[1024];
		
	HKEY hKey;
	char szMasterCD[_MAX_PATH];
	DWORD dwBufLen;
		
	RegOpenKeyEx( HKEY_CURRENT_USER,"Software\\Crytek",0,KEY_QUERY_VALUE,&hKey );
	if (RegQueryValueEx( hKey,"MasterCD", NULL, NULL,(LPBYTE)szMasterCD,&dwBufLen) != ERROR_SUCCESS)
	{
		strcpy( szMasterCD,"C:\\MasterCD" );
	}
	RegCloseKey( hKey );
	
	// Remove backslash if present.
	if (szMasterCD[strlen(szMasterCD)-1] == '\\')
		szMasterCD[strlen(szMasterCD)-1] = 0;
		
	sprintf( cmd,"%s\\XIsle.exe /preview %s",szMasterCD,(char*)fileName.data() );

	PROCESS_INFORMATION pi;
	STARTUPINFO si;
	memset( &si,0,sizeof(si) );
	si.cb = sizeof(si);
	CreateProcess( NULL,cmd,NULL,NULL,FALSE,0,NULL,szMasterCD,&si,&pi );
	Sleep(100);

	return false;
}

CStr CSExportUtility::GetRelativePath( const TSTR &fullPath )
{
	char path_buffer[_MAX_PATH];
	char drive[_MAX_DRIVE];
	char dir[_MAX_DIR];
	char fname[_MAX_FNAME];
	char fext[_MAX_EXT];

	strcpy( path_buffer,fullPath.data() );
	_splitpath( path_buffer,drive,dir,fname,fext );
	char *pos = dir;
	if (pos[0] == '\\')
		pos++;
	pos = strchr(pos,'\\');
	if (pos)
	{
		char temp[_MAX_DIR];
		strcpy( temp,pos+1 );
		strcpy( dir,temp );
	}
	else
	{
		strcpy( dir,"" );
	}
	_makepath( path_buffer,NULL,dir,fname,fext );
	return path_buffer;
}