// Implementation of the class that implements CryExportCgf Maya plugin command
#include "StdAfx.h"
#include "Cryheaders.h"
#include "cmdexportcgf.h"
#include "MayaCryMeshUtil.h"
#include "MayaCryShaderUtil.h"
#include "MayaCryUtils.h"
#include "crc32.h"


CCmdExportCgf::CCmdExportCgf(void):
	m_fMinLinkWeight(0.025f),
	m_hasBoneInfo(true)
{
}

CCmdExportCgf::~CCmdExportCgf(void)
{
}

//////////////////////////////////////////////////////////////////////////
// constructs a syntax object
MSyntax CCmdExportCgf::newSyntax()
{
	MSyntax syntax = CCmdExportCryFile::newSyntax ();
	
	return syntax;
}


//////////////////////////////////////////////////////////////////////////
// The main entry point
// Exports CGF file from the selection of bones and/or shapes.
// Normally, you would wish to select one or more bones out of a skeleton
// influencing the shape that is to be exported, or the shape itself, or both.
MStatus	CCmdExportCgf::doIt ( const MArgList& args)
{
	MStatus status = parseArgs(args);
	if (!status)
	{
		status.perror ("Cannot parse arguments");
		return status;
	}

	if (m_nMaxWeightsPerLink == 0)
		m_hasBoneInfo = false;

	// m_bExportIndividualFiles?"directory":"CGF", 
	Log ("Exporting to CGF \"%s\"", m_strFileName.asChar());

	MSelectionList list;
	if (m_lstSelection.length())
		list = m_lstSelection;
	else
	{
		status = MGlobal::getActiveSelectionList(list);

		if (!status)
		{
			Log ("*ERROR* couldn't get the current active selection list: %s", status.errorString().asChar());
			status.perror ("Cannot get the current active selection list");
			return status;
		}
	}

	//setHistoryOn (false);
	progressStart("Preparing mesh...", 3 + list.length() + m_setShapes.size());
	try
	{
		CMayaAutoAnimTempState mayaAAState;
		// prepare the shapes to export
		prepareObjectsToExport (list);

		dumpExportedObjects();
		progressIncrement();
		
		// prepare the bones to export
		if (m_hasBoneInfo)
		{
			initBones ();
			prepareAttachedLightsToExport();
		}
		progressIncrement();

		// check for multiple skeletons in the exported bone hierarchy
		if (m_Bones.numRoots () > 1)
		{
			status = MStatus::kFailure;
			status.perror ("More than 1 root bone found in the hierarchy: (" + m_Bones.getRootBoneDump() + "). Currently, multiple skeleton export is not supported");
			throw status;
		}

		if (m_Bones.numBones() == 0)
			m_hasBoneInfo = false;

		// set the whole model to the default pose (the pose will later be restored)
		//m_Bones.pushPosition ();
		//m_Bones.resetFromRestPosition ();
    
		// start writing the file
		m_pFile = new CChunkFileWriterRefTgt (m_strFileName.asChar(), FileType_Geom, GeomFileVersion);

		progressIncrement();
		exportTiming();

		progressEnd();
		progressStart("Exporting", m_setLights.size()*2 + m_setShapes.size()*4 + m_setHelpers.size()*2+4);
		// export all lights and their corresponding nodes
		for (MDagPathSet::const_iterator itLight = m_setLights.begin(); itLight != m_setLights.end(); ++itLight)
		{
			int nLightObjectId = exportLight (*itLight);
			prepareNodeForExport (*itLight, nLightObjectId);
			prepareLightBindingInfo (*itLight, nLightObjectId);
			progressIncrement();
		}

		// export all shapes and their corresponding nodes
		for (MDagPathSet::const_iterator itShape = m_setShapes.begin(); itShape != m_setShapes.end(); ++itShape)
		{
			int nNodeObjectId = exportShape (*itShape);
			progressIncrement(2);
			prepareNodeForExport (*itShape, nNodeObjectId);
			progressIncrement();
		}

		// export all helpers and their corresponding nodes
		for (MDagPathSet::const_iterator itHelper = m_setHelpers.begin(); itHelper != m_setHelpers.end(); ++itHelper)
		{
			int nNodeObjectId = exportHelper (*itHelper);
			if (nNodeObjectId != -1)
				prepareNodeForExport (*itHelper, nNodeObjectId);
			progressIncrement();
		}

		exportLightBinding();

		exportAllNodes();

		// after everything is exported, export the bone info: the names, the hierarchy
		if (m_hasBoneInfo)
		{
			exportBoneNameList ();
			progressIncrement();
			exportBoneAnim ();
		}
		progressIncrement();
		
		// close the file
		m_pFile = NULL;
		progressIncrement();

		// restore the bone poses
		//m_Bones.popPosition ();
	}
	catch (CChunkFileWriter::Error& e)
	{
		Log ("*ERROR* writing file: %s", e.c_str());
		std::cerr << "Cannot write to file: " << e.c_str () << '\n';
		status = MS::kFailure;
	}
	catch (MStatus& e)
	{
		Log ("*ERROR* status error %s", e.errorString().asChar());
		std::cerr << e << '\n';
		status = e;
		if (!status.error())
			status = MS::kFailure;
	}

	progressEnd();

	std::cout.flush();
	std::cerr.flush();
	return status;
}


// logs the exported shape names
void CCmdExportCgf::dumpExportedObjects()
{
	std::string strExportShapes  = toString (m_setShapes);
	std::string strExportLights  = toString (m_setLights);
	std::string strExportHelpers = toString (m_setHelpers);
	Log ("Exporting shapes: {%s}, lights: {%s}, helpers: {%s}", strExportShapes.c_str (), strExportLights.c_str(), strExportHelpers.c_str());
}


// prepares the array of shapes for export (fills the m_arrShapes in)
void CCmdExportCgf::prepareObjectsToExport(const MSelectionList& listSelection)
{
	for ( MItSelectionList listIter( listSelection ); !listIter.isDone(); listIter.next() )
	{
		MDagPath pathDagNode;
		MObject component;
		listIter.getDagPath( pathDagNode, component );

		//MFnDagNode fnDagNode ( pathDagNode );
		//Log ("Selected '%s': '%s'; component: '%s'", pathDagNode.node().apiTypeStr(), fnDagNode.name().asChar(), component.isNull() ? "NULL" : component.apiTypeStr() );

		MDagPath pathShape = pathDagNode;
		MStatus status = pathShape.extendToShape ();
		if (!status)
		{
			// this is not a shape - if it's a bone influencing some shape (s) then add those shapes
			prepareNonShapeToExport (pathDagNode);
		}
		else
		{
			Log0 ("Extended to shape: %s, apiType %s", pathShape.fullPathName ().asChar (), MFnDagNode(pathShape).object().apiTypeStr ());

			//if (pathShape.node().hasFn (MFn::kJoint))

			if (pathShape.node().hasFn (MFn::kMesh))
			{
				Log0("Exporting as a mesh");
				// the shape node is selected, export it
				m_setShapes.insert (pathShape);
			}

			if (pathShape.node().hasFn(MFn::kLight))
			{
				Log0 ("Exporting as a light");
				prepareLightToExport (pathShape);
			}

			if (pathShape.node().hasFn(MFn::kCamera))
			{
				Log0 ("Exporting as a camera (helper)");
				m_setHelpers.insert (pathShape);
			}

			if (pathShape.node().hasFn(MFn::kLocator))
			{
				Log0("Exporting as a locator (helper)");
				m_setHelpers.insert (pathShape);
			}
		}
		progressIncrement();
	}
}



// adds the lights attached to the bones
void CCmdExportCgf::prepareAttachedLightsToExport ()
{
	// iterate through the bones and if there are lights attached to a bone, then export them, too
	MFnLight fnLight;
	for (int nBoneId = 0; nBoneId < m_Bones.numBones(); ++nBoneId)
	{
		MDagPath pathBone = m_Bones.getBone(nBoneId).pathBone;
		unsigned nChild, numChildren = pathBone.childCount ();
		for (nChild = 0; nChild < numChildren; ++nChild)
		{
			MObject objLight = pathBone.child(nChild);
			if (fnLight.setObject(objLight))
			{
				// this is a light attached to a bone
				MDagPath pathLight = pathBone;
				pathLight.push (objLight);
				prepareLightToExport (pathLight);
			}
		}
	}
}

// prepares the given light for export - adds it to the light list
void CCmdExportCgf::prepareLightToExport (const MDagPath& pathDagNode	)
{
	m_setLights.insert (pathDagNode);
}

void CCmdExportCgf::prepareNonShapeToExport(const MDagPath& pathDagNode)
{
	if (prepareConnectedFleshToExport(pathDagNode))
		return;

	// this is a helper
	m_setHelpers.insert (pathDagNode);
}

//////////////////////////////////////////////////////////////////////////
// determines the shapes that this node (Bone) may influence
// and adds them to the set of shapes to export
bool CCmdExportCgf::prepareConnectedFleshToExport  (const MDagPath& pathDagNode)
{
	// Prepares the flesh on the bones to export
	//
	MStatus status;
	MFnGeometryFilter fnSkinCluster(findSkinCluster (pathDagNode, MItDependencyGraph::kDownstream), &status);
	if (!status)
	{
		Log ("*WARNING* Ignoring selection item %s - no skin cluster found", pathDagNode.fullPathName ().asChar ());
		return false;
	}

	// found a cluster - export all the shapes in it
	MObjectArray arrOutGeoms;
	if (!fnSkinCluster.getOutputGeometry (arrOutGeoms))
	{
		Log ("*WARNING* Ignoring selection item %s - no output geometry found: %s", pathDagNode.fullPathName ().asChar (), status.errorString ().asChar ());
		return false; // couldn't find the output geometry for the bone - ignore it
	}

	// for each geometry, determine the dag nodes and add to the set
	for (unsigned nGeom = 0; nGeom < arrOutGeoms.length (); ++nGeom)
	{
		MObject& objGeom = arrOutGeoms[nGeom];
		if (!objGeom.hasFn (MFn::kMesh))
		{
			Log ("*WARNING* Ignoring selection item's %s skin cluster %s output geometry %d of type %s: the object is not a mesh", pathDagNode.fullPathName ().asChar (), fnSkinCluster.name ().asChar (), nGeom, arrOutGeoms[nGeom].apiTypeStr ());
			continue;
		}

		MStatus status;
    MDagPathArray arrAllPaths;
		status = MDagPath::getAllPathsTo (objGeom, arrAllPaths);
		if (!status)
		{
			Log ("*WARNING Ignoring selection item's %s skin cluster %s output geometry %d of type %s: cannot retrieve all paths: %s", pathDagNode.fullPathName ().asChar (), fnSkinCluster.name ().asChar (), nGeom, arrOutGeoms[nGeom].apiTypeStr (), status.errorString ().asChar ());
			continue;
		}
		
		for (unsigned nPath = 0; nPath < arrAllPaths.length (); ++nPath)
      m_setShapes.insert (arrAllPaths[nPath]);
	}
	return true;
}


// prepares the m_Bones utility object: pushes all bones influencing shapes here
void CCmdExportCgf::initBones ()
{
	for (MDagPathSet::const_iterator itShape = m_setShapes.begin(); itShape != m_setShapes.end(); ++itShape)
	{
		progressIncrement();
		m_Bones.addShapeInfluences (itShape->node ());
	}
	
	if (m_Bones.numRoots() > 1)
	{
		// for all roots, find their world skeleton roots and add
		MDagPathSet setRoots;
		for (unsigned i = 0; i < m_Bones.numRoots(); ++i)
			setRoots.insert (m_Bones.getRoot(i));
		m_Bones.addRootBone (getCommonParent (setRoots));
	}
}


//////////////////////////////////////////////////////////////////////////
// returns the 0-based index of the given material in the file,
// given already saved material chunk id
int CCmdExportCgf::getMaterialIndexByChunkId (int nChunkId)
{
	if (m_bIgnoreMaterials)
		return -1;

	if (nChunkId == -1)
		return -1;

	for (int i = (int)m_arrExportMtlMap.size()-1; i >= 0; --i)
		if (m_arrExportMtlMap[i].second == nChunkId)
			return i;

	return -1;
}


//////////////////////////////////////////////////////////////////////////
// returns the 0-based index of the given material in the file
int CCmdExportCgf::getMaterialIndex (const MObject& objShader)
{
	if (m_bIgnoreMaterials)
		return -1;

	int nChunkId = getMaterialChunkId(objShader);
	
	return getMaterialIndexByChunkId (nChunkId);;
}


//////////////////////////////////////////////////////////////////////////
// returns the given material chunk id;
// exports the material if it's met first time.
// returns -1 if the material cannot be exported
int CCmdExportCgf::getMaterialChunkId (const MObject& objShader)
{
	if (m_bIgnoreMaterials)
		return -1;

	if (objShader.isNull ())
		return -1;

	// try to find the shader among those that were tried to be exported
	for (ExportMtlMap::iterator it = m_arrExportMtlMap.begin(); it != m_arrExportMtlMap.end(); ++it)
		if (it->first == objShader)
			return it->second;

	try
	{
		return exportMaterial (objShader);
	}
	catch (MStatus&)
	{
		return -1;
	}
}


//////////////////////////////////////////////////////////////////////////
// Exports the multimaterial, which is a set of materials.
// Returns the given multimaterial chunk id.
int CCmdExportCgf::exportMultiMaterial (const MObjectSet& setMtls)
{
	if (m_bIgnoreMaterials)
		return -1;

	assert (!setMtls.empty ());
	
	if (setMtls.empty ())
		return -1;

	// make the descriptor, name the new material automatically
	// and assign it the id; set and export the number of children
	MTL_CHUNK_DESC_EXPORTER chunk;
	memset(&chunk,0,sizeof(chunk));
	_snprintf (chunk.name, sizeof(chunk.name), "multiMtl(%d)", m_arrExportMtlMap.size ());
	chunk.multi.nChildren = setMtls.size ();
	chunk.MtlType = MTL_MULTI;

	int nMultiMtlId = newChunkId();
	m_pFile->addChunk (ChunkType_Mtl, chunk.VERSION, nMultiMtlId);
	chunk.chdr = m_pFile->getChunk ();
	m_pFile->write (chunk);

	for (unsigned i = 0; i < setMtls.size(); ++i)
	{
		exportMaterial (setMtls[i]);
	}

	//m_arrExportMtlMap.push_back (ExportMtlEntry(MObject::kNullObj, nMultiMtlId));

	return nMultiMtlId;
}



//////////////////////////////////////////////////////////////////////////
// Exports the given shader as a standard material chunk
// returns this chunk's id
// throws an error if fails to write the chunk for some reason
unsigned CCmdExportCgf::exportMaterial (const MObject& objShader)
{
	if (m_bIgnoreMaterials)
		return -1;

	unsigned nMtlID = newChunkId();
	CMayaCryShaderUtil CryShader (objShader);

	MTL_CHUNK_DESC_EXPORTER desc;
	memset (&desc, 0, sizeof(desc));
	
	strncpy (desc.name, CryShader.getNameStr(), sizeof(desc.name));
	desc.MtlType   = MTL_STANDARD;
	
	desc.std.col_d         = CryShader.m_rgbDiffuse;
	desc.std.col_s         = CryShader.m_rgbSpecular;
	desc.std.col_a         = CryShader.m_rgbAmbient;
	desc.std.specLevel     = CryShader.m_fSpecLevel;
	desc.std.specShininess = CryShader.m_fSpecShininess;
	desc.std.selfIllum     = CryShader.m_fSelfIllum;
	desc.std.opacity       = CryShader.m_fOpacity;
	desc.std.Dyn_Bounce		 = CryShader.m_fDynBounce;		// 1.0=physicalized, otherwise not

	CryShader.getTextureInfo (desc.std.tex_a, ID_AM );
	CryShader.getTextureInfo (desc.std.tex_d, ID_DI );
	CryShader.getTextureInfo (desc.std.tex_s, ID_SP );
	CryShader.getTextureInfo (desc.std.tex_o, ID_OP );
	CryShader.getTextureInfo (desc.std.tex_b, ID_BU );
	CryShader.getTextureInfo (desc.std.tex_g, ID_SS );
	CryShader.getTextureInfo (desc.std.tex_fl, ID_FI );
	CryShader.getTextureInfo (desc.std.tex_rl, ID_RL );
	CryShader.getTextureInfo (desc.std.tex_subsurf, ID_RR );
	CryShader.getTextureInfo (desc.std.tex_det, ID_DP );

	convertTextureNames (desc);

	Log ("Exporting material %d: diffuse(%d,%d,%d), specular(%d,%d,%d), ambient(%d,%d,%d), self-illumination(%.2f), opacity(%.2f)",
		nMtlID,
		(int)desc.std.col_d.r, (int)desc.std.col_d.g, (int)desc.std.col_d.b,
		(int)desc.std.col_s.r, (int)desc.std.col_s.g, (int)desc.std.col_s.b,
		(int)desc.std.col_a.r, (int)desc.std.col_a.g, (int)desc.std.col_a.b,
		desc.std.selfIllum,
		desc.std.opacity);

	// now we don't have the risk of an exception, so we can write the chunk out
	m_pFile->addChunk (ChunkType_Mtl, MTL_CHUNK_DESC_EXPORTER::VERSION, nMtlID);
	desc.chdr = m_pFile->getChunk();
	m_pFile->write (desc);

	m_arrExportMtlMap.push_back (ExportMtlEntry(objShader, nMtlID));

	return nMtlID;
}


//////////////////////////////////////////////////////////////////////////
// converts texture names, if needed, to CryEngine format
void CCmdExportCgf::convertTextureNames (MTL_CHUNK_DESC_EXPORTER& desc)
{
	if (m_bConvertTGAtoDDS)
	{
		convertTextureTGAtoDDS (desc.std.tex_a.name, ID_AM );
		convertTextureTGAtoDDS (desc.std.tex_d.name, ID_DI );
		convertTextureTGAtoDDS (desc.std.tex_s.name, ID_SP );
		convertTextureTGAtoDDS (desc.std.tex_o.name, ID_OP );
		convertTextureTGAtoDDS (desc.std.tex_b.name, ID_BU );
		convertTextureTGAtoDDS (desc.std.tex_g.name, ID_SS );
		convertTextureTGAtoDDS (desc.std.tex_fl.name, ID_FI );
		convertTextureTGAtoDDS (desc.std.tex_rl.name, ID_RL );
		convertTextureTGAtoDDS (desc.std.tex_subsurf.name, ID_RR );
		convertTextureTGAtoDDS (desc.std.tex_det.name, ID_DP );
	}
} 


//////////////////////////////////////////////////////////////////////////
// converts the given texture name from tga to dds extension
void CCmdExportCgf::convertTextureTGAtoDDS (char* szName, int nMap)
{
	if (m_bIgnoreMaterials)
		return;

	int nLength = strlen(szName);

	if (nLength > 4)
	{
		if (!stricmp(szName + nLength - 4, ".tga"))
			strcpy (szName + nLength - 4, ".dds");
	}
}


//////////////////////////////////////////////////////////////////////////
// prepares and exports material chunks out of the given geometry
// modifies the geometry face material references so that they point to the actual material chunks
unsigned CCmdExportCgf::prepareMaterials (CMayaCryMeshUtil& Mesh)
{
	// remap the faces in the Mesh to the new ids of the materials

	unsigned numFaces = Mesh.numCryFaces (), nFace;
	CryFace* pFaces = Mesh.getCryFaces ();

	// set of used materials
	MObjectSet setMtls;
	for (nFace = 0; nFace < numFaces; ++nFace)
	{
		CryFace& rFace = pFaces[nFace];
		if (rFace.MatID >= 0 && rFace.MatID < Mesh.numMaterials())
			setMtls.insert(Mesh.getMaterialShader (rFace.MatID));
	}

	// export the multimaterial or standard material
	if (setMtls.empty())
	{
		for (nFace = 0; nFace < numFaces; ++nFace)
		{
			CryFace& rFace = pFaces[nFace];
			rFace.MatID = -1;
		}
		Log ("*WARNING* No valid material assignments found for mesh %s", Mesh.name ().asChar ());
		return -1;
	}
	else
	if (setMtls.size() == 1)
	{
		// single (standard) material, no need to make multimaterial
		MObject objMtl = setMtls[0];
		unsigned nMaterialChunkId = getMaterialChunkId (objMtl);
		unsigned nMaterialIndex = getMaterialIndexByChunkId (nMaterialChunkId);
		for (nFace = 0; nFace < numFaces; ++nFace)
		{
			CryFace& rFace = pFaces[nFace];
			rFace.MatID = nMaterialIndex;
		}
		MFnDependencyNode fnNode (objMtl);
		Log ("A single material assignment found for mesh '%s': %s '%s'", Mesh.name().asChar (), objMtl.apiTypeStr (), fnNode.name ().asChar ());
		return nMaterialChunkId;
	}
	else
	{
		// multimaterial
		unsigned nMultiMtlChunkId = exportMultiMaterial (setMtls);
		for (nFace = 0; nFace < numFaces; ++nFace)
		{
			CryFace& rFace = pFaces[nFace];
			if (rFace.MatID < 0 || rFace.MatID >= Mesh.numMaterials())
				rFace.MatID = -1;
			else
			{
				int nMtlChunkId = getMaterialChunkId(Mesh.getMaterialShader (rFace.MatID));
				if (nMtlChunkId >= 0)
					rFace.MatID = getMaterialIndexByChunkId(nMtlChunkId);
			}
		}

		if (m_bIgnoreMaterials)
			return -1;
		// Make a list of materials to be exported
		//
		std::string strMtlList;
		for (unsigned i = 0; i < setMtls.size(); ++i)
		{
			if (i) strMtlList += ", ";
			strMtlList += setMtls[i].apiTypeStr ();
			MStatus status;
			MFnDependencyNode fnNode (setMtls[i], &status);
			if (status)
				strMtlList = strMtlList + " '" + fnNode.name ().asChar () + "'";
		}
		Log ("%d material assignments found for mesh %s: {%s}. Exporting multimaterial.",
			setMtls.size (), Mesh.name().asChar (), strMtlList.c_str ());
		return nMultiMtlChunkId;
	}
}


//////////////////////////////////////////////////////////////////////////
// exports the helper chunk HELPER_CHUNK_DESC
// returns the chunk id for the helper
// returns -1 if the object has not been exported (for example, if it
// shouldn't be exported according to the user)
int CCmdExportCgf::exportHelper (const MDagPath& pathHelper)
{
	MStatus status;
	MObject objTM = pathHelper.transform(&status);
	if (!status)
	{
		std::cerr << "Could not retreive transform for node " << pathHelper.partialPathName().asChar() << "\n";
		throw status;
	}

	MFnTransform fnTM (objTM, &status);
	if (!status)
		throw status;
	double dScale[3];
	status = fnTM.getScale (dScale);
	if (!status)
		throw status;

	HELPER_CHUNK_DESC desc;
	desc.type = HP_DUMMY;
	for (int i = 0; i < 3; ++i)
		desc.size[i] = (float)dScale[i];

	if (pathHelper.node().hasFn(MFn::kCamera))
		desc.type = HP_CAMERA;

	// don't export pure transforms if the user doesn't want it
	if (desc.type == HP_DUMMY && m_bIgnoreDummies)
		return -1;

	m_pFile->addChunk (ChunkType_Helper, desc.VERSION, newChunkId());
	desc.chdr = m_pFile->getChunk();

	m_pFile->write (desc);
	return desc.chdr.ChunkID;
}



//////////////////////////////////////////////////////////////////////////
// exports the light chunk LIGHT_CHUNK_DESC
// returns the chunk id for the light
int CCmdExportCgf::exportLight (const MDagPath& pathLight)
{
	MStatus status;
	MFnLight fnLight (pathLight, &status);
	if (!status)
	{
		Log ("*WARNING* object %s is not a light (apiType=%s), ignoring the object", pathLight.fullPathName().asChar(), pathLight.node().apiTypeStr());
		return -1;
	}

	int nChunkId = newChunkId();
	m_pFile->addChunk (ChunkType_Light, LIGHT_CHUNK_DESC::VERSION, nChunkId);
	LIGHT_CHUNK_DESC desc;
	memset (&desc, 0, sizeof(desc));
	desc.chdr = m_pFile->getChunk();

	desc.on = getAttrValueBool (pathLight.node(), "emitDiffuse", false) || getAttrValueBool (pathLight.node(), "emitSpecular", false);

	desc.color = MayaToCryIRGB (fnLight.color());
	desc.intens = fnLight.intensity();
	desc.shadow = false;

	MMatrix mxTM = pathLight.inclusiveMatrix(&status);
	if (!status)
		throw status;

	convertMayaToCryCS(mxTM);

	// Get the light image
	//
	MEulerRotation erOrientation = MEulerRotation::decompose(mxTM, MEulerRotation::kXYZ);
	desc.vDirection.x = (float)erOrientation.x;
	desc.vDirection.y = (float)erOrientation.y;
	desc.vDirection.z = (float)erOrientation.z;

	// Retrive the light color texture
	//
	MObject objTexture = findConnected (pathLight.node(), "color", true);
	if (!objTexture.isNull())
	{
		MStatus status;
		MString mstrFileName = getAttrValueString (objTexture, "fileTextureName", &status);

		if (status && mstrFileName.length() > 0)
		{
			// we have a plug and the string
			std::string strPath = mstrFileName.asChar();
			FixupPath(strPath);
			if (!GetRelativePath(desc.szLightImage, sizeof(desc.szLightImage), strPath.c_str()))
			{
				std::cerr << "Cannot retrieve relative path from the light texture " << strPath.c_str() << "\n";
				desc.szLightImage[0] = '\0';
			}
			else
			{
				desc.color = MayaToCryIRGB (MColor(1,1,1));
			}
		}
	}

	switch (pathLight.node().apiType())
	{
	case MFn::kAmbientLight:
		desc.type = LT_AMBIENT;
		break;

	case MFn::kDirectionalLight:
		{
			MFnDirectionalLight fnDirectLight (pathLight, &status);
			if (!status)
				throw status;
			desc.type = LT_DIRECT;
			desc.hotsize = float(fnDirectLight.shadowAngle() * 180/M_PI);
			desc.fallsize = float(fnDirectLight.shadowAngle() * 180/M_PI);
			desc.shadow = fnDirectLight.useDepthMapShadows() || fnDirectLight.useRayTraceShadows();
		}
		break;

	case MFn::kSpotLight:
		{
			MFnSpotLight fnSpotLight (pathLight, &status);
			if (!status)
				throw status;
			desc.type = LT_SPOT;
			desc.hotsize  = (float)(fnSpotLight.coneAngle() * 180/M_PI);
			desc.fallsize = (float)((fnSpotLight.coneAngle() + fnSpotLight.penumbraAngle())*180/M_PI);

			desc.shadow = fnSpotLight.useDepthMapShadows() || fnSpotLight.useRayTraceShadows();
		}
		break;

	case MFn::kPointLight:
		{
			MFnPointLight fnPointLight (pathLight, &status);
			if (!status)
				throw status;
			desc.type = LT_OMNI;

			desc.shadow = fnPointLight.useDepthMapShadows() || fnPointLight.useRayTraceShadows();
		}
		break;

	default:
		desc.type = LT_AMBIENT;
		break;
	}
	m_pFile->write (desc);

	return nChunkId;
}


//////////////////////////////////////////////////////////////////////////
// prepares the array of SBoneLightBind for exporting, when the bones are filled in (m_Bones)
// and the light has assigned chunk ids
void CCmdExportCgf::prepareLightBindingInfo (const MDagPath& pathLight, unsigned nLightChunkId)
{
	if (!m_Bones.numBones())
		return; // no light binding if there's no bones

	MDagPath pathParentBone = pathLight;
	pathParentBone.pop();
	int nBoneId = m_Bones.getBoneIndex(pathParentBone);
	if (nBoneId < 0)
	{
		std::cerr << "Warning: light-bone binding will not be exported for the light " << pathLight.partialPathName().asChar() << "\n";
		return; // no binding because no bone
	}

	SBoneLightBind bind;
	bind.nLightChunkId = nLightChunkId;
	bind.nBoneId = nBoneId;

	MStatus status;
	MFnTransform fnTransform (pathLight.transform(&status), &status);
	MMatrix tmLight = fnTransform.transformationMatrix(&status);
	if (!status)throw status;
	convertMayaToCryCS (tmLight);

	// get rid of the skew transformation to calculate scale
	MMatrix tmLightH = tmLight.homogenize ();

	bind.vLightOffset.x = (float)tmLightH[3][0];
	bind.vLightOffset.y = (float)tmLightH[3][1];
	bind.vLightOffset.z = (float)tmLightH[3][2];

	MQuaternion qRot;
	qRot = tmLightH;
	MQuaternion qRotLog = qRot.log();

	bind.vRotLightOrientation.x = (float)qRotLog.x;
	bind.vRotLightOrientation.y = (float)qRotLog.y;
	bind.vRotLightOrientation.z = (float)qRotLog.z;
	m_arrLightBinding.push_back(bind);
}

// exports the ChunkType_BoneLightBinding chunk 
// returns the chunk id of the light binding chunk
int CCmdExportCgf::exportLightBinding()
{
	if (m_arrLightBinding.empty())
		return -1; // nothing to export
		
	BONELIGHTBINDING_CHUNK_DESC_0001 Desc;
	int nChunkId = newChunkId();
	m_pFile->addChunk(ChunkType_BoneLightBinding, Desc.VERSION, nChunkId);
	Desc.numBindings = m_arrLightBinding.size();
	m_pFile->write (Desc);
	m_pFile->writeVector(m_arrLightBinding);

	return nChunkId;
}


//////////////////////////////////////////////////////////////////////////
// exports the mesh chunk MESH_CHUNK_DESC and material(s)
// exports the bone information, if present
// returns the chunk id for the shape
int CCmdExportCgf::exportShape (const MDagPath& pathMesh)
{
	MStatus status;

	CMayaCryMeshUtil CryMesh (pathMesh);

	if (m_bDontAllowMultiUV)
	{
		if (CryMesh.numCryUVs() > 0 && CryMesh.numCryVerts() < CryMesh.numCryUVs())
		{
			std::cerr << "There are multiple UV coordinates per vertex. Please don't use \"dontAllowMultiUV\" command-line option or set the checkbox \"Allow Multi UV/vertex\"";
			throw MS::kFailure;
		}
	}

	if (!getNodeParity(pathMesh))
		CryMesh.flipCryFaces();

	unsigned nMainMtlID = prepareMaterials (CryMesh);
	m_mapNodeMtl[pathMesh] = nMainMtlID;

	int nChunkId = newChunkId();
	m_pFile->addChunk (ChunkType_Mesh, MESH_CHUNK_DESC::VERSION, nChunkId);

	MESH_CHUNK_DESC desc;
	desc.chdr = m_pFile->getChunk ();
	desc.HasBoneInfo   = m_hasBoneInfo && m_Bones.numRoots () > 0;
	desc.HasVertexCol  = m_bExportVertexColors && CryMesh.numCryVertColors() == CryMesh.numCryVerts();
	desc.nFaces        = CryMesh.numCryFaces();
	desc.nTVerts       = CryMesh.numCryUVs();
	desc.nVerts        = CryMesh.numCryVerts();
	desc.VertAnimID    = -1; // TODO: attach it to the vertex animation chunk id when the vertex animation is exported

	m_pFile->write (desc);

	Log ( "Writing vertices, faces for %s: %d vertices, %d faces, %d texture vertices",
		CryMesh.name().asChar (),CryMesh.numCryVerts(), CryMesh.numCryFaces(), CryMesh.numTVerts());

	// before export, convert the mesh vertices from Maya to Cry CS
	// we don't convert it to avoid double-converting with node matrix; needs more thinking about
	//for (i = 0; i < CryMesh.numCryVerts (); ++i)
	//	convertMayaToCryCS(CryMesh.getCryVerts ()[i]);

	m_pFile->writeArray (CryMesh.getCryVerts (), CryMesh.numCryVerts ());
	m_pFile->writeArray (CryMesh.getCryFaces (), CryMesh.numCryFaces ());
	m_pFile->writeArray (CryMesh.getCryUVs(), CryMesh.numCryUVs());
	m_pFile->writeArray (CryMesh.getCryTexFaces (), CryMesh.numCryTexFaces ());

	if (desc.HasBoneInfo)
	{// export the bones
		exportMeshBoneInfoSubchunk(pathMesh, CryMesh);
	}

	if (desc.HasVertexCol)
	{// export the vertex colors
		exportVertColInfoSubchunk (pathMesh, CryMesh);
	}

	return nChunkId;

}



//////////////////////////////////////////////////////////////////////////
// exports the subchunk containing link/weight information for all vertices in the geometry
// It consists of one entry per vertex ->
//    number of links
//    array of CryLinks
void CCmdExportCgf::exportMeshBoneInfoSubchunk (const MDagPath& pathShape, CMayaCryMeshUtil& CryMesh)
{
	MStatus status;
  MFnSkinCluster fnSkinCluster (findSkinCluster (pathShape), &status);
	if (!status)
	{
		std::cerr << "Shape " << pathShape.partialPathName ().asChar () << " does not have a skin cluster\n";
		throw status; // we must have a skin cluster to call this function
	}
  
	// plug index of the pathShape shape
	unsigned nShapePlugIndex = fnSkinCluster.indexForOutputShape (pathShape.node (), &status);
	if (!status)
	{
		status.perror ("indexForOutputShape failed.");
		throw status;
	}

	// iterate through the shape geometry to output the link weight info
	MItGeometry itShapeGeom (pathShape, &status);
	if (!status)
	{
		std::cerr << "MItGeometry didn't initialize.";
		throw status; //failed to iterate through the shape geometry
	}

	// get the influence object map: influence in the skin cluster -> bone in the exported bone hierarchy
	std::vector<int> arrBoneMap;
	MDagPathArray arrInfluences;
	unsigned numInfluences = fnSkinCluster.influenceObjects (arrInfluences, &status);
	assert (numInfluences == arrInfluences.length ());
	if (!status)
	{
		std::cerr << "influenceObjects failed.";
		throw status;
	}
	m_Bones.getBoneIndexMap (arrInfluences, arrBoneMap);

	// also, we will need the inverse transform of each joint
	// to export vertex offsets from each joint
	std::vector<MMatrix> arrTMInvJoints;
	arrTMInvJoints.resize (arrInfluences.length ());
	for (unsigned nInfluence = 0; nInfluence < arrInfluences.length (); ++nInfluence)
	{
		MDagPath& pathInfluence = arrInfluences[nInfluence];
		//MFnTransform fnJoint (pathInfluence, &status);
		//if (!status)
		//	throw status;

		arrTMInvJoints[nInfluence] = pathInfluence.inclusiveMatrixInverse (&status);
		
		if (!status)
		{
			std::cerr << "inclusiveMatrixInverse failed.";
			throw status;
		}
		else
		{
			//MMatrix& m = arrTMInvJoints[nInfluence];
			//Log0 ("'%s' inclusiveMatrixInverse:", pathInfluence.partialPathName ().asChar ());

			//for (int i= 0; i < 4; ++i)
			//	Log (" %-7.3f %-7.3f %-7.3f %-7.3f", m[i][0], m[i][1], m[i][2], m[i][3]);
		}
	}

	typedef std::vector<CryLink>VertBinding;
	typedef std::vector<VertBinding> MeshBinding;
	MeshBinding arrVertBindings;
	arrVertBindings.resize(itShapeGeom.count ());

	int nPointIdx;
	for (nPointIdx = 0; !itShapeGeom.isDone (); itShapeGeom.next (), ++nPointIdx)
	{
		assert (itShapeGeom.index () == nPointIdx);

		VertBinding& arrLinks = arrVertBindings[itShapeGeom.index ()];

		MObject objComponent = itShapeGeom.component ();
		// get the weights
		MFloatArray arrWeights;
		
		status = fnSkinCluster.getWeights (pathShape, objComponent, arrWeights, numInfluences);
		if (!status)
			throw status;

		// the geometry iterator must have selected only one CV, so we must have the weight array
		// of the same size as the number of influences (* 1 CV)
		assert (numInfluences == arrWeights.length ());
		assert (numInfluences == arrBoneMap.size());

		// export the influences that are big enough after normalization
		std::vector<unsigned> arrSignificantLinkIndices;
		float fTotalLinkWeight = getSignificantLinks (arrWeights, arrSignificantLinkIndices);

		if (fTotalLinkWeight > m_fMinLinkWeight)
		{
			// influencing links found, export a CryLink array

			// for each CryLink, we need the position of the point in the corresponding bone space
			MPoint ptVertex = itShapeGeom.position (MSpace::kWorld, &status);
			if (!status)
				throw status;

			arrLinks.resize (arrSignificantLinkIndices.size ());

			// iterate through significant links and memorize them
			for (unsigned nLinkIter = 0; nLinkIter < arrLinks.size(); ++nLinkIter)
			{
				// the link in the vertex blending array
				CryLink& cryLink = arrLinks[nLinkIter];
				// this is the index of the link in the array of influences
				unsigned nLinkIndex = arrSignificantLinkIndices[nLinkIter];

				cryLink.Blending = arrWeights[nLinkIndex] / fTotalLinkWeight;  // normalize the weights
				cryLink.BoneID   = arrBoneMap[nLinkIndex]; // map the bone id to the cry bone hierarchy bone id
				
				MPoint ptLocalOffset = ptVertex * arrTMInvJoints[nLinkIndex];
				for (int nCoord = 0; nCoord < 3; ++nCoord)
					cryLink.offset[nCoord] = (float)ptLocalOffset[nCoord];
				//convertMayaToCryCS (cryLink.offset);
			}
		}
		else
		{
			// no influencing links found, export a zero-CryLink array
			arrLinks.clear();
		}
	}

	// this time iterate through the cry points
	for (unsigned nCryVertex = 0; nCryVertex < CryMesh.numCryVerts (); ++nCryVertex)
	{
		unsigned nMayaVertex = CryMesh.getMayaVertByCryVert (nCryVertex);
		VertBinding& arrLinks = arrVertBindings[nMayaVertex];
		m_pFile->write ((unsigned int)arrLinks.size());
		m_pFile->writeArray (&arrLinks[0], arrLinks.size());
	}
}


// exports the subchunk containing colors for vertices
void CCmdExportCgf::exportVertColInfoSubchunk (const MDagPath& pathShape, CMayaCryMeshUtil& CryMesh)
{
	m_pFile->writeArray (CryMesh.getCryVertColors(), CryMesh.numCryVertColors ());
}



//////////////////////////////////////////////////////////////////////////
// used to sort an array of indices of link weights
struct CLinkWeightOrder
{
	CLinkWeightOrder (const MFloatArray& arrWeights):
		m_arrWeights (arrWeights)
		{
		}

	bool operator () (unsigned left, unsigned right)const
	{
		assert (left >= 0 && left < m_arrWeights.length());
		assert (right >= 0 && right < m_arrWeights.length());
		return m_arrWeights[left] > m_arrWeights[right];
	}

protected:
	const MFloatArray& m_arrWeights;
};



//////////////////////////////////////////////////////////////////////////
// puts the indices of significant link weights into the given array
// returns the calculated sum of the "significant" links
float CCmdExportCgf::getSignificantLinks  (const MFloatArray& arrWeights, std::vector<unsigned>& arrSignificantLinkIndices)
{
	float fSumResult = 0;
	arrSignificantLinkIndices.clear();
	for (unsigned i = 0; i < arrWeights.length (); ++i)
	{
		if (arrWeights[i] > m_fMinLinkWeight)
		{
			arrSignificantLinkIndices.push_back (i);
			fSumResult += arrWeights[i];
		}
	}

	// sort them by weights
	std::sort (arrSignificantLinkIndices.begin(), arrSignificantLinkIndices.end(), CLinkWeightOrder(arrWeights));

	if (arrSignificantLinkIndices.size() > m_nMaxWeightsPerLink)
		arrSignificantLinkIndices.resize (m_nMaxWeightsPerLink);

	return fSumResult;
}



//////////////////////////////////////////////////////////////////////////
// normalize and export the influences that are big enough
//void CCmdExportCgf::exportCryLinks (const MFloatArray& arrWeights, const std::vector<int>& arrBoneMap)
//{
//	assert (arrWeights.length () == arrBoneMap.size ());
//
//	// find the influences that are big enough
//	CryLink
//}


// exports all nodes collected during preparation phase
// nodes are exported in 2 passes: prepare and export. During export, the whole information 
// about all nodes is necessary, because nodes may refer to each other
void CCmdExportCgf::exportAllNodes()
{
	for (NodePreExportMap::const_iterator it = m_mapNodePreExportInfo.begin(); it != m_mapNodePreExportInfo.end(); ++it)
	{
		exportNode (it->first, it->second);
		progressIncrement();
	}
}

// preapres the node for export: adds it to the array of exoprted nodes
// nodes are exported in 2 passes: prepare and export. During export, the whole information 
// about all nodes is necessary, because nodes may refer to each other
void CCmdExportCgf::prepareNodeForExport (MDagPath pathNode, int nNodeObjectId)
{
	// we want to export the higher-level transform
	if (!pathNode.node().hasFn(MFn::kTransform))
		pathNode.pop();

	if (!pathNode.node().hasFn(MFn::kTransform))
	{
		std::cerr << "Cannot export node " << pathNode.partialPathName ().asChar () << "\n";
		return;
	}
	m_mapNodePreExportInfo.insert (NodePreExportMap::value_type(pathNode, NodePreExportInfo (nNodeObjectId, newChunkId())));
}

//////////////////////////////////////////////////////////////////////////
// exports the node into NODE_CHUNK_DESC_VERSION. Uses the info collected by the preparation phase of node export:
// the nNodeObjectId is the chunk id of the object contained in this node (already exported)
// nodes are exported in 2 passes: prepare and export. During export, the whole information 
// about all nodes is necessary, because nodes may refer to each other
void CCmdExportCgf::exportNode (MDagPath pathNode, const NodePreExportInfo& NodeInfo)
{
	MStatus status;
	MFnDagNode fnDagNode (pathNode, &status);
	if (!status)
	{
		std::cerr << "Cannot export node " << pathNode.partialPathName ().asChar () << "\n";
		return;
	}

	m_pFile->addChunk (ChunkType_Node, NODE_CHUNK_DESC::VERSION, NodeInfo.nChunkId);

	MString strProperties = getAttrValueString (pathNode.node(), "cryProperties");

	std::vector<int> arrChildren; // array of chunk ids of this node children
	getNodeChildrenChunkIds (pathNode, arrChildren);

	NODE_CHUNK_DESC desc;
	memset (&desc, 0, sizeof(desc));
	desc.chdr = m_pFile->getChunk ();
	
	desc.MatID         = m_mapNodeMtl[pathNode];
	desc.nChildren     = arrChildren.size();
	desc.ObjectID      = NodeInfo.nObjectId;
	desc.ParentID      = getNodeParentChunkId(pathNode);
	desc.IsGroupHead   = false;
	desc.IsGroupMember = false;
	desc.pos_cont_id   = -1;
	desc.rot_cont_id   = -1;
	desc.scl_cont_id   = -1;
	desc.PropStrLen    = strProperties.length();

	// construct the name of the node by the attribute "crySector"
	MStatus statusSector;
	int nSector = getAttrValueInt (pathNode.node(), "crySector", &statusSector);
	if (statusSector && nSector >= 0)
	{
		// there is such attribute
		_snprintf (desc.name, sizeof(desc.name), "$%d_%s", nSector, fnDagNode.name().asChar ());
	}
	else
		strncpy (desc.name, fnDagNode.name().asChar (), sizeof(desc.name));

	// this is the transform that'll be saved for this node
	MMatrix tmNode;
	// if there's no parent exported for this node, get the full matrix;
	// otherwise, get the matrix of this node (relatively to the parent)
	if (desc.ParentID == -1)
	{
		if (m_bExportIndividualFiles)
		// exclusion: when we export individual files, we want each object exported with 0 in its pivot
			tmNode.setToIdentity();
		else
			tmNode = pathNode.inclusiveMatrix (&status);
	}
	else
	{
		MFnTransform fnTransform (pathNode.transform(&status), &status);
		tmNode = fnTransform.transformationMatrix(&status);
	}
	if (!status)throw status;
	convertMayaToCryCS (tmNode);

	// get rid of the skew transformation to calculate scale
	MMatrix tmNodeH = tmNode.homogenize ();

	int i, j;

	for (i = 0; i < 4; ++i)
		for (j = 0; j < 4; ++j)
			desc.tm[i][j] = (float)tmNode[i][j];

	for (i = 0; i < 3; ++i)
		desc.pos[i] = desc.tm[3][i];

	MQuaternion qRot;
	qRot = tmNodeH;
	desc.rot.v.x = (float)qRot.x;
	desc.rot.v.y = (float)qRot.y;
	desc.rot.v.z = (float)qRot.z;
	desc.rot.w = (float)qRot.w;

	for (i = 0; i < 3; ++i)
		desc.scl[i] = (float)sqrt (tmNode[i][0]*tmNode[i][0] + tmNode[i][1]*tmNode[i][1] + tmNode[i][2]*tmNode[i][2]);

	m_pFile->write (desc);

	// write the properties
	if (desc.PropStrLen > 0)
		m_pFile->write (strProperties.asChar(), desc.PropStrLen);

	// write the children
	m_pFile->writeVector (arrChildren);
}


// finds the node parent, if any, or returns -1
int CCmdExportCgf::getNodeParentChunkId (const MDagPath& pathNode)
{
	MDagPath pathParent = pathNode;
	
	if (!pathParent.pop())
		return -1;

	if (!pathParent.node().hasFn(MFn::kTransform))
		if (!pathParent.pop())
			return -1;

	return getNodeChunkId (pathParent);
}

// finds the node children and puts them all into the array
void CCmdExportCgf::getNodeChildrenChunkIds (const MDagPath& pathNode, std::vector<int>& arrChildren)
{
	MStatus status;
	unsigned numChildren = pathNode.childCount(&status);

	Log0(" CCmdExportCgf::getNodeChildrenChunkIds(\"%s\"), %u children", pathNode.partialPathName().asChar(), numChildren);
	if (!status)
		return;

	LOG_AUTO_INTEND();
  for (unsigned nChild = 0; nChild < numChildren; ++nChild)
	{
		MObject objChild = pathNode.child(nChild);
		if (!objChild.hasFn(MFn::kTransform))
			continue; // we're not interested in non-transform noded, since only they comprise the hierarchy

		MDagPath pathChild = pathNode;
		if (!pathChild.push(objChild))
			return;

		int nChildChunkId = getNodeChunkId(pathChild);
		if (nChildChunkId != -1)
			arrChildren.push_back(nChildChunkId);

		Log0 ("%u. ChunkId: %d, node \"%s\"", nChild, nChildChunkId, pathChild.partialPathName().asChar());
	}
}


// finds the node chunk id
int CCmdExportCgf::getNodeChunkId (const MDagPath& pathNode)
{
	NodePreExportMap::const_iterator it = m_mapNodePreExportInfo.find (pathNode);
	if (it != m_mapNodePreExportInfo.end())
		return it->second.nChunkId;
	else
		return -1;
}


// exports the bone name list chunk, using the bone info collected in the m_Bones
void CCmdExportCgf::exportBoneNameList ()
{
	m_pFile->addChunk (
		ChunkType_BoneNameList,
		BONENAMELIST_CHUNK_DESC_0745::VERSION,
		newChunkId());

	BONENAMELIST_CHUNK_DESC_0745 desc;
	//desc.chdr      = m_pFile->getChunk();
	desc.numEntities = m_Bones.numBones();
	
	m_pFile->write (desc);
	// this is the length of the chunk contents;
	unsigned nLength = 0;

	for (int i = 0; i < desc.numEntities; ++i)
	{
		MFnDagNode fnBone (m_Bones.getBone(i).pathBone);

		/*
		if (fnBone.name().length () + 1 >= sizeof(BoneName.name))
		{
			std::cerr << "Bone " << fnBone.name().asChar () << " name is too long (should be up to " << (sizeof(BoneName.name) - 1) << " characters).\n";
			throw MStatus::kFailure;
		}
		*/

		MString strBoneName = fnBone.name();
		m_pFile->write (strBoneName.asChar(), strBoneName.length()+1);
		nLength += strBoneName.length()+1;
	}
	m_pFile->write ('\0');
	nLength ++;

	if (nLength & 3)
	{
		int nPad = 0;
		m_pFile->write(&nPad, 4-(nLength&3));
	}
}


// exports the bone animation chunk: miscellaneous information about the bones
void CCmdExportCgf::exportBoneAnim ()
{
	m_pFile->addChunk (
		ChunkType_BoneAnim,
		BONEANIM_CHUNK_DESC::VERSION,
		newChunkId());

	BONEANIM_CHUNK_DESC desc;
	desc.chdr   = m_pFile->getChunk();
	desc.nBones = m_Bones.numBones ();
	
	m_pFile->write (desc);

	std::set<unsigned> setCtrlId; // set of controller ids that must be unique

	Log0("Exporting %d bones", m_Bones.numBones ());
	for (int i = 0; i < m_Bones.numBones (); ++i)
	{
		const CMayaCryBoneUtil::BoneEntry& BoneInfo = m_Bones.getBone (i);
		MStatus status;
		MFnDagNode fnBone (BoneInfo.pathBone, &status);
		if (!status)
		{
			Log ("*ERROR* Cannot export bone %s: \"%s\" is not a DAG node", BoneInfo.pathBone.fullPathName().asChar(), BoneInfo.pathBone.node().apiTypeStr());
			status.perror ("Bone is not a DAG node");
			throw status;
		}
		BONE_ENTITY BoneEntity;
		memset (&BoneEntity, 0, sizeof(BoneEntity));
		BoneEntity.BoneID   = i;
		BoneEntity.ParentID = BoneInfo.nParentId;
		BoneEntity.ControllerID = Crc32Gen::GetCRC32 (fnBone.name().asChar ());
		if (setCtrlId.find (BoneEntity.ControllerID) != setCtrlId.end())
		{
			std::cerr << "Bone " << fnBone.name ().asChar () << " has generated a duplicate id; please change the bone name.\n";
			throw MStatus::kFailure;
		}
		setCtrlId.insert (BoneEntity.ControllerID);
		BoneEntity.nChildren = BoneInfo.numChildren;
		BoneEntity.prop[0] = 0;
		BoneEntity.phys.flags = -1;
		BoneEntity.phys.nPhysGeom = -1;// geometry for bones is not implemented
		
		MFnIkJoint fnJoint (BoneInfo.pathBone, &status);
		if (status)
		{
			// this is a joint - get the joint-specific flags and parameters
			int i;
			BONE_PHYSICS_COMP& phys = BoneEntity.phys;
			phys.flags = 0;

			// get the stiffness
			double dStiffness[3];
			status = fnJoint.getStiffness (dStiffness);
			if (!status)
			{
				std::cerr << "Unexpected error encountered: cannot get bone " << fnBone.name().asChar() << " stiffness.\n";
				throw status;
			}
			for (i = 0; i < 3; ++i)
				phys.spring_tension[i] = (float)dStiffness[i];

			bool bDof[3];
			fnJoint.getDegreesOfFreedom ( bDof[0], bDof[1], bDof[2] );
			for (i = 0; i < 3; ++i)
				if (bDof[i])phys.flags |= JNT_XACTIVE << i;

			phys.min[0] = (float)(fnJoint.isLimited(MFnTransform::kRotateMinX)?fnJoint.limitValue(MFnTransform::kRotateMinX) : -M_PI);
			phys.max[0] = (float)(fnJoint.isLimited(MFnTransform::kRotateMaxX)?fnJoint.limitValue(MFnTransform::kRotateMaxX) :  M_PI);
			phys.min[1] = (float)(fnJoint.isLimited(MFnTransform::kRotateMinY)?fnJoint.limitValue(MFnTransform::kRotateMinY) : -M_PI/2);
			phys.max[1] = (float)(fnJoint.isLimited(MFnTransform::kRotateMaxY)?fnJoint.limitValue(MFnTransform::kRotateMaxY) :  M_PI/2);
			phys.min[2] = (float)(fnJoint.isLimited(MFnTransform::kRotateMinZ)?fnJoint.limitValue(MFnTransform::kRotateMinZ) : -M_PI);
			phys.max[2] = (float)(fnJoint.isLimited(MFnTransform::kRotateMaxZ)?fnJoint.limitValue(MFnTransform::kRotateMaxZ) :  M_PI);

			if (fnJoint.isLimited(MFnTransform::kRotateMinX)||fnJoint.isLimited(MFnTransform::kRotateMaxX))
				phys.flags |= JNT_XLIMITED;
			if (fnJoint.isLimited(MFnTransform::kRotateMinY)||fnJoint.isLimited(MFnTransform::kRotateMaxY))
				phys.flags |= JNT_YLIMITED;
			if (fnJoint.isLimited(MFnTransform::kRotateMinZ)||fnJoint.isLimited(MFnTransform::kRotateMaxZ))
				phys.flags |= JNT_ZLIMITED;

			double dPrefAngle[3];
			fnJoint.getPreferedAngle (dPrefAngle);
			for (i = 0; i < 3; ++i)
				phys.spring_angle[i] = (float)dPrefAngle[i];

			phys.damping[0] = float(fnJoint.minRotateDampXStrength () + fnJoint.maxRotateDampXStrength ())/2;
			phys.damping[1] = float(fnJoint.minRotateDampYStrength () + fnJoint.maxRotateDampYStrength ())/2;
			phys.damping[2] = float(fnJoint.minRotateDampZStrength () + fnJoint.maxRotateDampZStrength ())/2;
	
			MMatrix mtxBone = fnBone.transformationMatrix ();
			for (i = 0; i < 3; ++i)
			{
				for (int j = 0; j < 3; ++j)
					phys.framemtx[i][j] = (float)mtxBone[i][j];
			}
		}

		Log0 ("Writing bone entity %s", fnBone.name().asChar());
		m_pFile->write (BoneEntity);
	}
}
