////////////////////////////////////////////////////////////////////////////
// Implementation of the base abstract class that incapsulates the functionality common to 
// All Cry file exoprter commands.
// This includes:
//   parsing the command arguments
//   convert coordinate systems
//   export common chunks
//   misc. utilities, like progress bar tracking, command execution, keeping the track of chunk ids etc.
//////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include "CmdExportCryFile.h"
#include "MayaCryUtils.h"

// flags parsed by the MSyntax of this class
const char
	// The path to the file to write. This option has one string parameter. Note: the backslashes must be escaped by backslashes.
	*CCmdExportCryFile::g_szFlagFile             = "-f",
	*CCmdExportCryFile::g_szFlagFileLong         = "-file",
	// If this option is present, it overrides the default animation start time.
	*CCmdExportCryFile::g_szManualRangeStart     = "-mrs",
	*CCmdExportCryFile::g_szManualRangeStartLong = "-manualRangeStart",
	*CCmdExportCryFile::g_szManualRangeEnd       = "-mre",
	*CCmdExportCryFile::g_szManualRangeEndLong   = "-manualRangeEnd",
	*CCmdExportCryFile::g_szFlagIgnoreBones      = "-ib",
	*CCmdExportCryFile::g_szFlagIgnoreBonesLong  = "-ignoreBones",
	*CCmdExportCryFile::g_szDontConvertCS        = "-ncs",
	*CCmdExportCryFile::g_szDontConvertCSLong    = "-dontConvertCS",
	*CCmdExportCryFile::g_szExportVertexColors   = "-vc",
	*CCmdExportCryFile::g_szExportVertexColorsLong = "-vertexColors",
	*CCmdExportCryFile::g_szConvertTGAtoDDS      = "-td", // Targa to Dds
	*CCmdExportCryFile::g_szConvertTGAtoDDSLong  = "-convertTgaToDds",
	// If this option is present, no helper objects (pure transforms, locators and cameras) are exported.
	*CCmdExportCryFile::g_szIgnoreDummies        = "-id", // Ignore Dummies
	*CCmdExportCryFile::g_szIgnoreDummiesLong    = "-ignoreDummies",
	// If this option is present, then the exporter outputs an error if there is no one-to-one correspondence between texture and coordinate vertices.
	*CCmdExportCryFile::g_szDontAllowMultiUV     = "-nmt", // No Multi Texture coordinates
	*CCmdExportCryFile::g_szDontAllowMultiUVLong = "-dontAllowMultiUV",
	// Sets the step of sampling the animation curves. The smaller the step is, the better the curves are sampled, but the bigger the animation file can potentially be.
	*CCmdExportCryFile::g_szAnimationStep        = "-as",
	*CCmdExportCryFile::g_szAnimationStepLong    = "-animationStep",
	// 
	*CCmdExportCryFile::g_szRotationPrecision    = "-rp",
	*CCmdExportCryFile::g_szRotationPrecisionLong= "-rotationPrecision",
	*CCmdExportCryFile::g_szPositionPrecision    = "-pp",
	*CCmdExportCryFile::g_szPositionPrecisionLong= "-positionPrecision",
	*CCmdExportCryFile::g_szDisableKeyOptimization     = "-dko",
	*CCmdExportCryFile::g_szDisableKeyOptimizationLong = "-disableKeyOptimization",
	*CCmdExportCryFile::g_szIgnoreMaterials      = "-imt",
	*CCmdExportCryFile::g_szIgnoreMaterialsLong  = "-ignoreMaterials",
	*CCmdExportCryFile::g_szMaxWeightsPerLink    = "-wpl",
	*CCmdExportCryFile::g_szMaxWeightsPerLinkLong= "-maxWeightsPerLink",
	*CCmdExportCryFile::g_szProgressBarName      = "-pb",
	*CCmdExportCryFile::g_szProgressBarNameLong  = "-progressBar",
	*CCmdExportCryFile::g_szExportIndividualFiles     = "-ifi",
	*CCmdExportCryFile::g_szExportIndividualFilesLong = "-exportIndividualFiles"
	;

// only the descendant may construct this object
CCmdExportCryFile::CCmdExportCryFile ():
	m_bManualRangeStart (false), m_bManualRangeEnd (false),
	m_bIgnoreBones (false),
	m_bDontConvertCS (false),
	m_bExportVertexColors (false),
	m_bConvertTGAtoDDS (false),
	m_bIgnoreDummies (false),
	m_bDontAllowMultiUV (false),
	m_bDisableKeyOptimization (false),
	m_bPositionPrecision (false),
	m_bRotationPrecision (false),
	m_dRotationPrecision (5),
	m_dPositionPrecision (3.5),
	m_nMaxWeightsPerLink (0xFFFF),
	m_bIgnoreMaterials (false),
	m_nIsProgressBarMain(0),
	m_numChunkIdsGenerated (0),
	m_bExportIndividualFiles (false)
{
}

//////////////////////////////////////////////////////////////////////////
// constructs a syntax object
MSyntax CCmdExportCryFile::newSyntax()
{
	MSyntax syntax;
	syntax.addFlag (g_szFlagFile, g_szFlagFileLong, MSyntax::kString);
	syntax.addFlag (g_szManualRangeStart, g_szManualRangeStartLong, MSyntax::kTime);
	syntax.addFlag (g_szManualRangeEnd, g_szManualRangeEndLong, MSyntax::kTime);
	syntax.addFlag (g_szFlagIgnoreBones, g_szFlagIgnoreBonesLong);
	syntax.addFlag (g_szExportVertexColors, g_szExportVertexColorsLong);
	syntax.addFlag (g_szDontConvertCS, g_szDontConvertCSLong);
	syntax.addFlag (g_szConvertTGAtoDDS, g_szConvertTGAtoDDSLong);
	syntax.addFlag (g_szIgnoreDummies, g_szIgnoreDummiesLong);
	syntax.addFlag (g_szDontAllowMultiUV, g_szDontAllowMultiUVLong);
	syntax.addFlag (g_szAnimationStep, g_szAnimationStepLong, MSyntax::kTime);
	syntax.addFlag (g_szRotationPrecision, g_szRotationPrecisionLong, MSyntax::kDouble);
	syntax.addFlag (g_szPositionPrecision, g_szPositionPrecisionLong, MSyntax::kDouble);
	syntax.addFlag (g_szDisableKeyOptimization, g_szDisableKeyOptimizationLong);
	syntax.addFlag (g_szIgnoreMaterials, g_szIgnoreMaterialsLong);
	syntax.addFlag (g_szMaxWeightsPerLink, g_szMaxWeightsPerLinkLong, MSyntax::kLong);
	syntax.addFlag (g_szProgressBarName, g_szProgressBarNameLong, MSyntax::kString);
	syntax.addFlag (g_szExportIndividualFiles, g_szExportIndividualFilesLong);
	syntax.setObjectType(MSyntax::kSelectionList);
	syntax.useSelectionAsDefault (true);
	return syntax;
}

//////////////////////////////////////////////////////////////////////////
// parses the common arguments passed to the command (file name)
MStatus CCmdExportCryFile::parseArgs ( const MArgList& args)
{
	MStatus status = MS::kSuccess;
	
	MArgDatabase argDB(syntax(), args, &status);
	if (!status)
		return status;

	status = argDB.getFlagArgument (g_szFlagFile, 0, m_strFileName);

	double nFrame = 0;
	m_bManualRangeStart = MStatus::kSuccess == argDB.getFlagArgument (g_szManualRangeStart, 0, m_tManualRangeStart);
	m_bManualRangeEnd   = MStatus::kSuccess == argDB.getFlagArgument (g_szManualRangeEnd  , 0, m_tManualRangeEnd);

	if (m_bManualRangeStart && m_bManualRangeEnd)
		Log ("Manual range: [%g..%g] sec [%.1f..%.1f] frames",
			m_tManualRangeStart.as(MTime::kSeconds), m_tManualRangeEnd.as(MTime::kSeconds),
			m_tManualRangeStart.as(MTime::uiUnit ()), m_tManualRangeEnd.as(MTime::uiUnit ())
			);

	if (m_bManualRangeStart && m_bManualRangeEnd && (m_tManualRangeEnd < m_tManualRangeStart))
	{
		status.perror ("manual range start must be less or equal to the manual range end");
		return status;
	}

	if (argDB.isFlagSet (g_szFlagIgnoreBones))
		m_bIgnoreBones = true;

	if (argDB.isFlagSet(g_szExportIndividualFiles))
		m_bExportIndividualFiles = true;

	if (argDB.isFlagSet (g_szDontConvertCS))
		m_bDontConvertCS = true;

	if (argDB.isFlagSet(g_szIgnoreDummies))
		m_bIgnoreDummies = true;

	if (argDB.isFlagSet (g_szExportVertexColors))
		m_bExportVertexColors = true;

	if (argDB.isFlagSet (g_szConvertTGAtoDDS))
		m_bConvertTGAtoDDS = true;

	if (argDB.isFlagSet(g_szDontAllowMultiUV))
		m_bDontAllowMultiUV = true;

	if (argDB.isFlagSet(g_szIgnoreMaterials))
		m_bIgnoreMaterials = true;

	m_bAnimationStep = MStatus::kSuccess == argDB.getFlagArgument(g_szAnimationStep, 0, m_tAnimationStep);

	m_bRotationPrecision = MStatus::kSuccess == argDB.getFlagArgument(g_szRotationPrecision, 0, m_dRotationPrecision);
	m_bPositionPrecision = MStatus::kSuccess == argDB.getFlagArgument(g_szPositionPrecision, 0, m_dPositionPrecision);

	unsigned uMaxWeightsPerLink;
	if (MStatus::kSuccess == argDB.getFlagArgument(g_szMaxWeightsPerLink, 0, uMaxWeightsPerLink))
		m_nMaxWeightsPerLink = uMaxWeightsPerLink;

	if (argDB.isFlagSet(g_szDisableKeyOptimization))
	{
		std::cerr << "Key optimization disabled. The resulting file may be very large.\n";
		m_bDisableKeyOptimization = true;
	}

	if (!argDB.getFlagArgument (g_szProgressBarName, 0, m_strProgressBar))
		m_strProgressBar = "";

	if (m_bConvertTGAtoDDS)
		Log ("convert tga->dds flag is on");

	if (argDB.getObjects(m_lstSelection))
		Log ("exporting %d item(s)", m_lstSelection.length());

	return status;
}

//////////////////////////////////////////////////////////////////////////
// exports the timing chunk (scene start/end times etc.)
void CCmdExportCryFile::exportTiming()
{
	m_pFile->addChunk (
		ChunkType_Timing,
		TIMING_CHUNK_DESC::VERSION,
		newChunkId()
		);

	TIMING_CHUNK_DESC desc;
	desc.chdr = m_pFile->getChunk();
	desc.global_range.start	= MayaToCryTime (m_bManualRangeStart? m_tManualRangeStart : MAnimControl::minTime ()) / 160;
	desc.global_range.end		= (MayaToCryTime (m_bManualRangeEnd? m_tManualRangeEnd : MAnimControl::maxTime ()) +159)/ 160;
	desc.SecsPerTick = 1.0f/4800;
	desc.TicksPerFrame = 160; // assuming NTSC 30fps
	strcpy(desc.global_range.name,"GlobalRange");
	desc.nSubRanges = 0;

	m_pFile->write (desc);

	Log ("   Info: exported time range [%d..%d] ticks", desc.global_range.start, desc.global_range.end);
}


template <typename T>
void mayaToCryCSv (T* pt)
{
	std::swap (pt[1],pt[2]);
	//pt[0] = -pt[0];
	pt[1] = -pt[1];
}


// applies fixed rotation to the point, to convert from the Maya coordinate system (y up) to Cry coordinate system (z up)
void CCmdExportCryFile::convertMayaToCryCS (Vec3d& pt)
{
	if (!m_bDontConvertCS)
	{
		// (old)y -> (new)z, (old)z -> (new) -y
		//std::swap (pt.z, pt.y);
		//pt.y = -pt.y;
		mayaToCryCSv (&pt[0]);
	}
}

// applies fixed rotation to the matrix, to convert from the Maya coordinate system (y up) to Cry coordinate system (z up)
void CCmdExportCryFile::convertMayaToCryCS (MMatrix& mx)
{
	if (!m_bDontConvertCS)
	for (unsigned i = 0; i < 4; ++i)
	{
		//std::swap (mx[i][1],mx[i][2]);
		//mx[i][1] = -mx[i][1];
		mayaToCryCSv (mx[i]);
	}
}


// applies fixed rotation to the point and normal, to convert from the Maya coordinate system (y up) to Cry coordinate system (z up)
void CCmdExportCryFile::convertMayaToCryCS (CryVertex& cryVtx)
{
	convertMayaToCryCS (cryVtx.p);
	convertMayaToCryCS (cryVtx.n);
}

// applies a fixed rotation to the position and orientation, to convert from the Maya coordinate system (y up) to Cry coordinate system (z up)
void CCmdExportCryFile::convertMayaToCryCS (CryBoneKey& cryBoneKey)
{
	convertMayaToCryCS (cryBoneKey.abspos);
	convertMayaToCryCS (cryBoneKey.relpos);
	convertMayaToCryCS (cryBoneKey.relquat);
}


void CCmdExportCryFile::convertMayaToCryCS (CryQuat& q)
{
	if (!m_bDontConvertCS)
	{
		const double dSqrt2 = 1.4142135623730950488016887242097;
		CryQuat qRes;
		qRes.w = float (( q.w-q.v.x)/dSqrt2);
		qRes.v.x = float (( q.w+q.v.x)/dSqrt2);
		qRes.v.y = float (( q.v.y-q.v.z)/dSqrt2);
		qRes.v.z = float (( q.v.y+q.v.z)/dSqrt2);
		q = qRes;
	}
}


// Begins progress indication, if there is a progress bar assigned
void CCmdExportCryFile::progressStart (const char* szTitle, unsigned nTotalSteps)
{
	if (m_strProgressBar != "")
	{
		char szCmd[0x400];
		sprintf (szCmd, "progressBar -query -isMainProgressBar \"%s\"", m_strProgressBar.asChar());
		if (!MGlobal::executeCommand(szCmd, m_nIsProgressBarMain))
		{
			Log ("*ERROR* Cannot determine the origin of the progress bar %s", m_strProgressBar.asChar());
			m_strProgressBar = "";
			return;
		}
		if (m_nIsProgressBarMain)
			sprintf (szCmd, "progressBar -edit -beginProgress -isInterruptable true -status \"%s\" -maxValue %u \"%s\"",
				szTitle, nTotalSteps, m_strProgressBar.asChar());
		else
			sprintf (szCmd, "progressBar -edit -progress 0 -annotation \"%s\" -maxValue %u \"%s\"",
				szTitle, nTotalSteps, m_strProgressBar.asChar());
		executeCommand(szCmd);
	}
}


// Increments progress indication, if there is a progress bar assigned
void CCmdExportCryFile::progressIncrement(unsigned nStep)
{
	if (m_strProgressBar != "")
	{
		char szCmd[0x400];
		sprintf (szCmd, "progressBar -edit -step %u \"%s\"", nStep, m_strProgressBar.asChar());
		executeCommand(szCmd);
	}
}


// Stops progress indication, if there is a progress bar assigned
void CCmdExportCryFile::progressEnd()
{
	if (m_strProgressBar != "")
	{
		char szCmd[0x400];
		if (m_nIsProgressBarMain)
			sprintf (szCmd, "progressBar -edit -endProgress \"%s\"", m_strProgressBar.asChar());
		else
			sprintf (szCmd, "progressBar -edit -progress 0 \"%s\"", m_strProgressBar.asChar());
		executeCommand(szCmd);
	}
}

// executes a MEL command
void CCmdExportCryFile::executeCommand (const char* szCommand)
{
	//Log0 ("Executing: %s", szCommand);
	MGlobal::executeCommand (szCommand);
}

void CCmdExportCryFile::progressThrowOnCancel()
{
	if (m_strProgressBar == "")
		return;

	char szCmd[0x400];
	sprintf (szCmd, "progressBar -query -isCancelled \"%s\"", m_strProgressBar.asChar());
	int nCancelled;
	MGlobal::executeCommand(szCmd, nCancelled);
	if (nCancelled)
	{
		MStatus status (MS::kFailure);
		status.perror("User abort");
	}
}

// generates a unique chunk id
int CCmdExportCryFile::newChunkId()
{
	return m_numChunkIdsGenerated++;
}
