#include "stdafx.h"
#include "MayaUtils.h"
#include "MayaCryUtils.h"
#include "MayaCryKeyUtil.h"
#include "CryKeyInterpolation.h"

// retrieves the animation from the given transform (perhaps not animated) node
CMayaCryKeyUtil::CMayaCryKeyUtil (const MDagPath& pathTransform):
	m_pathTransform(pathTransform),
	m_bDisableOptimization (false),
	m_fPosError(1e-6f), m_fQuatError(0.9999f)
{
	setMinFrameStep (MTime (1, MTime::uiUnit ()));
	setRangeStart (MAnimControl::minTime ());
	setRangeEnd   (MAnimControl::maxTime ());
}

// optimizes the cry keys
void CMayaCryKeyUtil::optimizeAnimCryKeys ()
{
	// optimize the array by deleting unnecessary keys
	m_arrAnimCryKeys.resize (
		OptimizeKeys (&m_arrAnimCryKeys[0], (unsigned)m_arrAnimCryKeys.size(), m_fPosError, m_fQuatError)
		);
}


// animation start, in native CryAnimation integer units (4800 ticks per second)
//int CMayaCryKeyUtil::getAnimStart ()
//{
//	return MayaToCryTime(m_tAnimStart);
//}

// animation end, in native CryAnimation integer units (4800 ticks per second)
//int CMayaCryKeyUtil::getAnimEnd ()
//{
//	return MayaToCryTime(m_tAnimEnd);
//}

// returns the number of CryBoneKey elements in the array of keys
int CMayaCryKeyUtil::numCryKeys()
{
	return (int)m_arrAnimCryKeys.size();
}

// returns the pointer to the array of cry bone keys
CryBoneKey* CMayaCryKeyUtil::getCryKeys()
{
	return &m_arrAnimCryKeys[0];
}

// makes sure the animation keys have been fetched and optimized
void CMayaCryKeyUtil::prepareAnimationCache()
{
	if (m_arrAnimCryKeys.empty ())
	{	// retrieve the animation from Maya
		initAnimation ();	
		// optimize the animation
		if (!m_bDisableOptimization)
			optimizeAnimCryKeys();
	}
}


//////////////////////////////////////////////////////////////////////////
// retrieves the animation from the given dag path into start/end
// times and array of cry keys (unoptimized)
void CMayaCryKeyUtil::initAnimation()
{
	// the animated plugs
	MPlugArray arrPlugs;
	// retrieve the animated plugs
	MStatus status;
	bool bAnimated = MAnimUtil::findAnimatedPlugs (m_pathTransform, arrPlugs, false, &status);
	if (!status)
	{
		Log ("*ERROR* CMayaCryKeyUtil: cannot retrieve animated plugs on the node %s: %s", m_pathTransform.fullPathName().asChar(), status.errorString().asChar());
		throw (status);
	}

	// set the most basic MFn - transofrm, which will be used for most operations
	status = m_fnTM.setObject (m_pathTransform);
	if (!status)
	{
		Log ("*ERROR* supplied node %s is not a transform but a %s", m_pathTransform.fullPathName ().asChar (), m_pathTransform.node ().apiTypeStr ());
		throw status;
	}

	// retrieve the joint MFn, which, in the case of success, will be used
	// to determine the pre- and postrotational quaternions of the joint
	MStatus statusFnIkJoint;
	m_statusFnIkJoint = m_fnJoint.setObject (m_pathTransform);

	if (!bAnimated)
		initAnimationStatic ();
	else
	{
		initNodeAnimAttrs();
		convertAnimNodeToCryKeys();
	} // animated node

	// The actual CGF file contains inverse quaternions of hte movement.
	// This function makes all quaternions inverse, which is needed for the file.
	//negateCryKeyRotations ();

	// this is a necessary step, to avoid 180+ degree rotations in-between the keyframes
	// if such rotations are desirable, the artist should insert more keyframes in the middle
	// or export with a denser framerate
	cleanupCryKeyRotations ();
} // initAnimation()


// negate the quaternions in Cry keys
void CMayaCryKeyUtil::negateCryKeyRotations()
{
	CryBoneKeyArray::iterator it;
	for (it = m_arrAnimCryKeys.begin(); it != m_arrAnimCryKeys.end(); ++it)
	{
		CryBoneKey& key = *it;
		key.relquat.w = -key.relquat.w;
	}
}


// cleanup 180+ degree rotations because of quaternion representation parity:
// make all neighbor quaternions have positive dot product
void CMayaCryKeyUtil::cleanupCryKeyRotations()
{
	CryQuat qPrev (1,0,0,0); // must be identity at the beginning
	
	CryBoneKeyArray::iterator it;
	for (it = m_arrAnimCryKeys.begin(); it != m_arrAnimCryKeys.end(); ++it)
	{
		CryQuat& qThis = it->relquat;

		if ((qThis|qPrev) < 0)
			qThis = -qThis;

		qPrev = qThis;
	}
}


// retrieves the animated plug nodes: initializes m_arrNodeAnimAttr
void CMayaCryKeyUtil::initNodeAnimAttrs ()
{
	Log0 ("  Exporting %s", m_pathTransform.fullPathName().asChar());

	MStatus status;
	unsigned i;

	for (i = 0; i < kAA_Count; ++i)
		m_arrNodeAnimAttr[i] = MObject::kNullObj;

	MPlugArray arrPlugs;
	MAnimUtil::findAnimatedPlugs(m_pathTransform, arrPlugs, false, &status);
	if (!status) throw (status);
	Log0 ("%d animated plugs found", arrPlugs.length ());
	// we've found some plugs
	for (i = 0; i < arrPlugs.length(); ++i)
	{
		MPlug& plug = arrPlugs[i];
		Log0 ("Examining %s", plug.name ().asChar ());
		addAnimAttribute (plug);
	} // scan plugs
}

// uses the given animated attribute to initialize one or more anim attr elements in the animation array
void CMayaCryKeyUtil::addAnimAttribute (const MPlug& plug)
{
	MString strAttrName;
	MStatus status;
	MFnAttribute fnAttribute (plug.attribute(&status), &status);
	if (!status)
	{
		Log ("*ERROR* CMayaCryKeyUtil: cannot retrieve attribute name for plug %s", plug.info().asChar());
		return;
	}
	else
		strAttrName = fnAttribute.name();

	if (strAttrName == "rotateX")
		addAnimCurve (kAARotateX, plug);
	else
	if (strAttrName == "rotateY")
		addAnimCurve (kAARotateY, plug);
	else
	if (strAttrName == "rotateZ")
		addAnimCurve (kAARotateZ, plug);
	else
	if (strAttrName == "translateX")
		addAnimCurve (kAATranslateX, plug);
	else
	if (strAttrName == "translateY")
		addAnimCurve (kAATranslateY, plug);
	else
	if (strAttrName == "translateZ")
		addAnimCurve (kAATranslateZ, plug);
	else
	{
		// this is not a simple attribute, maybe it's compound
		if (strAttrName == "translate")
			addAnimMulticurve (kAATranslate, plug);
		else
		if (strAttrName == "rotate")
			addAnimMulticurve (kAARotate, plug);
	}
}

// adds a compound attribute: breaks it into children and adds them curve-by-curve
void CMayaCryKeyUtil::addAnimMulticurve (AnimAttributeEnum nAnimAttr, const MPlug& plug)
{
	MStatus status;
	MFnCompoundAttribute fnAttribute (plug.attribute(&status), &status);
	if (!status)
	{
		Log ("Cannot get multicurve compound attribute fnset: %s", status.errorString ().asChar ());
		return;
	}

	unsigned numChildren = fnAttribute.numChildren ();
	Log0("Compound attribute %s has %d children", fnAttribute.name ().asChar (), numChildren);
	for (unsigned i =0; i < numChildren; ++i)
	{
		MObject objAttrChild = fnAttribute.child (i);
		Log0 (" %s", objAttrChild.apiTypeStr ());
	}
}


// adds the animation curve
void CMayaCryKeyUtil::addAnimCurve (AnimAttributeEnum nAnimAttr, const MPlug& plug)
{
	// find the animation curve(s)
	MObjectArray arrAnimCurves;
	if (!MAnimUtil::findAnimation (plug, arrAnimCurves))
	{
		Log ("    Plug %s doesn't have an animation curve", plug.info().asChar());
		return;
	}

	unsigned int numCurves = arrAnimCurves.length();
	if (numCurves != 1)
		Log ("   Plug %s has %d animation curve(s)", plug.info().asChar(), numCurves);

	if (numCurves > 0)
		addAnimCurve (nAnimAttr, arrAnimCurves[0]);
}


// adds the animation curve
void CMayaCryKeyUtil::addAnimCurve (AnimAttributeEnum nAnimAttr, const MObject& objCurve)
{
	m_arrNodeAnimAttr[nAnimAttr] = objCurve;
	MStatus status = m_arrFnAnimAttr[nAnimAttr].setObject (objCurve);
	if (!status)
	{
		Log ("*WARNING* couldn't attach an object to the animated curve function set: %s", status.errorString().asChar());
		m_arrNodeAnimAttr[nAnimAttr] = MObject::kNullObj;
	}
}


// converts the animation contained in the array of nodes to the CryBoneKey array
void CMayaCryKeyUtil::convertAnimNodeToCryKeys ()
{
	MStatus status;
	// for both the rotational and positional component of the motion, retrieve the keys
	std::set<int> setKeys; // in the native Cry ticks
	addAnimNodeTicks(setKeys);

	// we have to have at least one keyframe; otherwise we're having a static motion
	if (setKeys.empty())
	{
		Log ("*WARNING* no keys found, resorting to static export");
		initAnimationStatic();
		return;
	}

	// now, for each key, retrieve its values and pack the cry bone key
	m_arrAnimCryKeys.reserve(setKeys.size());

	Log0 ("Converting to Cry Keys: %s (IkJoint: %s)", m_fnTM.name ().asChar (), m_statusFnIkJoint?"YES":"NO");

	// the default rotation order
	MTransformationMatrix::RotationOrder nRotOrd = m_fnTM.rotationOrder();
	MEulerRotation::RotationOrder nEulOrd = ToEulOrd (nRotOrd);
	Log0 ("Rotation Order: %s", getRotOrdStr (nRotOrd));

	// default rotation components (will go instead of missing animation curves)
	double dDefRotation[3];
	status = m_fnTM.getRotation(dDefRotation, nRotOrd);
	if (!status) throw status;
	Log0 ("Default rotation: (%.1f,%.1f,%.1f)", dDefRotation[0]*180/M_PI,dDefRotation[1]*180/M_PI,dDefRotation[2]*180/M_PI);

	// default translation components (will go instead of missing animation curves)
	MVector vmDefTranslation = m_fnTM.translation (MSpace::kPostTransform, &status);
	if (!status) throw status;
	Log0 ("Default translation: (%.2f,%.2f,%.2f)", vmDefTranslation [0],vmDefTranslation [1],vmDefTranslation [2]);

	for (std::set<int>::const_iterator it = setKeys.begin(); it != setKeys.end();)
	{
		// make sure the minimal step between the keyframes is achieved
		// find the current and the next tick and scavenge between them
		int nTickStart = *it, nTickEnd;
		
		if (++it == setKeys.end())
			nTickEnd = nTickStart+1; // no further keyframes
		else
			nTickEnd = *it;

		for (int nTick = nTickStart; nTick < nTickEnd; nTick = m_nMinFrameStepTicks ? nTick + m_nMinFrameStepTicks : nTickEnd)
		{
			// prepare the default transformation
			double dRotation[3] = { dDefRotation[0], dDefRotation[1], dDefRotation[2] };
			MVector vmTranslation = vmDefTranslation;

			// retrieve the animated transformation
			getTransform (nTick, dRotation, vmTranslation);

			// convert the transformation to the CryBoneKey
			MQuaternion qmRotation, qmJointOrient, qmRotateOrient;
			// the basic animated rotation
			qmRotation = MEulerRotation (dRotation[0],dRotation[1],dRotation[2],nEulOrd);
			fixupJointRotation (nTick, qmRotation);

			CryBoneKey key = ToCryBoneKey(nTick, vmTranslation, qmRotation);

			m_arrAnimCryKeys.push_back(key);
		}
	}
}

//////////////////////////////////////////////////////////////////////////
// fixes up the joint rotation matrix: forms the full rotation out of the 
// [R] matrix: sets the given [R]  to  [JO]*[R]*[RO]
void CMayaCryKeyUtil::fixupJointRotation (int nTick, MQuaternion& qmRotation)
{
	if (!m_statusFnIkJoint)
		return; // not a joint - nothing to fixup

	MQuaternion qmJointOrient, qmRotateOrient;
	// the joint orientation (pre-rotation)
	MStatus status = m_fnJoint.getOrientation (qmJointOrient);
	if (!status)
	{
		std::cerr << "Cannot retrieve the IkJoint joint orientation [JO] ";
		throw status;
	}
	//the rotate orientation (post-rotation)
	status = m_fnJoint.getScaleOrientation (qmRotateOrient);
	if (!status)
	{
		std::cerr << "Cannot retrieve the IkJoint rotate orientation [RO] ";
		throw status;
	}

#ifdef _DEBUG
	if (false)
	{
		MEulerRotation jo, r, ro;
		jo = qmJointOrient;
		r  = qmRotation;
		ro = qmRotateOrient;
		Log0 ("  % 5d: [JO](%.1f,%.1f,%.1f) * [R](%.1f,%.1f,%.1f) * [RO](%.1f,%.1f,%.1f)",
			nTick,
			jo.x*180/M_PI, jo.y*180/M_PI, jo.z*180/M_PI,
			r.x*180/M_PI,  r.y*180/M_PI,  r.z*180/M_PI,
			ro.x*180/M_PI, ro.y*180/M_PI, ro.z*180/M_PI
			);
	}
#endif
	// this is the reverse order, actually. I don't know why.
	qmRotation = qmRotateOrient * qmRotation * qmJointOrient;
}


//////////////////////////////////////////////////////////////////////////
// adds the animation node key times (in ticks) to the given set
void CMayaCryKeyUtil::addAnimNodeTicks (std::set<int>& setTicks)
{
	MStatus status;
	for (unsigned i = 0; i < kAA_Count; ++i)
		if (m_arrNodeAnimAttr[i] != MObject::kNullObj)
		{
			MFnAnimCurve fnCurve (m_arrNodeAnimAttr[i], &status);
			if (!status) throw status;

			unsigned numKeys = fnCurve.numKeys();
			Log ("Adding animation keys from \"%s\" (%d keys)%s", fnCurve.name ().asChar (), numKeys, fnCurve.isUnitlessInput()?" UNITLESS ":"");
			for (unsigned nKey = 0; nKey < numKeys; ++nKey)
			{
				// calculate the tick of the current key
				int nTick;
				if (fnCurve.isUnitlessInput())
					nTick = MayaToCryTime (fnCurve.unitlessInput (nKey));
				else
					nTick = MayaToCryTime (fnCurve.time (nKey));

				// clamp, if needed, by the start and the end of the manual range
				if (nTick < m_nRangeStart)
					nTick = m_nRangeStart;

				if (nTick > m_nRangeEnd)
					nTick = m_nRangeEnd;

				setTicks.insert (nTick);
			}
		}
}

// retrieves the animation if it's static (just retrieves the current position/rotation of the node)
void CMayaCryKeyUtil::initAnimationStatic ()
{
	Log0 ("  Exporting static %s", m_pathTransform.fullPathName().asChar());

	MStatus status;
	MQuaternion qmRotation;
	status = m_fnTM.getRotation(qmRotation, MSpace::kTransform);
	if (status.error())
		throw status;
	fixupJointRotation (0, qmRotation);

	MVector vmTranslation = m_fnTM.translation (MSpace::kTransform, &status);
	if (!status) throw status;

	CryBoneKey key = ToCryBoneKey (0, vmTranslation, qmRotation);

	m_arrAnimCryKeys.push_back(key);
}


// retrieves the transform of the node at the given time.
// fills in the components of the transform (rotation and position)
// which it can retrieve from the animation curves
void CMayaCryKeyUtil::getTransform (int time, double dRotation[3], MVector& vmTranslation)
{
	MStatus status;
	unsigned nCoord;

	for (nCoord = 0; nCoord < 3; ++nCoord)
	{
		if (m_arrNodeAnimAttr[kAARotate + nCoord] != MObject::kNullObj)
			getValue(time, m_arrFnAnimAttr[kAARotate + nCoord], dRotation[nCoord]);

		if (m_arrNodeAnimAttr[kAATranslate + nCoord] != MObject::kNullObj)
			getValue(time, m_arrFnAnimAttr[kAATranslate + nCoord], vmTranslation[nCoord]);
	}
}


// retrieves the animation curve Y at the given time
MStatus CMayaCryKeyUtil::getValue (int time, MFnAnimCurve& fnCurve, double& dValue)
{
	MStatus status;

	double dResult = 0;
	switch (fnCurve.animCurveType())
	{
	case MFnAnimCurve::kAnimCurveTA:
	case MFnAnimCurve::kAnimCurveTL:
	case MFnAnimCurve::kAnimCurveTU:
		dResult = fnCurve.evaluate (CryToMayaTime(time), &status);
		break;

	case MFnAnimCurve::kAnimCurveTT:
		{
			MTime newTime;
			status = fnCurve.evaluate(CryToMayaTime(time), newTime);
			dResult = newTime.as(MTime::kSeconds);
		}
		break;

	case MFnAnimCurve::kAnimCurveUA:
	case MFnAnimCurve::kAnimCurveUL:
	case MFnAnimCurve::kAnimCurveUU:
		status = fnCurve.evaluate(CryToMayaTime(time).as(MTime::kSeconds), dResult);
		break;

	case MFnAnimCurve::kAnimCurveUT:
		{
			MTime newTime;
			status = fnCurve.evaluate(CryToMayaTime(time).as(MTime::kSeconds), newTime);
			dResult = newTime.as(MTime::kSeconds);
		}
		break;

	default:
		dResult = fnCurve.evaluate (CryToMayaTime(time), &status);
		break;
	}

	if (!status)
	{
		Log ("*WARNING* CMayaCryKeyUtil: cannot evaluate: %s", status.errorString ().asChar());
		return status;
	}
	else
		dValue = dResult;
	return status;
}


// the start of the manual range - keys before that range are clamped
void CMayaCryKeyUtil::setRangeStart (MTime tStart)
{
	m_nRangeStart = MayaToCryTime (tStart);
}

// the end of the manual range - keys after that range are clamped
void CMayaCryKeyUtil::setRangeEnd (MTime tEnd)
{
	m_nRangeEnd = MayaToCryTime (tEnd);
}

// sets the minimal frame step to export, in frames in Maya
// 0 (default) means no minimum
void CMayaCryKeyUtil::setMinFrameStep (MTime tStep)
{
	m_nMinFrameStepTicks = MayaToCryTime (tStep);
}

