// Utility class for conversion between animation in Maya and Cry Engine

#ifndef _MAYA_CRY_EXPORT_MAYA_CRY_KEY_UTIL_HDR_
#define _MAYA_CRY_EXPORT_MAYA_CRY_KEY_UTIL_HDR_


/////////////////////////////////////////////////////////////////////////
// Given the transform node, retrieves the necessary keys and
// can return the pointer to the array of these keys.
// Used to convert the Maya representation of motion to the CryEngine
// representation that can be directly written to disk
// THROWS MStatus in the case of error
class CMayaCryKeyUtil
{
public:
	// retrieves the animation from the given transform (perhaps not animated) node
	CMayaCryKeyUtil (const MDagPath& pathTransform);

	// animation start and end, in native CryAnimation integer units (4800 ticks per second)
	// after clamping with the manual range
	//int getAnimStart ();
	//int getAnimEnd ();

	// before calling the cry key retrieval functions, the client can set the following parameters:

	// the start of the range - keys before that range are clamped
	void setRangeStart (MTime tStart);

	// the end of the range - keys after that range are clamped
	void setRangeEnd (MTime tEnd);

	// sets the rotation and position error
	void setPosError (float fError)
	{
		m_fPosError = fError;
	}
	void setRotError (float fError)
	{
		m_fQuatError = fError;
	}

	// sets the minimal frame step to export, in frames in Maya
	// 0 (default) means no minimum
	void setMinFrameStep (MTime tStep);

	// returns the number of CryBoneKey elements in the array of keys
	int numCryKeys();


	// returns the pointer to the array of cry bone keys
	CryBoneKey* getCryKeys();

	// negate the quaternions in Cry keys
	void negateCryKeyRotations();

	// cleanup 180+ degree rotations because of quaternion representation parity:
	// make all neighbor quaternions have positive dot product
	void cleanupCryKeyRotations();

	// creates the animation so that the subsequent calls to CryKey retrieval functions are valid
	void compute ()
	{
		prepareAnimationCache ();
	}

	// disables optimization of the keys with linear keys
	void disableOptimization()
	{
		m_bDisableOptimization = true;
	}
protected:

	// makes sure the animation keys have been fetched and optimized
	void prepareAnimationCache();

	// these are the attributes that may be animatable in Maya and thus will
	// be used to construct the animation cry bone array
	enum AnimAttributeEnum
	{
		kAARotate  = 0, // the subarray of rotation attributes
		kAARotateX = 0,
		kAARotateY = 1,
		kAARotateZ = 2,
		kAATranslate  = 3, // the subarray of translation attributes
		kAATranslateX = 3,
		kAATranslateY = 4,
		kAATranslateZ = 5,
		kAA_Count /*= 6*/,

		kAAUnknown = -1
	};

	// retrieves the animation from the given dag path into start/end tiem and array of cry keys (unoptimized)
	void initAnimation();

	// retrieves the animation if it's static (just retrieves the current position/rotation of the node)
	void initAnimationStatic ();

	// retrieves the animated plug nodes: initializes m_arrNodeAnimAttr
	void initNodeAnimAttrs ();

	// converts the animation contained in the array of nodes to the CryBoneKey array
	void convertAnimNodeToCryKeys ();

	// adds the animation node key times (in ticks) to the given set
	void addAnimNodeTicks (std::set<int>& setTicks);

	// optimizes the cry keys
	void optimizeAnimCryKeys ();

	// returns the animation attribute that this plug represents,
	// or kAAUnknown if the plug doesn't represent any of known attributes
	AnimAttributeEnum getAnimAttribute (const MPlug& plug);

	// adds a compound attribute: breaks it into children and adds them curve-by-curve
	void addAnimMulticurve (AnimAttributeEnum nAnimAttr, const MPlug& plug);

	// adds the animation curve
	void addAnimCurve (AnimAttributeEnum nAnimAttr, const MObject& objCurve);
	void addAnimCurve (AnimAttributeEnum nAnimAttr, const MPlug& plug);

	// uses the given animated attribute to initialize one or more anim attr elements in the animation array
	void addAnimAttribute (const MPlug& plug);

	// retrieves the transform of the node at the given time.
	// fills in the components of the transform (rotation and position)
	// which it can retrieve from the animation curves
	void getTransform (int time, double dRotation[3], MVector& vmTranslation);

	// retrieves the animation curve Y at the given time, if it can
	MStatus getValue (int time, MFnAnimCurve& fnCurve, double& fValue);

	// fixes up the joint rotation matrix: forms the full rotation out of the [R] matrix:
	// sets the given [R]  to  [JO]*[R]*[RO]
	void fixupJointRotation (int nTick, MQuaternion& qmRotation);

protected: // attributes

	// the array of animated attributes that define the animation of the node transformation
	MObject m_arrNodeAnimAttr[kAA_Count]; // they will get initialized to nulls
	MFnAnimCurve m_arrFnAnimAttr[kAA_Count];

	// the DAG path to the object being utilized
	MDagPath m_pathTransform;

	// status -whether this joint MFn is valid
	MStatus m_statusFnIkJoint;
	// joint MFn for the transform for which the keys are retrieved
	MFnIkJoint m_fnJoint;
	
	// transform function set - always valid (after initialization)
	MFnTransform m_fnTM;

	// the animation start and end
	//MTime m_tAnimStart, m_tAnimEnd;

	// the manual range animation start and end, and flags indicating whther to use this range
	// boundary(-ies)
	int m_nRangeStart, m_nRangeEnd;

	// the animation itself in the form of CryBoneKey's
	typedef std::vector<CryBoneKey> CryBoneKeyArray;
	CryBoneKeyArray m_arrAnimCryKeys;

	// If true, no initial optimization is performed
	bool m_bDisableOptimization;

	// the minimal step between frames exported, in ticks.
	// this is BEFORE the optimization, so the actual output may still be optimized
	int m_nMinFrameStepTicks;

	float m_fPosError, m_fQuatError;
};

#endif