// BigMatrix.cpp: implementation of the CBigMatrix class.
//
// See the header for detailed description
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "GLUtils.h"
#include "BigVector.h"
#include "BigMatrix.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////
// creates an empty matrix; needs to be resized
//////////////////////////////////////////////////////////////////////
CBigMatrix::CBigMatrix():
	m_nCapHeight(0), m_nCapWidth(0),
	m_nSizeHeight(0), m_nSizeWidth(0),
	m_fCells(NULL)
{}

//////////////////////////////////////////////////////////////////////
// prepares a matrix of the specified dimensions
// the content is undefined
//////////////////////////////////////////////////////////////////////
CBigMatrix::CBigMatrix(int nHeight, int nWidth):
	m_nCapHeight(0), m_nCapWidth(0),
	m_nSizeHeight(0), m_nSizeWidth(0),
	m_fCells(NULL)
{
	Resize (nHeight, nWidth);
}

CBigMatrix::~CBigMatrix()
{
	if (m_fCells)
		delete [m_nCapHeight*m_nCapWidth] m_fCells;
}

//////////////////////////////////////////////////////////////////////
// copies the size and contents of the matrix
// the new matrix is allocated with the same capacity as the source
//////////////////////////////////////////////////////////////////////
CBigMatrix::CBigMatrix (const CBigMatrix& mxThat):
	m_nCapHeight(0), m_nCapWidth(0),
	m_nSizeHeight(0), m_nSizeWidth(0),
	m_fCells(NULL)
{
	Resize (mxThat.m_nCapHeight, mxThat.m_nCapWidth);
	memcpy (m_fCells, mxThat.m_fCells, sizeof(*m_fCells)*m_nCapHeight*m_nCapWidth);
	m_nSizeHeight = mxThat.m_nSizeHeight;
	m_nSizeWidth = mxThat.m_nSizeWidth;
}

//////////////////////////////////////////////////////////////////////
// returns the pointer to a specified row in the matrix.
// the row is an array of doubles and can be indexed appropriately, e.g.
// bmMyMatrix[nRow][nColumn]
//////////////////////////////////////////////////////////////////////
double* CBigMatrix::operator [] (int nRow)
{
	assert(nRow >=0 && nRow < m_nSizeHeight);
	return m_fCells + m_nCapWidth * nRow;
}

//////////////////////////////////////////////////////////////////////
// returns the pointer to a specified row in the matrix.
// the row is an array of doubles and can be indexed appropriately, e.g.
// bmMyMatrix[nRow][nColumn]
//////////////////////////////////////////////////////////////////////
const double* CBigMatrix::operator [] (int nRow)const
{
	assert(nRow >=0 && nRow < m_nSizeHeight);
	return m_fCells + m_nCapWidth * nRow;
}

//////////////////////////////////////////////////////////////////////
// resizes the matrix; doesn't preserve its content
//////////////////////////////////////////////////////////////////////
void CBigMatrix::Resize (int nHeight, int nWidth)
{
	int nOldCap = m_nCapHeight*m_nCapWidth;
	int nNewCap = nHeight*nWidth;
	if (nHeight > m_nCapHeight || nWidth > m_nCapWidth)
	{
		if (m_fCells)
			delete [nOldCap] m_fCells;

		m_fCells = new double [nNewCap];
		m_nCapHeight = nHeight;
		m_nCapWidth  = nWidth;
	}
	m_nSizeHeight = nHeight;
	m_nSizeWidth  = nWidth;
}

//////////////////////////////////////////////////////////////////////
// composes inverse of this matrix; if in the process we have to divide by
// fThreshold or less then we return an empty matrix
//////////////////////////////////////////////////////////////////////
void CBigMatrix::ComposeInverse (CBigMatrix& mxInverse, double fThreshold)
{
	CBigMatrix mxCopyThis (*this);
	mxCopyThis.internalComposeInverse(mxInverse, fThreshold);
}

//////////////////////////////////////////////////////////////////////
// composes inverse of this matrix; if in the process we have to divide by
// fThreshold or less then we return an empty matrix
// SIDE EFFECT: sets this matrix to identity
// WARNING: this is for testing only, don't use as it uses gaussian
// step which is very unstable
//////////////////////////////////////////////////////////////////////
void CBigMatrix::internalComposeInverse (CBigMatrix& mxInverse, double fThreshold)
{
	assert (GetWidth() == GetHeight());
	mxInverse.Resize(GetWidth(), GetWidth());
	mxInverse.SetIdentity();

	int i, j;

	// stage = 0 (row was not operated) 1 (row was left-zeroed) 2 (row was right-zeroed) and 3 (row was adjusted to identity)
	// array of unprocessed rows
	std::vector<int> vRow;
	vRow.resize(GetWidth());
	for (i = 0; i < GetWidth(); ++i)
		vRow[i] = i;

	// on each step , choose a row with the smallest diagonal element that is greater than 1 and process that row
	while (!vRow.empty())
	{
		double fDiagBest = 0;
		int nRowBest = -1;
		// find the best row to divide
		std::vector<int>::iterator it, itBest = vRow.end();
		for (it = vRow.begin(); it != vRow.end(); ++it)
		{
			int nRow = *it;
			double fDiag = tabs((*this)[nRow][nRow]);
			if (fDiagBest >= 1)
			{
				if (fDiag >= 1 && fDiag < fDiagBest)
				{
					fDiagBest = fDiag;
					nRowBest = nRow;
				}
			}
			else
			{
				if (fDiag > fDiagBest)
				{
					fDiagBest = fDiag;
					nRowBest = nRow;
					itBest = it;
				}
			}
		}

		// check if we found a row
		if (itBest == vRow.end() || fDiagBest < fThreshold)
		{
			mxInverse.Resize(0,0);
			return;
		}

		vRow.erase(itBest);

		// we found the row - process it
		
		// first, normalize it
		double fDiag = (*this)[nRowBest][nRowBest];
		for (i = 0; i < GetWidth(); ++i)
		{
			(*this)[nRowBest][i] /= fDiag;
			mxInverse[nRowBest][i] /= fDiag;
		}
		
		// just get rid of additional errors
		(*this)[nRowBest][nRowBest] = 1;
		
		// second, subtract this row from each other row except itself
		for (i = 0; i < GetWidth(); ++i)
			if (i != nRowBest)
			{
				double fK = (*this)[i][nRowBest];
				if (fK != 0)
				for (j = 0; j < GetWidth(); ++j)
				{
					(*this)[i][j] -= (*this)[nRowBest][j]*fK;
					mxInverse[i][j] -= mxInverse[nRowBest][j] * fK;
				}
				// just get rid of accidential errors
				(*this)[i][nRowBest] = 0;
			}
	}
}

/////////////////////////////////////////////////////////////////////////////////
// sets this matrix to identity
/////////////////////////////////////////////////////////////////////////////////
void CBigMatrix::SetIdentity()
{
	assert (GetWidth() == GetHeight());
	SetZero();
	for (int i = 0; i < GetWidth(); ++i)
		(*this)[i][i] = 1;
}

/////////////////////////////////////////////////////////////////////////////////
// sets this matrix to the difference: mxA - mxB
/////////////////////////////////////////////////////////////////////////////////
void CBigMatrix::SetSubtract (CBigMatrix& mxA, CBigMatrix& mxB)
{
	assert(mxA.GetWidth() == mxB.GetWidth());
	assert(mxA.GetHeight() == mxB.GetHeight());
	Resize (mxA.GetHeight(), mxA.GetWidth());
	for (int i = 0; i < mxA.GetHeight(); ++i)
		for (int j = 0; j < mxA.GetWidth(); ++j)
			(*this)[i][j] = mxA[i][j] - mxB[i][j];
}

/////////////////////////////////////////////////////////////////////////////////
// returns the square root of the sum of squares of this matrix
/////////////////////////////////////////////////////////////////////////////////
double CBigMatrix::GetNorm () const
{
	double fNorm = 0;
	for (int i = 0; i < GetHeight(); ++i)
		for (int j = 0; j < GetWidth(); ++j)
			fNorm += tsqr((*this)[i][j]);
	return sqrt(fNorm);
}

///////////////////////////////////////////////////////////////////////////////
// returns the square root of the sum of squares of (this matrix minus identity
// matrix)
///////////////////////////////////////////////////////////////////////////////
double CBigMatrix::GetIdentityError()const
{
	assert (GetHeight() == GetWidth());
	double fError = 0;
	for (int i = 0; i < GetHeight(); ++i)
	{
		for (int j = 0; j < GetWidth(); ++j)
		{
			if (i == j)
				fError += tsqr ((*this)[i][j] - 1);
			else
				fError += tsqr ((*this)[i][j]);
		}
	}
	return fError;
}


///////////////////////////////////////////////////////////////////////////////
// sets this matrix to production of the given arguments (A*B)
///////////////////////////////////////////////////////////////////////////////
void CBigMatrix::SetMultiply (CBigMatrix& mxA, CBigMatrix& mxB)
{
	assert(mxA.GetWidth() == mxB.GetHeight());
	int nVectorSize = mxA.GetWidth();

	Resize (mxA.GetHeight(), mxB.GetWidth());
	SetZero();
	int i, j, k;
	for (i = 0; i < GetHeight(); ++i)
		for (j = 0; j < GetWidth(); ++j)
			for (k = 0; k < nVectorSize; ++k)
				(*this)[i][j] += mxA[i][k]*mxB[k][j];
}

///////////////////////////////////////////////////////////////////////////////
// assigns transpose of this matrix to the target matrix
// PARAMETERS:
//   mxT [OUT] - gets resized and copied the transposed content of this matrix
///////////////////////////////////////////////////////////////////////////////
void CBigMatrix::Transpose (CBigMatrix& mxT)
{
	mxT.Resize(GetWidth(), GetHeight());
	for (int i = 0; i < GetHeight(); ++i)
		for (int j = 0; j < GetWidth(); ++j)
			mxT[j][i] = (*this)[i][j];

}

double safesqrt(double x)
{
	assert (x>=0);
	return sqrt (x);
}

// non-destructive 
template <typename real>
real hypot(real a, real b)
{
	real at = a >= 0 ?a:-a, bt = b>=0?b:-b;
	return (at > bt ) ? 
		(at*sqrt(1.0+tsqr(bt/at))) :
		(bt != 0.0 ?
			(bt*sqrt(1.0+tsqr(at/bt))) :
			0.0);
}

template <typename real>
real hypot1(real a) // b==1
{
	double at = a>=0?a:-a;
	return (at > 1 ) ? 
		(at*sqrt(1.0+tsqr(1/at))) :
		(sqrt(1.0+tsqr(at)));
}


/////////////////////////////////////////////////////////////////////////
// Singular value decomposition of matrix A into A W V.
// taken from some place I don't exactly remember..
//
//   Given A is m by n, m >= n.						     
//   Output:								     
//	A is m by n modified in place into U,				     
//	W is a vector of length m,					     
//	V is a square matrix of size n by n.				     
//                         
// PARAMETERS:             
//   a:    A m by n matrix.
//   m, n: Dimensions of A, W and V
//   w:    A vector of length m
//   v:    A matrix of size n by n
/////////////////////////////////////////////////////////////////////////
bool CBigMatrix::SVDecomposition (CBigVector& w, CBigMatrix& V, int nMaxIterations)
{
	int m = GetHeight(), n = GetWidth();

	CBigMatrix& a = *this;

    int flag, i, its, j, jj, k, l, nm;
    long double c, f, h, s, x, y, z, 
		anorm = 0.0,
		g = 0.0,
		scale = 0.0;

    if (m < n)
		return false; // add extra zero rows to A

	CBigVector rv1(n);

	for (i = 0; i < n; ++i)
	{
		l = i + 1;
		rv1[i] = scale * g;
		g = s = scale = 0.0;
		if (i < m)
		{
			for (k = i; k < m; ++k)
				scale += fabs(a[k][i]);
			if (scale != 0.0)
			{
				for (k = i; k < m; ++k)
				{
					a[k][i] /= scale;
					s += a[k][i] * a[k][i];
				}
				f = a[i][i];
				g = -(f>=0?safesqrt(s):-safesqrt(s));
				h = f * g - s;
				a[i][i] = f - g;
				if (i != n)
				{
					for (j = l; j < n; ++j)
					{
						for (s = 0.0, k = i; k < m; ++k)
							s += a[k][i] * a[k][j];
						f = s / h;
						for (k = i; k < m; ++k)
							a[k][j] += f * a[k][i];
					}
				}
				for (k = i; k < m; ++k)
					a[k][i] *= scale;
			}
		}
		w[i] = scale * g;

		g = s = scale = 0.0;
		if (i < m && i != n-1)
		{
			for (k = l; k < n; ++k)
				scale += fabs(a[i][k]);
			if (scale != 0.0)
			{
				for (k = l; k < n; ++k)
				{
					a[i][k] /= scale;
					s += a[i][k] * a[i][k];
				}
				f = a[i][l];
				g = -(f>=0?safesqrt(s):-safesqrt(s));
				h = f * g - s;
				a[i][l] = f - g;
				for (k = l; k < n; ++k)
					rv1[k] = a[i][k] / h;
				if (i != m-1)
				{
					for (j = l; j < m; ++j)
					{
						for (s = 0.0, k = l; k < n; ++k)
							s += a[j][k] * a[i][k];
						for (k = l; k < n; ++k)
							a[j][k] += s * rv1[k];
					}
				}
				for (k = l; k < n; ++k)
					a[i][k] *= scale;
			}
		}
		s = fabs(w[i]) + fabs(rv1[i]);
		anorm = tmax(anorm, s);
    }

	for (i = n-1; i >= 0; --i)
	{
		if (i < n-1)
		{
			if (g != 0.0)
			{
				for (j = l; j < n; ++j)
					V[j][i] = (a[i][j] / a[i][l]) / g;
				for (j = l; j < n; ++j)
				{
					for (s = 0.0, k = l; k < n; ++k)
						s += a[i][k] * V[k][j];
					for (k = l; k < n; ++k)
					V[k][j] += s * V[k][i];
				}
			}
			for (j = l; j < n; ++j)
				V[i][j] = V[j][i] = 0.0;
		}
		V[i][i] = 1.0;
		g = rv1[i];
		l = i;
	}

	for (i = n-1; i >= 0; --i)
	{
		l = i + 1;
		g = w[i];
		if (i < n)
			for (j = l; j < n; ++j)
				a[i][j] = 0.0;
		if (g != 0.0)
		{
			g = 1.0 / g;
			if (i != n)
			{
				for (j = l; j < n; ++j)
				{
					for (s = 0.0, k = l; k < m; ++k)
						s += a[k][i] * a[k][j];
					f = (s / a[i][i]) * g;
					for (k = i; k < m; ++k)
						a[k][j] += f * a[k][i];
				}
			}
			for (j = i; j < m; ++j)
				a[j][i] *= g;
		}
		else
		{
			for (j = i; j < m; ++j)
				a[j][i] = 0.0;
		}
		++a[i][i];
	}

	for (k = n-1; k >= 0; --k)
	{
		for (its = 0; its < nMaxIterations; ++its)
		{
			flag = 1;
			for (l = k; l >= 0; --l)
			{
				nm = l-1;

				if (tabs(rv1[l]) + anorm == anorm)
				{
					flag = 0;
					break;
				}

				if ((nm>=0?tabs(w[nm]):0) + anorm == anorm)
				{
					break;
				}
			}

			if (flag)
			{
				c = 0.0;
				s = 1.0;
				for (i = l; i <= k; ++i)
				{
					f = s * rv1[i];
					if (tabs(f) + anorm != anorm)
					{
						g = w[i];
						h = safesqrt(tsqr(f) + tsqr(g));
						w[i] = h;
						h = 1.0 / h;
						c = g * h;
						s = (-f * h);
						for (j = 0; j < m; ++j)
						{
							y = nm>=0 ? a[j][nm] : 0;
							z = a[j][i];
							if (nm>=0) a[j][nm] = y * c + z * s;
							a[j][i] = z * c - y * s;
						}
					}
				}
			}
			z = w[k];
			if (l == k)
			{
				if (z < 0.0)
				{
					w[k] = -z;
					for (j = 0; j < n; ++j)
						V[j][k] = (-V[j][k]);
				}
				break;
			}

			if (its >= nMaxIterations)
				return false;
				
			x = w[l];
			nm = k - 1;
			y = nm>=0 ? w[nm] : 0;
			g = nm>=0 ? rv1[nm] : 0;
			h = rv1[k];
			f = ((y - z) * (y + z) + (g - h) * (g + h)) / (2.0 * h * y);
			g = safesqrt (tsqr(f) + 1.0);
			f = ((x - z) * (x + z) + h * ((y / (f + (f>=0?tabs(g):-tabs(f)))) - h)) / x;
			c = s = 1.0;
			for (j = 0; j <= nm; ++j)
			{
				i = j + 1;
				g = rv1[i];
				y = w[i];
				h = s * g;
				g = c * g;
				z = safesqrt(tsqr(f) + tsqr(h));
				rv1[j] = z;
				c = f / z;
				s = h / z;
				f = x * c + g * s;
				g = g * c - x * s;
				h = y * s;
				y = y * c;
				for (jj = 0; jj < n; ++jj)
				{
					x = V[jj][j];
					z = V[jj][i];
					V[jj][j] = x * c + z * s;
					V[jj][i] = z * c - x * s;
				}
				z = safesqrt(tsqr(f) + tsqr(h));
				w[j] = z;
				if (z != 0.0)
				{
					z = 1.0 / z;
					c = f * z;
					s = h * z;
				}
				f = (c * g) + (s * y);
				x = (c * y) - (s * g);
				for (jj = 0; jj < m; ++jj)
				{
					y = a[jj][j];
					z = a[jj][i];
					a[jj][j] = y * c + z * s;
					a[jj][i] = z * c - y * s;
				}
			}
			rv1[l] = 0.0;
			rv1[k] = f;
			w[k] = x;
		}
	}
	return true;
}

#define SIGN(a, b) ((b) < 0.0 ? -fabs(a): fabs(a))

// this one proved to be more stable over a series of experiments..
// though it's my personal feeling only.
bool CBigMatrix::SVDecompositionV2 (CBigVector& w, CBigMatrix& v, int nMaxIterations)
{
   /////////////////////////////////////////////////////////////////////
   // Given a matrix a[m][n], this routine computes its singular value
   // decomposition, A = U*W*V'.  The matrix U replaces a on output.
   // The diagonal matrix of singular values W is output as a vector w[n].
   // The matrix V  is output as v[n][n].
   // m must be greater or equal to n;  if it is smaller, then a should be
   // filled up to square with zero rows.
   /////////////////////////////////////////////////////////////////////////

	int m = GetHeight(), n = GetWidth();
	CBigMatrix& a = *this;

	int flag, i, its, j, jj, k, l, nm;
   long double c, f, h, s, x, y, z;
   long double anorm = 0.0, g = 0.0, scale = 0.0;

   CBigVector rv1(n);

    // Householder reduction to bidiagonal form.
   l = 0;    // added by T. Wang to avoid warning in g++
   nm = 0;   // added by T. Wang to avoid warning in g++
   for (i = 0; i < n; i++) {
      l = i + 1;
      rv1[i] = scale*g;
      g = s = scale = 0.0;
      if (i < m) {
         for (k = i; k < m; k++) scale += fabs(a[k][i]);
            if (scale != 0.0) {
               for (k = i; k < m; k++) {
                  a[k][i] /= scale;
                  s += a[k][i]*a[k][i];
               }
               f = a[i][i];
               g = -SIGN(sqrt(s), f);
               h = f*g - s;
               a[i][i] = f - g;
               if (i != n - 1) {
                  for (j = l; j < n; j++) {
                     for (s  = 0.0, k = i; k < m; k++) s += a[k][i]*a[k][j];
										 if (h != 0)
										 {
										   f = s/h;
                       for ( k = i; k < m; k++) a[k][j] += f*a[k][i];
										 }
                  }
               }
               for (k = i; k < m; k++) a[k][i] *= scale;
            }
         }
         w[i] = scale*g;
         g = s= scale = 0.0;
         if (i < m && i != n - 1) {
            for (k = l; k < n; k++)  scale += fabs(a[i][k]);
            if (scale != 0) {
               for (k = l; k < n; k++) {
                  a[i][k] /= scale;
                  s += a[i][k]*a[i][k];
               }
               f = a[i][l];
               g = -SIGN(sqrt(s), f);
               h = f*g - s;
               a[i][l] = f - g;
							 if ( h!= 0)
								for (k = l; k < n; k++)  rv1[k] = a[i][k]/h;
               if (i != m - 1) {
                  for (j = l; j < m; j++) {
                     for (s = 0.0, k = l; k < n; k++) s += a[j][k]*a[i][k];
                     for (k = l; k < n; k++) a[j][k] += s*rv1[k];
                  }
               }
               for (k = l; k < n; k++) a[i][k] *= scale;
            }
         }
         anorm = max(anorm, (fabs(w[i]) + fabs(rv1[i])));
      }
        /* Accumulation of right-hand transformations.	*/
      for (i = n - 1; 0 <= i; i--) {
         if (i < n - 1) {
	    if (g != 0) {
               for (j = l; j < n; j++)
							 {
								 long double tmp = a[i][l]*g;
								 if (tmp != 0)
									v[j][i] = a[i][j]/tmp;
							 }
		  /* Double division to avoid possible underflow: */
               for (j = l; j < n; j++) {
                  for (s = 0.0, k = l; k < n; k++) s += a[i][k]*v[k][j];
                  for (k = l; k < n; k++)  v[k][j] += s*v[k][i];
               }
	    }
	    for (j = l; j < n; j++) v[i][j] = v[j][i] = 0.0;
         }
         v[i][i] = 1.0;
         g = rv1[i];
         l = i;
      }
          /* Accumulation of left-hand transformations.	  */
      for (i = n - 1; 0 <= i; i--) {
         l = i + 1;
         g = w[i];
         if (i < n - 1) for (j = l; j < n; j++) a[i][j] = 0.0;
         if (g != 0.0) {
            g = 1.0/g;
            if (i != n - 1) {
               for (j = l; j < n; j++) {
		  for (s = 0.0, k = l; k < m; k++) s += a[k][i]*a[k][j];
                  if (a[i][i] != 0) f = (s/a[i][i])*g;
                  for (k = i; k < m; k++) a[k][j] += f*a[k][i];
               }
	    }
            for (j = i; j < m; j++)  a[j][i] *= g;
         }
         else {
            for (j = i; j < m; j++) a[j][i] = 0.0;
         }
         a[i][i] += 1.0;   // ++a[i][i]
      }
       /* Diagonalization of the bidiagonal form.  */
      for (k = n - 1; 0 <= k; k--) {        /* Loop over singular values. */
         for (its = 0; its < 30; its++) {    /* Loop over allowed iterations.*/
            flag = 1;
            for (l = k; 0 <= l; l--) {     // Test for splitting: 
               nm = l - 1;                 // Note that rv1[0] is always zero
               if (l == 0 || fabs(rv1[l]) + anorm == anorm) {//if (fabs(rv1[l]) + anorm == anorm) {
                  flag = 0;
                  break;
               }
               if (tabs(w[nm]) + anorm == anorm) break;
            }
            if (flag) {
            c = 0.0;	                   /* Cancellation of rv1[l], if l>0:*/
            s = 1.0;
            for (i = l; i <= k; i++) {
               f = s*rv1[i];
               if (fabs(f) + anorm != anorm) {
                  g = w[i];
                  h = hypot(f, g); 
                  w[i] = h;
									if (h != 0)
									{
                  h = 1.0/h;
                  c = g*h;
                  s = (-f*h);
                  for (j = 0; j < m; j++) {
                     y = a[j][nm];
                     z = a[j][i];
                     a[j][nm] = y*c + z*s;
                     a[j][i]  = z*c - y*s;
                  }
									}
               }
            }
         }
         z = w[k];
         if (l == k) {	     /* Convergence.  */
            if (z < 0.0) {        /* Singular value is made non-negative. */
               w[k] = -z;
               for (j = 0; j < n; j++) v[j][k] = (-v[j][k]);
            }
            break;
         }
         if (its >= nMaxIterations)
					 return false;
         x = w[l];               /* Shift from bottom 2-by-2 minor. */
         nm = k - 1;
         y = w[nm];
         g = rv1[nm];
         h = rv1[k];
         f = ((y - z)*(y + z) + (g - h)*(g + h))/(2.0*h*y);
         g = hypot1(f);
         f = ((x - z)*(x + z) + h*((y/(f + SIGN(g, f))) - h))/x;
	    /* Next QR transformation:	  */
         c = s = 1.0;
         for (j = l; j <= nm; j++) {
            i = j + 1;
            g = rv1[i];
            y = w[i];
            h = s*g;
            g = c*g;
            z = hypot(f, h);
            rv1[j] = z;
            c = f/z;
            s = h/z;
            f = x*c + g*s;
            g = g*c - x*s;
            h = y*s;
            y = y*c;
            for (jj = 0; jj < n;  jj++) {
               x = v[jj][j];
               z = v[jj][i];
               v[jj][j] = x*c + z*s;
               v[jj][i] = z*c - x*s;
            }
            z = hypot(f, h);
            w[j] = z;        /* Rotation can be arbitrary if z = 0.*/
            if (z) {
               z = 1.0/z;
               c = f*z;
               s = h*z;
            }
            f = (c*g) + (s*y);
            x = (c*y) - (s*g);
            for (jj = 0; jj < m; jj++) {
               y = a[jj][j];
               z = a[jj][i];
               a[jj][j] = y*c + z*s;
               a[jj][i] = z*c - y*s;
            }
         }
         rv1[l] = 0.0;
         rv1[k] = f;
         w[k] = x;
      }
   }
			return true;
}


// this one proved to be more stable over a series of experiments..
// though it's my personal feeling only.
bool CBigMatrix::SVDecompositionV3 (CBigVector& w, CBigMatrix& v, int nMaxIterations)
{
	// dgebrd() is the LAPACK routine for SVD
		

   /////////////////////////////////////////////////////////////////////
   // Given a matrix a[m][n], this routine computes its singular value
   // decomposition, A = U*W*V'.  The matrix U replaces a on output.
   // The diagonal matrix of singular values W is output as a vector w[n].
   // The matrix V  is output as v[n][n].
   // m must be greater or equal to n;  if it is smaller, then a should be
   // filled up to square with zero rows.
   /////////////////////////////////////////////////////////////////////////

	int m = GetHeight(), n = GetWidth();
	CBigMatrix& a = *this;
/*
	int flag, i, its, j, jj, k, l, nm;
   long double c, f, h, s, x, y, z;
   long double anorm = 0.0, g = 0.0, scale = 0.0;

   CBigVector rv1(n);*/
	 return false;
}

////////////////////////////////////////////////////////////////////
// pseudoinverse of a square or nonsquare matrix
// Multiplication of pinv(A) with A gives the identity matrix.
// This function replaces the matrix mxPseudoInverse with the TRANSPOSE of the 
//   pseudoinverse matrix.
// PARAMETERS:
//   mxPseudoInverse [OUT] - on successful output, the pseudoinverse;
//                           on error - empty (0x0) matrix
//   fThreshold      [IN]  - which eigenvalues must be clamped (1e-3 would go ok for most cases)
// REUTRNS:
//   the min eigenvalue / max eigenvalue
//   thus if it returns 1 - it's good
//                      0 - BAD
////////////////////////////////////////////////////////////////////
double CBigMatrix::ComposePseudoInverse (CBigMatrix& mxPseudoInverse, double fThreshold) const
{
	int i, j, k;

	assert (GetHeight() >= 1 && GetWidth() >= 1);

	// preare matrices for SVD

	CBigMatrix U, V (GetWidth(), GetWidth());
	CBigVector s (GetWidth());

	// for the SVD I have, the input must be wide, not tall
	if (GetHeight() < GetWidth())
	{
		// add spare lines of zeros
		U.Resize(GetWidth(), GetWidth());
		for (i = 0; i < GetHeight(); ++i)
			for (j = 0; j < GetWidth(); ++j)
				U[i][j] = (*this)[i][j];
		for (int i = GetHeight(); i < GetWidth(); ++i)
			for (j = 0; j < GetWidth(); ++j)
				U[i][j] = 0;
	}
	else
		U = (*this);

	// DO the SVD
	if (U.SVDecompositionV2 (s, V))
	{
		// calculate the pseudoinverse matrix
		// remove the near singular values from the vector s

		double fAbsMax = tabs(s[0]);

		for (i = 1; i < s.GetSize(); ++i)
		{
			if (tabs(s[i]) > fAbsMax)
				fAbsMax = tabs(s[i]);
		}

		double fAbsMin = fAbsMax;

		// compose the pseudoinverse itself, out of the SVD we've just made
		mxPseudoInverse.Resize (GetWidth (), GetHeight ());

		for (i=0;i<GetWidth();++i)
			for (j=0;j<GetHeight();++j)
			{
				mxPseudoInverse[i][j] = 0;
				for (k = 0; k < GetWidth(); ++k)
				{
					if (tabs(s[k]) / fAbsMax > fThreshold)
					{
						if (fAbsMin > tabs(s[k]))
							fAbsMin = tabs(s[k]);
						mxPseudoInverse[i][j] += V[i][k] * U[j][k] / s[k];
					}
				}
			};
		
		// compute the stability number - min Eigenvalue/ max Eigenvalue
		return fAbsMin/fAbsMax;
	}
	else
	{
		// SVD didn't converge.. the rare case. return error
		mxPseudoInverse.Resize (0,0);
		return 0;
	}
}

// copies contents of the matrix
CBigMatrix& CBigMatrix::operator = (const CBigMatrix& mxThat)
{
	Resize (mxThat.GetHeight(), mxThat.GetWidth());
	memcpy (m_fCells, mxThat.m_fCells, GetHeight()*GetWidth()*sizeof(m_fCells[0]));
	return *this;
}


///////////////////////////////////////////////////////////////////////////
// zeroes this matrix, including the spare cells (if capacity > dimensions)
///////////////////////////////////////////////////////////////////////////
void CBigMatrix::SetZero()
{
	memset (m_fCells, 0, m_nCapHeight*m_nCapWidth*sizeof(double));
}

///////////////////////////////////////////////////////////////////////////
// Slow but simple to use multiplication
///////////////////////////////////////////////////////////////////////////
CBigVector operator * (const CBigVector& bvIn, const CBigMatrix&bmIn)
{
	assert (bvIn.GetSize() == bmIn.GetHeight());
	CBigVector bvOut;
	bvOut.Resize(bmIn.GetWidth());
	bvOut.SetZero();
	for (int nMatrixColumn = 0; nMatrixColumn < bmIn.GetWidth(); ++nMatrixColumn)
		for (int nMatrixRow = 0; nMatrixRow < bvIn.GetSize(); ++nMatrixRow)
		{
			bvOut[nMatrixColumn] += bvIn[nMatrixRow] * bmIn[nMatrixRow][nMatrixColumn];
		}
	return bvOut;
}

///////////////////////////////////////////////////////////////////////////
// Slow but simple to use multiplication
///////////////////////////////////////////////////////////////////////////
CBigVector operator * (const CBigMatrix&bmIn, const CBigVector&bvIn)
{
	assert (bvIn.GetSize() == bmIn.GetWidth());
	CBigVector bvOut;
	bvOut.Resize(bmIn.GetHeight());
	bvOut.SetZero();
	for (int nMatrixRow = 0; nMatrixRow < bmIn.GetHeight(); ++nMatrixRow)
		for (int nMatrixColumn = 0; nMatrixColumn < bvIn.GetSize(); ++nMatrixColumn)
		{
			bvOut[nMatrixRow] += bmIn[nMatrixRow][nMatrixColumn] * bvIn[nMatrixColumn];
		}
	return bvOut;
}
