#pragma once

// Utility for calculation of the ticks spent by some operation.

// returns the counter core frequency, in Hertz, or 0 if cannot determine
unsigned __int64 PerfCounterFrequency();

// returns the current tick count, analog of the QueryPerformanceCounter
unsigned __int64 PerfCounterTick();

// This utility allows to easily track and sum up times of execution.
// It's a timer that starts automatically upon construction, can be paused
// and resumed
//
// To use it, create an instance and call resume() and then stop()
// periodically. After you called stop(), you can use seconds() to get the time
class CProfilerTimer
{
public:
	// initializes the timer (paused by default)
	CProfilerTimer();
	CProfilerTimer(bool bStart);
	// returns the current time, in seconds
	// can only be called after pause()
	double seconds();
	// returns the current time, in milliseconds
	double milliseconds() {return seconds() * 1000;}
	// resets the timer to 0 and sets it to a paused state
	void reset();
protected:
	// resumes the timer
	void resume();
	// pauses the timer
	void pause();
	friend class CProfilerTimerAutoClicker;
private:
	// this is the current time, 
	unsigned __int64 m_nTime;
};

// this class automatically resumes and pauses the given timer
class CProfilerTimerAutoClicker
{
public:
	CProfilerTimerAutoClicker (CProfilerTimer& timer):
		m_Timer (timer)
	{
		timer.resume();
	}
	~CProfilerTimerAutoClicker()
	{
		m_Timer.pause();
	}
protected:
	CProfilerTimer& m_Timer;
};

// include this string AUTO_PROFILE_SECTION(timer) for the section of code where the profiler timer must be turned on and off
#define AUTO_PROFILE_SECTION(timer) CProfilerTimerAutoClicker __profiler_for_##timer(timer)