// Triangulator.h: interface for the CTriangulator class.
//
// NOTE: THe code is taken from Graphics Gems and is publicly available
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_TRIANGULATOR_H__C36AFEF0_390E_4E9E_80A8_B2908BAB67AE__INCLUDED_)
#define AFX_TRIANGULATOR_H__C36AFEF0_390E_4E9E_80A8_B2908BAB67AE__INCLUDED_

#if _MSC_VER > 1000
#pragma once
#endif // _MSC_VER > 1000

#include <sys/types.h>
#include <time.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>

class CTriangulator  
{
public:
	struct Point
	{
		double x, y;

		Point (){}
		Point (double _x, double _y):x(_x),y(_y) {}

		double operator * (const Point& right) const {return x*right.x + y*right.y;}
		Point operator + (const Point& right) const {return Point (x+right.x, y+right.y);}
		Point operator - (const Point& right) const {return Point (x-right.x, y-right.y);}
		Point operator - () const {return Point (-x, -y);}
		// cross-product of two vectors
		double operator ^ (const Point& right) const {return x * right.y - y * right.x;}
		Point& operator += (const Point& right) {x+= right.x; y+= right.y;return *this;}
		Point& operator -= (const Point& right) {x-= right.x; y-= right.y;return *this;}

		double& operator [] (int nIndex) {assert (nIndex>= 0 && nIndex <2); return (&x)[nIndex];}
		const double& operator [] (int nIndex) const {assert (nIndex>= 0 && nIndex <2); return (&x)[nIndex];}
	};
	typedef Point Vector;

	struct Face
	{
		int v[3];
		int& operator [] (int i) {assert (i>=0 && i < 3); return v[i];}
		int operator [] (int i) const {assert (i>=0 && i < 3); return v[i];}
	};

	inline static double cross(const Point& v0, const Point& v1, const Point& v2)
	{
		return (v1.x - v0.x) * (v2.y - v0.y) - (v1.y - v0.y) * (v2.x - v0.x);
	}

	inline static double dot (const Point& v0, const Point& v1)
	{
		return v0.x * v1.x + v0.y * v1.y;
	}
private:

	enum { T_X     = 1 };
	enum { T_Y     = 2 };
	enum { T_SINK  = 3 };

	enum { QSIZE   = 800 }; // maximum table sizes 
	enum { TRSIZE  = 400 }; // max# trapezoids 
	enum { SEGSIZE = 100 }; // max# of segments 

	enum { FIRSTPT = 1 }; // checking whether pt. is inserted
	enum { LASTPT  = 2 };

	enum { INFINITY = 1<<30};

	enum { S_LEFT   = 1 }; // for merge-direction
	enum { S_RIGHT  = 2 };

	enum { ST_VALID = 1 }; // for trapezium table
	enum { ST_INVALID = 2};

	enum { SP_SIMPLE_LRUP =  1 }; /* for splitting trapezoids */
	enum { SP_SIMPLE_LRDN =  2 };
	enum { SP_2UP_2DN     =  3 };
	enum { SP_2UP_LEFT    =  4 };
	enum { SP_2UP_RIGHT   =  5 };
	enum { SP_2DN_LEFT    =  6 };
	enum { SP_2DN_RIGHT   =  7 };
	enum { SP_NOSPLIT     = -1 };

	enum { TR_FROM_UP = 1 }; // for traverse-direction
	enum { TR_FROM_DN = 2 }; 

	enum { TRI_LHS = 1 };
	enum { TRI_RHS = 2 };


	static const double gCoordEpsilon, gEpsilon;

	inline bool fp_equal(double s, double t)
	{
		return fabs(s - t) <= gEpsilon;
	}

	struct segment_t
	{
		Point v0, v1;
		int is_inserted;
		int root0, root1;
	};

	struct trap_t
	{
		int lseg, rseg;
		Point hi, lo;
		int u0, u1;
		int d0, d1;
		int sink;
		int usave, uside;
		int state;
	};

	struct node_t
	{
		int nodetype;
		int segnum;
		Point yval;
		int trnum;
		int parent;
		int left, right;
	};

	struct monchain_t
	{
		int vnum;
		int next;
		int prev;
	};

	struct vertexchain_t
	{
		Point pt;
		int vnext[4];			/* next vertices for the 4 chains */
		int vpos[4];			/* position of v in the 4 chains */
		int nextfree;
	};

	struct global_s
	{
		int nseg;
	};
	
	/////////////////////////////////////////////////////////////////
	
	// Query structure
	node_t qs[QSIZE];

	// Trapezoid structure
	trap_t tr[TRSIZE];

	// Segment table
	segment_t seg[SEGSIZE];

	struct global_s global;
	int choose_idx;
	int permute[SEGSIZE];
	int q_idx;
	int tr_idx;

	// Table to hold all the monotone polygons . Each monotone polygon is a circularly linked list.
	monchain_t mchain[TRSIZE];

	// chain init. information. This is used to decide which monotone polygon to split if
	// there are several other polygons touching at the same vertex.
	vertexchain_t vert[SEGSIZE];

	// Contains position of any vertex in the monotone chain for the polygon.
	int mon[SEGSIZE];

	//
	int visited[TRSIZE];

	// 
	int chain_idx, op_idx, mon_idx;

	inline static double crossSine (const Point& v0, const Point& v1)
	{
		return v0.x * v1.y - v1.x * v0.y;
	}

	inline static double length (const Point& v0)
	{
		return (sqrt((v0).x * (v0).x + (v0).y * (v0).y));
	}
private:
	//////////////////////// PRIVATE methods ////////////////////////
	int generate_random_ordering(int n);
	int choose_segment();
	int inserted(int segnum, int whichpt);
	int math_logstar_n(int n);
	int math_N(int n, int h);
	int newnode();
	int newtrap();
	int init_query_structure(int segnum);
	int is_collinear(int segnum, Point *v, int is_swapped);
	int locate_endpoint(Point *v, Point *vo, int r);
	int merge_trapezoids(int segnum, int tfirst, int tlast, int side);
	int add_segment(int segnum);
	int find_new_roots(int segnum);
	int construct_trapezoids(int nseg, segment_t *seg);
	int traverse_polygon(int mcur, int trnum, int from, int dir);
	int triangulate_single_polygon(int posmax, int side, Face op[]);
	int inside_polygon(trap_t *t);
	int newmon();
	int new_chain_element();
	double get_angle(Point *vp0, Point *vpnext, Point *vp1);
	int get_vertex_positions(int v0, int v1, int *ip, int *iq);
	int make_new_monotone_poly(int mcur, int v0, int v1);
	int monotonate_trapezoids(int n);
	int triangulate_monotone_polygons(int nmonpoly, Face op[]);
	int initialise(int nseg);
	int is_point_inside_polygon(double vertex[2]);
	/////////////////////////////////////////////////////////////////
public:
	/////////////////////// STATIC methods //////////////////////////
	int _max(Point *yval, Point *v0, Point *v1);
	int _min(Point *yval, Point *v0, Point *v1);
	bool _greater_than(Point *v0, Point *v1);
	bool _equal_to(Point *v0, Point *v1);
	bool _greater_than_equal_to(Point *v0, Point *v1);
	bool _less_than(Point *v0, Point *v1);
	bool is_left_of(int segnum, Point *v);
	/////////////////////////////////////////////////////////////////
public:
	/////////////// CONSTRUCTION / DESTRUCTION //////////////////////
	CTriangulator(){}
	virtual ~CTriangulator(){}
	/////////////////////////////////////////////////////////////////
public:
	/////////////////////// PUBLIC methods //////////////////////////
	int triangulatePolygon(int n, const Point vertices[], Face triangles[]);
	/////////////////////////////////////////////////////////////////
};

#endif // !defined(AFX_TRIANGULATOR_H__C36AFEF0_390E_4E9E_80A8_B2908BAB67AE__INCLUDED_)
