//////////////////////////////////////////////////////////////////////
//
//	Crytek SuperFramework Source code
//	
//	File:PolyBump.h
//
//	History: 	
//    * July 16,2001: Created by Marco Corbetta
//    * 2/15/2002:    modified by Martin Mittring, integrated in a 3D Studio Plugin
//		- 2/21/2002 modified to work in 3DStudio max as a thread
//
//////////////////////////////////////////////////////////////////////

#ifndef POLYBUMP_H
#define POLYBUMP_H

#if _MSC_VER > 1000
# pragma once
#endif

#include "PbTri.h"																		// CPbTri
#include "EveryObjectOnlyOnce.h"											// CEveryObjectOnlyOnce

#ifdef USE_RASTERCUBE
	#include "TGA.h"																		// PIX_LoadTGA32
	#include "RasterCube.h"															// CRasterCube
#endif


#define USE_MEMORY_BLOCKS

#ifdef USE_MEMORY_BLOCKS
	#define MAX_MEMORY_BLOCK 4*1024*1024
#endif

class CSimpleIndexedMesh;
class CPolyBumpWorkerThread;
class CPBCloneMap;





class CPbLowMesh
{
public:

	//! constructor
	CPbLowMesh() :m_pMesh(0), m_nNumTris(0) {}
		
	//! destructor
	~CPbLowMesh()
	{
		if(m_pMesh)
		{
			delete [] m_pMesh;
			m_pMesh=NULL;
		}    
	}

	//! copy the data, doesnt build a accelerator data structure
	//! /param mesh pointer to the input mesh
  void GetNormalsAndClearBaseVectors( CSimpleIndexedMesh *mesh );

	//! for clonespace
	//! copy the data, doesnt build a accelerator data structure (classes BuildNormalsAndClearBaseVectors)
	//! /param mesh pointer to the input mesh
	//! /param inpCloneMap pointer to the clonemap, must not be 0
	void BuildNormalsAndBaseVectors( CSimpleIndexedMesh *mesh, CPBCloneMap *inpCloneMap );

	//! for object and tangentspace
	//! copy the data, doesnt build a accelerator data structure (classes BuildNormalsAndClearBaseVectors)
	//! /param mesh pointer to the input mesh
	void BuildNormalsAndBaseVectors( CSimpleIndexedMesh *mesh );

	//! this methor builds the internal data structured for speedup the raycasting
	//! /param inbDebug debug output is generated
	//! /return status information that could be shown later (is terminated with '\n')
	std::string BuildRayAccelerator( const bool inbDebug );

	//! calculate the size of the bounding sphere (needed for BuildRayAccelerator())
	//! /return outMin minimum x,y,z of the bounding box
	//! /return outMin maximum x,y,z of the bounding box
	//! /return diameter size
	float CalcBoundingVolume( Vec3 &outMin, Vec3 &outMax );

	//! needed for low and high poly
	//! raytracing, gather unsorted hits
	//! call ClearListBefore and GetHitList afterwards
	//! /param inStart start vector in worldspace
	//! /param inDirection end vector of the ray
	//! /param outIntersections reference to the intersection list (make sure the list is empty)
	void GatherRayHits( const Vec3 invStart, const Vec3 invEnd, CIntInfoLowList &outIntersections, const CPbLowTri *inpIgnoreObject );
	
	// -------------------------------------------------------------------------------------------

	int															m_nNumTris;								//!< number of triangles in the mesh
  CPbLowTri *											m_pMesh;									//!< pointer to the triangle data

#ifdef USE_RASTERCUBE
		CRasterCube<CPbLowTri *>			m_Raster;									//!<
#endif // USE_RASTERCUBE
};




class CPbHighMesh
{
public:

	//! constructor
	CPbHighMesh() :m_pMesh(0), m_nNumTris(0) {}
		
	//! destructor
	~CPbHighMesh()
	{
#ifdef 	USE_MEMORY_BLOCKS	//win32 memory optimization
		for (unsigned long i = 0; i < m_nBlockCnt; i++)
		{
			if(m_pMesh)
			{
				delete [] m_pMesh[i];
				m_pMesh[i]=NULL;
			}    
		}

		delete [] m_pMesh;
		m_pMesh=NULL;
#else
		if(m_pMesh)
		{
			delete [] m_pMesh;
			m_pMesh=NULL;
		}    
#endif //USE_MEMORY_BLOCKS
	}

	//! raytracing, test if there is a hit or not
	//! /param inStart start vector in worldspace
	//! /param inDirection end vector of the ray
	//! /return true: ray has a intersection, false: otherwise
	bool CalcIntersection( const Vec3 invStart, const Vec3 invEnd, const CPbHighTri *inpIgnoreObject );

	//! copy the data, doesnt build a accelerator data structure
	//! /param mesh pointer to the input mesh
  void GetNormalsAndClearBaseVectors( CSimpleIndexedMesh *mesh );

	//! this methor builds the internal data structured for speedup the raycasting
	//! /param inbDebug debug output is generated
	//! /return status information that could be shown later (is terminated with '\n')
	std::string BuildRayAccelerator( const bool inbDebug );

	//! calculate the size of the bounding sphere (needed for BuildRayAccelerator())
	//! /return outMin minimum x,y,z of the bounding box
	//! /return outMin maximum x,y,z of the bounding box
	//! /return diameter size
	float CalcBoundingVolume( Vec3 &outMin, Vec3 &outMax );

	//! needed for low and high poly
	//! raytracing, gather unsorted hits
	//! call ClearListBefore and GetHitList afterwards
	//! /param inStart start vector in worldspace
	//! /param inDirection end vector of the ray
	//! /param outIntersections reference to the intersection list (make sure the list is empty)
	void GatherRayHits( const Vec3 invStart, const Vec3 invEnd, CIntInfoHighList &outIntersections, const CPbHighTri *inpIgnoreObject );

	
	// -------------------------------------------------------------------------------------------


	int															m_nNumTris;								//!< number of triangles in the mesh
#ifdef USE_MEMORY_BLOCKS
	unsigned long										m_nBlockCnt, m_nObjPerBlock;
  CPbHighTri **										m_pMesh;
#else
  CPbHighTri *										m_pMesh;									//!< pointer to the triangle data
#endif //USE_MEMORY_BLOCKS

#ifdef USE_RASTERCUBE
		CRasterCube<CPbHighTri *>			m_Raster;									//!<
#endif // USE_RASTERCUBE
};



	


bool CreateBumpMap( CPolyBumpWorkerThread *inpData );

//! \param invPos
//! \param invNormal
//! \param inpHighMesh
//! \param infRayLength
//! \param iniQuality 0=worst..10=very good
//! \param outvAverageNormal may not ne normalized (depending an accessibility)
float CalcAccessability( const Vec3 invPos, const Vec3 invNormal, CPbHighMesh *inpHighMesh, float infRayLength,
	int iniQuality, Vec3 &outvAverageNormal, const CPbHighTri *inpIgnoreObject );

#endif


