//////////////////////////////////////////////////////////////////////
//
//	Crytek SuperFramework Source code
//	
//	File: SimpleIndexedMesh.h
//
//	History:
//	-:Created by Vladimir Kajalin
//  -:Modified by Martin Mittring
//
//////////////////////////////////////////////////////////////////////

#ifndef __SIMPLEINDEXEDMESH2_H
#define __SIMPLEINDEXEDMESH2_H

#include <stdio.h>																		// sprintf()
#include <math.h>																			// fabs()
#include <assert.h>																		// assert()



#include <string>																			// STL string




struct CObjVert
{
	//! constructor
  CObjVert(){}

	//! destructor
  CObjVert(float _x, float _y, float _z)
  {
    x=_x;
    y=_y;
    z=_z;
  }

	//! compare operator
	bool operator != (CObjVert & other)
  {
		return(!(*this == other));
	}

	//! compare operator
	bool operator == (CObjVert & other)
  {
    if(fabs(x-other.x)<0.0001f)
    if(fabs(y-other.y)<0.0001f)
    if(fabs(z-other.z)<0.0001f)
      return(true);

    return(false);
  }
  
	float x,y,z;					//< position
};




struct CObjNorm : public CObjVert
{
	CObjNorm( float _x, float _y, float _z ): CObjVert(_x,_y,_z){}

  void Normalize();
};




struct CObjCoor
{
  float s,t;

	//! compare operator
  bool operator == (CObjCoor & other)
  {
    if(s == other.s)
    if(t == other.t)
      return(true);

    return(false);
  }
};





struct CObjFace
{
  CObjFace() 
	{ 
		v[0]=v[1]=v[2]=0;
		t[0]=t[1]=t[2]=0;
		n[0]=n[1]=n[2]=0;
		shader_id=0;
#ifdef USE_SMOOTH_TANGENTSPACE_BY_SMOOTHINGGROUPS
		m_smoothinggroup=0;
#endif
	}
  
  int v[3];										//!< 3 position indices
  int t[3];										//!< 3 texture indices
  int n[3];										//!< 3 normal indices
  int shader_id;							//!< material identifier index into m_pMaterials
#ifdef USE_SMOOTH_TANGENTSPACE_BY_SMOOTHINGGROUPS
	DWORD m_smoothinggroup;			//!< smoothing group
#endif
};



class CSimpleMaterial
{
public:

	//! constructor
	CSimpleMaterial()
	{
		m_iMaterialID=-1;
		m_AmbientColor[0]=1.0f;
		m_AmbientColor[1]=1.0f;
		m_AmbientColor[2]=1.0f;
		m_DiffuseColor[0]=1.0f;
		m_DiffuseColor[1]=1.0f;
		m_DiffuseColor[2]=1.0f;
		m_SpecularColor[0]=1.0f;
		m_SpecularColor[1]=1.0f;
		m_SpecularColor[2]=1.0f;
		m_fShininess=0.0f;
		m_fShiniStrength=0.0f;
		m_fSelfIllum=0.0f;
		m_dwFlags=0;
	}

	int						m_iMaterialID;					//!< material identifier (-1 is unset)
	std::string		m_sMaterialName;				//!< Material name

	float					m_AmbientColor[3];			//!< [0]=R [1]=G [2]=B
	float					m_DiffuseColor[3];			//!< [0]=R [1]=G [2]=B
	float					m_SpecularColor[3];			//!< [0]=R [1]=G [2]=B

	float					m_fShininess;						//!< like in MAX
	float					m_fShiniStrength;				//!< like in MAX
	float					m_fSelfIllum;						//!< like in MAX
	float					m_fTransparency;				//!< like in MAX

	std::string		m_sDiffuseTexturePath;	//!<
	std::string		m_sBumpTexturePath;			//!<

	DWORD					m_dwFlags;							//!< bit0: 1=marked as duplicate name, 0=unique
};	


class CSimpleIndexedMesh
{
public:

  // geometry data (alloceted with malloc)
  CObjFace *				m_pFaces;					// face (triangle) indices to m_pVerts,m_pCoors,m_pNorms, smoothing group information, shader_id
  CObjVert *				m_pVerts;					// vertex position
  CObjCoor *				m_pCoors;					// uv coordiantes
  CObjNorm *				m_pNorms;					// normals

	// allocated with new
	CSimpleMaterial *	m_pMaterials;			// materials [shader_id], may be 0

  int								m_FaceCount;			// face (triangle) indices to m_pVerts,m_pCoors,m_pNorms, smoothing group information, shader_id
  int								m_VertCount;			// vertex position
  int								m_CoorCount;			// uv coordiantes
  int								m_NormCount;			// normals
	int								m_MaterialCount;	// material
	

  //! constructor
	CSimpleIndexedMesh()
	{
		// init members
		m_pFaces=0;
		m_pVerts=0;
		m_pCoors=0;
		m_pNorms=0;
		m_pMaterials=0;
		m_FaceCount=0;
		m_VertCount=0;
		m_CoorCount=0;
		m_NormCount=0;
		m_MaterialCount=0;
	}

	//! free data 
	void FreeData( void )
	{
		free(m_pFaces);
		free(m_pVerts);
		free(m_pCoors);
		free(m_pNorms);
		delete [] m_pMaterials;

		m_pFaces=0;
		m_pVerts=0;
		m_pCoors=0;
		m_pNorms=0;
		m_pMaterials=0;
		m_FaceCount=0;
		m_VertCount=0;
		m_CoorCount=0;
		m_NormCount=0;
		m_MaterialCount=0;
	}

	//! destructor
	~CSimpleIndexedMesh()
	{
		FreeData();
	}

private:

	//! /param indwTriID1 number of the first triangle
	//! /param indwTriID2 number of the second triangle
	//! /param outiSide1[2] the normal number if there is at least one vertex that is shared, -1 otherwise
	//! /param outiSide2[2] the other normal number if there are at least two vertices that are shared, -1 otherwise
	void GetVerticesBetweenTriangles( const DWORD indwTriID1, const DWORD indwTriID2, int outiSide1[2], int outiSide2[2] )
	{
		outiSide1[0]=-1;outiSide1[1]=-1;
		outiSide2[0]=-1;outiSide2[1]=-1;

		// from tri1 to tri2
		for(DWORD dwSide1=0;dwSide1<3;dwSide1++)
		{
			DWORD dwVertexNo1=m_pFaces[indwTriID1].v[dwSide1];

			for(DWORD dwSide2=0;dwSide2<3;dwSide2++)
			{
				if(dwVertexNo1==m_pFaces[indwTriID2].v[dwSide2])
				{
					if(outiSide1[0]==-1)outiSide1[0]=m_pFaces[indwTriID1].n[dwSide1];
						else outiSide1[1]=m_pFaces[indwTriID1].n[dwSide1];

					if(outiSide2[0]==-1)outiSide2[0]=m_pFaces[indwTriID2].n[dwSide2];
						else outiSide2[1]=m_pFaces[indwTriID2].n[dwSide2];
				}
			}
		}
	}

public:

	//! /param indwTriID1
	//! /param indwTriID2
	//! /return
	bool IsSmoothBetweenTri( const DWORD indwTriID1, const DWORD indwTriID2 )
	{
		if(!m_pFaces)return(false);

#ifdef USE_SMOOTH_TANGENTSPACE_BY_SMOOTHINGGROUPS
		return((m_pFaces[indwTriID1].m_smoothinggroup & m_pFaces[indwTriID2].m_smoothinggroup)!=0);
#else // USE_SMOOTH_TANGENTSPACE_BY_SMOOTHINGGROUPS
		if(indwTriID1==indwTriID2)return(true);

		if(!m_pNorms)
			return(false);

		int iSide1[2],iSide2[2];

		GetVerticesBetweenTriangles(indwTriID1,indwTriID2,iSide1,iSide2);

		if(iSide1[0]!=-1)
		{
			assert(iSide2[0]!=-1);

			if(m_pNorms[iSide1[0]]!=m_pNorms[iSide2[0]])
				return(false);

			if(iSide1[1]!=-1)
			{
				assert(iSide2[1]!=-1);

				if(m_pNorms[iSide1[1]]!=m_pNorms[iSide2[1]])
					return(false);
			}
		}

		return(true);

#endif	// USE_SMOOTH_TANGENTSPACE_BY_SMOOTHINGGROUPS
	}

private:
	static const char *RemovePath( const char *inszPathName )
	{
		const char *pEnd=inszPathName;
		while(*inszPathName!=0)
		{
			if(*inszPathName=='\\')pEnd=inszPathName+1;
			inszPathName++;
		}

		return(pEnd);
	}

public:

	// stupid but simple implementation
	void MarkDuplicatedMaterialNames( void )
	{
		for(int i=0;i<m_MaterialCount;i++)
			for(int e=0;e<i;e++)			// was already in array?
				if(m_pMaterials[i].m_sMaterialName==m_pMaterials[e].m_sMaterialName)
				{
					m_pMaterials[i].m_dwFlags|=1;		// mark as duplicate
					break;
				}
	}


	// currently smoothing groups and material ID's are not supported,
	// vertex position, vertex normal and uv texturing is supported
	bool ExportAsOBJ( const char *inszOBJPathName )
	{
		MarkDuplicatedMaterialNames();	// is neccessary because material IDs may use the same Material

		char szMATPathName[1024];

		// build .mtl filename
		{
			int len=(int)strlen(inszOBJPathName);

			if(len>=1024)return(false);			// filename too long
			if(len<=3)return(false);			// filename too short

			strcpy(szMATPathName,inszOBJPathName);
			szMATPathName[len-3]='m';
			szMATPathName[len-2]='t';
			szMATPathName[len-1]='l';
		}
		
		// write material description
		{
			FILE *hOut=fopen(szMATPathName,"w");		if(!hOut)return(false);

			fprintf(hOut,"# SimpleIndexedMesh::ExportAsOBJ()\n#\n");
			if(!m_pMaterials)
			{
/*
				Ns = Phong specular component. Ranges from 0 to 1000. (I've seen various statements about this range (see below))
				Kd = Diffuse color weighted by the diffuse coefficient. (1,1,1) is white
				Ka = Ambient color weighted by the ambient coefficient. (1,1,1) is white
				Ks = Specular color weighted by the specular coefficient. (1,1,1) is white
				d = Dissolve factor (pseudo-transparency). Values are from 0-1. 0 is completely transparent, 1 is opaque.
				Ni = Refraction index. Values range from 1 upwards. A value of 1 will cause no refraction. A higher value implies refraction.
				illum = (0, 1, or 2) 0 to disable lighting, 1 for ambient & diffuse only (specular color set to black), 2 for full lighting (see below)
				sharpness = ? (see below)
				map_Kd = Diffuse color texture map.
				map_Ks = Specular color texture map.
				map_Ka = Ambient color texture map.
				map_Bump = Bump texture map.
				map_d = Opacity texture map.
				refl = reflection type and filename (?)
*/

				fprintf(hOut,"#\n");
				fprintf(hOut,"newmtl MatName\n");
				fprintf(hOut,"Ka  0.4 0.7 0.4\n");
				fprintf(hOut,"Kd  0.4 0.7 0.4\n");
				fprintf(hOut,"Ks  0.5 0.6 0.5\n");
				fprintf(hOut,"d  1\n");	
				fprintf(hOut,"Ns  0\n");
				fprintf(hOut,"illum 2\n");
				fprintf(hOut,"map_Kd texture.tga\n");
				fprintf(hOut,"#\n");
			}
			else
			{
				for(int i=0;i<m_MaterialCount;i++)
				if((m_pMaterials[i].m_dwFlags&1) == 0)				// not marked as duplicate name
				{
					fprintf(hOut,"#\n");
					fprintf(hOut,"newmtl %s\n",m_pMaterials[i].m_sMaterialName.c_str());
					fprintf(hOut,"Ka  %.3f %.3f %.3f\n",m_pMaterials[i].m_AmbientColor[0],m_pMaterials[i].m_AmbientColor[1],m_pMaterials[i].m_AmbientColor[2]);
					fprintf(hOut,"Kd  %.3f %.3f %.3f\n",m_pMaterials[i].m_DiffuseColor[0],m_pMaterials[i].m_DiffuseColor[1],m_pMaterials[i].m_DiffuseColor[2]);
					fprintf(hOut,"Ks  %.3f %.3f %.3f\n",m_pMaterials[i].m_SpecularColor[0]*m_pMaterials[i].m_fShiniStrength,
																							m_pMaterials[i].m_SpecularColor[1]*m_pMaterials[i].m_fShiniStrength,
																							m_pMaterials[i].m_SpecularColor[2]*m_pMaterials[i].m_fShiniStrength);
					fprintf(hOut,"d  %.3f\n",m_pMaterials[i].m_fTransparency);
					fprintf(hOut,"Ns  %.3f\n",m_pMaterials[i].m_fShininess);
					fprintf(hOut,"illum 2\n");
					if(!m_pMaterials[i].m_sDiffuseTexturePath.empty())fprintf(hOut,"map_Kd %s\n",RemovePath(m_pMaterials[i].m_sDiffuseTexturePath.c_str()));
					if(!m_pMaterials[i].m_sBumpTexturePath.empty())fprintf(hOut,"bump %s\n",RemovePath(m_pMaterials[i].m_sBumpTexturePath.c_str()));
					fprintf(hOut,"#\n");
				}
			}

			fprintf(hOut,"#\n# EOF\n");
			fclose(hOut);
		}

		FILE *hOut=fopen(inszOBJPathName,"w");		if(!hOut)return(false);

		fprintf(hOut,"# SimpleIndexedMesh::ExportAsOBJ()\n#\n");
		fprintf(hOut,"mtllib ./%s\n\n",RemovePath(szMATPathName));

		for(int v=0;v<m_VertCount;v++)
		{
			fprintf(hOut,"v %.4f %.2f %.2f\n",m_pVerts[v].x,m_pVerts[v].y,m_pVerts[v].z);
		}
		if(m_VertCount)fprintf(hOut,"# %d vertices\n\n",m_VertCount);

		for(int c=0;c<m_CoorCount;c++)
		{
			fprintf(hOut,"vt %.4f %.4f 0\n",m_pCoors[c].s,m_pCoors[c].t);
		}
		if(m_CoorCount)fprintf(hOut,"# %d texture vertices\n\n",m_CoorCount);

		for(int n=0;n<m_NormCount;n++)
		{
			fprintf(hOut,"vn %.3f %.3f %.3f\n",m_pNorms[n].x,m_pNorms[n].y,m_pNorms[n].z);
		}
		if(m_NormCount)fprintf(hOut,"# %d vertex normals\n\n",m_NormCount);
	
		int iPassCount=m_MaterialCount;			if(!iPassCount)iPassCount=0;

		for(int iCurrentMat=0;iCurrentMat<iPassCount;iCurrentMat++)
		{
			if(m_MaterialCount)
				fprintf(hOut,"usemtl %s\n",m_pMaterials[iCurrentMat].m_sMaterialName.c_str());
			 else
				fprintf(hOut,"usemtl MatName\n");

			for(int f=0;f<m_FaceCount;f++)
			{
				if(m_MaterialCount) if(iCurrentMat!=m_pFaces[f].shader_id)continue;

				fprintf(hOut,"f ");

				for(int d=0;d<3;d++)
				{
					assert(m_pVerts);
					
					fprintf(hOut,"%d",m_pFaces[f].v[d]+1);
					if(m_pCoors)fprintf(hOut,"/%d",m_pFaces[f].t[d]+1);
					if(m_pNorms)fprintf(hOut,"/%d",m_pFaces[f].n[d]+1);

					fprintf(hOut," ");
				}
				fprintf(hOut,"\n");
			}
		}

		fprintf(hOut,"# %d faces\n\n",m_FaceCount);

		fclose(hOut);
		return(true);
	}

	void Debug( void ) const
	{
		char str[80];

		OutputDebugString("CSimpleIndexedMesh Debug=\n{\n");

		for(int v=0;v<m_VertCount;v++)
		{
			sprintf(str,"  %d. Pos:(%.2f,%.2f,%.2f)\n",v,m_pVerts[v].x,m_pVerts[v].y,m_pVerts[v].z);
			OutputDebugString(str);
		}

		for(int c=0;c<m_CoorCount;c++)
		{
			sprintf(str,"  %d. UV:(%.2f,%.2f)\n",c,m_pCoors[c].s,m_pCoors[c].t);
			OutputDebugString(str);
		}

		for(int n=0;n<m_NormCount;n++)
		{
			sprintf(str,"  %d. Norm:(%.2f,%.2f,%.2f)\n",n,m_pNorms[n].x,m_pNorms[n].y,m_pNorms[n].z);
			OutputDebugString(str);
		}

		for(int f=0;f<m_FaceCount;f++)
		{
#ifdef USE_SMOOTH_TANGENTSPACE_BY_SMOOTHINGGROUPS
			sprintf(str,"  %d. Face %d/%d/%d %d/%d/%d %d/%d/%d M:%d SG:%p\n",f,
																										m_pFaces[f].v[0]+1,m_pFaces[f].t[0]+1,m_pFaces[f].n[0]+1,
																										m_pFaces[f].v[1]+1,m_pFaces[f].t[1]+1,m_pFaces[f].n[1]+1,
																										m_pFaces[f].v[2]+1,m_pFaces[f].t[2]+1,m_pFaces[f].n[2]+1,m_pFaces[f].shader_id,m_pFaces[f].m_smoothinggroup);
#else
			sprintf(str,"  %d. Face %d/%d/%d %d/%d/%d %d/%d/%d M:%d\n",f,
																										m_pFaces[f].v[0]+1,m_pFaces[f].t[0]+1,m_pFaces[f].n[0]+1,
																										m_pFaces[f].v[1]+1,m_pFaces[f].t[1]+1,m_pFaces[f].n[1]+1,
																										m_pFaces[f].v[2]+1,m_pFaces[f].t[2]+1,m_pFaces[f].n[2]+1,m_pFaces[f].shader_id);
#endif
			OutputDebugString(str);
		}

		OutputDebugString("}\n");
	}
};


#endif // __SIMPLEINDEXEDMESH2_H
