#include "stdafx.h"														// for precompiled headers (has to be in first place)
#include "3DApplication.h"										// C3DApplication
#include "3DTargetOrientation.h"							// C3DTargetOrientation
#include "Error.h"														// Error, Message
#include "resource.h"													// IDD_..
#include "windowsx.h"													// Button_SetCheck()
#include "commdlg.h"													// OPENFILENAME, GetOpenFileName


// constructor
C3DApplication::C3DApplication( void )
{
	m_View[0].m_eRenderMode=eRM_PolybumpSpecPixelShader_CM;
	m_View[0].m_bObjectSpace=false;
	m_View[1].m_eRenderMode=eRM_PolybumpSpecPixelShader_CM;
	m_View[1].m_bObjectSpace=false;
	m_bSplitWindowMode=false;
	m_hAppInstance=0;
	m_bAppQuiting=false;
	m_hInputDataDialog=0;
	m_bLightInitalized=false;
}


// destructor
C3DApplication::~C3DApplication( void )
{
} 


// initialization
bool C3DApplication::Init( HWND inhHWND, HINSTANCE inhInstance, const char *inszCommandLine )
{
	// double click on .obj to load it
	if(inszCommandLine[0]!=0)
	{
		if(MessageBox(0,"Do you want to use a object-space normalmap?\n\n(otherwise tangent-space = blue normaltexture, or clone-space)","Question",MB_YESNO)==IDYES)
			m_View[0].m_bObjectSpace=true;
		 else			
			m_View[0].m_bObjectSpace=false;
	}

#ifndef _DEBUG
	{
		m_SplashWindow.Create(inhInstance,inhHWND,IDB_SPLASHWINDOW,4000);								// splash window for 4 sec

		RunMessageLoop();																																// to paint the splash window
	}
#endif

	{
		char path_buffer[_MAX_PATH];
		char drive[_MAX_DRIVE];
		char dir[_MAX_DIR];
		char fname[_MAX_FNAME];
		char ext[_MAX_EXT];
		
		if(GetModuleFileName(NULL,path_buffer,_MAX_PATH)==0)
		{
				Error.Add("GetModuleFileName failed");
				return(false);
		}


		_splitpath( path_buffer, drive, dir, fname, ext );
		_makepath( path_buffer, drive, dir, 0, 0 );
		
		m_sExeFilePath=path_buffer;
	}

	char sCurrentPath[_MAX_PATH];

	GetCurrentDirectory(_MAX_PATH,sCurrentPath);
	SetCurrentDirectory(m_sExeFilePath.c_str());

	m_hAppInstance=inhInstance;
	m_hAccel=LoadAccelerators(m_hAppInstance,MAKEINTRESOURCE(IDR_ACCELERATORS));			// accelerators

	if(!m_Renderer.Init(inhHWND))																										// Renderer
	{
		Error.Add("m_Renderer.Init failed");
		return(false);
	}

//	m_View[0].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\foot\\foot.obj";
//	m_View[1].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\foot\\foot.obj";
//	m_View[0].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\box_reference\\test.obj";
//	m_View[0].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\Reference Objects\\generator\\generator.obj";
// 	m_View[1].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\foot_low\\foot_low.obj";

//	m_View[0].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\clone_low_tang\\clone.obj";
//	m_View[0].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\clone_low_tang_mirr\\clone.obj";
//	m_View[0].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\foot_low\\foot_low.obj";
//	m_View[0].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\foot\\foot.obj";
//	m_View[1].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\cube.obj";
//	m_View[0].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\foot_low_tang\\foot_low.obj";
//	m_View[1].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\foot_low_tang\\foot_low.obj";
//	m_View[1].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\foot\\foot.obj";
//	m_View[1].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\clone\\clone.obj";
//	m_View[1].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\clone_low\\clone.obj";
//	m_View[1].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\axis\\axis.obj";
//	m_View[1].m_OBJName="C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\box.obj";
//	"C:\\Documents and Settings\\martin.CRYTEK\\Desktop\\OBJ Fileformat for Polybump\\foot.obj"

	SetCurrentDirectory(sCurrentPath);


	// double click on .obj to load it
	if(inszCommandLine[0]!=0)
	if(strlen(inszCommandLine)+1<_MAX_PATH)
	{
		char path_buffer[_MAX_PATH];

		if(inszCommandLine[0]=='"')
		{
			strcpy(path_buffer,&(inszCommandLine[1]));

			char *ptr=strstr(path_buffer,"\"");
			*ptr=0;
		}
		else
		{
			strcpy(path_buffer,inszCommandLine);
		}

		m_View[0].m_OBJName=path_buffer;
//		m_View[1].m_OBJName=path_buffer;
	}

	CreateLights();

	ReloadObjects();
	
	return(true);
}



void C3DApplication::ReloadObjects( void )
{
	m_View[0].m_Scene.FreeObjects();
	m_View[1].m_Scene.FreeObjects();

	C3DObject *obj1=m_View[0].m_Scene.LoadOBJAndAdd(m_Renderer,m_View[0].m_OBJName.c_str(),m_View[0].m_bObjectSpace);
//	if(!obj1)	Error.Add("LoadOBJAndAdd(1) failed");

	{
		DWORD dwF=0,dwV=0; 
		if(obj1)obj1->GetStats(dwF,dwV);
//		m_View[0].m_Scene.SetStats(dwF,dwV);
	}

	C3DObject *obj2=m_View[1].m_Scene.LoadOBJAndAdd(m_Renderer,m_View[1].m_OBJName.c_str(),m_View[1].m_bObjectSpace);
//	if(!obj2)	Error.Add("LoadOBJAndAdd(2) failed");

	{
		DWORD dwF=0,dwV=0; 
		if(obj2)obj2->GetStats(dwF,dwV);
//		m_View[1].m_Scene.SetStats(dwF,dwV);

		if(dwF!=0)
			m_bSplitWindowMode=true;
	}

	if(obj1==0 && obj2==0)return;

	Command_ViewCenter();

	if(!m_bLightInitalized) Command_ResetLighting(); m_bLightInitalized=true;
}


// render everything
void C3DApplication::Render( void )
{
	char szText[1024];

	float fAspectRatio=m_bSplitWindowMode?0.5f:1.0f;
	HRESULT hRes;
	IDirect3DDevice8 *pDev=m_Renderer.GetDirectXDevice();			assert(pDev);

	D3DVIEWPORT8 ViewPort;
	DWORD dwWidth,dwHeight;

	m_Renderer.GetBackbufferSize(dwWidth,dwHeight);

	if(!dwWidth)return;		// minimzed ?
	if(!dwHeight)return;

	// left scene
	ViewPort.Width=dwWidth;ViewPort.Height=dwHeight;ViewPort.X=0;ViewPort.Y=0;ViewPort.MinZ=0.0f;ViewPort.MaxZ=1.0f;

	if(m_bSplitWindowMode)ViewPort.Width=dwWidth/2;


	// left/single
	{
		hRes=pDev->SetViewport(&ViewPort);
		if(FAILED(hRes))
			Error.AddDirectX("SetViewport failed: ",hRes);

		m_Renderer.SetRenderMode(m_View[0].m_eRenderMode);

		m_Renderer.BeginRender(m_Camera,fAspectRatio);
			{
					DWORD dwF,dwV;
					m_View[0].m_Scene.GetStats(dwF,dwV);
					sprintf(szText,"%d faces, %d vertices, %s",dwF,dwV,m_Renderer.AskShaderName());
			}
			m_View[0].m_Scene.Render(m_Renderer);
			m_Renderer.DrawText(10,10,szText);
		m_Renderer.EndRender();
	}


	// right scene
	if(m_bSplitWindowMode)
	{
		ViewPort.X=dwWidth/2;
		ViewPort.Width=dwWidth-dwWidth/2;

		hRes=pDev->SetViewport(&ViewPort);
		if(FAILED(hRes))
			Error.AddDirectX("SetViewport failed: ",hRes);

		m_Renderer.SetRenderMode(m_View[1].m_eRenderMode);

		m_Renderer.BeginRender(m_Camera,fAspectRatio);
			{
					DWORD dwF,dwV;
					m_View[1].m_Scene.GetStats(dwF,dwV);
					sprintf(szText,"%d faces, %d vertices, %s",dwF,dwV,m_Renderer.AskShaderName());
			}
			m_View[1].m_Scene.Render(m_Renderer);
			m_Renderer.DrawText(10,10,szText);
		m_Renderer.EndRender();

		// black line between both views (could be a performance loss)
		D3DRECT Rect;

		Rect.x1=dwWidth/2-1;Rect.x2=dwWidth/2+1;Rect.y1=0;Rect.y2=dwHeight;
		
		hRes=pDev->Clear( 1, &Rect, D3DCLEAR_TARGET, D3DCOLOR_XRGB(0,0,0), 1.0f, 0 );
		if(FAILED(hRes))Error.AddDirectX("Clear(2) failed: ",hRes);
	}


	DWORD ms100=m_Renderer.GetBlendedFrameTime100();


	sprintf(szText,"         %.2f ms %d fps",ms100*0.01f,100000/(ms100+1));

	m_Renderer.Present(szText);
}


// deinitialize
void C3DApplication::DeInit( void )
{
	SetCurrentDirectory(m_sExeFilePath.c_str());

	m_View[0].m_Scene.FreeObjects();																	// Scene
	m_View[1].m_Scene.FreeObjects();
	m_View[0].m_Scene.FreeLights();																		// Scene
	m_View[1].m_Scene.FreeLights();
	
	m_Renderer.DeInit();																							// Renderer
}


void C3DApplication::MouseMessage( DWORD indwButtons, int iniRelX , int iniRelY, int iniRelZ )
{
	int iCount=1;

	if(indwButtons&MK_CONTROL)iCount=2;
	if(indwButtons&MK_SHIFT)iCount=2;

	for(int i=0;i<iCount;i++)
	{
		C3DTargetOrientation *refObject = &m_Camera;
		
		// left and right mousebutton => middle mousebutton
		if(indwButtons&MK_LBUTTON)
			if(indwButtons&MK_RBUTTON)
			{
				indwButtons&=~MK_LBUTTON;
				indwButtons&=~MK_RBUTTON;
				indwButtons|=MK_MBUTTON;
			}

		if(indwButtons&MK_CONTROL) refObject=m_View[i].m_pLight;			// rotate primary light
		if(indwButtons&MK_SHIFT) refObject=m_View[i].m_pBackLight;		// rotate back light

		float x=(float)iniRelX;
		float y=(float)iniRelY;
		float z=(float)iniRelZ;

		// rotate with left mousebutton
		if(indwButtons&MK_LBUTTON)
		if((indwButtons&MK_RBUTTON)==0)
			refObject->RotateAroundTarget(-x*0.01f,-y*0.01f);

		D3DXVECTOR3 vRight,vUp,vIn;

		refObject->GetBase(vRight,vUp,vIn);

		// middle mousebutton has wheel like behaviour
		if(indwButtons&MK_MBUTTON)
			z=y*2.0f;

		// right mousebutton for strafing
		if(indwButtons&MK_RBUTTON)
		{
			D3DXVECTOR3 vFrom, vTo, vRel=-vRight*x*0.1f + vUp*y*0.1f;

			vFrom=refObject->GetFrom();
			vTo=refObject->GetTo();

			refObject->SetFrom(vFrom+vRel);
			refObject->SetTo(vTo+vRel);
		}

		// move in/out
		{
			D3DXVECTOR3 vFrom, vTo;
			float dist;

			vFrom=refObject->GetFrom();
			vTo=refObject->GetTo();

			dist=D3DXVec3Length(&(vFrom-vTo));

			dist+=z*0.5f;		if(dist<1.0)dist=1.0f;

			refObject->SetFrom(vTo-dist*vIn);
		}
	} 
}

void C3DApplication::CreateLights( void )
{
	m_View[0].m_pLight=m_View[0].m_Scene.AddLight();
	m_View[1].m_pLight=m_View[1].m_Scene.AddLight();

	if(false)			// used for no backlight
	{
		m_View[0].m_pBackLight=0;
		m_View[1].m_pBackLight=0;
	}
	else					//
	{
		m_View[0].m_pBackLight=m_View[0].m_Scene.AddLight();	assert(m_View[0].m_pBackLight);
		m_View[1].m_pBackLight=m_View[1].m_Scene.AddLight();	assert(m_View[1].m_pBackLight);
	}
}


void C3DApplication::Command_ResetLighting( void )
{
	// light colors *********************************

	m_View[0].m_Scene.SetAmbientColor(D3DXCOLOR(0.3f,0.3f,0.3f,0.0f));
	m_View[1].m_Scene.SetAmbientColor(D3DXCOLOR(0.3f,0.3f,0.3f,0.0f));

	m_View[0].m_pLight->SetColor(D3DXCOLOR(1,1,1,0));
	m_View[1].m_pLight->SetColor(D3DXCOLOR(1,1,1,0));

	if(m_View[0].m_pBackLight) m_View[0].m_pBackLight->SetColor(D3DXCOLOR(0.8f,0.5f,0.1f,0));
	if(m_View[1].m_pBackLight) m_View[1].m_pBackLight->SetColor(D3DXCOLOR(0.8f,0.5f,0.1f,0));

	// light pos *************************************

	D3DXVECTOR3 mid;
	float fRad=GetBoundingSphere(mid);	

	D3DXVECTOR3 vDirection(40,-30,40);

	D3DXVec3Normalize(&vDirection,&vDirection);
	vDirection*=fRad*1.5f;

	m_View[0].m_pLight->SetFrom(mid+vDirection);
	m_View[0].m_pLight->SetTo(mid);
	m_View[1].m_pLight->SetFrom(mid+vDirection);
	m_View[1].m_pLight->SetTo(mid);

	vDirection = D3DXVECTOR3(-60,-30,40);
	D3DXVec3Normalize(&vDirection,&vDirection);
	vDirection*=fRad*1.5f;

	if(m_View[0].m_pBackLight)
	{
		m_View[0].m_pBackLight->SetFrom(mid+vDirection);
		m_View[0].m_pBackLight->SetTo(mid);
	}

	if(m_View[1].m_pBackLight)
	{
		m_View[1].m_pBackLight->SetFrom(mid+vDirection);
		m_View[1].m_pBackLight->SetTo(mid);
	}
}

float	C3DApplication::GetBoundingSphere( D3DXVECTOR3 &outvMid ) const
{
	D3DXVECTOR3 min=D3DXVECTOR3(0,0,0),max=D3DXVECTOR3(0,0,0);

	if(m_View[0].m_Scene.GetBoundingBox(min,max))
	{
		D3DXVECTOR3 min1,max1;
		if(m_View[1].m_Scene.GetBoundingBox(min1,max1))
		{
			D3DXVec3Minimize(&min,&min1,&min);
			D3DXVec3Maximize(&max,&max1,&max);
		}
	}
	else
	{
		m_View[1].m_Scene.GetBoundingBox(min,max);
	}

	outvMid=(min+max)*0.5f;
	return( D3DXVec3Length(&(max-outvMid)) );
}


// move camera to fit scene
void C3DApplication::Command_ViewCenter( void )
{
	D3DXVECTOR3 mid;
	float fRad=GetBoundingSphere(mid);	

	m_Camera.SetTo(mid);
	m_Camera.SetFrom(mid+D3DXVECTOR3(0,fRad*3.5f,0));
}



void C3DApplication::Command_CycleRendermode( DWORD innViewNo, bool inbForward )
{
	assert(innViewNo==0 || innViewNo==1 || innViewNo==2);

	bool bSynconize=false;

	if(innViewNo==2)
	{
		innViewNo=0;
		bSynconize=true;
	}

	assert(innViewNo==0 || innViewNo==1);

	if(inbForward)
	{
		m_View[innViewNo].m_eRenderMode=(eRenderMode)((int)m_View[innViewNo].m_eRenderMode + 1);
		if(m_View[innViewNo].m_eRenderMode==eRM_Endmarker)m_View[innViewNo].m_eRenderMode=eRM_Wireframe;
	}
	else
	{
		if(m_View[innViewNo].m_eRenderMode==eRM_Wireframe)m_View[innViewNo].m_eRenderMode=(eRenderMode)((int)eRM_Endmarker-1);
			else m_View[innViewNo].m_eRenderMode=(eRenderMode)((int)m_View[innViewNo].m_eRenderMode - 1);
	}

	if(bSynconize)
	{
		m_View[1].m_eRenderMode=m_View[0].m_eRenderMode;
	}
}






// WndProc for preferences dialog
BOOL CALLBACK WndProcPreferencesDialog( HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam )
{
	HWND hMainWnd=GetParent(hWnd);

	C3DApplication *application=(C3DApplication *)GetWindowLong(hMainWnd,GWL_USERDATA);

 	switch (msg)
  {
		case WM_INITDIALOG:
			{
				SetWindowText(GetDlgItem(hWnd,IDC_FILENAME_LEFT),application->m_View[0].m_OBJName.c_str());
				SetWindowText(GetDlgItem(hWnd,IDC_FILENAME_RIGHT),application->m_View[1].m_OBJName.c_str());
				Button_SetCheck(GetDlgItem(hWnd,IDC_OBJECTSPACE_LEFT),application->m_View[0].m_bObjectSpace);
				Button_SetCheck(GetDlgItem(hWnd,IDC_OBJECTSPACE_RIGHT),application->m_View[1].m_bObjectSpace);
			}
			return(0);

		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
				case IDC_CHOOSEFILE_LEFT:
					{
						string sName;
						char str[1024];

						str[0]=0;GetWindowText(GetDlgItem(hWnd,IDC_FILENAME_LEFT),str,1024);sName=str;

						if(application->ChooseOBJFilename(sName))	SetWindowText(GetDlgItem(hWnd,IDC_FILENAME_LEFT),sName.c_str());
					}
					return(TRUE);

				case IDC_CHOOSEFILE_RIGHT:
					{
						string sName;
						char str[1024];

						str[0]=0;GetWindowText(GetDlgItem(hWnd,IDC_FILENAME_RIGHT),str,1024);sName=str;

						if(application->ChooseOBJFilename(sName))	SetWindowText(GetDlgItem(hWnd,IDC_FILENAME_RIGHT),sName.c_str());
					}
					return(TRUE);

				case IDOK:
					{
						char str[1024];

						str[0]=0;GetWindowText(GetDlgItem(hWnd,IDC_FILENAME_LEFT),str,1024);
						application->m_View[0].m_OBJName=str;

						str[0]=0;GetWindowText(GetDlgItem(hWnd,IDC_FILENAME_RIGHT),str,1024);
						application->m_View[1].m_OBJName=str;

						application->m_View[0].m_bObjectSpace = Button_GetCheck(GetDlgItem(hWnd,IDC_OBJECTSPACE_LEFT))!=0;
						application->m_View[1].m_bObjectSpace = Button_GetCheck(GetDlgItem(hWnd,IDC_OBJECTSPACE_RIGHT))!=0;
						application->m_hInputDataDialog=0;
						DestroyWindow(hWnd);
						application->ReloadObjects();
					}
					return(TRUE);

				case IDCANCEL:
					application->m_hInputDataDialog=0;
					DestroyWindow(hWnd);
					return(TRUE);
			}
			return(0);
		
		case WM_CLOSE:
			application->m_hInputDataDialog=0;
			DestroyWindow(hWnd);
			return(TRUE);
    }

    return(FALSE);
}





bool C3DApplication::ChooseOBJFilename( string &inoutFilename ) const
{
	char back[1024];
	if(GetCurrentDirectory(1024,back)==0)
	{
		Error.Add("GetCurrentDirectory failed");
		return(false);
	}

	char filestr[1024];		strcpy(filestr,inoutFilename.c_str());

	OPENFILENAME ofn;

	ofn.lStructSize=sizeof(OPENFILENAME);
	ofn.hwndOwner=m_Renderer.GetWindowHWND();
	ofn.hInstance=m_hAppInstance;
	ofn.lpstrFilter="Alias Wavefront object (*.obj)\0*.obj\0All Files (*.*)\0*.*\0";
	ofn.lpstrCustomFilter = (LPTSTR)NULL;
	ofn.nMaxCustFilter = 0L;
	ofn.nFilterIndex = 1L;
	ofn.lpstrInitialDir=NULL;
	if(filestr[0]==0)ofn.lpstrInitialDir=back;
	ofn.lpstrTitle="Choose the input file";

	ofn.lpstrFileTitle=0;
	ofn.nMaxFileTitle=0;

	ofn.lpstrFile=filestr;
	ofn.nMaxFile=sizeof(filestr);

	ofn.lCustData=0;
	ofn.lpstrDefExt=".obj";
	ofn.Flags=OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST | OFN_LONGNAMES | OFN_ENABLESIZING;
	ofn.nFileOffset = 0;
	ofn.nFileExtension = 0;

	if(GetOpenFileName(&ofn))
	{
		inoutFilename=filestr;

		SetCurrentDirectory(back);
		return(true);
	}

	SetCurrentDirectory(back);

	return(false);
}


//
void C3DApplication::Command_InputData( void )
{
	if(m_hInputDataDialog)return;

	m_hInputDataDialog=CreateDialog(m_hAppInstance,MAKEINTRESOURCE(IDD_INPUTDATA),m_Renderer.GetWindowHWND(),WndProcPreferencesDialog);

	if(!m_hInputDataDialog)return;

	ShowWindow(m_hInputDataDialog,SW_SHOW);
}








void C3DApplication::RunMessageLoop( void )
{
	// The message loop.
	MSG msg; 

	while(PeekMessage(&msg,NULL,0,0,PM_REMOVE))
	{
		if (!TranslateAccelerator(m_Renderer.GetWindowHWND(),m_hAccel,&msg))											// check accelerators
    {
      TranslateMessage( &msg );
			DispatchMessage( &msg );
		}
	}

	if(msg.message == WM_QUIT ){	m_bAppQuiting=true;return; }


	HWND hWnd=m_Renderer.GetWindowHWND();


	// Idle 
	InvalidateRect(hWnd,0,false);
	UpdateWindow(hWnd);

	if(GetFocus()!=hWnd)
		Sleep(10);
}


//
bool C3DApplication::IsAppQuiting( void )
{
	return(m_bAppQuiting);
}


// get attribute
bool C3DApplication::GetSecondViewOnOff( void ) const
{
	return(m_bSplitWindowMode);
}

// set attribute
void C3DApplication::SetSecondViewOnOff( const bool inbValue )
{
	m_bSplitWindowMode=inbValue;
}



// get attribute
bool C3DApplication::GetForceDirLights( void ) const
{
	return(m_Renderer.GetForceDirLight());
}

// set attribute
void C3DApplication::SetForceDirLights( const bool inbValue )
{
	m_Renderer.SetForceDirLight(inbValue);
}


//
CAppView &C3DApplication::GetLeftView( void )
{
	return(m_View[0]);
}

//
CAppView &C3DApplication::GetRightView( void )
{
	return(m_View[1]);
}

//
BOOL CALLBACK AboutWindowProc( HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam )
{
	switch(Msg)
  {
		case WM_COMMAND:
			switch(LOWORD(wParam))
			{
				case IDOK:
					EndDialog(hWnd,0);
					return(TRUE);
			}
			return(0);
  }

	return(FALSE);
}
  

//
void C3DApplication::Command_AboutBox( void )
{
	DialogBox(m_hAppInstance,MAKEINTRESOURCE(IDD_SPLASHABOUT),m_Renderer.GetWindowHWND(),AboutWindowProc);
}


bool C3DApplication::ChooseRGBColor( D3DXCOLOR &inoutOldColor )
{
	CHOOSECOLOR dlg;
	static COLORREF Array[16]={ RGB(0,0,0),RGB(0,0,255),RGB(0,255,0),RGB(0,255,255),
															RGB(255,0,0),RGB(255,0,255),RGB(255,255,0),RGB(255,255,255),
															RGB(32,32,32),RGB(64,64,64),RGB(128,128,128),RGB(196,196,196),
															RGB(255,255,255),RGB(255,128,64),RGB(128,64,32),RGB(64,32,16)	};

	memset(&dlg,0,sizeof(CHOOSECOLOR));
	dlg.lStructSize=sizeof(CHOOSECOLOR);
	dlg.hwndOwner=m_Renderer.GetWindowHWND();
	dlg.rgbResult=RGB((int)(inoutOldColor.r*255.0f),(int)(inoutOldColor.g*255.0f),(int)(inoutOldColor.b*255.0f));
	dlg.Flags=CC_FULLOPEN | CC_RGBINIT | CC_ANYCOLOR;
	dlg.lpCustColors=Array;

  bool bRet=(ChooseColor(&dlg)!=0);

	if(bRet)
		inoutOldColor=D3DXCOLOR(GetRValue(dlg.rgbResult)/255.0f,GetGValue(dlg.rgbResult)/255.0f,GetBValue(dlg.rgbResult)/255.0f,0.0f);

	return(bRet);
}





void C3DApplication::Command_ChooseAmbientColor( void )
{
	D3DXCOLOR col = m_View[0].m_Scene.GetAmbientColor();

	ChooseRGBColor(col);

	m_View[0].m_Scene.SetAmbientColor(col);
	m_View[1].m_Scene.SetAmbientColor(col);
}


void C3DApplication::Command_ChooseMainLightColor( void )
{
	D3DXCOLOR col = m_View[0].m_pLight->GetColor();

	ChooseRGBColor(col);

	m_View[0].m_pLight->SetColor(col);
	m_View[1].m_pLight->SetColor(col);
}




void C3DApplication::Command_ChooseBackLightColor( void )
{
	if(!m_View[0].m_pBackLight)return;
	if(!m_View[1].m_pBackLight)return;

	D3DXCOLOR col = m_View[0].m_pBackLight->GetColor();

	ChooseRGBColor(col);

	m_View[0].m_pBackLight->SetColor(col);
	m_View[1].m_pBackLight->SetColor(col);
}




// get attribute
bool C3DApplication::GetVertexBaseVectorDisplay( void ) const
{
	return(m_Renderer.GetShowVertexBaseSpace());
}

// set attribute
void C3DApplication::SetVertexBaseVectorDisplay( const bool inbValue )
{
	m_Renderer.SetShowVertexBaseSpace(inbValue);
}



// get attribute
bool C3DApplication::GetTriBaseVectorDisplay( void ) const
{
	return(m_Renderer.GetShowTriBaseSpace());
}

// set attribute
void C3DApplication::SetTriBaseVectorDisplay( const bool inbValue )
{
	m_Renderer.SetShowTriBaseSpace(inbValue);
}


// get attribute
bool C3DApplication::GetNormalVectorDisplay( void ) const
{
	return(m_Renderer.GetShowNormals());
}

// set attribute
void C3DApplication::SetNormalVectorDisplay( const bool inbValue )
{
	m_Renderer.SetShowNormals(inbValue);
}


void C3DApplication::Command_ChangeDebugScale( const float infRelative )
{
	m_Renderer.SetDebugScale(m_Renderer.GetDebugScale()+infRelative);
}
