#include "stdafx.h"											// for precompiled headers (has to be in first place)
#include "3drenderer.h"									// C3DRenderer
#include <d3d8.h>												// IDirect3D8
#include "3DTargetOrientation.h"				// C3DTargetOrientation
#include <stdio.h>											// sprintf()
#include "Error.h"											// Error, Message
#include <assert.h>											// assert()
#include <D3dx8math.h>									// D3DXMATRIX
#include "ProcessorClock.h"							// CProcessorClock



// #define USE_REFERENCERASTERIZER		//



// constructor
C3DRenderer::C3DRenderer( void )
{
	m_hWindow=0;																		// window handle
	m_pD3D=0;																				// Direct3D interface
	m_pd3dDevice=0;																	// Direct3DDevice interface
	m_eCurrentRenderMode=eRM_TexturedFullbright;		// rendermode textured
	m_BackbufferWidth=0;
	m_BackbufferHeight=0;
	m_dwBlendedFrameTime100=0;
	m_dwBlendedSamples=0;
	m_dwBlendedSum100=0;
	m_bShowVertexBaseSpace=false;
	m_bShowTriBaseSpace=false;
	m_bShowNormals=false;
	m_pd3dFont=0;
	m_bForceDirLight=false;
	m_fDebugScale=1.0f;
}



// destructor
C3DRenderer::~C3DRenderer( void )
{
	//  don't forget to free this
	assert(!m_pd3dDevice);
	assert(!m_pD3D);
	assert(!m_pd3dFont);
}


















void C3DRenderer::BeginRender( const C3DTargetOrientation &inCamera, float infAspectRatio )
{
	assert(m_pd3dDevice);
	if(m_pd3dDevice->TestCooperativeLevel()!=D3D_OK)		// in devicelost only Reset and TestCooperativeLevel is allowed
		return;

	assert(m_BackbufferHeight);
	assert(m_BackbufferWidth);

	infAspectRatio*=(float)m_BackbufferWidth/(float)m_BackbufferHeight;



	HRESULT hRes;

	// Direct3D applies the matrices to the scene in the following order: (1) World ,(2) View, (3) Projection

	D3DXMATRIX matWorld,matView,matProj;

	D3DXMatrixIdentity(&matWorld);
	inCamera.GetViewTransform(matView);
	D3DXMatrixPerspectiveFovLH( &matProj, D3DX_PI/4, infAspectRatio, 1.0f, 10000.0f );	// fov,aspect ratio,znear,zfar

	m_vEyePos=inCamera.GetFrom();
	m_vEyeDir=inCamera.GetDirection();

	D3DXMATRIX matTemp;
	D3DXMATRIX matWorldViewProj;
	D3DXMATRIX matWorldView;
	D3DXMATRIX matWorldInverse;

	D3DXMatrixMultiply(&matTemp, &matWorld, &matView);
	D3DXMatrixMultiply(&matWorldViewProj, &matTemp, &matProj);
	D3DXMatrixMultiply(&matWorldView, &matWorld, &matView);
	D3DXMatrixInverse(&matWorldInverse, NULL, &matWorld);
		
	// Projection to clip space
	D3DXMatrixTranspose(&m_mWVPTransform, &matWorldViewProj);

	// Transform to eye space
	D3DXMatrixTranspose(&m_mWVTransform, &matWorldView);

	// Transform to world space
	D3DXMatrixTranspose(&m_mWTransform, &matWorld);

	// timing start
	m_dwRenderStartTime=MyProcessorClock.GetActualTimeAccurate();

	// Clear the back buffer to a dark blue color
	hRes=m_pd3dDevice->Clear( 0, NULL, D3DCLEAR_TARGET | D3DCLEAR_ZBUFFER, D3DCOLOR_XRGB(0,0,128), 1.0f, 0 );
	if(FAILED(hRes))Error.AddDirectX("Clear(1) failed: ",hRes);

	// Begin the scene.
	hRes=m_pd3dDevice->BeginScene();
	if(FAILED(hRes))Error.AddDirectX("BeginScene failed: ",hRes);

	// wireframe
	m_pd3dDevice->SetRenderState(D3DRS_FILLMODE,AskWireframe() ? D3DFILL_WIREFRAME : D3DFILL_SOLID);

	m_pd3dDevice->SetRenderState(D3DRS_ZFUNC,D3DCMP_LESS);

#ifdef USE_SILHUETTECLIPPING		// this might cause problems with text rendering
	m_pd3dDevice->SetRenderState(D3DRS_ALPHAREF,202);											// 0..0xff
	m_pd3dDevice->SetRenderState(D3DRS_ALPHATESTENABLE,TRUE);
	m_pd3dDevice->SetRenderState(D3DRS_ALPHAFUNC,D3DCMP_LESSEQUAL);
#else
	m_pd3dDevice->SetRenderState(D3DRS_ALPHAREF,1); 
	m_pd3dDevice->SetRenderState(D3DRS_ALPHATESTENABLE,FALSE); 
  m_pd3dDevice->SetRenderState(D3DRS_ALPHAFUNC,D3DCMP_ALWAYS);
#endif

	m_pd3dDevice->SetRenderState(D3DRS_LIGHTING,FALSE);
	m_pd3dDevice->SetRenderState(D3DRS_DESTBLEND,D3DBLEND_ZERO);
	m_pd3dDevice->SetRenderState(D3DRS_ALPHABLENDENABLE,FALSE);
	m_pd3dDevice->SetRenderState(D3DRS_CULLMODE,D3DCULL_CW);
	m_pd3dDevice->SetRenderState(D3DRS_SPECULARENABLE,0);

	// bilinear filtering
	m_pd3dDevice->SetTextureStageState(0,D3DTSS_MAGFILTER,D3DTEXF_LINEAR);
	m_pd3dDevice->SetTextureStageState(0,D3DTSS_MINFILTER,D3DTEXF_LINEAR);
	m_pd3dDevice->SetTextureStageState(1,D3DTSS_MAGFILTER,D3DTEXF_LINEAR);
	m_pd3dDevice->SetTextureStageState(1,D3DTSS_MINFILTER,D3DTEXF_LINEAR);
	m_pd3dDevice->SetTextureStageState(2,D3DTSS_MAGFILTER,D3DTEXF_LINEAR);
	m_pd3dDevice->SetTextureStageState(2,D3DTSS_MINFILTER,D3DTEXF_LINEAR);
	m_pd3dDevice->SetTextureStageState(3,D3DTSS_MAGFILTER,D3DTEXF_LINEAR);
	m_pd3dDevice->SetTextureStageState(3,D3DTSS_MINFILTER,D3DTEXF_LINEAR);
	m_pd3dDevice->SetTextureStageState(4,D3DTSS_MAGFILTER,D3DTEXF_LINEAR);
	m_pd3dDevice->SetTextureStageState(4,D3DTSS_MINFILTER,D3DTEXF_LINEAR);

	m_pd3dDevice->SetRenderState(D3DRS_ZENABLE,1);
	m_pd3dDevice->SetRenderState(D3DRS_ZWRITEENABLE,1);
	m_pd3dDevice->SetRenderState(D3DRS_SHADEMODE,D3DSHADE_GOURAUD);
}



void C3DRenderer::EndRender( void )
{
	assert(m_pd3dDevice);
	if(m_pd3dDevice->TestCooperativeLevel()!=D3D_OK)		// in devicelost only Reset and TestCooperativeLevel is allowed
		return;

	DrawText(10.0f,(float)(m_BackbufferHeight-25),"Hold Crtl or Shift and drag mouse to move the lightsources");
			
	HRESULT hRes;

	// End the scene.
	hRes=m_pd3dDevice->EndScene();
	if(FAILED(hRes))Error.AddDirectX("EndScene failed: ",hRes);
}




void C3DRenderer::Present( const char *inszStatusText )
{
	assert(m_pd3dDevice);
	if(m_pd3dDevice->TestCooperativeLevel()!=D3D_OK)		// in devicelost only Reset and TestCooperativeLevel is allowed
		return;

	HRESULT hRes;

	// Blit or DoubleBuffer switch
	hRes=m_pd3dDevice->Present( NULL, NULL, NULL, NULL );
	if(FAILED(hRes))Error.AddDirectX("Present failed: ",hRes);

	// timing end
	m_dwLastFrameTime100=MyProcessorClock.GetDifference100(m_dwRenderStartTime);

	// Blended frame time calculation (10 frames)
	m_dwBlendedSum100+=m_dwLastFrameTime100;
	m_dwBlendedSamples++;
	if(m_dwBlendedSamples>=10)
	{
		m_dwBlendedFrameTime100=m_dwBlendedSum100/m_dwBlendedSamples;
		m_dwBlendedSamples=0;m_dwBlendedSum100=0;
	}

	// ms display
	if(m_hWindow)
	{
		TITLEBARINFO Info;

		Info.cbSize=sizeof(TITLEBARINFO);

		GetTitleBarInfo(m_hWindow,&Info);

		DWORD dwWidth=Info.rcTitleBar.right-Info.rcTitleBar.left;
		DWORD dwHeight=Info.rcTitleBar.bottom-Info.rcTitleBar.top;

		HDC hdc=GetWindowDC(m_hWindow);

		SetBkColor(hdc,GetSysColor(COLOR_MENU));

		HFONT hfnt, hOldFont; 
 
    hfnt = (HFONT)GetStockObject(ANSI_VAR_FONT);

		if(hOldFont = (HFONT)SelectObject(hdc, hfnt)) 
    { 
			SetTextAlign(hdc,TA_RIGHT);
			TextOut(hdc,dwWidth,GetSystemMetrics(SM_CYFIXEDFRAME)+3+dwHeight,inszStatusText,strlen(inszStatusText));
      SelectObject(hdc, hOldFont); 
    } 

		ReleaseDC(m_hWindow,hdc);
	}
}









bool C3DRenderer::Init( HWND inhHWND )
{
	assert(inhHWND);
	m_hWindow=inhHWND;

	// Get Direct3d8
	if( NULL == ( m_pD3D = Direct3DCreate8( D3D_SDK_VERSION ) ) )
	{
		assert(D3D_SDK_VERSION==220);
		Error.Add("Direct3DCreate8 failed (DirectX 8.1 is neccessary)");
		return(false);
	}

	D3DDISPLAYMODE d3ddm;
	if( FAILED( m_pD3D->GetAdapterDisplayMode( D3DADAPTER_DEFAULT, &d3ddm ) ) )
	{
		Error.Add("GetAdapterDisplayMode failed");
		return(false);
	}

	D3DPRESENT_PARAMETERS d3dpp; 
	ZeroMemory( &d3dpp, sizeof(d3dpp) );
	d3dpp.Windowed   = TRUE;
	d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;		// D3DSWAPEFFECT_COPY_VSYNC
	d3dpp.BackBufferFormat = d3ddm.Format;
  d3dpp.EnableAutoDepthStencil = TRUE;
  d3dpp.AutoDepthStencilFormat = D3DFMT_D16;

#ifdef USE_REFERENCERASTERIZER
	// use reference rasterizer
	if(FAILED(m_pD3D->CreateDevice(D3DADAPTER_DEFAULT,D3DDEVTYPE_REF,m_hWindow,D3DCREATE_SOFTWARE_VERTEXPROCESSING,&d3dpp,&m_pd3dDevice)))
#else
	// try hardware vertexshader, if this failes, try software vertex shader
	if(FAILED(m_pD3D->CreateDevice(D3DADAPTER_DEFAULT,D3DDEVTYPE_HAL,m_hWindow,D3DCREATE_HARDWARE_VERTEXPROCESSING,&d3dpp,&m_pd3dDevice)))
	if(FAILED(m_pD3D->CreateDevice(D3DADAPTER_DEFAULT,D3DDEVTYPE_HAL,m_hWindow,D3DCREATE_SOFTWARE_VERTEXPROCESSING,&d3dpp,&m_pd3dDevice)))
#endif
	{
		Error.Add("CreateDevice failed (Do you have a 3D graphics card, has your desktop true color depth?)");
		return(false);
	}

	if(!AskForPixelShaderSupport())
	{
		Error.Add("Graphics hardware has no pixel shader support.");
		return(false);
	}

	// get width and height
	{
		D3DSURFACE_DESC desc;
		LPDIRECT3DSURFACE8 pBackBuffer;
		m_pd3dDevice->GetBackBuffer( 0, D3DBACKBUFFER_TYPE_MONO, &pBackBuffer );

		if(pBackBuffer)
		{
			pBackBuffer->GetDesc( &desc );
			pBackBuffer->Release();

			assert(desc.Width);
			assert(desc.Height);

			m_BackbufferWidth=desc.Width;
			m_BackbufferHeight=desc.Height;
		}
		else		// e.g. minimized window
		{
			m_BackbufferWidth=0;
			m_BackbufferHeight=0;
		}
	}

	InitFont();


	m_eCurrentRenderMode=eRM_TexturedFullbright;		// rendermode textured

	if(!m_ShaderManager.Init(*this))
		return(false);


	ShowWindow(m_hWindow,SW_SHOW);
	return(true);
}



// deinitialization
void C3DRenderer::DeInit( void )
{
	// delete all dynamically created objects

	if(m_pd3dDevice)
	{
		if(m_pd3dDevice->TestCooperativeLevel()==D3D_OK)		// in devicelost only Reset and TestCooperativeLevel is allowed
		{
			m_pd3dDevice->SetTexture(0,0);
			m_pd3dDevice->SetTexture(1,0);
			m_pd3dDevice->SetTexture(2,0);

			m_pd3dDevice->SetStreamSource(0,0,0);

			m_pd3dDevice->SetIndices(0,0);

			m_ShaderManager.DeInit(*this);

			DeInitFont();
		}

		m_pd3dDevice->Release();
		m_pd3dDevice=0;

		assert(!m_pd3dFont);
	}

	// Direct3D
  if(m_pD3D)
	{
		m_pD3D->Release();
		m_pD3D=0;
	}
}



// get attribute
IDirect3DDevice8 *C3DRenderer::GetDirectXDevice( void )
{
	return(m_pd3dDevice);
}



// set attribute
void C3DRenderer::SetRenderMode( const enum eRenderMode ineRenderMode )
{
	m_eCurrentRenderMode=ineRenderMode;
}


// get attribute
enum eRenderMode C3DRenderer::GetRenderMode( void )
{
	return(m_eCurrentRenderMode);
}

//! get attribute
HWND C3DRenderer::GetWindowHWND( void ) const
{
	return(m_hWindow);
}


bool C3DRenderer::AskWireframe( void )
{
	return(m_eCurrentRenderMode==eRM_Wireframe);
}


bool C3DRenderer::AskTextured( void )
{
	return(m_eCurrentRenderMode!=eRM_Wireframe 
			&& m_eCurrentRenderMode!=eRM_GouraudSpecular 
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
			&& m_eCurrentRenderMode!=eRM_OverdrawMeasure
			&& m_eCurrentRenderMode!=eRM_ShadeTangentSpaceProblems
#endif
			&& m_eCurrentRenderMode!=eRM_Gouraud);
}


DWORD C3DRenderer::AskPixelShader( const DWORD indwPass, const bool inbMissingAssets )
{
	assert(indwPass==1 || indwPass==2);

	switch(m_eCurrentRenderMode)
	{
		case eRM_PolybumpDiffusePixelShader:
		case eRM_PolybumpSpecPixelShader:
			if(inbMissingAssets)return(0);
			return(m_ShaderManager.m_dwPS_PolyBumpSpecDir);
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
		case eRM_ShadeTangentSpaceProblems:
			return(m_ShaderManager.m_dwPS_TDebug);
#endif
		case eRM_PolybumpSpecPixelShader_CM:
			if(inbMissingAssets)return(0);
			if(indwPass==1) return(m_ShaderManager.m_dwPS_PolyBumpDiffuseDir_CM);
				else return(m_ShaderManager.m_dwPS_PolyBumpSpecDir_CM);

		default:
			return(0);
	}
}

DWORD C3DRenderer::AskVertexShader( const DWORD indwPass, const bool inbMissingAssets )
{
	DWORD dwRet=m_ShaderManager.m_dwVS_Standard;

	if(m_eCurrentRenderMode==eRM_TexturedLit)return(dwRet);

	if(!inbMissingAssets)
	{
		if(GetForceDirLight())
			dwRet=m_ShaderManager.m_dwVS_PolyBumpDir;				// directional light calculation
		 else
			dwRet=m_ShaderManager.m_dwVS_PolyBumpPoint;			// point light calculation
	}
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
	if(m_eCurrentRenderMode==eRM_ShadeTangentSpaceProblems)
		dwRet=m_ShaderManager.m_dwVS_TDebug;
#endif
	return(dwRet);
}

// Should the material use the non pixel shader (dot3 = GF1,GF2,GF4MX) fallback
// (depends on the current rendermode)bool C3DRenderer::AskPolyBumpFallback( void )
bool C3DRenderer::AskPolyBumpFallback( void )
{
	return(m_eCurrentRenderMode==eRM_PolybumpDiffuseNoPixelShader);
}


const char *C3DRenderer::AskShaderName( void )
{
	switch(m_eCurrentRenderMode)
	{
		case eRM_Wireframe:
			return("Wireframe");break;
		case eRM_Gouraud:
			return("Gouraud");break;
		case eRM_GouraudSpecular:
			return("GouraudSpecular");break;
		case eRM_TexturedFullbright:
			return("TexturedFullbright");break;
		case eRM_TexturedLit:
			return("TexturedLit");break;
		case eRM_NormalmapFullbright:
			return("NormalmapFullbright");break;
		case eRM_PolybumpDiffuseNoPixelShader:
			return("PolybumpDiffuseNoPixelShader");break;
		case eRM_PolybumpDiffusePixelShader:
			return("PolybumpDiffusePixelShader");break;
		case eRM_PolybumpSpecPixelShader:
			return("PolybumpSpecPixelShader 1 Pass");break;
		case eRM_PolybumpSpecPixelShader_CM:
			return("PolybumpSpecPixelShader CM 2 Passes");break;
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
		case eRM_OverdrawMeasure:
			return("OverdrawMeasure");
		case eRM_ShadeTangentSpaceProblems:
			return("ShadeTangentSpaceProblems");break;
#endif

		default:
			assert(0);
			return("");
	}
}


D3DXMATRIX C3DRenderer::GetWVPMatrix( void ) const
{
	return(m_mWVPTransform);
}


D3DXMATRIX C3DRenderer::GetWVMatrix( void ) const
{
	return(m_mWVTransform);
}

D3DXMATRIX C3DRenderer::GetWMatrix( void ) const
{
	return(m_mWTransform);
}



D3DXVECTOR3 C3DRenderer::GetEyePos( void ) const
{
	return(m_vEyePos);
}



D3DXVECTOR3 C3DRenderer::GetEyeDir( void ) const
{
	return(m_vEyeDir);
}





bool C3DRenderer::AskSpecularEnable( void )
{
	if(m_eCurrentRenderMode==eRM_PolybumpSpecPixelShader)return(true);
	if(m_eCurrentRenderMode==eRM_PolybumpSpecPixelShader_CM)return(true);
	if(m_eCurrentRenderMode==eRM_GouraudSpecular)return(true);

	return(false);
}


//
bool C3DRenderer::Resize3DEnvironment( void )
{
	if(!m_pD3D)return(true);		// no D3D, no work to do

	assert(m_pd3dDevice);

  HRESULT hRes;

	D3DDISPLAYMODE d3ddm;
	if( FAILED( m_pD3D->GetAdapterDisplayMode( D3DADAPTER_DEFAULT, &d3ddm ) ) )
	{
		Error.Add("GetAdapterDisplayMode failed");
		return(false);
	}

	DeInitFont();

	D3DPRESENT_PARAMETERS d3dpp; 
	ZeroMemory( &d3dpp, sizeof(d3dpp) );
	d3dpp.Windowed   = TRUE;
	d3dpp.SwapEffect = D3DSWAPEFFECT_DISCARD;		// D3DSWAPEFFECT_COPY_VSYNC
	d3dpp.BackBufferFormat = d3ddm.Format;
  d3dpp.EnableAutoDepthStencil = TRUE;
  d3dpp.AutoDepthStencilFormat = D3DFMT_D16;
	d3dpp.hDeviceWindow=m_hWindow;							// width and height is taken from this window

  // Reset the device
	hRes = m_pd3dDevice->Reset( &d3dpp );
  if(FAILED(hRes))
	{
		m_BackbufferWidth=0;
		m_BackbufferHeight=0;
		return(false);
//		Error.AddDirectX("Reset failed",hRes);
//		return(false);
	}

	// get width and height
  {
		D3DSURFACE_DESC desc;
		LPDIRECT3DSURFACE8 pBackBuffer;
		m_pd3dDevice->GetBackBuffer( 0, D3DBACKBUFFER_TYPE_MONO, &pBackBuffer );
		if(pBackBuffer)
		{
			pBackBuffer->GetDesc( &desc );
			pBackBuffer->Release();

			assert(desc.Width);
			assert(desc.Height);

			m_BackbufferWidth=desc.Width;
			m_BackbufferHeight=desc.Height;
		}
		else		// e.g. minimized
		{	
			m_BackbufferWidth=0;
			m_BackbufferHeight=0;
		}
	}

	InitFont();

	return(true);
}




// init m_pd3dDevice before using
bool C3DRenderer::AskForPixelShaderSupport( void )
{
	assert(m_pd3dDevice);

	D3DCAPS8 caps;

	m_pd3dDevice->GetDeviceCaps(&caps);						

	return( D3DSHADER_VERSION_MAJOR( caps.PixelShaderVersion ) >= 1 );
}



void C3DRenderer::GetBackbufferSize( DWORD &outdwWidth, DWORD &outdwHeight ) const
{
	outdwWidth=m_BackbufferWidth;
	outdwHeight=m_BackbufferHeight;
}




// /return frametime in 1/100 ms
DWORD C3DRenderer::GetFrameTime100( void ) const
{
	return(m_dwLastFrameTime100);
}



// /return blended frametime in 1/100 ms
DWORD C3DRenderer::GetBlendedFrameTime100( void ) const
{
	return(m_dwBlendedFrameTime100);
}


bool C3DRenderer::AskDrawLightSources( void )
{
	return(m_eCurrentRenderMode==eRM_Gouraud
			|| m_eCurrentRenderMode==eRM_GouraudSpecular
			|| m_eCurrentRenderMode==eRM_TexturedLit
			|| m_eCurrentRenderMode==eRM_PolybumpDiffuseNoPixelShader
			|| m_eCurrentRenderMode==eRM_PolybumpDiffusePixelShader
			|| m_eCurrentRenderMode==eRM_PolybumpSpecPixelShader
			|| m_eCurrentRenderMode==eRM_PolybumpSpecPixelShader_CM);
}


// set attribute
void C3DRenderer::SetShowVertexBaseSpace( const bool inbValue )
{
	m_bShowVertexBaseSpace=inbValue;
}

// get attribute
bool C3DRenderer::GetShowVertexBaseSpace( void ) const
{
	return(m_bShowVertexBaseSpace);
}


// get attribute
void C3DRenderer::SetShowTriBaseSpace( const bool inbValue )
{
	m_bShowTriBaseSpace=inbValue;
}


// get attribute
bool C3DRenderer::GetShowTriBaseSpace( void ) const
{
	return(m_bShowTriBaseSpace);
}



// set attribute
void C3DRenderer::SetShowNormals( const bool inbValue )
{
	m_bShowNormals=inbValue;
}

// get attribute
bool C3DRenderer::GetShowNormals( void ) const
{
	return(m_bShowNormals);
}

// set attribute
void C3DRenderer::SetForceDirLight( const bool inbValue )
{
	m_bForceDirLight=inbValue;
}



//
bool C3DRenderer::GetForceDirLight( void ) const
{
	return(m_bForceDirLight);
}


//
bool C3DRenderer::AskShowNormalmap( void )
{
	return(m_eCurrentRenderMode==eRM_NormalmapFullbright);
}


// output text on the backbuffer (could be optimized)
void C3DRenderer::DrawText( float infX, float infY, const char *inszText )
{
	assert(m_pd3dFont);			if(!m_pd3dFont)return;

	RECT rect;

	rect.left=(LONG)infX;
	rect.top=(LONG)infY;
	rect.right=(LONG)infX+1000;
	rect.bottom=(LONG)infY+1000;

	HRESULT hRes;

	hRes=m_pd3dFont->Begin();
	if(FAILED(hRes))Error.AddDirectX("DrawText(1) failed: ",hRes);

	hRes=m_pd3dFont->DrawText(inszText,-1,&rect,DT_LEFT,D3DCOLOR(0xffffffff));
	if(FAILED(hRes))Error.AddDirectX("DrawText(2) failed: ",hRes);

	hRes=m_pd3dFont->End();
	if(FAILED(hRes))Error.AddDirectX("DrawText(3) failed: ",hRes);
}


//
bool C3DRenderer::AskEachLightOnePass( void )
{
	return(m_eCurrentRenderMode!=eRM_TexturedFullbright
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
			&& m_eCurrentRenderMode!=eRM_OverdrawMeasure
			&& m_eCurrentRenderMode!=eRM_ShadeTangentSpaceProblems
#endif
			&& m_eCurrentRenderMode!=eRM_NormalmapFullbright);
}


//
bool C3DRenderer::AskFullbright( void )
{
	return(m_eCurrentRenderMode==eRM_Wireframe
			|| m_eCurrentRenderMode==eRM_TexturedFullbright
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
			|| m_eCurrentRenderMode==eRM_OverdrawMeasure
#endif
			|| m_eCurrentRenderMode==eRM_NormalmapFullbright);
}


void C3DRenderer::InitFont( void )
{
	// D3D Font
	HFONT hFont=(HFONT)GetStockObject(SYSTEM_FONT);				assert(hFont);

	D3DXCreateFont(m_pd3dDevice,hFont,&m_pd3dFont);
}




void C3DRenderer::DeInitFont( void )
{
	// D3D Font
	if(m_pd3dFont)
	{
		m_pd3dFont->Release();
		m_pd3dFont=0;
	}
}

//!
DWORD C3DRenderer::AskAmountOfPasses( void )
{
	if(m_eCurrentRenderMode==eRM_PolybumpSpecPixelShader_CM)return(2);

	return(1);
}





bool C3DRenderer::AskUseNormalizeCubemapT2T3( void )
{
	return(m_eCurrentRenderMode==eRM_PolybumpSpecPixelShader_CM);
}






// used for MeasureOverdraw
bool C3DRenderer::AskDestBlendOneOne( void )
{
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
		return(m_eCurrentRenderMode==eRM_OverdrawMeasure);
#endif
	return(false);
}



/*
bool C3DRenderer::AskWriteZBuffer( void )
{
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
		return(m_eCurrentRenderMode!=eRM_OverdrawMeasure);
#endif
	return(true);
}
*/

bool C3DRenderer::AskReadZBuffer( void )
{
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
		return(m_eCurrentRenderMode!=eRM_OverdrawMeasure);
#endif
	return(true);
}



//! /return D3DCULL_NONE | D3DCULL_CW | D3DCULL_CCW
DWORD C3DRenderer::AskCullMode( void )
{
#ifdef EXTENDED_FOR_DEBUGGING_PURPOSE
	if(m_eCurrentRenderMode==eRM_OverdrawMeasure)return(D3DCULL_NONE);
#endif
	return(D3DCULL_CW);
}


float C3DRenderer::GetDebugScale( void ) const
{
	return(m_fDebugScale);
}

void C3DRenderer::SetDebugScale( const float infValue )
{
	m_fDebugScale=infValue;
	if(m_fDebugScale<0.0f)m_fDebugScale=0.0f;
}
