#include "stdafx.h"
#include "SoundSceneNode.h"

#include "Ray.h"
#include "Sphere.h"

#include "SoundSystem.h"
#include "ResourceManager.h"
#include "DynamicSceneNode.h"
#include "SceneManager.h"
#include "OptionManager.h"

const float FADE_SPEED = 2.2f;

cSoundSceneNode::cSoundSceneNode( eType type )
: cSceneNode( type )
, mListenerEntered( false )
, mOldListenerEntered( false )
, mRadius( 1000.0f )
, mAttenDistance( 100.0f )
, mLoopCount( 0 )
, mTempCount( 0 )
, mLoopInterval( 0.0f )
, mTempInterval( 0.0f )
, mVolumeRatio( 1.0f )
, mGain( 0.0f )
, mTargetGain( 0.0f )
, mRemoved(false)
, mFadeOut(false)
, mpOwnerSceneNode(0)
{
}

cSoundSceneNode::~cSoundSceneNode()
{
	RemoveSelf();

	mSound = 0;
}

void cSoundSceneNode::OnProcess( unsigned long deltaTime, unsigned long accumTime )
{
	if( mFadeOut )
	{
		RemoveSelf();
	}

	///
	if( mpOwnerSceneNode )
	{
		SetTranslate( mpOwnerSceneNode->GetWorldTranslate() );
	}

	///
	float dt = float(deltaTime) * 0.001f;
	cSceneNode::OnProcess( deltaTime, accumTime );

	/// 
	if( mListenerEntered == true )
	{
		if( mOldListenerEntered == false )
		{
			mTempCount = 0;
			mTempInterval = mLoopInterval;
		}

		float dist = (mBoundSphere.GetCenter() - mListenerSphere.GetCenter()).Length();

		if( dist <= mAttenDistance )
			mTargetGain = mVolumeRatio;
		else
			mTargetGain = mVolumeRatio * (1.0f - (dist - mAttenDistance) / (mRadius + mListenerSphere.GetRadius() - mAttenDistance));
	}
	else
	{
		mTargetGain = 0.0f;
	}

	if( mGain != mTargetGain )
	{
		mGain += (mTargetGain - mGain) * min(deltaTime * FADE_SPEED, 1.0f);
		mSound->SetGain( mGain );
	}

	///  īƮ
	NiAudioSource::Status status = mSound->GetStatus();

	if( status != NiAudioSource::PLAYING )
	{
		if( mLoopCount == 0 )
		{
			mTempInterval += dt;

			if( mTempInterval >= mLoopInterval )
			{
				mTempInterval = 0.0f;
				mSound->Stop();
				mSound->Play();
			}
		}
		else
		{
			if( mTempCount < mLoopCount )
			{
				mTempInterval += dt;

				if( mTempInterval >= mLoopInterval )
				{
					mTempInterval = 0.0f;
					mTempCount += 1;
					mSound->Stop();
					mSound->Play();
				}
			}
			else
			{
				///  
				RemoveSelf();
			}
		}
	}

	mOldListenerEntered = mListenerEntered;
	mListenerEntered = false;
}

void cSoundSceneNode::OnListenerEntered( const cSphere& sphere )
{
	mListenerEntered = true;
	mListenerSphere = sphere;
}

bool cSoundSceneNode::Init( const cSoundSceneNodeParam& param )
{
	NiAudioSource* src = 0;
	if( param.mManaged == false )
	{
		///  带 
		NiAudioSystem* as = NiAudioSystem::GetAudioSystem();
		src = as->CreateSource( NiAudioSource::TYPE_AMBIENT );
		mSound = src;

		src->SetStreamed( false );
		src->SetFilename( param.mPathName.Cstr() );
		src->SetAllowSharing( true );

		/// 带 ε
		if( src->Load() == false )
		{
			assert( 0 );
			return false;
		}
	}
	else
	{
		/// Ʈ  ResourceManager ȹѴ.
		src = RESOURCEMAN->CloneStageSoundByName( param.mPathName.Cstr() );
		if( src == 0 )
		{
			assert( 0 );
			return false;
		}
		mSound = NiDynamicCast(NiAudioSource, src);
	}

	if( mSound == 0 )
		return false;

	mpOwnerSceneNode = param.mpOwnerSceneNode;
//	if( mpOwnerSceneNode )
//		mOwnerSceneNodeIdx = mpOwnerSceneNode->GetIndexByManger();

	///  ʱȭ
	mRadius = param.mRadius;
	mAttenDistance = param.mAttenDistance;
	mLoopCount = param.mLoopCount;
	mLoopInterval = param.mLoopInterval;
	mVolumeRatio = param.mVolumeRatio;
	mEffectSound = param.mEffectSound;

	///  ʱȭ
	GetFileName( &param.mPathName, param.mPathName );

	if( cSceneNode::Init( param ) == false )
	{
		return false;
	}

	/// Ӽ 
	src->SetLoopCount( 1 );

	//if( param.mManaged )
	{
		src->SetGain( 0.0f );
		src->Stop();
	}
	//else
	//{
	//	if( SOUNDSYS->IsSoundActive() )
	//	{
	//		src->SetGain( mVolumeRatio );
	//	}
	//	else
	//	{
	//		mGain = mVolumeRatio;
	//		src->SetGain( 0.0f );
	//	}
	//}
	return true;
}

bool cSoundSceneNode::Pick( const cRay& ray )
{
	return mBoundSphere.IntersectRay( ray );
}

void cSoundSceneNode::ActiveSound( bool active )
{
	if( active )
	{
		if( mSound )
			mSound->SetGain( mGain );
	}
	else
	{
		if( mSound )
			mSound->SetGain( 0.0f );
	}
}

void cSoundSceneNode::RemoveSelf()
{
	if( mRemoved == true )
		return;

	SCENEMAN->AddDeleteSoundNode( this );
	mRemoved = true;

	if( mpOwnerSceneNode )
	{
		mpOwnerSceneNode->UnLinkSound( this );
		mpOwnerSceneNode = 0;
	}

//	if( mOwnerSceneNodeIdx != (unsigned long)-1 )
//	{
//		cDynamicSceneNode* node = SCENEMAN->GetDynamicSceneNode( mOwnerSceneNodeIdx );
//		if( node )
//		{
//			assert( node == mpOwnerSceneNode );
//			node->UnLinkSound( this );
//		}
//	}
}

void cSoundSceneNode::Remove()
{
	if( mRemoved == true )
		return;

//	if( mOwnerSceneNodeIdx != (unsigned long)-1 )
//	{
//		cDynamicSceneNode* node = SCENEMAN->GetDynamicSceneNode( mOwnerSceneNodeIdx );
//		if( node )
//		{
//			assert( node == mpOwnerSceneNode );
//			node->UnLinkSound( this );
//		}
//	}

	if( mpOwnerSceneNode )
	{
		mpOwnerSceneNode->UnLinkSound( this );
		mpOwnerSceneNode = 0;
	}

	SCENEMAN->AddDeleteSoundNode( this );
	mRemoved = true;
}

void cSoundSceneNode::ParentNodeRemove()
{
	if( mRemoved == true )
		return;

	mpOwnerSceneNode = 0;
	SCENEMAN->AddDeleteSoundNode( this );
	mRemoved = true;
}

void cSoundSceneNode::UpdatVolumeRatio()
{
	if( mEffectSound )
		mVolumeRatio = OPTIONMAN->GetEffVolume();
	else
		mVolumeRatio = OPTIONMAN->GetEnvVolume();
}