#include "stdafx.h"
#include "LightSceneNode.h"

#include "Ray.h"
#include "Sphere.h"
#include "TerrainNode.h"

cLightSceneNode::cLightSceneNode( eSceneNodeType type )
: cSceneNode( type )
, mRadius( 0.0f )
, mDiffuse0( NiColor::WHITE )
, mDiffuse1( NiColor::WHITE )
, mDiffuseAnimTime( 0.0f )
, mDiffuseTime( 0.0f )
, mDiffuseDir( false )
, mFlickering( false )
, mConstantAtten0( 0.0f )
, mConstantAtten1( 0.0f )
, mConstantAttenAnimTime( 0.0f )
, mConstantAttenTime( 0.0f )
, mConstantAttenDir( false )
, mAffectedSet( 16 )
{
}

cLightSceneNode::~cLightSceneNode()
{
	for( unsigned int i = 0, iend = mTerrainArray.GetSize(); i < iend; ++i )
	{
		cTerrainLeafNode* n = mTerrainArray[i];

		n->mLight = 0;
		n->mIndexByLight = 0xFFFFFFFF;
	}
}

void cLightSceneNode::Process( float deltaTime, float accumTime )
{
	bool needUpdate = mNeedUpdateTransform;
	cSceneNode::Process( deltaTime, accumTime );

	if( needUpdate )
	{
		for( unsigned int i = 0, iend = mTerrainArray.GetSize(); i < iend; ++i )
		{
			cTerrainLeafNode* n = mTerrainArray[i];

			n->mLight = 0;
			n->mIndexByLight = 0xFFFFFFFF;
		}

		///
		mTerrainArray.Clear();
		mTerrainArray.Reserve(9);

		if( TERRAIN->CollideSphere( &mTerrainArray, mBoundSphere ) )
		{
			for( unsigned int i = 0, iend = mTerrainArray.GetSize(); i < iend; ++i )
			{
				cTerrainLeafNode* n = mTerrainArray[i];
				n->SetLight( this, i );
			}
		}
	}

	if( mDiffuseAnimTime != 0.0f )
	{
		if( mDiffuseDir )
			mDiffuseTime += deltaTime;
		else
			mDiffuseTime -= deltaTime;

		if( mDiffuseTime > mDiffuseAnimTime )
		{
			mDiffuseTime = mDiffuseAnimTime;
			mDiffuseDir = false;
		}
		else if( mDiffuseTime < 0.0f )
		{
			mDiffuseTime = 0.0f;
			mDiffuseDir = true;
		}

		float r = mDiffuseTime / mDiffuseAnimTime;
		mLight->SetDiffuseColor( mDiffuse0 + (mDiffuse1 - mDiffuse0) * r );
	}

	if( mConstantAttenAnimTime != 0.0f )
	{
		if( mConstantAttenDir )
			mConstantAttenTime += deltaTime;
		else
			mConstantAttenTime -= deltaTime;

		if( mConstantAttenTime > mConstantAttenAnimTime )
		{
			mConstantAttenTime = mConstantAttenAnimTime;
			mConstantAttenDir = false;
		}
		else if( mConstantAttenTime < 0.0f )
		{
			mConstantAttenTime = 0.0f;
			mConstantAttenDir = true;
		}

		float r = mConstantAttenTime / mConstantAttenAnimTime;
		mLight->SetConstantAttenuation( mConstantAtten0 + (mConstantAtten1 - mConstantAtten0) * r );
	}
}

bool cLightSceneNode::Save( cFileSaver& saver )
{
	///  (̸, ȯ ) 
	if( cSceneNode::Save( saver ) == false )
		return false;

	///  
	saver.WriteFloat( mRadius );

	/// ֺ 
	saver.Write( &(mLight->GetAmbientColor()), sizeof(NiColor) );

	///  
	saver.Write( &mDiffuse0, sizeof(NiColor) );
	saver.Write( &mDiffuse1, sizeof(NiColor) );
	saver.WriteFloat( mDiffuseAnimTime );
	saver.WriteUnsignedInt( mFlickering );

	/// ݻ 
	saver.Write( &(mLight->GetSpecularColor()), sizeof(NiColor) );

	///   
	saver.WriteFloat( mConstantAtten0 );
	saver.WriteFloat( mConstantAtten1 );
	saver.WriteFloat( mConstantAttenAnimTime );
	saver.WriteFloat( mLight->GetLinearAttenuation() );
	saver.WriteFloat( mLight->GetQuadraticAttenuation() );
	return true;
}

bool cLightSceneNode::Init( const cLightSceneNodeParam& param )
{
	///  带 
	NiPointLight* o = NiNew NiPointLight;
	mLight = o;
	mRadius = param.mRadius;
	mDiffuse0 = param.mDiffuse0;
	mDiffuse1 = param.mDiffuse1;
	mConstantAtten0 = param.mConstantAtten0;
	mConstantAtten1 = param.mConstantAtten1;

	if( mDiffuse0 != mDiffuse1 )
		mDiffuseAnimTime = param.mDiffuseAnimTime;
	else
		mDiffuseAnimTime = 0.0f;

	if( mConstantAtten0 != mConstantAtten1 )
		mConstantAttenAnimTime = param.mConstantAttenAnimTime;
	else
		mConstantAttenAnimTime = 0.0f;

	mFlickering = param.mFlickering;

	///  ʱȭ
	param.mNiObject = o;
	if( cSceneNode::Init( param ) == false )
	{
		return false;
	}

	///  
	o->SetAmbientColor( param.mAmbient );
	o->SetDiffuseColor( param.mDiffuse0 );
	o->SetSpecularColor( param.mSpecular );

	///   
	o->SetConstantAttenuation( param.mConstantAtten0 );
	o->SetLinearAttenuation( param.mLinearAtten );
	o->SetQuadraticAttenuation( param.mQuadricAtten );

	///    
	o->SetSelectiveUpdate( true );
	o->SetSelectiveUpdateTransforms( true );
	o->SetSelectiveUpdatePropertyControllers( true );
	o->SetSelectiveUpdateRigid( false );
	return true;
}

bool cLightSceneNode::Pick( const cRay& ray )
{
	return mBoundSphere.IntersectRay( &mPickPos, &mPickDistance, ray );
}

void cLightSceneNode::Attach( cSceneNode* node )
{
	assert( node );

	if( mAffectedSet.Find(node) == mAffectedSet.End() )
	{
		mLight->AttachAffectedNode( (NiNode*)node->GetNiObject() );
		mAffectedSet.Insert( node );
	}
}

void cLightSceneNode::Detach( cSceneNode* node )
{
	assert( node );

	if( mAffectedSet.Find(node) != mAffectedSet.End() )
	{
		mLight->DetachAffectedNode( (NiNode*)node->GetNiObject() );
		mAffectedSet.Erase( node );
	}
}

void cLightSceneNode::Detach( cTerrainLeafNode* node )
{
	assert( node );

	///  迭 
	unsigned int i = node->mIndexByLight;
	bool orderChanged = false;
	mTerrainArray.PopAt( &orderChanged, i );

	if( orderChanged )
	{
		/// ش ġ    ε 缳
		mTerrainArray[i]->mIndexByLight = i;
	}

	node->mLight = 0;
	node->mIndexByLight = 0xFFFFFFFF;
}

void cLightSceneNode::DetachAll()
{
	mLight->DetachAllAffectedNodes();
	mLight->DetachAllUnaffectedNodes();

	cAffectedSet::cIterator i = mAffectedSet.Begin();
	cAffectedSet::cIterator end = mAffectedSet.End();

	for( ; i != end; ++i )
	{
		cSceneNode* n = *i;
		n->RemoveLight( this );
		n->SetNeedUpdateTransform( false );
		n->SetNeedUpdateEffects( true );
		n->Update();
	}
	mAffectedSet.Clear();
}

void cLightSceneNode::SetAmbient( const NiColor& color )
{
	mLight->SetAmbientColor( color );
}

void cLightSceneNode::SetDiffuse0( const NiColor& color )
{
	mDiffuse0 = color;
	mLight->SetDiffuseColor( color );
}

void cLightSceneNode::SetDiffuse1( const NiColor& color )
{
	mDiffuse1 = color;
}

void cLightSceneNode::SetDiffuseAnimTime( float time )
{
	mDiffuseAnimTime = time;

	if( time == 0.0f )
		mLight->SetDiffuseColor( mDiffuse0 );
}

void cLightSceneNode::SetSpecular( const NiColor& color )
{
	mLight->SetSpecularColor( color );
}

void cLightSceneNode::SetConstantAtten0( float value )
{
	mConstantAtten0 = value;
	mLight->SetConstantAttenuation( value );
}

void cLightSceneNode::SetConstantAtten1( float value )
{
	mConstantAtten1 = value;
}

void cLightSceneNode::SetConstantAttenAnimTime( float time )
{
	mConstantAttenAnimTime = time;

	if( time == 0.0f )
		mLight->SetConstantAttenuation( mConstantAtten0 );
}

void cLightSceneNode::SetLinearAtten( float value )
{
	mLight->SetLinearAttenuation( value );
}

void cLightSceneNode::SetQuadricAtten( float value )
{
	mLight->SetQuadraticAttenuation( value );
}
