#include "stdafx.h"

#include "NaviMeshNode.h"
#include "SceneManager.h"
#include "SceneNode.h"
#include "Terrain.h"
#include "NaviMesh.h"
#include "Ray.h"

#ifdef MAP_EDITOR
#include "../Doing/NaviMeshBuilding.h"

bool cNaviMeshLeafNode::BackupBuilding( cNaviMeshNodeBuildingInfo* info, const NiPoint3& pos, float outerRadius )
{
	assert( info );

	unsigned int xbegin, ybegin, xend, yend;
	if( CalcRange( &xbegin, &ybegin, &xend, &yend, pos, outerRadius ) == false )
	{
		return false;
	}

	/// Undo  
	info->mNode = this;
	info->mXBegin = xbegin;
	info->mYBegin = ybegin;
	info->mXEnd = xend;
	info->mYEnd = yend;
	float* p = info->mUndoHeightArray = new float[(yend - ybegin) * (xend - xbegin)];

	for( unsigned int i = mLineCount * ybegin, yi = ybegin; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi, ++p )
		{
			NAVIMESH->GetHeight( p, mXIndex + xi, mYIndex + yi );
		}
	}
	return true;
}

void cNaviMeshLeafNode::UpdateBuilding( cNaviMeshNodeBuildingInfo* info, const NiPoint3& pos, float outerRadius )
{
	assert( info );

	unsigned int xbegin, ybegin, xend, yend;
	if( CalcRange( &xbegin, &ybegin, &xend, &yend, pos, outerRadius ) == false )
	{
		return;
	}

	/// Redo  
	float* p = info->mRedoHeightArray = new float[(yend - ybegin) * (xend - xbegin)];

	/// ׺޽ ̸ʰ ȭ
	/// ȭ    ũ Ѵ.
	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi, ++p )
		{
			NAVIMESH->GetHeight( p, mXIndex + xi, mYIndex + yi );
			mHeights[i + xi] = *p;
		}
	}

	///  ڸ 
	UpdateBoundUpward();

	///   
	UpdatePosCoords();
}

void cNaviMeshLeafNode::Undo( const cNaviMeshNodeBuilding& doing )
{
	unsigned int xbegin = doing.mXBegin;
	unsigned int ybegin = doing.mYBegin;
	unsigned int xend = doing.mXEnd;
	unsigned int yend = doing.mYEnd;
	float* p = doing.mUndoHeightArray;

	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi, ++p )
		{
			mHeights[i + xi] = *p;
			NAVIMESH->SetHeight( mXIndex + xi, mYIndex + yi, *p );
		}
	}

	///  ڸ 
	UpdateBoundUpward();

	///   
	UpdatePosCoords();
}

void cNaviMeshLeafNode::Redo( const cNaviMeshNodeBuilding& doing )
{
	unsigned int xbegin = doing.mXBegin;
	unsigned int ybegin = doing.mYBegin;
	unsigned int xend = doing.mXEnd;
	unsigned int yend = doing.mYEnd;
	float* p = doing.mRedoHeightArray;

	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi, ++p )
		{
			mHeights[i + xi] = *p;
			NAVIMESH->SetHeight( mXIndex + xi, mYIndex + yi, *p );
		}
	}

	///  ڸ 
	UpdateBoundUpward();

	///   
	UpdatePosCoords();
}

bool cNaviMeshLeafNode::Raise( const NiPoint3& pos, float innerRadius, float outerRadius, float maxValue )
{
	///  
	unsigned int xbegin, ybegin, xend, yend;
	if( CalcRange( &xbegin, &ybegin, &xend, &yend, pos, outerRadius ) == false )
	{
		return false;
	}

	///   鿡  
	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi )
		{
			float x = (mXIndex + xi) * 100.0f;
			float y = (mYIndex + yi) * 100.0f;
			float dx = x - pos.x;
			float dy = y - pos.y;
			float d = NiSqrt(dx*dx + dy*dy);
			float v;

			if( d > outerRadius )
				continue;
			if( innerRadius == outerRadius || d <= innerRadius )
				v = maxValue;
			else
				v = maxValue * (outerRadius - d) / (outerRadius - innerRadius);

			float& z = mHeights[i + xi];
			z += v;

			/// ׺޽ ̸ 
			NAVIMESH->SetHeight( mXIndex + xi, mYIndex + yi, z );
		}
	}
	return true;
}

bool cNaviMeshLeafNode::Lower( const NiPoint3& pos, float innerRadius, float outerRadius, float maxValue )
{
	///  
	unsigned int xbegin, ybegin, xend, yend;
	if( CalcRange( &xbegin, &ybegin, &xend, &yend, pos, outerRadius ) == false )
	{
		return false;
	}

	///   鿡  
	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi )
		{
			float x = (mXIndex + xi) * 100.0f;
			float y = (mYIndex + yi) * 100.0f;
			float dx = x - pos.x;
			float dy = y - pos.y;
			float d = NiSqrt(dx*dx + dy*dy);
			float v;

			if( d > outerRadius )
				continue;
			if( innerRadius == outerRadius || d <= innerRadius )
				v = maxValue;
			else
				v = maxValue * (outerRadius - d) / (outerRadius - innerRadius);

			float& z = mHeights[i + xi];
			z -= v;

			/// ׺޽ ̸ 
			NAVIMESH->SetHeight( mXIndex + xi, mYIndex + yi, z );
		}
	}
	return true;
}

bool cNaviMeshLeafNode::Flatten( const NiPoint3& pos, float innerRadius, float outerRadius, float maxValue )
{
	///  
	unsigned int xbegin, ybegin, xend, yend;
	if( CalcRange( &xbegin, &ybegin, &xend, &yend, pos, outerRadius ) == false )
	{
		return false;
	}

	///   鿡  
	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi )
		{
			float& z = mHeights[i + xi];

			if( z == pos.z )
				continue;

			float x = (mXIndex + xi) * 100.0f;
			float y = (mYIndex + yi) * 100.0f;
			float dx = x - pos.x;
			float dy = y - pos.y;
			float d = NiSqrt(dx*dx + dy*dy);
			float v;

			if( d > outerRadius )
				continue;
			if( innerRadius == outerRadius || d <= innerRadius )
				v = maxValue;
			else
				v = maxValue * (outerRadius - d) / (outerRadius - innerRadius);

			if( z > pos.z )
			{
				z -= v;
				if( z < pos.z )
					z = pos.z;
			}
			if( z < pos.z )
			{
				z += v;
				if( z > pos.z )
					z = pos.z;
			}

			/// ׺޽ ̸ 
			NAVIMESH->SetHeight( mXIndex + xi, mYIndex + yi, z );
		}
	}
	return true;
}

bool cNaviMeshLeafNode::Smooth( const NiPoint3& pos, float outerRadius, float ratio )
{
	if( ratio < 0.1f )
	{
		assert( "too small smooth ratio" );
		return false;
	}

	///  
	unsigned int xbegin, ybegin, xend, yend;
	if( CalcRange( &xbegin, &ybegin, &xend, &yend, pos, outerRadius ) == false )
	{
		return false;
	}

	///   鿡  
	float z[9], sumz;
	int count;

	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi )
		{
			unsigned int tx = mXIndex + xi;
			unsigned int ty = mYIndex + yi;
			float x = tx * 100.0f;
			float y = ty * 100.0f;
			float dx = x - pos.x;
			float dy = y - pos.y;
			float d = NiSqrt(dx*dx + dy*dy);

			if( d > outerRadius )
				continue;

			/// ֺ  ̰ 
			sumz = 0;
			count = 0;

			if( NAVIMESH->GetHeight( &z[0], tx - 1, ty - 1 ) )
			{
				sumz += z[0];
				++count;
			}
			if( NAVIMESH->GetHeight( &z[1], tx, ty - 1 ) )
			{
				sumz += z[1];
				++count;
			}
			if( NAVIMESH->GetHeight( &z[2], tx + 1, ty - 1 ) )
			{
				sumz += z[2];
				++count;
			}
			if( NAVIMESH->GetHeight( &z[3], tx - 1, ty ) )
			{
				sumz += z[3];
				++count;
			}
			if( NAVIMESH->GetHeight( &z[4], tx, ty ) )
			{
				sumz += z[4];
				++count;
			}
			if( NAVIMESH->GetHeight( &z[5], tx + 1, ty) )
			{
				sumz += z[5];
				++count;
			}
			if( NAVIMESH->GetHeight( &z[6], tx - 1, ty + 1) )
			{
				sumz += z[6];
				++count;
			}
			if( NAVIMESH->GetHeight( &z[7], tx, ty + 1) )
			{
				sumz += z[7];
				++count;
			}
			if( NAVIMESH->GetHeight( &z[8], tx + 1, ty + 1) )
			{
				sumz += z[8];
				++count;
			}

			/// հ 
			if( count )
			{
				float posz = mHeights[i + xi];
				float avgz = sumz / count;
				mHeights[i + xi] = posz + (avgz - posz) * ratio;
			}
		}
	}

	/// ׺޽ ̸ 
	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi )
		{
			NAVIMESH->SetHeight( mXIndex + xi, mYIndex + yi, mHeights[i + xi] );
		}
	}
	return true;
}
#endif /// MAP_EDITOR

bool cNaviMeshLeafNode::SyncToTerrain( const NiPoint3& pos, float innerRadius, float outerRadius )
{
	///  
	unsigned int xbegin, ybegin, xend, yend;
	if( CalcRange( &xbegin, &ybegin, &xend, &yend, pos, outerRadius ) == false )
	{
		return false;
	}

	///   鿡  
	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi )
		{
			float x = (mXIndex + xi) * 100.0f;
			float y = (mYIndex + yi) * 100.0f;
			float dx = x - pos.x;
			float dy = y - pos.y;
			float d = NiSqrt(dx*dx + dy*dy);

			if( d > outerRadius )
				continue;

			float& z = mHeights[i + xi];
			float tz = 0.0f;
			TERRAIN->GetHeight( &tz, mXIndex + xi, mYIndex + yi );

			if( innerRadius == outerRadius || d <= innerRadius )
			{
				z = tz;
			}
			else
			{
				float r = ( outerRadius - d ) / (outerRadius - innerRadius);
				z = z + r * (tz - z);
			}

			/// ׺޽ ̸ 
			NAVIMESH->SetHeight( mXIndex + xi, mYIndex + yi, z );
		}
	}
	return true;
}

bool cNaviMeshLeafNode::SyncToObject( const NiPoint3& pos, float outerRadius )
{
	static tArray<cSceneNode*> pickedArray;

	///  
	unsigned int xbegin, ybegin, xend, yend;
	if( CalcRange( &xbegin, &ybegin, &xend, &yend, pos, outerRadius ) == false )
	{
		return false;
	}

	///   鿡  
	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi )
		{
			float x = (mXIndex + xi) * 100.0f;
			float y = (mYIndex + yi) * 100.0f;
			float dx = x - pos.x;
			float dy = y - pos.y;
			float d = NiSqrt(dx*dx + dy*dy);

			if( d > outerRadius )
				continue;

			pickedArray.Clear();
			NiPoint3 origin;
			origin.x = x;
			origin.y = y;
			origin.z = 100000.0f;

			if( SCENEMAN->Pick( &pickedArray, cRay(origin, -NiPoint3::UNIT_Z), NI_INFINITY, true, SCENENODE_STATIC ) == false )
				continue;

			float& z = mHeights[i + xi];
			z = pickedArray[0]->GetPickPos().z;

			/// ׺޽ ̸ 
			NAVIMESH->SetHeight( mXIndex + xi, mYIndex + yi, z );
		}
	}
	return true;
}

bool cNaviMeshLeafNode::SyncToPickHeight( const NiPoint3& pos, float innerRadius, float outerRadius, float pickz )
{
	///  
	unsigned int xbegin, ybegin, xend, yend;
	if( CalcRange( &xbegin, &ybegin, &xend, &yend, pos, outerRadius ) == false )
	{
		return false;
	}

	///   鿡  
	for( unsigned int yi = ybegin, i = ybegin * mLineCount; yi < yend; ++yi, i += mLineCount )
	{
		for( unsigned int xi = xbegin; xi < xend; ++xi )
		{
			float x = (mXIndex + xi) * 100.0f;
			float y = (mYIndex + yi) * 100.0f;
			float dx = x - pos.x;
			float dy = y - pos.y;
			float d = NiSqrt(dx*dx + dy*dy);

			if( d > outerRadius )
				continue;

			float& z = mHeights[i + xi];

			if( innerRadius == outerRadius || d <= innerRadius )
			{
				z = pickz;
			}
			else
			{
				float r = ( outerRadius - d ) / (outerRadius - innerRadius);
				z = z + r * (pickz - z);
			}

			/// ׺޽ ̸ 
			NAVIMESH->SetHeight( mXIndex + xi, mYIndex + yi, z );
		}
	}
	return true;
}
