//////////////////////////////////////////////////////////////////////////////////////
// ConvertMtxFiles.cpp - 
//
// Author: Michael Starich   
//////////////////////////////////////////////////////////////////////////////////////
// THIS CODE IS PROPRIETARY PROPERTY OF SWINGIN' APE STUDIOS, INC.
// Copyright (c) 2002
//
// The contents of this file may not be disclosed to third
// parties, copied or duplicated in any form, in whole or in part,
// without the prior written permission of Swingin' Ape Studios, Inc.
//////////////////////////////////////////////////////////////////////////////////////
// Modification History:
//
// Date     Who         Description
// -------- ----------  --------------------------------------------------------------
// 04/03/02 Starich     Created.
//////////////////////////////////////////////////////////////////////////////////////
#include "stdafx.h"
#include <math.h>
#include "fang.h"
#include "ConvertMtxFiles.h"
#include "ErrorLog.h"
#include "fdata.h"
#include "fanim.h"
#include "fclib.h"

//====================
// private definitions

#define _ERROR_HEADING				".MTX FILE COMPILER "

//=================
// public variables

//==================
// private variables

//===================
// private prototypes

//=================
// public functions

CConvertMtxFile::CConvertMtxFile()  {
	m_nBytesAllocated = 0;
	m_pMemAllocated = NULL;
	m_pCompiledFile = NULL;
	m_nCompiledBytes = 0;
	m_nBoneCount = 0;
	m_nTotalFrameCount = 0;

	m_bCompressTime = TRUE;
	m_bCompressAnim = TRUE;
	m_fThreshold = 1.0f;
}

CConvertMtxFile::~CConvertMtxFile() {
	FreeData();
}

BOOL CConvertMtxFile::ConvertFile( const CFileInfo *pFileInfo, BOOL bConvertToBigEndian/*=FALSE*/ ) {
	CString sFilepath;
	u32 i, nNumBytesInFile;
	FILE *pFile;
	FData_MtxFileHeader_t *pHeader;
	FData_MtxFileBone_t *pBone;
	
	FreeData();

	if( !pFileInfo ) {
		return FALSE;
	}

	CErrorLog &rErrorLog = CErrorLog::GetCurrent();

	// gather some info about the file
	sFilepath = pFileInfo->GetFilePath();
	nNumBytesInFile = pFileInfo->GetLength();	

	// allocate memory
	m_nBytesAllocated = nNumBytesInFile;
	m_nBytesAllocated = FMATH_BYTE_ALIGN_UP( m_nBytesAllocated, 1024 );// allocate a little extra, just in case
	m_pMemAllocated = (u8 *)fang_MallocAndZero( m_nBytesAllocated, 16 );
	if( !m_pMemAllocated ) {
		FreeData();
		rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
		rErrorLog.WriteErrorLine( "Could not allocate memory for file image" );
		return FALSE;
	}

	// read the entire file into memory
	pFile = fopen( sFilepath, "rb" );
	if( !pFile ) {
		FreeData();
		rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
		rErrorLog.WriteErrorLine( "Could not open file for reading" );
		return FALSE;
	}
	if( fread( m_pMemAllocated, 1, nNumBytesInFile, pFile ) != nNumBytesInFile ) {
		fclose( pFile );
		FreeData();
		rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
		rErrorLog.WriteErrorLine( "Trouble reading the file" );
		return FALSE;
	}
	fclose( pFile );

	// fix up our pointers
	m_nCompiledBytes = nNumBytesInFile;
	m_pCompiledFile = m_pMemAllocated;
	pHeader = (FData_MtxFileHeader_t *)m_pCompiledFile;
	pBone = (FData_MtxFileBone_t *)&pHeader[1];
	
	FASSERT( pHeader->nDataType == FDATA_MTX_FILE_DATA_TYPE_MATRIX );

	// the post-john compression way (grab some file stats)
	m_nBoneCount = pHeader->nNumBones;
	m_nTotalFrameCount = 0;
	for( i=0; i < m_nBoneCount; i++ ) {
		m_nTotalFrameCount += pBone[i].nNumFrames;
	}

	// compress the data
	u32 nNewSize;
	void *pNewData = CompressAnimationData( sFilepath, pHeader, m_nCompiledBytes, bConvertToBigEndian, nNewSize );
	if( !pNewData ) {
		FreeData();
		// Error log generated by the compressor
		return FALSE;
	}
	// free our old data
	fang_Free( m_pMemAllocated );

	// save ptrs to our compressed data
	m_pMemAllocated = (u8 *)pNewData;
	m_pCompiledFile = m_pMemAllocated;
	m_nCompiledBytes = nNewSize;
	m_nBytesAllocated = nNewSize;	

	return TRUE;
}

u32 CConvertMtxFile::GetDataCRC() {

	if( !m_pMemAllocated ) {
		return 0;
	}

	u32 nReturnCRC = fmath_Crc32( 0, (u8 *)m_pMemAllocated, m_nCompiledBytes );

	return nReturnCRC;
}

u32 CConvertMtxFile::GetSizeOfConvertedFile() {

	if( !m_pMemAllocated ) {
		return 0;
	}
	return m_nCompiledBytes;
}

BOOL CConvertMtxFile::WriteConvertedFile( cchar *pszFilename, FILE *pFileStream/*=NULL*/ ) {

	if( !m_pMemAllocated ) {
		return FALSE;
	}
	BOOL bCloseFile = FALSE;
	if( !pFileStream ) {
		if( !pszFilename ) {
			// invalid filename
			return FALSE;
		}
		pFileStream = _tfopen( pszFilename, _T("wb") );
		if( !pFileStream ) {
			return FALSE;
		}
		bCloseFile = TRUE;
	}
	// write out our file
	fwrite( m_pCompiledFile, m_nCompiledBytes, 1, pFileStream );
	// close our file
	if( bCloseFile ) {
		fclose( pFileStream );
	}

	return TRUE;
}

void CConvertMtxFile::FreeData() {

	if( m_pMemAllocated ) {
		fang_Free( m_pMemAllocated );
		m_pMemAllocated = NULL;
	}
	m_nBytesAllocated = 0;
	m_pCompiledFile = NULL;
	m_nCompiledBytes = 0;
	m_nBoneCount = 0;
	m_nTotalFrameCount = 0;
}

void CConvertMtxFile::SetCompressionVars( BOOL bCompressTime, BOOL bCompressAnim, f32 fThreshold ) {
	m_bCompressTime = bCompressTime;
	m_bCompressAnim = bCompressAnim;
	m_fThreshold = fThreshold;
}

//==================
// private functions


//
// Utility function used to appropriately compress a time 
// step for a key frame (scale, translation or orientation)
//
static void _CompressTimeStep( u8 **pDest, u16 nAnimFlags, f32 fTotalAnimSecs, f32 fKeyStartingSecs, BOOL bLastFrame, BOOL bConvertEndian )
{
	if ( nAnimFlags & FANIM_BONEFLAGS_8BIT_SECS )
	{
		if ( bLastFrame )
		{
			*((u8 *)*pDest) = 128;
		}
		else
		{
			*((u8 *)*pDest) = (u8)(((fKeyStartingSecs * 128.f) / fTotalAnimSecs) + 0.5f);
		}
		(*pDest) = (*pDest) + 1;
	}
	else if ( nAnimFlags & FANIM_BONEFLAGS_16BIT_SECS )
	{
		if ( bLastFrame )
		{
			*((u16 *)*pDest) = 32768;
		}
		else
		{
			(*((u16 *)*pDest)) = (u16)(((fKeyStartingSecs * 32768.f) / fTotalAnimSecs) + 0.5f);
		}
		// Convert endian if needed
		if ( bConvertEndian )
		{
			*((u16 *)*pDest) = fang_ConvertEndian( *((u16 *)*pDest) );
		}
		(*pDest) = (*pDest) + 2;
	}
	else
	{
		if ( bLastFrame )
		{
			*((f32 *)*pDest) = 1.f;
		}
		else
		{
			*((f32 *)*pDest) = fKeyStartingSecs / fTotalAnimSecs;
		}
		// Convert endian if needed
		if ( bConvertEndian )
		{
			*((f32 *)*pDest) = fang_ConvertEndian( *((f32 *)*pDest) );
		}
		(*pDest) = (*pDest) + 4;
	}
}


FINLINE BOOL _SignificantOrientationDiff( CFQuatA *pOrient1, CFQuatA *pOrient2, BOOL bCompress, f32 fThreshhold )
{
#if FANIM_USE_ANIM_COMPRESSION
	if ( bCompress )
	{
		f32 fX1 = fanim_CompOrientf32( pOrient1->x );
		f32 fY1 = fanim_CompOrientf32( pOrient1->y );
		f32 fZ1 = fanim_CompOrientf32( pOrient1->z );
		f32 fW1 = fanim_CompOrientf32( pOrient1->w );
		f32 fX2 = fanim_CompOrientf32( pOrient2->x );
		f32 fY2 = fanim_CompOrientf32( pOrient2->y );
		f32 fZ2 = fanim_CompOrientf32( pOrient2->z );
		f32 fW2 = fanim_CompOrientf32( pOrient2->w );
		if ( fX1 == fX2 && fY1 == fY2 && fZ1 == fZ2 && fW1 == fW2 )
		{
			return FALSE;
		}
		f32 fMaxMag = (f32)(1<<FANIM_ORIENTATION_FRAC_BITS);
		f32 fDot = sqrtf((fX1 * fX2) + (fY1 * fY2) + (fZ1 * fZ2) + (fW1 * fW2));
		if ( fDot > fThreshhold * fMaxMag )
		{
			return FALSE;
		}
	}
	else
#endif // FANIM_USE_ANIM_COMPRESSION
	{
		f32 fX = pOrient1->x - pOrient2->x;
		f32 fY = pOrient1->y - pOrient2->y;
		f32 fZ = pOrient1->z - pOrient2->z;
		f32 fW = pOrient1->w - pOrient2->w;

		f32 fDiff2 = (fX * fX) + (fY * fY) + (fZ * fZ) + (fW * fW);
//			if ( fDiff2 < 0.00001f )
		if ( fDiff2 < 0.0000001f )
		{
			return FALSE;
		}
	}
	
	return TRUE;
}

// return a pointer to the compressed data and fill in rnNumCompressedBytes
// return NULL if there was a problem.
void *CConvertMtxFile::CompressAnimationData( const CString &sFilepath, const FData_MtxFileHeader_t *pHeader, u32 nNumHeaderBytes,
											  BOOL bConvertToBigEndian, u32 &rnNumCompressedBytes ) 
{
#if FANIM_USE_ANIM_COMPRESSION
	BOOL bCompress = m_bCompressAnim;
	BOOL bCompressSecs = m_bCompressTime;
#else
	BOOL bCompress = FALSE;
	BOOL bCompressSecs = FALSE;
#endif
	f32 fUnitTolerance;
	if ( m_fThreshold == -1.f )
	{
		fUnitTolerance = CONVERT_MTX_FILES_ANIM_QUAT_COMPRESSION_THRESHHOLD;
	}
	else
	{
		fUnitTolerance = cosf( FMATH_DEG2RAD( m_fThreshold ) );
	}

	u32 nBone, nBone2, nKey;

	BOOL bTCompress = FALSE;
	BOOL bOCompress = FALSE;

	if ( bCompress )
	{
		bTCompress = TRUE;
		bOCompress = TRUE;
	}

	////////////////////////////////////////////////////////////////
	// Verify Data
	////////////////////////////////////////////////////////////////
	if ( pHeader->nDataType != FDATA_MTX_FILE_DATA_TYPE_MATRIX ) 
	{
		return NULL;
	}

	char szTemp[128];
	FData_MtxFileBone_t *pFileBoneArray = (FData_MtxFileBone_t *)(pHeader + 1);
	CErrorLog &rErrorLog = CErrorLog::GetCurrent();

	// Verify we don't have any bones with the same name and total the name sizes
	u32 nNameBytes = 0;
	for( nBone = 0; nBone < pHeader->nNumBones; nBone++ ) 
	{
		nNameBytes += strlen( pFileBoneArray[nBone].szName ) + 1;
		for( nBone2 = nBone + 1; nBone2 < pHeader->nNumBones; nBone2++ ) 
		{
			if( !fclib_stricmp( pFileBoneArray[nBone].szName, pFileBoneArray[nBone2].szName ) ) 
			{
				// Redundant bone names...
				sprintf( szTemp, "Redundant bone names '%s' detected while compressing animation.\n", pFileBoneArray[nBone].szName );
				DEVPRINTF( "CompressAnimationData: %s\n", szTemp );
				rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
				rErrorLog.WriteErrorLine( szTemp );
				return NULL;
			}
		}

		// See if we should compress translation
		if ( bTCompress )
		{
			FData_MtxFileMatrix_t *pFileKeyArray = (FData_MtxFileMatrix_t *)((u32)pHeader + (u32)pFileBoneArray[nBone].nFrameArrayOffset);
			for( nKey = 0; nKey < (pFileBoneArray[nBone].nNumFrames - 1); nKey++ ) 
			{
				if ( pFileKeyArray[nKey].Mtx.m_vRight.Mag2() < 0.0001f ||
					pFileKeyArray[nKey].Mtx.m_vUp.Mag2() < 0.0001f ||
					pFileKeyArray[nKey].Mtx.m_vFront.Mag2() < 0.0001f )
				{
					sprintf( szTemp, "Invalid matrix detected at key %u of bone named '%s' while compressing animation.\n", nKey, pFileBoneArray[nBone].szName );
					DEVPRINTF( "CompressAnimationData: %s\n", szTemp );
					rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
					rErrorLog.WriteErrorLine( szTemp );
					return NULL;
				}

				if (	pFileKeyArray[nKey].Mtx.m_vPos.x < -(1<<(15 - FANIM_TRANSLATION_FRAC_BITS))
					||	pFileKeyArray[nKey].Mtx.m_vPos.x > (1<<(15 - FANIM_TRANSLATION_FRAC_BITS))
					||	pFileKeyArray[nKey].Mtx.m_vPos.y < -(1<<(15 - FANIM_TRANSLATION_FRAC_BITS))
					||	pFileKeyArray[nKey].Mtx.m_vPos.y > (1<<(15 - FANIM_TRANSLATION_FRAC_BITS))
					||	pFileKeyArray[nKey].Mtx.m_vPos.z < -(1<<(15 - FANIM_TRANSLATION_FRAC_BITS))
					||	pFileKeyArray[nKey].Mtx.m_vPos.z > (1<<(15 - FANIM_TRANSLATION_FRAC_BITS)) )
				{
					bTCompress = FALSE;
				}
			}
		}
	}

	// Verify Bone data and collect needed allocation vars
	u32 nMaxBoneSKeys = 0, nTotalSKeys = 0;
	u32 nMaxBoneTKeys = 0, nTotalTKeys = 0;
	u32 nMaxBoneOKeys = 0, nTotalOKeys = 0;
	u32 anSKeyFrameCount[256];
	u32 anTKeyFrameCount[256];
	u32 anOKeyFrameCount[256];
	f32 fTotalAnimSecs = -1.f;
	f32 fMinimumTimeBetweenFrames = FMATH_MAX_FLOAT;
	u32 nBoneCount = 0;
	CFTQuatA qTest, qOrient, qLastTrans, qLastOrient, qNextOrient;
	f32 fSlider, fLastTransTime, fLastScaleTime, fLastOrientTime, fFrameTime, fNextFrameTime;
	f32 fNextScale, fLastScale, fCurrScale;
	CFVec3 vTest;
	for( nBone = 0; nBone < pHeader->nNumBones; nBone++ ) 
	{
		if( pFileBoneArray[nBone].nNumFrames < 2 ) 
		{
			// Key count is too small. Skip this bone...
			continue;
		}

		FData_MtxFileMatrix_t *pFileKeyArray = (FData_MtxFileMatrix_t *)((u32)pHeader + (u32)pFileBoneArray[nBone].nFrameArrayOffset);

		anSKeyFrameCount[nBone] = 0;
		anTKeyFrameCount[nBone] = 0;
		anOKeyFrameCount[nBone] = 0;
		qNextOrient.BuildQuat( pFileKeyArray[0].Mtx, TRUE );
		fLastScale = -1.f; fCurrScale = -1.f;
		fNextScale = pFileKeyArray[0].Mtx.m_vRight.Mag();
		if ( fNextScale > 0.99f && fNextScale < 1.01f )
		{
			fNextScale = 1.f;
		}
		for( nKey = 0; nKey < (pFileBoneArray[nBone].nNumFrames - 1); nKey++ ) 
		{
			if( pFileKeyArray[nKey].fStartingSecs < 0.0f ) 
			{
				// Invalid...
				sprintf( szTemp, "Invalid time stamp %f detected for key %u, bone named '%s' while compressing animation.\n", pFileKeyArray[nKey].fStartingSecs, nKey, pFileBoneArray[nBone].szName );
				DEVPRINTF( "CompressAnimationData: %s\n", szTemp );
				rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
				rErrorLog.WriteErrorLine( szTemp );
				return NULL;
			}

			fFrameTime = pFileKeyArray[nKey].fStartingSecs;
			fNextFrameTime = pFileKeyArray[nKey+1].fStartingSecs;

			// Calculate the difference between the next frame and this frame
			f32 fFrameTimeDiff = fNextFrameTime - fFrameTime;

			if( fFrameTimeDiff <= 0.f ) 
			{
				// Invalid...
				sprintf( szTemp, "For bone %s, time stamp %f for key %u is greater than the next key's time of %f.\n", pFileBoneArray[nBone].szName, pFileKeyArray[nKey].fStartingSecs, nKey, pFileKeyArray[nKey+1].fStartingSecs );
				DEVPRINTF( "CompressAnimationData: %s\n", szTemp );
				rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
				rErrorLog.WriteErrorLine( szTemp );
				return NULL;
			}

			// See if this is a new minimum diff
			if ( fFrameTimeDiff < fMinimumTimeBetweenFrames )
			{
				fMinimumTimeBetweenFrames = fFrameTimeDiff;
			}

			// Determine the number of significant key frames for scale, translation and rotation

			// Check to see if we need to include this data as a scale keyframe
			fCurrScale = fNextScale;
			fNextScale = pFileKeyArray[nKey + 1].Mtx.m_vRight.Mag();
			FASSERT( fNextScale >= 0.f );
			if ( fNextScale > 0.99f && fNextScale < 1.01f )
			{
				fNextScale = 1.f;
			}
			if ( nKey == 0 )
			{
				anSKeyFrameCount[nBone]++;
				fLastScale = fCurrScale;
				fLastScaleTime = fFrameTime;
			}
			else
			{
				// See if the scale keyframe is linear when compared with the last
				// scale keyframe accepted and the next keyframe
				fSlider = (fFrameTime - fLastScaleTime)/(fNextFrameTime - fLastScaleTime);
				f32 fInterpScale = (fLastScale * (1 - fSlider)) + (fNextScale * fSlider);
				FASSERT( fInterpScale >= 0.f );
				if ( fInterpScale > 0.99f && fInterpScale < 1.01f )
				{
					fInterpScale = 1.f;
				}
				if ( fCurrScale != fInterpScale )
				{
					anSKeyFrameCount[nBone]++;
					fLastScale = fInterpScale;
					fLastScaleTime = fFrameTime;
				}
			}

			// Build a quat from the matrix
			qOrient.Set( qNextOrient );
			qNextOrient.BuildQuat( pFileKeyArray[nKey+1].Mtx, TRUE );

			if ( nKey == 0 )
			{
				anTKeyFrameCount[nBone]++;
				anOKeyFrameCount[nBone]++;
				qLastTrans.Set( qOrient );
				qLastOrient.Set( qOrient );
				fLastTransTime = fFrameTime;
				fLastOrientTime = fFrameTime;
				continue;
			}
			
			// Make sure the orientation quat didn't flip 
			f32 fFlip = (qOrient.x * qLastOrient.x) + (qOrient.y * qLastOrient.y) + (qOrient.z * qLastOrient.z) + (qOrient.w * qLastOrient.w);
			if ( fFlip < 0 )
			{
				qOrient.x = -qOrient.x;
				qOrient.y = -qOrient.y;
				qOrient.z = -qOrient.z;
				qOrient.w = -qOrient.w;
			}
			
			fSlider = (fFrameTime - fLastTransTime)/(fNextFrameTime - fLastTransTime);
			vTest.ReceiveLerpOf( fSlider, qLastTrans.m_Pos, qNextOrient.m_Pos );
			if ( fanim_SignificantTranslationDiff( &qOrient.m_Pos, &vTest, bTCompress ) )
			{
				anTKeyFrameCount[nBone]++;
				qLastTrans.Set( qOrient );
				fLastTransTime = fFrameTime;
			}
			
			fSlider = (fFrameTime - fLastOrientTime)/(fNextFrameTime - fLastOrientTime);
			qTest.ReceiveSlerpOf( fSlider, qLastOrient, qNextOrient );
			if ( _SignificantOrientationDiff( &qLastOrient, &qTest, bOCompress, fUnitTolerance ) )
			{
				anOKeyFrameCount[nBone]++;
				qLastOrient.Set( qOrient );
				fLastOrientTime = fFrameTime;
			}
		}

		// Add in a count for the last frame
		anSKeyFrameCount[nBone]++;
		anTKeyFrameCount[nBone]++;
		anOKeyFrameCount[nBone]++;

		// Fix Maxes
		if ( anSKeyFrameCount[nBone] > nMaxBoneSKeys )
		{
			nMaxBoneSKeys = anSKeyFrameCount[nBone];
		}
		if ( anTKeyFrameCount[nBone] > nMaxBoneTKeys )
		{
			nMaxBoneTKeys = anTKeyFrameCount[nBone];
		}
		if ( anOKeyFrameCount[nBone] > nMaxBoneOKeys )
		{
			nMaxBoneOKeys = anOKeyFrameCount[nBone];
		}

		// Accumulate totals
		nTotalSKeys += anSKeyFrameCount[nBone];
		nTotalTKeys += anTKeyFrameCount[nBone];
		nTotalOKeys += anOKeyFrameCount[nBone];

		// At this point, nKey is the index of the last key in this bone...

		if( fTotalAnimSecs == -1.0f ) 
		{
			// We haven't recorded the animation time yet...
			fTotalAnimSecs = pFileKeyArray[nKey].fStartingSecs;
		} 
		else 
		{
			// Every bone's last key time must be identical...

			if( fTotalAnimSecs != pFileKeyArray[nKey].fStartingSecs ) 
			{
				// Last key's time doesn't match...
				sprintf( szTemp, "For bone %s, last key time of %f isn't identical to other bones of %f.\n", pFileBoneArray[nBone].szName, pFileKeyArray[nKey].fStartingSecs, fTotalAnimSecs );
				DEVPRINTF( "CompressAnimationData: %s\n", szTemp );
				rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
				rErrorLog.WriteErrorLine( szTemp );
				return NULL;
			}
		}

		nBoneCount++;
	}

	// Insanity check
	FASSERT( fTotalAnimSecs != -1.0f );

	if( fTotalAnimSecs == 0.0f ) 
	{
		DEVPRINTF( "CompressAnimationData(): Animation has an invalid duration of 0 seconds.\n" );
		rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
		rErrorLog.WriteErrorLine( "Animation has an invalid duration of 0 seconds.\n" );
		return NULL;
	}

	if( nBoneCount == 0 ) 
	{
		DEVPRINTF( "CompressAnimationData(): Animation has no bones.\n" );
		rErrorLog.WriteErrorHeader( _ERROR_HEADING + sFilepath );
		rErrorLog.WriteErrorLine( "Animation has no bones.\n" );
		return NULL;
	}

	////////////////////////////////////////////////////////////////
	// Determine the size and addresses of data to allocate
	////////////////////////////////////////////////////////////////
	u32 nDataBytes = 0;

	u32 ADDR_FAnim =		FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
	nDataBytes =			ADDR_FAnim + sizeof( FAnim_t );

	u32 ADDR_FBoneArray =	FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
	nDataBytes =			ADDR_FBoneArray + (sizeof( FAnimBone_t ) * nBoneCount);

	u32 ADDR_FBoneNames =	FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
	nDataBytes =			ADDR_FBoneNames + nNameBytes;

	u32 ADDR_FSKeys =		FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
	nDataBytes =			ADDR_FSKeys + (sizeof( FAnimSKey_t ) * nTotalSKeys);

	u32 ADDR_FTKeys, ADDR_FOKeys;
	if ( bTCompress )
	{
		ADDR_FTKeys =		FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
		nDataBytes =		ADDR_FTKeys + (sizeof( FAnimCTKey_t ) * nTotalTKeys);
	}
	else
	{
		ADDR_FTKeys =		FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
		nDataBytes =		ADDR_FTKeys + (sizeof( FAnimTKey_t ) * nTotalTKeys);
	}

	if ( bOCompress )
	{
		ADDR_FOKeys =		FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
		nDataBytes =		ADDR_FOKeys + (sizeof( FAnimCOKey_t ) * nTotalOKeys);
	}
	else
	{
		ADDR_FOKeys =		FMATH_BYTE_ALIGN_UP( nDataBytes, FCLASS_BYTE_ALIGN );
		nDataBytes =		ADDR_FOKeys + (sizeof( FAnimOKey_t ) * nTotalOKeys);
	}

	u32 ADDR_FSTime = 0;
	u32 ADDR_FTTime = 0;
	u32 ADDR_FOTime = 0;  // FOT time?  Ha ha!  You know.... sounds a lot like fart time...
	u32 nBitsOfTime;

#if FANIM_USE_ANIM_COMPRESSION
	if ( bCompressSecs && fTotalAnimSecs < (128.f * fMinimumTimeBetweenFrames) )
	{
		// We can store the time stamps as u8's
		nBitsOfTime = 8;
		ADDR_FSTime = 		FMATH_BYTE_ALIGN_UP( nDataBytes, 1 );
		nDataBytes  =		ADDR_FSTime + sizeof( u8 ) * nTotalSKeys;
		ADDR_FTTime = 		FMATH_BYTE_ALIGN_UP( nDataBytes, 1 );
		nDataBytes  =		ADDR_FTTime + sizeof( u8 ) * nTotalTKeys;
		ADDR_FOTime = 		FMATH_BYTE_ALIGN_UP( nDataBytes, 1 );
		nDataBytes  =		ADDR_FOTime + sizeof( u8 ) * nTotalOKeys;
	}
	else if ( bCompressSecs && fTotalAnimSecs < (32768.f * fMinimumTimeBetweenFrames) )
	{
		// We can store the time stamps as u16's
		nBitsOfTime = 16;
		ADDR_FSTime = 		FMATH_BYTE_ALIGN_UP( nDataBytes, 2 );
		nDataBytes  =		ADDR_FSTime + sizeof( u16 ) * nTotalSKeys;
		ADDR_FTTime = 		FMATH_BYTE_ALIGN_UP( nDataBytes, 2 );
		nDataBytes  =		ADDR_FTTime + sizeof( u16 ) * nTotalTKeys;
		ADDR_FOTime = 		FMATH_BYTE_ALIGN_UP( nDataBytes, 2 );
		nDataBytes  =		ADDR_FOTime + sizeof( u16 ) * nTotalOKeys;
	}
	else
#endif // FANIM_USE_ANIM_COMPRESSION
	{
		// Store it as a float
		nBitsOfTime = 32;
		ADDR_FSTime = 		FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
		nDataBytes  =		ADDR_FSTime + sizeof( f32 ) * nTotalSKeys;
		ADDR_FTTime = 		FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
		nDataBytes  =		ADDR_FTTime + sizeof( f32 ) * nTotalTKeys;
		ADDR_FOTime = 		FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );
		nDataBytes  =		ADDR_FOTime + sizeof( f32 ) * nTotalOKeys;
	}

	// Round it up
	nDataBytes = FMATH_BYTE_ALIGN_UP( nDataBytes, 4 );

	// Allocate memory to contain export data
	u8 *pExportData = (u8 *)fang_MallocAndZero( nDataBytes, FCLASS_BYTE_ALIGN );
	if ( pExportData == NULL ) 
	{
		DEVPRINTF( "CompressAnimationData(): Could not allocate %u bytes.\n", nDataBytes );
		return NULL;
	}

	u8 *pDataEnd = pExportData + nDataBytes;

	////////////////////////////////////////////////////////////////
	// Fill in export data memory
	////////////////////////////////////////////////////////////////

	//////////////////////////
	// PACK THE FAnim_t
	FAnim_t *pAnim = (FAnim_t *)(pExportData + ADDR_FAnim);
	pAnim->szName[0] = 0;
	pAnim->nBoneCount = (u16)nBoneCount;
	pAnim->pBoneArray = (FAnimBone_t *)ADDR_FBoneArray;
	pAnim->fTotalSeconds = fTotalAnimSecs;
	pAnim->fOOTotalSeconds = 1.f / pAnim->fTotalSeconds;
	pAnim->nFlags = FANIM_BONEFLAGS_NONE;
	// Setup flags for the time steps
	if ( nBitsOfTime == 8 )
	{
		pAnim->nFlags |= FANIM_BONEFLAGS_8BIT_SECS;
	}
	else if ( nBitsOfTime == 16 )
	{
		pAnim->nFlags |= FANIM_BONEFLAGS_16BIT_SECS;
	}
	if ( bTCompress )
	{
		pAnim->nFlags |= FANIM_BONEFLAGS_COMP_TRANSLATION;
	}
	if ( bOCompress )
	{
		pAnim->nFlags |= FANIM_BONEFLAGS_COMP_ORIENTATION;
	}
	if ( nMaxBoneOKeys < 256 && nMaxBoneTKeys < 256 && nMaxBoneSKeys < 256 )
	{
		pAnim->nFlags |= FANIM_BONEFLAGS_8BIT_FRAMECOUNT;
	}

	//////////////////////////
	// PACK THE BONE DATA
	FAnimBone_t *pBoneArray = (FAnimBone_t *)(pExportData + ADDR_FBoneArray);
	u8 *pBoneName = pExportData + ADDR_FBoneNames;
	u8 *paSKeyUnitTime = pExportData + ADDR_FSTime;
	u8 *paTKeyUnitTime = pExportData + ADDR_FTTime;
	u8 *paOKeyUnitTime = pExportData + ADDR_FOTime;
	u8 *paSKeyData = pExportData + ADDR_FSKeys;
	u8 *paTKeyData = pExportData + ADDR_FTKeys;
	u8 *paOKeyData = pExportData + ADDR_FOKeys;

	// Cycle through the bones and fill out the data
	pFileBoneArray = (FData_MtxFileBone_t *)(pHeader + 1);
	for( nBone = 0; nBone < pHeader->nNumBones; nBone++, pFileBoneArray++ ) 
	{
		if( pFileBoneArray->nNumFrames < 2 ) 
		{
			// Key count is too small. Skip this bone...
			continue;
		}

		FASSERT( (u8 *)pBoneArray < pDataEnd );
		FASSERT( (u8 *)pBoneName < pDataEnd );

		// Copy the bone name over
		pBoneArray->pszName = (char *)((u32)pBoneName - (u32)pExportData);
		fclib_strcpy( (char *)pBoneName, pFileBoneArray->szName );
		pBoneName += strlen( pFileBoneArray->szName );
		(*pBoneName) = 0;
		pBoneName++;

		pBoneArray->nSKeyCount = 0;
		pBoneArray->nTKeyCount = 0;
		pBoneArray->nOKeyCount = 0;

		pBoneArray->paSKeyUnitTime = (void *)((u32)paSKeyUnitTime - (u32)pExportData);
		pBoneArray->paTKeyUnitTime = (void *)((u32)paTKeyUnitTime - (u32)pExportData);
		pBoneArray->paOKeyUnitTime = (void *)((u32)paOKeyUnitTime - (u32)pExportData);
		
		pBoneArray->paSKeyData = (void *)((u32)paSKeyData - (u32)pExportData);
		pBoneArray->paTKeyData = (void *)((u32)paTKeyData - (u32)pExportData);
		pBoneArray->paOKeyData = (void *)((u32)paOKeyData - (u32)pExportData);

		// Copy the data
		FData_MtxFileMatrix_t *pFileKeyArray = (FData_MtxFileMatrix_t *)((u32)pHeader + (u32)pFileBoneArray->nFrameArrayOffset);
		qNextOrient.BuildQuat( pFileKeyArray[0].Mtx, TRUE );
		fLastScale = -1.f; fCurrScale = -1.f;
		fNextScale = pFileKeyArray[0].Mtx.m_vRight.Mag();
		if ( fNextScale > 0.99f && fNextScale < 1.01f )
		{
			fNextScale = 1.f;
		}
		for ( nKey = 0; nKey < pFileBoneArray->nNumFrames; nKey++ ) 
		{
			fFrameTime = pFileKeyArray[nKey].fStartingSecs;
			fNextFrameTime = pFileKeyArray[nKey+1].fStartingSecs;

			// Check to see if we need to include this data as a scale keyframe
			fCurrScale = fNextScale;
			if ( nKey != pFileBoneArray->nNumFrames - 1 )
			{
				fNextScale = pFileKeyArray[nKey + 1].Mtx.m_vRight.Mag();
				if ( fNextScale > 0.99f && fNextScale < 1.01f )
				{
					fNextScale = 1.f;
				}
			}
			if ( nKey == 0 || nKey == pFileBoneArray->nNumFrames - 1 )
			{
				_CompressTimeStep( &paSKeyUnitTime, pAnim->nFlags, fTotalAnimSecs, pFileKeyArray[nKey].fStartingSecs, (nKey == pFileBoneArray->nNumFrames - 1), bConvertToBigEndian );
				
				if ( bConvertToBigEndian )
				{
					((FAnimSKey_t *)paSKeyData)->fScale = fang_ConvertEndian( fCurrScale );
				}
				else
				{
					((FAnimSKey_t *)paSKeyData)->fScale = fCurrScale;
				}
				pBoneArray->nSKeyCount++;
				paSKeyData += sizeof( FAnimSKey_t );
				fLastScale = fCurrScale;
				fLastScaleTime = fFrameTime;
			}
			else
			{
				fSlider = (fFrameTime - fLastScaleTime)/(fNextFrameTime - fLastScaleTime);
				FASSERT( fSlider >= 0.f && fSlider <= 1.f );
				f32 fInterpScale = (fLastScale * (1 - fSlider)) + (fNextScale * fSlider);
				FASSERT( fInterpScale >= 0.f );
				if ( fInterpScale > 0.99f && fInterpScale < 1.01f )
				{
					fInterpScale = 1.f;
				}
				if ( fCurrScale != fInterpScale )
				{
					FASSERT( fCurrScale >= 0.f );
					_CompressTimeStep( &paSKeyUnitTime, pAnim->nFlags, fTotalAnimSecs, pFileKeyArray[nKey].fStartingSecs, (nKey == pFileBoneArray->nNumFrames - 1), bConvertToBigEndian );
					if ( bConvertToBigEndian )
					{
						((FAnimSKey_t *)paSKeyData)->fScale = fang_ConvertEndian( fCurrScale );
					}
					else
					{
						((FAnimSKey_t *)paSKeyData)->fScale = fCurrScale;
					}
					pBoneArray->nSKeyCount++;
					paSKeyData += sizeof( FAnimSKey_t );
					fLastScale = fInterpScale;
					fLastScaleTime = fFrameTime;
				}
			}

			// Build a quat to represent the matrix
			qOrient.Set( qNextOrient );
			if (  nKey != pFileBoneArray->nNumFrames - 1 )
			{
				qNextOrient.BuildQuat( pFileKeyArray[nKey+1].Mtx, TRUE );
			}

			// Make sure the orientation quat didn't flip 
			if ( nKey > 0 )
			{
				f32 fFlip = (qOrient.x * qLastOrient.x) + (qOrient.y * qLastOrient.y) + (qOrient.z * qLastOrient.z) + (qOrient.w * qLastOrient.w);
				if (  fFlip < 0 )
				{
					qOrient.x = -qOrient.x;
					qOrient.y = -qOrient.y;
					qOrient.z = -qOrient.z;
					qOrient.w = -qOrient.w;
				}
			}

//			CFMtx43 mtxTest;
//			qOrient.BuildMtx( mtxTest );

			// Is this translation key frame significant enough to store?
			BOOL bStoreTrans = FALSE;
			if ( nKey == 0 || nKey == pFileBoneArray->nNumFrames - 1 )
			{
				// Always keep the first and last frames
				bStoreTrans = TRUE;
			}
			else
			{
				// Test other frames to see if they are linear
				fSlider = (fFrameTime - fLastTransTime)/(fNextFrameTime - fLastTransTime);
				vTest.ReceiveLerpOf( fSlider, qLastTrans.m_Pos, qNextOrient.m_Pos );
				if ( fanim_SignificantTranslationDiff( &qOrient.m_Pos, &vTest, bTCompress ) )
				{
					bStoreTrans = TRUE;
				}
			}

			if ( bStoreTrans )
			{
				// Yes it is!
				FASSERT( paTKeyUnitTime < pDataEnd );
				FASSERT( (u8 *)paTKeyData < pDataEnd );

				_CompressTimeStep( &paTKeyUnitTime, pAnim->nFlags, fTotalAnimSecs, pFileKeyArray[nKey].fStartingSecs, (nKey == pFileBoneArray->nNumFrames - 1), bConvertToBigEndian );

				if ( bTCompress )
				{
					((FAnimCTKey_t *)paTKeyData)->Set( qOrient.m_Pos );
					// Convert endian if needed
					if ( bConvertToBigEndian )
					{
						((FAnimCTKey_t *)paTKeyData)->ChangeEndian();
					}
					paTKeyData += sizeof( FAnimCTKey_t );
				}
				else
				{
					((FAnimTKey_t *)paTKeyData)->Set( qOrient.m_Pos );
					// Convert endian if needed
					if ( bConvertToBigEndian )
					{
						((FAnimTKey_t *)paTKeyData)->ChangeEndian();
					}
					paTKeyData += sizeof( FAnimTKey_t );
				}
				qLastTrans.Set( qOrient );
				fLastTransTime = fFrameTime;
				pBoneArray->nTKeyCount++;
			}

			// Is this orientation key frame significant enough to store?
			BOOL bStoreOrient = FALSE;
			if ( nKey == 0 || nKey == pFileBoneArray->nNumFrames - 1 )
			{
				// Always store the first and last frames
				bStoreOrient = TRUE;
			}
			else
			{
				// For the rest, check to see if they are linear
				fSlider = (fFrameTime - fLastOrientTime)/(fNextFrameTime - fLastOrientTime);
				qTest.ReceiveSlerpOf( fSlider, qLastOrient, qNextOrient );
				if ( _SignificantOrientationDiff( &qLastOrient, &qTest, bOCompress, fUnitTolerance ) )
				{
					bStoreOrient = TRUE;
				}
			}
			
			if ( bStoreOrient )
			{
				// Yes it is!
				FASSERT( paOKeyUnitTime < pDataEnd );
				FASSERT( (u8 *)paOKeyData < pDataEnd );
				FASSERT( paOKeyData < (paOKeyData + anOKeyFrameCount[nBone] ) );

				_CompressTimeStep( &paOKeyUnitTime, pAnim->nFlags, fTotalAnimSecs, pFileKeyArray[nKey].fStartingSecs, (nKey == pFileBoneArray->nNumFrames - 1), bConvertToBigEndian );

				if ( bOCompress )
				{
					((FAnimCOKey_t *)paOKeyData)->Set( qOrient );
					// Convert endian if needed
					if ( bConvertToBigEndian )
					{
						((FAnimCOKey_t *)paOKeyData)->ChangeEndian();
					}
					paOKeyData += sizeof( FAnimCOKey_t );
				}
				else
				{
					((FAnimOKey_t *)paOKeyData)->Set( qOrient );
					// Convert endian if needed
					if ( bConvertToBigEndian )
					{
						((FAnimOKey_t *)paOKeyData)->ChangeEndian();
					}
					paOKeyData += sizeof( FAnimOKey_t );
				}
				qLastOrient.Set( qOrient );
				fLastOrientTime = fFrameTime;
				pBoneArray->nOKeyCount++;
			}
			FASSERT( pBoneArray->nSKeyCount <= anSKeyFrameCount[nBone] );
			FASSERT( pBoneArray->nTKeyCount <= anTKeyFrameCount[nBone] );
			FASSERT( pBoneArray->nOKeyCount <= anOKeyFrameCount[nBone] );
		}

		FASSERT( pBoneArray->nSKeyCount == anSKeyFrameCount[nBone] );
		FASSERT( pBoneArray->nTKeyCount == anTKeyFrameCount[nBone] );
		FASSERT( pBoneArray->nOKeyCount == anOKeyFrameCount[nBone] );

		// Convert endian if needed
		if ( bConvertToBigEndian )
		{
			pBoneArray->ChangeEndian();
		}
		pBoneArray++;
	}

	// Convert endian if needed
	if ( bConvertToBigEndian )
	{
		pAnim->ChangeEndian();
	}

	// Record the new size
	rnNumCompressedBytes = nDataBytes;

	return pExportData;
}

