/*
Copyright (C) 2019 Apogee Software, Ltd.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, see <http://www.gnu.org/licenses/>.

*/

// scriplib.h

#ifndef _scriplib_public
#define _scriplib_public

#define        MAXTOKEN    128
extern	char	token[MAXTOKEN];

//====================================================================
//
// ScripLib
//
//====================================================================
// NOTE: all section names and entry names are case sensitive,
// so be careful when mixing and matching names, Music is not equivalent
// to MUSIC.


//
// loads a script into memory for READ functions
// if tokenized is true script isn't grabbed in a typical Windows format,
// instead it is accessible through GetToken, TokenAvailable and ResetScript.
//
void LoadScript ( char * filename, boolean tokenized );

//
// frees a script from memory. Subsequent Loadscripts will first call
// FreeScript if a script is already in memory.
//
void FreeScript ( void );

//
// Resets a script to the first line when accessing it in a tokenized format
//
void ResetScript (void);

//
// Signals that the current token was not used, and should be reported
// for the next GetToken.  Note that
//
// GetToken (true);
// UnGetToken ();
// GetToken (false);
//
// could cross a line boundary.
//
void UnGetToken (void);

//
// Get the next token
// crossline determines whether or not to go to the next line or stay on the
// current one.
// entireline determines whether or not to grab the entire line or just
// one token
//

void GetToken (boolean crossline, boolean entireline);

//
// whether or not their are any more tokens available
// if crossline is true lines are crossed
//
boolean TokenAvailable (boolean crossline);

//
// Number of sections in script
//

int32 NumberOfScriptSections( void );

//
// A specific section name by number
//
char * ScriptSection( int32 which );

//
// Number of entries in a particular section
//
int32 NumberOfScriptEntries( char * sectionname );

//
// A specific entry by number for a section
//
char * ScriptEntry( char * sectionname, int32 which );

//
// returns pointer to value of entry, NULL if it doesn't exist
//
char * ReadScriptRaw( char * sectionname, char * entryname );

//
// returns string denoted by entry, or unmodified if not found
//
void ReadScriptString( char * sectionname, char * entryname, char * dest );

//
// returns double strings denoted by entry, or unmodified if not found
//
void ReadScriptDoubleString( char * sectionname, char * entryname, char * dest1, char * dest2 );

//
// returns number denoted by entry, or unmodified if not found
// hexadecimal and decimal are automatically parsed.
// returns true if value is the default value
//
boolean ReadScriptNumber
   (
   char * sectionname,
   char * entryname,
   int32 * number
   );

//
// returns a floating point double pointed to by entryname
//
/*
void ReadScriptDouble
   (
   char * sectionname,
   char * entryname,
   double * number
   );
*/

//
// Write out directly to script file without using entries
//
void WriteScript( char * string );

//
// Write out printf style comment
//
void WriteScriptMultiComment( char * comment, ... );

//
// Write out a raw entry no special characters are added
//
void WriteScriptRaw( char * entryname, char * raw );

//
// Write out a string which is surround by string terminators
//
void WriteScriptString( char * entryname, char * string );

//
// Write out a double string which is surround by string terminators
//
void WriteScriptDoubleString( char * entryname, char * string1, char * string2 );

//
// Write out a Number either decimal or hexadecimal or a default value
//
void WriteScriptNumber
   (
   char * entryname,
   int32 number,
   boolean hexadecimal,
   boolean defaultvalue
   );

//
// Write out a floating point double
//

/*
void WriteScriptDouble
   (
   char * entryname,
   double number,
   boolean defaultvalue
   );
*/

//
// Write out a section header which causes all subsequent entries to be
// part of that section
//
void WriteScriptSectionHeader( char * sectionname );

//
// Write out a comment
//
void WriteScriptComment( char * comment );

//
// Write out an end of line character
//
void WriteScriptEOL( void );

//
// Open a script file for writing, must be accompanied by the Close function
//
void OpenSaveScript (char * filename);

//
// Close the script file OpenSaveScript must be called first
//
void CloseSaveScript ( void );

//
// Token form of get integer
//
void GetInteger(int32 * number, boolean crossline);

//
// Token form of get double
//
void GetDouble(double * number, boolean crossline);

//
// Look for a specific token and skip over the script until
// we find it
//
void    GetSpecificToken(char * );

//
// Count the number of lines in the file with valid data (non-comments)
//
int32 CountLinesInFile( void );

#endif
