/* This program is provided AS-IS with NO WARRANTY OF ANY KIND! */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define LINE_BUFFER_LEN 128
#define DATA_BUFFER_LEN 1024

#if ((defined __DOS__) || (defined MSDOS) || (defined WIN32) || (defined _WIN32))
#define PATH_SEPS "/\\"
#else
#define PATH_SEPS "/"
#endif

static void print_help_and_quit(const char *argzero) {
	puts("");
	puts("STRIPHDR - Code header stripper");
	puts("===============================");
	puts("");
	puts("Strips headers from C/C++ sources for which the following conditions are met:");
	puts("1. The very first line is a one-line comment, beginning with //.");
	puts("2. The next line begins with /*, creating a comments block ending with */.");
	puts("3. There are at least two more lines belonging to this block.");
	puts("4. The block is followed by another one-line comment (beginning with //).");
	printf("5. Each of the block's lines is shorter than %d characters long,\n", LINE_BUFFER_LEN);
	puts("including any newline character.");
	puts("");
	puts("If there's an additional following empty line, it's stripped as well.");
	puts("");
	puts("In case not all conditions are met for a given file, it's copied as-is.");
	puts("");
	puts("CR-LF and LF line endings are both supported.");
	puts("");
	printf("Usage: %s <destDir> <file1> [<file2> ...]\n", argzero);
	puts("");
	puts("Please note that ONLY the files' internal filenames are used for copying.");
	puts("");
	exit(0);
}

static const char *get_basename_ptr(const char *filepath) {
	const char *ptr = filepath + strlen(filepath) - 1;
	for (; ptr > filepath; --ptr) {
		if (strchr(PATH_SEPS, *ptr) != NULL) {
			return 1+ptr;
		}
	}
	return ptr;
}

/* ASSUMPTION: "buff" is a valid pointer to a
 * buffer WITH ENOUGH ROOM FOR ALL WRITTEN STRINGS!
 */
static void process_file(const char *dstdir, const char *filepath, char *fnamebuff, int fnamebufflen) {
	FILE *inf, *outf;
	int revert_file = 1;
	char line[LINE_BUFFER_LEN];
	char databuff[DATA_BUFFER_LEN];
	const char *basename;

	inf = fopen(filepath, "rb");
	if (!inf) {
		fprintf(stderr, "ERROR: Couldn't open file for reading: %s\n", filepath);
		exit(2);
	}

	/* Check first few lines lines */
	if ((fgets(line, sizeof(line), inf) != NULL) &&
	    (strncmp(line, "//", 2) == 0) &&
	    (fgets(line, sizeof(line), inf) != NULL) &&
	    (strncmp(line, "/*", 2) == 0) &&
	    (strstr(line+2, "*/") == NULL) &&
	    (fgets(line, sizeof(line), inf) != NULL) &&
	    (strstr(line, "*/") == NULL)
	) {
		/* Look for the line that ends the block */
		while (fgets(line, sizeof(line), inf) != NULL) {
			if ((strstr(line, "*/\r\n") != NULL) || (strstr(line, "*/\n") != NULL)) {
				revert_file = 0;
				break;
			}
		}

		if (!revert_file) { /* Check for another one-line comment */
			if ((fgets(line, sizeof(line), inf) == NULL) ||
			    (strncmp(line, "//", 2) != 0)
			) {
				revert_file = 1;
			}
		}

		if (!revert_file) { /* Skip additional empty line if there's any */
			long offset = ftell(inf);
			if ((fgets(line, sizeof(line), inf) != NULL) &&
			    ((*line == '\n') || ((*line == '\r') && (*(line+1) == '\n')))
			) {
				offset = ftell(inf);
			} else {
				fseek(inf, offset, SEEK_SET);
			}
		}
	}

	if (revert_file) {
		fseek(inf, 0, SEEK_SET);
	}

	
	basename = get_basename_ptr(filepath);
	strcpy(fnamebuff, dstdir);
	strcat(fnamebuff, PATH_SEPS + strlen(PATH_SEPS) - 1); // HACK
	strcat(fnamebuff, basename);

	outf = fopen(fnamebuff, "wb");
	if (!outf) {
		fprintf(stderr, "ERROR: Couldn't open file for writing: %s\n", fnamebuff);
		fclose(inf);
		exit(3);
	}

	while (1) {
		size_t read_len = fread(databuff, 1, DATA_BUFFER_LEN, inf);
		if (read_len == 0) {
			break;
		}
		if (fwrite(databuff, 1, read_len, outf) != read_len) {
			fprintf(stderr, "ERROR: Got error while writing to file: %s\n", fnamebuff);
			fclose(outf);
			fclose(inf);
			exit(4);
		}
	}

	fclose(outf);
	fclose(inf);
}

int main(int argc, char **argv) {
	const char *dstdir;
	char **arg;
	const char *argzero = argv[0];
	char *buff;
	int argnum, bufflen;
	if (argc < 3) {
		print_help_and_quit(argzero);
	}

	argv++;
	dstdir = *argv++;
	argc -= 2;
	if (dstdir == NULL) {
		print_help_and_quit(argzero);
	}
	bufflen = 0;
	for (argnum = 0, arg = argv; argnum < argc; ++argnum, ++arg) {
		const char *basename;
		if (*arg == NULL) {
			print_help_and_quit(argzero);
		}
		basename = get_basename_ptr(*arg);
		if (bufflen < *arg + strlen(*arg) - basename) {
			bufflen = *arg + strlen(*arg) - basename;
		}
	}
	bufflen += 1 + strlen(dstdir); /* Add dest. dir. + path separator */

	buff = (char *)malloc(bufflen);
	if (buff == NULL) {
		fprintf(stderr, "ERROR: OUT OF MEMORY!\n");
		exit(1);
	}

	for (; argc > 0; argv++, argc--) {
		process_file(dstdir, *argv, buff, bufflen);
	}

	return 0;
}
